/*
 * Copyright 2006-2008 The FLWOR Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <zorba/zorba.h>

#include <fstream>
#include <iostream>
#include <iomanip>
#include <sstream>

// tests are allowed to use internals
#include "api/unmarshaller.h"
#include "system/properties.h"

#include <zorba/store_manager.h>
#include <zorba/iterator.h>
#include <zorba/util/fs_util.h>
#include <zorba/xquery_exception.h>
#include <zorba/internal/unique_ptr.h>

// Global variable g_abort_on_error is used to generate an abort() when an
// error is encountered, to aid debugging
#ifndef NDEBUG
#ifdef BUILDING_ZORBA_STATIC
extern bool g_abort_on_error;
#else
ZORBA_DLL_PUBLIC bool g_abort_on_error;
#endif
#endif


using namespace zorba;
using namespace std;


void set_var (string name, string val, DynamicContext* dctx)
{
  if (name [name.size () - 1] == ':')
  {
    name = name.substr (0, name.size () - 1);
    Item lItem = Zorba::getInstance(NULL)->getItemFactory()->createString(val);
    if(name != ".") {
      dctx->setVariable(name, lItem);
    } else
      dctx->setContextItem(lItem);
  }
  else if (name[name.size () - 1] != ':')
  {
    ifstream is(val.c_str());
    assert (is);
    try {
      XmlDataManager_t lXmlMgr = Zorba::getInstance(NULL)->getXmlDataManager();
      Item lDoc = lXmlMgr->parseXML(is);
      assert (lDoc.getNodeKind() == zorba::store::StoreConsts::documentNode);
      if(name != ".")
        dctx->setVariable(name, lDoc);
      else
        dctx->setContextItem(lDoc);
    } catch (zorba::ZorbaException& e) {
      std::cerr << "could not set external variable "  << e << std::endl;
      exit(1);
    }
  }
}


#ifndef _WIN32_WCE
int main(int argc, char* argv[])
#else
int _tmain(int argc, _TCHAR* argv[])
#endif
{
  // read the command file properties
  if (! Properties::load(argc,argv))
    return 4;

  Properties* lProp = Properties::instance();
  if (! lProp->hasSingleQuery ()) {
    cout << "Error: either a single inline query or a single query file must be supplied" << endl;
    return 4;
  }
  Zorba_CompilerHints chints;
  switch (lProp->optimizer()) {
  case 0:
    chints.opt_level = ZORBA_OPT_LEVEL_O0;
    break;
  case 1:
    chints.opt_level = ZORBA_OPT_LEVEL_O1;
    break;
  case 2:
    chints.opt_level = ZORBA_OPT_LEVEL_O1;
    break;
  default:
    chints.opt_level = ZORBA_OPT_LEVEL_O1;
  }

  chints.for_serialization_only = false;

  if (Properties::instance()->serializeOnlyQuery() > 0)
  {
    chints.for_serialization_only = true;
  }

  // default is false
  if (lProp->libModule())
  {
    chints.lib_module = true;
  }

  // output file (either a file or the standard out if no file is specified)
  unique_ptr<ostream> outputFile (lProp->resultFile ().empty ()
                                ? NULL : new ofstream (lProp->resultFile().c_str()));
  ostream *resultFile = outputFile.get ();
  if (resultFile == NULL)
    resultFile = &cout;

  // input file (either from a file or given as parameter)
  unique_ptr<istream> qfile;
  std::string path;

  if (! lProp->inlineQuery()) 
  {
    path = lProp->queryFile ();
    fs::make_absolute( &path );
    qfile.reset (new ifstream (path.c_str ()));
    if (!qfile->good() || qfile->eof()) 
    {
      cerr << "no query given or not readable " << path << endl;
      return 3;
    }
  }
  else 
  {
    qfile.reset (new istringstream(lProp->query ()));
  }

  // print the query if requested
  if (lProp->printQuery()) {
    lProp->debug_out ()<< "Query text:\n";
    copy (istreambuf_iterator<char> (*qfile), istreambuf_iterator<char> (), ostreambuf_iterator<char> (lProp->debug_out ()));
    lProp->debug_out () << "\n" << endl;
    qfile->seekg(0); // go back to the beginning
  }

  // Instantiate the simple store
  void* store = zorba::StoreManager::getStore();

  // Set the g_abort_on_exception flag in error_manager.cpp
#ifndef NDEBUG
  if (lProp->abort())
    g_abort_on_error = true;
#endif

  // start processing
  Zorba* zengine = Zorba::getInstance(store);

  zorba::StaticContext_t staticContext = zengine->createStaticContext();

  // start parsing the query
  XQuery_t query = zengine->createQuery ();

#ifdef ZORBA_WITH_DEBUGGER
  if (lProp->debug()) 
  {
    query->setDebugMode(lProp->debug());
    Zorba_CompilerHints lHints;
    lHints.opt_level = ZORBA_OPT_LEVEL_O0;
  }
#endif

  if (! lProp->inlineQuery())
  {
    query->setFileName(path);
  }

  if (lProp->jsoniqParser())
  {
    staticContext->setJSONiqVersion(zorba::jsoniq_version_1_0);
  }

  try 
  {
    query->compile(*qfile, staticContext, chints);
  }
  catch (ZorbaException& e)
  {
    // no need to close because the object is not valid
    cerr << "Compilation error: " << e << endl;
    return 1;
  }

  if (lProp->testPlanSerialization())
  {
    try
    {
      std::string binary_path;
      if (lProp->inlineQuery())
        binary_path = path + ".plan";
      else
        binary_path = "./temp.plan";

      std::ofstream fbinary(binary_path.c_str(), std::ios_base::binary);
      if (!query->saveExecutionPlan(fbinary))
      {
        printf("save execution plan FAILED\n");
        return 0x0badc0de;
      }
      fbinary.close();
      printf("saved execution plan at: %s\n", binary_path.c_str());
    }
    catch(zorba::ZorbaException &err)
    {
      std::cout << err << std::endl;
      return -1;
    }

    // Now load back the plan
    try
    {
      std::string binary_path;
      if (lProp->inlineQuery())
        binary_path = path + ".plan";
      else
        binary_path = "./temp.plan";
      query = zengine->createQuery();
      std::ifstream ifbinary(binary_path.c_str(), std::ios_base::binary);
      if(!ifbinary.is_open())
      {
        std::cout << "cannot open plan " << binary_path << std::endl;
        return 15;
      }

      bool load_ret = query->loadExecutionPlan(ifbinary);

      if (!load_ret)
      {
        std::cout << "cannot load plan " << binary_path << std::endl;
        return 16;
      }

      printf("load execution plan: %s\n", binary_path.c_str());
    }
    catch(zorba::ZorbaException &err)
    {
      std::cout << err << std::endl;
      return -1;
    }
  }

  // set external variables
  vector<pair <string, string> > ext_vars = lProp->getExternalVars();
  DynamicContext* dctx = query->getDynamicContext ();
  dctx->setImplicitTimezone (lProp->tz ());
  for (vector<pair <string, string> >::const_iterator iter = ext_vars.begin ();
       iter != ext_vars.end (); iter++) 
  {
    set_var (iter->first, iter->second, dctx);
  }

  //if you want to print the plan into a file
  if( ! lProp->dotPlanFile().empty () ) 
  {
    unique_ptr<ostream> planFile (new ofstream (lProp->dotPlanFile().c_str()));
    ostream *printPlanFile = planFile.get ();

    query->printPlan(*printPlanFile, true);
  }

  int return_code = 0;
  if (! lProp->compileOnly() && ! lProp->libModule())
  {
    // output the result (either using xml serialization or using show)

    try 
    {
      if (lProp->useSerializer()) 
      {
        Zorba_SerializerOptions const opts(lProp->getSerializerParameters());
        query->execute(*resultFile, &opts);
      }
      else if (lProp->iterPlanTest())
      {
        Iterator_t result = query->iterator();
        result->open();
        Item lItem;
        while (result->next(lItem)) 
        {
          ;
        }
        result->close();
      }
      else
      {
        Iterator_t result = query->iterator();
        result->open();
        Item lItem;
        while (result->next(lItem)) 
        {
          ;
        }
        result->close();
      }
    }
    catch (ZorbaException const &e)
    {
      cerr << "Execution error: " << e << endl;
      return_code = 2;
    }
  }

  staticContext->removeReference();  // force destruction
  staticContext = NULL;
  query->close();  
  zengine->shutdown();
  zorba::StoreManager::shutdownStore(store);
  return return_code;
}
