xquery version "3.0";

(:
 : Copyright 2006-2010 The FLWOR Foundation.
 :
 : Licensed under the Apache License, Version 2.0 (the "License");
 : you may not use this file except in compliance with the License.
 : You may obtain a copy of the License at
 :
 : http://www.apache.org/licenses/LICENSE-2.0
 :
 : Unless required by applicable law or agreed to in writing, software
 : distributed under the License is distributed on an "AS IS" BASIS,
 : WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 : See the License for the specific language governing permissions and
 : limitations under the License.
:)

(:~
 : <p>
 : The goal of xqDoc is to provide a simple vendor neutral solution for
 : documenting XQuery modules, as well as tools to generate a user friendly
 : presentation of this documentation and cross referencing information.
 : Therefore, xqDoc proposes a new commenting convention that extends the
 : currently defined XQuery comment style. This convention is modeled
 : after Java's Javadoc commenting style, and provides a simple, uniform
 : way to document XQuery source code. You can find more information about
 : xqDoc on the website of the <a href="http://xqdoc.org/">xqDoc project</a>.
 : This library module provides XQDoc utility functions.
 : </p>
 : 
 : <p>
 : Generating a user friendly presentation of the documentation is
 : accomplished in the following steps:
 : <ol>
 :  <li>Module, variable, function, collection, and index declarations need
 :      to be commented using the xqDoc commenting conventions. For example,
 :      this module contains xqDoc-style comments</li>
 :  <li>A xqDoc-enabled processor can parse such documentation and generate
 :      a vendor neutral XML document which stores all the information about
 :      the code and the comments. Such a document adheres to the xqDoc
 :      Schema.</li>
 :  <li>The information of an XML document generated by the second step,
 :      can be transformed into arbitrary presentation formats 
 :      (e.g. html).</li>
 : </ol>
 : </p>
 :
 : <p>
 : This module implements the first and second step of this process.
 : That is, Zorba can parse XQuery modules which are annotated with
 : xqDoc-style documentation and generate the vendor neutral
 : XML representation. 
 : </p>
 :
 : @see <a href="http://xqdoc.org/" target="_blank">xqDoc specification</a>
 : @see <a href="http://www.zorba-xquery.com/tutorials/xqdoc.html" target="_blank">xqDoc tutorial with Zorba</a>
 : @author Gabriel Petrovay
 : @project Zorba/XQDoc/XQDoc Core
 :
 :)
module namespace xqd = "http://zorba.io/modules/xqdoc";

import module namespace fetch = "http://zorba.io/modules/fetch";
import module namespace schema = "http://zorba.io/modules/schema";

import schema namespace opt =
  "http://zorba.io/modules/xqdoc-options";

declare namespace an = "http://zorba.io/annotations";
declare namespace ver = "http://zorba.io/options/versioning";

declare namespace err = "http://www.w3.org/2005/xqt-errors";
declare namespace zerr = "http://zorba.io/errors";

declare option ver:module-version "2.0";

(:~
 : Generates an XQDoc XML document for the module located
 : at the URI provided as parameter to this function.
 :
 : @param $module-uri The URL of the module for which to
 :        generate XQDoc.
 : @return An element according to the xqdoc schema
 :  (<tt>http://zorba.io/modules/xqdoc.xsd</tt>).
 : @error zerr::ZXQD0002 if the xqdoc comments in the
 :  module contain invalid XML
 : @deprecated please use the fetch:content#1 and xqd:xqdoc-content#1 function
 :)
declare %an:nondeterministic function xqd:xqdoc(
  $module-uri as xs:string
) as element()
{
  let $content := fetch:content($module-uri, "MODULE")
  return xqd:xqdoc-content-impl($content, $module-uri)
};

(:~
 : Generates an XQDoc XML document for the module located
 : at the URI provided as parameter to this function.
 : In comparison to the single parameter version, this function does not
 : generate XQDoc for all language components. By default, the 
 : following components are deactivated: XQuery comments, import
 : statements, variable declarations, function declarations, collection
 : declarations,  and index declarations. The second parameter is used to
 : enable the XQDoc generation of those components.
 :
 : @param $module-uri The URL of the module for which to
 :        generate XQDoc.
 : @param $options XQDoc generation options, e.g.:
 : <pre>
 : &lt;enable xmlns="http://zorba.io/modules/xqdoc-options"
 :   comments="true"
 :   functions="true"
 :   indexes="true"
 : /&gt;
 : </pre>
 : @return An element according to the xqdoc schema
 :  (<tt>http://zorba.io/modules/xqdoc.xsd</tt>).
 : @error zerr::ZXQD0002 if the xqdoc comments in the
 :  module contain invalid XML
 : @deprecated please use the fetch:content#1 and xqd:xqdoc-content#2 function
 :)
declare function xqd:xqdoc(
  $module-uri as xs:string,
  $options as element(opt:enable)
) as element()
{
  let $content := fetch:content($module-uri, "MODULE")
  let $xqdoc-options := if ( schema:is-validated( $options ) ) then
                              $options
                            else
                              validate { $options }
  return xqd:xqdoc-content-options-impl($content, $module-uri, $xqdoc-options)
};

declare %private function xqd:xqdoc-content-impl(
  $module as xs:string,
  $filename as xs:string
) as element() external;


(:~
 : Generated the an XQDoc XML document for the module provided
 : as parameter to this function.
 :
 : @param $module The module (as string) for which to generate
 :  the XQDoc documentation.
 : @return An element according to the xqdoc schema
 :  (<tt>http://zorba.io/modules/xqdoc.xsd</tt>).
 : @error zerr::ZXQD0002 if the xqdoc comments in the
 :  module contain invalid XML
 :)
declare function xqd:xqdoc-content(
  $module as xs:string
) as element()
{
  xqd:xqdoc-content-impl($module, "")
};


(:~
 : Generated the an XQDoc XML document for the module provided
 : as parameter to this function.
 : In comparison to the single parameter version, this function does not
 : generate XQDoc for all language components. By default, the 
 : following components are deactivated: XQuery comments, import
 : statements, variable declarations, function declarations, collection
 : declarations,  and index declarations. The second parameter is used to
 : enable the XQDoc generation of those components.
 :
 : @param $module The module (as string) for which to generate
 :  the XQDoc documentation.
 : @param $options XQDoc generation options, e.g.:
 : <pre>
 : &lt;enable xmlns="http://zorba.io/modules/xqdoc-options"
 :   comments="true"
 :   functions="true"
 :   indexes="true"
 : &gt;
 : </pre>
 : @return An element according to the xqdoc schema
 :  (<tt>http://zorba.io/modules/xqdoc.xsd</tt>).
 : @error zerr::ZXQD0002 if the xqdoc comments in the
 :  module contain invalid XML
 :)
declare function xqd:xqdoc-content(
  $module as xs:string,
  $options as element(opt:enable)
) as element()
{
  let $xqdoc-options := if ( schema:is-validated( $options ) ) then
                              $options
                            else
                              validate { $options }
  return xqd:xqdoc-content-options-impl($module, "", $xqdoc-options)
};

declare %private function xqd:xqdoc-content-options-impl(
  $module as xs:string,
  $filename as xs:string,
  $options as element()
) as element() external;

