/*
    libzint - the open source barcode library
    Copyright (C) 2019-2022 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/*
 * Adapted from the GNU LIBICONV library and patched to make compatible with
 * https://www.unicode.org/Public/MAPPINGS/OBSOLETE/EASTASIA/JIS/SHIFTJIS.TXT
 * with the exception of one duplicate mapping to reverse solidus 0x815F
 * and the mapping of user-defined PUA codepoints U+E000..E757.
 */
/*
 * Copyright (C) 1999-2002, 2016 Free Software Foundation, Inc.
 * This file is part of the GNU LIBICONV Library.
 *
 * The GNU LIBICONV Library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The GNU LIBICONV Library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the GNU LIBICONV Library; see the file COPYING.LIB.
 * If not, see <https://www.gnu.org/licenses/>.
 */
#ifdef _MSC_VER
#include <malloc.h>
#endif
#include "common.h"
#include "sjis.h"
#include "eci.h"

/*
 * JISX0201.1976-0 (libiconv-1.16/lib/jisx0201.h)
 */

static int jisx0201_wctomb(unsigned int *r, const unsigned int wc) {
    if (wc < 0x0080 && !(wc == 0x005c || wc == 0x007e)) {
        *r = wc;
        return 1;
    }
    if (wc == 0x00a5) {
        *r = 0x5c;
        return 1;
    }
    if (wc == 0x203e) {
        *r = 0x7e;
        return 1;
    }
    if (wc >= 0xff61 && wc < 0xffa0) {
        *r = wc - 0xfec0;
        return 1;
    }
    return 0;
}

/*
 * JISX0208.1990-0 (libiconv-1.16/lib/jisx0208.h)
 */

/* ZINT: Table converted from JIS X 0208 to Shift JIS values using tools/cnv_sjis.php:
 *
while ($line = fgets(STDIN)) {
    echo preg_replace_callback('/0x([0-9a-f]{4})/', function ($matches) {
        $dec = hexdec($matches[1]);
        $c1 = $dec >> 8;
        $c2 = ($dec & 0xFF);
        $t1 = ($c1 - 0x21) >> 1;
        $t2 = ((($c1 - 0x21) & 1) ? 0x5e : 0) + ($c2 - 0x21);
        $r = (($t1 < 0x1f ? ($t1 + 0x81) : ($t1 + 0xc1)) << 8) | ($t2 < 0x3f ? ($t2 + 0x40) : ($t2 + 0x41));
        return '0x' . dechex($r);
    }, $line);
}
 */
static const unsigned short jisx0208_2charset[6879] = {
  0x8191, 0x8192, 0x8198, 0x814e, 0x81ca, 0x818b, 0x817d, 0x814c,
  0x81f7, 0x817e, 0x8180, 0x839f, 0x83a0, 0x83a1, 0x83a2, 0x83a3,
  0x83a4, 0x83a5, 0x83a6, 0x83a7, 0x83a8, 0x83a9, 0x83aa, 0x83ab,
  0x83ac, 0x83ad, 0x83ae, 0x83af, 0x83b0, 0x83b1, 0x83b2, 0x83b3,
  0x83b4, 0x83b5, 0x83b6, 0x83bf, 0x83c0, 0x83c1, 0x83c2, 0x83c3,
  0x83c4, 0x83c5, 0x83c6, 0x83c7, 0x83c8, 0x83c9, 0x83ca, 0x83cb,
  0x83cc, 0x83cd, 0x83ce, 0x83cf, 0x83d0, 0x83d1, 0x83d2, 0x83d3,
  0x83d4, 0x83d5, 0x83d6, 0x8446, 0x8440, 0x8441, 0x8442, 0x8443,
  0x8444, 0x8445, 0x8447, 0x8448, 0x8449, 0x844a, 0x844b, 0x844c,
  0x844d, 0x844e, 0x844f, 0x8450, 0x8451, 0x8452, 0x8453, 0x8454,
  0x8455, 0x8456, 0x8457, 0x8458, 0x8459, 0x845a, 0x845b, 0x845c,
  0x845d, 0x845e, 0x845f, 0x8460, 0x8470, 0x8471, 0x8472, 0x8473,
  0x8474, 0x8475, 0x8477, 0x8478, 0x8479, 0x847a, 0x847b, 0x847c,
  0x847d, 0x847e, 0x8480, 0x8481, 0x8482, 0x8483, 0x8484, 0x8485,
  0x8486, 0x8487, 0x8488, 0x8489, 0x848a, 0x848b, 0x848c, 0x848d,
  0x848e, 0x848f, 0x8490, 0x8491, 0x8476, 0x815d, 0x815c, 0x8161,
  0x8165, 0x8166, 0x8167, 0x8168, 0x81f5, 0x81f6, 0x8164, 0x8163,
  0x81f1, 0x818c, 0x818d, 0x81a6, 0x818e, 0x81f0, 0x81a9, 0x81aa,
  0x81a8, 0x81ab, 0x81cb, 0x81cc, 0x81cd, 0x81dd, 0x81ce, 0x81de,
  0x81b8, 0x81b9, 0x817c, 0x81e3, 0x81e5, 0x8187, 0x81da, 0x81c8,
  0x81c9, 0x81bf, 0x81be, 0x81e7, 0x81e8, 0x8188, 0x81e6, 0x81e4,
  0x81e0, 0x8182, 0x81df, 0x8185, 0x8186, 0x81e1, 0x81e2, 0x81bc,
  0x81bd, 0x81ba, 0x81bb, 0x81db, 0x81dc, 0x849f, 0x84aa, 0x84a0,
  0x84ab, 0x84a1, 0x84ac, 0x84a2, 0x84ad, 0x84a4, 0x84af, 0x84a3,
  0x84ae, 0x84a5, 0x84ba, 0x84b5, 0x84b0, 0x84a7, 0x84bc, 0x84b7,
  0x84b2, 0x84a6, 0x84b6, 0x84bb, 0x84b1, 0x84a8, 0x84b8, 0x84bd,
  0x84b3, 0x84a9, 0x84b9, 0x84be, 0x84b4, 0x81a1, 0x81a0, 0x81a3,
  0x81a2, 0x81a5, 0x81a4, 0x819f, 0x819e, 0x819b, 0x819d, 0x819c,
  0x81fc, 0x819a, 0x8199, 0x818a, 0x8189, 0x81f4, 0x81f3, 0x81f2,
  0x8140, 0x8141, 0x8142, 0x8156, 0x8158, 0x8159, 0x815a, 0x8171,
  0x8172, 0x8173, 0x8174, 0x8175, 0x8176, 0x8177, 0x8178, 0x8179,
  0x817a, 0x81a7, 0x81ac, 0x816b, 0x816c, 0x8160, 0x829f, 0x82a0,
  0x82a1, 0x82a2, 0x82a3, 0x82a4, 0x82a5, 0x82a6, 0x82a7, 0x82a8,
  0x82a9, 0x82aa, 0x82ab, 0x82ac, 0x82ad, 0x82ae, 0x82af, 0x82b0,
  0x82b1, 0x82b2, 0x82b3, 0x82b4, 0x82b5, 0x82b6, 0x82b7, 0x82b8,
  0x82b9, 0x82ba, 0x82bb, 0x82bc, 0x82bd, 0x82be, 0x82bf, 0x82c0,
  0x82c1, 0x82c2, 0x82c3, 0x82c4, 0x82c5, 0x82c6, 0x82c7, 0x82c8,
  0x82c9, 0x82ca, 0x82cb, 0x82cc, 0x82cd, 0x82ce, 0x82cf, 0x82d0,
  0x82d1, 0x82d2, 0x82d3, 0x82d4, 0x82d5, 0x82d6, 0x82d7, 0x82d8,
  0x82d9, 0x82da, 0x82db, 0x82dc, 0x82dd, 0x82de, 0x82df, 0x82e0,
  0x82e1, 0x82e2, 0x82e3, 0x82e4, 0x82e5, 0x82e6, 0x82e7, 0x82e8,
  0x82e9, 0x82ea, 0x82eb, 0x82ec, 0x82ed, 0x82ee, 0x82ef, 0x82f0,
  0x82f1, 0x814a, 0x814b, 0x8154, 0x8155, 0x8340, 0x8341, 0x8342,
  0x8343, 0x8344, 0x8345, 0x8346, 0x8347, 0x8348, 0x8349, 0x834a,
  0x834b, 0x834c, 0x834d, 0x834e, 0x834f, 0x8350, 0x8351, 0x8352,
  0x8353, 0x8354, 0x8355, 0x8356, 0x8357, 0x8358, 0x8359, 0x835a,
  0x835b, 0x835c, 0x835d, 0x835e, 0x835f, 0x8360, 0x8361, 0x8362,
  0x8363, 0x8364, 0x8365, 0x8366, 0x8367, 0x8368, 0x8369, 0x836a,
  0x836b, 0x836c, 0x836d, 0x836e, 0x836f, 0x8370, 0x8371, 0x8372,
  0x8373, 0x8374, 0x8375, 0x8376, 0x8377, 0x8378, 0x8379, 0x837a,
  0x837b, 0x837c, 0x837d, 0x837e, 0x8380, 0x8381, 0x8382, 0x8383,
  0x8384, 0x8385, 0x8386, 0x8387, 0x8388, 0x8389, 0x838a, 0x838b,
  0x838c, 0x838d, 0x838e, 0x838f, 0x8390, 0x8391, 0x8392, 0x8393,
  0x8394, 0x8395, 0x8396, 0x8145, 0x815b, 0x8152, 0x8153, 0x88ea,
  0x929a, 0x8eb5, 0x969c, 0x8fe4, 0x8e4f, 0x8fe3, 0x89ba, 0x9573,
  0x975e, 0x98a0, 0x894e, 0x8a8e, 0x98a1, 0x90a2, 0x99c0, 0x8b75,
  0x95b8, 0x8fe5, 0x97bc, 0x95c0, 0x98a2, 0x9286, 0x98a3, 0x8bf8,
  0x98a4, 0x8adb, 0x924f, 0x8ee5, 0x98a5, 0x98a6, 0x98a7, 0x9454,
  0x8b76, 0x9456, 0x93e1, 0x8cc1, 0x9652, 0xe568, 0x98a8, 0x8fe6,
  0x98a9, 0x89b3, 0x8be3, 0x8cee, 0x96e7, 0x9ba4, 0x9790, 0x93fb,
  0x8aa3, 0x8b54, 0x98aa, 0x98ab, 0x97b9, 0x975c, 0x9188, 0x98ad,
  0x8e96, 0x93f1, 0x98b0, 0x895d, 0x8cdd, 0x8cdc, 0x88e4, 0x986a,
  0x9869, 0x8db1, 0x889f, 0x98b1, 0x98b2, 0x98b3, 0x9653, 0x98b4,
  0x8cf0, 0x88e5, 0x9692, 0x8b9c, 0x8b9d, 0x8b9e, 0x92e0, 0x97ba,
  0x98b5, 0x98b6, 0x98b7, 0x906c, 0x8f59, 0x906d, 0x98bc, 0x98ba,
  0x98bb, 0x8b77, 0x8da1, 0x89ee, 0x98b9, 0x98b8, 0x95a7, 0x8e65,
  0x8e64, 0x91bc, 0x98bd, 0x9574, 0x90e5, 0x8157, 0x98be, 0x98c0,
  0x91e3, 0x97df, 0x88c8, 0x98bf, 0x89bc, 0x8bc2, 0x9287, 0x8c8f,
  0x98c1, 0x9443, 0x8ae9, 0x98c2, 0x88c9, 0x8cde, 0x8aea, 0x959a,
  0x94b0, 0x8b78, 0x89ef, 0x98e5, 0x9360, 0x948c, 0x98c4, 0x94ba,
  0x97e0, 0x904c, 0x8e66, 0x8e97, 0x89be, 0x92cf, 0x9241, 0x98c8,
  0x88ca, 0x92e1, 0x8f5a, 0x8db2, 0x9743, 0x91cc, 0x89bd, 0x98c7,
  0x975d, 0x98c3, 0x98c5, 0x8dec, 0x98c6, 0x9b43, 0x98ce, 0x98d1,
  0x98cf, 0x89c0, 0x95b9, 0x98c9, 0x98cd, 0x8cf1, 0x8e67, 0x8aa4,
  0x98d2, 0x98ca, 0x97e1, 0x8e98, 0x98cb, 0x98d0, 0x98d3, 0x98cc,
  0x8b9f, 0x88cb, 0x8ba0, 0x89bf, 0x9b44, 0x9699, 0x958e, 0x8cf2,
  0x904e, 0x97b5, 0x95d6, 0x8c57, 0x91a3, 0x89e2, 0x8f72, 0x98d7,
  0x98dc, 0x98da, 0x98d5, 0x91ad, 0x98d8, 0x98db, 0x98d9, 0x95db,
  0x98d6, 0x904d, 0x9693, 0x98dd, 0x98de, 0x8f43, 0x98eb, 0x946f,
  0x9555, 0x98e6, 0x95ee, 0x89b4, 0x98ea, 0x98e4, 0x98ed, 0x9171,
  0x8cc2, 0x947b, 0xe0c5, 0x98ec, 0x937c, 0x98e1, 0x8cf4, 0x8cf3,
  0x98df, 0x8ed8, 0x98e7, 0x95ed, 0x926c, 0x98e3, 0x8c91, 0x98e0,
  0x98e8, 0x98e2, 0x97cf, 0x98e9, 0x9860, 0x8be4, 0x8c90, 0x98ee,
  0x98ef, 0x98f3, 0x88cc, 0x95ce, 0x98f2, 0x98f1, 0x98f5, 0x98f4,
  0x92e2, 0x8c92, 0x98f6, 0x8ec3, 0x91a4, 0x92e3, 0x8bf4, 0x98f7,
  0x8b55, 0x98f8, 0x98fa, 0x9654, 0x8c86, 0x8e50, 0x94f5, 0x98f9,
  0x8dc3, 0x9762, 0x98fc, 0x9942, 0x98fb, 0x8dc2, 0x8f9d, 0x8c58,
  0x9943, 0x8bcd, 0x9940, 0x9941, 0x93ad, 0x919c, 0x8ba1, 0x966c,
  0x9944, 0x97bb, 0x9945, 0x9948, 0x9946, 0x916d, 0x9947, 0x9949,
  0x994b, 0x994a, 0x95c6, 0x8b56, 0x994d, 0x994e, 0x89ad, 0x994c,
  0x8ef2, 0x9951, 0x9950, 0x994f, 0x98d4, 0x9952, 0x8f9e, 0x9953,
  0x9744, 0x96d7, 0x9955, 0x9954, 0x9957, 0x9956, 0x9958, 0x9959,
  0x88f2, 0x8cb3, 0x8c5a, 0x8f5b, 0x929b, 0x8ba2, 0x90e6, 0x8cf5,
  0x8d8e, 0x995b, 0x96c6, 0x9365, 0x8e99, 0x995a, 0x995c, 0x937d,
  0x8a95, 0x995d, 0x93fc, 0x9153, 0x995f, 0x9960, 0x94aa, 0x8cf6,
  0x985a, 0x9961, 0x8ba4, 0x95ba, 0x91b4, 0x8bef, 0x9354, 0x8c93,
  0x9962, 0x9963, 0x93e0, 0x897e, 0x9966, 0x8dfb, 0x9965, 0x8dc4,
  0x9967, 0xe3ec, 0x9968, 0x9660, 0x9969, 0x996a, 0x996b, 0x8fe7,
  0x8eca, 0x8aa5, 0x996e, 0x996c, 0x96bb, 0x996d, 0x9579, 0x996f,
  0x9970, 0x9971, 0x937e, 0x9975, 0x9973, 0x9974, 0x9972, 0x8de1,
  0x9976, 0x96e8, 0x97e2, 0x9977, 0x90a6, 0x9978, 0x8f79, 0x9979,
  0x929c, 0x97bd, 0x9380, 0x99c3, 0x997a, 0xeaa3, 0x8bc3, 0x997b,
  0x967d, 0x8f88, 0x91fa, 0x997d, 0x93e2, 0x997e, 0x9980, 0x8a4d,
  0x9981, 0x8ba5, 0x93ca, 0x899a, 0x8f6f, 0x949f, 0x9982, 0x9381,
  0x906e, 0x9983, 0x95aa, 0x90d8, 0x8aa0, 0x8aa7, 0x9984, 0x9986,
  0x8c59, 0x9985, 0x97f1, 0x8f89, 0x94bb, 0x95ca, 0x9987, 0x9798,
  0x9988, 0x9989, 0x939e, 0x998a, 0x90a7, 0x8dfc, 0x8c94, 0x998b,
  0x8e68, 0x8d8f, 0x92e4, 0x998d, 0x91a5, 0x8ded, 0x998e, 0x998f,
  0x914f, 0x998c, 0x9991, 0x9655, 0x8d84, 0x9990, 0x8c95, 0x8ddc,
  0x948d, 0x9994, 0x9992, 0x959b, 0x8fe8, 0x999b, 0x8a84, 0x9995,
  0x9993, 0x916e, 0x9997, 0x9996, 0x8a63, 0x8c80, 0x999c, 0x97ab,
  0x9998, 0x999d, 0x999a, 0x9999, 0x97cd, 0x8cf7, 0x89c1, 0x97f2,
  0x8f95, 0x9377, 0x8d85, 0x99a0, 0x99a1, 0x97e3, 0x984a, 0x99a3,
  0x8cf8, 0x99a2, 0x8a4e, 0x99a4, 0x9675, 0x92ba, 0x9745, 0x95d7,
  0x99a5, 0xe8d3, 0x93ae, 0x99a6, 0x8aa8, 0x96b1, 0x8f9f, 0x99a7,
  0x95e5, 0x99ab, 0x90a8, 0x99a8, 0x8bce, 0x99a9, 0x8aa9, 0x8c4d,
  0x99ac, 0x99ad, 0x99ae, 0x99af, 0x8ed9, 0x8cf9, 0x96dc, 0x96e6,
  0x93f5, 0x95ef, 0x99b0, 0x99b1, 0x99b3, 0x99b5, 0x99b4, 0x99b6,
  0x89bb, 0x966b, 0x8dfa, 0x99b7, 0x9178, 0x8fa0, 0x8ba7, 0x99b8,
  0x94d9, 0x99b9, 0x99ba, 0x99bb, 0x99bc, 0x9543, 0x8be6, 0x88e3,
  0x93bd, 0x99bd, 0x8f5c, 0x90e7, 0x99bf, 0x99be, 0x8fa1, 0x8cdf,
  0x99c1, 0x94bc, 0x99c2, 0x94da, 0x91b2, 0x91ec, 0x8ba6, 0x93ec,
  0x9250, 0x948e, 0x966d, 0x99c4, 0x90e8, 0x8c54, 0x99c5, 0x99c6,
  0x894b, 0x88f3, 0x8aeb, 0x91a6, 0x8b70, 0x9791, 0x99c9, 0x89b5,
  0x99c8, 0x8ba8, 0x99ca, 0x96ef, 0x99cb, 0x97d0, 0x8cfa, 0x8cb4,
  0x99cc, 0x99ce, 0x99cd, 0x907e, 0x8958, 0x897d, 0x99cf, 0x99d0,
  0x8cb5, 0x99d1, 0x8b8e, 0x8e51, 0x99d2, 0x9694, 0x8db3, 0x8b79,
  0x9746, 0x916f, 0x94bd, 0x8efb, 0x8f66, 0x8ee6, 0x8ef3, 0x8f96,
  0x94be, 0x99d5, 0x8962, 0x9170, 0x8cfb, 0x8cc3, 0x8be5, 0x99d9,
  0x9240, 0x91fc, 0x8ba9, 0x8fa2, 0x99da, 0x99d8, 0x89c2, 0x91e4,
  0x8eb6, 0x8e6a, 0x8945, 0x8a90, 0x8d86, 0x8e69, 0x99db, 0x99dc,
  0x8b68, 0x8a65, 0x8d87, 0x8b67, 0x92dd, 0x8944, 0x93af, 0x96bc,
  0x8d40, 0x9799, 0x9366, 0x8cfc, 0x8c4e, 0x99e5, 0x8be1, 0x9669,
  0x94db, 0x99e4, 0x8adc, 0x99df, 0x99e0, 0x99e2, 0x99e3, 0x8b7a,
  0x9081, 0x95ab, 0x99e1, 0x99dd, 0x8ce1, 0x99de, 0x9843, 0x95f0,
  0x92e6, 0x8ce0, 0x8d90, 0x99e6, 0x93db, 0x99ea, 0x8efc, 0x8ef4,
  0x99ed, 0x99eb, 0x96a1, 0x99e8, 0x99f1, 0x99ec, 0x99ef, 0x8cc4,
  0x96bd, 0x99f0, 0x99f2, 0x99f4, 0x8dee, 0x9861, 0x99e9, 0x99e7,
  0x99f3, 0x99ee, 0x99f6, 0x9a42, 0x99f8, 0x99fc, 0x9a40, 0x99f9,
  0x9a5d, 0x8de7, 0x8a50, 0x99f7, 0x9a44, 0x88f4, 0x9a43, 0x88a3,
  0x9569, 0x9a41, 0x99fa, 0x99f5, 0x99fb, 0x8dc6, 0x9a45, 0x88f5,
  0x9a4e, 0x9a46, 0x9a47, 0x8fa3, 0x9689, 0x9a4c, 0x9a4b, 0x934e,
  0x9a4d, 0x9a4a, 0x8953, 0x8db4, 0x904f, 0x9a48, 0x9382, 0x9a49,
  0x88a0, 0x9a53, 0x9742, 0x8fa5, 0x9a59, 0x9a58, 0x9a4f, 0x91c1,
  0x9a50, 0x91ed, 0x9a55, 0x8fa4, 0x9a52, 0x96e2, 0x8c5b, 0x9a56,
  0x9a57, 0x9a54, 0x9a5a, 0x9a51, 0x9a60, 0x9a65, 0x9a61, 0x9a5c,
  0x9a66, 0x9150, 0x9a68, 0x8d41, 0x9a5e, 0x929d, 0x9a62, 0x9a5b,
  0x8aab, 0x8aec, 0x8a85, 0x9a63, 0x9a5f, 0x8c96, 0x9a69, 0x9a67,
  0x9172, 0x8b69, 0x8baa, 0x9a64, 0x8bf2, 0x8963, 0x9a6d, 0x9a6b,
  0x9aa5, 0x9a70, 0x9a6a, 0x9a6e, 0x9a6c, 0x8e6b, 0x9a6f, 0x9a72,
  0x9a77, 0x9a75, 0x9a74, 0x9251, 0x89c3, 0x9a71, 0x9a73, 0x8fa6,
  0x8952, 0x9a76, 0x89dc, 0x9a82, 0x8ffa, 0x9a7d, 0x9a7b, 0x9a7c,
  0x9a7e, 0x895c, 0x9158, 0x9a78, 0x9a79, 0x8a9a, 0x9a81, 0x8aed,
  0x9a84, 0x9a80, 0x9a83, 0x95ac, 0x93d3, 0x94b6, 0x9a86, 0x9a85,
  0x8a64, 0x9a87, 0x9a8a, 0x9a89, 0x9a88, 0x9458, 0x9a8b, 0x9a8c,
  0x9a8e, 0x9a8d, 0x9a90, 0x9a93, 0x9a91, 0x9a8f, 0x9a92, 0x9a94,
  0x9a95, 0x9a96, 0x9a97, 0x9a98, 0x9964, 0x8efa, 0x8e6c, 0x89f1,
  0x88f6, 0x9263, 0x9a99, 0x8da2, 0x88cd, 0x907d, 0x9a9a, 0x8cc5,
  0x8d91, 0x9a9c, 0x9a9b, 0x95de, 0x9a9d, 0x9a9f, 0x9a9e, 0x9aa0,
  0x9aa1, 0x8c97, 0x8980, 0x9aa2, 0x9aa4, 0x9aa3, 0x9aa6, 0x9379,
  0x9aa7, 0x88b3, 0x8ddd, 0x8c5c, 0x926e, 0x9aa8, 0x9aa9, 0x9aab,
  0x9aac, 0x8de2, 0x8bcf, 0x9656, 0x9aaa, 0x9aad, 0x8dbf, 0x8d42,
  0x9ab1, 0x8da3, 0x9252, 0x9aae, 0x92d8, 0x9ab2, 0x9082, 0x9ab0,
  0x9ab3, 0x8c5e, 0x9ab4, 0x9ab5, 0x8d43, 0x8a5f, 0x9ab7, 0x9ab8,
  0x9ab9, 0x9ab6, 0x9aaf, 0x9aba, 0x9abb, 0x9684, 0x8fe9, 0x9abd,
  0x9abe, 0x9abc, 0x9ac0, 0x9457, 0x88e6, 0x9575, 0x9ac1, 0x8ffb,
  0x8eb7, 0x947c, 0x8aee, 0x8de9, 0x9678, 0x93b0, 0x8c98, 0x91cd,
  0x9abf, 0x9ac2, 0x91c2, 0x9ac3, 0x9ac4, 0x9ac6, 0x92e7, 0x8aac,
  0xea9f, 0x8981, 0x95f1, 0x8fea, 0x9367, 0x8de4, 0x9acc, 0x95bb,
  0x97db, 0x89f2, 0x9ac8, 0x9159, 0x9acb, 0x9383, 0x9368, 0x9384,
  0x94b7, 0x92cb, 0x8dc7, 0x9ac7, 0x8996, 0x9355, 0x9ac9, 0x9ac5,
  0x906f, 0x9acd, 0x8f6d, 0x8bab, 0x9ace, 0x95e6, 0x919d, 0x92c4,
  0x9ad0, 0x966e, 0x9ad1, 0x9ad6, 0x95ad, 0x9ad5, 0x9acf, 0x9ad2,
  0x9ad4, 0x8da4, 0x95c7, 0x9ad7, 0x9264, 0x89f3, 0x8feb, 0x9ad9,
  0x9ad8, 0x8d88, 0x9ada, 0x9adc, 0x9adb, 0x9ade, 0x9ad3, 0x9ae0,
  0x9adf, 0x9add, 0x8e6d, 0x9070, 0x9173, 0x9ae1, 0x90ba, 0x88eb,
  0x9484, 0x92d9, 0x9ae3, 0x9ae2, 0x9ae4, 0x9ae5, 0x9ae6, 0x9ae7,
  0x95cf, 0x9ae8, 0x89c4, 0x9ae9, 0x975b, 0x8a4f, 0x99c7, 0x8f67,
  0x91bd, 0x9aea, 0x96e9, 0x96b2, 0x9aec, 0x91e5, 0x9356, 0x91be,
  0x9576, 0x9aed, 0x9aee, 0x899b, 0x8eb8, 0x9aef, 0x88ce, 0x9af0,
  0x9af1, 0x8982, 0x8aef, 0x93de, 0x95f2, 0x9af5, 0x9174, 0x9af4,
  0x8c5f, 0x967a, 0x9af3, 0x9385, 0x9af7, 0x9af6, 0x9af9, 0x9af8,
  0x899c, 0x9afa, 0x8fa7, 0x9afc, 0x9244, 0x9afb, 0x95b1, 0x8f97,
  0x937a, 0x9b40, 0x8d44, 0x9b41, 0x9440, 0x94dc, 0x96cf, 0x9444,
  0x9b4a, 0x8b57, 0x9764, 0x96ad, 0x9baa, 0x9b42, 0x9b45, 0x91c3,
  0x9657, 0x9369, 0x9b46, 0x9685, 0x8dc8, 0x8fa8, 0x9b47, 0x8e6f,
  0x8e6e, 0x88b7, 0x8cc6, 0x90a9, 0x88cf, 0x9b4b, 0x9b4c, 0x9b49,
  0x8957, 0x8aad, 0x9b48, 0x96c3, 0x9550, 0x88a6, 0x88f7, 0x8e70,
  0x88d0, 0x88a1, 0x9b51, 0x9b4f, 0x96ba, 0x9b52, 0x9b50, 0x9b4e,
  0x9050, 0x9b4d, 0x95d8, 0x8ce2, 0x9b56, 0x9b57, 0x8fa9, 0x9b53,
  0x984b, 0x946b, 0x9b55, 0x8da5, 0x9b58, 0x9577, 0x9b59, 0x9b54,
  0x96b9, 0x947d, 0x9b5a, 0x9551, 0x9b5b, 0x9b5f, 0x9b5c, 0x89c5,
  0x9b5e, 0x8eb9, 0x9b5d, 0x8c99, 0x9b6b, 0x9b64, 0x9b61, 0x9284,
  0x9b60, 0x9b62, 0x9b63, 0x9b65, 0x9b66, 0x8af0, 0x9b68, 0x9b67,
  0x9b69, 0x8fec, 0x9b6c, 0x92da, 0x8964, 0x9b6a, 0x9b6d, 0x9b6e,
  0x9b71, 0x9b6f, 0x9b70, 0x8e71, 0x9b72, 0x8d45, 0x9b73, 0x8e9a,
  0x91b6, 0x9b74, 0x9b75, 0x8e79, 0x8d46, 0x96d0, 0x8b47, 0x8cc7,
  0x9b76, 0x8a77, 0x9b77, 0x91b7, 0x9b78, 0x9ba1, 0x9b79, 0x9b7a,
  0x9b7b, 0x9b7d, 0x9b7e, 0x9b80, 0x91ee, 0x8946, 0x8ee7, 0x88c0,
  0x9176, 0x8aae, 0x8eb3, 0x8d47, 0x9386, 0x8f40, 0x8aaf, 0x9288,
  0x92e8, 0x88b6, 0x8b58, 0x95f3, 0x8ec0, 0x8b71, 0x90e9, 0x8eba,
  0x9747, 0x9b81, 0x8b7b, 0x8dc9, 0x8a51, 0x8983, 0x8faa, 0x89c6,
  0x9b82, 0x9765, 0x8f68, 0x8ee2, 0x9b83, 0x8af1, 0x93d0, 0x96a7,
  0x9b84, 0x9b85, 0x9578, 0x9b87, 0x8aa6, 0x8bf5, 0x9b86, 0x8ab0,
  0x9051, 0x9b8b, 0x8e40, 0x89c7, 0x9b8a, 0x9b88, 0x9b8c, 0x9b89,
  0x944a, 0x9ecb, 0x9052, 0x9b8d, 0x97be, 0x9b8e, 0x9b90, 0x929e,
  0x9b8f, 0x90a1, 0x8e9b, 0x91ce, 0x8ef5, 0x9595, 0x90ea, 0x8ecb,
  0x9b91, 0x8fab, 0x9b92, 0x9b93, 0x88d1, 0x91b8, 0x9071, 0x9b94,
  0x93b1, 0x8fac, 0x8fad, 0x9b95, 0x90eb, 0x8fae, 0x9b96, 0x9b97,
  0x96de, 0x9b98, 0x8bc4, 0x8f41, 0x9b99, 0x9b9a, 0x8eda, 0x904b,
  0x93f2, 0x9073, 0x94f6, 0x9441, 0x8bc7, 0x9b9b, 0x8b8f, 0x9b9c,
  0x8bfc, 0x93cd, 0x89ae, 0x8e72, 0x9b9d, 0x9ba0, 0x9b9f, 0x8bfb,
  0x9b9e, 0x9357, 0x91ae, 0x936a, 0x8ec6, 0x9177, 0x979a, 0x9ba2,
  0x9ba3, 0x93d4, 0x8e52, 0x9ba5, 0x9ba6, 0x9ba7, 0x8af2, 0x9ba8,
  0x9ba9, 0x89aa, 0x915a, 0x8ae2, 0x9bab, 0x96a6, 0x91d0, 0x8a78,
  0x9bad, 0x9baf, 0x8add, 0x9bac, 0x9bae, 0x9bb1, 0x9bb0, 0x9bb2,
  0x9bb3, 0x93bb, 0x8bac, 0x89e3, 0x9bb4, 0x9bb9, 0x9bb7, 0x95f5,
  0x95f4, 0x9387, 0x9bb6, 0x8f73, 0x9bb5, 0x9092, 0x9bba, 0x8de8,
  0x9bc0, 0x9bc1, 0x9bbb, 0x8a52, 0x9bbc, 0x9bc5, 0x9bc4, 0x9bc3,
  0x9bbf, 0x9bbe, 0x9bc2, 0x95f6, 0x9bc9, 0x9bc6, 0x9bc8, 0x9792,
  0x9bc7, 0x9bbd, 0x9093, 0x9bca, 0x8db5, 0x9bcb, 0x9bcc, 0x9bcf,
  0x9bce, 0x9bcd, 0x9388, 0x9bb8, 0x9bd5, 0x9bd1, 0x9bd0, 0x9bd2,
  0x9bd3, 0x9bd6, 0x97e4, 0x9bd7, 0x9bd4, 0x9bd8, 0x8ade, 0x9bd9,
  0x9bdb, 0x9bda, 0x9bdc, 0x9bdd, 0x90ec, 0x8f42, 0x8f84, 0x9183,
  0x8d48, 0x8db6, 0x8d49, 0x8b90, 0x9bde, 0x8db7, 0x8cc8, 0x9bdf,
  0x96a4, 0x9462, 0x9be0, 0x8d4a, 0x8aaa, 0x9246, 0x8bd0, 0x8e73,
  0x957a, 0x94bf, 0x9be1, 0x8af3, 0x9be4, 0x929f, 0x9be3, 0x9be2,
  0x9be5, 0x92e9, 0x9083, 0x8e74, 0x90c8, 0x91d1, 0x8b41, 0x92a0,
  0x9be6, 0x9be7, 0x8fed, 0x9658, 0x9bea, 0x9be9, 0x9be8, 0x959d,
  0x9bf1, 0x9679, 0x9beb, 0x9bed, 0x968b, 0x9bec, 0x9bee, 0x94a6,
  0x9bef, 0x95bc, 0x9bf0, 0x8ab1, 0x95bd, 0x944e, 0x9bf2, 0x9bf3,
  0x8d4b, 0x8ab2, 0x9bf4, 0x8cb6, 0x9763, 0x9748, 0x8af4, 0x9bf6,
  0x92a1, 0x8d4c, 0x8faf, 0x94dd, 0x8fb0, 0x8f98, 0x92ea, 0x95f7,
  0x9358, 0x8d4d, 0x957b, 0x9bf7, 0x9378, 0x8dc0, 0x8cc9, 0x92eb,
  0x88c1, 0x8f8e, 0x8d4e, 0x9766, 0x9bf8, 0x9bf9, 0x9470, 0x9bfa,
  0x97f5, 0x984c, 0x9bfc, 0x9bfb, 0x8a66, 0x9c40, 0x9c43, 0x9c44,
  0x9c42, 0x955f, 0x8fb1, 0x9c46, 0x9c45, 0x9c41, 0x9c47, 0x9c48,
  0x9c49, 0x9c4c, 0x9c4a, 0x9c4b, 0x9c4d, 0x8984, 0x92ec, 0x9c4e,
  0x8c9a, 0x89f4, 0x9455, 0x9c4f, 0x93f9, 0x95d9, 0x9c50, 0x984d,
  0x9c51, 0x95be, 0x9c54, 0x989f, 0x98af, 0x8eae, 0x93f3, 0x9c55,
  0x8b7c, 0x92a2, 0x88f8, 0x9c56, 0x95a4, 0x8d4f, 0x926f, 0x92ed,
  0x96ed, 0x8cb7, 0x8cca, 0x9c57, 0x9c58, 0x9c5e, 0x8ee3, 0x92a3,
  0x8bad, 0x9c59, 0x954a, 0x9265, 0x9c5a, 0x9c5b, 0x8bae, 0x9c5c,
  0x9c5d, 0x9c5f, 0x9396, 0x9c60, 0x9c61, 0x9c62, 0x9c53, 0x9c52,
  0x9c63, 0x8c60, 0x9546, 0x8dca, 0x9556, 0x92a4, 0x956a, 0x9c64,
  0x8fb2, 0x8965, 0x9c65, 0x9c66, 0x96f0, 0x94de, 0x9c69, 0x899d,
  0x90aa, 0x9c68, 0x9c67, 0x8c61, 0x91d2, 0x9c6d, 0x9c6b, 0x9c6a,
  0x97a5, 0x8ce3, 0x8f99, 0x9c6c, 0x936b, 0x8f5d, 0x93be, 0x9c70,
  0x9c6f, 0x9c6e, 0x9c71, 0x8ce4, 0x9c72, 0x959c, 0x8f7a, 0x9c73,
  0x94f7, 0x93bf, 0x92a5, 0x934f, 0x9c74, 0x8b4a, 0x9053, 0x954b,
  0x8af5, 0x9445, 0x9c75, 0x8e75, 0x9659, 0x965a, 0x899e, 0x9c7a,
  0x9289, 0x9c77, 0x89f5, 0x9cab, 0x9c79, 0x944f, 0x9c78, 0x9c76,
  0x8d9a, 0x9c7c, 0x9c83, 0x9c89, 0x9c81, 0x937b, 0x9c86, 0x957c,
  0x9c80, 0x9c85, 0x97e5, 0x8e76, 0x91d3, 0x9c7d, 0x8b7d, 0x9c88,
  0x90ab, 0x8985, 0x9c82, 0x89f6, 0x9c87, 0x8baf, 0x9c84, 0x9c8a,
  0x9c8c, 0x9c96, 0x9c94, 0x9c91, 0x9c90, 0x97f6, 0x9c92, 0x8bb0,
  0x8d50, 0x8f9a, 0x9c99, 0x9c8b, 0x9c8f, 0x9c7e, 0x89f8, 0x9c93,
  0x9c95, 0x9270, 0x8da6, 0x89b6, 0x9c8d, 0x9c98, 0x9c97, 0x8bb1,
  0x91a7, 0x8a86, 0x8c62, 0x9c8e, 0x9c9a, 0x9c9d, 0x9c9f, 0x8ebb,
  0x9ca5, 0x92ee, 0x9c9b, 0x9ca3, 0x89f7, 0x9ca1, 0x9ca2, 0x9c9e,
  0x9ca0, 0x8ce5, 0x9749, 0x8ab3, 0x8978, 0x9ca4, 0x9459, 0x88ab,
  0x94df, 0x9c7b, 0x9caa, 0x9cae, 0x96e3, 0x9ca7, 0x9389, 0x9cac,
  0x8fee, 0x9cad, 0x93d5, 0x9866, 0x9ca9, 0x9caf, 0x8d9b, 0x90c9,
  0x88d2, 0x9ca8, 0x9ca6, 0x9179, 0x9c9c, 0x8e53, 0x91c4, 0x9cbb,
  0x917a, 0x9cb6, 0x9cb3, 0x9cb4, 0x8ee4, 0x9cb7, 0x9cba, 0x9cb5,
  0x8f44, 0x9cb8, 0x9cb2, 0x96fa, 0x96f9, 0x9cbc, 0x9cbd, 0x88d3,
  0x9cb1, 0x8bf0, 0x88a4, 0x8ab4, 0x9cb9, 0x9cc1, 0x9cc0, 0x9cc5,
  0x9cc6, 0x9cc4, 0x9cc7, 0x9cbf, 0x9cc3, 0x9cc8, 0x9cc9, 0x9cbe,
  0x8e9c, 0x9cc2, 0x91d4, 0x8d51, 0x9cb0, 0x9054, 0x9cd6, 0x95e7,
  0x9ccc, 0x9ccd, 0x9cce, 0x9cd5, 0x9cd4, 0x969d, 0x8ab5, 0x9cd2,
  0x8c64, 0x8a53, 0x9ccf, 0x97b6, 0x9cd1, 0x88d4, 0x9cd3, 0x9cca,
  0x9cd0, 0x9cd7, 0x8c63, 0x9ccb, 0x977c, 0x974a, 0x9cda, 0x9cde,
  0x919e, 0x97f7, 0x9cdf, 0x9cdc, 0x9cd9, 0x9cd8, 0x9cdd, 0x95ae,
  0x93b2, 0x8c65, 0x9ce0, 0x9cdb, 0x9ce1, 0x8c9b, 0x89af, 0x9ce9,
  0x8ab6, 0x9ce7, 0x9ce8, 0x8da7, 0x9ce6, 0x9ce4, 0x9ce3, 0x9cea,
  0x9ce2, 0x9cec, 0x89f9, 0x9cee, 0x9ced, 0x92a6, 0x9cf1, 0x9cef,
  0x9ce5, 0x8c9c, 0x9cf0, 0x9cf4, 0x9cf3, 0x9cf5, 0x9cf2, 0x9cf6,
  0x9cf7, 0x9cf8, 0x95e8, 0x9cfa, 0x9cf9, 0x8f5e, 0x90ac, 0x89e4,
  0x89fa, 0x9cfb, 0x88bd, 0x90ca, 0x9cfc, 0xe6c1, 0x9d40, 0x8c81,
  0x9d41, 0x90ed, 0x9d42, 0x9d43, 0x8b59, 0x9d44, 0x9d45, 0x9d46,
  0x91d5, 0x8ccb, 0x96df, 0x965b, 0x8f8a, 0x9d47, 0x90ee, 0xe7bb,
  0x94e0, 0x8ee8, 0x8dcb, 0x9d48, 0x91c5, 0x95a5, 0x91ef, 0x9d4b,
  0x9d49, 0x9d4c, 0x9d4a, 0x9d4d, 0x95af, 0x88b5, 0x957d, 0x94e1,
  0x9d4e, 0x9d51, 0x8fb3, 0x8b5a, 0x9d4f, 0x9d56, 0x8fb4, 0x9d50,
  0x9463, 0x977d, 0x9d52, 0x9d53, 0x9d57, 0x938a, 0x9d54, 0x8d52,
  0x90dc, 0x9d65, 0x94b2, 0x91f0, 0x94e2, 0x9dab, 0x95f8, 0x92ef,
  0x9695, 0x9d5a, 0x899f, 0x928a, 0x9d63, 0x9253, 0x9d5d, 0x9d64,
  0x9d5f, 0x9d66, 0x9d62, 0x9d61, 0x948f, 0x9d5b, 0x89fb, 0x9d59,
  0x8b91, 0x91f1, 0x9d55, 0x9d58, 0x8d53, 0x90d9, 0x8fb5, 0x9d60,
  0x9471, 0x8b92, 0x8a67, 0x8a87, 0x9040, 0x9d68, 0x9d6d, 0x9d69,
  0x8c9d, 0x9d6e, 0x8e41, 0x8d89, 0x8f45, 0x9d5c, 0x8e9d, 0x9d6b,
  0x8e77, 0x9d6c, 0x88c2, 0x9d67, 0x92a7, 0x8b93, 0x8bb2, 0x9d6a,
  0x88a5, 0x8dc1, 0x9055, 0x92f0, 0x94d2, 0x9d70, 0x917d, 0x91a8,
  0x8e4a, 0x9d71, 0x9d73, 0x9d6f, 0x95df, 0x92bb, 0x917b, 0x95f9,
  0x8ecc, 0x9d80, 0x9d7e, 0x9098, 0x8c9e, 0x9d78, 0x8fb7, 0x93e6,
  0x9450, 0x9d76, 0x917c, 0x8ef6, 0x9d7b, 0x8fb6, 0x9d75, 0x9d7a,
  0x9472, 0x9d74, 0x8c40, 0x8a7c, 0x9d7c, 0x97a9, 0x8dcc, 0x9254,
  0x9d79, 0x90da, 0x8d54, 0x9084, 0x8986, 0x915b, 0x9d77, 0x8b64,
  0x8c66, 0x92cd, 0x9d7d, 0x917e, 0x9d81, 0x9d83, 0x91b5, 0x9d89,
  0x9d84, 0x9d86, 0x9560, 0x92f1, 0x9d87, 0x974b, 0x9767, 0x8ab7,
  0x88ac, 0x9d85, 0x9d82, 0x8af6, 0x8987, 0x9d88, 0x9768, 0x9d8c,
  0x91b9, 0x9d93, 0x9d8d, 0x9d8a, 0x9d91, 0x9d72, 0x9d8e, 0x9d92,
  0x94c0, 0x938b, 0x9d8b, 0x9d8f, 0x8c67, 0x8def, 0x90db, 0x9d97,
  0x9345, 0x9d94, 0x9680, 0x9d95, 0x9d96, 0x96cc, 0x90a0, 0x8c82,
  0x9d9d, 0x8e54, 0x9d9a, 0x9d99, 0x9451, 0x93b3, 0x9350, 0x9d9b,
  0x9d9c, 0x958f, 0x9464, 0x8e42, 0x90ef, 0x966f, 0x8a68, 0x9da3,
  0x9d9e, 0x9769, 0x9da5, 0x9da1, 0x9da2, 0x9180, 0x9da0, 0x9d5e,
  0x9da4, 0x9d9f, 0x9da9, 0x9daa, 0x9346, 0x9dac, 0x8e43, 0x9da7,
  0x8b5b, 0x9dad, 0x9da6, 0x9db1, 0x9db0, 0x9daf, 0x9db2, 0x9db4,
  0x8fef, 0x9db3, 0x9db7, 0x9db5, 0x9db6, 0x9d90, 0x9db9, 0x9db8,
  0x9d98, 0x9dba, 0x9dae, 0x8e78, 0x9dbb, 0x9dbc, 0x9dbe, 0x9dbd,
  0x9dbf, 0x89fc, 0x8d55, 0x95fa, 0x90ad, 0x8ccc, 0x9dc1, 0x9dc4,
  0x9571, 0x8b7e, 0x9dc3, 0x9dc2, 0x9473, 0x9dc5, 0x8bb3, 0x9dc7,
  0x9dc6, 0x8ab8, 0x8e55, 0x93d6, 0x8c68, 0x9094, 0x9dc8, 0x90ae,
  0x9347, 0x957e, 0x9dc9, 0x9dca, 0x9dcb, 0x95b6, 0x9b7c, 0x90c4,
  0x956b, 0x8dd6, 0x94e3, 0x94c1, 0x936c, 0x97bf, 0x9dcd, 0x8ece,
  0x9dce, 0x88b4, 0x8bd2, 0x90cb, 0x9580, 0x9dcf, 0x8e61, 0x9266,
  0x8e7a, 0x9056, 0x9dd0, 0x95fb, 0x8997, 0x8e7b, 0x9dd3, 0x9dd1,
  0x9dd4, 0x97b7, 0x9dd2, 0x90f9, 0x9dd5, 0x91b0, 0x9dd6, 0x8af8,
  0x9dd8, 0x9dd7, 0x9dd9, 0x9dda, 0x8af9, 0x93fa, 0x9255, 0x8b8c,
  0x8e7c, 0x9181, 0x8f7b, 0x88ae, 0x9ddb, 0x89a0, 0x9ddf, 0x8d56,
  0x9dde, 0x8da9, 0x8fb8, 0x9ddd, 0x8fb9, 0x96be, 0x8da8, 0x88d5,
  0x90cc, 0x9de4, 0x90af, 0x8966, 0x8f74, 0x9686, 0x8df0, 0x8fba,
  0x90a5, 0x9de3, 0x9de1, 0x9de2, 0x928b, 0x9e45, 0x9de8, 0x8e9e,
  0x8d57, 0x9de6, 0x9de7, 0x9057, 0x9de5, 0x8e4e, 0x9dea, 0x9de9,
  0x9dee, 0x9def, 0x9deb, 0x8a41, 0x9dec, 0x9ded, 0x94d3, 0x9581,
  0x8c69, 0x9df0, 0x90b0, 0x8fbb, 0x9271, 0x8bc5, 0x9df1, 0x9df5,
  0x89c9, 0x9df2, 0x9df4, 0x9df3, 0x8f8b, 0x9267, 0x88c3, 0x9df6,
  0x9df7, 0x92a8, 0x97ef, 0x8e62, 0x95e9, 0x965c, 0x9e41, 0x9df9,
  0x9dfc, 0x9dfb, 0x9df8, 0x9e40, 0x93dc, 0x9dfa, 0x9e42, 0x8f8c,
  0x9e43, 0x976a, 0x9498, 0x9e44, 0x9e46, 0x9e47, 0x9e48, 0x8bc8,
  0x8967, 0x8d58, 0x9e49, 0x9e4a, 0x8f91, 0x9182, 0x99d6, 0x915d,
  0x915c, 0x91d6, 0x8dc5, 0x98f0, 0x8c8e, 0x974c, 0x95fc, 0x959e,
  0x9e4b, 0x8df1, 0x92bd, 0x9e4c, 0x984e, 0x965d, 0x92a9, 0x9e4d,
  0x8afa, 0x9e4e, 0x9e4f, 0x96d8, 0x96a2, 0x9696, 0x967b, 0x8e44,
  0x9e51, 0x8ee9, 0x9670, 0x9e53, 0x9e56, 0x9e55, 0x8af7, 0x8b80,
  0x9e52, 0x9e54, 0x9e57, 0x9099, 0x979b, 0x88c7, 0x8dde, 0x91ba,
  0x8edb, 0x8ff1, 0x9e5a, 0x936d, 0x9e58, 0x91a9, 0x9e59, 0x8ff0,
  0x96db, 0x9e5b, 0x9e5c, 0x9788, 0x9e61, 0x8d59, 0x9474, 0x9e5e,
  0x938c, 0x9ddc, 0x9de0, 0x8b6e, 0x9466, 0x9e60, 0x8fbc, 0x94c2,
  0x9e66, 0x94f8, 0x9e5d, 0x9e63, 0x9e62, 0x90cd, 0x968d, 0x97d1,
  0x9687, 0x89ca, 0x8e7d, 0x9867, 0x9e65, 0x9095, 0x9e64, 0x9e5f,
  0x8ccd, 0x9e6b, 0x9e69, 0x89cb, 0x9e67, 0x9e6d, 0x9e73, 0x91c6,
  0x95bf, 0x9e75, 0x9541, 0x9e74, 0x9490, 0x965e, 0x8ab9, 0x90f5,
  0x8f5f, 0x92d1, 0x974d, 0x9e70, 0x9e6f, 0x9e71, 0x9e6e, 0x9e76,
  0x9e6c, 0x9e6a, 0x9e72, 0x9e68, 0x928c, 0x96f6, 0x8ec4, 0x8df2,
  0x8db8, 0x968f, 0x8a60, 0x92cc, 0x93c8, 0x8968, 0x90f0, 0x90b2,
  0x8c49, 0x9e78, 0x8d5a, 0x8a9c, 0x9e7a, 0x8a94, 0x9e81, 0x9e7d,
  0x90f1, 0x8a6a, 0x8daa, 0x8a69, 0x8dcd, 0x9e7b, 0x8c85, 0x8c6a,
  0x938d, 0x9e79, 0x88c4, 0x9e7c, 0x9e7e, 0x8bcb, 0x8c4b, 0x8aba,
  0x8b6a, 0x9e82, 0x8df7, 0x9691, 0x8e56, 0x9e83, 0x954f, 0x9e8f,
  0x89b1, 0x9e84, 0x9e95, 0x9e85, 0x97c0, 0x9e8c, 0x947e, 0x9e94,
  0x9e87, 0x88b2, 0x9e89, 0x8d5b, 0x9e8b, 0x9e8a, 0x9e86, 0x9e91,
  0x8fbd, 0x9aeb, 0x8ce6, 0x979c, 0x9e88, 0x92f2, 0x8a42, 0x8dab,
  0x9e80, 0x9e90, 0x8a81, 0x9e8e, 0x9e92, 0x938e, 0x8afc, 0x9eb0,
  0x96c7, 0x9e97, 0x8afb, 0x9e9e, 0x965f, 0x9e9f, 0x9ea1, 0x9ea5,
  0x9e99, 0x9249, 0x938f, 0x9ea9, 0x9e9c, 0x9ea6, 0x9ea0, 0x9058,
  0x9eaa, 0x90b1, 0x9ea8, 0x8abb, 0x986f, 0x9e96, 0x9ea4, 0x88d6,
  0x9e98, 0x96b8, 0x9e9d, 0x9041, 0x92c5, 0x9e93, 0x9ea3, 0x909a,
  0x9ead, 0x8a91, 0x8c9f, 0x9eaf, 0x9e9a, 0x9eae, 0x9ea7, 0x9e9b,
  0x9eab, 0x9eac, 0x9ebd, 0x93cc, 0x9ea2, 0x9eb9, 0x9ebb, 0x92d6,
  0x976b, 0x9596, 0x9eb6, 0x91c8, 0x9ebc, 0x915e, 0x9eb3, 0x9ec0,
  0x9ebf, 0x93ed, 0x9ebe, 0x93e8, 0x9ec2, 0x9eb5, 0x8bc6, 0x9eb8,
  0x8f7c, 0x9480, 0x9eba, 0x8bc9, 0x9eb2, 0x9eb4, 0x9eb1, 0x984f,
  0x8a79, 0x9eb7, 0x9ec1, 0x8a54, 0x8de5, 0x897c, 0x9ed2, 0x9850,
  0x9ed5, 0x9059, 0x9ed4, 0x9ed3, 0x9ed0, 0x9ec4, 0x9ee1, 0x9ec3,
  0x9ed6, 0x9ece, 0x9ec9, 0x9ec6, 0x9ec7, 0x9ecf, 0xeaa0, 0x9ecc,
  0x8d5c, 0x92c6, 0x9184, 0x9eca, 0x9ec5, 0x9ec8, 0x976c, 0x968a,
  0x9ecd, 0x9ed7, 0x9edf, 0x9ed8, 0x9ee5, 0x9ee3, 0x9ede, 0x9edd,
  0x92ce, 0x9185, 0x9edb, 0x9ed9, 0x9ee0, 0x9ee6, 0x94f3, 0x9eec,
  0x9ee7, 0x9eea, 0x9ee4, 0x9294, 0x9557, 0x9eda, 0x9ee2, 0x8fbe,
  0x96cd, 0x9ef6, 0x9ee9, 0x8ca0, 0x89a1, 0x8a7e, 0x9ed1, 0x8fbf,
  0x9eee, 0x9ef5, 0x8ef7, 0x8a92, 0x924d, 0x9eeb, 0x9ef0, 0x9ef4,
  0x8bb4, 0x8b6b, 0x9ef2, 0x8b40, 0x93c9, 0x9ef1, 0x9ef3, 0x9eed,
  0x9eef, 0x8a80, 0x9268, 0x9efa, 0x9ef8, 0x8ce7, 0x9ef7, 0x9f40,
  0x9e77, 0x9ef9, 0x9efb, 0x9efc, 0x9f4b, 0x9f47, 0x9e8d, 0x9f46,
  0x9f45, 0x9f42, 0x9ee8, 0x9f44, 0x9f43, 0x9f49, 0x9845, 0x9f4c,
  0x8bf9, 0x9f48, 0x9f4a, 0x94a5, 0x9f4d, 0x9f51, 0x9f4e, 0x9793,
  0x9f4f, 0x9edc, 0x9f52, 0x9f53, 0x8954, 0x9f55, 0x8c87, 0x8e9f,
  0x8bd3, 0x89a2, 0x977e, 0x9f57, 0x9f56, 0x9f59, 0x8b5c, 0x8bd4,
  0x8abc, 0x9f5c, 0x9f5b, 0x9f5d, 0x89cc, 0x9256, 0x9f5e, 0x8abd,
  0x9f60, 0x9f5f, 0x9f61, 0x9f62, 0x9f63, 0x8e7e, 0x90b3, 0x8d9f,
  0x9590, 0x95e0, 0x9863, 0x8e95, 0x8dce, 0x97f0, 0x9f64, 0x9f65,
  0x8e80, 0x9f66, 0x9f67, 0x9f69, 0x9f68, 0x9677, 0x8f7d, 0x8eea,
  0x8e63, 0x9f6a, 0x9f6c, 0x9042, 0x9f6b, 0x9f6d, 0x9f6e, 0x9f6f,
  0x9f70, 0x9f71, 0x9f73, 0x9f72, 0x9f74, 0x89a3, 0x9269, 0x9f75,
  0x8e45, 0x8a6b, 0x9f76, 0x9361, 0x9aca, 0x8b42, 0x9f77, 0x9f78,
  0x95ea, 0x9688, 0x93c5, 0x9f79, 0x94e4, 0x94f9, 0x96d1, 0x9f7a,
  0x9f7c, 0x9f7b, 0x9f7e, 0x9f7d, 0x9f81, 0x8e81, 0x96af, 0x9f82,
  0x9f83, 0x8b43, 0x9f84, 0x9f86, 0x9f85, 0x9085, 0x9558, 0x8969,
  0x94c3, 0x92f3, 0x8f60, 0x8b81, 0x94c4, 0x8eac, 0x9f88, 0x8abe,
  0x8998, 0x93f0, 0x9f87, 0x8d5d, 0x9272, 0x9f89, 0x9f91, 0x9f8a,
  0x91bf, 0x8b82, 0x9f92, 0x8c88, 0x8b44, 0x9f90, 0x9f8e, 0x9f8b,
  0x9780, 0x92be, 0x93d7, 0x9f8c, 0x9f94, 0x9f93, 0x8c42, 0x89ab,
  0x8db9, 0x9f8d, 0x9f8f, 0x9676, 0x91f2, 0x9697, 0x9f9c, 0x9f9d,
  0x89cd, 0x95a6, 0x96fb, 0x9f9f, 0x8ea1, 0x8fc0, 0x9f98, 0x9f9e,
  0x8988, 0x8bb5, 0x9f95, 0x9f9a, 0x90f2, 0x9491, 0x94e5, 0x9f97,
  0x9640, 0x9f99, 0x9fa2, 0x9fa0, 0x9f9b, 0x9641, 0x9467, 0x8b83,
  0x9344, 0x928d, 0x9fa3, 0x9fa1, 0x91d7, 0x9f96, 0x896a, 0x976d,
  0x9fae, 0x9fad, 0x90f4, 0x9faa, 0x978c, 0x93b4, 0x9fa4, 0x92c3,
  0x896b, 0x8d5e, 0x9fa7, 0x8f46, 0x9fac, 0x9fab, 0x9fa6, 0x9fa9,
  0x8a88, 0x9fa8, 0x9468, 0x97ac, 0x8ff2, 0x90f3, 0x9fb4, 0x9fb2,
  0x956c, 0x9faf, 0x9fb1, 0x8959, 0x8d5f, 0x9851, 0x8a5c, 0x9582,
  0x9781, 0x8a43, 0x905a, 0x9fb3, 0x9fb8, 0x8fc1, 0x974f, 0x9fb5,
  0x9fb0, 0x9fb6, 0x97dc, 0x9393, 0x93c0, 0x8a55, 0x8974, 0x9fbc,
  0x9fbf, 0x97c1, 0x9784, 0x9fc6, 0x9fc0, 0x9fbd, 0x97d2, 0x9fc3,
  0x8f69, 0x9fc5, 0x9fca, 0x9391, 0x9fc8, 0x9fc2, 0x9257, 0x9fc9,
  0x9fbe, 0x9fc4, 0x9fcb, 0x88fa, 0x9fc1, 0x9fcc, 0x905b, 0x8f7e,
  0x95a3, 0x8dac, 0x9fb9, 0x9fc7, 0x9359, 0x90b4, 0x8a89, 0x8dcf,
  0x8fc2, 0x9fbb, 0x8f61, 0x8c6b, 0x9fba, 0x9fd0, 0x8f8d, 0x8cb8,
  0x9fdf, 0x9fd9, 0x8b94, 0x936e, 0x9fd4, 0x9fdd, 0x88ad, 0x8951,
  0x89b7, 0x9fd6, 0x91aa, 0x9fcd, 0x9fcf, 0x8d60, 0x9fe0, 0x9fdb,
  0x9fd3, 0x9fda, 0x96a9, 0x9fd8, 0x9fdc, 0x8cce, 0x8fc3, 0x9258,
  0x9fd2, 0x974e, 0x9fd5, 0x9fce, 0x9392, 0x9fd1, 0x9fd7, 0x9870,
  0x8ebc, 0x969e, 0x9fe1, 0x94ac, 0x9fed, 0x8cb9, 0x8f80, 0x9fe3,
  0x97ad, 0x8d61, 0x9ff0, 0x88ec, 0x9fee, 0x9fe2, 0x9fe8, 0x9fea,
  0x976e, 0x9fe5, 0x934d, 0x9fe7, 0x9fef, 0x9fe9, 0x96c5, 0x9fe4,
  0x8ea0, 0x9ffc, 0x8a8a, 0x9fe6, 0x9feb, 0x9fec, 0x91ea, 0x91d8,
  0x9ff4, 0x9ffa, 0x9ff8, 0x9348, 0xe042, 0x9ff5, 0x9ff6, 0x9fde,
  0x8b99, 0x9559, 0x8ebd, 0x8d97, 0x9852, 0x9ff2, 0xe041, 0x8989,
  0x9186, 0x9499, 0x8abf, 0x97f8, 0x969f, 0x92d0, 0x9ff9, 0x9ffb,
  0x9151, 0xe040, 0x9ff7, 0x9ff1, 0x8ac1, 0x8c89, 0xe04e, 0xe049,
  0x90f6, 0x8a83, 0x8f81, 0xe052, 0xe04b, 0x92aa, 0xe048, 0x92d7,
  0xe06b, 0xe045, 0xe044, 0xe04d, 0xe047, 0xe046, 0xe04c, 0x909f,
  0xe043, 0xe04f, 0xe050, 0x8ac0, 0xe055, 0xe054, 0xe056, 0xe059,
  0x9362, 0xe053, 0xe057, 0x8c83, 0x91f7, 0xe051, 0x945a, 0xe058,
  0xe05d, 0xe05b, 0xe05e, 0xe061, 0xe05a, 0x8d8a, 0x9447, 0x9fb7,
  0x9794, 0xe05c, 0xe060, 0x91f3, 0xe05f, 0xe04a, 0xe889, 0xe064,
  0xe068, 0xe066, 0xe062, 0xe063, 0xe067, 0xe065, 0x956d, 0xe06d,
  0xe06a, 0xe069, 0xe06c, 0x93d2, 0xe06e, 0x9295, 0x91eb, 0x90a3,
  0xe06f, 0xe071, 0xe070, 0x9ff3, 0xe072, 0x93e5, 0xe073, 0x89ce,
  0x9394, 0x8a44, 0x8b84, 0x8edc, 0x8dd0, 0x9846, 0x9086, 0x898a,
  0xe075, 0xe074, 0xe078, 0x9259, 0xe07b, 0xe076, 0xe07a, 0xe079,
  0x935f, 0x88d7, 0x97f3, 0xe07d, 0x8947, 0xe080, 0xe07e, 0xe07c,
  0xe077, 0x9642, 0xe082, 0xe081, 0x898b, 0xe084, 0x95b0, 0xe083,
  0x96b3, 0x8fc5, 0x9152, 0x8fc4, 0x97f9, 0xe08a, 0x90f7, 0xe086,
  0xe08b, 0x898c, 0xe089, 0x9481, 0xe085, 0xe088, 0x8fc6, 0x94cf,
  0xe08c, 0x8ecf, 0x90f8, 0xe08f, 0xe087, 0x8c46, 0xe08d, 0x976f,
  0xe090, 0xeaa4, 0x8f6e, 0xe091, 0xe092, 0x944d, 0xe094, 0xe095,
  0x9452, 0x9395, 0xe097, 0xe099, 0x97d3, 0xe096, 0xe098, 0x898d,
  0xe093, 0x9a7a, 0xe09a, 0x9187, 0x8e57, 0xe09c, 0xe09b, 0x9043,
  0x99d7, 0xe09d, 0xe09f, 0xe08e, 0xe09e, 0xe0a0, 0x949a, 0xe0a1,
  0xe0a2, 0xe0a3, 0xe0a4, 0x92dc, 0xe0a6, 0xe0a5, 0xe0a7, 0xe0a8,
  0x8edd, 0x9583, 0x96ea, 0xe0a9, 0xe0aa, 0x9175, 0x8ea2, 0xe0ab,
  0xe0ac, 0xe0ad, 0x95d0, 0x94c5, 0xe0ae, 0x9476, 0x92ab, 0xe0af,
  0x89e5, 0x8b8d, 0x96c4, 0x96b4, 0x89b2, 0x9853, 0x9671, 0x95a8,
  0x90b5, 0xe0b0, 0x93c1, 0x8ca1, 0xe0b1, 0x8dd2, 0xe0b3, 0xe0b2,
  0xe0b4, 0xe0b5, 0xe0b6, 0x8b5d, 0xe0b7, 0xe0b8, 0x8ca2, 0x94c6,
  0xe0ba, 0x8ff3, 0xe0b9, 0x8bb6, 0xe0bb, 0xe0bd, 0xe0bc, 0xe0be,
  0x8ccf, 0xe0bf, 0x8be7, 0x915f, 0x8d9d, 0xe0c1, 0xe0c2, 0xe0c0,
  0x8eeb, 0x93c6, 0x8bb7, 0xe0c4, 0x924b, 0xe0c3, 0x9854, 0x9482,
  0xe0c7, 0xe0c9, 0xe0c6, 0x96d2, 0xe0c8, 0xe0ca, 0x97c2, 0xe0ce,
  0xe0cd, 0x9296, 0x944c, 0x8ca3, 0xe0cc, 0xe0cb, 0x9750, 0x9751,
  0xe0cf, 0x898e, 0x8d96, 0x8e82, 0xe0d0, 0xe0d1, 0xe0d3, 0x8f62,
  0xe0d5, 0xe0d4, 0xe0d6, 0x8a6c, 0xe0d8, 0xe0d7, 0xe0da, 0xe0d9,
  0x8cba, 0x97a6, 0x8bca, 0x89a4, 0x8be8, 0x8adf, 0x97e6, 0xe0dc,
  0xe0de, 0xe0df, 0x89cf, 0xe0db, 0x8e58, 0x92bf, 0xe0dd, 0xe0e2,
  0x8eec, 0xe0e0, 0x8c5d, 0x94c7, 0xe0e1, 0xe0fc, 0xe0e7, 0x8cbb,
  0x8b85, 0xe0e4, 0x979d, 0x97ae, 0x91f4, 0xe0e6, 0xe0e8, 0x97d4,
  0x8bd5, 0x94fa, 0x9469, 0xe0e9, 0xe0eb, 0xe0ee, 0xe0ea, 0xe0ed,
  0x8ce8, 0x896c, 0xe0ef, 0x9090, 0xe0ec, 0x97da, 0xe0f2, 0xeaa2,
  0xe0f0, 0xe0f3, 0xe0e5, 0xe0f1, 0x8dba, 0xe0f4, 0xe0f5, 0x979e,
  0xe0f6, 0xe0f7, 0xe0e3, 0xe0f8, 0x8ac2, 0x8ea3, 0xe0f9, 0xe0fa,
  0xe0fb, 0x895a, 0xe140, 0x955a, 0xe141, 0x8aa2, 0xe142, 0xe143,
  0xe144, 0xe146, 0xe147, 0xe145, 0x9572, 0xe149, 0xe148, 0xe14b,
  0xe14a, 0xe14c, 0xe14d, 0xe14f, 0xe14e, 0x8d99, 0xe151, 0xe150,
  0x8ac3, 0x9072, 0x935b, 0xe152, 0x90b6, 0x8e59, 0x8999, 0xe153,
  0x9770, 0x95e1, 0xe154, 0x9363, 0x9752, 0x8d62, 0x905c, 0x926a,
  0x99b2, 0x92ac, 0x89e6, 0xe155, 0xe156, 0xe15b, 0xe159, 0xe158,
  0x9dc0, 0x8a45, 0xe157, 0x88d8, 0x94a8, 0x94c8, 0x97af, 0xe15c,
  0xe15a, 0x927b, 0x90a4, 0x94a9, 0x954c, 0xe15e, 0x97aa, 0x8c6c,
  0xe15f, 0xe15d, 0x94d4, 0xe160, 0xe161, 0x88d9, 0x8ff4, 0xe166,
  0xe163, 0x93eb, 0xe162, 0x8b45, 0xe169, 0xe164, 0xe165, 0xe168,
  0xe167, 0x9544, 0x9161, 0x9160, 0x8b5e, 0xe16a, 0xe16b, 0xe16c,
  0xe16e, 0xe16d, 0x8975, 0xe176, 0x94e6, 0xe170, 0xe172, 0xe174,
  0x905d, 0xe175, 0xe173, 0x8ebe, 0xe16f, 0xe171, 0x9561, 0x8fc7,
  0xe178, 0xe177, 0xe179, 0x8ea4, 0x8dad, 0x9397, 0xe17a, 0x92c9,
  0xe17c, 0x979f, 0xe17b, 0x9189, 0xe182, 0xe184, 0xe185, 0x9273,
  0xe183, 0xe180, 0xe17d, 0xe17e, 0xe181, 0xe188, 0xe186, 0xe187,
  0xe189, 0xe18b, 0xe18c, 0xe18d, 0xe18e, 0xe18a, 0xe190, 0xe18f,
  0xe191, 0x97c3, 0xe194, 0xe192, 0xe193, 0x8ae0, 0x96fc, 0x95c8,
  0xe196, 0xe195, 0xe197, 0xe198, 0xe19c, 0xe199, 0xe19a, 0xe19b,
  0xe19d, 0xe19e, 0xe19f, 0xe1a0, 0xe1a1, 0x94ad, 0x936f, 0xe1a2,
  0x9492, 0x9553, 0xe1a3, 0xe1a4, 0x9349, 0x8a46, 0x8d63, 0xe1a5,
  0xe1a6, 0xe1a7, 0x8e48, 0xe1a9, 0xe1a8, 0xe1aa, 0xe1ab, 0x94e7,
  0xe1ac, 0xe1ad, 0xea89, 0xe1ae, 0xe1af, 0xe1b0, 0x8e4d, 0xe1b1,
  0x9475, 0x967e, 0x896d, 0x8976, 0xe1b2, 0xe1b4, 0xe1b3, 0x9390,
  0x90b7, 0x9f58, 0xe1b5, 0x96bf, 0xe1b6, 0x8ac4, 0x94d5, 0xe1b7,
  0xe1b8, 0xe1b9, 0x96da, 0x96d3, 0x92bc, 0x918a, 0xe1bb, 0x8f82,
  0x8fc8, 0xe1be, 0xe1bd, 0xe1bc, 0x94fb, 0x8ac5, 0x8ca7, 0xe1c4,
  0xe1c1, 0x905e, 0x96b0, 0xe1c0, 0xe1c2, 0xe1c3, 0xe1bf, 0xe1c5,
  0xe1c6, 0x92ad, 0x8ae1, 0x9285, 0xe1c7, 0xe1c8, 0xe1cb, 0x9087,
  0x93c2, 0xe1cc, 0x9672, 0xe1c9, 0xe1ca, 0xe1cf, 0xe1ce, 0xe1cd,
  0xe1d1, 0xe1d0, 0xe1d2, 0xe1d4, 0xe1d3, 0x95cb, 0x8f75, 0x97c4,
  0xe1d5, 0x93b5, 0xe1d6, 0xe1d7, 0xe1db, 0xe1d9, 0xe1da, 0xe1d8,
  0xe1dc, 0xe1dd, 0xe1de, 0xe1df, 0x96b5, 0xe1e0, 0x96ee, 0xe1e1,
  0x926d, 0x948a, 0x8be9, 0x925a, 0xe1e2, 0x8bb8, 0x90ce, 0xe1e3,
  0x8dbb, 0xe1e4, 0xe1e5, 0x8ca4, 0x8dd3, 0xe1e7, 0x9375, 0x8dd4,
  0x8b6d, 0x9643, 0x946a, 0x9376, 0x8d7b, 0xe1e9, 0x8fc9, 0x97b0,
  0x8d64, 0x8ca5, 0x94a1, 0xe1eb, 0xe1ed, 0x8ce9, 0xe1ec, 0x92f4,
  0xe1ef, 0x8a56, 0xe1ea, 0x94e8, 0x894f, 0x8dea, 0x9871, 0xe1ee,
  0xe1f0, 0x95c9, 0x90d7, 0xe1f2, 0xe1f3, 0xe1f1, 0x8a6d, 0xe1f9,
  0xe1f8, 0x8ea5, 0xe1fa, 0xe1f5, 0xe1fb, 0xe1f6, 0x94d6, 0xe1f4,
  0xe1f7, 0xe241, 0xe240, 0x9681, 0xe1fc, 0x88e9, 0xe243, 0xe242,
  0x8fca, 0xe244, 0x9162, 0xe246, 0xe245, 0xe247, 0xe1e6, 0xe1e8,
  0xe249, 0xe248, 0x8ea6, 0x97e7, 0x8ed0, 0xe24a, 0x8c56, 0x8b5f,
  0x8b46, 0x8e83, 0x9753, 0xe250, 0xe24f, 0x9163, 0xe24c, 0xe24e,
  0x8f6a, 0x905f, 0xe24d, 0xe24b, 0x9449, 0x8fcb, 0x955b, 0x8dd5,
  0x9398, 0xe251, 0xe252, 0xe268, 0x8bd6, 0x985c, 0x9154, 0xe253,
  0x89d0, 0x92f5, 0x959f, 0xe254, 0x8b9a, 0xe255, 0xe257, 0xe258,
  0x9448, 0xe259, 0xe25a, 0xe25b, 0x8bd7, 0x89d1, 0x93c3, 0x8f47,
  0x8e84, 0xe25c, 0x8f48, 0x89c8, 0x9562, 0xe25d, 0x94e9, 0x9164,
  0xe260, 0xe261, 0x9489, 0x9060, 0xe25e, 0x9281, 0xe25f, 0x8fcc,
  0x88da, 0x8b48, 0xe262, 0x92f6, 0xe263, 0x90c5, 0x96ab, 0x9542,
  0xe264, 0xe265, 0x9274, 0x97c5, 0xe267, 0xe266, 0x8eed, 0xe269,
  0x88ee, 0xe26c, 0xe26a, 0x89d2, 0x8c6d, 0xe26b, 0x8d65, 0x8d92,
  0x95e4, 0xe26d, 0x9673, 0xe26f, 0x90cf, 0x896e, 0x89b8, 0x88aa,
  0xe26e, 0xe270, 0xe271, 0x8ff5, 0xe272, 0x8a6e, 0xe274, 0x8c8a,
  0x8b86, 0xe275, 0x8bf3, 0xe276, 0x90fa, 0x93cb, 0x90de, 0x8df3,
  0xe277, 0x9282, 0x918b, 0xe279, 0xe27b, 0xe278, 0xe27a, 0x8c41,
  0xe27c, 0x8c45, 0x8b87, 0x9771, 0xe27e, 0xe280, 0x894d, 0xe283,
  0x8a96, 0xe282, 0xe281, 0xe285, 0xe27d, 0xe286, 0x97a7, 0xe287,
  0xe288, 0x9af2, 0xe28a, 0xe289, 0xe28b, 0xe28c, 0x97b3, 0xe28d,
  0xe8ed, 0x8fcd, 0xe28e, 0xe28f, 0x8f76, 0x93b6, 0xe290, 0x9247,
  0xe291, 0x925b, 0xe292, 0x8ba3, 0x995e, 0x927c, 0x8eb1, 0x8ac6,
  0xe293, 0xe2a0, 0xe296, 0x8b88, 0xe295, 0xe2a2, 0xe294, 0x8fce,
  0xe298, 0xe299, 0x934a, 0xe29a, 0x8a7d, 0x9079, 0x9584, 0xe29c,
  0x91e6, 0xe297, 0xe29b, 0xe29d, 0x8df9, 0xe2a4, 0x954d, 0x94a4,
  0x9399, 0x8bd8, 0xe2a3, 0xe2a1, 0x94b3, 0xe29e, 0x927d, 0x939b,
  0x939a, 0x8df4, 0xe2b6, 0xe2a6, 0xe2a8, 0xe2ab, 0xe2ac, 0xe2a9,
  0xe2aa, 0xe2a7, 0xe2a5, 0xe29f, 0x95cd, 0x89d3, 0xe2b3, 0xe2b0,
  0xe2b5, 0xe2b4, 0x9493, 0x96a5, 0x8e5a, 0xe2ae, 0xe2b7, 0xe2b2,
  0xe2b1, 0xe2ad, 0xe2af, 0x8ac7, 0x925c, 0x90fb, 0x94a0, 0xe2bc,
  0x94a2, 0x90df, 0xe2b9, 0x94cd, 0xe2bd, 0x95d1, 0x927a, 0xe2b8,
  0xe2ba, 0xe2bb, 0xe2be, 0x8ec2, 0x93c4, 0xe2c3, 0xe2c2, 0xe2bf,
  0x9855, 0xe2c8, 0xe2cc, 0xe2c9, 0xe2c5, 0xe2c6, 0xe2cb, 0xe2c0,
  0x99d3, 0xe2c7, 0xe2c1, 0xe2ca, 0xe2d0, 0x8ac8, 0xe2cd, 0xe2ce,
  0xe2cf, 0xe2d2, 0xe2d1, 0x94f4, 0xe2d3, 0x97fa, 0x95eb, 0xe2d8,
  0xe2d5, 0xe2d4, 0x90d0, 0xe2d7, 0xe2d9, 0xe2d6, 0xe2dd, 0xe2da,
  0xe2db, 0xe2c4, 0xe2dc, 0xe2de, 0xe2df, 0x95c4, 0xe2e0, 0x96e0,
  0x8bcc, 0x8c48, 0xe2e1, 0x95b2, 0x9088, 0x96ae, 0xe2e2, 0x97b1,
  0x9494, 0x9165, 0x9453, 0x8f6c, 0x88be, 0xe2e7, 0xe2e5, 0xe2e3,
  0x8a9f, 0x8fcf, 0xe2e8, 0xe2e6, 0xe2e4, 0xe2ec, 0xe2eb, 0xe2ea,
  0xe2e9, 0xe2ed, 0xe2ee, 0x90b8, 0xe2ef, 0xe2f1, 0xe2f0, 0x8cd0,
  0x9157, 0xe2f3, 0x939c, 0xe2f2, 0xe2f4, 0x95b3, 0x918c, 0x8d66,
  0xe2f5, 0x97c6, 0xe2f7, 0xe2f8, 0xe2f9, 0xe2fa, 0x8e85, 0xe2fb,
  0x8c6e, 0x8b8a, 0x8b49, 0xe340, 0x96f1, 0x8d67, 0xe2fc, 0xe343,
  0x96e4, 0x945b, 0x9552, 0x8f83, 0xe342, 0x8ed1, 0x8d68, 0x8e86,
  0x8b89, 0x95b4, 0xe341, 0x9166, 0x9661, 0x8df5, 0x8e87, 0x92db,
  0xe346, 0x97dd, 0x8dd7, 0xe347, 0x9061, 0xe349, 0x8fd0, 0x8dae,
  0xe348, 0x8f49, 0x8cbc, 0x9167, 0xe344, 0xe34a, 0xe345, 0x8c6f,
  0xe34d, 0xe351, 0x8c8b, 0xe34c, 0xe355, 0x8d69, 0x978d, 0x88ba,
  0xe352, 0x8b8b, 0xe34f, 0xe350, 0x939d, 0xe34e, 0xe34b, 0x8a47,
  0x90e2, 0x8ca6, 0xe357, 0xe354, 0xe356, 0xe353, 0x8c70, 0x91b1,
  0xe358, 0x918e, 0xe365, 0xe361, 0xe35b, 0xe35f, 0x8ef8, 0x88db,
  0xe35a, 0xe362, 0xe366, 0x8d6a, 0x96d4, 0x92d4, 0xe35c, 0xe364,
  0xe359, 0x925d, 0xe35e, 0x88bb, 0x96c8, 0xe35d, 0x8bd9, 0x94ea,
  0x918d, 0x97ce, 0x8f8f, 0xe38e, 0xe367, 0x90fc, 0xe363, 0xe368,
  0xe36a, 0x92f7, 0xe36d, 0xe369, 0x95d2, 0x8ac9, 0x96c9, 0x88dc,
  0xe36c, 0x97fb, 0xe36b, 0x898f, 0x93ea, 0xe36e, 0xe375, 0xe36f,
  0xe376, 0xe372, 0x949b, 0x8ec8, 0xe374, 0xe371, 0xe377, 0xe370,
  0x8f63, 0x9644, 0x8f6b, 0xe373, 0xe380, 0xe37b, 0xe37e, 0xe37c,
  0xe381, 0xe37a, 0xe360, 0x90d1, 0x94c9, 0xe37d, 0xe378, 0x9140,
  0x8c71, 0x8f4a, 0x9044, 0x9155, 0xe384, 0xe386, 0xe387, 0xe383,
  0xe385, 0xe379, 0xe382, 0xe38a, 0xe389, 0x969a, 0x8c4a, 0xe388,
  0xe38c, 0xe38b, 0xe38f, 0xe391, 0x8e5b, 0xe38d, 0xe392, 0xe393,
  0xe394, 0xe39a, 0x935a, 0xe396, 0xe395, 0xe397, 0xe398, 0xe399,
  0xe39b, 0xe39c, 0x8aca, 0xe39d, 0xe39e, 0xe39f, 0xe3a0, 0xe3a1,
  0xe3a2, 0xe3a3, 0xe3a4, 0xe3a6, 0xe3a5, 0xe3a7, 0xe3a8, 0xe3a9,
  0xe3ac, 0xe3aa, 0xe3ab, 0x8ddf, 0x8c72, 0x9275, 0x94b1, 0x8f90,
  0x946c, 0x94eb, 0xe3ad, 0x9ceb, 0xe3ae, 0xe3b0, 0x9785, 0xe3af,
  0xe3b2, 0xe3b1, 0x9772, 0xe3b3, 0x94fc, 0xe3b4, 0xe3b7, 0xe3b6,
  0xe3b5, 0xe3b8, 0x8c51, 0x9141, 0x8b60, 0xe3bc, 0xe3b9, 0xe3ba,
  0xe3bd, 0xe3be, 0xe3bb, 0x8948, 0x89a5, 0xe3c0, 0xe3c1, 0xe3c2,
  0x9782, 0x8f4b, 0xe3c4, 0xe3c3, 0x9089, 0xe3c5, 0xe3c6, 0xe3c7,
  0x8ae3, 0x8acb, 0xe3c8, 0xe3c9, 0x967c, 0x9783, 0x9773, 0x9856,
  0x8d6c, 0xe3cc, 0x8ed2, 0xe3cb, 0xe3cd, 0x8ea7, 0x91cf, 0xe3ce,
  0x8d6b, 0x96d5, 0xe3cf, 0xe3d0, 0xe3d1, 0xe3d2, 0xe3d3, 0x8ea8,
  0x96eb, 0xe3d5, 0x925e, 0xe3d4, 0xe3d7, 0xe3d6, 0xe3d8, 0x90b9,
  0xe3d9, 0xe3da, 0x95b7, 0xe3db, 0x918f, 0xe3dc, 0xe3dd, 0x97fc,
  0xe3e0, 0xe3df, 0xe3de, 0x92ae, 0xe3e1, 0x9045, 0xe3e2, 0xe3e3,
  0x9857, 0xe3e4, 0xe3e5, 0xe3e7, 0xe3e6, 0x94a3, 0x93f7, 0x985d,
  0x94a7, 0xe3e9, 0x8fd1, 0x9549, 0xe3ea, 0xe3e8, 0x8acc, 0x8cd2,
  0x8e88, 0x94ec, 0x8ca8, 0x9662, 0xe3ed, 0xe3eb, 0x8d6d, 0x8d6e,
  0x88e7, 0x8de6, 0x9478, 0x88dd, 0xe3f2, 0x925f, 0x9477, 0x91d9,
  0xe3f4, 0xe3f0, 0xe3f3, 0xe3ee, 0xe3f1, 0x9645, 0x8cd3, 0x88fb,
  0xe3ef, 0xe3f6, 0xe3f7, 0x93b7, 0x8bb9, 0xe445, 0x945c, 0x8e89,
  0x8bba, 0x90c6, 0x9865, 0x96ac, 0xe3f5, 0x90d2, 0x8b72, 0xe3f8,
  0xe3fa, 0xe3f9, 0xe3fb, 0x9245, 0x945d, 0x92af, 0xe442, 0xe441,
  0xe3fc, 0x9074, 0x9585, 0xe444, 0xe443, 0x8d6f, 0x9872, 0xe454,
  0xe448, 0xe449, 0x8eee, 0xe447, 0x8d98, 0xe446, 0xe44a, 0x92b0,
  0x95a0, 0x9142, 0x91da, 0xe44e, 0xe44f, 0xe44b, 0xe44c, 0xe44d,
  0x8d70, 0xe455, 0xe451, 0x9586, 0x968c, 0x9547, 0xe450, 0xe453,
  0xe452, 0x9663, 0xe456, 0xe457, 0x9156, 0xe458, 0xe45a, 0xe45e,
  0xe45b, 0xe459, 0x945e, 0xe45c, 0xe45d, 0x89b0, 0xe464, 0xe45f,
  0xe460, 0xe461, 0x919f, 0xe463, 0xe462, 0xe465, 0xe466, 0xe467,
  0x9062, 0x89e7, 0xe468, 0x97d5, 0x8ea9, 0x8f4c, 0x8e8a, 0x9276,
  0xe469, 0xe46a, 0x8950, 0xe46b, 0xe46c, 0xe46d, 0xe46e, 0xe46f,
  0x8bbb, 0x9da8, 0xe470, 0x90e3, 0xe471, 0x8ec9, 0xe472, 0x98ae,
  0xe473, 0x95dc, 0x8ada, 0x9143, 0x8f77, 0x9591, 0x8f4d, 0xe474,
  0x8d71, 0xe475, 0x94ca, 0xe484, 0xe477, 0x91c7, 0x9495, 0x8cbd,
  0xe476, 0x9144, 0xe478, 0x92f8, 0xe47a, 0xe479, 0xe47c, 0xe47b,
  0xe47d, 0xe480, 0xe47e, 0x8acd, 0xe481, 0xe482, 0xe483, 0x8daf,
  0x97c7, 0xe485, 0x9046, 0x8990, 0xe486, 0xe487, 0xe488, 0x88f0,
  0xe489, 0xe48a, 0x9587, 0x8ec5, 0xe48c, 0x8a48, 0x88b0, 0xe48b,
  0xe48e, 0x946d, 0x9063, 0x89d4, 0x9646, 0x8c7c, 0x8bda, 0xe48d,
  0x89e8, 0x8aa1, 0x8991, 0xe492, 0x97e8, 0x91db, 0x9563, 0xe49e,
  0x89d5, 0xe49c, 0xe49a, 0xe491, 0xe48f, 0xe490, 0x8ee1, 0x8bea,
  0x9297, 0x93cf, 0x8970, 0xe494, 0xe493, 0xe499, 0xe495, 0xe498,
  0x96ce, 0xe497, 0x89d6, 0x8a9d, 0xe49b, 0xe49d, 0x8c73, 0xe4a1,
  0xe4aa, 0xe4ab, 0x88a9, 0xe4b2, 0x88ef, 0xe4a9, 0xe4a8, 0xe4a3,
  0xe4a2, 0xe4a0, 0xe49f, 0x9283, 0x91f9, 0xe4a5, 0xe4a4, 0xe4a7,
  0x9190, 0x8c74, 0x8960, 0xe4a6, 0x8d72, 0x9191, 0xe4b8, 0xe4b9,
  0x89d7, 0x89ac, 0xe4b6, 0xe4ac, 0xe4b4, 0xe4bb, 0xe4b5, 0xe4b3,
  0xe496, 0xe4b1, 0xe4ad, 0x8ace, 0xe4af, 0xe4ba, 0xe4b0, 0xe4bc,
  0xe4ae, 0x949c, 0x9789, 0xe4b7, 0xe4cd, 0xe4c5, 0x909b, 0x8b65,
  0x8bdb, 0xe4c0, 0x89d9, 0x8fd2, 0xe4c3, 0x8dd8, 0x9370, 0xe4c8,
  0x95ec, 0xe4bf, 0x89d8, 0x8cd4, 0x9548, 0xe4c9, 0xe4bd, 0xe4c6,
  0xe4d0, 0xe4c1, 0xe4c2, 0x93b8, 0xe4c7, 0xe4c4, 0x9647, 0xe4ca,
  0x88de, 0xe4be, 0xe4cc, 0xe4cb, 0x948b, 0xe4d2, 0xe4dd, 0x8a9e,
  0xe4e0, 0xe4ce, 0xe4d3, 0x978e, 0xe4dc, 0x9774, 0x97a8, 0x9298,
  0x8a8b, 0x9592, 0xe4e2, 0x939f, 0x88af, 0xe4db, 0xe4d7, 0x9192,
  0xe4d1, 0xe4d9, 0xe4de, 0x944b, 0x88a8, 0xe4d6, 0xe4df, 0x9598,
  0xe4da, 0xe4d5, 0x8fd3, 0x8f4e, 0x8eaa, 0x96d6, 0x9566, 0xe4e5,
  0xe4ee, 0xe4d8, 0x8a97, 0x8ff6, 0xe4e3, 0xe4e8, 0x9193, 0xe4e4,
  0xe4eb, 0x927e, 0xe4ec, 0x9775, 0xe4e1, 0x8a57, 0xe4e7, 0xe4ea,
  0x96aa, 0xe4ed, 0xe4e6, 0xe4e9, 0x9648, 0x9840, 0xe4f1, 0xe4f8,
  0xe4f0, 0x8ec1, 0xe4cf, 0x95cc, 0x96a0, 0xe4f7, 0xe4f6, 0xe4f2,
  0xe4f3, 0x8955, 0xe4f5, 0xe4ef, 0x92d3, 0xe4f4, 0x88fc, 0x91a0,
  0x95c1, 0xe4f9, 0xe540, 0x94d7, 0xe4fc, 0x8fd4, 0x8ec7, 0xe542,
  0x8bbc, 0xe543, 0x9599, 0xe4fb, 0xe4d4, 0xe4fa, 0x986e, 0x93a0,
  0x9593, 0xe54a, 0xe550, 0xe551, 0xe544, 0x9496, 0xe54e, 0xe546,
  0xe548, 0xe552, 0xe547, 0xe54b, 0x8992, 0x93e3, 0xe54c, 0xe54f,
  0xe545, 0x9145, 0xe549, 0x8e46, 0x9064, 0x8c4f, 0x96f2, 0x96f7,
  0x8f92, 0xe556, 0xe554, 0x986d, 0xe553, 0x9795, 0xe555, 0xe557,
  0xe558, 0xe55b, 0xe559, 0x93a1, 0xe55a, 0x94cb, 0xe54d, 0x8f93,
  0xe55c, 0xe561, 0x9194, 0xe560, 0xe541, 0xe562, 0x9168, 0xe55d,
  0xe55f, 0xe55e, 0x9f50, 0x9f41, 0xe564, 0xe563, 0x9796, 0xe1ba,
  0xe565, 0xe566, 0xe567, 0x8cd5, 0x8b73, 0xe569, 0x997c, 0x8b95,
  0x97b8, 0x8bf1, 0xe56a, 0xe56b, 0x928e, 0xe56c, 0x93f8, 0x88b8,
  0x89e1, 0xe571, 0xe572, 0xe56d, 0x8e5c, 0xe56e, 0x9461, 0xe56f,
  0xe570, 0xe57a, 0xe574, 0xe577, 0xe573, 0xe575, 0xe576, 0x8ed6,
  0xe578, 0x9260, 0x8c75, 0x8a61, 0xe57b, 0x8a5e, 0xe581, 0xe57c,
  0xe580, 0x94b8, 0xe57d, 0xe57e, 0x9567, 0x94d8, 0xe582, 0x91fb,
  0xe58c, 0xe588, 0x89e9, 0xe586, 0x9649, 0xe587, 0xe584, 0xe585,
  0xe58a, 0xe58d, 0xe58b, 0xe589, 0xe583, 0x9277, 0xe594, 0x96a8,
  0xe592, 0xe593, 0xe58e, 0xe590, 0xe591, 0xe58f, 0x90e4, 0x9858,
  0xe598, 0xe599, 0xe59f, 0x9049, 0xe59b, 0xe59e, 0xe596, 0xe595,
  0xe5a0, 0x89da, 0xe59c, 0xe5a1, 0xe59d, 0xe59a, 0x92b1, 0xe597,
  0x9488, 0xe5a5, 0x975a, 0xe5a4, 0xe5a3, 0xe5ac, 0xe5a6, 0xe5ae,
  0x9786, 0xe5b1, 0xe5a8, 0xe5a9, 0xe5ad, 0xe5b0, 0xe5af, 0xe5a7,
  0xe5aa, 0xe5bb, 0xe5b4, 0xe5b2, 0xe5b3, 0xe5b8, 0xe5b9, 0x8a49,
  0x8b61, 0xe5b7, 0xe5a2, 0xe5b6, 0xe5ba, 0xe5b5, 0xe5bc, 0xe5be,
  0xe5bd, 0xe5c0, 0xe5bf, 0xe579, 0xe5c4, 0xe5c1, 0xe5c2, 0xe5c3,
  0xe5c5, 0x8c8c, 0xe5c7, 0xe5c6, 0x8f4f, 0x8d73, 0x9fa5, 0xe5c8,
  0x8f70, 0x8a58, 0xe5c9, 0x8971, 0x8fd5, 0xe5ca, 0x8d74, 0xe5cb,
  0x88df, 0x955c, 0xe5cc, 0x908a, 0xe5d3, 0xe5d0, 0x928f, 0xe5d1,
  0xe5ce, 0x8bdc, 0xe5cd, 0xe5d4, 0x8c55, 0x91dc, 0xe5da, 0xe5d6,
  0x91b3, 0xe5d5, 0xe5d8, 0xe5cf, 0xe5d9, 0xe5db, 0x94ed, 0xe5d7,
  0xe5dc, 0xe5de, 0x8cd1, 0xe5d2, 0x88bf, 0xe5dd, 0x8dd9, 0x97f4,
  0xe5df, 0xe5e0, 0x9195, 0x97a0, 0xe5e1, 0x9754, 0xe5e2, 0xe5e3,
  0x95e2, 0xe5e4, 0x8dbe, 0x97a1, 0xe5e9, 0xe5ea, 0x8fd6, 0xe5e8,
  0x9787, 0xe5e5, 0xe5e7, 0x90bb, 0x909e, 0xe5e6, 0xe5eb, 0x95a1,
  0xe5ed, 0xe5ec, 0x8a8c, 0x964a, 0xe5ee, 0xe5fa, 0xe5f0, 0xe5f1,
  0xe5f2, 0xe5f3, 0xe5f7, 0xe5f8, 0xe5f6, 0xe5f4, 0xe5ef, 0xe5f5,
  0xe5f9, 0xe8b5, 0x89a6, 0xe5fc, 0x8bdd, 0xe5fb, 0xe641, 0xe640,
  0xe643, 0xe642, 0xe644, 0x8f50, 0xe645, 0xe646, 0xe647, 0x90bc,
  0x9776, 0xe648, 0x95a2, 0x9465, 0xe649, 0xe64a, 0x8ca9, 0x8b4b,
  0xe64b, 0x8e8b, 0x9460, 0xe64c, 0x8a6f, 0xe64d, 0xe64f, 0x9797,
  0xe64e, 0x9065, 0xe650, 0xe651, 0xe652, 0x8acf, 0xe653, 0xe654,
  0xe655, 0xe656, 0x8a70, 0xe657, 0xe658, 0xe659, 0x89f0, 0x9047,
  0xe65a, 0xe65b, 0xe65c, 0x8cbe, 0x92f9, 0xe65d, 0x8c76, 0x9075,
  0xe660, 0x93a2, 0xe65f, 0x8c50, 0xe65e, 0x91f5, 0x8b4c, 0xe661,
  0xe662, 0x8fd7, 0x8c8d, 0xe663, 0x964b, 0x90dd, 0x8b96, 0x96f3,
  0x9169, 0xe664, 0x9066, 0x9290, 0x8fd8, 0xe665, 0xe668, 0xe669,
  0x8dbc, 0x91c0, 0xe667, 0x8fd9, 0x955d, 0xe666, 0x8e8c, 0x8972,
  0xe66d, 0x8c77, 0x8e8e, 0x8e8d, 0x986c, 0xe66c, 0xe66b, 0x9146,
  0x8b6c, 0x9862, 0x8a59, 0x8fda, 0xe66a, 0xe66f, 0xe670, 0xe66e,
  0x8cd6, 0x975f, 0x8e8f, 0x9446, 0xe673, 0x90be, 0x9261, 0x9755,
  0xe676, 0x8cea, 0x90bd, 0xe672, 0xe677, 0x8ceb, 0xe674, 0xe675,
  0xe671, 0x90e0, 0x93c7, 0x924e, 0x89db, 0x94ee, 0x8b62, 0x92b2,
  0xe67a, 0xe678, 0x926b, 0x90bf, 0x8ad0, 0xe679, 0x907a, 0x97c8,
  0x985f, 0xe67b, 0xe687, 0x92b3, 0xe686, 0xe683, 0xe68b, 0xe684,
  0xe680, 0x92fa, 0xe67e, 0xe67c, 0x9740, 0x8e90, 0xe681, 0xe67d,
  0xe685, 0x8f94, 0x8cbf, 0x91f8, 0x9664, 0x8979, 0x88e0, 0x93a3,
  0xe689, 0xe688, 0x93e4, 0xe68d, 0xe682, 0xe68c, 0xe68e, 0x8caa,
  0xe68a, 0x8d75, 0x8ed3, 0xe68f, 0x9777, 0xe692, 0xe695, 0xe693,
  0x9554, 0xe690, 0x8bde, 0xe694, 0xe696, 0xe69a, 0xe697, 0xe699,
  0xe698, 0xe69b, 0x8eaf, 0xe69d, 0xe69c, 0x9588, 0xe69f, 0x8c78,
  0xe69e, 0xe6a0, 0xe6a1, 0x8b63, 0xe3bf, 0x8ff7, 0xe6a2, 0x8cec,
  0xe6a3, 0xe6a4, 0x8e5d, 0x9dcc, 0xe6a5, 0xe6a6, 0x8f51, 0xe6a7,
  0xe6a8, 0xe6a9, 0xe6aa, 0xe6ab, 0x924a, 0xe6ac, 0xe6ae, 0xe6ad,
  0x93a4, 0xe6af, 0x964c, 0xe6b0, 0xe6b1, 0xe6b2, 0xe6b3, 0x93d8,
  0x8fdb, 0xe6b4, 0x8d8b, 0x98ac, 0xe6b5, 0xe6b6, 0x955e, 0xe6b7,
  0xe6bf, 0xe6b8, 0xe6ba, 0xe6b9, 0xe6bb, 0x9665, 0xe6bc, 0xe6bd,
  0xe6be, 0xe6c0, 0x8a4c, 0x92e5, 0x9589, 0x8de0, 0x8d76, 0x956e,
  0x89dd, 0x94cc, 0xe6c3, 0x8ad1, 0x90d3, 0xe6c2, 0xe6c7, 0x9299,
  0x96e1, 0xe6c5, 0xe6c6, 0x8b4d, 0xe6c8, 0x9483, 0x91dd, 0x94ef,
  0x935c, 0xe6c4, 0x9666, 0x89ea, 0xe6ca, 0x9847, 0x92c0, 0x9864,
  0x8e91, 0xe6c9, 0x91af, 0xe6da, 0x9147, 0x93f6, 0x956f, 0xe6cd,
  0x8e5e, 0x8e92, 0x8fdc, 0x9485, 0x8cab, 0xe6cc, 0xe6cb, 0x958a,
  0x8ebf, 0x9371, 0xe6cf, 0xe6d0, 0x8d77, 0xe6ce, 0xe6d1, 0xe6d2,
  0xe6d4, 0x91a1, 0xe6d3, 0x8ae4, 0xe6d6, 0xe6d5, 0xe6d7, 0xe6d9,
  0xe6db, 0xe6dc, 0x90d4, 0x8ecd, 0xe6dd, 0x8a71, 0xe6de, 0x9196,
  0xe6df, 0xe6e0, 0x958b, 0x8b4e, 0xe6e1, 0x92b4, 0x897a, 0xe6e2,
  0x8eef, 0x9096, 0x91ab, 0xe6e5, 0xe6e4, 0xe6e3, 0xe6eb, 0xe6e9,
  0xe6e6, 0xe6e8, 0xe6e7, 0xe6ea, 0x8b97, 0xe6ee, 0x90d5, 0xe6ef,
  0x8cd7, 0xe6ec, 0xe6ed, 0x9848, 0x92b5, 0x9148, 0xe6f0, 0xe6f3,
  0xe6f1, 0xe6f2, 0x9778, 0x93a5, 0xe6f6, 0xe6f4, 0xe6f5, 0xe6f7,
  0xe748, 0xe6fa, 0xe6fb, 0xe6f9, 0xe6f8, 0x92fb, 0xe740, 0xe744,
  0xe741, 0xe6fc, 0xe742, 0xe743, 0xe74a, 0xe745, 0x90d6, 0xe747,
  0xe749, 0xe746, 0xe74c, 0x8f52, 0xe74b, 0xe74d, 0xe74e, 0xe751,
  0xe750, 0xe74f, 0xe753, 0xe752, 0x96f4, 0xe755, 0xe754, 0xe756,
  0xe757, 0xe759, 0xe758, 0x9067, 0xe75a, 0x8beb, 0xe75b, 0xe75d,
  0xe75e, 0xe75f, 0xe75c, 0xe760, 0x8ed4, 0xe761, 0x8b4f, 0x8c52,
  0x8cac, 0xe762, 0x93ee, 0x935d, 0xe763, 0xe766, 0x8eb2, 0xe765,
  0xe764, 0x8c79, 0xe767, 0x8a72, 0xe769, 0x8dda, 0xe768, 0xe771,
  0xe76b, 0xe76d, 0x95e3, 0xe76a, 0xe76c, 0xe770, 0xe76e, 0x8b50,
  0xe76f, 0xe772, 0x9479, 0x97d6, 0x8f53, 0xe773, 0x9741, 0xe775,
  0xe774, 0xe778, 0x9760, 0xe777, 0x8a8d, 0xe776, 0xe77b, 0xe77a,
  0xe779, 0x9351, 0xe77c, 0xe77d, 0xe77e, 0x8d8c, 0x8c44, 0xe780,
  0xe781, 0xe782, 0x9068, 0xe783, 0x8eab, 0xe784, 0xe785, 0x999f,
  0x999e, 0xe786, 0xe390, 0xe787, 0x9243, 0x904a, 0x945f, 0xe788,
  0x95d3, 0x92d2, 0x8d9e, 0x9248, 0x8949, 0x9698, 0x9076, 0x8c7d,
  0x8bdf, 0x95d4, 0xe789, 0xe78b, 0xe78a, 0x89de, 0x93f4, 0xe78c,
  0x9497, 0x9352, 0xe78d, 0x8f71, 0xe78f, 0x96c0, 0xe79e, 0xe791,
  0xe792, 0x92c7, 0x91de, 0x9197, 0x93a6, 0xe790, 0x8b74, 0xe799,
  0xe796, 0xe7a3, 0x93a7, 0x9280, 0xe793, 0x92fc, 0x9372, 0xe794,
  0xe798, 0x9080, 0x9487, 0x92ca, 0x90c0, 0xe797, 0x91ac, 0x91a2,
  0xe795, 0x88a7, 0x9841, 0xe79a, 0x91df, 0x8f54, 0x9069, 0xe79c,
  0xe79b, 0x88ed, 0xe79d, 0x954e, 0xe7a5, 0x93d9, 0x908b, 0x9278,
  0x8bf6, 0xe7a4, 0x9756, 0x895e, 0x95d5, 0x89df, 0xe79f, 0xe7a0,
  0xe7a1, 0xe7a2, 0x93b9, 0x9242, 0x88e1, 0xe7a6, 0xe7a7, 0xeaa1,
  0x91bb, 0xe7a8, 0x8993, 0x916b, 0x8cad, 0x9779, 0xe7a9, 0x934b,
  0x9198, 0x8ed5, 0xe7aa, 0xe7ad, 0x8f85, 0xe7ab, 0x914a, 0x9149,
  0x88e2, 0x97c9, 0xe7af, 0x94f0, 0xe7b1, 0xe7b0, 0xe7ae, 0xe284,
  0x8ad2, 0xe78e, 0xe7b3, 0xe7b2, 0xe7b4, 0x9757, 0x93df, 0x964d,
  0xe7b5, 0x8ed7, 0xe7b6, 0xe7b7, 0xe7b8, 0x9340, 0x88e8, 0x8d78,
  0x9859, 0xe7bc, 0x8c53, 0xe7b9, 0xe7ba, 0x9594, 0x8a73, 0x9758,
  0x8bbd, 0x9373, 0xe7bd, 0xe7be, 0xe7bf, 0x9341, 0xe7c1, 0xe7c0,
  0x93d1, 0xe7c2, 0x8f55, 0x8ede, 0x947a, 0x9291, 0x8ef0, 0x908c,
  0xe7c3, 0xe7c4, 0x907c, 0xe7c5, 0xe7c6, 0xe7c7, 0x978f, 0x8f56,
  0xe7c9, 0xe7c8, 0x8d79, 0x8d93, 0x8e5f, 0xe7cc, 0x8f86, 0xe7cb,
  0xe7ca, 0x91e7, 0x8ced, 0x90c1, 0x94ae, 0x8f58, 0xe7cd, 0x8fdd,
  0xe7d0, 0xe7ce, 0xe7cf, 0xe7d2, 0xe7d1, 0x8ff8, 0xe7d3, 0xe7d4,
  0xe7d5, 0x94ce, 0x8dd1, 0x8edf, 0xe7d6, 0xe7d7, 0x97a2, 0x8f64,
  0x96ec, 0x97ca, 0xe7d8, 0x8be0, 0xe7d9, 0x9342, 0xe7dc, 0x8a98,
  0x906a, 0xe7da, 0xe7db, 0x92de, 0x9674, 0x8bfa, 0xe7de, 0xe7df,
  0xe7dd, 0xe7e1, 0x93dd, 0x8a62, 0xe7e5, 0xe7e2, 0xe7e4, 0xe7e0,
  0xe86e, 0xe7e3, 0x97e9, 0x8cd8, 0xe7ed, 0x9353, 0xe7e8, 0xe7eb,
  0xe7e9, 0xe7ee, 0xe7ef, 0xe7e7, 0xe7f4, 0x8994, 0xe7e6, 0x94ab,
  0xe7ea, 0x8fde, 0x8d7a, 0x9667, 0x8be2, 0x8f65, 0x93ba, 0x914c,
  0xe7f2, 0xe7ec, 0xe7f1, 0x96c1, 0x92b6, 0xe7f3, 0xe7f0, 0x914b,
  0xe7f7, 0xe7f6, 0xe7f5, 0x964e, 0x8f9b, 0xe7f8, 0x95dd, 0x8973,
  0x9565, 0x9292, 0x8b98, 0xe7fa, 0x8d7c, 0x8e4b, 0xe7f9, 0x908d,
  0x908e, 0xe840, 0xe842, 0x8ff9, 0xe841, 0xe843, 0x8bd1, 0x9564,
  0x8ee0, 0x9842, 0xe7fc, 0x8df6, 0x985e, 0xe845, 0xe844, 0xe846,
  0xe7fb, 0x93e7, 0x9374, 0x92d5, 0xe84b, 0x9262, 0xe847, 0xe848,
  0x8c4c, 0xe84a, 0x8cae, 0xe849, 0x8fdf, 0x8a99, 0xe84f, 0x8dbd,
  0x9199, 0x92c8, 0x8a5a, 0xe84d, 0xe84e, 0x92c1, 0xe84c, 0xe850,
  0xe856, 0xe859, 0xe858, 0x934c, 0xe851, 0xe852, 0xe855, 0xe857,
  0x8bbe, 0xe85a, 0xe854, 0xe853, 0xe85e, 0xe85f, 0xe860, 0xe85d,
  0xe85c, 0x8fe0, 0x93a8, 0xe85b, 0xe864, 0xe862, 0xe863, 0xe861,
  0x91f6, 0xe865, 0xe866, 0xe868, 0x8ad3, 0xe867, 0x96f8, 0xe873,
  0xe869, 0xe86c, 0xe86a, 0xe86b, 0xe86d, 0xe86f, 0xe870, 0xe871,
  0xe874, 0xe872, 0xe875, 0xe877, 0xe876, 0x92b7, 0x96e5, 0xe878,
  0x914d, 0xe879, 0x95c2, 0xe87a, 0x8a4a, 0x895b, 0x8ad5, 0x8ad4,
  0xe87b, 0xe87c, 0xe87d, 0xe87e, 0xe880, 0x8ad6, 0x8a74, 0x8d7d,
  0x94b4, 0xe882, 0xe881, 0xe883, 0x897b, 0xe886, 0xe885, 0xe884,
  0xe887, 0xe88a, 0x88c5, 0xe888, 0xe88c, 0xe88b, 0xe88e, 0xe88d,
  0xe88f, 0x93ac, 0xe890, 0xe891, 0xe893, 0xe892, 0x958c, 0xe894,
  0xe895, 0x8de3, 0xe896, 0xe897, 0x9668, 0x916a, 0x88a2, 0x91c9,
  0xe898, 0x958d, 0xe89b, 0xe899, 0x8d7e, 0xe89a, 0x8cc0, 0x95c3,
  0xe89d, 0xe89f, 0xe89e, 0xe8a0, 0x8940, 0x9077, 0x8f9c, 0x8ad7,
  0xe8a1, 0x9486, 0xe8a3, 0x8941, 0xe8a2, 0x92c2, 0x97cb, 0x93a9,
  0xe89c, 0x97a4, 0x8caf, 0x977a, 0x8bf7, 0x97b2, 0x8c47, 0x91e0,
  0xe440, 0xe8a4, 0x8a4b, 0x908f, 0x8a75, 0xe8a6, 0xe8a7, 0xe8a5,
  0x8c84, 0x8ddb, 0x8fe1, 0x8942, 0x97d7, 0xe8a9, 0xe7ac, 0xe8a8,
  0xe8ac, 0xe8aa, 0xe8ab, 0xe8ad, 0xe8ae, 0x97ea, 0xe8af, 0xe8b0,
  0x90c7, 0x94b9, 0x909d, 0x8ae5, 0x9759, 0x89eb, 0x8f57, 0x8cd9,
  0xe8b3, 0xe8b2, 0x8e93, 0xe8b4, 0xe8b1, 0x8e47, 0xe8b8, 0xe5ab,
  0x99d4, 0x9097, 0xe8b6, 0x97a3, 0x93ef, 0x894a, 0x90e1, 0x8eb4,
  0x95b5, 0x895f, 0x97eb, 0x978b, 0xe8b9, 0x9364, 0x8ef9, 0xe8ba,
  0xe8bb, 0x906b, 0xe8bc, 0x97ec, 0xe8b7, 0xe8be, 0xe8c0, 0xe8bf,
  0xe8bd, 0xe8c1, 0xe8c2, 0x919a, 0x89e0, 0xe8c3, 0x96b6, 0xe8c4,
  0xe8c5, 0x9849, 0x9e50, 0xe8c6, 0xe8c7, 0xe8c8, 0xe8cc, 0xe8c9,
  0xe8ca, 0xe8cb, 0xe8cd, 0x90c2, 0x96f5, 0x90c3, 0xe8ce, 0x94f1,
  0xe8cf, 0xea72, 0x96ca, 0xe8d0, 0xe8d1, 0xe8d2, 0x8a76, 0xe8d4,
  0x9078, 0xe8d5, 0x8c43, 0xe8d6, 0xe8da, 0xe8d8, 0xe8d9, 0x8a93,
  0xe8d7, 0xe8db, 0xe8dc, 0x88c6, 0xe8dd, 0xe8de, 0x8fe2, 0xe8df,
  0x8b66, 0xe8e2, 0xe8e1, 0xe8e0, 0xe691, 0x95da, 0xe8e3, 0xe8e4,
  0xe8e5, 0xe8e6, 0xe8e7, 0xe8e8, 0x8ad8, 0xe8e9, 0xe8ea, 0x9442,
  0xe8ec, 0x89b9, 0xe8ef, 0xe8ee, 0x8943, 0x8bbf, 0x95c5, 0x92b8,
  0x8da0, 0x8d80, 0x8f87, 0x907b, 0xe8f1, 0xe8f0, 0x9761, 0x8ae6,
  0x94d0, 0x93da, 0x909c, 0x97cc, 0x8c7a, 0xe8f4, 0xe8f3, 0x966a,
  0x93aa, 0x896f, 0xe8f5, 0xe8f2, 0x9570, 0x978a, 0xe8f6, 0xe8f7,
  0xe8f9, 0x91e8, 0x8a7a, 0x8a7b, 0xe8f8, 0x8ae7, 0x8cb0, 0x8ae8,
  0x935e, 0x97de, 0x8cda, 0xe8fa, 0xe8fb, 0xe8fc, 0xe940, 0xe942,
  0xe941, 0x9597, 0xe943, 0xe944, 0xe945, 0xe946, 0xe948, 0xe947,
  0xe949, 0x94f2, 0xe3ca, 0x9048, 0x8b51, 0xe94a, 0xe94b, 0x99aa,
  0x9f5a, 0x94d1, 0x88f9, 0x88b9, 0x8e94, 0x964f, 0x8ffc, 0xe94c,
  0x96dd, 0xe94d, 0x977b, 0x8961, 0x8e60, 0xe94e, 0x89ec, 0xe94f,
  0xe950, 0xe952, 0xe953, 0xe955, 0xe951, 0xe954, 0x8ad9, 0xe956,
  0xe957, 0xe958, 0xe959, 0xe95a, 0xe95c, 0xe95b, 0xe95e, 0xe961,
  0xe95d, 0xe95f, 0xe960, 0xe962, 0x8bc0, 0x8ef1, 0xe963, 0xe964,
  0x8d81, 0xe965, 0x8a5d, 0x946e, 0xe966, 0xe967, 0x9279, 0x93e9,
  0xe968, 0x949d, 0x91ca, 0x8977, 0x8bec, 0x8bed, 0x9293, 0xe96d,
  0x8bee, 0x89ed, 0xe96c, 0xe96a, 0xe96b, 0xe969, 0xe977, 0xe96e,
  0xe96f, 0xe970, 0xe971, 0xe973, 0xe972, 0x8f78, 0xe974, 0xe976,
  0x8b52, 0xe975, 0x919b, 0x8cb1, 0xe978, 0x91cb, 0xe979, 0x93ab,
  0xe97a, 0xe980, 0xe97d, 0xe97c, 0xe97e, 0xe97b, 0xe982, 0xe981,
  0xe984, 0x8bc1, 0xe983, 0xe985, 0xe986, 0xe988, 0xe987, 0xe989,
  0xe98b, 0xe98a, 0x8d9c, 0xe98c, 0xe98d, 0x8a5b, 0xe98e, 0xe98f,
  0x9091, 0xe990, 0xe991, 0xe992, 0xe993, 0x8d82, 0xe994, 0xe995,
  0xe996, 0xe997, 0xe998, 0x94af, 0xe99a, 0x9545, 0xe99b, 0xe999,
  0xe99d, 0xe99c, 0xe99e, 0xe99f, 0xe9a0, 0xe9a1, 0xe9a2, 0xe9a3,
  0xe9a4, 0xe9a5, 0xe9a6, 0xe9a7, 0xe9a8, 0xe9a9, 0xe9aa, 0xe9ab,
  0xe9ac, 0x9f54, 0xe9ad, 0xe2f6, 0x8b53, 0x8a40, 0x8db0, 0xe9af,
  0xe9ae, 0x96a3, 0xe9b1, 0xe9b2, 0xe9b0, 0xe9b3, 0x9682, 0xe9b4,
  0x8b9b, 0x9844, 0xe9b5, 0xe9b7, 0x88bc, 0xe9b8, 0x95a9, 0xe9b6,
  0xe9b9, 0xe9ba, 0xe9bb, 0xe9bc, 0xe9bd, 0x968e, 0x8e4c, 0x8df8,
  0x914e, 0xe9be, 0xe9c1, 0xe9bf, 0xe9c2, 0x8cef, 0xe9c0, 0xe9c3,
  0xe9c4, 0xe9c5, 0xe9c9, 0x8e49, 0x91e2, 0xe9ca, 0xe9c7, 0xe9c6,
  0xe9c8, 0x8c7e, 0xe9ce, 0xe9cd, 0xe9cc, 0x88b1, 0xe9d8, 0xe9d4,
  0xe9d5, 0xe9d1, 0xe9d7, 0xe9d3, 0x8a82, 0x986b, 0xe9d6, 0xe9d2,
  0xe9d0, 0xe9cf, 0xe9da, 0xe9dd, 0xe9dc, 0xe9db, 0x9568, 0xe9d9,
  0x88f1, 0xe9de, 0xe9e0, 0x8a8f, 0xe9cb, 0x8956, 0xe9e2, 0xe9e1,
  0xe9df, 0x924c, 0x9690, 0x97d8, 0xe9e3, 0xe9e4, 0xe9e5, 0xe9e6,
  0xe9e7, 0x92b9, 0xe9e8, 0x94b5, 0xe9ed, 0xe9e9, 0xe9ea, 0x9650,
  0x96c2, 0x93ce, 0xe9ee, 0xe9ef, 0x93bc, 0xe9ec, 0xe9eb, 0x89a8,
  0xe9f7, 0xe9f6, 0x8995, 0xe9f4, 0xe9f3, 0xe9f1, 0x8a9b, 0xe9f0,
  0x8eb0, 0x89a7, 0x8d83, 0xe9fa, 0xe9f9, 0xe9f8, 0xe9f5, 0xe9fb,
  0xe9fc, 0xea44, 0xea43, 0xea45, 0x894c, 0xea40, 0xea41, 0x8d94,
  0x96b7, 0xea42, 0x9651, 0xea4a, 0xea46, 0xea4b, 0xea48, 0xea47,
  0x8c7b, 0xea4c, 0xea4d, 0xea4e, 0xea49, 0xe9f2, 0xea4f, 0x92df,
  0xea53, 0xea54, 0xea52, 0xea51, 0xea57, 0xea50, 0xea55, 0xea56,
  0xea59, 0xea58, 0xea5b, 0xea5c, 0xea5d, 0x9868, 0xea5a, 0x91e9,
  0x8deb, 0xea5e, 0xea5f, 0xea60, 0xea61, 0xea62, 0x8cb2, 0xea63,
  0xea64, 0x8ead, 0xea65, 0xea66, 0xea67, 0xea68, 0xea6b, 0xea69,
  0x985b, 0xea6a, 0x97ed, 0xea6c, 0x97d9, 0xea6d, 0x949e, 0xea6e,
  0xea70, 0xea71, 0xea6f, 0x8d8d, 0x96cb, 0x9683, 0x9bf5, 0x9f80,
  0x969b, 0x89a9, 0xea73, 0x8b6f, 0xea74, 0xea75, 0xea76, 0x8d95,
  0xea77, 0xe0d2, 0x96d9, 0x91e1, 0xea78, 0xea7a, 0xea79, 0xea7b,
  0xea7c, 0xea7d, 0xea7e, 0xea80, 0xea81, 0xea82, 0xea83, 0xea84,
  0xea85, 0xea86, 0xea87, 0xea88, 0x9343, 0x8cdb, 0xea8a, 0x916c,
  0xea8b, 0xea8c, 0x9540, 0xea8d, 0xea8e, 0xe256, 0xe6d8, 0xe8eb,
  0xea8f, 0xea90, 0xea92, 0xea93, 0xea94, 0x97ee, 0xea91, 0xea95,
  0xea96, 0xea98, 0xea97, 0xea9a, 0xea9b, 0xea99, 0x97b4, 0xea9c,
  0xea9d, 0xe273, 0xea9e, 0x8149, 0x8194, 0x8190, 0x8193, 0x8195,
  0x8169, 0x816a, 0x8196, 0x817b, 0x8143, 0x8144, 0x815e, 0x824f,
  0x8250, 0x8251, 0x8252, 0x8253, 0x8254, 0x8255, 0x8256, 0x8257,
  0x8258, 0x8146, 0x8147, 0x8183, 0x8181, 0x8184, 0x8148, 0x8197,
  0x8260, 0x8261, 0x8262, 0x8263, 0x8264, 0x8265, 0x8266, 0x8267,
  0x8268, 0x8269, 0x826a, 0x826b, 0x826c, 0x826d, 0x826e, 0x826f,
  0x8270, 0x8271, 0x8272, 0x8273, 0x8274, 0x8275, 0x8276, 0x8277,
  0x8278, 0x8279, 0x816d, 0x815f, 0x816e, 0x814f, 0x8151, 0x814d,
  0x8281, 0x8282, 0x8283, 0x8284, 0x8285, 0x8286, 0x8287, 0x8288,
  0x8289, 0x828a, 0x828b, 0x828c, 0x828d, 0x828e, 0x828f, 0x8290,
  0x8291, 0x8292, 0x8293, 0x8294, 0x8295, 0x8296, 0x8297, 0x8298,
  0x8299, 0x829a, 0x816f, 0x8162, 0x8170, 0x8150, 0x818f,
};

typedef struct {
  unsigned short indx; /* index into big table */
  unsigned short used; /* bitmask of used entries */
} Summary16;

static const Summary16 jisx0208_uni2indx_page00[16] = {
  /* 0x0000 */
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 },
  /* ZINT: Patched reverse solidus (backslash) mapping U+005C to 0x815F */
  {    0, 0x0000 }, { 6843, 0x1000 }, {    0, 0x0000 }, {    0, 0x0000 },
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x118c }, {    5, 0x0053 },
  {    9, 0x0000 }, {    9, 0x0080 }, {   10, 0x0000 }, {   10, 0x0080 },
};
static const Summary16 jisx0208_uni2indx_page03[22] = {
  /* 0x0300 */
  {   11, 0x0000 }, {   11, 0x0000 }, {   11, 0x0000 }, {   11, 0x0000 },
  {   11, 0x0000 }, {   11, 0x0000 }, {   11, 0x0000 }, {   11, 0x0000 },
  {   11, 0x0000 }, {   11, 0xfffe }, {   26, 0x03fb }, {   35, 0xfffe },
  {   50, 0x03fb }, {   59, 0x0000 }, {   59, 0x0000 }, {   59, 0x0000 },
  /* 0x0400 */
  {   59, 0x0002 }, {   60, 0xffff }, {   76, 0xffff }, {   92, 0xffff },
  {  108, 0xffff }, {  124, 0x0002 },
};
static const Summary16 jisx0208_uni2indx_page20[50] = {
  /* 0x2000 */
  {  125, 0x0000 }, {  125, 0x3361 }, {  132, 0x0063 }, {  136, 0x080d },
  {  140, 0x0000 }, {  140, 0x0000 }, {  140, 0x0000 }, {  140, 0x0000 },
  {  140, 0x0000 }, {  140, 0x0000 }, {  140, 0x0000 }, {  140, 0x0000 },
  {  140, 0x0000 }, {  140, 0x0000 }, {  140, 0x0000 }, {  140, 0x0000 },
  /* 0x2100 */
  {  140, 0x0008 }, {  141, 0x0000 }, {  141, 0x0800 }, {  142, 0x0000 },
  {  142, 0x0000 }, {  142, 0x0000 }, {  142, 0x0000 }, {  142, 0x0000 },
  {  142, 0x0000 }, {  142, 0x000f }, {  146, 0x0000 }, {  146, 0x0000 },
  {  146, 0x0000 }, {  146, 0x0014 }, {  148, 0x0000 }, {  148, 0x0000 },
  /* 0x2200 */
  {  148, 0x098d }, {  154, 0x6404 }, {  158, 0x1f81 }, {  165, 0x2030 },
  {  168, 0x0000 }, {  168, 0x0004 }, {  169, 0x0cc3 }, {  175, 0x0000 },
  {  175, 0x00cc }, {  179, 0x0000 }, {  179, 0x0020 }, {  180, 0x0000 },
  {  180, 0x0000 }, {  180, 0x0000 }, {  180, 0x0000 }, {  180, 0x0000 },
  /* 0x2300 */
  {  180, 0x0000 }, {  180, 0x0004 },
};
static const Summary16 jisx0208_uni2indx_page25[23] = {
  /* 0x2500 */
  {  181, 0x900f }, {  187, 0x3999 }, {  195, 0x9939 }, {  203, 0x9999 },
  {  211, 0x0804 }, {  213, 0x0000 }, {  213, 0x0000 }, {  213, 0x0000 },
  {  213, 0x0000 }, {  213, 0x0000 }, {  213, 0x0003 }, {  215, 0x300c },
  {  219, 0xc8c0 }, {  224, 0x0000 }, {  224, 0x8000 }, {  225, 0x0000 },
  /* 0x2600 */
  {  225, 0x0060 }, {  227, 0x0000 }, {  227, 0x0000 }, {  227, 0x0000 },
  {  227, 0x0005 }, {  229, 0x0000 }, {  229, 0xa400 },
};
static const Summary16 jisx0208_uni2indx_page30[16] = {
  /* 0x3000 */
  {  232, 0xffef }, {  247, 0x103f }, {  254, 0x0000 }, {  254, 0x0000 },
  {  254, 0xfffe }, {  269, 0xffff }, {  285, 0xffff }, {  301, 0xffff },
  {  317, 0xffff }, {  333, 0x780f }, {  341, 0xfffe }, {  356, 0xffff },
  {  372, 0xffff }, {  388, 0xffff }, {  404, 0xffff }, {  420, 0x787f },
};
static const Summary16 jisx0208_uni2indx_page4e[1307] = {
  /* 0x4e00 */
  {  431, 0x6f8b }, {  441, 0x43f3 }, {  450, 0x2442 }, {  454, 0x9b46 },
  {  462, 0xe82c }, {  469, 0xe3e0 }, {  477, 0x0004 }, {  478, 0x400a },
  {  481, 0x5f65 }, {  491, 0xdb36 }, {  501, 0x7977 }, {  512, 0x0449 },
  {  516, 0xecd7 }, {  527, 0xe3f0 }, {  536, 0x6038 }, {  541, 0x08c5 },
  /* 0x4f00 */
  {  546, 0xe602 }, {  552, 0x3403 }, {  557, 0x8000 }, {  558, 0x3551 },
  {  565, 0xe0c8 }, {  571, 0x7eab }, {  582, 0x8200 }, {  584, 0x9869 },
  {  591, 0xa948 }, {  597, 0x2942 }, {  602, 0xe803 }, {  608, 0x8060 },
  {  611, 0x441c }, {  616, 0xad93 }, {  625, 0xc03a }, {  631, 0x4568 },
  /* 0x5000 */
  {  637, 0xaa60 }, {  643, 0x8656 }, {  650, 0x3f7a }, {  661, 0x0240 },
  {  663, 0x8388 }, {  668, 0x1461 }, {  673, 0x1020 }, {  675, 0x2174 },
  {  681, 0x2021 }, {  684, 0x0702 }, {  688, 0x3000 }, {  690, 0x40bc },
  {  696, 0xa624 }, {  702, 0x4462 }, {  707, 0x60a8 }, {  712, 0x0a20 },
  /* 0x5100 */
  {  715, 0x0217 }, {  720, 0x8574 }, {  727, 0x0402 }, {  729, 0x9c84 },
  {  735, 0x7bfb }, {  748, 0x1415 }, {  753, 0x7f24 }, {  762, 0x11e2 },
  {  768, 0xb665 }, {  777, 0x02ef }, {  785, 0x1f75 }, {  795, 0x20ff },
  {  804, 0x3a70 }, {  811, 0x3840 }, {  815, 0x26c3 }, {  822, 0x6763 },
  /* 0x5200 */
  {  831, 0x4dd9 }, {  840, 0x2092 }, {  844, 0x46b0 }, {  850, 0x0fc9 },
  {  858, 0xbc98 }, {  866, 0x4850 }, {  870, 0x8638 }, {  876, 0xa03f },
  {  884, 0x2388 }, {  889, 0x8816 }, {  894, 0x3e09 }, {  901, 0x5232 },
  {  907, 0x22aa }, {  913, 0xe3a4 }, {  921, 0x00dd }, {  927, 0xc72c },
  /* 0x5300 */
  {  935, 0xa166 }, {  942, 0x26e1 }, {  949, 0x840b }, {  954, 0x8f0a },
  {  961, 0x27eb }, {  971, 0x559e }, {  980, 0xc241 }, {  985, 0x89bb },
  {  994, 0x0014 }, {  996, 0x8540 }, { 1000, 0x6361 }, { 1007, 0x0849 },
  { 1011, 0x7f0c }, { 1020, 0x8ad0 }, { 1026, 0xff3e }, { 1039, 0x05cf },
  /* 0x5400 */
  { 1047, 0xff1a }, { 1058, 0xa803 }, { 1063, 0x7a41 }, { 1070, 0x7b40 },
  { 1077, 0x4745 }, { 1084, 0x8002 }, { 1086, 0x0500 }, { 1088, 0x38eb },
  { 1097, 0xd851 }, { 1104, 0x0005 }, { 1106, 0x9934 }, { 1113, 0x710c },
  { 1119, 0x0397 }, { 1126, 0x0100 }, { 1127, 0x6366 }, { 1135, 0x2404 },
  /* 0x5500 */
  { 1138, 0x80d0 }, { 1142, 0x0051 }, { 1145, 0xc000 }, { 1147, 0x430a },
  { 1152, 0x9071 }, { 1158, 0x30c8 }, { 1163, 0x0008 }, { 1164, 0x5800 },
  { 1167, 0x0e99 }, { 1174, 0xf700 }, { 1181, 0x5f80 }, { 1188, 0x0041 },
  { 1190, 0x00b0 }, { 1193, 0x9410 }, { 1197, 0x0018 }, { 1199, 0x6280 },
  /* 0x5600 */
  { 1203, 0x0240 }, { 1205, 0x09d0 }, { 1210, 0x8200 }, { 1212, 0x0156 },
  { 1217, 0x5004 }, { 1220, 0x0801 }, { 1222, 0x1d10 }, { 1227, 0x0510 },
  { 1230, 0x84c1 }, { 1235, 0x0010 }, { 1236, 0x4025 }, { 1240, 0x1050 },
  { 1243, 0x410f }, { 1249, 0x4d8a }, { 1256, 0x4009 }, { 1259, 0xa60d },
  /* 0x5700 */
  { 1266, 0xab19 }, { 1274, 0x914c }, { 1280, 0x21c0 }, { 1284, 0x0981 },
  { 1288, 0xc485 }, { 1294, 0x0003 }, { 1296, 0x0652 }, { 1301, 0x8000 },
  { 1302, 0x0b04 }, { 1306, 0x0008 }, { 1307, 0x041d }, { 1312, 0x0009 },
  { 1314, 0x4849 }, { 1319, 0x905c }, { 1325, 0x0009 }, { 1327, 0x1690 },
  /* 0x5800 */
  { 1332, 0x0c65 }, { 1338, 0x2220 }, { 1341, 0x8412 }, { 1345, 0x2433 },
  { 1351, 0x0c03 }, { 1355, 0x4796 }, { 1363, 0x0a04 }, { 1366, 0x4225 },
  { 1371, 0x0028 }, { 1373, 0x9088 }, { 1377, 0x4900 }, { 1380, 0x4f08 },
  { 1386, 0x14a2 }, { 1391, 0xd3aa }, { 1400, 0xd830 }, { 1406, 0x3e87 },
  /* 0x5900 */
  { 1415, 0x8604 }, { 1419, 0x1f61 }, { 1427, 0x7ea4 }, { 1436, 0x4186 },
  { 1441, 0xc390 }, { 1447, 0x05b3 }, { 1454, 0x57a5 }, { 1463, 0x2118 },
  { 1467, 0x241e }, { 1473, 0x2a48 }, { 1478, 0x1128 }, { 1482, 0x4a04 },
  { 1486, 0x0a40 }, { 1489, 0x161b }, { 1496, 0x0d60 }, { 1501, 0x8840 },
  /* 0x5a00 */
  { 1504, 0x020a }, { 1507, 0x9502 }, { 1512, 0x8221 }, { 1516, 0x1060 },
  { 1519, 0x0243 }, { 1523, 0x0400 }, { 1524, 0x1444 }, { 1528, 0x8000 },
  { 1529, 0x0000 }, { 1529, 0x0c04 }, { 1532, 0x0000 }, { 1532, 0x7000 },
  { 1535, 0x1a06 }, { 1540, 0x00c1 }, { 1543, 0x024a }, { 1547, 0x0c00 },
  /* 0x5b00 */
  { 1549, 0x1a00 }, { 1552, 0x0040 }, { 1553, 0x1404 }, { 1556, 0x4045 },
  { 1560, 0x0029 }, { 1563, 0xbdb3 }, { 1574, 0x0a78 }, { 1580, 0x052b },
  { 1586, 0xbba9 }, { 1596, 0xbfa0 }, { 1605, 0x407c }, { 1611, 0x8379 },
  { 1619, 0x12fc }, { 1627, 0xe81d }, { 1635, 0x4bf6 }, { 1645, 0xc569 },
  /* 0x5c00 */
  { 1653, 0xeff6 }, { 1666, 0x044a }, { 1670, 0x2115 }, { 1675, 0xff02 },
  { 1684, 0xed63 }, { 1694, 0x402b }, { 1699, 0xd033 }, { 1706, 0x0242 },
  { 1709, 0x1000 }, { 1710, 0x0013 }, { 1713, 0x1b02 }, { 1718, 0x59ca },
  { 1726, 0x00a0 }, { 1728, 0x0200 }, { 1729, 0xa703 }, { 1736, 0x2c41 },
  /* 0x5d00 */
  { 1741, 0x4880 }, { 1744, 0x8ff2 }, { 1754, 0x0204 }, { 1756, 0x0000 },
  { 1756, 0x5800 }, { 1759, 0x1005 }, { 1762, 0x9200 }, { 1765, 0x0048 },
  { 1767, 0x1894 }, { 1772, 0x2001 }, { 1774, 0x5004 }, { 1777, 0x3480 },
  { 1781, 0x3200 }, { 1784, 0x684c }, { 1790, 0x49ea }, { 1798, 0x68be },
  /* 0x5e00 */
  { 1807, 0x184c }, { 1812, 0x2e42 }, { 1818, 0xa820 }, { 1822, 0x21c9 },
  { 1828, 0x50b9 }, { 1835, 0x80b0 }, { 1839, 0x001e }, { 1843, 0xff7c },
  { 1856, 0x849a }, { 1862, 0x14e0 }, { 1867, 0x28c1 }, { 1872, 0x01e0 },
  { 1876, 0x870e }, { 1883, 0xac49 }, { 1890, 0x130f }, { 1897, 0xdddb },
  /* 0x5f00 */
  { 1909, 0xbe1a }, { 1918, 0x89fb }, { 1928, 0xa2e0 }, { 1934, 0x51a2 },
  { 1940, 0x5502 }, { 1945, 0x32ca }, { 1952, 0x3e46 }, { 1960, 0x928b },
  { 1967, 0x1dbf }, { 1978, 0x438f }, { 1986, 0x6703 }, { 1993, 0x3218 },
  { 1998, 0x3028 }, { 2002, 0x33c0 }, { 2008, 0x0811 }, { 2011, 0xa923 },
  /* 0x6000 */
  { 2018, 0xc000 }, { 2020, 0x3a65 }, { 2028, 0x8fe3 }, { 2038, 0x0402 },
  { 2040, 0x2c4e }, { 2047, 0x8625 }, { 2053, 0xbf3d }, { 2065, 0x00a1 },
  { 2068, 0x3a1a }, { 2075, 0x8cd4 }, { 2082, 0x06c9 }, { 2088, 0x317c },
  { 2096, 0x00e0 }, { 2099, 0x950a }, { 2105, 0x018b }, { 2110, 0x0edb },
  /* 0x6100 */
  { 2119, 0xe34b }, { 2128, 0x8c20 }, { 2132, 0x1182 }, { 2136, 0xf010 },
  { 2141, 0x7d94 }, { 2150, 0xa728 }, { 2157, 0xc9ac }, { 2165, 0x40fb },
  { 2173, 0x4484 }, { 2177, 0x0653 }, { 2183, 0x5a90 }, { 2189, 0x4444 },
  { 2193, 0x3fc8 }, { 2202, 0x0001 }, { 2203, 0x0048 }, { 2205, 0xf5d4 },
  /* 0x6200 */
  { 2215, 0x7701 }, { 2222, 0xec57 }, { 2232, 0xc442 }, { 2237, 0x891d },
  { 2244, 0x6b83 }, { 2252, 0x4928 }, { 2257, 0x4109 }, { 2261, 0xd242 },
  { 2267, 0x061d }, { 2273, 0x59fe }, { 2284, 0x1800 }, { 2286, 0x3a22 },
  { 2292, 0xb7e4 }, { 2302, 0x3b9f }, { 2313, 0xf003 }, { 2319, 0xc0ea },
  /* 0x6300 */
  { 2326, 0x1386 }, { 2332, 0x8202 }, { 2335, 0x8980 }, { 2339, 0xe400 },
  { 2343, 0xb200 }, { 2347, 0x10a1 }, { 2351, 0x4b80 }, { 2356, 0x0cc4 },
  { 2361, 0xd309 }, { 2368, 0x8944 }, { 2373, 0x1faf }, { 2384, 0x4834 },
  { 2389, 0x8259 }, { 2395, 0x0c45 }, { 2400, 0x420a }, { 2404, 0x0450 },
  /* 0x6400 */
  { 2407, 0xa040 }, { 2410, 0x10c8 }, { 2414, 0x3140 }, { 2418, 0x4450 },
  { 2422, 0x4004 }, { 2424, 0x0100 }, { 2425, 0x8280 }, { 2428, 0x0540 },
  { 2431, 0x0108 }, { 2433, 0x442c }, { 2438, 0x6a30 }, { 2444, 0x1a05 },
  { 2449, 0x20a6 }, { 2454, 0x0514 }, { 2458, 0x90cf }, { 2466, 0x6456 },
  /* 0x6500 */
  { 2473, 0x0021 }, { 2475, 0x3100 }, { 2478, 0x9c18 }, { 2484, 0xcbf0 },
  { 2493, 0xa120 }, { 2497, 0x63e2 }, { 2505, 0x104c }, { 2509, 0x01b5 },
  { 2515, 0x538c }, { 2522, 0x9a83 }, { 2529, 0xb8b2 }, { 2537, 0x3281 },
  { 2542, 0x987a }, { 2550, 0x0a84 }, { 2554, 0x33e7 }, { 2564, 0x0c02 },
  /* 0x6600 */
  { 2567, 0xd4cc }, { 2575, 0x9018 }, { 2579, 0xa1a1 }, { 2585, 0x9070 },
  { 2590, 0x8a1e }, { 2597, 0xe004 }, { 2601, 0xc3d4 }, { 2609, 0x0451 },
  { 2613, 0x439a }, { 2620, 0x21c2 }, { 2625, 0x4844 }, { 2629, 0x5310 },
  { 2634, 0x0292 }, { 2638, 0x3640 }, { 2643, 0x0241 }, { 2646, 0xf3bd },
  /* 0x6700 */
  { 2658, 0xab09 }, { 2665, 0xe8f0 }, { 2673, 0x7dc0 }, { 2681, 0xa5d2 },
  { 2689, 0xc242 }, { 2694, 0xd24b }, { 2702, 0xa43f }, { 2711, 0xd0af },
  { 2720, 0x1aa0 }, { 2725, 0x34a1 }, { 2731, 0x8247 }, { 2737, 0x03d8 },
  { 2743, 0xc452 }, { 2749, 0x651b }, { 2757, 0xd294 }, { 2764, 0xc83a },
  /* 0x6800 */
  { 2771, 0x001c }, { 2774, 0x40c8 }, { 2778, 0x0e06 }, { 2783, 0x3314 },
  { 2789, 0x614f }, { 2797, 0xb21b }, { 2805, 0x0088 }, { 2807, 0xc0d0 },
  { 2812, 0xa02a }, { 2817, 0xa898 }, { 2823, 0xa1c5 }, { 2830, 0x166b },
  { 2838, 0x2e50 }, { 2844, 0x85b4 }, { 2851, 0xc08b }, { 2857, 0x0604 },
  /* 0x6900 */
  { 2860, 0xf933 }, { 2870, 0x1e04 }, { 2875, 0x056e }, { 2882, 0xa251 },
  { 2888, 0x0400 }, { 2889, 0x7638 }, { 2897, 0xec07 }, { 2905, 0x73b8 },
  { 2914, 0x4406 }, { 2918, 0x1832 }, { 2923, 0x4081 }, { 2926, 0xc816 },
  { 2932, 0x7c8a }, { 2940, 0x6309 }, { 2946, 0x2980 }, { 2950, 0xaa04 },
  /* 0x6a00 */
  { 2955, 0x1c24 }, { 2960, 0xca9c }, { 2968, 0x4e0e }, { 2975, 0x2760 },
  { 2981, 0x0990 }, { 2985, 0x8300 }, { 2988, 0x0046 }, { 2991, 0x8104 },
  { 2994, 0x6011 }, { 2998, 0x1081 }, { 3001, 0x540d }, { 3007, 0x0908 },
  { 3010, 0x000e }, { 3013, 0xcc0a }, { 3019, 0x0500 }, { 3021, 0x0c00 },
  /* 0x6b00 */
  { 3023, 0x0430 }, { 3026, 0xa044 }, { 3030, 0x008b }, { 3034, 0x6784 },
  { 3041, 0x5288 }, { 3046, 0x8a19 }, { 3052, 0x865e }, { 3060, 0x8b18 },
  { 3066, 0x2e59 }, { 3074, 0x4160 }, { 3078, 0x8c10 }, { 3082, 0x9cbe },
  { 3092, 0x6861 }, { 3098, 0x891c }, { 3104, 0x9800 }, { 3107, 0x0008 },
  /* 0x6c00 */
  { 3108, 0x8100 }, { 3110, 0x089a }, { 3115, 0x0018 }, { 3117, 0x4190 },
  { 3121, 0x4007 }, { 3125, 0xe4a1 }, { 3132, 0x0505 }, { 3136, 0x640d },
  { 3142, 0x310e }, { 3148, 0x0e4d }, { 3155, 0x4806 }, { 3159, 0xff0a },
  { 3169, 0x1632 }, { 3175, 0x2aa8 }, { 3181, 0x852e }, { 3188, 0x000b },
  /* 0x6d00 */
  { 3191, 0x1800 }, { 3193, 0xca84 }, { 3199, 0x0e20 }, { 3203, 0x696c },
  { 3211, 0x0032 }, { 3214, 0x1600 }, { 3217, 0x5658 }, { 3224, 0x0390 },
  { 3228, 0x5120 }, { 3232, 0x1a28 }, { 3237, 0x8000 }, { 3238, 0x1124 },
  { 3242, 0x18e1 }, { 3248, 0x4326 }, { 3254, 0x5d52 }, { 3262, 0x0eaa },
  /* 0x6e00 */
  { 3269, 0x0fa0 }, { 3275, 0xae28 }, { 3282, 0xfa7b }, { 3294, 0x4500 },
  { 3297, 0x6408 }, { 3301, 0x8940 }, { 3305, 0xc880 }, { 3309, 0xc044 },
  { 3313, 0x9005 }, { 3317, 0xb141 }, { 3323, 0x8424 }, { 3327, 0x24c4 },
  { 3332, 0x1a34 }, { 3338, 0x603a }, { 3344, 0x9000 }, { 3346, 0xc194 },
  /* 0x6f00 */
  { 3352, 0x8246 }, { 3357, 0x003a }, { 3361, 0x180d }, { 3366, 0xc106 },
  { 3371, 0x0022 }, { 3373, 0x9910 }, { 3378, 0xe050 }, { 3383, 0x1511 },
  { 3388, 0x4057 }, { 3394, 0x0082 }, { 3396, 0x041a }, { 3400, 0x020a },
  { 3403, 0x004f }, { 3408, 0x8930 }, { 3413, 0xd813 }, { 3420, 0x444a },
  /* 0x7000 */
  { 3425, 0x8a02 }, { 3429, 0xed22 }, { 3437, 0x10c0 }, { 3440, 0x4005 },
  { 3443, 0x1000 }, { 3444, 0x0102 }, { 3446, 0x8808 }, { 3449, 0x3101 },
  { 3453, 0x4600 }, { 3456, 0x0204 }, { 3458, 0xf000 }, { 3462, 0x0708 },
  { 3466, 0x8900 }, { 3469, 0xa200 }, { 3472, 0x0000 }, { 3472, 0x2202 },
  /* 0x7100 */
  { 3475, 0x0200 }, { 3476, 0x1610 }, { 3480, 0x0042 }, { 3482, 0x1040 },
  { 3484, 0x5200 }, { 3487, 0x0260 }, { 3490, 0x52f4 }, { 3498, 0x2000 },
  { 3499, 0x8510 }, { 3503, 0x8230 }, { 3507, 0x1100 }, { 3509, 0x4202 },
  { 3512, 0x4308 }, { 3516, 0x80b5 }, { 3522, 0x70e1 }, { 3529, 0x9a20 },
  /* 0x7200 */
  { 3534, 0x2040 }, { 3536, 0x0801 }, { 3538, 0x3500 }, { 3542, 0xfc65 },
  { 3552, 0x19c1 }, { 3558, 0xab04 }, { 3564, 0x0286 }, { 3568, 0x6214 },
  { 3573, 0x0087 }, { 3577, 0x0044 }, { 3579, 0x9085 }, { 3584, 0x0244 },
  { 3587, 0x405c }, { 3592, 0x0a85 }, { 3597, 0x3207 }, { 3603, 0x3380 },
  /* 0x7300 */
  { 3608, 0x0400 }, { 3609, 0xb8c0 }, { 3615, 0xce20 }, { 3621, 0xc0d0 },
  { 3626, 0xc030 }, { 3630, 0x0080 }, { 3631, 0x0508 }, { 3634, 0x0d25 },
  { 3640, 0x0a90 }, { 3644, 0x0040 }, { 3645, 0x0200 }, { 3646, 0x080c },
  { 3649, 0x6505 }, { 3655, 0x4000 }, { 3656, 0x6421 }, { 3661, 0x4102 },
  /* 0x7400 */
  { 3664, 0x0268 }, { 3668, 0x0000 }, { 3668, 0x0024 }, { 3670, 0x847c },
  { 3677, 0x0002 }, { 3678, 0xde20 }, { 3685, 0x8619 }, { 3691, 0x4049 },
  { 3695, 0x0808 }, { 3697, 0x4000 }, { 3698, 0x0084 }, { 3700, 0x2001 },
  { 3702, 0x8400 }, { 3704, 0x1010 }, { 3706, 0x42cd }, { 3713, 0x01c7 },
  /* 0x7500 */
  { 3719, 0x7038 }, { 3725, 0xd52a }, { 3733, 0x1968 }, { 3739, 0x1d8f },
  { 3748, 0xbe50 }, { 3756, 0x3e12 }, { 3763, 0x2ef5 }, { 3773, 0x81d9 },
  { 3780, 0xcec4 }, { 3788, 0x2412 }, { 3792, 0x0828 }, { 3795, 0x732e },
  { 3804, 0x24ac }, { 3810, 0x4b34 }, { 3817, 0x020c }, { 3820, 0xd41d },
  /* 0x7600 */
  { 3828, 0x2a02 }, { 3832, 0x8000 }, { 3833, 0x0097 }, { 3838, 0x0811 },
  { 3841, 0x11c4 }, { 3846, 0x1144 }, { 3850, 0x1786 }, { 3857, 0x7d45 },
  { 3866, 0x49d9 }, { 3874, 0x0649 }, { 3879, 0x4000 }, { 3880, 0x8791 },
  { 3887, 0x254c }, { 3893, 0xd8c4 }, { 3900, 0x44ba }, { 3907, 0x4914 },
  /* 0x7700 */
  { 3912, 0x1b92 }, { 3919, 0xc800 }, { 3922, 0x0271 }, { 3927, 0x1580 },
  { 3931, 0x0081 }, { 3933, 0x0c00 }, { 3935, 0x096a }, { 3941, 0xc200 },
  { 3944, 0x4800 }, { 3946, 0x4002 }, { 3948, 0x3021 }, { 3952, 0xba49 },
  { 3960, 0x2080 }, { 3962, 0x1c80 }, { 3966, 0xe2ac }, { 3974, 0x1008 },
  /* 0x7800 */
  { 3976, 0x1004 }, { 3978, 0x0034 }, { 3981, 0x00e1 }, { 3985, 0x8414 },
  { 3989, 0x0020 }, { 3990, 0x2000 }, { 3991, 0x9800 }, { 3994, 0x1014 },
  { 3997, 0x70c2 }, { 4003, 0x04aa }, { 4008, 0x8688 }, { 4013, 0x5420 },
  { 4017, 0x0c62 }, { 4022, 0x0413 }, { 4026, 0x9180 }, { 4030, 0x2010 },
  /* 0x7900 */
  { 4032, 0x4082 }, { 4035, 0x0206 }, { 4038, 0x1c40 }, { 4042, 0x5400 },
  { 4045, 0x0383 }, { 4050, 0xe4e9 }, { 4059, 0x2125 }, { 4064, 0x8480 },
  { 4067, 0xe433 }, { 4075, 0x2000 }, { 4076, 0x44c0 }, { 4080, 0xe609 },
  { 4087, 0x0a03 }, { 4091, 0x8126 }, { 4096, 0x12da }, { 4103, 0x0801 },
  /* 0x7a00 */
  { 4105, 0x6901 }, { 4110, 0x9790 }, { 4117, 0x4001 }, { 4119, 0xf886 },
  { 4127, 0xe24d }, { 4135, 0x0081 }, { 4137, 0x0a0e }, { 4142, 0xa651 },
  { 4149, 0x011a }, { 4153, 0x81ec }, { 4160, 0xc600 }, { 4164, 0x8441 },
  { 4168, 0xadb8 }, { 4177, 0xb62c }, { 4185, 0xa46f }, { 4194, 0x8741 },
  /* 0x7b00 */
  { 4200, 0x8d54 }, { 4207, 0x4b02 }, { 4212, 0x1161 }, { 4217, 0x0268 },
  { 4221, 0xbb60 }, { 4229, 0x2057 }, { 4235, 0x50a0 }, { 4239, 0x0433 },
  { 4244, 0xa8c0 }, { 4249, 0xb7b4 }, { 4259, 0x2402 }, { 4262, 0x0112 },
  { 4265, 0x9ad3 }, { 4274, 0x2000 }, { 4275, 0x2271 }, { 4281, 0x00c8 },
  /* 0x7c00 */
  { 4284, 0x2081 }, { 4287, 0x809e }, { 4293, 0x0c8a }, { 4298, 0xe180 },
  { 4303, 0xb009 }, { 4308, 0x8151 }, { 4313, 0x1031 }, { 4317, 0x4028 },
  { 4320, 0x2a0e }, { 4326, 0x89a5 }, { 4333, 0x69b6 }, { 4342, 0x620e },
  { 4348, 0x4425 }, { 4353, 0xd144 }, { 4359, 0x8085 }, { 4363, 0x4d54 },
  /* 0x7d00 */
  { 4370, 0x2c75 }, { 4378, 0x1fb1 }, { 4387, 0xd807 }, { 4394, 0x862d },
  { 4401, 0xd87c }, { 4410, 0x4841 }, { 4414, 0x414e }, { 4420, 0x226e },
  { 4427, 0x8200 }, { 4429, 0x9e08 }, { 4435, 0xf80c }, { 4442, 0xed37 },
  { 4453, 0x8c80 }, { 4457, 0x7526 }, { 4465, 0x9313 }, { 4472, 0x0814 },
  /* 0x7e00 */
  { 4475, 0x0e32 }, { 4481, 0xc804 }, { 4485, 0x484e }, { 4491, 0x6ea6 },
  { 4500, 0x2c4a }, { 4506, 0x6670 }, { 4513, 0x26c0 }, { 4518, 0xba01 },
  { 4524, 0xd30c }, { 4531, 0x185d }, { 4538, 0x0000 }, { 4538, 0x0000 },
  { 4538, 0x0000 }, { 4538, 0x0000 }, { 4538, 0x0000 }, { 4538, 0x0000 },
  /* 0x7f00 */
  { 4538, 0x0000 }, { 4538, 0x0000 }, { 4538, 0x0000 }, { 4538, 0x0540 },
  { 4541, 0x7020 }, { 4545, 0x8133 }, { 4551, 0x4f81 }, { 4558, 0x03a5 },
  { 4564, 0x55ec }, { 4573, 0x6410 }, { 4577, 0xc318 }, { 4583, 0x2344 },
  { 4588, 0x1462 }, { 4593, 0x0034 }, { 4596, 0x0a43 }, { 4601, 0x1a09 },
  /* 0x8000 */
  { 4606, 0x187b }, { 4614, 0x13a5 }, { 4621, 0x0102 }, { 4623, 0xa848 },
  { 4628, 0x0440 }, { 4630, 0xc544 }, { 4636, 0x8106 }, { 4640, 0xe2dd },
  { 4650, 0x1af0 }, { 4657, 0x2d48 }, { 4663, 0xb626 }, { 4671, 0x0416 },
  { 4675, 0x5058 }, { 4680, 0x6e40 }, { 4686, 0x8032 }, { 4690, 0x3112 },
  /* 0x8100 */
  { 4695, 0x07e4 }, { 4702, 0x0c00 }, { 4704, 0x8208 }, { 4707, 0x420a },
  { 4711, 0x4840 }, { 4714, 0x803b }, { 4720, 0x4860 }, { 4724, 0x8713 },
  { 4731, 0x850d }, { 4737, 0x3428 }, { 4742, 0x0319 }, { 4747, 0xe529 },
  { 4755, 0x2345 }, { 4761, 0x870a }, { 4767, 0x25a9 }, { 4774, 0x5c18 },
  /* 0x8200 */
  { 4780, 0x77a6 }, { 4790, 0xd9c5 }, { 4799, 0x5e00 }, { 4804, 0x03e8 },
  { 4810, 0x0081 }, { 4812, 0xa700 }, { 4817, 0xcd54 }, { 4825, 0x41c6 },
  { 4831, 0x2800 }, { 4833, 0xa204 }, { 4837, 0xb860 }, { 4843, 0x2b0a },
  { 4849, 0x0020 }, { 4850, 0xda9e }, { 4860, 0x08ea }, { 4866, 0x0e1a },
  /* 0x8300 */
  { 4872, 0x427c }, { 4879, 0x11c0 }, { 4883, 0x8908 }, { 4887, 0x0376 },
  { 4894, 0x8621 }, { 4899, 0x0105 }, { 4902, 0x0000 }, { 4902, 0x18a8 },
  { 4907, 0x46a0 }, { 4912, 0xc448 }, { 4917, 0x0d05 }, { 4922, 0x2022 },
  { 4925, 0x5422 }, { 4930, 0x9148 }, { 4935, 0x8a01 }, { 4939, 0x2897 },
  /* 0x8400 */
  { 4946, 0x7898 }, { 4953, 0x0008 }, { 4954, 0x1605 }, { 4959, 0x3122 },
  { 4964, 0x4240 }, { 4967, 0x0880 }, { 4969, 0xfa4e }, { 4979, 0x06a2 },
  { 4984, 0x0814 }, { 4987, 0x9211 }, { 4992, 0x2002 }, { 4994, 0x9b04 },
  { 5000, 0x2e52 }, { 5007, 0x0643 }, { 5012, 0x5000 }, { 5014, 0x9010 },
  /* 0x8500 */
  { 5017, 0x0041 }, { 5019, 0x85ba }, { 5027, 0x3042 }, { 5031, 0x2020 },
  { 5033, 0x4f0b }, { 5041, 0x05a0 }, { 5045, 0x2708 }, { 5050, 0x4080 },
  { 5052, 0x0591 }, { 5057, 0x1a93 }, { 5064, 0xdf50 }, { 5073, 0x0600 },
  { 5075, 0xa202 }, { 5079, 0x3021 }, { 5083, 0x0630 }, { 5087, 0x4e80 },
  /* 0x8600 */
  { 5092, 0x0cc4 }, { 5097, 0x04c8 }, { 5101, 0xa004 }, { 5104, 0x8001 },
  { 5106, 0x6000 }, { 5108, 0xd431 }, { 5115, 0x0880 }, { 5117, 0x0a02 },
  { 5120, 0x1c00 }, { 5123, 0x0028 }, { 5125, 0x8e18 }, { 5131, 0x0041 },
  { 5133, 0x6ad0 }, { 5140, 0xca10 }, { 5145, 0xf210 }, { 5151, 0x4b00 },
  /* 0x8700 */
  { 5155, 0x274d }, { 5163, 0x1506 }, { 5168, 0x0220 }, { 5170, 0x8890 },
  { 5174, 0x5a00 }, { 5178, 0x82a8 }, { 5183, 0x4549 }, { 5189, 0x8150 },
  { 5193, 0x2004 }, { 5195, 0x8000 }, { 5196, 0x8804 }, { 5199, 0x2c08 },
  { 5203, 0x08d1 }, { 5208, 0x0005 }, { 5210, 0x8001 }, { 5212, 0x4ac4 },
  /* 0x8800 */
  { 5218, 0xe020 }, { 5222, 0x0062 }, { 5225, 0x008e }, { 5229, 0x0a42 },
  { 5233, 0x3055 }, { 5239, 0x6a8c }, { 5246, 0x090e }, { 5251, 0xe0a5 },
  { 5258, 0x2906 }, { 5263, 0x42c4 }, { 5268, 0x4814 }, { 5272, 0x80b3 },
  { 5278, 0x803e }, { 5284, 0xb330 }, { 5291, 0x0102 }, { 5293, 0x731c },
  /* 0x8900 */
  { 5301, 0x1494 }, { 5306, 0x600d }, { 5311, 0x0c20 }, { 5314, 0x0940 },
  { 5317, 0x301a }, { 5322, 0xc040 }, { 5325, 0xa451 }, { 5331, 0xc094 },
  { 5336, 0x8dca }, { 5344, 0x05c8 }, { 5349, 0x96c2 }, { 5356, 0xa40c },
  { 5361, 0x0001 }, { 5362, 0x3404 }, { 5366, 0x00c8 }, { 5369, 0x0110 },
  /* 0x8a00 */
  { 5371, 0x550d }, { 5378, 0xa9c9 }, { 5386, 0x2428 }, { 5390, 0x1c5a },
  { 5397, 0x0142 }, { 5400, 0x4837 }, { 5407, 0x7a4d }, { 5416, 0x100f },
  { 5421, 0x32b4 }, { 5428, 0x452a }, { 5434, 0x317b }, { 5443, 0x9205 },
  { 5448, 0xb894 }, { 5455, 0x5c44 }, { 5461, 0x68d7 }, { 5470, 0x458a },
  /* 0x8b00 */
  { 5476, 0x5097 }, { 5483, 0x2ed1 }, { 5491, 0x1943 }, { 5497, 0x4208 },
  { 5500, 0xd202 }, { 5505, 0x9d40 }, { 5511, 0x9840 }, { 5515, 0x2097 },
  { 5521, 0x5409 }, { 5526, 0x064d }, { 5532, 0x0000 }, { 5532, 0x0000 },
  { 5532, 0x0000 }, { 5532, 0x0000 }, { 5532, 0x0000 }, { 5532, 0x0000 },
  /* 0x8c00 */
  { 5532, 0x0000 }, { 5532, 0x0000 }, { 5532, 0x0000 }, { 5532, 0x8480 },
  { 5535, 0x5542 }, { 5541, 0x0421 }, { 5544, 0x1c06 }, { 5549, 0x1700 },
  { 5553, 0x7624 }, { 5560, 0x6110 }, { 5564, 0xff87 }, { 5576, 0xb9dd },
  { 5587, 0x659f }, { 5597, 0x5c0a }, { 5603, 0x245d }, { 5610, 0x3c00 },
  /* 0x8d00 */
  { 5614, 0xadb0 }, { 5622, 0x0059 }, { 5626, 0x0000 }, { 5626, 0x0000 },
  { 5626, 0x0000 }, { 5626, 0x0000 }, { 5626, 0x28d0 }, { 5631, 0x009b },
  { 5636, 0x0422 }, { 5639, 0x0200 }, { 5640, 0x0108 }, { 5642, 0x4408 },
  { 5645, 0x9804 }, { 5649, 0xac40 }, { 5654, 0x8d0a }, { 5660, 0x9028 },
  /* 0x8e00 */
  { 5664, 0x8700 }, { 5668, 0xe001 }, { 5672, 0x0400 }, { 5673, 0x0031 },
  { 5676, 0x1794 }, { 5683, 0x8221 }, { 5687, 0x0019 }, { 5690, 0x1054 },
  { 5694, 0x2cb2 }, { 5701, 0x021a }, { 5705, 0x9c02 }, { 5710, 0x4003 },
  { 5713, 0x3d60 }, { 5720, 0x8804 }, { 5723, 0x080c }, { 5726, 0x7900 },
  /* 0x8f00 */
  { 5731, 0x1628 }, { 5736, 0xba3c }, { 5745, 0x8640 }, { 5749, 0xcb08 },
  { 5755, 0x7274 }, { 5763, 0x9080 }, { 5766, 0x001e }, { 5770, 0x0000 },
  { 5770, 0x0000 }, { 5770, 0xd800 }, { 5774, 0xe188 }, { 5780, 0x9c87 },
  { 5788, 0x4034 }, { 5792, 0x0412 }, { 5795, 0xae64 }, { 5803, 0x2791 },
  /* 0x9000 */
  { 5810, 0xe86b }, { 5819, 0xe6fb }, { 5831, 0x408f }, { 5837, 0x5366 },
  { 5845, 0xeea6 }, { 5855, 0x537f }, { 5866, 0xe32b }, { 5875, 0xb5e4 },
  { 5884, 0x869f }, { 5893, 0x0002 }, { 5894, 0x8548 }, { 5899, 0x0122 },
  { 5902, 0x4402 }, { 5905, 0x0800 }, { 5906, 0x2116 }, { 5911, 0x20a0 },
  /* 0x9100 */
  { 5914, 0x0004 }, { 5915, 0x0204 }, { 5917, 0x2000 }, { 5918, 0x0005 },
  { 5920, 0x7e00 }, { 5926, 0x0154 }, { 5930, 0x162c }, { 5936, 0x01ac },
  { 5941, 0x2a84 }, { 5946, 0x1085 }, { 5950, 0x8c14 }, { 5955, 0x0530 },
  { 5959, 0xfbc3 }, { 5970, 0xb943 }, { 5978, 0x00ca }, { 5982, 0x9060 },
  /* 0x9200 */
  { 5986, 0x6000 }, { 5988, 0x4032 }, { 5992, 0x1200 }, { 5994, 0x8090 },
  { 5997, 0x0b30 }, { 6002, 0x4c81 }, { 6007, 0x0054 }, { 6010, 0x4002 },
  { 6012, 0x0029 }, { 6015, 0x1d6a }, { 6023, 0x2000 }, { 6024, 0x0280 },
  { 6026, 0x8000 }, { 6027, 0x0004 }, { 6028, 0x2610 }, { 6032, 0x150c },
  /* 0x9300 */
  { 6037, 0x8040 }, { 6039, 0x0701 }, { 6043, 0xd94d }, { 6052, 0x0c24 },
  { 6056, 0x2810 }, { 6059, 0x1850 }, { 6063, 0x5001 }, { 6066, 0x5020 },
  { 6069, 0x1000 }, { 6070, 0x04d0 }, { 6074, 0x7080 }, { 6078, 0x0201 },
  { 6080, 0x0108 }, { 6082, 0x21c3 }, { 6088, 0x0132 }, { 6092, 0x0000 },
  /* 0x9400 */
  { 6092, 0x0088 }, { 6094, 0x0719 }, { 6100, 0x0802 }, { 6102, 0x0560 },
  { 6106, 0x0012 }, { 6108, 0x4c0e }, { 6114, 0x0405 }, { 6117, 0xf0a1 },
  { 6124, 0x0002 }, { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0000 },
  { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0000 },
  /* 0x9500 */
  { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0000 },
  { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0000 }, { 6125, 0x0080 },
  { 6126, 0x8e8d }, { 6134, 0x035a }, { 6140, 0x21bd }, { 6148, 0x5a04 },
  { 6153, 0x3488 }, { 6158, 0x1170 }, { 6163, 0x0026 }, { 6166, 0x0000 },
  /* 0x9600 */
  { 6166, 0x0000 }, { 6166, 0x1000 }, { 6167, 0xc502 }, { 6172, 0x8804 },
  { 6175, 0xb815 }, { 6182, 0xf801 }, { 6188, 0x147c }, { 6195, 0x25ed },
  { 6204, 0xed60 }, { 6212, 0x1bb0 }, { 6219, 0x0589 }, { 6224, 0x1bd7 },
  { 6234, 0x7af3 }, { 6245, 0x1a62 }, { 6251, 0x0d0c }, { 6256, 0x0ac5 },
  /* 0x9700 */
  { 6262, 0xe5d1 }, { 6271, 0x524a }, { 6277, 0x0490 }, { 6280, 0x6305 },
  { 6286, 0x0354 }, { 6291, 0x5244 }, { 6296, 0x2b57 }, { 6305, 0x1612 },
  { 6310, 0xa872 }, { 6317, 0x1101 }, { 6320, 0x2949 }, { 6326, 0x0018 },
  { 6328, 0x0948 }, { 6332, 0x1008 }, { 6334, 0x6000 }, { 6336, 0x886c },
  /* 0x9800 */
  { 6342, 0x916e }, { 6350, 0x058f }, { 6357, 0x3012 }, { 6361, 0x3990 },
  { 6367, 0xf840 }, { 6373, 0x4930 }, { 6378, 0x8880 }, { 6381, 0x001b },
  { 6385, 0x0000 }, { 6385, 0x0000 }, { 6385, 0x8500 }, { 6388, 0x0042 },
  { 6390, 0x0058 }, { 6393, 0x9800 }, { 6396, 0xea04 }, { 6402, 0x7014 },
  /* 0x9900 */
  { 6407, 0x1628 }, { 6412, 0x611d }, { 6419, 0x5113 }, { 6425, 0x6000 },
  { 6427, 0x1a24 }, { 6432, 0x00a7 }, { 6437, 0x0000 }, { 6437, 0x0000 },
  { 6437, 0x0000 }, { 6437, 0x03c0 }, { 6441, 0x7120 }, { 6446, 0x1018 },
  { 6449, 0x0172 }, { 6454, 0xa927 }, { 6462, 0x6004 }, { 6465, 0x8906 },
  /* 0x9a00 */
  { 6470, 0xc022 }, { 6474, 0x020c }, { 6477, 0x0900 }, { 6479, 0x4081 },
  { 6482, 0x202d }, { 6487, 0x8ca0 }, { 6492, 0x0e34 }, { 6498, 0x0000 },
  { 6498, 0x0000 }, { 6498, 0x0000 }, { 6498, 0x2100 }, { 6500, 0x1101 },
  { 6503, 0x8011 }, { 6506, 0xc11a }, { 6512, 0xec4c }, { 6520, 0x0892 },
  /* 0x9b00 */
  { 6524, 0x0040 }, { 6525, 0x8500 }, { 6528, 0xc7ac }, { 6537, 0x1806 },
  { 6541, 0xe03e }, { 6549, 0x0512 }, { 6553, 0x8000 }, { 6554, 0x0010 },
  { 6555, 0x4008 }, { 6557, 0x80ce }, { 6563, 0x6d01 }, { 6569, 0x0210 },
  { 6571, 0x8641 }, { 6576, 0x0856 }, { 6581, 0x011e }, { 6586, 0x0027 },
  /* 0x9c00 */
  { 6590, 0x3750 }, { 6597, 0x083d }, { 6603, 0xe032 }, { 6609, 0x4e05 },
  { 6615, 0x01c0 }, { 6618, 0x0484 }, { 6621, 0x0081 }, { 6623, 0x0140 },
  { 6625, 0x0000 }, { 6625, 0x0000 }, { 6625, 0x0000 }, { 6625, 0x0000 },
  { 6625, 0x0000 }, { 6625, 0x0000 }, { 6625, 0x1aa0 }, { 6630, 0x0059 },
  /* 0x9d00 */
  { 6634, 0x43c8 }, { 6640, 0x8824 }, { 6644, 0x1d48 }, { 6650, 0xc800 },
  { 6653, 0x0152 }, { 6657, 0x7203 }, { 6663, 0x9013 }, { 6668, 0x0404 },
  { 6670, 0x8280 }, { 6673, 0x0400 }, { 6674, 0x8a10 }, { 6678, 0x0d14 },
  { 6683, 0x8056 }, { 6688, 0x0208 }, { 6690, 0xa040 }, { 6693, 0x2704 },
  /* 0x9e00 */
  { 6698, 0x0000 }, { 6698, 0x4c00 }, { 6701, 0x0000 }, { 6701, 0x0000 },
  { 6701, 0x0000 }, { 6701, 0x0000 }, { 6701, 0x0000 }, { 6701, 0xa320 },
  { 6706, 0x1902 }, { 6710, 0xa0ae }, { 6717, 0x2660 }, { 6722, 0xdf00 },
  { 6729, 0xf010 }, { 6734, 0x7b15 }, { 6743, 0x8121 }, { 6747, 0x3ad0 },
  /* 0x9f00 */
  { 6754, 0x4180 }, { 6757, 0x0028 }, { 6759, 0x1003 }, { 6762, 0x4800 },
  { 6764, 0xcc00 }, { 6768, 0x8014 }, { 6771, 0x14cf }, { 6779, 0x00c4 },
  { 6782, 0x2000 }, { 6783, 0x3020 }, { 6786, 0x0001 },
};

static const Summary16 jisx0208_uni2indx_pageff[15] = {
  /* 0xff00 */
  { 6787, 0xdf7a }, { 6799, 0xffff }, { 6815, 0xffff }, { 6831, 0xffff },
  { 6847, 0xffff }, { 6863, 0x3fff }, { 6877, 0x0000 }, { 6877, 0x0000 },
  { 6877, 0x0000 }, { 6877, 0x0000 }, { 6877, 0x0000 }, { 6877, 0x0000 },
  { 6877, 0x0000 }, { 6877, 0x0000 }, { 6877, 0x0028 },
};

static int jisx0208_wctomb(unsigned int *r, const unsigned int wc) {
    const Summary16 *summary = NULL;
    if (wc < 0x0100) {
        summary = &jisx0208_uni2indx_page00[(wc>>4)];
    } else if (wc >= 0x0300 && wc < 0x0460) {
        summary = &jisx0208_uni2indx_page03[(wc>>4)-0x030];
    } else if (wc >= 0x2000 && wc < 0x2320) {
        summary = &jisx0208_uni2indx_page20[(wc>>4)-0x200];
    } else if (wc >= 0x2500 && wc < 0x2670) {
        summary = &jisx0208_uni2indx_page25[(wc>>4)-0x250];
    } else if (wc >= 0x3000 && wc < 0x3100) {
        summary = &jisx0208_uni2indx_page30[(wc>>4)-0x300];
    } else if (wc >= 0x4e00 && wc < 0x9fb0) {
        summary = &jisx0208_uni2indx_page4e[(wc>>4)-0x4e0];
    } else if (wc >= 0xff00 && wc < 0xfff0) {
        summary = &jisx0208_uni2indx_pageff[(wc>>4)-0xff0];
    }
    if (summary) {
        unsigned short used = summary->used;
        unsigned int i = wc & 0x0f;
        if (used & ((unsigned short) 1 << i)) {
            /* Keep in 'used' only the bits 0..i-1. */
            used &= ((unsigned short) 1 << i) - 1;
            /* Add 'summary->indx' and the number of bits set in 'used'. */
            used = (used & 0x5555) + ((used & 0xaaaa) >> 1);
            used = (used & 0x3333) + ((used & 0xcccc) >> 2);
            used = (used & 0x0f0f) + ((used & 0xf0f0) >> 4);
            used = (used & 0x00ff) + (used >> 8);
            *r = jisx0208_2charset[summary->indx + used];
            return 2;
        }
    }
    return 0;
}

/*
 * SHIFT_JIS (libiconv-1.16/lib/sjis.h)
 */

/* Returns 1 or 2 on success, 0 if no mapping */
INTERNAL int sjis_wctomb_zint(unsigned int *r, const unsigned int wc) {
    int ret;

    /* Try JIS X 0201-1976. */
    ret = jisx0201_wctomb(r, wc);
    if (ret) {
        return ret;
    }

    /* Try JIS X 0208-1990. */
    /* ZINT: Note leaving mapping of full-width reverse solidus U+FF3C to 0x815F (duplicate of patched U+005C) to
     * avoid having to regen tables */
    ret = jisx0208_wctomb(r, wc);
    if (ret) {
        return ret;
    }

    /* User-defined range. See
    * Ken Lunde's "CJKV Information Processing", table 4-66, p. 206. */
    /* ZINT: https://file.allitebooks.com/20160708/CJKV%20Information%20Processing.pdf (table 4-86, p.286, 2nd ed.) */
    if (wc >= 0xe000 && wc < 0xe758) {
        unsigned char c1, c2;
        c1 = (unsigned int) (wc - 0xe000) / 188;
        c2 = (unsigned int) (wc - 0xe000) % 188;
        *r = ((c1 + 0xf0) << 8) | (c2 < 0x3f ? c2 + 0x40 : c2 + 0x41);
        return 2;
    }

    return 0;
}

/* Convert UTF-8 string to Shift JIS and place in array of ints */
INTERNAL int sjis_utf8(struct zint_symbol *symbol, const unsigned char source[], int *p_length,
                unsigned int *ddata) {
    int error_number;
    unsigned int i, length;
#ifndef _MSC_VER
    unsigned int utfdata[*p_length + 1];
#else
    unsigned int *utfdata = (unsigned int *) _alloca((*p_length + 1) * sizeof(unsigned int));
#endif

    error_number = utf8_to_unicode(symbol, source, utfdata, p_length, 1 /*disallow_4byte*/);
    if (error_number != 0) {
        return error_number;
    }

    for (i = 0, length = *p_length; i < length; i++) {
        if (!sjis_wctomb_zint(ddata + i, utfdata[i])) {
            strcpy(symbol->errtxt, "800: Invalid character in input data");
            return ZINT_ERROR_INVALID_DATA;
        }
    }

    return 0;
}

/* Convert UTF-8 string to ECI and place in array of ints */
INTERNAL int sjis_utf8_to_eci(const int eci, const unsigned char source[], int *p_length, unsigned int *ddata,
                const int full_multibyte) {

    if (is_eci_convertible(eci)) {
        int error_number;
        const int eci_length = get_eci_length(eci, source, *p_length);
#ifndef _MSC_VER
        unsigned char converted[eci_length + 1];
#else
        unsigned char *converted = (unsigned char *) _alloca(eci_length + 1);
#endif

        error_number = utf8_to_eci(eci, source, converted, p_length);
        if (error_number != 0) {
            /* Note not setting `symbol->errtxt`, up to caller */
            return error_number;
        }

        sjis_cpy(converted, p_length, ddata, full_multibyte || eci == 20);
    } else {
        sjis_cpy(source, p_length, ddata, full_multibyte);
    }

    return 0;
}

/* If `full_multibyte` set, copy byte input stream to array of ints, putting double-bytes that match QR Kanji mode in
 * a single entry. If `full_multibyte` not set, do a straight copy */
INTERNAL void sjis_cpy(const unsigned char source[], int *p_length, unsigned int *ddata, const int full_multibyte) {
    unsigned int i, j, jis, length;
    unsigned char c;

    if (full_multibyte) {
        for (i = 0, j = 0, length = *p_length; i < length; i++, j++) {
            c = source[i];
            if (((c >= 0x81 && c <= 0x9F) || (c >= 0xE0 && c <= 0xEB)) && length - i >= 2) {
                jis = (c << 8) | source[i + 1];
                if ((jis >= 0x8140 && jis <= 0x9FFC) || (jis >= 0xE040 && jis <= 0xEBBF)) {
                    /* This may or may not be valid Shift JIS, but don't care as long as it can be encoded in
                     * QR Kanji mode */
                    ddata[j] = jis;
                    i++;
                } else {
                    ddata[j] = c;
                }
            } else {
                ddata[j] = c;
            }
        }
        *p_length = j;
    } else {
        /* Straight copy */
        for (i = 0, length = *p_length; i < length; i++) {
            ddata[i] = source[i];
        }
    }
}

/* Call `sjis_cpy()` for each segment */
INTERNAL void sjis_cpy_segs(struct zint_seg segs[], const int seg_count, unsigned int *ddata,
                const int full_multibyte) {
    int i;
    unsigned int *dd = ddata;

    for (i = 0; i < seg_count; i++) {
        sjis_cpy(segs[i].source, &segs[i].length, dd, full_multibyte);
        dd += segs[i].length;
    }
}

/* vim: set ts=4 sw=4 et : */
