import registerMenubar from './register-menu-bar'
import registerToolbar, { ToolbarControl } from './register-toolbar'
import registerWindowControls from './register-window-controls'
import registerGlobals from './register-globals'
import loadI18nRenderer from '../../load-i18n-renderer'
import registerThemes from './register-themes'
import registerDefaultContextMenu from './register-default-context'
import loadIcons from './load-icons'

export interface RegistrationOptions {
  showMenubar?: boolean
  showWindowControls?: boolean
  toolbarControls?: ToolbarControl[]
}

// Provide the registerToolbar function to programmatically update it afterwards
export { registerToolbar }

/**
 * This function is the renderer's counterpart to the main process's window
 * configuration and registers stuff like custom window controls and the menu
 * bar (on Windows and Linux, if native is off)
 */
export default function windowRegister (options?: RegistrationOptions): void {
  // First of all, add the correct class to the body element. This ensures
  // certain styling, for instance a minimal top-bar for Windows and Linux non-
  // native styles.
  document.body.classList.add(process.platform)

  // Load the clarity icons
  loadIcons().catch(e => { console.error(e) })

  // Prevent any funny navigation -- Zettlr does not allow any other file
  // to be open here, so if anyone wants to navigate, open the link externally
  // instead.
  // TODO: Currently, this also prevents reloads
  // window.addEventListener('beforeunload', (event) => {
  //   event.preventDefault()
  //   event.returnValue = false // equivalent to `return false` but not recommended
  //   return false
  // })

  // Determine if the menubar should be shown (default: yes)
  let shouldShowMenubar: boolean = true
  if (options !== undefined) {
    if (options.showMenubar !== undefined) {
      shouldShowMenubar = options.showMenubar
    }
  }

  // Determine if the window controls should be shown (default: yes)
  let shouldShowWindowControls: boolean = true
  if (options !== undefined) {
    if (options.showWindowControls !== undefined) {
      shouldShowWindowControls = options.showWindowControls
    }
  }

  // Determine if this code should handle the toolbar (default: no).
  // The default is set to give this code backward compatibility (only
  // touch the toolbar where we explicitly set this)
  let shouldHandleToolbar: boolean = false
  if (options !== undefined) {
    if (options.toolbarControls !== undefined) {
      shouldHandleToolbar = true // Existence of toolbarControls implies a toolbar
    }
  }

  // Load the translation strings
  loadI18nRenderer()

  // Register globals (such as global.config, etc.)
  registerGlobals()
  // Then, we also need to listen to clicks onto the window controls
  registerWindowControls(shouldShowWindowControls)
  // ... register the menu bar ...
  registerMenubar(shouldShowMenubar)
  // ... the toolbar ...
  if (shouldHandleToolbar && options?.toolbarControls !== undefined) {
    registerToolbar(options.toolbarControls)
  }
  // ... the theming functionality ...
  registerThemes()
  // ... the default context menus
  registerDefaultContextMenu()
}
