#! /usr/bin/env bash

# It's possible to use this script locally from the zeek repo's root dir.
# The parallelism level when running tests locally is $1 if provided, else
# the value of `nproc` if available, otherwise just a single core.

result=0
BTEST=$(pwd)/auxil/btest/btest

if [[ -z "${CIRRUS_CI}" ]]; then
    # Set default values to use in place of env. variables set by Cirrus CI.
    ZEEK_CI_CPUS=1
    [[ $(which nproc) ]] && ZEEK_CI_CPUS=$(nproc)
    [[ -n "${1}" ]] && ZEEK_CI_CPUS=${1}
    ZEEK_CI_BTEST_JOBS=${ZEEK_CI_CPUS}
    ZEEK_CI_BTEST_RETRIES=2
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" &>/dev/null && pwd)"
. ${SCRIPT_DIR}/common.sh

function pushd {
    command pushd "$@" >/dev/null || exit 1
}

function popd {
    command popd "$@" >/dev/null || exit 1
}

function banner {
    local msg="${1}"
    printf "+--------------------------------------------------------------+\n"
    printf "| %-60s |\n" "$(date)"
    printf "| %-60s |\n" "${msg}"
    printf "+--------------------------------------------------------------+\n"
}

function run_unit_tests {
    banner "Running unit tests"

    pushd build
    (. ./zeek-path-dev.sh && zeek --test --no-skip) || result=1
    popd
    return 0
}

function prep_artifacts {
    banner "Prepare artifacts"
    [[ -d .tmp ]] && rm -rf .tmp/script-coverage && tar -czf tmp.tar.gz .tmp
    junit2html btest-results.xml btest-results.html
}

function run_btests {
    banner "Running baseline tests: zeek"

    pushd testing/btest

    # Commenting out this line in btest.cfg causes the script profiling/coverage
    # to be disabled. We do this for the sanitizer build right now because of a
    # fairly significant performance bug when running tests.
    if [ "${ZEEK_CI_DISABLE_SCRIPT_PROFILING}" = "1" ]; then
        sed -i 's/^ZEEK_PROFILER_FILE/#ZEEK_PROFILER_FILE/g' btest.cfg
    fi

    ${BTEST} -z ${ZEEK_CI_BTEST_RETRIES} -d -b -x btest-results.xml -j ${ZEEK_CI_BTEST_JOBS} || result=1
    make coverage
    prep_artifacts
    popd
    return 0
}

function run_external_btests {
    # Commenting out this line in btest.cfg causes the script profiling/coverage
    # to be disabled. We do this for the sanitizer build right now because of a
    # fairly significant performance bug when running tests.
    if [ "${ZEEK_CI_DISABLE_SCRIPT_PROFILING}" = "1" ]; then
        pushd testing/external
        sed -i 's/^ZEEK_PROFILER_FILE/#ZEEK_PROFILER_FILE/g' subdir-btest.cfg
        popd
    fi

    local zeek_testing_pid=""
    local zeek_testing_pid_private=""
    pushd testing/external/zeek-testing
    ${BTEST} -d -b -x btest-results.xml -j ${ZEEK_CI_BTEST_JOBS} >btest.out 2>&1 &
    zeek_testing_pid=$!
    popd

    if [[ -d testing/external/zeek-testing-private ]]; then
        pushd testing/external/zeek-testing-private
        # Note that we don't use btest's "-d" flag or generate/upload any
        # artifacts to prevent leaking information about the private pcaps.
        ${BTEST} -b -j ${ZEEK_CI_BTEST_JOBS} >btest.out 2>&1 &
        zeek_testing_private_pid=$!
        popd
    fi

    banner "Running baseline tests: external/zeek-testing"
    wait ${zeek_testing_pid} || result=1
    pushd testing/external/zeek-testing
    cat btest.out
    make coverage
    prep_artifacts
    popd

    if [[ -n "${zeek_testing_private_pid}" ]]; then
        banner "Running baseline tests: external/zeek-testing-private"
        wait ${zeek_testing_private_pid} || result=1
        pushd testing/external/zeek-testing-private
        make coverage
        cat btest.out
        popd
    else
        banner "Skipping private tests (not available for PRs)"
    fi
}

banner "Start tests: ${ZEEK_CI_CPUS} cpus, ${ZEEK_CI_BTEST_JOBS} btest jobs"

run_unit_tests
run_btests
run_external_btests

exit ${result}
