/*
 * Decompiled with CFR 0.152.
 */
package org.zanata.client.commands;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.io.File;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Optional;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.HierarchicalINIConfiguration;
import org.apache.commons.configuration.SubnodeConfiguration;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zanata.client.commands.ConfigurableGlossaryOptions;
import org.zanata.client.commands.ConfigurableOptions;
import org.zanata.client.commands.ConfigurableProjectOptions;
import org.zanata.client.commands.ConsoleInteractor;
import org.zanata.client.commands.ConsoleInteractorImpl;
import org.zanata.client.commands.FileMappingRuleHandler;
import org.zanata.client.commands.Messages;
import org.zanata.client.commands.OptionMismatchChecker;
import org.zanata.client.config.ConfigUtil;
import org.zanata.client.config.FileMappingRule;
import org.zanata.client.config.LocaleList;
import org.zanata.client.config.LocaleMapping;
import org.zanata.client.config.ZanataConfig;
import org.zanata.client.exceptions.ConfigException;
import org.zanata.rest.client.ProjectIterationLocalesClient;
import org.zanata.rest.client.RestClientFactory;
import org.zanata.util.VersionUtility;

public class OptionsUtil {
    private static final Logger log = LoggerFactory.getLogger(OptionsUtil.class);

    public static void applyConfigFiles(ConfigurableOptions opts) throws ConfigurationException, JAXBException {
        Optional<ZanataConfig> zanataConfig = Optional.empty();
        if (opts instanceof ConfigurableProjectOptions) {
            ConfigurableProjectOptions projOpts = (ConfigurableProjectOptions)opts;
            zanataConfig = OptionsUtil.applyProjectConfigToProjectOptions(projOpts);
        } else if (opts instanceof ConfigurableGlossaryOptions) {
            OptionsUtil.applyConfigToGlossaryOptions((ConfigurableGlossaryOptions)opts);
        }
        boolean shouldFetchLocalesFromServer = OptionsUtil.shouldFetchLocalesFromServer(zanataConfig, opts);
        if (opts.getUserConfig() != null) {
            if (opts.getUserConfig().exists()) {
                log.info("Loading user config from {}", (Object)opts.getUserConfig());
                HierarchicalINIConfiguration dataConfig = new HierarchicalINIConfiguration(opts.getUserConfig());
                OptionsUtil.applyUserConfig(opts, dataConfig);
            } else {
                System.err.printf("User config file '%s' not found; ignoring.%n", opts.getUserConfig());
            }
        }
        if (shouldFetchLocalesFromServer) {
            ConfigurableProjectOptions projectOptions = (ConfigurableProjectOptions)opts;
            LocaleList localeMappings = OptionsUtil.fetchLocalesFromServer(projectOptions, OptionsUtil.createClientFactoryWithoutVersionCheck(projectOptions));
            projectOptions.setLocaleMapList(localeMappings);
        }
    }

    public static Optional<ZanataConfig> applyProjectConfigToProjectOptions(ConfigurableProjectOptions opts) throws JAXBException {
        Optional<ZanataConfig> projectConfig = OptionsUtil.readProjectConfigFile(opts);
        if (projectConfig.isPresent()) {
            OptionsUtil.applyProjectConfig(opts, projectConfig.get());
        } else if (opts.getProjectConfig() != null) {
            log.warn("Project config file '{}' not found; ignoring.", (Object)opts.getProjectConfig());
        }
        return projectConfig;
    }

    public static boolean shouldFetchLocalesFromServer(Optional<ZanataConfig> projectConfig, ConfigurableOptions opts) {
        boolean localesDefinedInFile;
        if (!projectConfig.isPresent()) {
            if (opts instanceof ConfigurableProjectOptions) {
                ConfigurableProjectOptions projectOptions = (ConfigurableProjectOptions)opts;
                return StringUtils.isNotEmpty((CharSequence)projectOptions.getProj()) && StringUtils.isNotEmpty((CharSequence)projectOptions.getProjectVersion());
            }
            return false;
        }
        ZanataConfig zanataConfig = projectConfig.get();
        boolean bl = localesDefinedInFile = zanataConfig.getLocales() != null && !zanataConfig.getLocales().isEmpty();
        if (localesDefinedInFile) {
            ConsoleInteractorImpl console = new ConsoleInteractorImpl(opts);
            console.printfln(ConsoleInteractor.DisplayMode.Warning, Messages.get("locales.in.config.deprecated"), new Object[0]);
            return false;
        }
        return true;
    }

    public static void applyConfigToGlossaryOptions(ConfigurableGlossaryOptions opts) throws JAXBException {
        File configFile = opts.getConfig();
        if (configFile != null) {
            JAXBContext jc = JAXBContext.newInstance((Class[])new Class[]{ZanataConfig.class});
            Unmarshaller unmarshaller = jc.createUnmarshaller();
            if (configFile.exists()) {
                log.info("Loading config from {}", (Object)configFile);
                ZanataConfig projectConfig = (ZanataConfig)unmarshaller.unmarshal(configFile);
                OptionsUtil.applyBasicConfig(opts, projectConfig);
            } else {
                log.warn("Config file '{}' not found; ignoring.", (Object)configFile);
            }
        }
    }

    public static Optional<ZanataConfig> readProjectConfigFile(ConfigurableProjectOptions projOpts) throws JAXBException {
        File projectConfigFile;
        if (projOpts.getProjectConfig() != null && (projectConfigFile = projOpts.getProjectConfig()).exists()) {
            JAXBContext jc = JAXBContext.newInstance((Class[])new Class[]{ZanataConfig.class});
            Unmarshaller unmarshaller = jc.createUnmarshaller();
            log.info("Loading project config from {}", (Object)projectConfigFile);
            return Optional.of((ZanataConfig)unmarshaller.unmarshal(projectConfigFile));
        }
        return Optional.empty();
    }

    public static LocaleList fetchLocalesFromServer(ConfigurableProjectOptions projectOpts, RestClientFactory restClientFactory) {
        LocaleList localeList = new LocaleList();
        ProjectIterationLocalesClient projectIterationLocalesClient = restClientFactory.getProjectLocalesClient(projectOpts.getProj(), projectOpts.getProjectVersion());
        List localeMappings = Lists.transform((List)projectIterationLocalesClient.getLocales(), input -> input == null ? null : new LocaleMapping(input.getLocaleId().getId(), input.getAlias()));
        localeList.addAll(localeMappings);
        return localeList;
    }

    private static void applyProjectConfig(ConfigurableProjectOptions opts, ZanataConfig config) {
        OptionsUtil.applyBasicConfig(opts, config);
        if (opts.getProj() == null) {
            opts.setProj(config.getProject());
        }
        if (opts.getProjectVersion() == null) {
            opts.setProjectVersion(config.getProjectVersion());
        }
        if (opts.getProjectType() == null) {
            opts.setProjectType(config.getProjectType());
        }
        OptionsUtil.applySrcDirAndTransDirFromProjectConfig(opts, config);
        OptionsUtil.applyIncludesAndExcludesFromProjectConfig(opts, config);
        LocaleList localesInFile = config.getLocales();
        opts.setLocaleMapList(localesInFile);
        if (opts.getCommandHooks().isEmpty() && config.getHooks() != null) {
            opts.setCommandHooks(config.getHooks());
        }
        opts.setFileMappingRules(config.getRules());
        OptionsUtil.checkPotentialMistakesInRules(opts, new ConsoleInteractorImpl(opts));
    }

    private static void applyBasicConfig(ConfigurableOptions opts, ZanataConfig config) {
        if (opts.getUrl() == null) {
            opts.setUrl(config.getUrl());
        }
    }

    @VisibleForTesting
    protected static void checkPotentialMistakesInRules(ConfigurableProjectOptions opts, ConsoleInteractor console) {
        boolean potentialProblem = false;
        boolean invalid = false;
        for (FileMappingRule mappingRule : opts.getFileMappingRules()) {
            String rule = mappingRule.getRule();
            if (!FileMappingRuleHandler.isRuleValid(rule)) {
                console.printfln(ConsoleInteractor.DisplayMode.Warning, Messages.get("invalid.rule"), rule);
                invalid = true;
            }
            if (!OptionsUtil.ruleMayHaveProblem(rule)) continue;
            console.printfln(ConsoleInteractor.DisplayMode.Warning, Messages.get("unrecognized.variables"), FileMappingRuleHandler.Placeholders.allHolders(), rule);
            potentialProblem = true;
        }
        Preconditions.checkState((!invalid ? 1 : 0) != 0);
        if (potentialProblem && opts.isInteractiveMode()) {
            console.printfln(ConsoleInteractor.DisplayMode.Question, Messages.get("confirm.rule"), new Object[0]);
            console.expectYes();
        }
    }

    private static boolean ruleMayHaveProblem(String rule) {
        String remains = OptionsUtil.stripValidHolders(rule);
        return remains.contains("{") || remains.contains("}");
    }

    @VisibleForTesting
    protected static void applySrcDirAndTransDirFromProjectConfig(ConfigurableProjectOptions opts, ZanataConfig config) {
        OptionMismatchChecker<File> srcDirChecker = OptionMismatchChecker.from(opts.getSrcDir(), config.getSrcDirAsFile(), "Source directory");
        if (srcDirChecker.hasValueInConfigOnly()) {
            opts.setSrcDir(config.getSrcDirAsFile());
        }
        srcDirChecker.logHintIfNotDefinedInConfig(String.format("<src-dir>%s</src-dir>", opts.getSrcDir()));
        srcDirChecker.logWarningIfValuesMismatch();
        OptionMismatchChecker<File> transDirChecker = OptionMismatchChecker.from(opts.getTransDir(), config.getTransDirAsFile(), "Translation directory");
        if (transDirChecker.hasValueInConfigOnly()) {
            opts.setTransDir(config.getTransDirAsFile());
        }
        transDirChecker.logHintIfNotDefinedInConfig(String.format("<trans-dir>%s</trans-dir>", opts.getTransDir()));
        transDirChecker.logWarningIfValuesMismatch();
    }

    protected static void applyIncludesAndExcludesFromProjectConfig(ConfigurableProjectOptions opts, ZanataConfig config) {
        OptionMismatchChecker<ImmutableList<String>> includesChecker = OptionMismatchChecker.from(opts.getIncludes(), config.getIncludesAsList(), "Includes");
        if (includesChecker.hasValueInConfigOnly()) {
            opts.setIncludes(config.getIncludes());
        }
        Joiner commaJoiner = Joiner.on((String)",");
        includesChecker.logHintIfNotDefinedInConfig(String.format("<includes>%s</includes>", commaJoiner.join(opts.getIncludes())));
        includesChecker.logWarningIfValuesMismatch();
        OptionMismatchChecker<ImmutableList<String>> excludesChecker = OptionMismatchChecker.from(opts.getExcludes(), config.getExcludesAsList(), "Excludes");
        if (excludesChecker.hasValueInConfigOnly()) {
            opts.setExcludes(config.getExcludes());
        }
        excludesChecker.logHintIfNotDefinedInConfig(String.format("<excludes>%s</excludes>", commaJoiner.join(opts.getExcludes())));
        excludesChecker.logWarningIfValuesMismatch();
    }

    public static void applyUserConfig(ConfigurableOptions opts, HierarchicalINIConfiguration config) {
        SubnodeConfiguration servers;
        String prefix;
        Boolean quiet;
        Boolean errors;
        Boolean debug;
        if (!opts.isDebugSet() && (debug = config.getBoolean("defaults.debug", null)) != null) {
            opts.setDebug(debug);
        }
        if (!opts.isErrorsSet() && (errors = config.getBoolean("defaults.errors", null)) != null) {
            opts.setErrors(errors);
        }
        if (!opts.isQuietSet() && (quiet = config.getBoolean("defaults.quiet", null)) != null) {
            opts.setQuiet(quiet);
        }
        if ((opts.getUsername() == null || opts.getKey() == null) && opts.getUrl() != null && (prefix = ConfigUtil.findPrefix(servers = config.getSection("servers"), opts.getUrl())) != null) {
            if (opts.getUsername() == null) {
                opts.setUsername(servers.getString(prefix + ".username", null));
            }
            if (opts.getKey() == null) {
                opts.setKey(servers.getString(prefix + ".key", null));
            }
        }
    }

    private static void checkMandatoryOptsForRequestFactory(ConfigurableOptions opts) {
        if (opts.getUrl() == null) {
            throw new ConfigException("Server URL must be specified");
        }
        if (opts.isAuthRequired() && opts.getUsername() == null) {
            throw new ConfigException("Username must be specified");
        }
        if (opts.isAuthRequired() && opts.getKey() == null) {
            throw new ConfigException("API key must be specified");
        }
        if (opts.isDisableSSLCert()) {
            log.warn("SSL certificate verification will be disabled. You should consider adding the certificate instead of disabling it.");
        }
    }

    public static String stripValidHolders(String rule) {
        String temp = rule;
        for (FileMappingRuleHandler.Placeholders placeholder : FileMappingRuleHandler.Placeholders.values()) {
            temp = temp.replace(placeholder.holder(), "");
        }
        return temp;
    }

    public static <O extends ConfigurableOptions> RestClientFactory createClientFactory(O opts) {
        OptionsUtil.checkMandatoryOptsForRequestFactory(opts);
        try {
            RestClientFactory restClientFactory = new RestClientFactory(opts.getUrl().toURI(), opts.getUsername(), opts.getKey(), VersionUtility.getAPIVersionInfo(), opts.getLogHttp(), opts.isDisableSSLCert());
            restClientFactory.performVersionCheck();
            return restClientFactory;
        }
        catch (URISyntaxException e) {
            throw new ConfigException(e);
        }
    }

    public static <O extends ConfigurableOptions> RestClientFactory createClientFactoryWithoutVersionCheck(O opts) {
        OptionsUtil.checkMandatoryOptsForRequestFactory(opts);
        try {
            return new RestClientFactory(opts.getUrl().toURI(), opts.getUsername(), opts.getKey(), VersionUtility.getAPIVersionInfo(), opts.getLogHttp(), opts.isDisableSSLCert());
        }
        catch (URISyntaxException e) {
            throw new ConfigException(e);
        }
    }
}

