#ifndef __XRDPOSIXXROOTD_H__
#define __XRDPOSIXXROOTD_H__
/******************************************************************************/
/*                                                                            */
/*                        X r d P o s i x X r o o t d                         */
/*                                                                            */
/* (c) 2010 by the Board of Trustees of the Leland Stanford, Jr., University  */
/*                            All Rights Reserved                             */
/*   Produced by Andrew Hanushevsky for Stanford University under contract    */
/*              DE-AC02-76-SFO0515 with the Department of Energy              */
/*                                                                            */
/* This file is part of the XRootD software suite.                            */
/*                                                                            */
/* XRootD is free software: you can redistribute it and/or modify it under    */
/* the terms of the GNU Lesser General Public License as published by the     */
/* Free Software Foundation, either version 3 of the License, or (at your     */
/* option) any later version.                                                 */
/*                                                                            */
/* XRootD is distributed in the hope that it will be useful, but WITHOUT      */
/* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or      */
/* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public       */
/* License for more details.                                                  */
/*                                                                            */
/* You should have received a copy of the GNU Lesser General Public License   */
/* along with XRootD in a file called COPYING.LESSER (LGPL license) and file  */
/* COPYING (GPL license).  If not, see <http://www.gnu.org/licenses/>.        */
/*                                                                            */
/* The copyright holder's institutional names and contributor's names may not */
/* be used to endorse or promote products derived from this software without  */
/* specific prior written permission of the institution or contributor.       */
/* Modified by Frank Winklmeier to add the full Posix file system definition. */
/******************************************************************************/

#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <sys/types.h>

#if defined(__APPLE__) || defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/mount.h>
#else
#include <sys/statfs.h>
#endif

#include "XrdPosix/XrdPosixOsDep.hh"
#include "XrdSys/XrdSysPthread.hh"

class XrdOucCache;
class XrdOucEnv;
class XrdPosixCallBack;
class XrdPosixFile;
class XrdPosixDir;

class XrdPosixXrootd
{
public:

// POSIX methods
//
static int     Close(int fildes, int Stream=0);

static int     Closedir(DIR *dirp);

static int     Fstat(int fildes, struct stat *buf);

static int     Fsync(int fildes);

static int     Ftruncate(int fildes, off_t offset);

static long long Getxattr (const char *path, const char *name,
                           void *value, unsigned long long size);

static off_t   Lseek(int fildes, off_t offset, int whence);

static int     Mkdir(const char *path, mode_t mode);

static const int isStream = 0x40000000; // Internal for Open oflag

static int     Open(const char *path, int oflag, mode_t mode=0,
                    XrdPosixCallBack *cbP=0);

static DIR*    Opendir(const char *path);
  
static ssize_t Pread(int fildes, void *buf, size_t nbyte, off_t offset);
  
static ssize_t Read(int fildes, void *buf, size_t nbyte);

static ssize_t Readv(int fildes, const struct iovec *iov, int iovcnt);

static struct dirent*   Readdir  (DIR *dirp);
static struct dirent64* Readdir64(DIR *dirp);

static int     Readdir_r  (DIR *dirp, struct dirent   *entry, struct dirent   **result);
static int     Readdir64_r(DIR *dirp, struct dirent64 *entry, struct dirent64 **result);

static int     Rename(const char *oldpath, const char *newpath);

static void    Rewinddir(DIR *dirp);

static int     Rmdir(const char *path);

static void    Seekdir(DIR *dirp, long loc);

static int     Stat(const char *path, struct stat *buf);

static int     Statfs(const char *path, struct statfs *buf);

static int     Statvfs(const char *path, struct statvfs *buf);

static ssize_t Pwrite(int fildes, const void *buf, size_t nbyte, off_t offset);

static long    Telldir(DIR *dirp);

static int     Truncate(const char *path, off_t offset);

static int     Unlink(const char *path);

static ssize_t Write(int fildes, const void *buf, size_t nbyte);

static ssize_t Write(int fildes, void *buf, size_t nbyte, off_t offset);

static ssize_t Writev(int fildes, const struct iovec *iov, int iovcnt);

// Some non POSIX methods
//
static int     Access(const char *path, int amode);

static int     endPoint(int FD, char *Buff, int Blen);

static bool    isXrootdDir(DIR *dirp);

static int     mapError(int rc);

static
inline bool    myFD(int fd) {return fd >= baseFD && fd <= (highFD+baseFD)
                                      && myFiles && myFiles[fd-baseFD];
                            }

static void    OpenCB(XrdPosixFile *fp, void *cbArg, int res);

static int     QueryChksum(const char *, time_t &, char *, int);

static long long QueryOpaque(const char*, char*, int);

static void    setDebug(int val);

static void    setEnv(const char *var, const char *val);

static void    setEnv(const char *var, long val);

static void    setCache(XrdOucCache *cP);

static int     Debug;

/* There must be one instance of this object per executable image. Typically,
   this object is declared in main() or at file level. This is necessary to
   properly do one-time initialization of the static members. When declaring
   this object, you can pass the following information:
   maxfd  - maximum number of simultaneous files to support (i.e. fdlimit).
            The value returned by getrlimit() over-rides the passed value
            unless maxfd is negative. When negative, abs(maxfd) becomes the
            absolute maximum and shadow file descriptors are not used.
   maxdir - maximum number of open directories that can be supported. This
            is independent of maxfd.
   maxthr - maximum number of threads to use for the callback function.
*/
               XrdPosixXrootd(int maxfd=255, int maxdir=255, int maxthr=255);
              ~XrdPosixXrootd();

private:

static void                  initEnv();
static void                  initEnv(char *eData);
static void                  initEnv(XrdOucEnv &, const char *, long long &);
static int                   Fault(XrdPosixFile *fp, int complete=1);
static XrdPosixFile         *findFP(int fildes, int glk=0);
static XrdPosixDir          *findDIR(DIR *dirp, int glk=0);
static void                  initStat(struct stat *buf);
static void                  initXdev(dev_t &st_dev, dev_t &st_rdev);
static int                   mapFlags(int flags);
static int                   mapMode(mode_t Mode);

static XrdSysMutex    myMutex;
static XrdPosixFile **myFiles;
static XrdPosixDir  **myDirs;
static XrdOucCache   *myCache;
static int            lastFD;
static int            highFD;
static int            baseFD;
static int            freeFD;
static int            lastDir;
static int            highDir;
static int            devNull;
static int            pllOpen;
static int            maxThreads;
static int            initDone;
};
#endif
