%{
/*_
 * Copyright (c) 2005, 2006, Markus W. Weissmann <mww@opendarwin.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of OpenDarwin nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: xmlpars.y,v 1.9 2006/02/22 17:42:34 mww Exp $
 *
 */

#include <assert.h>
#include <errno.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "item.h"
#include "pkg.h"
#include "util.h"
#include "xml.h"
#include "xpkg.h"

/* called in case of parse-error */
int32_t
xpkgxmlerror(xpkg_pkg_t *pkg, char *s) {
	DEBUG(s);
	return 0;
}

/* prototype for flex-generated lexer */
int32_t
xpkgxmllex();

/**/
int32_t
xpkgxml_prepare(char *s);

/* helper function to care for xml-attributes (e. g. <xpkg version="xyz">..) */
void
handle_attribute(xpkg_pkg_t pkg, const char *attr, const char *value) {
	DEBUG(attr);
	DEBUG(value);
	if (strncmp("version", attr, 7))
		pkg->xpkg_version=strdup(value);
}

%}

%locations
%name-prefix="xpkgxml"
%debug
%union { char *string; xpkg_item_t xitem; xpkg_items_t xitems; long number; };
%token ATTR XPKG NAME VERS REVI REQU PROV ITEM MINO MAJO
%token <string> STRING ATTR
%type <string> name
%type <number> major minor
%type <xitems> items
%type <xitem> aitem aitem_m
%destructor { FREE($$); } ATTR STRING name
%parse-param {xpkg_pkg_t *pkg}

%%
s    : xpkg
     |
;

xpkg : '<' XPKG attrs '>' expr_list '/' XPKG '>'
     ;

attrs:
     | ATTR '=' STRING					{ handle_attribute( *pkg, $1, $3 ); }
     | attrs ATTR '=' STRING			{ handle_attribute( *pkg, $2, $4 ); }
     ;

expr_list : expr expr_list
     | expr
     ;

expr : '<' NAME '>' STRING '/' NAME '>'	{ (*pkg)->name=strdup($4); FREE($4); }
     | '<' VERS '>' STRING '/' VERS '>'	{ (*pkg)->version=strdup($4); FREE($4); }
     | '<' REVI '>' STRING '/' REVI '>'	{ (*pkg)->revision=strtol($4,NULL,10); FREE($4); }
     | '<' MAJO '>' STRING '/' MAJO '>'	{ (*pkg)->major=strtol($4,NULL,10); FREE($4); }
     | '<' MINO '>' STRING { printf("%s\n",$4); } '/' MINO '>' { (*pkg)->minor=strtol($4,NULL,10); FREE($4); }
     | '<' REQU '>' items '/' REQU '>'	{ (*pkg)->requires = $4; }
     | '<' PROV '>' items '/' PROV '>'	{ (*pkg)->provides = $4; }
     ;

items: items aitem							{ LIST_INSERT_HEAD($1, $2, entries); $$=$1; };
     | aitem								{ xpkg_items_t i = items_create(); LIST_INSERT_HEAD(i, $1, entries); $$=i; };
     ;

aitem: '<' ITEM '>' aitem_m '/' ITEM '>'	{ $$ = $4 }
     ;

aitem_m : name major minor		{ $$ = item_create($1, $2, $3); FREE($1); }
     | name minor major			{ $$ = item_create($1, $3, $2); FREE($1); }
     | major minor name			{ $$ = item_create($3, $1, $2); FREE($3); }
     | major name minor			{ $$ = item_create($2, $1, $3); FREE($2); }
     | minor major name			{ $$ = item_create($3, $2, $1); FREE($3); }
     | minor name major			{ $$ = item_create($2, $3, $1); FREE($2); }
     ;

name : '<' NAME '>' STRING '/' NAME '>'		{ $$ = $4; }
     ;

major: '<' MAJO '>' STRING '/' MAJO '>'		{ $$ = strtol($4,NULL,10); FREE($4); }
     ;

minor: '<' MINO '>' STRING '/' MINO '>'		{ $$ = strtol($4,NULL,10); FREE($4); }
     ;

%%

/* parse-function - execs parser on given char*, fills given package-struct */
int32_t
xml_parse(char *xml, xpkg_pkg_t pkg) {
	int rc = XPKG_ERROR;

	assert(pkg);
	assert(xml);
	assert(0 < strlen(xml));

	if (NULL!=getenv("XPKG_DEBUG"))
		xpkgxmldebug = 1;
	xpkgxml_prepare(xml);
	pkg->requires = items_create();
	pkg->provides = items_create();
	if (0 == yyparse(&pkg))
		rc = XPKG_OK;
	return rc;
}

