/*_
 * Copyright (c) 2005, Markus W. Weissmann <mww@opendarwin.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of OpenDarwin nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: registry.c,v 1.25 2006/02/22 22:21:28 mww Exp $
 *
 */

#include <assert.h>
#include <errno.h>
#include <sqlite3.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "dbinit.sqlstr.h"
#include "file.h"
#include "item.h"
#include "pkg.h"
#include "registry.h"
#include "util.h"
#include "xpkg.h"

int32_t
registry_check_sqlite3db(const char *filename) {
	int rc = XPKG_ERROR;
	FILE *f;
	char buffer[15];

	DEBUG("checking reg-file");
	DEBUG(filename);
	if (0 != access(filename, F_OK)) {
		/* file does not exist */
		DEBUG("reg-file inexistent");
		rc = XPKG_OK;
	} else if (0 == access(filename, F_OK | W_OK)) {
		DEBUG("checking exitent reg-file");
		/* file exists, we can write - check if it is a sqlite3 db */
		memset(buffer, 0, 15);
		f = fopen(filename, "r");
		fread(buffer, sizeof(char), 15, f);
		fclose(f);
		if (0 == strncmp(buffer, "SQLite format 3", 15) )
			rc = XPKG_OK;
	}
	DEBUG("leaving check reg-file");
	return rc;
}

xpkg_registry_t
xpkg_registry_open(const char *filename) {
	sqlite3 *db;
	xpkg_registry_t reg;

	reg = NULL;
	DEBUG("open db");
	assert(*filename);
	if (XPKG_OK == registry_check_sqlite3db(filename)) {
		if (SQLITE_OK == sqlite3_open(filename, &db)) {
			reg = calloc(1,sizeof(struct xpkg_registry_t));
			reg->db = db;
			reg->in_transaction = false;
			reg->dberrmsg = NULL;
			reg->dberrmsg = "no error\n";
			/* just dont know yet, so mark it as inconsistent */
			reg->consistent = false;
		}
	} else {
		errno = XPKG_REGISTRY_UNAVAILABLE;
	}
	return reg;
}

int32_t
xpkg_registry_close(xpkg_registry_t *reg) {
	int rc = XPKG_ERROR;

	DEBUG("close db");
	if (NULL == *reg) {
		DEBUG("registry not open");
		rc = XPKG_REGISTRY_CLOSED;
	} else if (true == (*reg)->in_transaction) {
		DEBUG("transaction in progress");
		rc = XPKG_TRANSACTION_INPROGRESS;
	} else {
		DEBUG("trying to close sqlite3-db");
		if (SQLITE_OK == sqlite3_close((*reg)->db)) {
			DEBUG("closed db");
			FREE(*reg);
			rc = XPKG_OK;
		} else {
			DEBUG(sqlite3_errmsg((*reg)->db));
		}
	}

	return rc;
}

int32_t
xpkg_registry_transaction_begin(xpkg_registry_t reg) {
	int rc = XPKG_ERROR;
	const char* sql = "BEGIN EXCLUSIVE TRANSACTION XPKG;";

	DEBUG("begin transaction");
	assert(reg);
	if (false == reg->in_transaction) {
		if (SQLITE_OK == sqlite3_exec(reg->db, sql, NULL, NULL, &(reg)->dberrmsg)) {
			reg->in_transaction = true;
			rc = XPKG_OK;
		}
	} else {
		rc= XPKG_TRANSACTION_INPROGRESS;
	}
	return rc;
}

int32_t
xpkg_registry_transaction_commit(xpkg_registry_t reg) {
	int rc = XPKG_ERROR;
	const char* sql = "COMMIT TRANSACTION XPKG;";

	assert(reg);
	if (true == reg->in_transaction) {
		if (false == reg->consistent) {
			/* if consistency is ot given, 1st check */
			xpkg_registry_checkconsistence(reg);
		}
		if (true == reg->consistent) {
			/* if inconsistent after check, abort - else commit */
			if (SQLITE_OK == sqlite3_exec(reg->db, sql, NULL, NULL, &(reg)->dberrmsg)) {
				reg->in_transaction = false;
				rc = XPKG_OK;
			}
		} else {
			rc = XPKG_REGISTRY_INCONSISTENT;
		}
	} else {
		rc = XPKG_NO_TRANSACTION;
	}
	return rc;
}

int32_t
xpkg_registry_transaction_rollback(xpkg_registry_t reg) {
	int rc = XPKG_ERROR;
	const char* sql = "ROLLBACK TRANSACTION XPKG;";

	DEBUG("rolling back transaction");
	if (NULL == reg) {
		rc = XPKG_REGISTRY_CLOSED;
	} else if (true != reg->in_transaction) {
		rc = XPKG_NO_TRANSACTION;
	} else {
		if (SQLITE_OK == sqlite3_exec(reg->db, sql, NULL, NULL, &(reg)->dberrmsg)) {
			reg->in_transaction = false;
			rc = XPKG_OK;
		}
	}
	return rc;
}

int32_t
xpkg_registry_init(xpkg_registry_t reg) {
	int rc = XPKG_ERROR;
	const char *sql = SQL_STR_dbinit;

	DEBUG("creating tables etc.");
	assert(reg);
	if (false == reg->in_transaction) {
		if (SQLITE_OK == sqlite3_exec(reg->db, sql, NULL, NULL, &(reg)->dberrmsg)) {
			rc = XPKG_OK;
		} else {
			errno = XPKG_REGISTRY_CANT_INIT;
		}
	}
	return rc;
}

int32_t
xpkg_registry_checkrequirements(xpkg_registry_t reg, xpkg_pkg_t pkg, xpkg_items_t *items) {
	int n, count;
	const char *sql = "SELECT * FROM PKGPROVIDES WHERE ITEM=:iname AND MAJOR=? AND MINOR>=?;";
	xpkg_item_t i;
	xpkg_item_t o;
	xpkg_items_t items_out;
	xpkg_items_t requires;
	sqlite3_stmt* stmt;

	DEBUG("checking requirements");
	assert(reg);
	if (SQLITE_OK != sqlite3_prepare( reg->db, sql, -1, &stmt, NULL)) {
		assert(false);
	}
	requires = pkg->requires;
	items_out = items_create();
	count = 0;

	for (i = (*requires).lh_first; NULL != i; i = i->entries.le_next) {
		if (SQLITE_OK != sqlite3_bind_text(stmt, 1, i->name, -1, NULL)) {
			assert(false);
		}
		if (SQLITE_OK == sqlite3_bind_int(stmt, 2, i->major)) {
			assert(false);
		}
		if (SQLITE_OK == sqlite3_bind_int(stmt, 3, i->minor)) {
			assert(false);
		}
		n = sqlite3_step( stmt );
		if (SQLITE_DONE == n) {
			/* required item not installed - 0 rows returned */
			o = item_cpy(&o, i);
			item_to_items(items_out, o);
			count++;
		} else if (SQLITE_ROW != n) {
			assert(false);
		}
		if (SQLITE_OK != sqlite3_reset(stmt)) {
			assert(false);
		}
	}
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}

	*items = items_out;
	return count;
}

int32_t
add_packageentry(xpkg_registry_t reg, xpkg_pkg_t pkg, int64_t *pkgid) {
	int rc = XPKG_ERROR;
	int n;
	sqlite3_stmt *stmt_in, *stmt_id;
	const char *sql_in = "INSERT INTO PACKAGES( NAME, VERSION, REVISION, MAJOR, MINOR ) VALUES( :name, :version, ?, ?, ? );";
	const char *sql_id = "SELECT last_insert_rowid();";

	DEBUG("adding pkg entry");
	assert(reg);
	reg->consistent = false;
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql_in, -1, &stmt_in, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_text(stmt_in, 1, pkg->name, -1, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_text(stmt_in, 2, pkg->version, -1, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_int(stmt_in, 3, pkg->revision)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_int(stmt_in, 4, pkg->major)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_int(stmt_in, 5, pkg->minor)) {
		assert(false);
	}
	DEBUG("stepping through results" );
	n = sqlite3_step(stmt_in);
	if (SQLITE_DONE == n) {
		DEBUG("stepping through results" );
		if (SQLITE_OK != sqlite3_finalize(stmt_in)) {
			assert(false);
		}
		if (SQLITE_OK != sqlite3_prepare(reg->db, sql_id, -1, &stmt_id, NULL)) {
			assert(false);
		}
		if (SQLITE_ROW != sqlite3_step(stmt_id)) {
			assert(false);
		}
		*pkgid = sqlite3_column_int64(stmt_id, 0);
		if (SQLITE_OK != sqlite3_finalize(stmt_id)) {
			assert(false);
		}
		assert(*pkgid > 0);
		rc = XPKG_OK;
	} else {
		DEBUG(sqlite3_errmsg(reg->db));
		if (SQLITE_CONSTRAINT != sqlite3_finalize(stmt_in)) {
			assert(false);
		}
		rc = XPKG_PKG_ALREADY_REGISTERED;
		DEBUG("package already registered");
	}

	return rc;
}

int32_t
add_files(xpkg_registry_t reg, int64_t pkgid, xpkg_files_t files) {
	int rc = XPKG_ERROR;
	struct xpkg_files_t fs;
	xpkg_file_t f;
	sqlite3_stmt *stmt_infile, *stmt_ownership, *stmt_ownership_d;
	const char *sql_infile = "INSERT INTO FILES (PATH, FTYPE, CHKSUM) VALUES (:path, ?, :chksum);";
	const char *sql_ownership = "INSERT INTO FILEOWNERSHIP (PKGID, FILEID) SELECT ?, last_insert_rowid();";
	const char *sql_ownership_d = "INSERT INTO FILEOWNERSHIP (PKGID, FILEID) SELECT ?, ID FROM FILES WHERE PATH=:path;";

	assert(reg);
	assert(files);
	fs = *files;

	reg->consistent = false;
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql_infile, -1, &stmt_infile, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql_ownership, -1, &stmt_ownership, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql_ownership_d, -1, &stmt_ownership_d, NULL)) {
		assert(false);
	}

	for (f = fs.lh_first; NULL != f; f = f->entries.le_next) {
		assert(f);
		assert(f->ftype);

		/* bind path, type & optionally teh checksum */
		if (SQLITE_OK != sqlite3_bind_text(stmt_infile, 1, f->path, -1, NULL)) {
			assert(false);
		}
		if (SQLITE_OK != sqlite3_bind_int(stmt_infile, 2, f->ftype)) {
			assert(false);
		}
		if (XPKG_FILE_DIR == f->ftype) {
			if (SQLITE_OK != sqlite3_bind_null(stmt_infile, 3)) {
				assert(false);
			}
		} else {
			if (SQLITE_OK != sqlite3_bind_text(stmt_infile, 3, f->chksum, -1, NULL)) {
				assert(false);
			}
		}
		/* just try to exec */
		if (SQLITE_DONE == sqlite3_step(stmt_infile)) {
			if (SQLITE_OK != sqlite3_bind_int64(stmt_ownership, 1, pkgid)) {
				assert(false);
			}
			if (SQLITE_DONE != sqlite3_step(stmt_ownership)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_reset(stmt_ownership)) {
				assert(false);
			}
			/* reset INSERT statement normally */
			if (SQLITE_OK != sqlite3_reset(stmt_infile)) {
				assert(false);
			}
		} else if (XPKG_FILE_DIR == f->ftype) {
			/* directory with multiple owning packages */
			if (SQLITE_OK != sqlite3_bind_int64(stmt_ownership_d, 1, pkgid)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_bind_text(stmt_ownership_d, 2, f->path, -1, NULL)) {
				assert(false);
			}
			if (SQLITE_DONE != sqlite3_step(stmt_ownership_d)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_reset(stmt_ownership_d)) {
				assert(false);
			}
			/* expect constraint error - the directory was already listed */
			if (SQLITE_CONSTRAINT != sqlite3_reset(stmt_infile)) {
				assert(false);
			}
		} else {
			/* duplicate file - error */
			DEBUG(sqlite3_errmsg(reg->db));
			rc = XPKG_FILETYPE_CONFLICT;
			break;
		}
	}

	if (SQLITE_OK != sqlite3_finalize(stmt_ownership)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_finalize(stmt_ownership_d)) {
		assert(false);
	}
	if (XPKG_FILE_CONFLICT == rc) {
		/* should be constraint error */
		if (SQLITE_CONSTRAINT != sqlite3_finalize(stmt_infile)) {
			assert(false);
		}
	} else if (XPKG_FILETYPE_CONFLICT == rc) {
		DEBUG(sqlite3_errmsg(reg->db));
		assert(false);
	} else {
		if (SQLITE_OK != sqlite3_finalize(stmt_infile)) {
			/* constraint error */
			DEBUG(sqlite3_errmsg(reg->db));
			assert(false);
		}
		rc = XPKG_OK;
	}

	return rc;
}

int32_t
add_items(xpkg_registry_t reg, int64_t pkgid, xpkg_pkg_t pkg) {
	int rc = XPKG_ERROR;
	int c;
	xpkg_items_t items;
	xpkg_item_t i;
	sqlite3_stmt *stmt_it, *stmt_de;
	char *sql_de;
	const char *sql_it = "INSERT OR IGNORE INTO ITEMS( NAME, MAJOR, MINOR ) VALUES( :name, ?, ? );";
	const char *sql_re = "INSERT INTO REQUIRES( PKGID, ITEMID ) SELECT ?, last_insert_rowid();";
	const char *sql_pr = "INSERT INTO PROVIDES( PKGID, ITEMID ) SELECT ?, last_insert_rowid();";

	assert(reg);
	reg->consistent = false;
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql_it, -1, &stmt_it, NULL)) {
		assert(false);
	}
	for (c = 0; c < 2; c++) {
		if( 0 == c ) {
			/* provides-item(s) */
			items = pkg->requires;
			sql_de = (char*)sql_re;
		} else {
			/* requires-item(s) */
			items = pkg->provides;
			sql_de = (char*)sql_pr;
		}
		if (SQLITE_OK != sqlite3_prepare(reg->db, sql_de, -1, &stmt_de, NULL)) {
			assert(false);
		}
		for (i =(*items).lh_first; NULL != i; i = i->entries.le_next ) {
			if (SQLITE_OK != sqlite3_bind_text(stmt_it, 1, i->name, -1, NULL)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_bind_int(stmt_it, 2, i->major)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_bind_int(stmt_it, 3, i->minor)) {
				assert(false);
			}
			if (SQLITE_DONE != sqlite3_step(stmt_it)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_reset(stmt_it)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_bind_int64( stmt_de, 1, pkgid)) {
				assert(false);
			}
			if (SQLITE_DONE != sqlite3_step(stmt_de)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_reset(stmt_de)) {
				assert(false);
			}
		}
		if (SQLITE_OK != sqlite3_finalize(stmt_de)) {
			assert(false);
		}
	}
	if (SQLITE_OK != sqlite3_finalize(stmt_it)) {
		assert(false);
	}

	return rc;
}

int32_t
xpkg_registry_registerpkg(xpkg_registry_t reg, xpkg_pkg_t pkg) {
	int rc = XPKG_ERROR;
	int k, n, m;
	int64_t id;

	assert(reg);
	reg->consistent = false;
	k = add_packageentry(reg, pkg, &id);
	if (XPKG_OK == k) {
		DEBUG( "added package" );
		assert(pkg->files);
		n = add_files(reg, id, pkg->files);
		if (XPKG_OK == n) {
			DEBUG( "added files" );
			m = add_items(reg, id, pkg);
			if (XPKG_OK == m)
				DEBUG( "added items" );
			rc = XPKG_OK;
		}
	}
	return rc;
}

/* fill package with its associated files */
int32_t
get_files_for_pkg(xpkg_registry_t reg, xpkg_pkg_t pkg) {
	int rc = XPKG_OK;
	char *path, *chksum;
	int n, ftype;
	sqlite3_stmt *stmt;
	xpkg_file_t file;
	xpkg_files_t files;
	const char *sql = "SELECT PATH, FTYPE, CHKSUM FROM FILESPACKAGES WHERE NAME=:pkgname ORDER BY PATH";

	assert(reg);
	assert(pkg);
	files = files_create();
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql, -1, &stmt, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_text(stmt, 1, pkg->name, -1, NULL)) {
		assert(false);
	}

	for (;;) {
		n = sqlite3_step(stmt);
		if (SQLITE_ROW != n)
			break;
		DEBUG(sqlite3_errmsg(reg->db)); 
		path = (char*)sqlite3_column_text(stmt, 0);
		ftype = sqlite3_column_int(stmt, 1);
		chksum = (char*)sqlite3_column_text(stmt, 2);
		file = file_create(path, ftype, chksum);
		file_to_files(files, file);
	}
	if (SQLITE_DONE != n) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}
	DEBUG(sqlite3_errmsg(reg->db));
	pkg->files = files;

	return rc;
}

int32_t
xpkg_registry_querypkgs(xpkg_registry_t reg, const char *query, int mode, xpkg_pkgs_t *pkgs) {
	int n, count;
	xpkg_pkg_t p;
	xpkg_pkgs_t pkgs_out;
	sqlite3_stmt *stmt;
	const char *sql_a = "SELECT NAME, VERSION, REVISION, MAJOR, MINOR FROM PACKAGES;";
	const char *sql_n = "SELECT NAME, VERSION, REVISION, MAJOR, MINOR FROM PACKAGES WHERE NAME GLOB :pkgname;";
	const char *sql_p = "SELECT NAME, VERSION, REVISION, MAJOR, MINOR FROM FILESPACKAGES WHERE PATH GLOB :path;";

	DEBUG("querying db");
	assert(reg);
	switch (mode) {
		case XPKG_QUERYMODE_LISTALL:
			assert(NULL == query);
			if (SQLITE_OK != sqlite3_prepare(reg->db, sql_a, -1, &stmt, NULL)) {
				assert(false);
			}
			break;
		case XPKG_QUERYMODE_BYNAME:
			assert(query);
			if (SQLITE_OK != sqlite3_prepare(reg->db, sql_n, -1, &stmt, NULL)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_bind_text(stmt, 1, query, -1, NULL)) {
				assert(false);
			}
			break;
		case XPKG_QUERYMODE_BYPATH:
			assert(query);
			if (SQLITE_OK != sqlite3_prepare(reg->db, sql_p, -1, &stmt, NULL)) {
				assert(false);
			}
			if (SQLITE_OK != sqlite3_bind_text(stmt, 1, query, -1, NULL)) {
				assert(false);
			}
			break;
		default:
			assert(false);
	}

	count = 0;
	pkgs_out = pkgs_create();
	for (;;) {
		n = sqlite3_step(stmt);
		if(SQLITE_ROW != n)
			break;
		p = pkg_create((char*)sqlite3_column_text(stmt, 0), /* name */
				(char*)sqlite3_column_text(stmt, 1), /* version */
				sqlite3_column_int(stmt, 2), /* revision */
				sqlite3_column_int(stmt, 3), /* major */
				sqlite3_column_int(stmt, 4)); /* minor */
		if (XPKG_OK != get_files_for_pkg(reg, p)) {
			assert(false);
		}
		pkg_to_pkgs(pkgs_out, p);
		count++;
	}
	assert(SQLITE_DONE == n);
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}
	DEBUG(sqlite3_errmsg(reg->db));

	*pkgs = pkgs_out;
	return count;
}

int32_t
xpkg_registry_removepkg(xpkg_registry_t reg, const char *pkgname) {
	int rc = XPKG_OK;
	const char *sql = "DELETE FROM PACKAGES WHERE NAME=:name;";
	sqlite3_stmt* stmt;

	assert(reg);
	reg->consistent = false;
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql, -1, &stmt, NULL)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_bind_text(stmt, 1, pkgname, -1, NULL)) {
		assert(false);
	}
	if (SQLITE_DONE != sqlite3_step( stmt )) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}
	return rc;
}

int32_t
xpkg_registry_getstalefiles(xpkg_registry_t reg, xpkg_files_t *files) {
	int n, ftype, count;
	char *path, *chksum;
	xpkg_file_t file;
	xpkg_files_t f;
	sqlite3_stmt* stmt;
	const char *sql = "SELECT PATH, FTYPE, CHKSUM FROM STALEFILES ORDER BY PATH ASC;";

	assert(reg);

	DEBUG("searching stale files");
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql, -1, &stmt, NULL)) {
		assert(false);
	}
	count = 0;
	f = files_create();
	for (;;) {
		n = sqlite3_step(stmt);
		if (SQLITE_ROW != n)
			break;
		path = (char*)sqlite3_column_text(stmt, 0);
		DEBUG(path);
		ftype = sqlite3_column_int(stmt, 1);
		chksum = (char*)sqlite3_column_text(stmt, 2);
		file = file_create(path, ftype, chksum);
		file_to_files(f, file);
		count++;
	}
	assert(SQLITE_DONE == n);
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}

	return count;
}

int32_t
xpkg_registry_removestalefiles(xpkg_registry_t reg) {
	int rc = XPKG_OK;
	sqlite3_stmt* stmt;
	const char *sql = "DELETE FROM FILES WHERE ID IN (SELECT ID FROM STALEFILES);";

	assert(reg);
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql, -1, &stmt, NULL)) {
		assert(false);
	}
	if (SQLITE_DONE != sqlite3_step(stmt)) {
		assert(false);
	}
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}
	return rc;
}

int32_t
xpkg_registry_checkconsistence(xpkg_registry_t reg) {
	int rc = XPKG_ERROR;
	int n;
	const char *sql = "SELECT ITEM, MAJOR, MINOR FROM MISSING_DEPENDENCIES;";
	sqlite3_stmt* stmt;

	assert(reg);
	if (SQLITE_OK != sqlite3_prepare(reg->db, sql, -1, &stmt, NULL)) {
		assert(false);
	}
	n = sqlite3_step(stmt);
	if (SQLITE_ROW == n) {
		/* if there is something in this view, we miss some item(s) */
		rc = XPKG_CONSTRAINT_ERROR;
	} else {
		reg->consistent = true;
		assert(SQLITE_DONE == n);
		rc = XPKG_OK;
	}
	if (SQLITE_OK != sqlite3_finalize(stmt)) {
		assert(false);
	}

	return rc;
}

