--_
-- Copyright (c) 2005, Markus W. Weissmann <mww@opendarwin.org>
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in the
--    documentation and/or other materials provided with the distribution.
-- 3. Neither the name of OpenDarwin nor the names of its contributors
--    may be used to endorse or promote products derived from this software
--    without specific prior written permission.
--
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
-- ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
-- TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
-- PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
-- CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
-- EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
-- PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
-- OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
-- WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
-- OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
-- ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--
-- $Id: dbinit.sql,v 1.9 2006/02/16 18:41:33 mww Exp $
--

PRAGMA AUTO_VACUUM = 1;

BEGIN TRANSACTION CREATEDB;

CREATE TABLE OPTIONS (
	XPKGVERSION TEXT,
	CREATIONDATE TEXT
);

CREATE TABLE FILES (
	ID INTEGER PRIMARY KEY,
	PATH TEXT UNIQUE NOT NULL,
	FTYPE INTEGER NOT NULL,
	CHKSUM TEXT,
	CHECK(FTYPE>=0)
);

CREATE TABLE PACKAGES (
	ID INTEGER PRIMARY KEY,
	NAME TEXT UNIQUE NOT NULL,
	VERSION TEXT NOT NULL,
	REVISION INTEGER NOT NULL,
	MAJOR INTEGER NOT NULL,
	MINOR INTEGER NOT NULL,
	INSTALLDATE TEXT,
	CHECK(REVISION>=0),
	CHECK(MAJOR>=0),
	CHECK(MINOR>=0)
);

CREATE TABLE ITEMS (
	ID INTEGER PRIMARY KEY,
	NAME TEXT NOT NULL,
	MAJOR INTEGER NOT NULL,
	MINOR INTEGER NOT NULL,
	CHECK(MAJOR>=0),
	CHECK(MINOR>=0)
);

CREATE TABLE PROVIDES (
	PKGID INTEGER NOT NULL,		-- package id that provides
	ITEMID INTEGER NOT NULL,	-- item
	PRIMARY KEY(PKGID, ITEMID)
);

CREATE TABLE REQUIRES (
	PKGID INTEGER NOT NULL,		-- package id that requires
	ITEMID INTEGER NOT NULL,	-- item
	PRIMARY KEY( PKGID, ITEMID )
);

CREATE TABLE FILEOWNERSHIP (
	PKGID INTEGER NOT NULL,		-- package owns
	FILEID INTEGER NOT NULL,	-- file
	PRIMARY KEY( PKGID, FILEID )
);

CREATE VIEW PKGREQUIRES AS		-- all items that are required by some package
	SELECT P.ID AS ID, P.NAME AS NAME, I.NAME AS ITEM, I.MAJOR AS MAJOR, I.MINOR AS MINOR
		FROM PACKAGES AS P
		JOIN REQUIRES AS R ON P.ID=R.PKGID
		JOIN ITEMS AS I ON R.ITEMID=I.ID;

CREATE VIEW PKGPROVIDES AS		-- all items that are provided by some package
	SELECT P.ID AS ID, P.NAME AS NAME, I.NAME AS ITEM, I.MAJOR AS MAJOR, I.MINOR AS MINOR
		FROM PACKAGES AS P
		JOIN PROVIDES AS R ON P.ID=R.PKGID
		JOIN ITEMS AS I ON R.ITEMID=I.ID;

CREATE VIEW PKGINUSE AS			-- all items that are provided & are required
	SELECT P.ID AS ID, P.NAME AS NAME, P.ITEM AS ITEM, P.MAJOR AS MAJOR, P.MINOR AS MINOR
		FROM PKGPROVIDES AS P
		JOIN PKGREQUIRES AS R ON P.ITEM=R.ITEM
		WHERE P.MAJOR=R.MAJOR AND P.MINOR>=R.MINOR;

CREATE VIEW MISSING_DEPENDENCIES AS	-- all items that are required but not provided
	SELECT ID, NAME, ITEM, MAJOR, MINOR
		FROM PKGREQUIRES AS R
		WHERE NOT EXISTS (
			SELECT ID FROM PKGPROVIDES AS P
				WHERE P.ITEM=R.ITEM AND P.MAJOR=R.MAJOR AND P.MINOR>=R.MINOR );

-- remove left over items & relations, NOT files
CREATE TRIGGER DELETE_CASCADE_PACKAGE AFTER DELETE ON PACKAGES FOR EACH ROW
	BEGIN
		DELETE FROM FILEOWNERSHIP WHERE FILEOWNERSHIP.PKGID=OLD.ID;
		DELETE FROM PROVIDES WHERE PROVIDES.PKGID=OLD.ID;
		DELETE FROM REQUIRES WHERE REQUIRES.PKGID=OLD.ID;
		DELETE FROM ITEMS WHERE
			NOT EXISTS ( SELECT ITEMID FROM PROVIDES AS P WHERE P.ITEMID=ITEMS.ID ) AND
			NOT EXISTS ( SELECT ITEMID FROM REQUIRES AS R WHERE R.ITEMID=ITEMS.ID );
	END;

CREATE TRIGGER INSERT_CURRENTDATE_PACKAGE AFTER INSERT ON PACKAGES FOR EACH ROW
	BEGIN
		UPDATE PACKAGES SET INSTALLDATE=CURRENT_TIMESTAMP WHERE ID=NEW.ID;
	END;

CREATE VIEW FILESPACKAGES AS
	SELECT P.NAME AS NAME, P.VERSION AS VERSION, P.REVISION AS REVISION, P.MAJOR AS MAJOR, P.MINOR AS MINOR, F.PATH AS PATH, F.FTYPE AS FTYPE, F.CHKSUM AS CHKSUM
		FROM FILES AS F
		JOIN FILEOWNERSHIP AS O ON F.ID=O.FILEID
		JOIN PACKAGES AS P ON O.PKGID=P.ID;

CREATE VIEW STALEFILES AS
	SELECT ID, PATH, FTYPE, CHKSUM FROM FILES
		WHERE ID NOT IN (SELECT FILEID FROM FILEOWNERSHIP);

END TRANSACTION CREATEDB;

