/* normalmode.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2009 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "normalmode.h"
#include "worker_locale.h"
#include "wconfig.h"
#include "worker.h"
#include <aguix/bevelbox.h>
#include "wpucontext.h"
#include "execlass.h"
#include "configheader.h"
#include "configparser.tab.h"
#include "worker_shm.h"
#include "nmrowdata.h"
#include "basic_actions.h"
#include "worker_filefunc.h"
#include <aguix/kartei.h>
#include "startprogop.h"
#include <stdarg.h>
#include <aguix/acontainer.h>
#include <aguix/acontainerbb.h>
#include "wcfiletype.hh"
#include "copyopwin.hh"
#include "nwc_fsentry.hh"
#include <aguix/popupmenu.hh>
#include "run_custom_action.hh"
#include "filenameshrinker.hh"
#include "bookmarkdbentry.hh"
#include "nmbookmarkcb.hh"
#include "bookmarkdbproxy.hh"
#include "datei.h"
#include "deleteop.h"
#include "copyop.h"
#include "nmcacheentry.hh"
#include "nmcopyopdir.hh"
#include "nmextlist.hh"
#include "nmspecialsourceext.hh"
#include <algorithm>
#include <aguix/karteibutton.h>
#include "nwc_path.hh"

const char *NormalMode::type="NormalMode";

int NormalMode::maxfilestrlen=1;
int NormalMode::maxdirstrlen=1;
int NormalMode::maxbytesstrlen=1;
int NormalMode::maxwbytesstrlen=1;
int NormalMode::maxnochanges=0;

static void *NM_slavestart( void *arg )
{
  NormalMode *nm = (NormalMode*)arg;
  nm->slavehandler();
  return NULL;
}

NormalMode::NormalMode(Lister *tparent):ListerMode(tparent)
{
  ce=NULL;
  cache=new List();
  req=new Requester(aguix);
  searchmodeon=false;
  lastsearch=NULL;
  sortmode=SORT_NAME;
  lv=NULL;
  lastrow=-1;
  showfreespace=true;
  freespacestr = NULL;
  lastfsupdate=0;
  updatetime=10;
  parentlister->getWorker()->PS_setLifetime( (double)updatetime );
  ssh_allow = SSH_ASK;

  lasteagain = false;
  
  // to preventing slave access
  reclist = new ft_rec_list();
  recreslist = new ft_recres_list();

  slave.filetypes = new List();
  slave.filetype_ex.lock();
  slave.running = 0;
  slave.status = THREAD_NOP;
  slave.order = THREAD_REINIT;
  if ( pthread_create( &slave.th, NULL, NM_slavestart, this ) != 0 ) {
    fprintf( stderr, "Worker: Can't create thread! Aborting\n" );
    exit( EXIT_FAILURE );
  }

  slave.filetype_ex.wait();
  ft_list_update();
  slave.filetype_ex.signal();
  slave.filetype_ex.unlock();
  
  chownUserList = NULL;
  chownGroupList = NULL;

  visChanged = true;
  oldlvy = -1;
  
  busyFlag = 0;
  lastBusyFlagUpdate = time( NULL );

  quicksearch_enabled = true;
  _filtered_search_enabled = true;
  m_filtered_search_active = false;

  _lv_ids_serial = -1;
  
  m_last_statebar_update = (time_t)0;

  m_bookmarks_has_been_changed = false;
  m_bookmark_cb = new NMBookmarkCB( this );
  Worker::getBookmarkDBInstance().registerChangeCallback( m_bookmark_cb );

  m_cont = NULL;
  m_cont2 = NULL;
  m_tab_b = NULL;
  m_tab_new = NULL;
  m_tab_close = NULL;

  m_off_store.active_tab = -1;

  m_shrinker = RefCount<TextShrinker>( new HalfShrinker() );
  m_filtered_search_infix_search = false;
}

NormalMode::~NormalMode()
{
  switchOrder( THREAD_EXIT );
  pthread_join( slave.th, NULL );
  ft_list_clear();
  delete slave.filetypes;
  ft_rec_list_clear();
  ft_recres_list_clear();
  delete reclist;
  delete recreslist;

  if(searchmodeon==true) finishsearchmode();
  deleteCache();
  delete cache;
  delete req;
  if ( freespacestr != NULL ) _freesafe( freespacestr );

  Worker::getBookmarkDBInstance().unregisterChangeCallback( m_bookmark_cb );
  delete m_bookmark_cb;
}

void NormalMode::messageHandler(AGMessage *msg)
{
  bool ma=false;
  switch(msg->type) {
    case AG_STRINGGADGET_OK:
      if(searchmodeon==false) {
        if(msg->stringgadget.sg==sg) {
	  //TODO: better not replacing env vars at all here?
	  std::string tstr1 = sg->getText();
	  if ( tstr1[0] == '$' ) {
	    char *tstr2 = HandlePathExt( tstr1.c_str() );
	    tstr1 = tstr2;
	    _freesafe( tstr2 );
	  }
          enterDir( tstr1.c_str() );
        }
        break;
      }
    case AG_STRINGGADGET_CANCEL:
      if(searchmodeon==false) {
        if(msg->stringgadget.sg==sg) {
          if ( getCurrentDir() != NULL ) sg->setText( getCurrentDir() );
        }
        break;
      }
    case AG_STRINGGADGET_DEACTIVATE:
      if(searchmodeon==true) {
        finishsearchmode();
      }
      break;
    case AG_STRINGGADGET_CONTENTCHANGE:
      if ( searchmodeon == true ) {
          if ( shownextentry( sg->getText(), false, true ) == true ) {
              if ( lastsearch != NULL )
                  _freesafe( lastsearch );
              else
                  debugmsg("oops1\n");
              lastsearch = dupstring( sg->getText() );
          } else {
              sg->setText( lastsearch );
          }
      }
      break;
    case AG_SIZECHANGED:
      break;
    case AG_STRINGGADGET_ACTIVATE:
      if(msg->stringgadget.sg==sg) ma=true;
      break;
    case AG_BUTTONCLICKED:
      if(msg->button.button==hb[0]) {
        showcache(-1);
        ma=true;
      } else if(msg->button.button==hb[1]) { 
        showcache(1);
        ma=true;
      } else if ( msg->button.button == parentb ) { 
        parent();
        ma = true;
      } else if ( msg->button.button == m_tab_new ) {
          newTab();
      } else if ( msg->button.button == m_tab_close ) {
          closeCurrentTab();
      }
      break;
    case AG_FIELDLV_PRESSED:
      if(msg->fieldlv.lv==lv) ma=true;
      break;
    case AG_FIELDLV_ONESELECT:
      if(msg->fieldlv.lv==lv) {
        if(lastrow==-1) {
          lastrow=msg->fieldlv.row;
          gettimeofday(&lastclick,NULL);
        } else {
          if(msg->fieldlv.row==lastrow) {
            struct timeval acttime;
            gettimeofday(&acttime,NULL);
            if ( aguix->isDoubleClick( &acttime, &lastclick ) == true ) {
              //TODO: Wir haben einen Doppelclick
              // Aus lvc->data==fe->nr bekommen wir das fe (nr beschreibt Position)
              if(ce!=NULL) {
                FileEntry *tfe = getFEForRow( lastrow );
                startAction(tfe);
              }
              // when had a doubleclick the next click on the same should not be a
              // doubleclick again
              lastrow=-1;
            } else {
              lastrow=msg->fieldlv.row;
              gettimeofday(&lastclick,NULL);
            }
          } else {
            lastrow=msg->fieldlv.row;
            gettimeofday(&lastclick,NULL);
          }
        }
      }
      break;
    case AG_FIELDLV_HEADERCLICKED:
      if ( ( msg->fieldlv.lv == lv ) &&
           ( msg->fieldlv.button == Button1 ) ) {
        changeSortModeForField( msg->fieldlv.row );
      }
      break;
    case AG_FIELDLV_ENTRY_PRESSED:
        if ( msg->fieldlv.lv == lv ) {
            lv_pressed( msg->fieldlv.row );
        }
        break;
    case AG_POPUPMENU_CLICKED:
        if ( msg->popupmenu.menu == m_current_popup_settings.lv_popup_menu ) {
            startLVPopUpAction( msg );
        } else if ( msg->popupmenu.menu == m_current_popup_settings.label_popup_menu.get() ) {
            handleLabelPopUp( msg );
        }
        break;
    case AG_KEYPRESSED:
        if ( parentlister->isActive() == true ) {
            if ( msg->key.key == XK_Prior && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                switchToTabDirection( -1 );
            } else if ( msg->key.key == XK_Next && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                switchToTabDirection( 1 );
            } else if ( msg->key.key == XK_t && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                newTab();
            } else if ( msg->key.key == XK_w && KEYSTATEMASK( msg->key.keystate ) == ControlMask ) {
                closeCurrentTab();
            } else if ( searchmodeon == false &&
                        msg->key.keybuf != NULL &&
                        quicksearch_enabled == true ) {
                // key press msg only arrives if a string gadget doesn't have the focus
                // but it won't hurt to check for active search mode
                if ( strlen( msg->key.keybuf ) > 0 ) {
                    //TODO mh, this is not the best way to set case insensitive
                    //     but currently it's the easiest way
                    //     an argument would be better for sure
                    setEntrySearchCaseSensitive( false );
                    shownextentry( msg->key.keybuf );
                }
            }
        }
        break;
    case AG_KARTEIBUTTONCLICKED:
        if ( msg->karteibutton.karteibutton == m_tab_b ) {
            ma = true;
            switchToTab( msg->karteibutton.option );
        }
        break;
    default:
        break;
  }
  if(ma==true) parentlister->makeActive();

  //TODO: Brauche ich das hier noch? So richtig Sinn macht es nicht
  //showFreeSpace( false );
}

void NormalMode::on()
{
  int side=parentlister->getSide();
  AGUIXFont *afont=aguix->getFont(wconfig->getFont(2+side));
  int sgh = ( ( afont != NULL ) ? afont->getCharHeight() : aguix->getCharHeight() ) +8;
  int hbw = aguix->getTextWidth( "<", afont ) + 10;
  int m;

  m_cont = new AContainer( parentawindow, 1, 3 );
  m_cont->setBorderWidth( 0 );
  m_cont->setMinSpace( 0 );
  m_cont->setMaxSpace( 0 );
  parentlister->setContainer( m_cont );

  m_cont2 = m_cont->add( new AContainer( parentawindow, 4, 1 ), 0, 2 );
  m_cont2->setBorderWidth( 0 );
  m_cont2->setMinSpace( 0 );
  m_cont2->setMaxSpace( 0 );

  parentb = new Button( aguix, 0, 0, aguix->getTextWidth( "..", afont ) + 10, "..", 1, 0, 5 );
  hb[0] = new Button( aguix, 0, 0, hbw, "<", 1, 0, 2 );
  hb[1] = new Button( aguix, 0, 0, hbw, ">", 1, 0, 3 );

  m_tab_cont = m_cont->add( new AContainer( parentawindow, 3, 1 ), 0, 0 );
  m_tab_cont->setBorderWidth( 0 );
  m_tab_cont->setMinSpace( 0 );
  m_tab_cont->setMaxSpace( 0 );

  m_tab_entries.clearEntries();
  m_tab_b = (KarteiButton*)m_tab_cont->add( new KarteiButton( aguix,
                                                              0, 0,
                                                              100, 1, 0, 0 ), 1, 0, AContainer::ACONT_MINH + AContainer::ACONT_MAXH );
  m_tab_b->setAcceptFocus( false );
  addTabOption( "" );

  m_tab_b->setTextShrinker( m_shrinker );
  m_tab_b->setFont( wconfig->getFont( 2 + side ) );

  m_tab_new = (Button*)m_tab_cont->add( new Button( aguix, 0, 0,
                                                    "N", 1, 0, 0 ), 0, 0, AContainer::ACONT_MINW + AContainer::ACONT_MAXW );
  m_tab_new->setAcceptFocus( false );
  m_tab_new->setFont( wconfig->getFont( 2 + side ) );
  m_tab_new->resize( aguix->getTextWidth( "N", afont ) + 10, m_tab_new->getHeight() );

  m_tab_close = (Button*)m_tab_cont->add( new Button( aguix, 0, 0,
                                                    "X", 1, 0, 0 ), 2, 0, AContainer::ACONT_MINW + AContainer::ACONT_MAXW );
  m_tab_close->setAcceptFocus( false );
  m_tab_close->setFont( wconfig->getFont( 2 + side ) );
  m_tab_close->resize( aguix->getTextWidth( "X", afont ) + 10, m_tab_new->getHeight() );

  m_tab_cont->readLimits();

  lv = (NMFieldListViewDND*)m_cont->add( new NMFieldListViewDND( aguix,
                                                                 0, 0,
                                                                 50, 50,
                                                                 1 ), 0, 1 );
  lv->setAcceptFocus( false );
  lv->setMBG(wconfig->getListerBG());
  m=(wconfig->getHBarTop(side)==true)?1:2;
  lv->setHBarState(m);
  m=(wconfig->getVBarLeft(side)==true)?1:2;
  lv->setVBarState(m);
  m=wconfig->getHBarHeight(side);
  lv->setHBarHeight(m);
  m=wconfig->getVBarWidth(side);
  lv->setVBarWidth(m);
  lv->setShowHeader( wconfig->getShowHeader( side ) );
  lv->setFont(wconfig->getFont(2+side));
  lv->setHeaderFG( wconfig->getLVHeader( 0 ) );
  lv->setHeaderBG( wconfig->getLVHeader( 1 ) );
  
  parentb->resize( parentb->getWidth(), sgh );
  m_cont2->add( parentb, ( side == 0 ) ? 0 : 3, 0, AContainer::CFIX );
  parentb->setAcceptFocus( false );
  parentb->setFont(wconfig->getFont(2+side));
  
  hb[0]->resize( hb[0]->getWidth(), sgh );
  m_cont2->add( hb[0], ( side == 0 ) ? 1 : 0, 0, AContainer::CFIX );
  hb[0]->setAcceptFocus( false );
  hb[0]->setFont(wconfig->getFont(2+side));
  
  hb[1]->resize( hb[1]->getWidth(), sgh );
  m_cont2->add( hb[1], ( side == 0 ) ? 3 : 2, 0, AContainer::CFIX );
  hb[1]->setAcceptFocus( false );
  hb[1]->setFont(wconfig->getFont(2+side));

  sg = (StringGadget*)m_cont2->add( new StringGadget( aguix,
                                                      0,
                                                      0,
                                                      50, "", 4 ),
                                    ( side == 0 ) ? 2 : 1, 0 );
  sg->setAcceptFocus( false );
  sg->resize( sg->getWidth(), sgh );
  sg->setFont(wconfig->getFont(2+side));

  parentlister->setActiveMode(this);
  setName();
  lv->setSelectHandler(this);

  setupLVFields();
  
  if ( m_off_store.list_of_used_tabs.size() > 0 ) {
      std::list<std::string>::iterator it1 = m_off_store.list_of_used_tabs.begin();
      if ( it1 != m_off_store.list_of_used_tabs.end() ) {
          setTabOption( 0, *it1 );
          it1++;
      }
      for ( ; it1 != m_off_store.list_of_used_tabs.end(); it1++ ) {
          addTabOption( *it1 );
      }

      if ( m_off_store.active_tab >= 0 ) {
          switchToTab( m_off_store.active_tab );
      }
  }

  parentawindow->updateCont();
}

void NormalMode::off()
{
  finishsearchmode();

  // backup field width for later reuse or save
  writeFieldWidthsToLoadMap();

  // store the current pos for reactivating
  if ( ( ce != NULL ) && ( lv != NULL ) ) {
      ce->setPos( lv->getYOffset() );
      ce->setXPos( lv->getXOffset() );
      ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );

      m_off_store.active_tab = m_tab_b->getSelectedOption();
      m_off_store.list_of_used_tabs.clear();
      for ( int i = 0; i < m_tab_b->getNrOfOptions(); i++ ) {
          m_off_store.list_of_used_tabs.push_back( m_tab_entries.getEntry( i ) );
      }
  }
  ce = NULL;

  freePopUpMenus();
  
  delete lv;
  lv = NULL;
  delete hb[0];
  hb[0] = NULL;
  delete hb[1];
  hb[1] = NULL;
  delete sg;
  sg = NULL;
  delete parentb;
  parentb = NULL;

  parentlister->setContainer( NULL );
  delete m_cont;

  m_cont = m_cont2 = NULL;

  parentlister->setActiveMode(NULL);
  parentlister->setName("");
  if ( parentlister->getFocus() == true )
    parentlister->getWorker()->setTitle(NULL);
}

void NormalMode::activate()
{
  int row;

  if ( ce != NULL ) {
      if ( ce->getActiveFE() != NULL ) {
          row = 0;
          while ( row < lv->getElements() ) {
              if ( lv->getData( row ) == ce->getActiveFE()->getID() ) break;
              row++;
          }
          if( row < lv->getElements() ) lv->setActiveRow( row );
      }
  }
  showCacheState();
  parentlister->getWorker()->setTitle( sg->getText() );
}

void NormalMode::deactivate()
{
  int row=lv->getActiveRow();
  if ( row >= 0 ) {
    lv->setActiveRow( -1);
  }
  finishsearchmode();
  parentlister->getWorker()->setTitle(NULL);
}

void NormalMode::buildListView()
{
  if ( ce == NULL ) return;
  if ( lv == NULL ) return;

  int side = parentlister->getSide();
  const std::vector<WorkerTypes::listcol_t> *dis;
  int nr_of_entries = ce->getNrOfFiles( 0 ) + ce->getNrOfDirs( 0 ) + 1;

  lv->setSize( nr_of_entries );
  lv->setSizeDNDText( nr_of_entries );
  lv->setActiveRowQ( -1 );

  dis = wconfig->getVisCols( side );

  int tpos = 0;

  for ( Verzeichnis::verz_it it1 = ce->begin();
        it1 != ce->end();
        it1++ ) {
      FileEntry *fe = *it1;
      if ( fe->use == true ) {
          fe->setLVC_DND( lv, tpos, dis );
          lv->setData( tpos, fe->getID());
          if ( fe->select == true ) lv->setSelectQ( tpos, true );
          else lv->setSelectQ( tpos, false );
          if ( fe == ce->getActiveFE() ) {
              // only make the lvc active if we are active
              if ( parentlister->isActive() == true ) lv->setActiveRowQ( tpos );
          }
          tpos++;
      }
  }

  _lv_ids_serial = ce->getSerialOfIDs();

  if ( tpos != nr_of_entries ) {
      fprintf( stderr, "Worker error: Nr of entries does not match real value!\n" );
      //TODO remove potential remaining entries?
  }
  
  lv->redraw();
  
  visChanged = true;
}

void NormalMode::deleteCache()
{
  NMCacheEntry *tce=(NMCacheEntry*)cache->getFirstElement();
  while(tce!=NULL) {
    delete tce;
    tce=(NMCacheEntry*)cache->getNextElement();
  }
}

bool NormalMode::checkEnterDir( const char *dir )
{
    std::string str1;
    bool allow_entering = true;
    
#ifdef HAVE_AVFS
    if ( strncmp( dir, "/#ssh:", 6 ) == 0 && ssh_allow != SSH_ALWAYS ) {
        if ( ssh_allow == SSH_NEVER ) {
            allow_entering = false;
        } else {
            bool var1;
            int erg;
            
            // show requester
            str1 = catalog.getLocale( 629 );
            str1 += "|";
            str1 += catalog.getLocale( 8 );
            
            var1 = false;
            
            erg = request_choose( catalog.getLocale( 123 ),
                                  catalog.getLocale( 630 ),
                                  catalog.getLocale( 631 ), var1,
                                  str1.c_str() );
            if ( var1 == true ) {
                if ( erg != 0 ) {
                    ssh_allow = SSH_NEVER;
                } else {
                    ssh_allow = SSH_ALWAYS;
                }
            }
            
            if ( erg != 0 ) {
                allow_entering = false;
            }
        }
    }
#endif

    if ( allow_entering == true ) {
        if ( worker_access( dir, R_OK | X_OK ) != 0 ) {
            char *tstr2;
            const char *myerror;
            
            // show requester
            myerror = strerror( errno );
            tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 532 ) ) +
                                       strlen( dir ) +
                                       strlen( myerror ) + 1 );
            sprintf( tstr2, catalog.getLocale( 532 ), dir, myerror );
            request( catalog.getLocale( 347 ), tstr2, catalog.getLocale( 11 ) );
            _freesafe( tstr2 );
            
            allow_entering = false;
        }
    }

    return allow_entering;
}

int NormalMode::enterDir(const char *dirstr)
{
    char *tstr;
    std::auto_ptr<Verzeichnis> tverz;
    NMCacheEntry *tce = NULL;
    int comperg;
    bool reset;
    int returnvalue = 0;  /* 0 = OK
                           * 1 = Not found
                           * 2 = not found, but lv cleared (was actually)
                           */
    std::string str1;
    bool find_new_active_row = false;
  
    if ( lv == NULL ) return 1;

    if ( dirstr == NULL ) return 1;

    parentlister->getWorker()->setWaitCursor();
    aguix->Flush();

    // only reset string filter if not entering same directory
    if ( ce == NULL || strcmp( ce->getDir(), dirstr ) != 0 ) {
        setEntrySearchString( "", false );
    }
    
    // normalize given dir
    if ( dirstr[0] != '/' ) {
        // resolve current dir
        char *cwd = HandlePathExt( "." );
        str1 = cwd;
        str1 += "/";
        str1 += dirstr;
        
        _freesafe( cwd );
        tstr = HandlePath( str1.c_str() );
    } else {
        tstr = HandlePath( dirstr );
    }
    
    // check whether it is okay to enter the dir
    if ( checkEnterDir( tstr ) == false ) {
        // restore stringgadget
        if ( ce != NULL ) {
            _freesafe( tstr );
            tstr = HandlePath( ce->getDir() );
            sg->setText( tstr );
        } else sg->setText( "" );
      
        _freesafe( tstr );
        parentlister->getWorker()->unsetWaitCursor();
        return 1;
    }
    
    finishsearchmode();
    
    // clear the reclist so no wrong pointers are in the list
    ft_rec_list_clear();
    
    // search for existing cache entry
    if ( cache->size() > 0 ) {
        tce = (NMCacheEntry*)cache->getFirstElement();
        while ( tce != NULL ) {
            if ( strcmp( tce->getDir(), tstr ) == 0 ) {
                break;
            }
            tce = (NMCacheEntry*)cache->getNextElement();
        }
    }
    
    // read directory
    tverz = std::auto_ptr<Verzeichnis>( new Verzeichnis() );
    if ( tverz->readDir( tstr ) != 0 ) {
        tverz.release();
    }
    
    if ( tverz.get() != NULL ) {
        if ( tce == NULL ) {
            // no cache entry found
            while ( cache->size() >= (int)wconfig->getCacheSize() ) {
                //TODO maybe use max cache size plus number of tabs as limit
                tce = findOldestCacheEntryNotInTab();
                if ( tce == NULL ) break;

                cache->removeElement( tce );
                delete tce;
                if ( ce == tce ) ce = NULL;
            }
            tce = new NMCacheEntry( tverz,
                                    &_dir_filter_sets,
                                    &_dir_sort_sets,
                                    &m_dir_bookmarks_sets );
            cache->addElement( tce );
            tce->restartcheck();

            if ( tce->dirOpened() == true ) {
                if ( tce->begin() != tce->end() ) {
                    tce->setActiveFE( *( tce->begin() ) );
                }
            }
            tce->checkfordcd();
        } else {
            // put tce at the end of the cache to realize Not-Recently-Used
            cache->removeElement(tce);
            cache->addElement(tce);
            
            // insert old values into to new tverz
            // first sort both for names so we can easily (and fast) get old values
            // set new sort mode so getFiles will init resort
            int old_sort_mode = _dir_sort_sets.getSortMode();
            _dir_sort_sets.setSortMode( SORT_NAME | SORT_DIRMIXED );

            tverz->sort( SORT_NAME | SORT_DIRMIXED );
            
            Verzeichnis::verz_it old_it1 = tce->begin();
            Verzeichnis::verz_it new_it1 = tverz->begin();

            FileEntry *old_activefe = tce->getActiveFE();
            FileEntry *new_activefe = NULL;
            
            while ( new_it1 != tverz->end() ) {
                FileEntry *nfe = *new_it1;
                if ( old_activefe != NULL )
                    if ( strcmp( nfe->fullname, old_activefe->fullname ) == 0 )
                        new_activefe = nfe;
                while ( old_it1 != tce->end() ) {
                    FileEntry *ofe = *old_it1;
                    comperg = strcmp( ofe->fullname, nfe->fullname );
                    if ( comperg > 0 ) break;
                    else if ( comperg == 0 ) {
                        // found an entry matching the current
                        // now update values

                        if ( ofe->select == true )
                            nfe->select = true;
                        else
                            nfe->select = false;

                        if ( ofe->isDir() == true )
                            if ( ofe->dirsize >= 0 )
                                nfe->dirsize = ofe->dirsize;

                        if ( ofe->filetype != NULL )
                            nfe->filetype = ofe->filetype;

                        nfe->setColor( ofe->getColor() );
                    }
                    old_it1++;
                    if ( strcmp( nfe->name, ".." ) == 0 )
                        break; /* the ".." is always first in new and old so stop here
                                * because following elements can be "smaller" then ".."
                                * f.i. "#343" < ".."
                                */
                }

                if ( old_it1 == tce->end() )
                    break; // there are no further old files => no need to continue

                new_it1++;
            }
            
            if ( new_activefe == NULL && old_activefe != NULL ) {
                // corresponding active entry wasn't found
                find_new_active_row = true;
            }
            
            // update done
            // reset original sort mode
            _dir_sort_sets.setSortMode( old_sort_mode );

            // now set new new Verzechnis
            tce->setVerz( tverz );
            // and active FE
            tce->setActiveFE( new_activefe );
        }

        if ( tce->getActiveFE() != NULL ) {
            if ( tce->getActiveFE()->use == false ) {
                // this will never happen because the fallback is
                // the first entry in the filelist which is always ".."
                // but this doesn't hurt
                tce->setActiveFE( NULL );
            }
        }

        if ( ce != NULL ) {
            ce->setPos( lv->getYOffset() );
            ce->setXPos( lv->getXOffset() );
            ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );
            ce->setLastActiveRow( lv->getActiveRow() );
        }

        ce = tce;
        rebuildView();

        if ( find_new_active_row == true ) {
            // find used entry around old_act_row

            int new_row = ce->getLastActiveRow() - 1;
            while ( lv->isValidRow( new_row ) == false && new_row > 0 ) {
                new_row--;
            }
            
            makeRowActive( new_row );
        }

        lv->setYOffset( ce->getPos() );
        lv->setXOffset( ce->getXPos() );
        if ( ce->getActiveIsVisible() == true &&
             lv->isRowVisible( lv->getActiveRow() ) == false ) {
            lv->centerActive();
        }
        sg->setText( tstr );
    } else { /* tverz == NULL */
        // can't read the dir
        returnvalue = 1;
        reset = true;
        if ( tce != NULL ) {
            //TODO: Verzeichnis gibt es nicht mehr, also aus Cache entfernen
            // klappt jetzt so
            if ( tce == ce ) {
                // it's the current cacheentry
                reset = false;
                returnvalue = 2;
                cache->removeElement( ce );
                delete ce;
                ce = NULL;
                lv->setSize( 0 );
                lv->redraw();
                sg->setText( "" );
                updateTabs();
            } else {
                cache->removeElement( tce );
                delete tce;
            }
        }
        if ( reset == true ) {
            // wiederherstellen
            if ( ce != NULL ) {
                _freesafe( tstr );
                tstr = HandlePath( ce->getDir() );
                sg->setText( tstr );
            } else sg->setText( "" );
        }
    }
    _freesafe( tstr );
    
    showFreeSpace( true );
    
    if ( parentlister->getFocus() == true )
        parentlister->getWorker()->setTitle( sg->getText() );
    parentlister->getWorker()->unsetWaitCursor();

    return returnvalue;
}

void NormalMode::showcache(int direction)
{
  NMCacheEntry *tce;
  int cs=cache->size();
  int i=0,id;
  if(cs<1) return;

  finishsearchmode();
  
  if(ce!=NULL) {
    ce->setPos( lv->getYOffset() );
    ce->setXPos( lv->getXOffset() );
    ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );
  }
  int t = cache->getIndex( ce );
  if ( t < 0 ) t = 0;
  
  id=cache->initEnum();
  do {
    i++;
    t+=direction;
    if(t>=cache->size()) t=0;
    else if(t<0) t=cache->size()-1;
    tce=(NMCacheEntry*)cache->getElementAt(id,t);
    if(tce!=NULL) {
      if(enterDir(tce->getDir())==0) {
        break;
      }
    }
  } while(i<cs);
  cache->closeEnum(id);
}

void NormalMode::selhandler(FieldListView *tlv, int row)
{
  if(ce==NULL) return;
  if ( ce->dirOpened() == false ) return;
  FileEntry *fe;
  fe = getFEForRow( row );

  if(fe==NULL) return;

  bool state=lv->getSelect( row );
  bool oldstate=fe->select;
  
  if(state!=oldstate) {
    // Aenderung
    
    if( row == 0 ) {
      lv->setSelect( row, false );
    } else {
      if(state==true) {
          // Aus -> An
          ce->updateStatEntrySelected( fe );
      } else {
          // An -> Aus
          ce->updateStatEntryDeselected( fe );
      }
      fe->select=state;
    }
  }
  showCacheState();
  if ( lv->getActiveRow() == row ) {
    ce->setActiveFE( fe );
  }
}

NMFieldListViewDND::NMFieldListViewDND(AGUIX *taguix,int tx,int ty,int width,int height,int tdata):FieldListViewDND(taguix,tx,ty,width,height,tdata)
{
  nm=NULL;
}

NMFieldListViewDND::~NMFieldListViewDND()
{
}

void NMFieldListViewDND::setSelectHandler(NormalMode *nnm)
{
  nm=nnm;
}

void NMFieldListViewDND::runSelectHandler( int row )
{
  if(nm!=NULL) nm->selhandler(this,row );
}

void NormalMode::showCacheState()
{
  if(parentlister->isActive()==false) return;  // do nothing when we are not active
  if ( ce == NULL ) {
    // no directory but at least clear statebar
    parentlister->setStatebarText( "" );
    return;
  }
  const char *tstr1=catalog.getLocale(107),*tstr2=catalog.getLocale(108);
  const char *tstr3=catalog.getLocale(13);
  char *str;
  std::string buf1, buf2, buf3, buf4;
  bool human_strings = false;
  loff_t l1,l2;
  int l,tl;
  char *formatstr;
  
  l1 = ce->getSizeOfFiles( 1 ) + ce->getSizeOfDirs( 1 );
  l2 = ce->getSizeOfFiles( 0 ) + ce->getSizeOfDirs( 0 );

  buf3 = AGUIXUtils::bytes_to_human_readable( l1, 10 );
  buf4 = AGUIXUtils::bytes_to_human_readable( l2, 10 );

  if ( l2 > 10 * 1024 ) {
      human_strings = true;
  }

  MakeLong2NiceStr( l1, buf1 );
  MakeLong2NiceStr( l2, buf2 );
  
  // wenn mehr als 100 mal nichts an diesen Variablen
  // geaendert wurde, jeweils um 1 verringern, so dass
  // nicht ewig zu viel Platz verwendet wird, nur weil
  // einmal eine besonders grosse Datei da war
  if(maxnochanges>100) {
    maxnochanges=0;
    // auf >0 muss nicht geprueft werden, da danach sowieso nochmal geprueft
    // wird
    maxfilestrlen--;
    maxdirstrlen--;
    maxbytesstrlen--;
    maxwbytesstrlen--;
  }

  tl=0;
  l = (int)( log10( (double)ce->getNrOfFiles( 0 ) + 1 ) + 1.0 );
  if(l>maxfilestrlen) {
    maxfilestrlen=l;
    tl++;
  }
  l = (int)( log10( (double)ce->getNrOfDirs( 0 ) + 1 ) +1.0 );
  if(l>maxdirstrlen) {
    maxdirstrlen=l;
    tl++;
  }
  l = buf2.length();
  if(l>maxbytesstrlen) {
    maxbytesstrlen=l;
    tl++;
  }
  
  // we have to check for buf3 and buf4 because this is possibly:
  //  1200 KB / 100 MB
  // the left part takes more space
  l = buf3.length();
  if(l>maxwbytesstrlen) {
    maxwbytesstrlen=l;
    tl++;
  }
  l = buf4.length();
  if(l>maxwbytesstrlen) {
    maxwbytesstrlen=l;
    tl++;
  }

  // this are the "double" format strings for the statebar
  // the first is when we will also print some nice byte-values (in B, KB or MB)
  // the second otherwise
  // they have to be same length and modifiers
  // from this strings the real format string is build according to the needed string/number lengths
  // NOTE: the two last %s in _SLIM will be replaced by spaces
  //       this is needed to keep the string at the same place in the statebar when switching between listers
#define STATEBAR_FORMAT      "[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds   (%%%ds / %%%ds) ]"
#define STATEBAR_FORMAT_SLIM "[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds]   %%%ds   %%%ds   "

  formatstr = (char*)_allocsafe( strlen( STATEBAR_FORMAT ) +
                                 8 * ( A_BYTESFORNUMBER( maxfilestrlen ) ) + 1 );
  // for a 4 byte value I need 11 decimals
  // there will be 8 replacements

  str = (char*)_allocsafe( sizeof( char ) *
                           ( strlen( STATEBAR_FORMAT ) +     // space of the format string
                             strlen( tstr1 ) +               // length of string for "files" (first %s)
                             strlen( tstr2 ) +               // length of string for "dirs" (second %s)
                             strlen( tstr3 ) +               // length of string for "bytes" (third %s)
                             2 * maxfilestrlen +             // space needed for "sel files / files"
                             2 * maxdirstrlen +              // space needed for "sel dirs / dirs"
                             2 * maxbytesstrlen +            // space needed for "sel bytes / bytes"
                             2 * maxwbytesstrlen +           // space needed for "sel bytes / bytes" (in B/KB/MB)
                             1 ) );                          // and not to forget the final zero
  //NOTE: this is more than we will need because the modifiers in the formatstring will be replaced
  //      but it's okay to alloc some more bytes than needed
  if ( human_strings == true ) {
    // this only when the field is visible

    sprintf( formatstr, STATEBAR_FORMAT,
             maxfilestrlen,
             maxfilestrlen,
             maxdirstrlen,
             maxdirstrlen,
             maxbytesstrlen,
             maxbytesstrlen,
             maxwbytesstrlen,
             maxwbytesstrlen );
    sprintf( str, formatstr,
             tstr1,
             ce->getNrOfFiles( 1 ),
             ce->getNrOfFiles( 0 ),
             tstr2,
             ce->getNrOfDirs( 1 ),
             ce->getNrOfDirs( 0 ),
             tstr3,
             buf1.c_str(),
             buf2.c_str(),
             buf3.c_str(),
             buf4.c_str() );
  } else {
    sprintf( formatstr, STATEBAR_FORMAT_SLIM,
             maxfilestrlen,
             maxfilestrlen,
             maxdirstrlen,
             maxdirstrlen,
             maxbytesstrlen,
             maxbytesstrlen,
             maxwbytesstrlen,
             maxwbytesstrlen );
    sprintf( str, formatstr,
             tstr1,
             ce->getNrOfFiles( 1 ),
             ce->getNrOfFiles( 0 ),
             tstr2,
             ce->getNrOfDirs( 1 ),
             ce->getNrOfDirs( 0 ),
             tstr3,
             buf1.c_str(),
             buf2.c_str(),
             "",    // print just spaces
             "" );  // print just spaces
  }
  _freesafe(formatstr);
  
  if(tl==0) // no changes
    maxnochanges++;
  
  parentlister->setStatebarText(str);
  _freesafe(str);
#undef STATEBAR_FORMAT
#undef STATEBAR_FORMAT_SLIM
}

bool NormalMode::isType(const char *str)
{
  if(strcmp(str,type)==0) return true; else return false;
}

const char *NormalMode::getType()
{
  return type;
}

const char *NormalMode::getStaticType()
{
  return type;
}

int NormalMode::configure()
{
  Button *fb;
  AWindow *win;
  CycleButton *cyb[2];
  ChooseButton *chb, *ucb, *no_qs_cb;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  bool tshowhidden;
  StringGadget *tsg;
  int tut;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  std::list<NM_Filter> tfilters;
  
  tstr = (char*)_allocsafe( strlen( catalog.getLocale( 293 ) ) + strlen( getLocaleName() ) + 1 );
  sprintf( tstr, catalog.getLocale( 293 ), getLocaleName() );
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 5 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainerBB *ac1_1 = static_cast<AContainerBB*>( ac1->add( new AContainerBB( win, 1, 3 ), 0, 0 ) );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );

  AContainer *ac1_1_1 = ac1_1->add( new AContainer( win, 2, 1 ), 0, 0 );
  ac1_1_1->setMinSpace( 5 );
  ac1_1_1->setMaxSpace( 5 );
  ac1_1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 158 ), 1 ), 0, 0, cfix );
  cyb[0]=(CycleButton*)ac1_1_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
  cyb[0]->addOption(catalog.getLocale(160));
  cyb[0]->addOption(catalog.getLocale(161));
  cyb[0]->addOption(catalog.getLocale(162));
  cyb[0]->addOption(catalog.getLocale(163));
  cyb[0]->addOption(catalog.getLocale(164));
  cyb[0]->addOption(catalog.getLocale( 432 ));
  cyb[0]->addOption(catalog.getLocale( 433 ));
  cyb[0]->addOption(catalog.getLocale( 552 ));
  cyb[0]->addOption(catalog.getLocale( 553 ));
  cyb[0]->resize(cyb[0]->getMaxSize(),cyb[0]->getHeight());
  if((sortmode&0xff)==SORT_SIZE) cyb[0]->setOption(1);
  else if((sortmode&0xff)==SORT_ACCTIME) cyb[0]->setOption(2);
  else if((sortmode&0xff)==SORT_MODTIME) cyb[0]->setOption(3);
  else if((sortmode&0xff)==SORT_CHGTIME) cyb[0]->setOption(4);
  else if((sortmode&0xff)==SORT_TYPE) cyb[0]->setOption(5);
  else if((sortmode&0xff)==SORT_OWNER) cyb[0]->setOption(6);
  else if ( ( sortmode & 0xff ) == SORT_INODE ) cyb[0]->setOption( 7 );
  else if ( ( sortmode & 0xff ) == SORT_NLINK ) cyb[0]->setOption( 8 );
  else cyb[0]->setOption(0);

  chb=(ChooseButton*)ac1_1->add( new ChooseButton( aguix, 0, 0,
						   ((sortmode&SORT_REVERSE)==SORT_REVERSE)?1:0,
						   catalog.getLocale(165),LABEL_RIGHT,1,0), 0, 1, cincwnr );

  AContainer *ac1_1_2 = ac1_1->add( new AContainer( win, 2, 1 ), 0, 2 );
  ac1_1_2->setMinSpace( 5 );
  ac1_1_2->setMaxSpace( 5 );
  ac1_1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 353 ), 1 ), 0, 0, cfix );
  cyb[1]=(CycleButton*)ac1_1_2->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
  cyb[1]->addOption(catalog.getLocale(354));
  cyb[1]->addOption(catalog.getLocale(355));
  cyb[1]->addOption(catalog.getLocale(356));
  cyb[1]->resize(cyb[1]->getMaxSize(),cyb[1]->getHeight());
  if((sortmode&SORT_DIRLAST)==SORT_DIRLAST) cyb[1]->setOption(1);
  else if((sortmode&SORT_DIRMIXED)==SORT_DIRMIXED) cyb[1]->setOption(2);
  else cyb[1]->setOption(0);

  ac1_1_1->readLimits();
  ac1_1_2->readLimits();

  fb = (Button*)ac1->add( new Button( aguix, 0, 0, catalog.getLocale( 159 ), 1, 0, 0 ), 0, 1, cincw );

  AContainerBB *ac1_2 = static_cast<AContainerBB*>( ac1->add( new AContainerBB( win, 1, 2 ), 0, 2 ) );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( 5 );

  ucb=(ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
						   (showfreespace==true)?1:0,
						   catalog.getLocale(128),LABEL_RIGHT,1,0), 0, 0, cincwnr );

  AContainer *ac1_2_1 = ac1_2->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_2_1->setMinSpace( 5 );
  ac1_2_1->setMaxSpace( 5 );
  ac1_2_1->add( new Text( aguix, 0, 0, catalog.getLocale( 129 ), 1 ), 0, 0, cfix );
  tstr = (char*)_allocsafe( A_BYTESFORNUMBER( int ) );
  sprintf(tstr,"%d",updatetime);
  tsg=(StringGadget*)ac1_2_1->add( new StringGadget( aguix, 0, 0, 100, tstr, 0 ), 1, 0, cincw );
  _freesafe(tstr);

  AContainer *ac1_4 = ac1->add( new AContainer( win, 2, 1 ), 0, 3 );
  ac1_4->setMinSpace( 5 );
  no_qs_cb = (ChooseButton*)ac1_4->add( new ChooseButton( aguix, 0, 0,
                                                          ( quicksearch_enabled == true) ? 1 : 0,
                                                          catalog.getLocale( 649 ), LABEL_RIGHT, 1, 0 ), 0, 0, AContainer::CFIXNR );
  ChooseButton *fs_cb = (ChooseButton*)ac1_4->add( new ChooseButton( aguix, 0, 0,
                                                                     ( _filtered_search_enabled == true) ? 1 : 0,
                                                                     catalog.getLocale( 754 ), LABEL_RIGHT, 1, 0 ), 1, 0, AContainer::CFIXNR );

  AContainer *ac1_3 = ac1->add( new AContainer( win, 2, 1 ), 0, 4 );
  ac1_3->setMinSpace( 5 );
  ac1_3->setMaxSpace( -1 );
  ac1_3->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_3->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cb = (Button*)ac1_3->add( new Button( aguix,
						0,
						0,
						catalog.getLocale( 8 ),
						1,
						0,
						0 ), 1, 0, cfix );
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();

  tshowhidden = _dir_filter_sets.getShowHidden();
  tfilters = _dir_filter_sets.getFilters();

  DirFilterSettings t_dir_settings( _dir_filter_sets );

  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          else if(msg->button.button==fb) {
              configureFilters( &tshowhidden, tfilters,
                                t_dir_settings );
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    int tsortmode=0;
    switch(cyb[0]->getSelectedOption()) {
      case 1:
        tsortmode=SORT_SIZE;
        break;
      case 2:
        tsortmode=SORT_ACCTIME;
        break;
      case 3:
        tsortmode=SORT_MODTIME;
        break;
      case 4:
        tsortmode=SORT_CHGTIME;
        break;
      case 5:
        tsortmode=SORT_TYPE;
        break;
      case 6:
        tsortmode=SORT_OWNER;
        break;
      case 7:
        tsortmode = SORT_INODE;
        break;
      case 8:
        tsortmode = SORT_NLINK;
        break;
      default:
        tsortmode=SORT_NAME;
        break;
    }
    if ( chb->getState() == true ) tsortmode |= SORT_REVERSE;
    switch(cyb[1]->getSelectedOption()) {
      case 1:
        tsortmode|=SORT_DIRLAST;
        break;
      case 2:
        tsortmode|=SORT_DIRMIXED;
        break;
      default:
        break;
    }
    setSortmode(tsortmode);
    setShowHidden(tshowhidden);
    setFilters( tfilters );
    setShowFreeSpace( ucb->getState() );
    tut=atoi(tsg->getText());
    setUpdatetime(tut);

    setQuicksearchEnabled( no_qs_cb->getState() );
    setFilteredSearchEnabled( fs_cb->getState() );

    setHighlightBookmarkPrefix( t_dir_settings.getHighlightBookmarkPrefix() );
    setBookmarkFilter( t_dir_settings.getBookmarkFilter() );
    setSpecificBookmarkLabel( t_dir_settings.getSpecificBookmarkLabel() );
  }
  
  delete win;

  return endmode;
}

void NormalMode::down()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow();
        row++;
        if ( lv->isValidRow( row ) == true ) {
            makeRowActive( row );
            lv->showActive();
        }
    }
}

void NormalMode::up()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow();
        row--;
        if ( lv->isValidRow( row ) == true ) {
            makeRowActive( row );
            lv->showActive();
        }
        
    }
}

void NormalMode::toggleHidden()
{
    setShowHidden( ( _dir_filter_sets.getShowHidden() == true ) ? false : true );
}

void NormalMode::setShowHidden( bool nhf )
{
  finishsearchmode();
  
  _dir_filter_sets.setShowHidden( nhf );
  rebuildView();
  setName();
  if ( lv != NULL ) lv->showActive();
}

void NormalMode::startAction(FileEntry *tfe)
{
  WCFiletype *ft;
  bool fallbackEnterDir;
  if(tfe==NULL) return;
  if(tfe->isCorrupt==true) return; // Ziel existiert sowieso nicht, alles koennen wir nichts
                                   // machen

  finishsearchmode();
  
  fallbackEnterDir = false;
  ft = tfe->filetype;
  if ( ft == NULL ) {
    if ( tfe->isDir() == true ) {
      ft = wconfig->getdirtype();
    } else {
      ft = wconfig->getnotyettype();
    }
  }

  if ( ft != NULL ) {
    if ( tfe->isDir() == true ) {
      List *l = ft->getDoubleClickActions();
      if ( l != NULL ) {
	if ( l->size() < 1 ) fallbackEnterDir = true;
      }
    }
    if ( fallbackEnterDir == false ) {
      if ( checkForEmptyActionList( ft ) == true ) {
	// empty action list
	ActionMessage amsg( parentlister->getWorker() );
	StartProgOp spop;

	// prepare command list with startprogop
	List *templist = new List();
	templist->addElement( &spop );
	spop.setStart( StartProgOp::STARTPROGOP_START_IN_TERMINAL_AND_WAIT4KEY );
	spop.setRequestFlags( true );
	spop.setGlobal( true );
	spop.setGUIMsg( catalog.getLocale( 646 ) );
	//TODO Vielleicht auch Moeglichkeit haben, um Initialfenster im startprogop einzustellen

	amsg.mode = amsg.AM_MODE_SPECIAL;
	
	amsg.setFE( tfe );
	amsg.filetype = ft;
        amsg.m_action_descr = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );
	parentlister->getWorker()->interpret( templist, &amsg );

	delete templist;
      } else {
	ActionMessage amsg( parentlister->getWorker() );
	
	//amsg.mode=amsg.AM_MODE_ONLYACTIVE;
	amsg.mode=amsg.AM_MODE_SPECIAL;
	
	amsg.setFE( tfe );
	amsg.filetype = ft;
        amsg.m_action_descr = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );
	parentlister->getWorker()->interpret( ft->getDoubleClickActions(), &amsg );
      }
    }
  } else {
    if ( tfe->isDir() == true ) {
      fallbackEnterDir = true;
    }
  }
  if ( fallbackEnterDir == true ) {
    // duplicate fullname because enterDir could delete
    // the directory and with it the fullname
    std::string s1( tfe->fullname );

    // fallback for directories
    enterDir( s1.c_str() );
  }
}

void NormalMode::cyclicfunc(cyclicfunc_mode_t mode)
{
  WCFiletype *ft;
  int trow;
  int side;
  const std::vector<WorkerTypes::listcol_t> *dis;
  bool doflush, dontCheckVirtual;
  time_t now;

  if(mode==CYCLICFUNC_MODE_NORMAL) {
    if(ce!=NULL) {
      FileEntry *fe = ce->getCheckFE();

      side = parentlister->getSide();
      dis = wconfig->getVisCols( side );
      dontCheckVirtual = wconfig->getDontCheckVirtual();
  
      ft = NULL;

      doflush = false;
      
      if ( visChanged == false ) {
        if ( lv->getYOffset() != oldlvy ) visChanged = true;
      }

      // first fill the slave buffer
      reclist->lock();
        
      if ( ( visChanged == true ) && ( reclist->isFull_locked() == false ) ) {
        FileEntry *topfe;
        int toppos;
        int useelements = a_min( lv->getMaxDisplayV(), lv->getElements() - lv->getYOffset() );
          
        toppos = lv->getYOffset();

        topfe = getFEForRow( toppos );
        while ( ( topfe != NULL ) && ( useelements > 0 ) && ( reclist->isFull_locked() == false ) ) {
          if ( ( topfe->reclistQueued == false ) &&
               ( topfe->filetype == NULL ) &&
               ( topfe->isDir() == false ) ) {
            // add topfe because:
            // 1.it's visible
            // 2.type is not checked
            // 3.it's not queued
            // 4.reclist is not full
            // 5.it's a file
            ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
            te->name = topfe->fullname;
            te->fe = topfe;
            te->dontCheckContent = ce->getDontCheck();
            if ( ( te->dontCheckContent == false ) &&
                 ( dontCheckVirtual == true ) ) {
              if ( worker_islocal( te->name ) == 0 ) te->dontCheckContent = true;
            }

            reclist->put_locked( te );
          }
          useelements--;
            
          // find next visible entry
          toppos++;
          if ( lv->isValidRow( toppos ) ) {
            topfe = getFEForRow( toppos );
          } else topfe = NULL;
        }
        if (  useelements == 0 ) {
          // all visible are already known or we added them
          visChanged = false;
          oldlvy = lv->getYOffset();
        }
      }
        
      while ( ( fe != NULL ) && ( reclist->isFull_locked() == false ) ){
        if ( ( fe->reclistQueued == false ) && ( fe->filetype == NULL ) ) {
          // when fe was visible we already added it to the list
          // it's not wrong to still add it but it's just double work
          ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
          te->name = fe->fullname;
          te->fe = fe;
          te->dontCheckContent = ce->getDontCheck();
          if ( ( te->dontCheckContent == false ) &&
               ( dontCheckVirtual == true ) ) {
            if ( worker_islocal( te->name ) == 0 ) te->dontCheckContent = true;
          }
  
          reclist->put_locked( te );
        }
        ce->next_checkfe();
  
        fe = ce->getCheckFE();
      }
      reclist->signal();
      reclist->unlock();
      
      // now check for results
      trow = 0;
        
      recreslist->lock();
      ft_recres_list::ft_recres_list_t *rte;
      ft_rec_list::ft_rec_list_t *te;
      while ( recreslist->isEmpty_locked() == false ) {
        // alle Resultate entnehmen
        rte = recreslist->remove_locked();
        if ( rte != NULL ) {
          // mit oberstem Element von reclist vergleichen
          reclist->lock();
  
          te = reclist->gettop_locked();
          if ( te != NULL ) {
            if ( strcmp( te->name, rte->name ) == 0 ) {
              // correct answer
              te = reclist->remove_locked();
#ifdef DEBUG
              //                printf("type for %s:%d\n",te->name,rte->ft_index);
#endif
              if ( rte->ft_index != NULL ) {
                ft = findFiletype( rte->ft_index );
              } else {
                  ft = wconfig->getunknowntype();
              }
                
              if ( ft != NULL ) {
                // change fe and lvc to filetype
                fe = te->fe;
                fe->filetype = ft;
                if ( rte->custom_color.get() != NULL &&
                     rte->custom_color->getColorSet() != 0 ) {
                    fe->setCustomColor( *rte->custom_color );
                }
  
                while( lv->isValidRow( trow ) == true ) {
                  if ( lv->getData( trow ) == fe->getID() ) break;
                  trow++;
                }
                if ( lv->isValidRow( trow ) == false ) {
                  // lvc not found
                  // because we start always at the last position
                  // let's search again from 0

#ifdef DEBUG
                  printf( "filetype lvc search rollover\n" );
#endif
  
                  trow = 0;
                  while( lv->isValidRow( trow ) == true ) {
                    if ( lv->getData( trow ) == fe->getID() ) break;
                    trow++;
                  }
                }
                  
                if ( lv->isValidRow( trow ) == true ) {
                  fe->setLVC_DND( lv, trow, dis );
                  //lv->redraw( trow );
                  doflush = true;
                } else {
                  // lvc not found
                  // this is not a problem because filters,... could be changed
                  
                  // reset pos for next element
                  trow = 0;
                }
                  
                if ( fe == ce->getFirstNULLFT() ) {
                  // firstnullft is no longer the first element with null ft
                  ce->next_firstnullft();
                }
              }
                
              delete te;
            } else {
#ifdef DEBUG
              printf( "wrong answer: %s\n", rte->name );
#endif
            }
          }
          delete rte;

          reclist->unlock();
        }
      }
        
      recreslist->unlock();
      
      if ( doflush == true ) {
        #ifdef DEBUG
        printf("Flush\n");
        #endif
        lv->simpleRedrawUpdatedRows();
      }
      if ( ce->getFirstNULLFT() != NULL ) {
        // still some work to do so update busyFlag
        now = time( NULL );
        if ( ( lastBusyFlagUpdate < now ) || ( busyFlag == 0 ) ) {
          lastBusyFlagUpdate = now;
          busyFlag++;
          if ( busyFlag > 4 ) busyFlag = 1;
          setName();
        }
      } else {
        // recognition done so clear busyFlag
        if ( busyFlag > 0 ) {
          busyFlag = 0;
          setName();
        }
      }
      showFreeSpace(false);

      updateOnBookmarkChange();
    }
  } else {
    // get slave in reinit state

    // first wait for the slave to process previous order
    waitForNoOrder();
    
    // put it in reinit mode
    switchOrder( THREAD_REINIT );

    slave.filetype_ex.lock();
    // update filetype list
    ft_list_update();
    
    // clear lists
    ft_rec_list_clear();
    ft_recres_list_clear();
    
    // wake up slave
    slave.filetype_ex.signal();
    slave.filetype_ex.unlock();
    
    // wait for slave in wait mode
    // slave will go from reinit in wait mode after wake up
    waitForNoOrder();
    waitForStatus( THREAD_WAIT );
    
    switchOrder( THREAD_RUN );

    setupLVFields();
  
    NMCacheEntry *tce;
    int id=cache->initEnum();
    tce=(NMCacheEntry*)cache->getFirstElement(id);
    while(tce!=NULL) {
      tce->restartcheck();
      tce->checkfordcd();
      tce=(NMCacheEntry*)cache->getNextElement(id);
    }
    cache->closeEnum(id);

    bookmarksChanged();
    updateOnBookmarkChange();
  }
}

const char* NormalMode::getLocaleName()
{
  return getStaticLocaleName();
}

const char* NormalMode::getStaticLocaleName()
{
  return catalog.getLocale(173);
}

int NormalMode::load()
{
  int found_error = 0;
  int tsortmode;
  NM_Filter *tfi;
  std::string str1;
  std::list<NM_Filter> tfilters;

  tsortmode = SORT_NAME;
  for (;;) {
    if ( worker_token == HIDDENFILES_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == SHOW_WCP ) {
        setShowHidden( true );
      } else if ( worker_token == HIDE_WCP ) {
        setShowHidden( false );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SORTBY_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == SIZE_WCP ) {
        tsortmode = SORT_SIZE | ( tsortmode & ~0xff );
      } else if ( worker_token == ACCTIME_WCP ) {
        tsortmode = SORT_ACCTIME | ( tsortmode & ~0xff );
      } else if ( worker_token == MODTIME_WCP ) {
        tsortmode = SORT_MODTIME | ( tsortmode & ~0xff );
      } else if ( worker_token == CHGTIME_WCP ) {
        tsortmode = SORT_CHGTIME | ( tsortmode & ~0xff );
      } else if ( worker_token == TYPE_WCP ) {
        tsortmode = SORT_TYPE | ( tsortmode & ~0xff );
      } else if ( worker_token == OWNER_WCP ) {
        tsortmode = SORT_OWNER | ( tsortmode & ~0xff );
      } else if ( worker_token == INODE_WCP ) {
        tsortmode = SORT_INODE | ( tsortmode & ~0xff );
      } else if ( worker_token == NLINK_WCP ) {
        tsortmode = SORT_NLINK | ( tsortmode & ~0xff );
      } else if ( worker_token == NAME_WCP ) {
        tsortmode = SORT_NAME | ( tsortmode & ~0xff );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SORTFLAG_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == REVERSE_WCP ) {
        tsortmode |= SORT_REVERSE;
      } else if ( worker_token == DIRLAST_WCP ) {
        tsortmode |= SORT_DIRLAST;
        tsortmode &= ~SORT_DIRMIXED;
      } else if ( worker_token == DIRMIXED_WCP ) {
        tsortmode |= SORT_DIRMIXED;
        tsortmode &= ~SORT_DIRLAST;
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SHOWFREESPACE_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == YES_WCP ) {
        showfreespace = true;
      } else if ( worker_token == NO_WCP ) {
        showfreespace = false;
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == UPDATETIME_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == NUM_WCP ) {
        setUpdatetime( yylval.num );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == FILTER_WCP ) {
      readtoken();

      if ( worker_token != LEFTBRACE_WCP ) {
        found_error = 1;
        break;
      }
      readtoken();
      
      tfi = new NM_Filter();
      if ( tfi->load() == 0 ) {
        tfilters.push_back( *tfi );
      } else {
        delete tfi;
        found_error = 1;
        break;
      }
      delete tfi;

      if ( worker_token != RIGHTBRACE_WCP ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == SSHALLOW_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == NEVER_WCP ) {
        ssh_allow = SSH_NEVER;
      } else if ( worker_token == ALWAYS_WCP ) {
        ssh_allow = SSH_ALWAYS;
      } else if ( worker_token == ASK_WCP ) {
        ssh_allow = SSH_ASK;
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == FIELD_WIDTH_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != STRING_WCP ) {
	found_error = 1;
	break;
      }

      str1 = yylval.strptr;

      readtoken();
      if ( worker_token != ',' ) {
	found_error = 1;
	break;
      }
      readtoken();
      if ( worker_token != NUM_WCP ) {
	found_error = 1;
	break;
      }
      _loaded_field_width[str1] = yylval.num;

      readtoken();
      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == QUICKSEARCHENABLED_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == YES_WCP ) {
        quicksearch_enabled = true;
      } else if ( worker_token == NO_WCP ) {
        quicksearch_enabled = false;
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == FILTEREDSEARCHENABLED_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token == YES_WCP ) {
        setFilteredSearchEnabled( true );
      } else if ( worker_token == NO_WCP ) {
        setFilteredSearchEnabled( false );
      } else {
        found_error = 1;
        break;
      }
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else {
      break;
    }
  }
  setSortmode( tsortmode );
  setFilters( tfilters );
  
  return found_error;
}

//TODO kann das nicht mal weg?
int NormalMode::loadBin(Datei *fh)
{
  int chunksize=fh->getInt();

  int cs=Datei::getUCharSize(),
      is=Datei::getIntSize();
  unsigned char c1;
  int size;
  int tsortmode=0;
  bool tshowHidden=false;
  NM_Filter *tfi;
  std::list<NM_Filter> tfilters;

  if(chunksize>=(4*cs+is)) {
    tshowHidden=(fh->getUChar()==1)?true:false;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        tsortmode=SORT_SIZE;
        break;
      case 2:
        tsortmode=SORT_ACCTIME;
        break;
      case 3:
        tsortmode=SORT_MODTIME;
        break;
      case 4:
        tsortmode=SORT_CHGTIME;
        break;
      case 5:
        tsortmode=SORT_TYPE;
        break;
      case 6:
        tsortmode=SORT_OWNER;
        break;
      default:
        tsortmode=SORT_NAME;
        break;
    }
    if(fh->getUChar()==1) tsortmode|=SORT_REVERSE;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        tsortmode|=SORT_DIRLAST;
        break;
      case 2:
        tsortmode=SORT_DIRMIXED;
        break;
      default:
        break;
    }
    size=fh->getInt();
    chunksize-=4*cs+is;
    
    for(int i=0;i<size;i++) {
      tfi=new NM_Filter();
      tfi->loadBin( fh );
      tfilters.push_back( *tfi );
      delete tfi;
    }
    // apply new values
    setShowHidden(tshowHidden);
    setSortmode(tsortmode);
    setFilters( tfilters );
    // this is added from version 2.0.2 so for file from older version
    if(chunksize>=(cs+is)) {
      showfreespace=(fh->getUChar()==1)?true:false;
      chunksize-=cs;
      setUpdatetime(fh->getInt());
      chunksize-=is;
    }
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return 0;
}

bool NormalMode::save(Datei *fh)
{
  char numbuf[A_BYTESFORNUMBER( int )];
  std::map<std::string,int>::iterator it1;
  std::string str1;

  if ( fh == NULL ) return false;
 
  fh->configPutPair( "hiddenfiles" , ( _dir_filter_sets.getShowHidden() == true ) ? "show" : "hide" );
  switch ( sortmode & 0xff ) {
    case SORT_SIZE:
      fh->configPutPair( "sortby", "size" );
      break;
    case SORT_ACCTIME:
      fh->configPutPair( "sortby", "acctime" );
      break;
    case SORT_MODTIME:
      fh->configPutPair( "sortby", "modtime" );
      break;
    case SORT_CHGTIME:
      fh->configPutPair( "sortby", "chgtime" );
      break;
    case SORT_TYPE:
      fh->configPutPair( "sortby", "type" );
      break;
    case SORT_OWNER:
      fh->configPutPair( "sortby", "owner" );
      break;
    case SORT_INODE:
      fh->configPutPair( "sortby", "inode" );
      break;
    case SORT_NLINK:
      fh->configPutPair( "sortby", "nlink" );
      break;
    default:
      fh->configPutPair( "sortby", "name" );
      break;
  }
  if ( ( sortmode & SORT_REVERSE ) != 0 ) fh->configPutPair( "sortflag", "reverse" );
  if ( ( sortmode & SORT_DIRLAST ) != 0 ) fh->configPutPair( "sortflag", "dirlast" );
  else if ( ( sortmode & SORT_DIRMIXED ) != 0 ) fh->configPutPair( "sortflag", "dirmixed" );
  
  fh->configPutPairBool( "showfreespace", showfreespace );
  fh->configPutPairNum( "updatetime", updatetime );

  fh->configPutPairBool( "quicksearchenabled", quicksearch_enabled );
  fh->configPutPairBool( "filteredsearchenabled", _filtered_search_enabled );

  switch ( ssh_allow ) {
    case SSH_NEVER:
      fh->configPutPair( "sshallow", "never" );
      break;
    case SSH_ALWAYS:
      fh->configPutPair( "sshallow", "always" );
      break;
    default:
      fh->configPutPair( "sshallow", "ask" );
      break;
  }

  if ( lv != NULL ) {
    writeFieldWidthsToLoadMap();
  }

  for ( it1 = _loaded_field_width.begin(); it1 != _loaded_field_width.end(); it1++ ) {
    sprintf( numbuf, "%d", (*it1).second );
    str1 = "field_width = ";
    str1 += "\"";
    str1 += (*it1).first;
    str1 += "\"";
    str1 += ",";
    str1 += numbuf;
    str1 += ";";
    fh->configPutInfo( str1.c_str(), false );
  }

  std::list<NM_Filter> filters = _dir_filter_sets.getFilters();
  std::list<NM_Filter>::iterator fil_it;

  for ( fil_it = filters.begin();
        fil_it != filters.end();
        fil_it++ ) {
      fh->configOpenSection( "filter" );
      (*fil_it).save( fh );
      fh->configCloseSection(); //filter
  }
  return false;
}

void NormalMode::top()
{
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    makeRowActive( 0 );
    lv->showActive();
}

void NormalMode::last()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();
    
    row = lv->getElements() - 1;
    if ( lv->isValidRow( row ) == true ) {
        makeRowActive( row );
        lv->showActive();
    }
}

void NormalMode::pageup()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();

    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow() - lv->getMaxDisplayV() + 1;
        if ( lv->isValidRow( row ) == false )
            row = 0;
        
        makeRowActive( row );
        lv->showActive();
    }
}

void NormalMode::pagedown()
{
    int row;
    
    if ( ce == NULL ) return;
    
    finishsearchmode();

    if ( ce->getActiveFE() == NULL ) {
        makeRowActive( 0 );
        lv->showActive();
    } else {
        row = lv->getActiveRow() + lv->getMaxDisplayV() - 1;
        if ( lv->isValidRow( row ) == false )
            row = lv->getElements() - 1;
        if ( lv->isValidRow( row ) == false )
            row = 0;
        
        makeRowActive( row );
        lv->showActive();
    }
}

void NormalMode::select()
{
  if(ce==NULL) return;
  if ( ce->getActiveFE() == NULL ) return;

  finishsearchmode();
  
  int row=lv->getActiveRow();
  if( row < 0 ) {
    printf("oops, this should never happen!\n");
    return;
  }
  if(lv->getSelect(row)==false) lv->setSelect(row, true); else lv->setSelect(row,false);
  selhandler(lv,row );
  down();
}

void NormalMode::selectall()
{
  if(ce==NULL) return;
  int row;
  
  finishsearchmode();
  
  /* LVCs auswaehlen */
  row = 1;
  while( row < lv->getElements() ) {
    lv->setSelect(row,true);
    row++;
  }
  /* FEs auswaehlen */

  for ( Verzeichnis::verz_it it1 = ce->begin();
        it1 != ce->end();
        it1++ ) {
      FileEntry *fe = *it1;
      if ( fe->use == true && strcmp( fe->name, ".." ) != 0 )
          fe->select = true;
  }
  ce->recalcStats();
  lv->redraw();
  showCacheState();
}

void NormalMode::selectnone()
{
  if(ce==NULL) return;
  int row;
  
  finishsearchmode();
  
  /* LVCs deselecten */
  row = 0;
  while( row < lv->getElements()) {
    lv->setSelect(row,false);
    row++;
  }
  /* FEs deselecten */
  for ( Verzeichnis::verz_it it1 = ce->begin();
        it1 != ce->end();
        it1++ ) {
      FileEntry *fe = *it1;
      if ( fe->use == true )
          fe->select = false;
  }
  ce->recalcStats();
  lv->redraw();
  showCacheState();
}

void NormalMode::invertall()
{
  if(ce==NULL) return;
  int row;
  
  finishsearchmode();
  
  /* LVCs invertieren */
  row = 1;
  while(row<lv->getElements()) {
    lv->setSelect(row,lv->getSelect(row)==true?false:true);
    row++;
  }
  /* FEs invertieren */
  for ( Verzeichnis::verz_it it1 = ce->begin();
        it1 != ce->end();
        it1++ ) {
      FileEntry *fe = *it1;
      if ( fe->use == true && strcmp( fe->name, ".." ) != 0 )
          fe->select = ( fe->select == true ? false : true );
  }
  ce->recalcStats();
  lv->redraw();
  showCacheState();
}

void NormalMode::parent()
{
  char oentry[128];
  if(ce==NULL) return;
  int row;
  
  const char *adir=ce->getDir();
  char *ndir=ParentDir(adir,oentry,127);
  enterDir(ndir);
  /* oentry finden und aktivieren */
  FileEntry *fe = NULL;
  for ( Verzeichnis::verz_it it1 = ce->begin();
        it1 != ce->end();
        it1++ ) {
      fe = *it1;
      if ( fe->use == true )
          if ( strcmp( fe->name, oentry ) == 0 )
              break;
      fe = NULL;
  }
  if ( fe!=NULL) {
    /* Eintrag gefunden */
    ce->setActiveFE( fe );
    row = 0;
    while( row < lv->getElements() ) {
      if ( lv->getData(row) == fe->getID() ) break;
      row++;
    }
    if( row < lv->getElements()) {
      makeListViewRowActive(row);
      lv->showActive();
      aguix->Flush();
    }
  }
  _freesafe(ndir);
}

void NormalMode::enterDirActive()
{
  if(ce==NULL) return;
  if ( ce->getActiveFE() != NULL ) {
    if ( ce->getActiveFE()->isDir() == true ) {
      // copy fullname because enterDir can delete directory
      std::string s1( ce->getActiveFE()->fullname );
      enterDir( s1.c_str() );
    }
  }
}

void NormalMode::filterselect(char *filter,int mode)
{
  /* Steht in filter[0] ein /, werden Verzeichnisse betrachtet, sonst Dateien */
  if(ce==NULL) return;
  int row;
  char *realfilter;
  bool tselect=(mode==0)?true:false;
  bool usedir;
  
  finishsearchmode();
  
  if(filter[0]=='/') {
    usedir=true;
    realfilter=&filter[1];
  } else {
    usedir=false;
    realfilter=filter;
  }
  
  for ( Verzeichnis::verz_it it1 = ce->begin();
        it1 != ce->end();
        it1++ ) {
      FileEntry *fe = *it1;
      if ( fe->use == true && strcmp( fe->name, ".." ) != 0 )
          if ( fe->match( realfilter ) == true ) 
              if ( fe->isDir() == usedir )
                  fe->select = tselect;
  }

  row = 0;
  Verzeichnis::verz_it it2 = ce->begin();
  while( row < lv->getElements()) {
      while ( (*it2)->getID() != lv->getData( row ) )
          it2++;
      lv->setSelect( row, (*it2)->select );
      row++;
  }
  lv->redraw();
  ce->recalcStats();
  showCacheState();
}

const char *NormalMode::getCurrentDir()
{
  if ( ce != NULL ) return ce->getDir();
  return NULL;
}

void
NormalMode::copy(struct NM_copyorder *copyorder)
{
  Lister *olister=NULL;
  NormalMode *nm2=NULL,*updatenm=NULL;
  ListerMode *lm=NULL;
  NM_specialsourceInt *ss1, *tss;
  const FileEntry *tfe;
  bool enter;
  NM_CopyOp_Dir *cod1,*tcod;
  unsigned long files,dirs,gf,gd;
  loff_t bytes;
  bool skip,cancel=false;
  CopyOpWin *cowin;
  nm_copy_t ce1;
  bool nodatabase;
  int erg;
  char *textstr,*tstr,*buttonstr;
  std::list<NM_specialsourceInt*> *copylist;
  std::list<NM_specialsourceInt*>::iterator iti1, iti2;
  std::list<NM_specialsourceInt*> *copy_lvc_list;
  std::list<const FileEntry*>::iterator itfe1;
  
  if(copyorder==NULL) return;
  if(copyorder->destdir==NULL) return;
  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to copy
    return;
  }

  if ( ce->dirOpened() == false ) return;

  copyorder->overwrite=copyorder->NM_OVERWRITE_NORMAL;  // force to start in this mode

  finishsearchmode();
  
  if(Datei::fileExistsExt(copyorder->destdir)!=Datei::D_FE_DIR) {
    // no dir as destination
    textstr=(char*)_allocsafe(strlen(catalog.getLocale(67))+strlen(copyorder->destdir)+1);
    sprintf(textstr,catalog.getLocale(67),copyorder->destdir);
    buttonstr = dupstring( catalog.getLocale(11) );
    req->request(catalog.getLocale(125),textstr,buttonstr);
    _freesafe(textstr);
    _freesafe( buttonstr );
    return;
  }
  
  // clear the reclist so the slace doesn't block us because he is still reading
  // from disk
  ft_rec_list_clear();
  
  if ( strcmp( getCurrentDir(), copyorder->destdir ) == 0 ) {
    copyorder->do_rename=true;
    updatenm=this;
  } else {
    olister=parentlister->getWorker()->getOtherLister(parentlister);
    if(olister!=NULL) {
      lm=olister->getActiveMode();
      if(lm!=NULL) {
        if(lm->isType("NormalMode")==true) {
          nm2=(NormalMode*)lm;
          if ( nm2->getCurrentDir() != NULL ) {
            if ( strcmp( nm2->getCurrentDir(), copyorder->destdir ) == 0 ) {
              updatenm=nm2;
            }
          }
        }
      }
    }
  }
  buf=_allocsafe(BUFSIZE);
  
  if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;
  
  // no sense for follow_symlinks when moving
  if(copyorder->move==true) copyorder->follow_symlinks=false;

  /* now presupposition checked
     next search for the files to copy
     therefore collect entries to copy in a list together with the LVC (when on root-level)
     then for each entry do recursive call to copy it
     after success deactivate LVC */

  /* first: create List contains NM_specialsource
     when source==NM_SPECIAL then the given list but check for existing of the FEs
     else source==NM_ONLYACTIVE => take the activefe (from ce) in this list
     else take all selected entries (except the "..")
  */
  cowin=copyorder->cowin;
  if(cowin!=NULL) {
    cowin->open(nodatabase);
    cowin->setmessage(catalog.getLocale(122),0);
    if ( cowin->redraw() != 0 ) cancel = true;
  }
  copylist = new std::list<NM_specialsourceInt*>;
  switch(copyorder->source) {
    case NM_copyorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( copyorder->sources ),
                                   *copylist );
        break;
    case NM_copyorder::NM_ONLYACTIVE:
      getSelFiles( copylist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( copylist, NM_GETFILES_SELORACT, false );
      break;
  }
  if ( cowin != NULL ) {
    if ( cowin->redraw() != 0 ) cancel = true;
  }

  // create the NM_CopyOp_Dir for each dir in copylist
  files=dirs=gf=gd=0;
  bytes = 0;
  
  if(nodatabase==false) {
    for ( iti1 = copylist->begin(); iti1 != copylist->end(); iti1++ ) {
      if ( cancel == true ) break;
      ss1 = *iti1;
      enter=false;
      if ( ss1->entry()->isDir() == true ) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
	// entry is a dir so it cannot be a corrupt link so no need to check
        if ( ss1->entry()->isLink == false ) enter = true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1 = new NM_CopyOp_Dir( ss1->entry() );
        if ( cod1->user_abort == false ) {
          // recursive call
          if(cod1->createSubDirs(copyorder,&gf,&gd)!=0) cancel=true;
        } else cancel = true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;

        ss1->cod=cod1;
        // this is a dir so inc the counter
        dirs++;
        gd++;
        if(cowin!=NULL) {
          cowin->set_files_to_copy(gf);
          cowin->set_dirs_to_copy(gd);
          if(cowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        gf++;
        if ( ( ss1->entry()->isLink == true ) &&
             ( copyorder->follow_symlinks == true ) &&
             ( ss1->entry()->isCorrupt == false ) ) {
	  bytes += ss1->entry()->dsize();
        } else {
          bytes += ss1->entry()->size();
        }
        ss1->cod=NULL;
      }
    }
  }

  if(cowin!=NULL) {
    cowin->set_files_to_copy(files);
    cowin->set_dirs_to_copy(dirs);
    cowin->set_bytes_to_copy(bytes);
    cowin->starttimer();
  }
  // now database ready, can start copy-process

  copy_deletefe_list = new std::list<const FileEntry*>;
  copy_lvc_list = new std::list<NM_specialsourceInt*>;

  skip=false;

  for ( iti1 = copylist->begin(); ( iti1 != copylist->end() ) && ( cancel == false ); iti1++ ) {
    ss1 = *iti1;
    if(nodatabase==false) {
      tcod=ss1->cod;
    } else {
      enter=false;
      tcod=NULL;
      if ( ss1->entry()->isDir() == true ) {
	// entry is a dir so it cannot be a corrupt link so no need to check
        if ( ss1->entry()->isLink == false ) enter = true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if ( enter == true ) {
        tcod = new NM_CopyOp_Dir( ss1->entry() );
        if ( tcod->user_abort == true ) {
          delete tcod;
          tcod = NULL;
          cancel = true;
        }
      }
    }
    if ( cancel == true ) break;
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row, false );
      lv->timeLimitedRedraw();
    }
    if(tcod!=NULL) {
      ce1 = copydir( ss1->entry(), tcod, copyorder, true, copyorder->destdir );
      if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
        // success
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
      if(copyorder->move==true) {
        if((ce1==NM_COPY_OK)&&(tcod->error_counter==0)) {
          if ( ss1->row >= 0 ) lv->showRow( ss1->row, false );
          removeEntry( ss1->entry(), ss1->row );
          fixSpecialSourceList( copylist, ss1->row );
        } else if(ce1==NM_COPY_NEED_DELETE) {
          if(ss1->row>=0) {
	    tss = new NM_specialsourceInt( ss1->entry() );
            tss->row=ss1->row;
            copy_lvc_list->push_back(tss);
          }
        }
      } else {
        if((tcod->error_counter==0)&&(cancel==false)) {
          // only deselect when dir totally copied
          if(ss1->row>=0) {
            lv->setSelect( ss1->row, false );
            lv->showRow( ss1->row, false );
          }
          ss1->entry()->select=false;
          ce->recalcStats();
          showCacheState();
        }
      }
      if(nodatabase==true) delete tcod;
    } else {
      ce1 = copyfile( ss1->entry(), copyorder, true, copyorder->destdir );
      if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
        // success
        if(copyorder->move==true) {
          if(ce1==NM_COPY_OK) {
            if ( ss1->row >= 0 ) lv->showRow( ss1->row, false );
            removeEntry( ss1->entry(), ss1->row );
            fixSpecialSourceList( copylist, ss1->row );
          } else {
            if(ss1->row>=0) {
	      tss = new NM_specialsourceInt( ss1->entry() );
              tss->row=ss1->row;
              copy_lvc_list->push_back(tss);
            }
          }
        } else {
          if(ss1->row>=0) {
            lv->setSelect(ss1->row,false);
            lv->showRow( ss1->row, false );
          }
          ss1->entry()->select = false;
          ce->recalcStats();
          showCacheState();
        }
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
    }
    if ( ss1->row >= 0 ) {
      // this can and will changed the wrong row when the entry
      // was moved successfully
      // but for this case this doesn't matter because
      // the disabled visual mark is the default
      lv->setVisMark( ss1->row, false );
    }
    if(cowin!=NULL) {
      if(cowin->redraw()!=0) cancel=true;
    }
  }
  lv->redraw();
  
  // now delete remaining files/dirs when moving
  if(copyorder->move==true) {
    bool delcancel; // additional cancel so we don't need to
                    // clear cancel for the special case when move
                    // has to copy&delete and was aborted somewhere
                    // I ask the user what to do with the correctly copied
                    // files
    std::string str1;
  
    if(cowin!=NULL) cowin->setmessage("",1);

    itfe1 = copy_deletefe_list->begin();

    if ( itfe1 != copy_deletefe_list->end() )
      tfe = *itfe1;
    else
      tfe = NULL;
    iti2 = copy_lvc_list->begin();
    if ( iti2 != copy_lvc_list->end() )
      tss = *iti2;
    else
      tss = NULL;
    
    delcancel = false;
    if ( ( cancel == true ) && ( tfe != NULL ) ) {
      // copy was canceled
      // ask the user what to do with the correctly moved files
      // (exactly what to do with the sources!!)
      str1 = catalog.getLocale( 11 );
      str1 += "|";
      str1 += catalog.getLocale( 8 );
      erg = request( cowin, catalog.getLocale( 123 ), catalog.getLocale( 528 ), str1.c_str() );
      if ( erg != 0 ) {
        delcancel = true;
      }
    }
    
    while ( ( tfe != NULL ) && ( delcancel == false ) ) {
      if(tfe->isDir()==true) {
        if(cowin!=NULL) {
          tstr=(char*)_allocsafe(strlen(catalog.getLocale(141))+strlen(tfe->fullname)+1);
          sprintf(tstr,catalog.getLocale(141),tfe->fullname);
          cowin->setmessage(tstr,0);
          _freesafe(tstr);
        }
      }
      // remove tfe
      if ( ( tfe->isDir() == true ) && ( tfe->isLink == false ) ) erg = worker_rmdir( tfe->fullname );
      else erg = worker_unlink( tfe->fullname );
      if(erg==0) {
        // success
        if(tss!=NULL) {
          if ( tfe == tss->entry() ) {
            lv->showRow(tss->row);
            removeEntry( tss->entry(), tss->row );
            fixSpecialSourceList( copy_lvc_list, tss->row );
	    iti2++;
	    if ( iti2 != copy_lvc_list->end() )
	      tss = *iti2;
	    else
	      tss = NULL;
          }
        }
      } else {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(tfe->fullname)+1);
        sprintf(textstr,catalog.getLocale(291),tfe->fullname);
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                  catalog.getLocale(8));
        erg = request( cowin, catalog.getLocale( 347 ), textstr, buttonstr );
        _freesafe(buttonstr);
        _freesafe(textstr);
        if ( erg == 1 ) delcancel = true;
      }
      if ( itfe1 != copy_deletefe_list->end() ) itfe1++;
      if ( itfe1 != copy_deletefe_list->end() )
	tfe = *itfe1;
      else
	tfe = NULL;
    }
    if ( delcancel == true ) cancel = true; // as I write this cancel is not used
                                            // but perhaps later I will use it and
                                            // so set it also to true in this case
  }
  
  if(cowin!=NULL) cowin->close();
  
  for ( iti1 = copylist->begin(); iti1 != copylist->end(); iti1++ ) {
    ss1 = *iti1;
    if(ss1->cod!=NULL) delete ss1->cod;
    delete ss1;
  }
  delete copylist;
  delete copy_deletefe_list;

  for ( iti1 = copy_lvc_list->begin(); iti1 != copy_lvc_list->end(); iti1++ ) {
    tss = *iti1;
    delete tss;
  } 

  delete copy_lvc_list;
  _freesafe(buf);
  if(copyorder->move==true) {
    ce->restartcheck();
    rebuildView();
  }
  if(updatenm!=NULL) updatenm->update(false);
  aguix->Flush();
}

NormalMode::nm_newname_t NormalMode::getNewName(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool requestDir,bool acceptrename)
{
  if(requestDir==true)
    return getNewName4Dir(oldfe,copyorder,dest,newname_return,acceptrename);
  else
    return getNewName4File(oldfe,copyorder,dest,newname_return,acceptrename);
}

NormalMode::nm_newname_t NormalMode::getNewName4Dir(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool acceptrename)
{
  char *oldname=oldfe->name;
  char *newname=dupstring(oldname);
  bool nameok=false,skip=false,cancel=false;
  char *return_str=NULL,*buttonstr,*textstr,*newdest;
  int erg;
  Datei::d_fe_t se,lse;
  int round=1;
  nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
  bool trydel,do_rename,strongdirs;
  char *title;
  
  title = (char*)_allocsafe( strlen( catalog.getLocale( 500 ) ) + strlen( oldfe->fullname ) + 1 );
  sprintf( title, catalog.getLocale( 500 ), oldfe->fullname );
  
  do_rename=false;
  if(copyorder->do_rename==false) do_rename=false;
  else if(acceptrename==true) do_rename=true;
  
  strongdirs=false;

  do {
    if(!((round==1)&&(do_rename==false))) {
      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                  strlen(catalog.getLocale(225))+1+
                                  strlen(catalog.getLocale(8))+1);
      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                   catalog.getLocale(225),
                                   catalog.getLocale(8));
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
      sprintf(textstr,catalog.getLocale(148),oldname);
      erg = string_request( copyorder->cowin,
                            catalog.getLocale( 149 ),
                            textstr,
                            newname,
                            buttonstr,
                            &return_str,
                            Requester::REQUEST_SELECTALL );
      _freesafe(buttonstr);
      _freesafe(textstr);
      if(return_str!=NULL) {
        _freesafe(newname);
        newname=return_str;
      }
      if(erg==1) {
        skip=true;
        returnvalue=NM_NEWNAME_SKIP;
        break;
      } else if(erg==2) {
        cancel=true;
        returnvalue=NM_NEWNAME_CANCEL;
        break;
      }
    }
    
    if ( strlen( newname ) < 1 ) {
      // empty name->request
      round++;  // not needed since empty names should only come from user input
                // but perhaps there is (will be) a OS which allows empty names
      continue;
    }
    
    newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
    strcpy(newdest,dest);
    if(strlen(dest)>1) strcat(newdest,"/");
    strcat(newdest,newname);
    se=Datei::fileExistsExt(newdest);
    lse=Datei::lfileExistsExt(newdest);

    switch(lse) {
      case Datei::D_FE_DIR:
        if(strongdirs==true) {
          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 190 ) ) + strlen( newdest ) + 1 );
          sprintf( textstr, catalog.getLocale( 190 ), newdest );

          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(273))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                          catalog.getLocale(273),
                                          catalog.getLocale(225),
                                          catalog.getLocale(8));

          erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) {
            nameok=true;
            returnvalue=NM_NEWNAME_USE;
          } else if(erg==2) {
            skip=true;
            returnvalue=NM_NEWNAME_SKIP;
          } else if(erg==3) {
            cancel=true;
            returnvalue=NM_NEWNAME_CANCEL;
          }
        } else {
          // just use the dir
          nameok=true;
          returnvalue=NM_NEWNAME_USE;
        }
        break;
      case Datei::D_FE_LINK:
      case Datei::D_FE_FILE:
        if((lse==Datei::D_FE_LINK)&&(se==Datei::D_FE_DIR)) {
          // Symlink with dir as destination 
          if(strongdirs==true) {
//            textstr="There is already a symlink called %s, which points to a dir|You can use this link or delete it to|create a real directory";
//            buttonstr="Enter new name|Use this dir|Delete link|Skip|Cancel";

            textstr = (char*)_allocsafe( strlen( catalog.getLocale( 275 ) ) + strlen( newdest ) + 1 );
            sprintf( textstr, catalog.getLocale( 275 ), newdest );
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(273))+1+
                                        strlen(catalog.getLocale(274))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(230),
                                            catalog.getLocale(273),
                                            catalog.getLocale(274),
                                            catalog.getLocale(225),
                                            catalog.getLocale(8));

            erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe(textstr);
            switch(erg) {
              case 1:
                nameok=true;
                returnvalue=NM_NEWNAME_USE;
                break;
              case 2:
                // try to remove the link
                // if success set nameok to true
                // in case of failure show a request and repeat the whole
                if ( worker_unlink( newdest ) == 0 ) {
                  nameok=true;
                  returnvalue=NM_NEWNAME_OK;
                } else {
//                  textstr="Failed to remove this file|Please enter new name!";
//                  buttonstr="Ok|Skip|Cancel";

                  textstr = dupstring( catalog.getLocale(276) );
                  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                              strlen(catalog.getLocale(225))+1+
                                              strlen(catalog.getLocale(8))+1);
                  sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                               catalog.getLocale(225),
                                               catalog.getLocale(8));

                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
                  _freesafe(buttonstr);
                  _freesafe( textstr );
                  if(erg==1) {
                    skip=true;
                    returnvalue=NM_NEWNAME_SKIP;
                  } else if(erg==2) {
                    cancel=true;
                    returnvalue=NM_NEWNAME_CANCEL;
                  }
                }
                break;
              case 3:
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
              case 4:
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
          } else {
            // just use the link to the dir
            nameok=true;
            returnvalue=NM_NEWNAME_USE;
          }
        } else {
          trydel=false;
          if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
            trydel=true;
          } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
            // skip
            skip=true;
            returnvalue=NM_NEWNAME_SKIP;
          } else {
            if(lse==Datei::D_FE_LINK) {
//              textstr="There is already a link named %s";
//              buttonstr="Enter new name|Delete link|Skip|Cancel";

              textstr = (char*)_allocsafe( strlen( catalog.getLocale( 278 ) ) + strlen( newdest ) + 1 );
              sprintf( textstr, catalog.getLocale( 278 ), newdest );
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                          strlen(catalog.getLocale(274))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                              catalog.getLocale(274),
                                              catalog.getLocale(225),
                                              catalog.getLocale(8));
            } else {
//              textstr="There is already a file named %s";
//              buttonstr="Enter new name|Delete file|Skip|Cancel";

              textstr = (char*)_allocsafe( strlen( catalog.getLocale( 279 ) ) + strlen( newdest ) + 1 );
              sprintf( textstr, catalog.getLocale( 279 ), newdest );
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                          strlen(catalog.getLocale(277))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                              catalog.getLocale(277),
                                              catalog.getLocale(225),
                                              catalog.getLocale(8));
            }
            erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe(textstr);
            switch(erg) {
              case 1:
                trydel=true;
                break;
              case 2:
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
              case 3:
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
          }
          if(trydel==true) {
            if ( worker_unlink( newdest ) == 0 ) {
              nameok=true;
              returnvalue=NM_NEWNAME_OK;
            } else {
//              textstr="Failed to remove this file|Please enter new name!";
//              buttonstr="Ok|Skip|Cancel";

              textstr = dupstring( catalog.getLocale(276) );
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                           catalog.getLocale(225),
                                           catalog.getLocale(8));

              erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
              _freesafe(buttonstr);
              _freesafe( textstr );
              if(erg==1) {
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
              } else if(erg==2) {
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
              }
            }
          }
        }
        break;
      default:
        nameok=true;
        returnvalue=NM_NEWNAME_OK;
        break;
    }
    _freesafe(newdest);
    round++;
  } while((nameok==false)&&(skip==false)&&(cancel==false));
  if((skip==true)||(cancel==true)) {
    _freesafe(newname);
    *newname_return=NULL;
  } else {
    *newname_return=newname;
  }
  _freesafe( title );
  return returnvalue;
}

NormalMode::nm_newname_t NormalMode::getNewName4File(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool acceptrename)
{
  char *oldname=oldfe->name;
  char *newname=dupstring(oldname);
  bool nameok=false,skip=false,cancel=false;
  char *return_str=NULL,*buttonstr,*textstr,*newdest;
  int erg;
  Datei::d_fe_t se,lse;
  int round=1;
  nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
  bool do_rename,trydel;
  char *extrastr,*tstr,*newtime,*oldtime;
  int newsizelen,oldsizelen,maxsizelen;
  char *title;
  time_t tvar;
  std::string newsize, oldsize;
  
  title = (char*)_allocsafe( strlen( catalog.getLocale( 116 ) ) + strlen( oldfe->fullname ) + 1 );
  sprintf( title, catalog.getLocale( 116 ), oldfe->fullname );
  
  do_rename=false;
  if(copyorder->do_rename==false) do_rename=false;
  else if(acceptrename==true) do_rename=true;

  do {
    if(!((round==1)&&(do_rename==false))) {
//      buttonstr="Ok|Skip|Cancel";
//      textstr="Enter new name for ...";
      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                  strlen(catalog.getLocale(225))+1+
                                  strlen(catalog.getLocale(8))+1);
      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                   catalog.getLocale(225),
                                   catalog.getLocale(8));
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
      sprintf(textstr,catalog.getLocale(148),oldname);
      erg = string_request( copyorder->cowin,
                            catalog.getLocale( 149 ),
                            textstr,
                            newname,
                            buttonstr,
                            &return_str,
                            Requester::REQUEST_SELECTALL );
      _freesafe(buttonstr);
      _freesafe(textstr);
      if(return_str!=NULL) {
        _freesafe(newname);
        newname=return_str;
      }
      if(erg==1) {
        skip=true;
        returnvalue=NM_NEWNAME_SKIP;
        break;
      } else if(erg==2) {
        cancel=true;
        returnvalue=NM_NEWNAME_CANCEL;
        break;
      }
    }
    
    if ( strlen( newname ) < 1 ) {
      // empty name->request
      round++;  // not needed since empty names should only come from user input
                // but perhaps there is (will be) a OS which allows empty names
      continue;
    }
    
    newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
    strcpy(newdest,dest);
    if(strlen(dest)>1) strcat(newdest,"/");
    strcat(newdest,newname);
    se=Datei::fileExistsExt(newdest);
    lse=Datei::lfileExistsExt(newdest);

    switch(lse) {
      case Datei::D_FE_DIR:
//        textstr="there is already a dir called %s";
//        buttonstr="Enter new name|Skip|Cancel";

        textstr = (char*)_allocsafe( strlen( catalog.getLocale( 190 ) ) + strlen( newdest ) + 1 );
        sprintf( textstr, catalog.getLocale( 190 ), newdest );

        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));

        erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
        _freesafe(buttonstr);
        _freesafe(textstr);
        if(erg==1) {
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else if(erg==2) {
          cancel=true;
          returnvalue=NM_NEWNAME_CANCEL;
        }
        break;
      case Datei::D_FE_LINK:
        trydel=false;
        if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
          trydel=true;
        } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
          // skip
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else {
//          textstr="there is already a link named %s";
//          buttonstr="Enter new name|Delete link|Skip|Cancel";

          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 278 ) ) + strlen( newdest ) + 1 );
          sprintf( textstr, catalog.getLocale( 278 ), newdest );
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(274))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                          catalog.getLocale(274),
                                          catalog.getLocale(225),
                                          catalog.getLocale(8));

          erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
          _freesafe(buttonstr);
          _freesafe(textstr);
          switch(erg) {
            case 1:
              trydel=true;
              break;
            case 2:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              break;
            case 3:
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
              break;
          }
        }
        if(trydel==true) {
          if ( worker_unlink( newdest ) == 0 ) {
            nameok=true;
            returnvalue=NM_NEWNAME_OK;
          } else {
//            textstr="Failed to remove this file|Please enter new name!";
//            buttonstr="Ok|Skip|Cancel";

            textstr = dupstring( catalog.getLocale(276) );
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));

            erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
            _freesafe(buttonstr);
            _freesafe( textstr );
            if(erg==1) {
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
            } else if(erg==2) {
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
            }
          }
        }
        break;
      case Datei::D_FE_FILE:
        if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
          nameok=true;
          returnvalue=NM_NEWNAME_OVERWRITE;
        } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else {
          FileEntry *newfe=new FileEntry();
          if(newfe->name!=NULL) _freesafe(newfe->name);
          if(newfe->fullname!=NULL) _freesafe(newfe->fullname);
          newfe->name=dupstring(newname);
          newfe->fullname=dupstring(newdest);
          newfe->readInfos();
          
	  MakeLong2NiceStr( newfe->size(), newsize );
	  MakeLong2NiceStr( oldfe->size(), oldsize );
	  oldsizelen = oldsize.length();
	  newsizelen = newsize.length();
	  maxsizelen = ( newsizelen > oldsizelen ) ? newsizelen : oldsizelen;
	  if ( maxsizelen > newsizelen )
	    newsize.insert( newsize.begin(), maxsizelen - newsizelen, ' ' );
	  if ( maxsizelen > oldsizelen )
	    oldsize.insert( oldsize.begin(), maxsizelen - oldsizelen, ' ' );

          tvar = newfe->lastmod();
          newtime = dupstring( ctime( &( tvar ) ) );
          tvar = oldfe->lastmod();
          oldtime = dupstring( ctime( &( tvar ) ) );
          newtime[strlen(newtime)-1]='\0';  // remove return
          oldtime[strlen(oldtime)-1]='\0';  // remove return
          
          extrastr=(char*)_allocsafe(strlen(catalog.getLocale(137))+newsize.length()+oldsize.length()+
                                     strlen(newtime)+strlen(oldtime)+1);
          sprintf(extrastr,catalog.getLocale(137),newsize.c_str(),oldsize.c_str(),newtime,oldtime);
          
          _freesafe(newtime);
          _freesafe(oldtime);
          delete newfe;
//          textstr="There is already a file named %s";
//          buttonstr="Enter new name|Overwrite|Overwrite all|Overwrite none|Skip|Cancel";

          textstr = (char*)_allocsafe( strlen( catalog.getLocale( 279 ) ) + strlen( newdest ) + 1 );
          sprintf( textstr, catalog.getLocale( 279 ), newdest );

          tstr=catstring(textstr,"|");
          _freesafe(textstr);
          textstr=catstring(tstr,extrastr);
          _freesafe(tstr);
          _freesafe(extrastr);

          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(280))+1+
                                      strlen(catalog.getLocale(135))+1+
                                      strlen(catalog.getLocale(136))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s|%s|%s",catalog.getLocale(230),
                                                catalog.getLocale(280),
                                                catalog.getLocale(135),
                                                catalog.getLocale(136),
                                                catalog.getLocale(225),
                                                catalog.getLocale(8));

          erg = request( copyorder->cowin, title/*catalog.getLocale(123)*/, textstr, buttonstr );
          _freesafe(buttonstr);
          _freesafe(textstr);
          switch(erg) {
            case 1:
              nameok=true;
              returnvalue=NM_NEWNAME_OVERWRITE;
              break;
            case 2:
              nameok=true;
              returnvalue=NM_NEWNAME_OVERWRITE;
              copyorder->overwrite=copyorder->NM_OVERWRITE_ALWAYS;
              break;
            case 3:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              copyorder->overwrite=copyorder->NM_OVERWRITE_NEVER;
              break;
            case 4:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              break;
            case 5:
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
              break;
          }
        }
        break;
      default:
        nameok=true;
        returnvalue=NM_NEWNAME_OK;
        break;
    }
    _freesafe(newdest);
    round++;
  } while((nameok==false)&&(skip==false)&&(cancel==false));
  if((skip==true)||(cancel==true)) {
    _freesafe(newname);
    *newname_return=NULL;
  } else {
    *newname_return=newname;
  }
  _freesafe( title );
  return returnvalue;
}

NormalMode::nm_copy_t
NormalMode::copydir(const FileEntry *fe,NM_CopyOp_Dir *cod,struct NM_copyorder *copyorder,bool acceptrename,char *destdir)
{
  NM_CopyOp_Dir *cod1=NULL,*tcod;
  bool nameok,createdir;
  nm_newname_t e1;
  char *destsubdir;
  Datei::d_fe_t de1;
  char *newname=NULL;
  int id1;
  FileEntry *subfe;
  bool isdir,skip,cancel;
  char *buttonstr,*textstr;
  int erg;
  nm_copy_t ce1;
  CopyOpWin *cowin=copyorder->cowin;
  bool nodatabase;
  bool docopy,failedmove=false;
  struct utimbuf utb;

  if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  skip = cancel = false;
  if ( cowin != NULL ) {
    // call newfile without destination so the user
    // can see the filename in the copyopwin when he will
    // be asked for the new name
    cowin->newfile( fe->name, NULL );
    if ( cowin->redraw() != 0 ) cancel = true;
  }

  if ( cancel == false ) {
    e1=getNewName(fe,copyorder,destdir,&newname,true,acceptrename);
    if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_USE)) {
      // build new name
      destsubdir=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
      strcpy(destsubdir,destdir);
      if(strlen(destsubdir)>1) strcat(destsubdir,"/");
      strcat(destsubdir,newname);
      
      nameok=false;
      createdir=true;
      if(e1==NM_NEWNAME_USE) {
        // there is already such dir (or a symlink to a dir) -> use it
        nameok=true;
        createdir=false;
      } else if(e1==NM_NEWNAME_OK) {
        // check for fileexists (should not)
        // if nethertheless then skip this dir because the rename-func wasn't able to remove it
        de1=Datei::lfileExistsExt(destsubdir);
        if(de1==Datei::D_FE_NOFILE) {
          // everything is ok
          nameok=true;
        } else if(de1==Datei::D_FE_DIR) {
          // dest is a dir
          // should not happend but ok
          nameok=true;
          createdir=false;
          e1=NM_NEWNAME_USE; // for moving it's important to set this, so it doesn't try to
          // move
        }
      }
      if(nameok==true) {
        docopy=true;
        failedmove=false;
        if(copyorder->move==true) {
          if(e1==NM_NEWNAME_USE) {
            // because of using the destination dir
            // we need to do as failed rename
            failedmove=true;
          } else {
            if ( worker_rename( fe->fullname, destsubdir ) == 0 ) {
              // success
              docopy=false;
            } else {
              // failure
              failedmove=true;
            }
          }
        }
        
        if(docopy==true) {
          if(cowin!=NULL) cowin->newfile(fe->name,destsubdir);
          if(createdir==true) {
            if ( worker_mkdir( destsubdir, 0700 ) != 0 ) {
              // failed to create dir -> skip
              //            textstr="Failed to create dir|I will skip this dir!";
              //            buttonstr="Ok|Cancel";
              
              if(cowin!=NULL) cowin->stoptimer();
              textstr=(char*)_allocsafe(strlen(catalog.getLocale(126))+strlen(destsubdir)+1);
              sprintf(textstr,catalog.getLocale(126),destsubdir);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                      catalog.getLocale(8));
              erg = request( copyorder->cowin, catalog.getLocale( 347 ), textstr, buttonstr );
              _freesafe(buttonstr);
              _freesafe(textstr);
              if(erg==1) cancel=true;
              nameok=false;
              if(cowin!=NULL) cowin->conttimer();
            }
          }
          if(nameok==true) {
            // first copy all subdirs
            // then copy all files 
            if(nodatabase==false) {
              id1=cod->subdirs->initEnum();
              cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
              while(cod1!=NULL) {
                ce1=copydir(cod1->fileentry,cod1,copyorder,false,destsubdir);
                if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                  // success
                  cod->error_counter+=cod1->error_counter;
                } else {
                  cod->error_counter+=cod1->error_counter+1;
                  if(ce1==NM_COPY_CANCEL) {
                    cancel=true;
                    break;
                  }
                }
                cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
              }
              cod->subdirs->closeEnum(id1);
            }
            
            // next only if read this dir correctly
            if((cod->ok==true)&&(cancel==false)) {
              for ( Verzeichnis::verz_it subfe_it1 = cod->verz->begin();
                    subfe_it1 != cod->verz->end();
                    subfe_it1++ ) {
                subfe = *subfe_it1;
                if(strcmp(subfe->name,"..")!=0) {
                  isdir=false;
                  if(subfe->isDir()==true) {
                    if(subfe->isLink==false) isdir=true;
                    else if(copyorder->follow_symlinks==true) isdir=true;
                  }
                  if(isdir==false) {
                    ce1=copyfile(subfe,copyorder,false,destsubdir);
                    if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                      // success
                    } else {
                      cod->error_counter++;
                      if(ce1==NM_COPY_CANCEL) {
                        cancel=true;
                        break;
                      }
                    }
                  } else {
                    if(nodatabase==true) {
                      // no database so copy the dir
                      tcod=new NM_CopyOp_Dir(subfe);
                      if ( tcod->user_abort == false ) {
                        ce1=copydir(subfe,tcod,copyorder,false,destsubdir);
                        if(ce1==NM_COPY_OK) {
                          // success
                          cod->error_counter+=tcod->error_counter;
                        } else {
                          cod->error_counter+=tcod->error_counter+1;
                          if(ce1==NM_COPY_CANCEL) {
                            cancel=true;
                          }
                        }
                      } else cancel = true;
                      delete tcod;
                      if(cancel==true) break;
                    }
                  }
                }
              }
            }
            // finally change the permissions of the dir
            if((createdir==true)&&(copyorder->preserve_attr==true)) {
              // fe is a dir so if it is a link it is not corrupt
              worker_chown( destsubdir, ( fe->isLink == true ) ? fe->duserid() : fe->userid(),
                            ( fe->isLink == true ) ? fe->dgroupid() : fe->groupid() );
              worker_chmod( destsubdir, ( fe->isLink == true ) ? fe->dmode() : fe->mode() );
              utb.actime = ( fe->isLink == true ) ? fe->dlastaccess() : fe->lastaccess();
              utb.modtime = ( fe->isLink == true ) ? fe->dlastmod() : fe->lastmod();
              worker_utime(destsubdir,&utb);
            }
          } else skip=true;
        }
      } else skip=true;
      _freesafe(destsubdir);
    } else {
      switch(e1) {
        case NM_NEWNAME_CANCEL:
          cancel=true;
        default:
          skip=true;
          break;
      }
    }
  }
  if((skip==true)&&(cowin!=NULL)) {
    // dir skiped so dec CopyOpWin counter by the files/dirs skiped
    cowin->dec_file_counter(cod->files);
    cowin->dec_dir_counter(cod->dirs);
    cowin->dec_byte_counter(cod->bytes);
  }
  if ( cod->verz->getInvalidEntries() == NULL ) {
    skip = true;
  } else {
    if ( cod->verz->getInvalidEntries()->empty() == false ) {
#if 0
      // same comment as in ::enterDir
      ArrayList *invalid_entries = cod->verz->getInvalidEntries();
      char *my_msg;
      
      my_msg = (char*)_allocsafe( strlen( catalog.getLocale( 530 ) ) +
                                  strlen( cod->verz->getDir() ) + 1 );
      sprintf( my_msg, catalog.getLocale( 530 ), cod->verz->getDir() );
      showInvalidList( invalid_entries, my_msg );
      _freesafe( my_msg );
#endif
      // when there are some invalid files mark this dir as skipped
      // But I do this after the test above because this is no user initiated
      // skip so all normal files are copied (if everything was okay)
      skip = true;
    }
  }
  if(cowin!=NULL) {
    cowin->dir_finished();
    if(cowin->redraw()!=0) cancel=true;
  }
  if(newname!=NULL) _freesafe(newname);
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) {
    cod->error_counter++;
    return NM_COPY_SKIP;
  }
  if(copyorder->move==true) {
    if((cod->error_counter==0)&&(failedmove==true)) {
      // put fe in list

      // Man koennte das FE auch also Kopie benutzen
      // man muesste nur den Vergleich beim spaeteren Loeschen anpassen
      copy_deletefe_list->push_back( fe );
      return NM_COPY_NEED_DELETE;
    }
  }
  return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::copyfile(const FileEntry *fe,struct NM_copyorder *copyorder,bool acceptrename,char *destdir)
{
  bool nameok;
  nm_newname_t e1;
  char *newfullname;
  Datei::d_fe_t de1;
  char *newname=NULL;
  bool useregcopy, skip, cancel;
  int erg;
  CopyOpWin *cowin=copyorder->cowin;
  bool docopy,failedmove=false,opok=false;
  struct utimbuf utb;
  const char *myerrstr;
  loff_t bytesToCopy;
  mode_t modeCheck;
  bool useLinkDest;
  std::string text, button;
  
  if ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) )
    bytesToCopy = fe->dsize();
  else
    bytesToCopy = fe->size();

  skip = cancel = false;
  if ( cowin != NULL ) {
    // call newfile without destination so the user
    // can see the filename in the copyopwin when he will
    // be asked for the new name
    cowin->newfile( fe->name, NULL );
    if ( cowin->redraw() != 0 ) cancel = true;
  }

  if ( cancel == false ) {
    e1=getNewName(fe,copyorder,destdir,&newname,false,acceptrename);
    
    if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_OVERWRITE)) {
      // build new name
      newfullname=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
      strcpy(newfullname,destdir);
      if(strlen(newfullname)>1) strcat(newfullname,"/");
      strcat(newfullname,newname);
      
      nameok=false;
      skip=false;
      if(e1==NM_NEWNAME_OVERWRITE) {
        nameok=true;
      } else if(e1==NM_NEWNAME_OK) {
        // check for fileexists (should not)
        de1=Datei::lfileExistsExt(newfullname);
        if(de1==Datei::D_FE_NOFILE) {
          // everything is ok
          nameok=true;
        } else if(de1==Datei::D_FE_FILE) {
          // dest is a dir
          // should not happend but ok try to overwrite this
          nameok=true;
        }
      }
      if(nameok==true) {
        if ( fe->isSameFile( newfullname, copyorder->follow_symlinks ) == false ) {
          // wenn move true, dann rename probieren
          //        klappt rename nicht, muss flag gesetzt werden und normal weitergemacht
          //        werden und bei korrekten kopieren FE in liste gepackt werden und NEED_DELETE
          //        zurueckgeliefert werden
          //      also: 1.flag docopy am Anfang auf true setzen
          //            2.bei move==true rename probieren
          //            2.1.bei Erfolg docopy auf false, Rueckgabewert ist automatisch OK
          //            2.2.bei Misslingen failedmove auf true setzen
          //            3.bei docopy==true normal kopieren
          //            3.1.bei Misslingen kommt halt skip zurueck
          //            3.2.Bei Erfolg und failedmove==true muss FE in liste gepackt werden
          //                und NEED_DELETE zurueckgeliefert werden
          docopy=true;
          failedmove=false;
          
          if ( cowin != NULL ) cowin->newfile( fe->name, newfullname );
          
          if(copyorder->move==true) {
            if ( worker_rename( fe->fullname, newfullname ) == 0 ) {
              // success
              docopy=false;
            } else {
              // failure
              failedmove=true;
            }
          }
          
          if(docopy==true) {
            // first try to open the file
            // in case of failure give a requester with choose "Delete"
            // then try to delete it and retry to open
            // if all fails, skip
            useregcopy = false;
            if ( S_ISREG( fe->mode() ) ) useregcopy = true;
            else if ( ( fe->isLink == true ) &&
                      ( copyorder->follow_symlinks == true ) &&
                      ( fe->isCorrupt == false ) &&
                      ( S_ISREG( fe->dmode() ) ) ) useregcopy = true;
            
            if ( useregcopy == true ) {
              reg_copy_erg_t rce = copyfile_reg( fe->fullname,
                                                 newfullname,
                                                 destdir,
                                                 newname,
                                                 ( fe->isLink == true ) ? fe->dmode() : fe->mode(),
                                                 copyorder->cowin,
                                                 bytesToCopy );
              if ( rce == REG_COPY_OK ) {
                opok = true;
              } else if ( rce == REG_COPY_SKIP ) {
                skip = true;
              } else {
                // everything else (including ERROR) is cancel
                cancel = true;
              }
            } else {
              // no regular copy
              if ( e1 == NM_NEWNAME_OVERWRITE ) {
                // there is already a file but because we cannot overwrite
                // with special file I will delete it
                // remember the user already choose to overwrite in getNewName!
                if ( worker_remove( newfullname ) != 0 ) {
                  // remove failed => cannot continue
                  
                  if ( cowin != NULL ) cowin->stoptimer();
                  myerrstr = strerror( errno );
                  text = AGUIXUtils::formatStringToString( catalog.getLocale( 516 ), fe->fullname, newfullname, myerrstr );
                  button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                  if ( erg == 0 ) {
                    skip = true;
                  } else {
                    cancel = true;
                  }
                  if ( cowin != NULL ) cowin->conttimer();
                }
              }
              //TODO: mc will skip corrupt links when follow_symlinks is used
              //      I could do this too if I set modeCheck = 0 so it will got to
              //      the else part
              if ( ( fe->isLink == true ) &&
                   ( copyorder->follow_symlinks == true ) &&
                   ( fe->isCorrupt == false ) ) {
                useLinkDest = true;
                modeCheck = fe->dmode();
              } else {
                useLinkDest = false;
                modeCheck = fe->mode();
              }
              
              if ( ( skip == false ) && ( cancel == false ) ) {
                if ( ( S_ISCHR( modeCheck ) ) ||
#ifdef S_ISSOCK
                     ( S_ISSOCK( modeCheck ) ) ||
#endif
                     ( S_ISBLK( modeCheck ) ) ) {
                  worker_struct_stat statbuf;
                  
                  if ( useLinkDest == true ) {
                    erg = worker_stat( fe->fullname, &statbuf );
                  } else {
                    erg = worker_lstat( fe->fullname, &statbuf );
                  }
                  if ( erg == 0 ) {
                    // mknod
                    if ( worker_mknod( newfullname, statbuf.st_mode, statbuf.st_rdev ) != 0 ) {
                      if ( cowin != NULL ) cowin->stoptimer();
                      myerrstr = strerror( errno );
                      text = AGUIXUtils::formatStringToString( catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                      button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                      erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                      if ( erg == 0 ) {
                        skip = true;
                      } else {
                        cancel = true;
                      }
                      if ( cowin != NULL ) cowin->conttimer();
                    } else opok = true;
                  } else {
                    if ( cowin != NULL ) cowin->stoptimer();
                    myerrstr = strerror( errno );
                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 518 ), fe->fullname, newfullname, myerrstr );
                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                    if ( erg == 0 ) {
                      skip = true;
                    } else {
                      cancel = true;
                    }
                    if ( cowin != NULL ) cowin->conttimer();
                  }
                } else if ( S_ISFIFO( modeCheck ) ) {
                  // mkfifo
                  if ( mkfifo( newfullname, modeCheck ) != 0 ) {
                    if ( cowin != NULL ) cowin->stoptimer();
                    myerrstr = strerror( errno );
                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                    if ( erg == 0 ) {
                      skip = true;
                    } else {
                      cancel = true;
                    }
                    if ( cowin != NULL ) cowin->conttimer();
                  } else opok = true;
                } else if ( S_ISLNK( modeCheck ) ) {
                  // it's a link so create a new symlink which points to the same destination
                  char *linkdest = fe->getDestination();
                  if ( linkdest != NULL ) {
                    if ( worker_symlink( linkdest, newfullname ) != 0 ) {
                      if ( cowin != NULL ) cowin->stoptimer();
                      myerrstr = strerror( errno );
                      text = AGUIXUtils::formatStringToString( catalog.getLocale( 517 ), fe->fullname, newfullname, myerrstr );
                      button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                      erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                      if ( erg == 0 ) {
                        skip = true;
                      } else {
                        cancel = true;
                      }
                      if ( cowin != NULL ) cowin->conttimer();
                    } else opok = true;
                    _freesafe( linkdest );
                  } else {
                    if ( cowin != NULL ) cowin->stoptimer();
                    myerrstr = strerror( errno );
                    text = AGUIXUtils::formatStringToString( catalog.getLocale( 519 ), fe->fullname, newfullname, myerrstr );
                    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                    erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                    if ( erg == 0 ) {
                      skip = true;
                    } else {
                      cancel = true;
                    }
                    if ( cowin != NULL ) cowin->conttimer();
                  }
                } else {
                  if ( cowin != NULL ) cowin->stoptimer();
                  myerrstr = strerror( errno );
                  text = AGUIXUtils::formatStringToString( catalog.getLocale( 520 ), fe->fullname );
                  button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
                  erg = request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
                  if ( erg == 0 ) {
                    skip = true;
                  } else {
                    cancel = true;
                  }
                  if ( cowin != NULL ) cowin->conttimer();
                }
              }
              //TODO: muss hier nicht eher add_curbytes_copied aufgerufen werden?
              //            if(cowin!=NULL) cowin->set_bytes_to_copy_curfile(fe->size);
              if ( cowin != NULL ) cowin->add_curbytes_copied( fe->size() );
            }
          }
        } else {
          text = AGUIXUtils::formatStringToString( catalog.getLocale( 113 ), fe->fullname );
          button = catalog.getLocale( 11 );
          
          request( copyorder->cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
        }
      } else skip=true;
      
      if ( opok == true ) {
        // try to change the mode because open modify it with umask
        // it doesn't matter if this fails 
        if(copyorder->preserve_attr==true) {
          //TODO:currently deactivated
          //     chown is necessary because link can have an owner
          //     but chmod could be optional
          // only apply new mode and time if it's not a link
          // doing this would not do anything wrong
          // but chown/chmod/utime would update the access/change time
          // of the destination file although we don't have to touch it
          //if ( ( fe->isLink == false ) || ( copyorder->follow_symlinks == true ) ) {
          tryApplyOwnerPerm( newfullname,
                             ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->duserid() : fe->userid(),
                             ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dgroupid() : fe->groupid(),
                             ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dmode() : fe->mode(),
                             copyorder );
          utb.actime = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dlastaccess() : fe->lastaccess();
          utb.modtime = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dlastmod() : fe->lastmod();
          worker_utime(newfullname,&utb);
          //}
        }
      }
      
      _freesafe(newfullname);
    } else {
      switch(e1) {
        case NM_NEWNAME_CANCEL:
          cancel=true;
        default:
          skip=true;
          break;
      }
    }
  }

  if(cowin!=NULL) {
    if ( skip == true ) {
      // the CopyOpWin need to substract bytesToCopy from global bytes sum
      // and already copied bytes from copyied bytes sum
      // if we called newfile() (which resets this counter)
      cowin->file_skipped( bytesToCopy, true );
    } else {
      cowin->file_finished();
    }
    if(cowin->redraw()!=0) cancel=true;
  }
  if(newname!=NULL) _freesafe(newname);
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) return NM_COPY_SKIP;
  if(copyorder->move==true) {
    if((opok==true)&&(failedmove==true)) {
      // remember: failedmove is true when OS function rename failed
      //           in this case we have to copy the file and delete the source
      //           so add this entry only when copy is complete (opok) AND
      //           rename failed (failedmove)
      // put fe in list
      copy_deletefe_list->push_back( fe );
      return NM_COPY_NEED_DELETE;
    }
  }
  return NM_COPY_OK;
}

bool
NormalMode::isCorrectViewProg(const char *str)
{
  const char *pos;
  if(strlen(str)<1) return true;
  pos=strstr(str,"%s");
  if(pos!=NULL) return true;
  return false;
}

void
NormalMode::deletef(struct NM_deleteorder *delorder)
{
  NM_specialsourceInt *ss1;
//  bool found;
  bool enter;
  NM_CopyOp_Dir *cod1,*tcod;
  unsigned long files,dirs,gf,gd;
  bool skip,cancel=false;
  DeleteOpWin *dowin;
  nm_copy_t ce1;
  char *buttonstr,*textstr;
  int erg;
  bool empty;
  int tnrfiles,tnrdirs;
  std::list<NM_specialsourceInt*> *dellist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  
  if(delorder==NULL) return;
  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to delete
    return;
  }

  if ( ce->dirOpened() == false ) return;

  tnrfiles = ce->getNrOfFiles( 1 );
  tnrdirs = ce->getNrOfDirs( 1 );

  if ( ( delorder->source == delorder->NM_ALLENTRIES ) &&
       ( ( tnrfiles + tnrdirs ) < 1 ) ) {
    if ( ce->getActiveFE() != NULL ) {
      if ( strcmp( ce->getActiveFE()->name, ".." ) != 0 ) {
        if ( ce->getActiveFE()->isDir() == true ) tnrdirs++;
        else tnrfiles++;
      }
    }
  }
  
  if ( ( tnrfiles + tnrdirs ) < 1 )
    return;

  finishsearchmode();
  
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 145 ) ) + A_BYTESFORNUMBER( tnrfiles ) +
                               A_BYTESFORNUMBER( tnrdirs ) + 1 );
  sprintf(textstr,catalog.getLocale(145),tnrfiles,tnrdirs);
  erg = request( catalog.getLocale( 123 ), textstr, buttonstr );
  _freesafe(buttonstr);
  _freesafe(textstr);
  if(erg!=0) return;

  // clear the reclist so the slace doesn't block us because he is still reading
  // from disk
  ft_rec_list_clear();
  
  dowin=delorder->dowin;
  if(dowin!=NULL) {
    dowin->open();
    dowin->setmessage(catalog.getLocale(140));
    if ( dowin->redraw() != 0 ) cancel = true;
  }
  dellist = new std::list<NM_specialsourceInt*>;
  switch(delorder->source) {
    case NM_deleteorder::NM_SPECIAL:
        break;
    case NM_deleteorder::NM_ONLYACTIVE:
/*OPT:At the moment I doesn't allow onlyactive
      if(ce->activefe!=NULL) {
        lvc=lv->getActiveLVC();
        if(lvc!=NULL) {
          if(lvc->getData()___wrong___==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
	      ss2 = new NM_specialsource( ce->activefe );
              ss2->lvc=lvc;
              ss2->cod=NULL;
              dellist->addElement(ss2);
            }
          }
        }
      }*/
      break;
    case NM_deleteorder::NM_ONLYSELECTED:
      getSelFiles( dellist, NM_GETFILES_ONLYSELECT, false );
      break;
    default:  // all selected entries
      getSelFiles( dellist, NM_GETFILES_SELORACT, false );
      break;
  }
  if(dowin!=NULL) dowin->redraw();

  // create the NM_CopyOp_Dir for each dir in dellist
  files=dirs=gf=gd=0;
  
  for ( iti1 = dellist->begin(); iti1 != dellist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    enter=false;
    if ( ss1->entry()->isDir() == true ) {
      // fe is a dir, check if it is a link and take it only when follow_symlinks==true
      if ( ss1->entry()->isLink == false ) enter = true;
    }
    if(enter==true) {
      // fe is a dir so creating corresponding entry
      cod1 = new NM_CopyOp_Dir( ss1->entry() );
      if ( cod1->user_abort == false ) {
        // recursive call
        if(cod1->createSubDirs(delorder,&gf,&gd)!=0) cancel=true;
      } else cancel = true;
      // add the values from this subdir to this dir
      files+=cod1->files;
      dirs+=cod1->dirs;
      
      ss1->cod=cod1;
      // this is a dir so inc the counter
      dirs++;
      gd++;
      if(dowin!=NULL) {
        dowin->set_files_to_delete(gf);
        dowin->set_dirs_to_delete(gd);
        if(dowin->redraw()!=0) cancel=true;;
      }
    } else {
      // is not dir (mostly a file but can also be links ...)
      files++;
      gf++;
      ss1->cod=NULL;
    }
  }

  if(dowin!=NULL) {
    dowin->set_files_to_delete(files);
    dowin->set_dirs_to_delete(dirs);
  }
  // now database ready, can start copy-process
  skip=false;
  for ( iti1 = dellist->begin(); ( iti1 != dellist->end() ) && ( cancel == false ); iti1++ ) {
    ss1 = *iti1;
    tcod=ss1->cod;
    if ( cancel == true ) break;
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(tcod!=NULL) {
      // determine if the dir is empty or not
      if((tcod->files+tcod->dirs)>0) empty=false;
      else empty=true;
      skip=false;
      // skip if the user doesn't want to delete non-empty dirs
      if((delorder->dirdel==delorder->NM_DIRDELETE_NONE)&&(empty==false)) skip=true;
      // if the user doesn't select ALL or NONE and the the dir isn't empty show a request
      if((delorder->dirdel==delorder->NM_DIRDELETE_NORMAL)&&(empty==false)) {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(146))+
                                  strlen( ss1->entry()->fullname ) + 1 );
        sprintf( textstr, catalog.getLocale( 146 ), ss1->entry()->fullname );
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(287))+1+
                                    strlen(catalog.getLocale(288))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(11),
                                           catalog.getLocale(287),
                                           catalog.getLocale(288),
                                           catalog.getLocale(225),
                                           catalog.getLocale(8));
        erg = request( delorder->dowin, catalog.getLocale( 123 ), textstr, buttonstr );
        _freesafe(buttonstr);
        _freesafe(textstr);
        switch(erg) {
          case 1:
            delorder->dirdel=delorder->NM_DIRDELETE_ALL;
            break;
          case 2:
            delorder->dirdel=delorder->NM_DIRDELETE_NONE;
            skip=true;
            break;
          case 3:
            skip=true;
            break;
          case 4:
            cancel=true;
            skip=true;
            break;
        }
      }
      if(skip==false) {
        ce1 = deldir( ss1->entry(), tcod, delorder );
        if(ce1==NM_COPY_OK) {
          // success
        } else if(ce1==NM_COPY_CANCEL) {
          cancel=true;
        }
        if((tcod->error_counter==0)&&(cancel==false)) {
          // successful delete this dir so remove it form the lv
          // first show this lvc
          if(ss1->row>=0) lv->showRow(ss1->row);
          removeEntry( ss1->entry(), ss1->row );
          fixSpecialSourceList( dellist, ss1->row );
        }
      }
    } else {
      if ( dowin != NULL ) dowin->setfilename( ss1->entry()->fullname );
      ce1 = delfile( ss1->entry(), delorder, false );
      if(ce1==NM_COPY_OK) {
        // success
        if(ss1->row>=0) lv->showRow(ss1->row);
        removeEntry( ss1->entry(), ss1->row );
        fixSpecialSourceList( dellist, ss1->row );
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
    }
    if ( ss1->row >= 0 ) {
      // this can and will changed the wrong row when the entry
      // was deleted successfully
      // but for this case this doesn't matter because
      // the disabled visual mark is the default
      lv->setVisMark( ss1->row, false );
    }
    if(dowin!=NULL) {
      if(dowin->redraw()!=0) cancel=true;
    }
  }
  if(dowin!=NULL) dowin->close();
  
  for ( iti1 = dellist->begin(); iti1 != dellist->end(); iti1++ ) {
    ss1 = *iti1;
    if(ss1->cod!=NULL) delete ss1->cod;
    delete ss1;
  }
  delete dellist;

  ce->restartcheck();
  rebuildView();
  aguix->Flush();
}

void NormalMode::removeEntry( FileEntry *fe, int row )
{
    if ( ce == NULL ) return;
    if ( fe == NULL ) return;
    if ( ce->dirOpened() == false ) return;
    if ( lv->isValidRow( row ) == false ||
         row == 0 )
        return;
    
    if ( getFEForRow( row ) != fe )
        return;
    
    // we will remove an entry so clear this list
    // so it won't contain wrong pointers
    ft_rec_list_clear();
    
    // if it's the active one then move the active up by one
    if ( fe == ce->getActiveFE() ) {
        int prev_row = row - 1;
        
        ce->setActiveFE( getFEForRow( prev_row ) );
        makeListViewRowActive( prev_row );
        lv->showActive();
    }
    
    ce->removeEntry( fe );
    
    lv->deleteRow( row );
    //TODO make it optional to use FPS limit
    lv->timeLimitedRedraw();
    
    delete fe;
    
    // update cache info
    showCacheState();
}

NormalMode::nm_copy_t
NormalMode::deldir(const FileEntry *fe,NM_CopyOp_Dir *cod,struct NM_deleteorder *delorder)
{
  NM_CopyOp_Dir *cod1=NULL;
  int id1;
  FileEntry *subfe;
  bool isdir,skip,cancel;
//  char *buttonstr,*textstr;
//  int erg;
  nm_copy_t ce1;
  DeleteOpWin *dowin=delorder->dowin;
  bool delerror;

  skip=cancel=false;

  if(dowin!=NULL) dowin->setfilename(fe->fullname);
  // first del all subdirs
  // then del all files 
  id1=cod->subdirs->initEnum();
  cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
  while(cod1!=NULL) {
    ce1=deldir(cod1->fileentry,cod1,delorder);
    if(ce1==NM_COPY_OK) {
      // success
      cod->error_counter+=cod1->error_counter;
    } else {
      cod->error_counter+=cod1->error_counter+1;
      if(ce1==NM_COPY_CANCEL) {
        cancel=true;
        break;
      }
    }
    cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
  }
  cod->subdirs->closeEnum(id1);

  // next only if read this dir correctly
  if((cod->ok==true)&&(cancel==false)) {
    for ( Verzeichnis::verz_it subfe_it1 = cod->verz->begin();
          subfe_it1 != cod->verz->end();
          subfe_it1++ ) {
      subfe = *subfe_it1;
      if(strcmp(subfe->name,"..")!=0) {
        isdir=false;
        if(subfe->isDir()==true) {
          if(subfe->isLink==false) isdir=true;
        }
        if(isdir==false) {
          ce1=delfile(subfe,delorder,false);
          if(ce1==NM_COPY_OK) {
            // success
          } else {
            cod->error_counter++;
            if(ce1==NM_COPY_CANCEL) {
              cancel=true;
              break;
            }
          }
        }
      }
    }
  }
  if((skip==true)&&(dowin!=NULL)) {
    // dir skiped so dec CopyOpWin counter by the files/dirs skiped
    dowin->dec_file_counter(cod->files);
    dowin->dec_dir_counter(cod->dirs);
  }
  if ( cod->verz->getInvalidEntries() == NULL ) {
    // in difference to copydir don't skip but try to delete the directory
    // this case will happen when the directory isn't readable or executeable
    // if this unreadable dir is empty the delete will work
    //skip = true;
  } else {
    if ( cod->verz->getInvalidEntries()->empty() == false ) {
      // when there are some invalid files mark this dir as skipped
      // But I do this after the test above because this is no user initiated
      // skip so all normal files are deleted (if everything was okay)
      skip = true;
    }
  }
  delerror=true;
  if((skip==false)&&(cancel==false)) {
    if(cod->error_counter==0) {
      ce1=delfile(fe,delorder,true);
      if(ce1==NM_COPY_OK) {
        // success
        delerror=false;
      } else {
        cod->error_counter++;
        if(ce1==NM_COPY_CANCEL) {
          cancel=true;
        }
      }
    }
  }
  if(delerror==true) {
    char *textstr=(char*)_allocsafe(strlen(catalog.getLocale(289))+strlen(fe->fullname)+1);
    sprintf(textstr,catalog.getLocale(289),fe->fullname);
    char *buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    int erg = request( delorder->dowin, catalog.getLocale( 125 ), textstr, buttonstr );
    _freesafe(buttonstr);
    _freesafe(textstr);
    if(erg==1) cancel=true;
    else skip=true;
  }
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) {
    cod->error_counter++;
    return NM_COPY_SKIP;
  }
  return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::delfile(const FileEntry *fe,struct NM_deleteorder *delorder,bool dir)
{
  bool skip,cancel;
  char *textstr,*buttonstr;
  int erg;
  DeleteOpWin *dowin=delorder->dowin;

  skip=cancel=false;

  if ( dir == true ) erg = worker_rmdir( fe->fullname );
  else erg = worker_unlink( fe->fullname );
  if(erg!=0) {
//    textstr="Failed to remove this file!";
//    buttonstr="Ok|Cancel";

    if(dir==true) {
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(147))+strlen(fe->fullname)+1);
      sprintf(textstr,catalog.getLocale(147),fe->fullname);
    } else { 
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(286))+strlen(fe->fullname)+1);
      sprintf(textstr,catalog.getLocale(286),fe->fullname);
    }
    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    erg = request( delorder->dowin, catalog.getLocale( 347 ), textstr, buttonstr );
    _freesafe(buttonstr);
    _freesafe(textstr);
    if(erg==1) cancel=true;
    else skip=true;
  }
  if(dowin!=NULL) {
    if(dir==true) {
      dowin->dir_finished();
      if(dowin->redraw()!=0) cancel=true;
    } else dowin->file_finished();
  }
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) return NM_COPY_SKIP;
  return NM_COPY_OK;
}

void
NormalMode::update(bool reset_dirsizes)
{
  update( reset_dirsizes, false );
}

void
NormalMode::update( bool reset_dirsizes, bool keep_filetypes )
{
    if ( ce == NULL )
        return;
    
    char *dir;
  
    finishsearchmode();
    
    dir = dupstring( ce->getDir() );
    for(;;) {
        int e = enterDir( dir );
        if ( e == 0 ) break;  // read ok, so finish
        if ( strcmp( dir, "/" ) == 0 ) break;  // there is no dir we can display anymore, so finish
        char *tstr = ParentDir( dir, NULL );
        _freesafe( dir );
        dir = tstr;
    }
    _freesafe( dir );

    if ( ce != NULL ) {
        if ( keep_filetypes == true ) {
            ce->reset_checkfe();
        } else {
            ce->restartcheck();
        }
        if ( reset_dirsizes == true )
            ce->reset_dirsizes();
    }
    
    rebuildView();
}

void
NormalMode::makedir()
{
  if(ce==NULL) return;
  FileEntry *tfe;
  int row;
  char *buttonstr,*textstr,*return_str,*destdir;
  int erg;
  
  finishsearchmode();
  
  textstr = dupstring( catalog.getLocale(152) );
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocaleCom(27),textstr,"",buttonstr,&return_str);
  _freesafe(buttonstr);
  _freesafe( textstr );
  if((erg==0)&&(strlen(return_str)>0)) {
    destdir=(char*)_allocsafe(strlen(ce->getDir())+1+strlen(return_str)+1);
    strcpy(destdir,ce->getDir());
    if(strlen(destdir)>1) strcat(destdir,"/");
    strcat(destdir,return_str);

    if ( worker_mkdir( destdir, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH ) != 0 ) {
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(127))+strlen(destdir)+1);
      sprintf(textstr,catalog.getLocale(127),destdir);
      buttonstr = dupstring( catalog.getLocale(11) );
      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
      _freesafe(textstr);
      _freesafe( buttonstr );
    } else {
      // update the dir
      update(false);
      // find the new dir
      for ( Verzeichnis::verz_it tfe_it1 = ce->begin();
            tfe_it1 != ce->end();
            tfe_it1++ ) {
        tfe = *tfe_it1;
        if(strcmp(tfe->name,return_str)==0) {
          // found
          ce->setActiveFE( tfe );
          row = 0;
          while( row < lv->getElements() ) {
            if ( lv->getData( row ) == tfe->getID() ) {
              // lvc found
              makeListViewRowActive( row );
              break;
            }
            row++;
          }
          break;
        }
      }
      lv->showActive();
      lv->redraw();
    }
    _freesafe(destdir);
  }
  _freesafe(return_str);
}

void
NormalMode::externprog(class NM_externorder *extorder)
{
  char *tmpname,*tmpoutput,*exestr,*tstr;
  Datei *fp;
  int lastsize[2];
  bool useint;
  bool cancel=false;
  WPUContext *wpu;
  bool oldrec, oldtd;
  std::string string1;
  bool removeTMPFiles;

  if(extorder==NULL) return;
  if(extorder->com_str==NULL) return;  // nothing to do
  if ( extorder->wpu == NULL ) return;
  wpu = extorder->wpu;

  oldrec = wpu->getRecursive();
  oldtd = wpu->getTakeDirs();
  wpu->setRecursive( extorder->recursive );
  wpu->setTakeDirs( extorder->take_dirs );

  finishsearchmode();

  tmpname=Datei::createTMPName();
  tmpoutput=Datei::createTMPName();

  if ( tmpname != NULL && tmpoutput != NULL ) {
    // runCommand will remove the tmpfiles but if we do runCommand
    // (because of an error) remove them here
    removeTMPFiles = true;
    fp=new Datei();
    if(fp->open(tmpname,"w")==0) {
      fp->putString("#! /bin/sh\n");
      if ( wpu->getBaseDir() != NULL ) {
        if ( ( worker_islocal( wpu->getBaseDir() ) == 1 ) &&
             ( extorder->dontcd == false ) ) {
          tstr = AGUIX_catTrustedAndUnTrusted( "cd ", wpu->getBaseDir() );
          fp->putString( tstr );
          _freesafe( tstr );
          fp->putString( "\n" );
        }
      }
      lastsize[0] = wpu->filelistRealSize( false );
      lastsize[1] = wpu->filelistRealSize( true );
      for(int loopnr=1;;loopnr++) {
        exestr = wpu->parse( extorder->com_str, EXE_STRING_LEN - strlen( tmpoutput ) - 3 );
        if(exestr!=NULL) {
          string1 = Worker_secureCommandForShell( exestr );
          if ( string1.length() > 0 ) {
            if(extorder->extstart==extorder->NM_EXT_SHOW_OUTPUT) {
              tstr=(char*)_allocsafe(strlen(string1.c_str())+3+strlen(tmpoutput)+1);
              if(loopnr==1) sprintf(tstr,"%s >%s",string1.c_str(),tmpoutput);
              else sprintf(tstr,"%s >>%s",string1.c_str(),tmpoutput);
              fp->putString( tstr);
              _freesafe(tstr);
            } else {
              fp->putString( string1.c_str() );
            }
            fp->putString("\n");
          }
          _freesafe(exestr);
          if(extorder->separate_each_entry==false) break;
          if ( wpu->filelistRealSize( false ) == 0 ) break;
          if ( wpu->filelistRealSize( false ) == lastsize[0] ) {
            // endless loop
            //TODO: requester
            break;
          }
        } else {
          cancel=true;
          break;
        }
        lastsize[0] = wpu->filelistRealSize( false );
      }
      if(cancel==false) {
        if((extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL_AND_WAIT4KEY)||
           (extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL)) {
          /* in terminal starten */
          if(extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL_AND_WAIT4KEY) {
            fp->putStringExt("echo %s\n",catalog.getLocale(201));
            fp->putString("read x\n");
          }
          exestr=(char*)_allocsafe(strlen(wconfig->getTerminalStr())+strlen(tmpname)+1);
          sprintf(exestr,wconfig->getTerminalStr(),tmpname);
        } else {
          exestr=dupstring(tmpname);
        }
        if(extorder->extstart==extorder->NM_EXT_SHOW_OUTPUT) {
          useint=true;
          if(extorder->view_str!=NULL)
            if(strlen(extorder->view_str)>2)
              if(strstr(extorder->view_str,"%s")!=NULL) useint=false;
          if(useint==true) fp->putStringExt(OUTPUT_BIN,tmpoutput);
          else fp->putStringExt(extorder->view_str,tmpoutput);
          fp->putString("\n");
        }
        fp->close();
      
        if ( fp->errors() == 0 ) {
          worker_chmod( tmpname, 0700 );
          removeTMPFiles = false;
          parentlister->getWorker()->runCommand( exestr, tmpname, tmpoutput, extorder->inbackground );
        } else {
          std::string text = AGUIXUtils::formatStringToString( catalog.getLocale( 647 ), tmpname );
          std::string button = catalog.getLocale( 11 );
          request( catalog.getLocale( 347 ), text.c_str(), button.c_str() );
        }
        _freesafe(exestr);
      } else fp->close();
    }
    delete fp;
    if ( removeTMPFiles == true ) {
      worker_unlink( tmpname );
      worker_unlink( tmpoutput );
    }
  } else {
    std::string text = AGUIXUtils::formatStringToString( catalog.getLocale( 647 ), "" );
    std::string button = catalog.getLocale( 11 );
    request( catalog.getLocale( 347 ), text.c_str(), button.c_str() );
  }
  
  if ( tmpname != NULL ) _freesafe(tmpname);
  if ( tmpoutput != NULL ) _freesafe(tmpoutput);

  wpu->setRecursive( oldrec );
  wpu->setTakeDirs( oldtd );
}

#define PARSECAT( str1 ) \
do { \
  if ( ( quote == true ) && ( force_noquoting == false ) ) { \
    dstr[dpos] = '\0'; \
    newdstr = AGUIX_catTrustedAndUnTrusted( dstr, ( str1 ) ); \
    snprintf( dstr, maxlen, "%s", newdstr ); \
    _freesafe( newdstr ); \
    dstr[maxlen] = '\0'; \
    dpos = strlen( dstr ); \
    if ( dpos >= maxlen ) ende = true; \
  } else { \
    tlen = strlen( ( str1 ) ); \
    if ( ( dpos + tlen ) < maxlen ) { \
      strcpy( &dstr[dpos], ( str1 ) ); \
      dpos += strlen( ( str1 ) ); \
    } else ende = true; \
  } \
} while ( 0 );

#define PUTCHARDSTR( ch ) \
do { \
  if ( dpos < maxlen ) { \
    dstr[dpos++] = ch; \
  } else ende = true; \
} while ( 0 );

char *NormalMode::parseComStr( const char *sstr,class NM_externorder *extorder,int maxlen,NMExtList *entries[2], bool quote )
{
  char *dstr = NULL, *buf1, ch, *newdstr;
  int spos=0,dpos=0,bracketcount=0,bufpos=0;
  int mode=0,myside;
  bool otherside,nounselect,noext;
  int filemode,tpos,tmode;
  NormalMode *nm2,*tnm;
  Lister *l2;
  ListerMode *lm1;
  int tlen;
  bool ende,takefirst;
  char *tstr;
  const char *ctstr;
  NM_extern_fe *efe1;
  NMExtList *l;
  int id;
  bool cancel=false;
  int num;
  std::string str1;
  char buf2[ A_BYTESFORNUMBER( int ) + 2 ];
  bool efeskip;
  char *flagbuf;
  int flagbuf_size;
  bool force_noquoting;
  
  nm2=NULL;
  l2=parentlister->getWorker()->getOtherLister(parentlister);
  lm1=l2->getActiveMode();
  if(lm1!=NULL)
    if(lm1->isType("NormalMode")==true)
      nm2=(NormalMode*)lm1;
  
  myside=parentlister->getSide();
  dstr=(char*)_allocsafe(maxlen+1);
  flagbuf_size = strlen( sstr );
  flagbuf = (char*)_allocsafe( flagbuf_size + 1 );
  if(dstr!=NULL) {
    for(ende=false;ende==false;) {
      ch=sstr[spos++];
      if(ch=='\0') break;
      switch(mode) {
        case 1:
          // we are in a open bracket and waiting for close bracket
          if(ch=='{') {
            // other flag, perhaps useless, but perhaps it is in a String-requester
            // in any case just overtake it in the buffer
            if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
            bracketcount++;
          } else if(ch=='}') {
            // a closeing bracket
            if(bracketcount>1) {
              // this is a bracket in the bracket
              if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
            } else {
              // this is the awaited closebracket
              flagbuf[ bufpos ] = 0;
              // now flagbuf contains a flag which must be parsed
              mode=3;
            }
            bracketcount--;
          } else if(ch=='\\') {
            // backslash are only resolved on toplevel (bracketcount==0)
            // so overtake also this
            if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
            mode=4;
          } else {
            if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
          }
          break;
        case 2:
          PUTCHARDSTR( ch );
          mode=0;
          break;
        case 4:
          if ( bufpos < flagbuf_size ) flagbuf[ bufpos++ ] = ch;
          mode=1;
          break;
        default:
          // we are in no bracket
          if(ch=='\\') {
            // next character is protected and will be overtaken
            mode=2;
          } else if(ch!='{') {
            // a normal character
            PUTCHARDSTR( ch );
          } else {
            mode=1;
            bracketcount++;
            bufpos=0;
          }
          break;
      }
      if(mode==3) {
        if ( flagbuf[0] == '-' ) {
          force_noquoting = true;
          buf1 = flagbuf + 1;
        } else {
          force_noquoting = false;
          buf1 = flagbuf;
        }
        // parse buf1
        if ( strncmp( buf1, "Rs", 2 ) == 0 ) {
          // Stringrequest
          tstr = StringRequest( buf1, extorder, maxlen - dpos, entries );
          if(tstr!=NULL) {
            PARSECAT( tstr );
            _freesafe(tstr);
          } else {
            cancel=true;
            ende=true;
          }
        } else if ( strncmp( buf1, "scripts", 7 ) == 0 ) {
          // dir of the scripts
          str1 = Worker::getDataDir();
          str1 += "/scripts";
          PARSECAT( str1.c_str() );
        } else if ( strncmp( buf1, "pop", 3 ) == 0 ) {
          if ( strlen( buf1 ) > 3 ) {
            num = atoi( buf1 + 3 );
            if ( num < 0 ) num = 0;
          } else {
            num = 0;
          }
          if ( extorder->wpu != NULL ) {
            if ( extorder->wpu->size( num ) > 0 ) {
              str1 = extorder->wpu->pop( num );
              PARSECAT( str1.c_str() );
            }
          }
        } else if ( strncmp( buf1, "top", 3 ) == 0 ) {
          if ( strlen( buf1 ) > 3 ) {
            num = atoi( buf1 + 3 );
            if ( num < 0 ) num = 0;
          } else {
            num = 0;
          }
          if ( extorder->wpu != NULL ) {
            if ( extorder->wpu->size( num ) > 0 ) {
              str1 = extorder->wpu->top( num );
              PARSECAT( str1.c_str() );
            }
          }
        } else if ( strncmp( buf1, "size", 4 ) == 0 ) {
          if ( strlen( buf1 ) > 4 ) {
            num = atoi( buf1 + 4 );
            if ( num < 0 ) num = 0;
          } else {
            num = 0;
          }
          if ( extorder->wpu != NULL ) {
            sprintf( buf2, "%d", extorder->wpu->size( num ) );
            str1 = buf2;
            PARSECAT( str1.c_str() );
          }
        } else if ( strncmp( buf1, "p", 1 ) == 0 ) {
          // path
          ctstr = getCurrentDir();
          if ( ctstr != NULL ) {
            PARSECAT( ctstr );
          }
        } else if ( strncmp( buf1, "op", 2 ) == 0 ) {
          // other side
          if ( nm2 == NULL ) ctstr = NULL;
          else ctstr = nm2->getCurrentDir();
          if ( ctstr != NULL ) {
            PARSECAT( ctstr );
          }
        } else if ( strncmp( buf1, "lp", 2 ) == 0 ) {
          // left side
          if(myside==0) tnm=this;
          else tnm=nm2;
          if ( tnm == NULL ) ctstr = NULL;
          else ctstr = tnm->getCurrentDir();
          if ( ctstr != NULL ) {
            PARSECAT( ctstr );
          }
        } else if ( strncmp( buf1, "rp", 2 ) == 0 ) {
          // right side
          if(myside==1) tnm=this;
          else tnm=nm2;
          if ( tnm == NULL ) ctstr = NULL;
          else ctstr = tnm->getCurrentDir();
          if ( ctstr != NULL ) {
            PARSECAT( ctstr );
          }
        } else if ( strncmp( buf1, "dp", 2 ) == 0 ) {
          // destination path
          if ( extorder->destmode == NULL ) ctstr = NULL;
          else ctstr = extorder->destmode->getCurrentDir();
          if ( ctstr != NULL ) {
            PARSECAT( ctstr );
          }
        } else if ( strncmp( buf1, "v", 1 ) == 0 ) {
          // env variable
          tstr = getenv( buf1 + 1 );
          if ( tstr != NULL ) {
            PARSECAT( tstr );
          }
        } else {
          // file flag or unknown
          filemode=-1;
          otherside=false;
          nounselect=false;
          noext=false;
          tpos=0;
          tmode=0;
          for(;(tmode<4)&&(tpos<4);) {
            ch = buf1[ tpos++ ];
            if(ch=='o') {
              if(tmode<1) {
                otherside=true;
                tmode=1;
              } else {
                // an "o" at the wrong position
                tmode=6;
              }
            } else if(ch=='u') {
              if(tmode<2) {
                nounselect=true;
                tmode=2;
              } else {
                tmode=6;
              }
            } else if((ch=='f')&&(tmode<3)) {
              filemode=0;
              tmode=3;
            } else if((ch=='a')&&(tmode<3)) {
              filemode=1;
              tmode=3;
            } else if((ch=='F')&&(tmode<3)) {
              filemode=2;
              tmode=3;
            } else if((ch=='A')&&(tmode<3)) {
              filemode=3;
              tmode=3;
            } else if((ch=='E')&&(tmode==3)) {
              noext=true;
              tmode=4;
            } else if ( ( ch == 't' ) && ( tmode < 3 ) ) {
              filemode = 4;
              tmode = 4;  // no E allowed so directly change to mode 4
            } else if(ch=='\0') tmode=4;
          }
          if((filemode>=0)&&(filemode<5)) {
            // we have a valid flag
            takefirst=true;
            if((filemode==1)||(filemode==3)) takefirst=false;
            if(otherside==true) l=entries[1];
            else l=entries[0];
            id=l->initEnum();
            efe1=(NM_extern_fe*)l->getFirstElement(id);
            for(int c=0;;) {
              if(efe1==NULL) break;
              
              efeskip = false;
              if ( ( efe1->getDirFinished() != 0 ) &&
                   ( extorder->take_dirs == false ) ) {
                efeskip = true;
              } else {
                // this is a valid entry
		if ( filemode == 4 ) {
		  if ( extorder->wpu != NULL ) {
		    ctstr = extorder->wpu->getTempName4File( efe1->getFullname( false ) );
		  } else {
		    ctstr = NULL;
		  }
		} else if ( filemode < 2 ) ctstr = efe1->getName( noext );
                else ctstr=efe1->getFullname(noext);

                if ( ctstr != NULL ) {
                  if ( c > 0 ) PUTCHARDSTR( ' ' );
                  if ( dpos >= maxlen ) ende = true;
                  PARSECAT( ctstr );
                }               
                if(ende==true) break;
              }
              for (;;) {
                if(nounselect==false) {
                  // remove efe1;
                  l->removeElement( efe1 );
                  // now deactivate LVC if not NULL
                  if ( efe1->getFE() != NULL ) {
                    if(otherside==false) {
                      // in my list
		      deselect( efe1->getFE() );
                    } else {
		      if ( nm2 != NULL ) nm2->deselect( efe1->getFE() );
                    }
                  }
                  delete efe1;
                  efe1=(NM_extern_fe*)l->getFirstElement(id);
                } else {
                  efe1=(NM_extern_fe*)l->getNextElement(id);
                }
                if ( efe1 == NULL ) break;
                if ( ( efe1->getDirFinished() == 0 ) ||
                     ( extorder->take_dirs == true ) ) {
                  break;
                }
              }

              if ( efeskip == true ) continue;
              if((c==0)&&(takefirst==true)) break;
              c=1;
            }
            l->closeEnum(id);
          }
        }
        mode=0;
      }
    }
    dstr[dpos]=0;
  }
  _freesafe( flagbuf );
  if(cancel==true) {
    _freesafe(dstr);
    return NULL;
  }
  return dstr;
}

void NormalMode::deselect( FileEntry *fe, int row)
{
  if((lv->isValidRow(row)==false)||(fe==NULL)) return;
  if(ce==NULL) return;
  if(fe->select==false) return;
  if(lv->isValidRow(row)) {
    lv->setSelect(row,false);
    lv->showRow(row);
  }
  fe->select=false;
  ce->recalcStats();
  showCacheState();
}

char *NormalMode::StringRequest(char *sstr,class NM_externorder *extorder,int maxlen,NMExtList *entries[2])
{
  char *istr=NULL,ch;
  char *defstr=NULL;
  char *ristr=NULL,*rdefstr=NULL;
  int spos=0,bracketcount=0;
  int istart,iend,defstart,defend;
  int mode;
  char *buttonstr,*textstr,*return_str;
  int erg;
  
  spos=2; // buf1 starts with Rs
  istart=iend=defstart=defend=-1;
  mode=0;
  for(;mode<4;) {
    ch=sstr[spos++];
    if(ch=='\0') break;
    switch(mode) {
      case 0:
        // we are in no bracket
        if(ch=='{') {
          mode=1;
          bracketcount++;
          istart=spos;
        }
        break;
      case 1:
        // in the infotext
        if(ch=='{') bracketcount++;
        else if(ch=='}') {
          if(bracketcount==1) {
            iend=spos-2;
            mode=2;
          }
          bracketcount--;
        }
        break;
      case 2:
        // we are in no bracket
        if(ch=='{') {
          mode=3;
          bracketcount++;
          defstart=spos;
        }
        break;
      case 3:
        // in the defaulttext
        if(ch=='{') bracketcount++;
        else if(ch=='}') {
          if(bracketcount==1) {
            defend=spos-2;
            mode=4;
          }
          bracketcount--;
        }
        break;
      default:
        break;
    }
  }
  if((istart>=0)&&(iend>=istart)) {
    // we have an infotext
    istr=dupstring(&sstr[istart]);
    istr[iend-istart+1]=0;
    ristr=parseComStr(istr,extorder,maxlen,entries, false );
    _freesafe(istr);
  }
  if((defstart>=0)&&(defend>=istart)) {
    // we have a defaulttext
    defstr=dupstring(&sstr[defstart]);
    defstr[defend-defstart+1]=0;
    rdefstr=parseComStr(defstr,extorder,maxlen,entries, false );
    _freesafe(defstr);
  }
  
  textstr=NULL;
  if(ristr!=NULL) {
    if(strlen(ristr)>0) textstr=ristr;
    else _freesafe(ristr);
  }
  if(textstr==NULL) textstr=dupstring(catalog.getLocale(200));
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocale(123),textstr,
                          (rdefstr!=NULL)?rdefstr:"",
                          buttonstr,&return_str);
  _freesafe(buttonstr);
  _freesafe(textstr);
  if(rdefstr!=NULL) _freesafe(rdefstr);
  if(erg!=0) {
    _freesafe(return_str);
    return NULL;
  }
  return return_str;
}

void NormalMode::renamef(struct NM_renameorder *renorder)
{
  NM_specialsourceInt *ss1;
  FileEntry *fe;
  int erg;
  bool skip,cancel,nameok;
  char *newname,*buttonstr,*textstr,*newfullname,*return_str;
  const std::vector<WorkerTypes::listcol_t> *dis;
  std::list<NM_specialsourceInt*> *renlist;
  std::list<NM_specialsourceInt*>::iterator iti1;
#if 0
  int oldx,oldy;
#endif

  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to rename
    return;
  }

  finishsearchmode();
  
  dis = wconfig->getVisCols( parentlister->getSide() );
  
  renlist = new std::list<NM_specialsourceInt*>;
  switch(renorder->source) {
    case NM_renameorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( renorder->sources ),
                                   *renlist );
        break;
    case NM_renameorder::NM_ONLYACTIVE:
      getSelFiles( renlist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( renlist, NM_GETFILES_SELORACT, false );
      break;
  }
  
  cancel=skip=false;

  for ( iti1 = renlist->begin(); iti1 != renlist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    // now rename ss1->entry
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    fe = ss1->entry();
    if(fe!=NULL) {
      newname=dupstring(fe->name);
      newfullname=NULL;
      for(nameok=false;nameok==false;) {
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(fe->name)+1);
        sprintf(textstr,catalog.getLocale(148),fe->name);
        erg=req->string_request(catalog.getLocale(149),textstr,newname,buttonstr,&return_str, Requester::REQUEST_SELECTALL );
        _freesafe(buttonstr);
        _freesafe(textstr);
        _freesafe(newname);
        newname=return_str;
        if(erg==2) {
          cancel=true;
          break;
        } else if(erg==1) {
          break;
        } else {
          if(strcmp(newname,fe->name)==0) {
            // same name as orig
            // so skip this entry
            break;
          } else if ( strlen( newname ) < 1 ) {
            // empty string->repeat
            continue;
          }
          newfullname = (char*)_allocsafe( strlen( getCurrentDir() ) + 1 + strlen( newname ) + 1 );
          if ( strlen( getCurrentDir() ) > 1 ) sprintf( newfullname, "%s/%s", getCurrentDir(), newname );
          else sprintf( newfullname, "%s%s", getCurrentDir(), newname );
          if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(279),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) break;
            else if(erg==2) {
              cancel=true;
              break;
            }
            _freesafe(newfullname);
            newfullname=NULL;
          } else {
            nameok=true;
          }
        }
      }
      if((nameok==true)&&(newfullname!=NULL)) {
        erg = worker_rename( fe->fullname, newfullname );
        if(erg!=0) {
          textstr = dupstring( catalog.getLocale(151) );
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe( textstr );
          if(erg==1) cancel=true;
        } else {
          // deselect and change lvc temp.
          if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
	  //TODO: Man koennte eine helper-Funktion einfuehren, um
	  //      name/fullname zu setzen
          _freesafe(fe->name);
          _freesafe(fe->fullname);
          fe->name=newname;
          newname=NULL;
          if(fe->name[0]=='.') fe->isHidden=true;
          else fe->isHidden=false;          
          fe->fullname=newfullname;
          newfullname=NULL;
          if(lv->isValidRow(ss1->row)==true) {
            //TODO: Eigentlich muesste hier auch das DND neugesetzt werden, aber ich weiss nicht die Position
            // d.h. ich muesste erst suchen.
            // allerdings kann das auch so bleiben,da am Ende ohnehin buildLister gemacht wird
            fe->setLVC( lv, ss1->row, dis );
            lv->redraw();
            aguix->Flush();
          }
        }
      }
      if(newname!=NULL) _freesafe(newname);
      if(newfullname!=NULL) _freesafe(newfullname);
    }
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, false );
    }
  }
  
  for ( iti1 = renlist->begin(); iti1 != renlist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete renlist;
  // now update
  
  update(false);
  lv->showActive();
}

void NormalMode::dirsize(struct NM_dirsizeorder *dsorder)
{
  NM_specialsourceInt *ss1;
  bool enter;
  const std::vector<WorkerTypes::listcol_t> *dis;
  int oldx,oldy;
  std::list<NM_specialsourceInt*> *dslist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  
  finishsearchmode();
  
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();

  if(dsorder==NULL) return;
  if ( getCurrentDir() == NULL ) return;
  
  dis = wconfig->getVisCols( parentlister->getSide() );
  
  dslist = new std::list<NM_specialsourceInt*>;
  switch(dsorder->source) {
    case NM_dirsizeorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( dsorder->sources ),
                                   *dslist );
        break;
    case NM_dirsizeorder::NM_ONLYACTIVE:
      getSelFiles( dslist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( dslist, NM_GETFILES_SELORACT, false );
      break;
  }

  // create the NM_CopyOp_Dir for each dir in copylist
  
  parentlister->getWorker()->setWaitCursor();
  aguix->getLastKeyRelease();
  aguix->getLastMouseRelease();

  bool cancel = false;

  for ( iti1 = dslist->begin(); iti1 != dslist->end() && cancel == false; iti1++ ) {
    ss1 = *iti1;
    enter=false;
    if ( ss1->entry()->isDir() == true ) {
      if ( ss1->entry()->isLink == false ) enter = true;
    }
    if(enter==true) {
      if ( getDirSize( ss1->entry(), ss1->entry()->dirsize, cancel ) == 0 ) {
        if(lv->isValidRow(ss1->row)==true) {
          ss1->entry()->setLVC( lv, ss1->row, dis );
          lv->redraw();
          aguix->Flush();
        }
      } else {
        ss1->entry()->dirsize = -1;
      }
    }
  }

  for ( iti1 = dslist->begin(); iti1 != dslist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete dslist;

  // now update
  ce->forceUpdate();
  rebuildView();
  lv->setYOffset(oldy);
  lv->setXOffset(oldx);
  lv->redraw();

  parentlister->getWorker()->unsetWaitCursor();
}

int
NormalMode::getDirSize( FileEntry*rootfe, loff_t &return_size, bool &cancel )
{
  FileEntry *fe;
  bool enter;
  loff_t size,recsize;
  Verzeichnis *verz=new Verzeichnis();
  time_t now;
  
  if(verz->readDir(rootfe->fullname)!=0) {
    delete verz;
    return -1;
  }
  
  if ( verz->dirOpened() == false ) {
    delete verz;
    return -1;
  }

  now = time( NULL );
  if ( m_last_statebar_update != now ) {
      FileNameShrinker fs;
      AFontWidth fw( aguix, NULL );
      std::string shrinked_text;
      
      shrinked_text = fs.shrink( rootfe->fullname,
                                 parentlister->getWorker()->getStatebarWidth(),
                                 fw );

      parentlister->setStatebarText( shrinked_text.c_str() );
      m_last_statebar_update = now;
  }

  size=0;
  for ( Verzeichnis::verz_it fe_it1 = verz->begin();
        fe_it1 != verz->end() && cancel == false;
        fe_it1++ ) {
    fe = *fe_it1;
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        recsize = 0;
        if ( getDirSize( fe, recsize, cancel ) == 0 ) {
          if(recsize>0) size+=recsize;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        size += fe->size();
      }
    }
  }

  //TODO this is a rather ugly hack to handle redraws and key presses
  aguix->doXMsgs( NULL, false );
  if ( aguix->getLastKeyRelease() == XK_Escape ) {
      cancel = true;
  }

  delete verz;
  return_size = size;

  // return a negative value to indicate that the return_size is not valid
  return ( cancel == true ) ? -2 : 0;
}

void NormalMode::simdd()
{
  if ( getCurrentDir() == NULL ) {
    return;
  }
  if ( ce->getActiveFE() != NULL ) startAction( ce->getActiveFE() );
}

void NormalMode::searchentry( bool ignore_case, bool reverse_search, bool infix_search )
{
  setEntrySearchCaseSensitive( ( ignore_case == true ) ? false : true );

  m_filtered_search_infix_search = infix_search;

  if(searchmodeon==false) startsearchmode();
  else {
    shownextentry( NULL, reverse_search );
  }
}

void NormalMode::startsearchmode()
{
  if ( getCurrentDir() == NULL ) {
    return;
  }
  oldstr=dupstring(sg->getText());
  lastsearch=dupstring("");
  sg->setText("");
  sg->setForbidPosChange(true);
  sg->setStrongKeyCapture(false);
  sg->activate();
  searchmodeon=true;

  setEntrySearchString( "" );
}

void NormalMode::finishsearchmode()
{
  if(searchmodeon==false) return;
  sg->setForbidPosChange(false);
  sg->setStrongKeyCapture(true);
  sg->setText(oldstr);
  sg->deactivate();
  _freesafe(oldstr);
  if(lastsearch!=NULL) _freesafe(lastsearch);
  else debugmsg("oops2\n");
  lastsearch=NULL;
  searchmodeon=false;
}

bool NormalMode::shownextentry( const char *str, bool reverse_search, bool start_at_same )
{
  FileEntry *fe;
  bool found;
  int row;
  std::string old_string_filter, string_filter;
  
  found = false;

  if ( getCurrentDir() == NULL ) {
    return found;
  }

  if ( str == NULL )
      str = lastsearch;
  if ( str == NULL )
      return found;

  old_string_filter = getEntrySearchString();

  if ( m_filtered_search_infix_search == true ) {
      string_filter = "*";
  } else {
      string_filter = "";
  }

  string_filter += str;
  string_filter += "*";

  if ( string_filter == "**" ) {
      string_filter = "*";
  }

  /* first test whether an entry exists so don't use setEntrySearchString here
   * as it would also modify dirfilter */
  _entrysearch_matcher.setMatchString( string_filter );

  Verzeichnis::verz_it pos_it1 = ce->begin();
  
  if ( ce->getActiveFE() != NULL ) {
      for ( pos_it1 = ce->begin();
            pos_it1 != ce->end();
            pos_it1++ ) {
          if ( (*pos_it1) == ce->getActiveFE() )
              break;
      }
      if ( start_at_same == false ) {
          if ( reverse_search == false ) {
              if ( pos_it1 != ce->end() )
                  pos_it1++;
              if ( pos_it1 == ce->end() )
                  pos_it1 = ce->begin();
          } else {
              if ( pos_it1 != ce->begin() )
                   pos_it1--;
              else {
                  pos_it1 = ce->end();
                  pos_it1--;
              }
          }
      }
  }
  
  for ( int i = 0; i < 2 && found == false; i++ ) {
      while ( pos_it1 != ce->end() && found == false ) {
          fe = *pos_it1;
          if ( fe->use == true ) {
              if ( _entrysearch_matcher.match( fe->name ) == true ) {
                  // hit
                  ce->setActiveFE( fe );
                  row = 0;
                  while(lv->isValidRow(row)==true) {
                      if ( lv->getData( row ) == fe->getID() ) {
                          makeListViewRowActive( row );
                          break;
                      }
                      row++;
                  }
                  found = true;
              }
          }
          if ( reverse_search == true ) {
              if ( pos_it1 != ce->begin() )
                  pos_it1--;
              else
                  pos_it1 = ce->end();  // set to end to stop while loop
          } else {
              pos_it1++;
          }
      }
      if ( reverse_search == true ) {
          pos_it1 = ce->end();
          pos_it1--;
      } else {
          pos_it1 = ce->begin(); // if not found try from pos 0 again
      }
  }
  
  if ( found == true ) {
      setEntrySearchString( string_filter );
      //TODO do rebuildView only if filtered search is enabled
      lv->showActive();
      lv->redraw();
  } else {
      _entrysearch_matcher.setMatchString( old_string_filter );
  }
  
  return found;
}

void NormalMode::enterpath()
{
  sg->activate();
}

void NormalMode::scrolllister(int dir)
{
  lv->setXOffset( lv->getXOffset() + dir * lv->getHScrollStep() );
  aguix->Flush();
}

void NormalMode::setActiveDir2Other()
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;

  if(ce==NULL) return;
  if ( ce->getActiveFE() != NULL ) {
    if ( ce->getActiveFE()->isDir() == true ) {
      l1=parentlister->getWorker()->getOtherLister(parentlister);
      if(l1!=NULL) {
        l1->switch2Mode(0);
        lm1=l1->getActiveMode();
        if(lm1!=NULL) {
          if(lm1->isType("NormalMode")==true) {
            std::string s1( ce->getActiveFE()->fullname );

            nm1=(NormalMode*)lm1;
            nm1->enterDir( s1.c_str() );
          }
        }
      }
    }
  }
}

void NormalMode::createsymlink(struct NM_createsymlinkorder *cslorder)
{
  NM_specialsourceInt *ss1;
  FileEntry *fe;
  int erg;
  bool skip,cancel,nameok;
  char *newname,*buttonstr,*textstr,*newfullname,*return_str;
  char *sldest;
  Lister *olister=NULL;
  NormalMode *nm2=NULL,*updatenm=NULL;
  ListerMode *lm=NULL;
  std::list<NM_specialsourceInt*> *csllist;
  std::list<NM_specialsourceInt*>::iterator iti1;

  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to rename
    return;
  }

  if ( ce->dirOpened() == false )
      return;

  finishsearchmode();
  
  if ( strcmp( getCurrentDir(), cslorder->destdir ) == 0 ) {
    updatenm=this;
  }
  olister=parentlister->getWorker()->getOtherLister(parentlister);
  if(olister==NULL) return;
  lm=olister->getActiveMode();
  if(lm!=NULL) {
    if(lm->isType("NormalMode")==true) {
      nm2=(NormalMode*)lm;
      if ( nm2->getCurrentDir() != NULL ) {
        if ( strcmp( nm2->getCurrentDir(), cslorder->destdir ) == 0 ) {
          updatenm=nm2;
        }
      }
    }
  }

  csllist = new std::list<NM_specialsourceInt*>;
  switch(cslorder->source) {
    case NM_createsymlinkorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( cslorder->sources ),
                                   *csllist );
        break;
    case NM_createsymlinkorder::NM_ONLYACTIVE:
      getSelFiles( csllist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( csllist, NM_GETFILES_SELORACT, false );
      break;
  }
  
  cancel=skip=false;
  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    fe = ss1->entry();
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(fe!=NULL) {
      newname=dupstring(fe->name);
      newfullname=NULL;
      for(nameok=false;nameok==false;) {
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(207))+strlen(fe->name)+1);
        sprintf(textstr,catalog.getLocale(207),fe->name);
        erg=req->string_request(catalog.getLocaleCom(36),textstr,newname,buttonstr,&return_str);
        _freesafe(buttonstr);
        _freesafe(textstr);
        _freesafe(newname);
        newname=return_str;
        if(erg==2) {
          cancel=true;
          break;
        } else if(erg==1) {
          break;
        } else {
          newfullname=(char*)_allocsafe(strlen(cslorder->destdir)+1+strlen(newname)+1);
          if(strlen(cslorder->destdir)>1) sprintf(newfullname,"%s/%s",cslorder->destdir,newname);
          else sprintf(newfullname,"%s%s",cslorder->destdir,newname);
          if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(279),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) break;
            else if(erg==2) {
              cancel=true;
              break;
            }
            _freesafe(newfullname);
            newfullname=NULL;
          } else {
            nameok=true;
          }
        }
      }
      if((nameok==true)&&(newfullname!=NULL)) {
        // now create symlink newfullname which points to fe->fullname or fe->name
        if(cslorder->local==true) {
          sldest = Datei::getRelativePath( fe->fullname, cslorder->destdir );
          if ( sldest == NULL ) {
            //TODO: I currently can't imagine a situation but the call
            //      can return NULL (shouldn't happen because I give
            //      correct info) so in this case just use the fullname
            //      But give the user a requester?
            sldest = dupstring( fe->fullname );
          }
        } else sldest = dupstring( fe->fullname );
        erg = worker_symlink(sldest,newfullname);
        _freesafe( sldest );
        if(erg!=0) {
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(newfullname)+1);
          sprintf(textstr,catalog.getLocale(205),newfullname);
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) cancel=true;
        } else {
          // deselect
          if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
        }
      }
      if(newname!=NULL) _freesafe(newname);
      if(newfullname!=NULL) _freesafe(newfullname);
    }
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, false );
    }
  }
  
  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete csllist;
  // now update
  if(updatenm!=NULL) updatenm->update(false);
}

void NormalMode::changesymlink(struct NM_changesymlinkorder *chslorder)
{
  NM_specialsourceInt *ss1;
  FileEntry *fe;
  int erg;
  bool cancel;
  char *buttonstr,*textstr,*return_str;
  char buffer[4096];
  int tx;
  std::list<NM_specialsourceInt*> *csllist;
  std::list<NM_specialsourceInt*>::iterator iti1;

  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to rename
    return;
  }

  finishsearchmode();
  
  csllist = new std::list<NM_specialsourceInt*>;
  switch(chslorder->source) {
    case NM_changesymlinkorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( chslorder->sources ),
                                   *csllist );
        break;
    case NM_changesymlinkorder::NM_ONLYACTIVE:
      getSelFiles( csllist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( csllist, NM_GETFILES_SELORACT, false );
      break;
  }
  
  cancel=false;

  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    if ( cancel == true ) break;
    ss1 = *iti1;
    fe = ss1->entry();
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, true );
      lv->showRow( ss1->row );
    }
    if(fe!=NULL) {
      if(fe->isLink==true) {
        tx = worker_readlink( fe->fullname, buffer, 4095 );
        if(tx>=0) {
          buffer[tx]=0;
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                       catalog.getLocale(225),
                                       catalog.getLocale(8));
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(206))+strlen(fe->name)+1);
          sprintf(textstr,catalog.getLocale(206),fe->name);
          erg=req->string_request(catalog.getLocaleCom(37),textstr,buffer,buttonstr,&return_str);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==2) {
            // cancel
            cancel=true;
          } else if(erg==0) {
            // ok
            if(strcmp(buffer,return_str)!=0) {
              // only if new value differs
              if ( worker_unlink( fe->fullname ) == 0 ) {
                if ( worker_symlink( return_str, fe->fullname ) == 0 ) {
                  // deselect
                  if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
                } else {
                  textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(fe->fullname)+1);
                  sprintf(textstr,catalog.getLocale(205),fe->fullname);
                  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                              strlen(catalog.getLocale(8))+1);
                  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                            catalog.getLocale(8));
                  erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                  _freesafe(buttonstr);
                  _freesafe(textstr);
                  if(erg==1) cancel=true;
                }
              } else {
                textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(fe->fullname)+1);
                sprintf(textstr,catalog.getLocale(291),fe->fullname);
                buttonstr = dupstring( catalog.getLocale(11) );
                req->request(catalog.getLocale(347),textstr,buttonstr);
                _freesafe(textstr);
                _freesafe( buttonstr );
              }
            }
          }
          _freesafe(return_str);
        }
      }
    }
    if ( ss1->row >= 0 ) {
      lv->setVisMark( ss1->row, false );
    }
  }
  
  for ( iti1 = csllist->begin(); iti1 != csllist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete csllist;

  update( false );
}

void NormalMode::chmodf(struct NM_chmodorder *cmorder)
{
  NM_specialsourceInt *ss1;
  changemod_info_t *cminfo;
  std::list<NM_specialsourceInt*> *cmlist;
  std::list<NM_specialsourceInt*>::iterator iti1;
   
  if(cmorder==NULL) return;
  if ( getCurrentDir() == NULL ) {
    // no current dir => no files to copy
    return;
  }

  finishsearchmode();
  
  cmlist = new std::list<NM_specialsourceInt*>;
  switch(cmorder->source) {
    case NM_chmodorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( cmorder->sources ),
                                   *cmlist );
        break;
    case NM_chmodorder::NM_ONLYACTIVE:
      getSelFiles( cmlist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( cmlist, NM_GETFILES_SELORACT, false );
      break;
  }

  cminfo = new changemod_info_t;

  for ( iti1 = cmlist->begin(); iti1 != cmlist->end(); iti1++ ) {
    ss1 = *iti1;
    if ( worker_changemod( ss1, cminfo, cmorder ) != 0 ) {
      break;
    }
  }
  for ( iti1 = cmlist->begin(); iti1 != cmlist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete cminfo;
  delete cmlist;
  update(false);
}

int NormalMode::worker_changemod( const struct NM_specialsourceInt *ss1,
                                  changemod_info_t *cminfo,
                                  const NM_chmodorder *cmorder )
{
  bool enter, dontChange, skip, cancel, changemodFirst;
  int erg;
  changemod_info_t newmode;
  FileEntry *subfe;
  NM_specialsourceInt *ss2;
  NM_CopyOp_Dir *cod;
  
  if ( ( ss1 == NULL ) || ( cminfo == NULL ) || ( cmorder == NULL ) ) return -1;

  newmode.mode = 0700; //default
  newmode.op = changemod_info::CHMOD_SET;

  // do we have to enter this entry?
  enter = false;
  if ( ( ss1->entry()->isDir() == true ) && ( cmorder->recursive == true ) ) {
    if ( ss1->entry()->isLink == false ) enter = true;
  }
  
  // check operation applies to this entry
  dontChange = false;
  if ( ( ss1->entry()->isDir() == true ) && ( cmorder->ondirs == false ) ) dontChange = true;
  else if ( ( ss1->entry()->isDir() == false ) && ( cmorder->onfiles == false ) ) dontChange = true;

  /* skip means skip entry AND all sub entries
   * cancel is clear
   * dontchange means to normally handle this entry
   *   but actually don't change mod
   */
  skip = cancel = false;
  if ( dontChange == false ) {
    // ask for new mode
    if ( cminfo->forAll == true ) {
      newmode = *cminfo;
    } else {
      erg = requestNewMode( ss1->entry(), &newmode );
      if ( erg == 1 ) {
        *cminfo = newmode;
        cminfo->forAll = true;
      } else if ( erg == 2 ) {
        skip = true;
      } else if ( erg == 3 ) {
        cancel = true;
      }
    }
  }
  if ( skip == true ) return 0;
  if ( cancel == true ) return 1;

  if ( ss1->entry()->checkAccess( R_OK|W_OK|X_OK ) != true ) {
    // cannot access directory to change mod so apply
    // newmode first
    changemodFirst = true;
  } else {
    changemodFirst = false;
  }
  if ( ( dontChange == false ) && ( changemodFirst == true ) ) {
    if ( applyNewMode( ss1, newmode ) != 0 ) cancel = true;
  }
  if ( ( enter == true ) && ( cancel == false ) ) {
    cod = new NM_CopyOp_Dir( ss1->entry() );
    if ( cod->user_abort == true ) {
      cancel = true;
    } else if ( cod->ok == true ) {
      for ( Verzeichnis::verz_it subfe_it1 = cod->verz->begin();
            subfe_it1 != cod->verz->end() && cancel == false;
            subfe_it1++ ) {
        subfe = *subfe_it1;
        if ( strcmp( subfe->name, ".." ) != 0 ) {
	  ss2 = new NM_specialsourceInt( subfe );
          ss2->row = -1;
          if ( worker_changemod( ss2, cminfo, cmorder ) != 0 ) {
            cancel = true;
          }
          delete ss2;
        }
      }
    }
    delete cod;
  }
  if ( ( dontChange == false ) && ( changemodFirst == false ) ) {
    if ( applyNewMode( ss1, newmode ) != 0 ) cancel = true;
  }
  return ( cancel == true ) ? 1 : 0;
}

int NormalMode::applyNewMode( const NM_specialsourceInt *ss1, changemod_info_t newmode )
{
  char *textstr, *buttonstr;
  int erg;
  bool cancel = false;
  
  if ( ss1 == NULL ) return -1;

  mode_t apply_mode = ( ss1->entry()->isLink == true &&
                        ss1->entry()->isCorrupt == false ) ? ss1->entry()->dmode() : ss1->entry()->mode();
  if ( newmode.op == changemod_info::CHMOD_SET ) {
      apply_mode = newmode.mode;
  } else if ( newmode.op == changemod_info::CHMOD_ADD ) {
      apply_mode |= newmode.mode;
  } else if ( newmode.op == changemod_info::CHMOD_REM ) {
      apply_mode &= ~newmode.mode;
  } else {
      return -1;
  }

  if ( worker_chmod( ss1->entry()->fullname, apply_mode ) != 0 ) {
    // error
    buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                   strlen( catalog.getLocale( 8 ) ) + 1 );
    sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                                 catalog.getLocale( 8 ) );
    textstr = (char*)_allocsafe( strlen( catalog.getLocale( 226 ) ) + strlen( ss1->entry()->fullname ) + 1 );
    sprintf( textstr, catalog.getLocale( 226 ), ss1->entry()->fullname );
    erg = req->request( catalog.getLocale( 347 ), textstr, buttonstr );
    _freesafe( buttonstr );
    _freesafe( textstr );
    if ( erg == 1 ) cancel = true;
  } else {
    if ( lv->isValidRow( ss1->row ) == true ) deselect( ss1->entry(), ss1->row );
  }
  return ( cancel == true ) ? 1 : 0;
}

static void setChooseButtonMatrixForMode( ChooseButton **mcb, mode_t mode )
{
    mcb[0]->setState( ( ( mode & S_IRUSR ) == 0 ) ? false : true );
    mcb[1]->setState( ( ( mode & S_IWUSR ) == 0 ) ? false : true );
    mcb[2]->setState( ( ( mode & S_IXUSR ) == 0 ) ? false : true );
    mcb[3]->setState( ( ( mode & S_IRGRP ) == 0 ) ? false : true );
    mcb[4]->setState( ( ( mode & S_IWGRP ) == 0 ) ? false : true );
    mcb[5]->setState( ( ( mode & S_IXGRP ) == 0 ) ? false : true );
    mcb[6]->setState( ( ( mode & S_IROTH ) == 0 ) ? false : true );
    mcb[7]->setState( ( ( mode & S_IWOTH ) == 0 ) ? false : true );
    mcb[8]->setState( ( ( mode & S_IXOTH ) == 0 ) ? false : true );
    mcb[9]->setState( ( ( mode & S_ISUID ) == 0 ) ? false : true );
    mcb[10]->setState( ( ( mode & S_ISGID ) == 0 ) ? false : true );
    mcb[11]->setState( ( ( mode & S_ISVTX ) == 0 ) ? false : true );
}

int NormalMode::requestNewMode( FileEntry *fe, changemod_info_t *return_mode )
{
    mode_t tmode, newmode;
    AWindow *win;
    ChooseButton *mcb[12];
    AGMessage *msg;
    int endmode = -1;
    char *tstr;
    
    tmode = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dmode() : fe->mode();
    win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocaleCom( 38 ) );
    win->create();
    
    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 4 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );
    
    tstr=(char*)_allocsafe(strlen(catalog.getLocale(213))+strlen(fe->fullname)+1);
    sprintf(tstr,catalog.getLocale(213),fe->fullname);
    
    ac1->add( new Text( aguix, 0, 0, tstr, 1 ), 0, 0, AContainer::CINCWNR );
    _freesafe(tstr);
    
    AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( 5 );

    ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 780 ), 1 ), 0, 0, AContainer::CFIX );
    CycleButton *apply_mode_cb = static_cast<CycleButton*>( ac1_1->add( new CycleButton( aguix,
                                                                                         0, 0,
                                                                                         10,
                                                                                         1, 0, 0 ),
                                                                        1, 0, AContainer::CFIX ) );
    apply_mode_cb->addOption( catalog.getLocale( 781 ) );
    apply_mode_cb->addOption( catalog.getLocale( 782 ) );
    apply_mode_cb->addOption( catalog.getLocale( 783 ) );
    apply_mode_cb->setOption( 0 );
    apply_mode_cb->resize( apply_mode_cb->getMaxSize(), apply_mode_cb->getHeight() );
    ac1_1->readLimits();
    
    AContainer *ac1_2 = ac1->add( new AContainer( win, 4, 6 ), 0, 2 );
    ac1_2->setMinSpace( 5 );
    ac1_2->setMaxSpace( 5 );
    
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 218 ), 1 ), 1, 0, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 219 ), 1 ), 2, 0, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 220 ), 1 ), 3, 0, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 214 ), 1 ), 0, 1, AContainer::CFIX );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 215 ), 1 ), 0, 2, AContainer::CFIX );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 216 ), 1 ), 0, 3, AContainer::CFIX );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 217 ), 1 ), 0, 5, AContainer::CFIX );
    
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 221 ), 1 ), 1, 4, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 222 ), 1 ), 2, 4, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    ac1_2->add( new Text( aguix, 0, 0, catalog.getLocale( 223 ), 1 ), 3, 4, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    
    mcb[0] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IRUSR ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 1, 1, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[1] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IWUSR ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 2, 1, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[2] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IXUSR ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 3, 1, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[3] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IRGRP ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 1, 2, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[4] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IWGRP ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 2, 2, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[5] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IXGRP ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 3, 2, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[6] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IROTH ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 1, 3, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[7] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IWOTH ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 2, 3, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[8] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_IXOTH ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 3, 3, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[9] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                          ( ( tmode & S_ISUID ) == 0 ) ? 0 : 1,
                                                          "", LABEL_RIGHT, 0, 0 ), 1, 5, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[10] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                           ( ( tmode & S_ISGID ) == 0 ) ? 0 : 1,
                                                           "", LABEL_RIGHT, 0, 0 ), 2, 5, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    mcb[11] = (ChooseButton*)ac1_2->add( new ChooseButton( aguix, 0, 0,
                                                           ( ( tmode & S_ISVTX ) == 0 ) ? 0 : 1,
                                                           "", LABEL_RIGHT, 0, 0 ), 3, 5, AContainer::CFIXNR + AContainer::ACONT_CENTER );
    setChooseButtonMatrixForMode( mcb, tmode );
    
    AContainer *ac1_3 = ac1->add( new AContainer( win, 4, 1 ), 0, 3 );
    ac1_3->setMinSpace( 5 );
    ac1_3->setMaxSpace( -1 );
    ac1_3->setBorderWidth( 0 );
    Button *okb =(Button*)ac1_3->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 11 ),
                                                  1,
                                                  0,
                                                  0 ), 0, 0, AContainer::CFIX );
    Button *ok2allb = (Button*)ac1_3->add( new Button( aguix,
                                                       0,
                                                       0,
                                                       catalog.getLocale( 224 ),
                                                       1,
                                                       0,
                                                       0 ), 1, 0, AContainer::CFIX );
    Button *skipb = (Button*)ac1_3->add( new Button( aguix,
                                                     0,
                                                     0,
                                                     catalog.getLocale( 225 ),
                                                     1,
                                                     0,
                                                     0 ), 2, 0, AContainer::CFIX );
    Button *cb = (Button*)ac1_3->add( new Button( aguix,
                                                  0,
                                                  0,
                                                  catalog.getLocale( 8 ),
                                                  1,
                                                  0,
                                                  0 ), 3, 0, AContainer::CFIX );
    win->setDoTabCycling( true );
    win->contMaximize( true, true );
    win->show();
  
    for( ; endmode == -1; ) {
        msg = aguix->WaitMessage( win );
        if ( msg != NULL ) {
            switch ( msg->type ) {
              case AG_CLOSEWINDOW:
                  if ( msg->closewindow.window == win->getWindow() ) endmode = 3;
                  break;
              case AG_BUTTONCLICKED:
                  if ( msg->button.button == okb ) endmode = 0;
                  else if ( msg->button.button == ok2allb ) endmode = 1;
                  else if ( msg->button.button == skipb ) endmode = 2;
                  else if ( msg->button.button == cb ) endmode = 3;
                  break;
              case AG_KEYPRESSED:
                  if ( win->isParent( msg->key.window, false ) == true ) {
                      switch ( msg->key.key ) {
                        case XK_1:
                        case XK_KP_1:
                        case XK_KP_End:
                            mcb[6]->setState( ( mcb[6]->getState() == true ) ? false : true );
                            break;
                        case XK_2:
                        case XK_KP_2:
                        case XK_KP_Down:
                            mcb[7]->setState( ( mcb[7]->getState() == true ) ? false : true );
                            break;
                        case XK_3:
                        case XK_KP_3:
                        case XK_KP_Next:
                            mcb[8]->setState( ( mcb[8]->getState() == true ) ? false : true );
                            break;
                        case XK_4:
                        case XK_KP_4:
                        case XK_KP_Left:
                            mcb[3]->setState( ( mcb[3]->getState() == true ) ? false : true );
                            break;
                        case XK_5:
                        case XK_KP_5:
                        case XK_KP_Begin:
                            mcb[4]->setState( ( mcb[4]->getState() == true ) ? false : true );
                            break;
                        case XK_6:
                        case XK_KP_6:
                        case XK_KP_Right:
                            mcb[5]->setState( ( mcb[5]->getState() == true ) ? false : true );
                            break;
                        case XK_7:
                        case XK_KP_7:
                        case XK_KP_Home:
                            mcb[0]->setState( ( mcb[0]->getState() == true ) ? false : true );
                            break;
                        case XK_8:
                        case XK_KP_8:
                        case XK_KP_Up:
                            mcb[1]->setState( ( mcb[1]->getState() == true ) ? false : true );
                            break;
                        case XK_9:
                        case XK_KP_9:
                        case XK_KP_Prior:
                            mcb[2]->setState( ( mcb[2]->getState() == true ) ? false : true );
                            break;
                        case XK_Return:
                        case XK_KP_Enter:
                            if ( ok2allb->getHasFocus() == false &&
                                 skipb->getHasFocus() == false &&
                                 cb->getHasFocus() == false ) {
                                endmode = 0;
                            }
                            break;
                        case XK_F1:
                            endmode = 0;
                            break;
                        case XK_Escape:
                        case XK_F4:
                            endmode = 3;
                            break;
                        case XK_F2:
                            endmode = 1;
                            break;
                        case XK_F3:
                            endmode = 2;
                            break;
                      }
                  }
                  break;
              case AG_CYCLEBUTTONCLICKED:
                  if ( msg->cyclebutton.cyclebutton == apply_mode_cb ) {
                      switch ( msg->cyclebutton.option ) {
                        case 2:
                        case 1:
                            setChooseButtonMatrixForMode( mcb, 0 );
                            break;
                        default:
                            setChooseButtonMatrixForMode( mcb, tmode );
                            break;
                      }
                  }
                  break;
            }
            aguix->ReplyMessage( msg );
        }
    }
    
    if ( endmode == 0 || endmode == 1 ) {
        // ok
        newmode = 0;
        newmode |= ( mcb[0]->getState() == true ) ? S_IRUSR : 0;
        newmode |= ( mcb[1]->getState() == true ) ? S_IWUSR : 0;
        newmode |= ( mcb[2]->getState() == true ) ? S_IXUSR : 0;
        newmode |= ( mcb[3]->getState() == true ) ? S_IRGRP : 0;
        newmode |= ( mcb[4]->getState() == true ) ? S_IWGRP : 0;
        newmode |= ( mcb[5]->getState() == true ) ? S_IXGRP : 0;
        newmode |= ( mcb[6]->getState() == true ) ? S_IROTH : 0;
        newmode |= ( mcb[7]->getState() == true ) ? S_IWOTH : 0;
        newmode |= ( mcb[8]->getState() == true ) ? S_IXOTH : 0;
        newmode |= ( mcb[9]->getState() == true ) ? S_ISUID : 0;
        newmode |= ( mcb[10]->getState() == true ) ? S_ISGID : 0;
        newmode |= ( mcb[11]->getState() == true ) ? S_ISVTX : 0;
        return_mode->mode = newmode;

        switch ( apply_mode_cb->getSelectedOption() ) {
          case 2:
              return_mode->op = changemod_info::CHMOD_REM;
              break;
          case 1:
              return_mode->op = changemod_info::CHMOD_ADD;
              break;
          default:
              return_mode->op = changemod_info::CHMOD_SET;
              break;
        }
    }
    
    delete win;
    
    return endmode;
}

int NormalMode::getSelFiles( std::list<NM_specialsourceExt*> *list, nm_getfiles_t selmode, bool unselect )
{
  int res;
  std::list<NM_specialsourceInt*> *tlist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  NM_specialsourceInt *ss1;
  NM_specialsourceExt *sse1;

  if ( list == NULL ) return -1;

  tlist = new std::list<NM_specialsourceInt*>;
  res = getSelFiles( tlist, selmode, unselect );
  if ( res < 0 ) {
    delete tlist;
    return -1;
  }

  for ( iti1 = tlist->begin(); iti1 != tlist->end(); iti1++ ) {
    ss1 = *iti1;
    sse1 = new NM_specialsourceExt( ss1->entry() );
    
    list->push_back( sse1 );

    delete ss1;
  }
  delete tlist;
  return res;
}

void NormalMode::setSortmode(int nmode)
{
  if ( ISVALID_SORTMODE( nmode ) ) {
    nmode&=0xff|SORT_REVERSE|SORT_DIRLAST|SORT_DIRMIXED;
    sortmode=nmode;

    _dir_sort_sets.setSortMode( sortmode );
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
      if ( ce->getActiveFE() != NULL )
        lv->centerActive();
    setName();
  }
}

void NormalMode::setFilters( const std::list<NM_Filter> &filters )
{
    _dir_filter_sets.setFilters( filters );
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
        if ( ce->getActiveFE() != NULL )
            lv->centerActive();
    setName();
}

void NormalMode::configureFilters( bool *tshowhidden, std::list<NM_Filter> &filters,
                                   DirFilterSettings &dir_settings )
{
  int trow;
  int pos;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;
  std::list<NM_Filter> usetf;
  std::list<NM_Filter>::iterator fil_it1;

  usetf = filters;

  AWindow *win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 159 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 6 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ChooseButton *chb=(ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0,
							       (*tshowhidden==true)?1:0,
							       catalog.getLocale(357),LABEL_RIGHT,1,0), 0, 0, cincwnr );

  FieldListView *filv = (FieldListView*)ac1->add( new FieldListView( aguix,
                                                                     0,
                                                                     0,
                                                                     100,
                                                                     10 * aguix->getCharHeight(),
                                                                     0 ), 0, 1, cmin );
  filv->setHBarState(2);
  filv->setVBarState(2);
  filv->setNrOfFields( 3 );
  filv->setFieldWidth( 1, 1 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 3, 1 ), 0, 2 );
  ac1_1->setMinSpace( 0 );
  ac1_1->setMaxSpace( 0 );
  ac1_1->setBorderWidth( 0 );

  Button *newb = (Button*)ac1_1->add( new Button( aguix, 0, 0,
						  catalog.getLocale( 167 ), 1, 0, 0 ), 0, 0, cincw );
  Button *delb = (Button*)ac1_1->add( new Button( aguix, 0, 0,
						  catalog.getLocale( 169 ), 1, 0, 0 ), 1, 0, cincw );
  Button *editb = (Button*)ac1_1->add( new Button( aguix, 0, 0,
						   catalog.getLocale( 168 ), 1, 0, 0 ), 2, 0, cincw );

  Button *uallb = (Button*)ac1->add( new Button( aguix, 0, 0,
						 catalog.getLocale( 194 ), 1, 0, 0 ), 0, 3, cincw );

  AContainerBB *ac1_3 = (AContainerBB*)ac1->add( new AContainerBB( win, 1, 4 ), 0, 4 );
  ac1_3->setMinSpace( 5 );
  ac1_3->setMaxSpace( -1 );
  ac1_3->setBorderWidth( 5 );

  ac1_3->add( new Text( aguix, 0, 0, catalog.getLocale( 845 ), 1 ), 0, 0, AContainer::CINCWNR );

  AContainer *ac1_3_1 = ac1_3->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_3_1->setMinSpace( 5 );
  ac1_3_1->setMaxSpace( 5 );
  ac1_3_1->setBorderWidth( 0 );

  ac1_3_1->add( new Text( aguix, 0, 0, catalog.getLocale( 840 ), 1 ), 0, 0, AContainer::CFIX );
  CycleButton *bookmark_filter_cycb = (CycleButton*)ac1_3_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, AContainer::CINCW );
  bookmark_filter_cycb->addOption( catalog.getLocale( 841 ) );
  bookmark_filter_cycb->addOption( catalog.getLocale( 842 ) );
  bookmark_filter_cycb->addOption( catalog.getLocale( 843 ) );
  switch ( dir_settings.getBookmarkFilter() ) {
      case DirFilterSettings::SHOW_ONLY_BOOKMARKS:
          bookmark_filter_cycb->setOption( 1 );
          break;
      case DirFilterSettings::SHOW_ONLY_LABEL:
          bookmark_filter_cycb->setOption( 2 );
          break;
      case DirFilterSettings::SHOW_ALL:
      default:
          bookmark_filter_cycb->setOption( 0 );
          break;
  }
  bookmark_filter_cycb->resize( bookmark_filter_cycb->getMaxSize(),
                                bookmark_filter_cycb->getHeight() );
  ac1_3_1->readLimits();

  AContainer *ac1_3_2 = ac1_3->add( new AContainer( win, 3, 2 ), 0, 2 );
  ac1_3_2->setMinSpace( 0 );
  ac1_3_2->setMaxSpace( 0 );
  ac1_3_2->setBorderWidth( 0 );

  ac1_3_2->setMinWidth( 5, 1, 0 );
  ac1_3_2->setMaxWidth( 5, 1, 0 );

  ac1_3_2->add( new Text( aguix, 0, 0, catalog.getLocale( 844 ), 1 ), 0, 1, AContainer::CFIX );

  FieldListView *label_lv = (FieldListView*)ac1_3_2->add( new FieldListView( aguix,
                                                                             0, 0,
                                                                             40, 40, 0 ),
                                                          2, 0, AContainer::CMIN );
  label_lv->setHBarState( 2 );
  label_lv->setVBarState( 2 );

  std::list<std::string> cats = Worker::getBookmarkDBInstance().getCats();
  const std::map<std::string, WConfig::ColorDef::label_colors_t> labels = wconfig->getColorDefs().getLabelColors();
  std::map<std::string, WConfig::ColorDef::label_colors_t>::const_iterator label_it;
  
  for ( label_it = labels.begin(); label_it != labels.end(); ++label_it ) {
      if ( std::find( cats.begin(), cats.end(),
                      label_it->first ) == cats.end() ) {
          cats.push_back( label_it->first );
      }
  }
  if ( dir_settings.getSpecificBookmarkLabel().length() > 0 &&
       std::find( cats.begin(), cats.end(),
                  dir_settings.getSpecificBookmarkLabel() ) == cats.end() ) {
      cats.push_back( dir_settings.getSpecificBookmarkLabel() );
  }
  for ( std::list<std::string>::iterator it1 = cats.begin();
        it1 != cats.end();
        ++it1 ) {
      int label_row = label_lv->addRow();
      label_lv->setText( label_row, 0, *it1 );
      label_lv->setPreColors( label_row, FieldListView::PRECOLOR_ONLYACTIVE );
  }
  label_lv->resize( label_lv->getWidth(), 6 * aguix->getCharHeight() );
  ac1_3_2->readLimits();

  StringGadget *bookmark_label_sg = (StringGadget*)ac1_3_2->add( new StringGadget( aguix, 0, 0, 100,
                                                                                   dir_settings.getSpecificBookmarkLabel().c_str(), 0 ), 2, 1, AContainer::CINCW );

  ChooseButton *highlight_prefix_cb = (ChooseButton*)ac1_3->add( new ChooseButton( aguix, 0, 0,
                                                                                   dir_settings.getHighlightBookmarkPrefix(),
                                                                                   catalog.getLocale( 846 ), LABEL_LEFT,
                                                                                   1, 0 ), 0, 3, AContainer::CINCWNR );

  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 5 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );

  pos = 0;
  for ( fil_it1 = usetf.begin();
        fil_it1 != usetf.end();
        fil_it1++ ) {
      trow = filv->addRow();
      setLVC4Filter( filv, trow, *fil_it1 );
      filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
  }
  filv->redraw();

  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;

      } else if ( msg->type == AG_STRINGGADGET_CONTENTCHANGE ) {
          if ( msg->stringgadget.sg == bookmark_label_sg ) {
              bookmark_filter_cycb->setOption( 2 );
          }
      } else if ( msg->type == AG_FIELDLV_ONESELECT ||
                  msg->type == AG_FIELDLV_MULTISELECT ) {
          if ( msg->fieldlv.lv == label_lv ) {
              int label_row = label_lv->getActiveRow();
              if ( label_lv->isValidRow( label_row ) == true ) {
                  bookmark_label_sg->setText( label_lv->getText( label_row, 0 ).c_str() );
              }
              bookmark_filter_cycb->setOption( 2 );
          }
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
        else if(msg->button.button==newb) {
            NM_Filter tfi;
          
            if ( configureFilter( tfi ) == 0 ) {
                trow = filv->addRow();
                setLVC4Filter( filv, trow, tfi );
                filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
                filv->setActiveRow( trow );
                filv->showActive();
                usetf.push_back( tfi );
                filv->redraw();
            }
        } else if(msg->button.button==delb) {
          pos = 0;
          while ( filv->isValidRow( pos ) == true ) {
            if ( filv->getSelect( pos ) == true ) {
                int poscopy = pos;
                fil_it1 = usetf.begin();
                while ( fil_it1 != usetf.end() && poscopy > 0 ) {
                    fil_it1++;
                    poscopy--;
                }
                
                if ( fil_it1 != usetf.end() ) {
                    usetf.erase( fil_it1 );
                    filv->deleteRow( pos );
                    filv->redraw();
                    pos--;
                }
            }
            pos++;
          }
        } else if(msg->button.button==editb) {
          pos = 0;
          while ( filv->isValidRow( pos ) == true ) {
            if ( filv->getSelect( pos ) == true ) {
                int poscopy = pos;
                fil_it1 = usetf.begin();
                while ( fil_it1 != usetf.end() && poscopy > 0 ) {
                    fil_it1++;
                    poscopy--;
                }

                if ( fil_it1 != usetf.end() ) {
                    if ( configureFilter( *fil_it1 ) == 0 ) {
                        setLVC4Filter( filv, pos, *fil_it1 );
                        filv->redraw();
                    }
                }
            }
            pos++;
          }
        } else if(msg->button.button==uallb) {
          pos = 0;
          for ( fil_it1 = usetf.begin();
                fil_it1 != usetf.end();
                fil_it1++ ) {
              fil_it1->setCheck( NM_Filter::INACTIVE );
              if ( filv->isValidRow( pos ) == true ) {
                  setLVC4Filter( filv, pos, *fil_it1 );
                  pos++;
              }
          }
          filv->redraw();
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
      // insert usetf in tfilters
      filters = usetf;
      *tshowhidden = chb->getState();

      dir_settings.setHighlightBookmarkPrefix( highlight_prefix_cb->getState() );
      dir_settings.setSpecificBookmarkLabel( bookmark_label_sg->getText() );

      switch ( bookmark_filter_cycb->getSelectedOption() ) {
          case 1:
              dir_settings.setBookmarkFilter( DirFilterSettings::SHOW_ONLY_BOOKMARKS );
              break;
          case 2:
              dir_settings.setBookmarkFilter( DirFilterSettings::SHOW_ONLY_LABEL );
              break;
          case 0:
          default:
              dir_settings.setBookmarkFilter( DirFilterSettings::SHOW_ALL );
              break;
      }
  }
  
  delete win;
}

void NormalMode::setLVC4Filter( FieldListView *filv, int row, NM_Filter &fi )
{
  if ( filv == NULL ) return;
  const char *p = fi.getPattern();
  
  filv->setText( row, 0, ( p != NULL ) ? p : "" );
  if ( fi.getCheck() == 1 ) filv->setText( row, 2, catalog.getLocale( 170 ) );
  else if ( fi.getCheck() == 2 ) filv->setText( row, 2, catalog.getLocale( 171 ) );
  else filv->setText( row, 2, "" );
}

int NormalMode::configureFilter( NM_Filter &fi )
{
  const char *tstr;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;

  AWindow *win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 172 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );
  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 93 ), 1 ), 0, 0, cfix );

  tstr = fi.getPattern();
  StringGadget *tsg = (StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 100,
								   ( tstr != NULL ) ? tstr : "", 0 ), 1, 0, cincw );

  CycleButton *cyb = (CycleButton*)ac1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 0, 1, cincw );
  cyb->addOption(catalog.getLocale(358));
  cyb->addOption(catalog.getLocale(170));
  cyb->addOption(catalog.getLocale(171));
  if ( fi.getCheck() == 1 ) cyb->setOption(1);
  else if ( fi.getCheck() == 2 ) cyb->setOption(2);
  else cyb->setOption(0);
  cyb->resize(cyb->getMaxSize(),cyb->getHeight());
  ac1->readLimits();

  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 2 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );
  
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    switch(cyb->getSelectedOption()) {
      case 1:
        fi.setCheck( NM_Filter::INCLUDE );
        break;
      case 2:
        fi.setCheck( NM_Filter::EXCLUDE );
        break;
      default:
        fi.setCheck( NM_Filter::INACTIVE );
        break;
    }
    fi.setPattern( tsg->getText() );
  }
  
  delete win;
  return (ende==1)?0:1;
}

/*
 * NormalMode::rebuildView
 *
 * rebuild listview and update statebar infos
 */
void NormalMode::rebuildView()
{
  if ( lv != NULL ) {
      int oldx = lv->getXOffset();
      int oldy = lv->getYOffset();
      buildListView();
      lv->setYOffset( oldy );
      lv->setXOffset( oldx );
      showCacheState();
      setName();
      updateTabs();
  }
}

void NormalMode::setName()
{
  bool filteractive;
  std::string tstr;
  bool doUpdate;
  
  filteractive = _dir_filter_sets.filterActive();
  
  switch ( busyFlag ) {
    case 1:
      tstr += '|';
      break;
    case 2:
      tstr += '/';
      break;
    case 3:
      tstr += '-';
      break;
    case 4:
      tstr += '\\';
      break;
    default:
      tstr += ' ';
      break;
  }

  tstr += ' ';

  switch(sortmode&0xff) {
    case SORT_NAME:
      tstr += 'N';
      break;
    case SORT_SIZE:
      tstr += 'S';
      break;
    case SORT_CHGTIME:
      tstr += 'C';
      break;
    case SORT_MODTIME:
      tstr += 'M';
      break;
    case SORT_ACCTIME:
      tstr += 'A';
      break;
    case SORT_TYPE:
      tstr += 'T';
      break;
    case SORT_OWNER:
      tstr += 'O';
      break;
    case SORT_INODE:
      tstr += 'I';
      break;
    case SORT_NLINK:
      tstr += 'L';
      break;
    default:
      tstr += '?';
      break;
  }

  if ( ( sortmode & SORT_REVERSE ) == SORT_REVERSE ) tstr += 'R';
  else tstr += ' ';

  if ( _dir_filter_sets.getShowHidden() == false ) tstr += 'H';
  else tstr += ' ';

  if ( filteredSearchActive() == true ) {
      tstr += '+';
  } else if ( filteractive == true ) {
      tstr += '*';
  } else {
      tstr += ' ';
  }

  if ( _dir_filter_sets.getBookmarkFilter() != DirFilterSettings::SHOW_ALL ) {
      tstr += 'B';
  } else {
      tstr += ' ';
  }

  tstr += "   ";
  tstr += catalog.getLocale( 173 );
  
  if ( tstr != namestr ) doUpdate = true;
  else doUpdate = false;

  namestr = tstr;

  if ( doUpdate == true ) updateName();
}

/*
 * update lvb name with current name and freespace str
 */
void NormalMode::updateName()
{
  char *tstr;
  const char *t1, *t2;
  
  t1 = namestr.c_str();
  t2 = ( freespacestr != NULL ) ? freespacestr : "";
  tstr = catstring( t1, t2 );
  //TODO: Vielleicht mit aktuellen String vergleichen, um redraw
  //      zu vermeiden?
  parentlister->setName( tstr );
  _freesafe( tstr );
}

void NormalMode::reconfig()
{
  int side = parentlister->getSide();
  AGUIXFont *afont = aguix->getFont( wconfig->getFont( 2 + side ) );
  int sgh = ( ( afont != NULL ) ? afont->getCharHeight() : aguix->getCharHeight() ) +8;
  int hbw;

  lv->setMBG(wconfig->getListerBG());
  lv->setHBarState( ( wconfig->getHBarTop( side ) == true ) ? 1 : 2 );
  lv->setVBarState( ( wconfig->getVBarLeft( side ) == true ) ? 1 : 2 );
  lv->setHBarHeight( wconfig->getHBarHeight( side ) );
  lv->setVBarWidth( wconfig->getVBarWidth( side ) );
  lv->setShowHeader( wconfig->getShowHeader( side ) );
  lv->setFont( wconfig->getFont( 2 + side ) );
  sg->setFont( wconfig->getFont( 2 + side ) );
  hb[0]->setFont( wconfig->getFont( 2 + side ) );
  hb[1]->setFont( wconfig->getFont( 2 + side ) );
  parentb->setFont( wconfig->getFont( 2 + side ) );
  lv->setHeaderFG( wconfig->getLVHeader( 0 ) );
  lv->setHeaderBG( wconfig->getLVHeader( 1 ) );
  
  sg->resize( sg->getWidth(), sgh );
  hbw = aguix->getTextWidth( "<", afont ) + 10;
  hb[0]->resize( hbw, sgh );
  hb[1]->resize( hbw, sgh );
  parentb->resize( aguix->getTextWidth( "..", afont ) + 10, sgh );

  m_tab_b->setFont( wconfig->getFont( 2 + side ) );
  m_tab_new->setFont( wconfig->getFont( 2 + side ) );
  m_tab_new->resize( aguix->getTextWidth( "N", afont ) + 10, m_tab_new->getHeight() );
  m_tab_close->setFont( wconfig->getFont( 2 + side ) );
  m_tab_close->resize( aguix->getTextWidth( "X", afont ) + 10, m_tab_new->getHeight() );
  m_tab_cont->readLimits();

  m_cont->readLimits();
  m_cont2->readLimits();

  parentawindow->updateCont();
  lv->centerActive();
}

void NormalMode::showFreeSpace(bool force)
{
  int erg;
  if ( force == true ) erg = updateFreeSpaceStr( 0 );
  else erg = updateFreeSpaceStr( updatetime );
  if ( erg != 0 ) updateName();
}

/*
 * method updates freespacestr
 * returns 0 for no change, 1 for change
 */
int NormalMode::updateFreeSpaceStr( int ti )
{
  char *tstr;
  int len;
  time_t now;
  int erg, rv = 0;

  if(showfreespace==true) {
    now=time(NULL);
    if ( ( now - lastfsupdate >= ti ) || ( lasteagain == true ) ) {
      if ( getCurrentDir() != NULL ) {
        erg = parentlister->getWorker()->PS_readSpace( getCurrentDir() );
        if ( erg == 0 ) {
            std::string spaceh, freeh;

            spaceh = parentlister->getWorker()->PS_getSpaceH();
            freeh = parentlister->getWorker()->PS_getFreeSpaceH();
            
            len = freeh.length() +
                spaceh.length() +
                strlen( catalog.getLocale( 849 ) ) + 1;

            tstr = (char*)_allocsafe( len );
            sprintf( tstr, catalog.getLocale( 849 ), freeh.c_str(), spaceh.c_str() );
            if ( freespacestr != NULL ) _freesafe( freespacestr );
            freespacestr = tstr;
            lasteagain = false;
            rv = 1;
        } else if ( erg == EAGAIN ) {
          // no valid value so lets try again next call
          lasteagain = true;
        }
        lastfsupdate=now;
      }
    }
  } else {
    if ( freespacestr != NULL ) {
      _freesafe( freespacestr );
      freespacestr = NULL;
      rv = 1;
    }
  }
  return rv;
}

void NormalMode::setShowFreeSpace(bool v)
{
  showfreespace=v;
  setName();
  showFreeSpace(true);
}

void NormalMode::setUpdatetime(int nv)
{
  updatetime=nv;
  if(updatetime<1) updatetime=1;
  parentlister->getWorker()->PS_setLifetime( (double)updatetime );
}

bool NormalMode::startdnd(DNDMsg *dm)
{
  bool returnvalue=false;
  if(dm->getStart()->element==lv) {
    returnvalue=true;

    int row = dm->getStart()->specialinfo.value;
    if(lv->isValidRow(row)==true) {
      if(ce!=NULL) {
        if ( ce->dirOpened() == true ) {
          FileEntry *fe = getFEForRow( row );
          if(fe!=NULL) {
            // found entry for dnd
            // now use the type to call the dnd action

            WCFiletype *ft;
            ActionMessage amsg( parentlister->getWorker() );
            amsg.startLister=parentlister;
            amsg.mode=amsg.AM_MODE_DNDACTION;

            if(fe->isDir()==false) {
              ft=fe->filetype;
              if ( ft == NULL ) ft = wconfig->getnotyettype(); // not yet checked
            } else {
              ft = wconfig->getdirtype();
            }
            if ( ft == NULL ) ft = wconfig->getvoidtype();
            if(ft!=NULL) {
              if ( ft->getDNDActions()->size() < 1 ) ft = wconfig->getvoidtype();
              if(ft!=NULL) {
                amsg.dndmsg = dm;
		amsg.filetype = ft;
                amsg.m_action_descr = RefCount<ActionDescr>( new DNDAction::DNDActionDescr() );
                parentlister->getWorker()->interpret(ft->getDNDActions(),&amsg);
              }
            }
          }
        }
      }
    }
  }
  return returnvalue;
}

bool NormalMode::isyours( Widget *elem )
{
  if(elem==lv) return true;
  return false;
}

FileEntry *NormalMode::getFE4DNDSTART( const AGDNDSTART *st )
{
  int row;
  FileEntry *fe=NULL;
  NMRowData *rdp;

  if ( st == NULL ) return NULL;
  if ( st->specialinfo.rowDataP == NULL ) return NULL;
  if( ( lv != NULL ) && ( st->element == lv ) ) {
    // lookup strategy:
    // 1.check if row (value) contains FE
    // 2.if not, find FE
    // 3.check fullname with found FE
    //TODO:search for fullname
    //     this is no bug but would improve it
    //     this would also only help users which
    //     try to trigger this
    row = st->specialinfo.value;
    if(lv->isValidRow(row)==true) {
      if(ce!=NULL) {
        if ( ce->dirOpened() == true ) {
          fe = getFEForRow( row );
          //TODO:for more secure cast
          //if ( typeid( *(st->specialinfo.rowDataP) ) == typeid( NMRowData ) );
          rdp = (NMRowData*)st->specialinfo.rowDataP;
          if ( fe->equals( rdp->getFE() ) == false ) {
            fe = NULL;
            for ( Verzeichnis::verz_it fe_it1 = ce->begin();
                  fe_it1 != ce->end();
                  fe_it1++ ) {
              fe = *fe_it1;
              if ( fe->equals( rdp->getFE() ) == true ) break;
              fe = NULL;
            }
          }
          if ( fe != NULL ) {
	    // The idea for this test was that the DND msg contains a pointer to an old
	    // FileEntry which could be overwritten by another instance so I check the fullname
	    // As I now store copies of the FE inside the NMRowData and use equals() to find
	    // the corresponding FE this test is obsolete but doesn't hurt
            if ( strcmp( fe->fullname, rdp->getFullname() ) != 0 ) fe = NULL;
          }
        }
      }
    }
  }
  return fe;
}

void NormalMode::unsetAllFilters()
{
    _dir_filter_sets.unsetAllFilters();
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
        if ( ce->getActiveFE() != NULL )
            lv->centerActive();
    setName();
}

void NormalMode::setFilter(const char *filter,nm_filter_t mode)
{
    NM_Filter::check_t newmode;

    switch ( mode ) {
      case NM_FILTER_EXCLUDE:
          newmode = NM_Filter::EXCLUDE;
          break;
      case NM_FILTER_UNSET:
          newmode = NM_Filter::INACTIVE;
          break;
      default:
          newmode = NM_Filter::INCLUDE;
          break;
    }
    _dir_filter_sets.setFilter( filter, newmode );
    
    // aktualisieren
    rebuildView();
    if ( ce != NULL )
        if ( ce->getActiveFE() != NULL )
            lv->centerActive();
    setName();
}

void NormalMode::slavehandler()
{
  const char *fname;
  char *filename;
  ft_recres_list::ft_recres_list_t *te;
  bool ende;
  int erg;
  bool dontCheckContent;
  
  // to get from REINIT directly to RUN
  bool firstrun = true;

  if ( slave.running != 0 ) {
    fprintf( stderr, "Worker: another thread already running!\n");
    return;
  }
  slave.running = 1;
  switchStatus( THREAD_REINIT );
  switchOrder( THREAD_NOP );

  slave.filetype_ex.lock();

#ifdef DEBUG
  printf("entering slave handler\n");
#endif

  for( ende = false; ende == false; ) {

    switch ( slave.status ) {
      case THREAD_RUN:
#ifdef DEBUG
        printf("waiting for element\n");
#endif

        reclist->lock();
        // wait for next element or stop
        while ( ( reclist->elemAvail_locked() == false ) && ( slave.order == THREAD_NOP ) )
          reclist->wait();

#ifdef DEBUG
        printf("wait finished\n");
#endif

        if ( slave.order != THREAD_NOP ) {
          // new command
          // switch mode for new command
          if ( slave.order == THREAD_EXIT ) {
            switchStatus( THREAD_EXIT );
            ende = true;
          } else if ( slave.order == THREAD_REINIT ) {
            switchStatus( THREAD_REINIT );
          } else if ( slave.order == THREAD_WAIT ) {
            switchStatus( THREAD_WAIT );
          }
          // mark this command read
          switchOrder( THREAD_NOP );
          reclist->unlock();
        } else {
          // an element to check
          erg = reclist->slaveread_locked( &fname, &dontCheckContent );
          filename = NULL;
          if ( erg == 0 ) {
            if ( fname != NULL ) {
              filename = dupstring( fname );
            }
          }
          reclist->unlock();

          if ( ( erg == 0 ) && ( filename != NULL ) ) {
	    checkFiletypeRes_t res;
#ifdef DEBUG
//            printf("erkenne %s\n",filename);
#endif
            slave_checkFiletype( filename, dontCheckContent, res );
            recreslist->lock();
    
            te = new ft_recres_list::ft_recres_list_t;
            te->name = filename;
            te->ft_index = res.v;
	    if ( res.custom_color.get() != NULL &&
                 res.custom_color->getColorSet() != 0 ) {
                te->custom_color = res.custom_color;
	    }
    
            recreslist->put_locked( te );
            recreslist->unlock();
          }
        }
        break;
      case THREAD_WAIT:
        waitForNewOrder();
        if ( slave.order == THREAD_EXIT ) {
          switchStatus( THREAD_EXIT );
          ende = true;
        } else if ( slave.order == THREAD_REINIT ) {
          switchStatus( THREAD_REINIT );
        } else if ( slave.order == THREAD_RUN ) {
          switchStatus( THREAD_RUN );
        }
        switchOrder( THREAD_NOP );
        break;
      case THREAD_REINIT:
        // wait for change
        slave.filetype_ex.signal();
        slave.filetype_ex.wait();
        
        // change mode
        if ( firstrun == true ) {
          firstrun = false;
          switchStatus( THREAD_RUN );
        } else {
          switchStatus( THREAD_WAIT );
        }
        break;
      default:
        break;
    }
  }
    
  slave.filetype_ex.unlock();

#ifdef DEBUG
  printf("leaving slave handler\n");
#endif
}

int NormalMode::ft_rec_list::slaveread_locked( const char **name, bool *dontCheckContent )
{
  if ( ( name == NULL ) || ( dontCheckContent == NULL ) ) return 1;
  if ( readp != NULL ) {
    *name = readp->name;
    *dontCheckContent = readp->dontCheckContent;
    readp = readp->next;
  } else return 1;
  return 0;
}

bool NormalMode::ft_rec_list::isFull_locked()
{
  if ( elements >= size ) return true;
  return false;
}

bool NormalMode::ft_rec_list::isEmpty_locked()
{
  if ( elements < 1 ) return true;
  return false;
}

int NormalMode::ft_rec_list::put_locked( ft_rec_list_t *elem )
{
  int pos;

  if ( elem == NULL ) return -1;
  if ( isFull_locked() == true ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
    // if readp is null, slave has read all elems, so set it to this new elem
    if ( readp == NULL )
      readp = tail;
  } else {
    head = tail = elem;
    readp = tail;
  }

  if ( elem->fe != NULL )
    elem->fe->reclistQueued = true;

  pos = elements++;
  return pos;
}

NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::remove_locked()
{
  ft_rec_list_t *te;

  if ( elements == 0 ) return NULL;
  
  // readp could be the element to remove
  if ( readp == head )
    readp = readp->next;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  if ( te->fe != NULL )
    te->fe->reclistQueued = false;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

NormalMode::ft_rec_list::ft_rec_list()
{
  size = 10;
  head = NULL;
  tail = NULL;
  readp = NULL;
  elements = 0;
}

NormalMode::ft_rec_list::~ft_rec_list()
{
  ft_rec_list_t *te;

  lock();
  while ( isEmpty_locked() == false ) {
    te = remove_locked();
    delete te;
  }
  unlock();
}

void NormalMode::ft_rec_list::lock()
{
  ex.lock();
}

void NormalMode::ft_rec_list::unlock()
{
  ex.unlock();
}

void NormalMode::ft_rec_list::wait()
{
  ex.wait();
}

void NormalMode::ft_rec_list::signal()
{
  ex.signal();
}

bool NormalMode::ft_rec_list::elemAvail_locked()
{
  if ( readp != NULL ) return true;
  return false;
}

NormalMode::ft_recres_list::ft_recres_list()
{
  head = tail = NULL;
  elements = 0;
}

NormalMode::ft_recres_list::~ft_recres_list()
{
  ft_recres_list_t *rte;

  lock();
  while ( isEmpty_locked() == false ) {
    rte = remove_locked();
    if ( rte != NULL ) {
      delete rte;
    }
  }
  unlock();
}

bool NormalMode::ft_recres_list::isEmpty_locked()
{
  if ( elements == 0 ) return true;
  return false;
}

int NormalMode::ft_recres_list::put_locked( ft_recres_list_t *elem )
{
  if ( elem == NULL ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
  } else {
    head = tail = elem;
  }

  elements++;
  return 0;
}

NormalMode::ft_recres_list::ft_recres_list_t *NormalMode::ft_recres_list::remove_locked()
{
  ft_recres_list_t *te;

  if ( elements == 0 ) return NULL;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

void NormalMode::ft_recres_list::lock()
{
  ex.lock();
}

void NormalMode::ft_recres_list::unlock()
{
  ex.unlock();
}


NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::gettop_locked()
{
  return head;
}

void NormalMode::ft_rec_list_clear()
{
  ft_rec_list::ft_rec_list_t *te;
  NMCacheEntry *tce;
  int id;

  reclist->lock();
  while ( reclist->isEmpty_locked() == false ) {
    te = reclist->remove_locked();
    delete te;
  }
  // reset checkfe for all caches
  id = cache->initEnum();
  tce = (NMCacheEntry*)cache->getFirstElement( id );
  while ( tce != NULL ) {
      tce->resetCheckFE2FirstNULLFT();
      tce = (NMCacheEntry*)cache->getNextElement( id );
  }
  cache->closeEnum( id );
  reclist->unlock();
  
  visChanged = true;
}

void NormalMode::ft_recres_list_clear()
{
  ft_recres_list::ft_recres_list_t *rte;

  recreslist->lock();
  while ( recreslist->isEmpty_locked() == false ) {
    // alle Resultate entnehmen
    rte = recreslist->remove_locked();
    if ( rte != NULL ) {
      delete rte;
    }
  }
  recreslist->unlock();
}

/*
 * ft_list_clear
 *
 * will delete all filetypes from list
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_clear()
{
  WCFiletype *ft;
  int id;

  slave.filetype_ex.lock();
  
  id = slave.filetypes->initEnum();
  ft = (WCFiletype*)slave.filetypes->getFirstElement( id );
  while ( ft != NULL ) {
    delete ft;
    slave.filetypes->removeFirstElement();
    ft = (WCFiletype*)slave.filetypes->getFirstElement( id );
  }
  slave.filetypes->closeEnum( id );
  
  slave.filetype_ex.unlock();
}

/*
 * ft_list_update
 *
 * will update the filetype-list to the current config
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_update()
{
  List *ftlist;
  WCFiletype *ft;
  int id;
  
  slave.filetype_ex.lock();

  ft_list_clear();

  ftlist = wconfig->getFiletypes();
  id = ftlist->initEnum();
  ft = (WCFiletype*)ftlist->getFirstElement( id );
  while ( ft != NULL ) {
    slave.filetypes->addElement( ft->duplicate() );
    ft = (WCFiletype*)ftlist->getNextElement( id );
  }
  ftlist->closeEnum( id );
  
  slave.filetype_ex.unlock();
}

int NormalMode::slave_checkFiletype( const char *fullname, bool dontCheckContent, checkFiletypeRes_t &res_return )
{
  FileEntry *fe1;
  int pos = -1;
  WCFiletype *ft, *parentft;
  std::vector<unsigned int> *v = NULL;

  slave.filetype_ex.lock();

  fe1 = new FileEntry();
  fe1->fullname = dupstring( fullname );
  fe1->name = Datei::getFilenameFromPath( fullname );
  if( fe1->readInfos() == 0 ) {
    ft = fe1->checkFiletype( slave.filetypes, dontCheckContent, &condparser );
    if ( ft != NULL ) {
      // first get vector with filetype position for each child
      v = ft->getTypePos();
      // now find root filetype
      
      for ( parentft = ft; parentft->getParentType() != NULL; parentft = parentft->getParentType() );
      pos = slave.filetypes->getIndex( parentft );
      if ( pos >= 0 ) {
	if ( v != NULL ) v->push_back( (unsigned int)pos );
      } else {
	if ( v != NULL ) {
	  delete v;
	  v = NULL;
	}
      }
    }
  }
  res_return.v = v;

  if ( fe1->getCustomColorSet() != 0 ) {
      try {
          res_return.custom_color.reset( new FileEntryCustomColor( fe1->getCustomColor() ) );
      } catch ( int ) {
      }
  }
  delete fe1;

  slave.filetype_ex.unlock();
  return 0;
}

void NormalMode::setupLVFields()
{
  int side = parentlister->getSide();
  const std::vector<WorkerTypes::listcol_t> *sets;
  int i;
  int used_fields, li;
  AGUIXFont *afont;
  int space_width;
  int cur_field;
  int tw;
  
  sets = wconfig->getVisCols( side );
  if ( sets == NULL ) return;
  
  used_fields = sets->size();

  afont = aguix->getFont( wconfig->getFont( 2 + side ) );
  space_width = aguix->getTextWidth( " ", afont );

  if ( used_fields == 0 ) {
    // oops, what's this?
    lv->setNrOfFields( 1 );  // 0 fields are not supported, blame the author (me ;-) )
  } else {
    lv->setNrOfFields( used_fields * 2 - 1 );
  
    cur_field = 0;
    for( i = 0; i < (int)sets->size(); i++ ) {
      if ( i > 0 ) {
        lv->setFieldWidthQ( cur_field++, space_width );
      }
      lv->setFieldTextMergedQ( cur_field, true );
      lv->setFieldTextResizeable( cur_field, true );
      li = WorkerTypes::getAvailListColEntry( (*sets)[i] );
      if ( li >= 0 ) {
        lv->setFieldTextQ( cur_field, catalog.getLocale( WorkerTypes::availListCols[li].catalogid ) );
      }
      switch ( (*sets)[i] ) {
        case WorkerTypes::LISTCOL_NAME:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_NAME" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;

          lv->setFieldWidthQ( cur_field, tw );
	  
	  _field_map["LISTCOL_NAME"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_SIZE:
#ifdef LEFTJUSTIFY
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );
#else
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );
#endif
	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_SIZE" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;

          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_SIZE"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_TYPE:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_TYPE" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_TYPE"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_PERM:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_PERM" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_PERM"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_OWNER:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_OWNER" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_OWNER"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_DEST:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_DEST" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_DEST"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_MOD:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_MOD" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_MOD"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_ACC:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_ACC" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_ACC"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_CHANGE:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_LEFT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_CHANGE" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_CHANGE"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_INODE:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_INODE" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_INODE"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_NLINK:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_NLINK" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_NLINK"] = cur_field;
          break;
        case WorkerTypes::LISTCOL_BLOCKS:
          lv->setFieldAlignQ( cur_field, FieldListView::ALIGN_RIGHT );

	  tw = getAndEraseLoadedFieldWidth( "LISTCOL_BLOCKS" );
	  if ( tw < -1 || tw > 10000 ) tw = -1;
          lv->setFieldWidthQ( cur_field, tw );

	  _field_map["LISTCOL_BLOCKS"] = cur_field;
          break;
        default:
          break;
      }
      cur_field++;
    }
    lv->redraw();
  }
}

void NormalMode::chownf(struct NM_chownorder *coorder)
{
  NM_specialsourceInt *ss1;
  changeown_info_t *coinfo;
  std::list<NM_specialsourceInt*> *colist;
  std::list<NM_specialsourceInt*>::iterator iti1;
  
  if ( coorder == NULL ) return;
  if ( getCurrentDir() == NULL ) return;

  finishsearchmode();
  
  colist = new std::list<NM_specialsourceInt*>;
  switch ( coorder->source ) {
    case NM_chownorder::NM_SPECIAL:
        buildSpecialSourceIntList( *( coorder->sources ),
                                   *colist );
        break;
    case NM_chownorder::NM_ONLYACTIVE:
      getSelFiles( colist, NM_GETFILES_ONLYACTIVE, false );
      break;
    default:  // all selected entries
      getSelFiles( colist, NM_GETFILES_SELORACT, false );
      break;
  }

  buildOwnerRequestInfos();

  coinfo = new changeown_info_t;
  for ( iti1 = colist->begin(); iti1 != colist->end(); iti1++ ) {
    ss1 = *iti1;
    if ( worker_changeown( ss1, coinfo, coorder ) != 0 ) {
      break;
    }
  }

  for ( iti1 = colist->begin(); iti1 != colist->end(); iti1++ ) {
    ss1 = *iti1;
    delete ss1;
  }
  delete coinfo;
  delete colist;
  freeOwnerRequestInfos();
  update(false);
}

int NormalMode::worker_changeown( const struct NM_specialsourceInt *ss1,
                                  changeown_info_t *coinfo,
                                  const NM_chownorder *coorder )
{
  bool enter, dontChange, skip, cancel;
  int erg;
  uid_t newuid = (uid_t)-1;
  gid_t newgid = (gid_t)-1;
  FileEntry *subfe;
  NM_specialsourceInt *ss2;
  NM_CopyOp_Dir *cod;
  
  if ( ( ss1 == NULL ) || ( coinfo == NULL ) || ( coorder == NULL ) ) return -1;

  // do we have to enter this entry?
  enter = false;
  if ( ( ss1->entry()->isDir() == true ) && ( coorder->recursive == true ) ) {
    if ( ss1->entry()->isLink == false ) enter = true;
  }
  
  // check operation applies to this entry
  dontChange = false;
  if ( ( ss1->entry()->isDir() == true ) && ( coorder->ondirs == false ) ) dontChange = true;
  else if ( ( ss1->entry()->isDir() == false ) && ( coorder->onfiles == false ) ) dontChange = true;

  /* skip means skip entry AND all sub entries
   * cancel is clear
   * dontchange means to normally handle this entry
   *   but actually don't change mod
   */
  skip = cancel = false;
  if ( dontChange == false ) {
    // ask for new owner
    if ( coinfo->forAll == true ) {
      newuid = coinfo->newuid;
      newgid = coinfo->newgid;
    } else {
      erg = requestNewOwner( ss1->entry(), &newuid, &newgid );
      if ( erg == 1 ) {
        coinfo->forAll = true;
        coinfo->newuid = newuid;
        coinfo->newgid = newgid;
      } else if ( erg == 2 ) {
        skip = true;
      } else if ( erg == 3 ) {
        cancel = true;
      }
    }
  }
  if ( skip == true ) return 0;
  if ( cancel == true ) return 1;

  if ( applyNewOwner( ss1, newuid, newgid ) != 0 ) cancel = true;
  if ( ( enter == true ) && ( cancel == false ) ) {
    cod = new NM_CopyOp_Dir( ss1->entry() );
    if ( cod->user_abort == true ) {
      cancel = true;
    } else if ( cod->ok == true ) {
      for ( Verzeichnis::verz_it subfe_it1 = cod->verz->begin();
            subfe_it1 != cod->verz->end() && cancel == false;
            subfe_it1++ ) {
        subfe = *subfe_it1;
        if ( strcmp( subfe->name, ".." ) != 0 ) {
	  ss2 = new NM_specialsourceInt( subfe );
          ss2->row = -1;
          if ( worker_changeown( ss2, coinfo, coorder ) != 0 ) {
            cancel = true;
          }
          delete ss2;
        }
      }
    }
    delete cod;
  }
  return ( cancel == true ) ? 1 : 0;
}

int NormalMode::applyNewOwner( const NM_specialsourceInt *ss1, uid_t newuid, gid_t newgid )
{
  char *textstr, *buttonstr;
  int erg;
  bool cancel = false;
  
  if ( ss1 == NULL ) return -1;

  if ( worker_chown( ss1->entry()->fullname, newuid, newgid ) != 0 ) {
    if ( errno == EPERM ) {
      buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                     strlen( catalog.getLocale( 8 ) ) + 1 );
      sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                                   catalog.getLocale( 8 ) );
      textstr = (char*)_allocsafe( strlen( catalog.getLocale( 446 ) ) + strlen( ss1->entry()->fullname ) + 1 );
      sprintf( textstr, catalog.getLocale( 446 ), ss1->entry()->fullname );
      erg = req->request( catalog.getLocale( 347 ), textstr, buttonstr );
      _freesafe( buttonstr );
      _freesafe( textstr );
      if ( erg == 1 ) cancel = true;
    } else {
      // error
      buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 11 ) ) + 1 +
                                     strlen( catalog.getLocale( 8 ) ) + 1 );
      sprintf( buttonstr, "%s|%s", catalog.getLocale( 11 ),
                                   catalog.getLocale( 8 ) );
      textstr = (char*)_allocsafe( strlen( catalog.getLocale( 447 ) ) + strlen( ss1->entry()->fullname ) + 1 );
      sprintf( textstr, catalog.getLocale( 447 ), ss1->entry()->fullname );
      erg = req->request( catalog.getLocale( 347 ), textstr, buttonstr );
      _freesafe( buttonstr );
      _freesafe( textstr );
      if ( erg == 1 ) cancel = true;
    }
  } else {
    if ( lv->isValidRow( ss1->row ) == true ) deselect( ss1->entry(), ss1->row );
  }
  return ( cancel == true ) ? 1 : 0;
}

void NormalMode::freeOwnerRequestInfos()
{
  chownrequest_id_name_t *elem;

  if ( chownUserList != NULL ) {
    elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
    while ( elem != NULL ) {
      _freesafe( elem->name );
      _freesafe( elem );
      chownUserList->removeFirstElement();
      elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
    }
    delete chownUserList;
    chownUserList = NULL;
  }
  
  if ( chownGroupList != NULL ) {
    elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
    while ( elem != NULL ) {
      _freesafe( elem->name );
      _freesafe( elem );
      chownGroupList->removeFirstElement();
      elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
    }
    delete chownGroupList;
    chownGroupList = NULL;
  }
}

void NormalMode::buildOwnerRequestInfos()
{
  struct passwd *pwdp;
  struct group *grpp;
  chownrequest_id_name_t *elem;

  freeOwnerRequestInfos();

  chownUserList = new List();
  chownGroupList = new List();
  
  //setpwent();   // is this needed?
  pwdp = getpwent();
  while ( pwdp != NULL ) {
    elem = (chownrequest_id_name_t*)_allocsafe( sizeof( chownrequest_id_name_t ) );
    elem->name = dupstring( pwdp->pw_name );
    elem->id.uid = pwdp->pw_uid;
    chownUserList->addElement( elem );
    pwdp = getpwent();
  }
  endpwent();

  //setgrent();  // is this needed?
  grpp = getgrent();
  while ( grpp != NULL ) {
    elem = (chownrequest_id_name_t*)_allocsafe( sizeof( chownrequest_id_name_t ) );
    elem->name = dupstring( grpp->gr_name );
    elem->id.gid = grpp->gr_gid;
    chownGroupList->addElement( elem );
    grpp = getgrent();
  }
  endgrent();
}

int NormalMode::requestNewOwner( FileEntry *fe, uid_t *return_owner, gid_t *return_group )
{
  uid_t towner;
  gid_t tgroup;
  Button *okb, *cb, *ok2allb, *skipb;
  AWindow *win;
  Text *ttext, *utext, *gtext;
  int tw, ttw, tth, ttx, tty;
  AGMessage *msg;
  int endmode = -1;
  char *tstr;
  GUIElement *ba[4];
  FieldListView *lvu, *lvg;
  chownrequest_id_name_t *elem;
  int row, pos, i;
  int w;
  
  if ( ( chownUserList == NULL ) || ( chownGroupList == NULL ) )
    buildOwnerRequestInfos();

  towner = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->duserid() : fe->userid();
  tgroup = ( ( fe->isLink == true ) && ( fe->isCorrupt == false ) ) ? fe->dgroupid() : fe->groupid();

  ttw = tth = 10;
  ttx = tty = 5;
  win = new AWindow( aguix, 10, 10, ttw, tth, 0, catalog.getLocaleCom( 43 ) );
  win->create();
  tstr = (char*)_allocsafe( strlen( catalog.getLocale( 448 ) ) + strlen( fe->fullname ) + 1 );
  sprintf( tstr, catalog.getLocale( 448 ), fe->fullname );

  ttext = (Text*)win->add( new Text( aguix, ttx, tty, tstr, 1 ) );
  _freesafe( tstr );
  
  tty += ttext->getHeight() + 5;
  
  utext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 214 ), 1 ) );
  gtext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 215 ), 1 ) );
  tty += utext->getHeight() + 5;

  lvu = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 50, 10 * aguix->getCharHeight(), 0 ) );
  lvg = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 50, 10 * aguix->getCharHeight(), 0 ) );
  
  lvu->setNrOfFields( 1 );
  lvg->setNrOfFields( 1 );

  elem = (chownrequest_id_name_t*)chownUserList->getFirstElement();
  pos = 0;
  while ( elem != NULL ) {
    row = lvu->addRow();
    lvu->setText( row, 0, elem->name );
    lvu->setData( row, pos );
    lvu->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    if ( towner == elem->id.uid ) {
      lvu->setActiveRow( row );
    }
    elem = (chownrequest_id_name_t*)chownUserList->getNextElement();
    pos++;
  }
  elem = (chownrequest_id_name_t*)chownGroupList->getFirstElement();
  pos = 0;
  while ( elem != NULL ) {
    row = lvg->addRow();
    lvg->setText( row, 0, elem->name );
    lvg->setData( row, pos );
    lvg->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    if ( tgroup == elem->id.gid ) {
      lvg->setActiveRow( row );
    }
    elem = (chownrequest_id_name_t*)chownGroupList->getNextElement();
    pos++;
  }
  
  lvu->setVBarState( 2 );
  lvu->setHBarState( 0 );
  lvg->setVBarState( 2 );
  lvg->setHBarState( 0 );
  lvu->maximizeX();
  lvg->maximizeX();
  lvu->setDisplayFocus( true );
  lvg->setDisplayFocus( true );
  lvu->setAcceptFocus( true );
  lvg->setAcceptFocus( true );
  
  tw = a_max( utext->getWidth(), lvu->getWidth() );
  lvu->resize( a_max( tw, 20 * aguix->getTextWidth( "x" ) ), lvu->getHeight() );
  tw = a_max( gtext->getWidth(), lvg->getWidth() );
  lvg->resize( a_max( tw, 20 * aguix->getTextWidth( "x" ) ), lvg->getHeight() );
  lvg->move( lvu->getX() + lvu->getWidth() + 10, lvg->getY() );
  lvu->showActive();
  lvg->showActive();
  
  gtext->move( lvg->getX(), gtext->getY() );
  
  lvu->takeFocus();

  tty += lvu->getHeight() +5;
  
  win->maximizeX();
  w = win->getWidth();

  okb = (Button*)win->add( new Button( aguix,
                                       5,
                                       tty,
                                       catalog.getLocale( 11 ),
                                       1,
                                       0,
                                       0 ) );
  ok2allb = (Button*)win->add( new Button( aguix,
                                           0,
                                           tty,
                                           catalog.getLocale( 224 ),
                                           1,
                                           0,
                                           0 ) );
  skipb = (Button*)win->add( new Button( aguix,
                                         0,
                                         tty,
                                         catalog.getLocale( 225 ),
                                         1,
                                         0,
                                         0 ) );
  cb = (Button*)win->add( new Button( aguix,
                                      0,
                                      tty,
                                      catalog.getLocale( 8 ),
                                      1,
                                      0,
                                      0 ) );

  ba[0] = okb;
  ba[1] = ok2allb;
  ba[2] = skipb;
  ba[3] = cb;
  tw = AGUIX::scaleElementsW( w, 5, 5, -1, false, false, ba, NULL, 4 );
  if ( tw > w ) {
    w = tw;
    win->resize( w, win->getHeight() );
  }
  
  tty += okb->getHeight() + 5;
  
  for ( i = 0; i < 4; i++ ) {
    ba[i]->setAcceptFocus( true );
  }
  
  tth = tty;
  okb->takeFocus();
  win->setDoTabCycling( true );
  win->resize( w, tth );
  win->setMaxSize( w, tth );
  win->setMinSize( w, tth );
  win->show();
  for( ; endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 3;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == okb ) endmode = 0;
          else if ( msg->button.button == ok2allb ) endmode = 1;
          else if ( msg->button.button == skipb ) endmode = 2;
          else if ( msg->button.button == cb ) endmode = 3;
          break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_Return:
              case XK_KP_Enter:
                if ( ( ok2allb->getHasFocus() == false ) &&
                     ( skipb->getHasFocus() == false ) &&
                     ( cb->getHasFocus() == false ) ) {
                  endmode = 0;
                }
                break;
              case XK_F1:
                endmode = 0;
                break;
              case XK_Escape:
              case XK_F4:
                endmode = 3;
                break;
              case XK_F2:
                endmode = 1;
                break;
              case XK_F3:
                endmode = 2;
                break;
            }
          }
          break;
        case AG_FIELDLV_PRESSED:
          if ( msg->fieldlv.lv == lvu ) {
            lvu->takeFocus();
          } else if ( msg->fieldlv.lv == lvg ) {
            lvg->takeFocus();
          }
      }
      aguix->ReplyMessage( msg );
    }
  }
  
  if ( ( endmode == 0 ) || ( endmode == 1 ) ) {
    // ok
    if ( return_owner != NULL )
      *return_owner = (uid_t)-1;  // -1 means no change for chown
    if ( return_group != NULL )
      *return_group = (gid_t)-1;

    row = lvu->getActiveRow();
    if ( lvu->isValidRow( row ) == true ) {
      elem = (chownrequest_id_name_t*)chownUserList->getElementAt( lvu->getData( row ) );
      if ( elem != NULL ) {
        if ( return_owner != NULL )
          *return_owner = elem->id.uid;
      }
    }

    row = lvg->getActiveRow();
    if ( lvg->isValidRow( row ) == true ) {
      elem = (chownrequest_id_name_t*)chownGroupList->getElementAt( lvg->getData( row ) );
      if ( elem != NULL ) {
        if ( return_group != NULL )
          *return_group = elem->id.gid;
      }
    }
  }
  
  delete win;

  return endmode;
}

void NormalMode::fixSpecialSourceList( const std::list<NM_specialsourceInt*> *l, int deleted_row )
{
  std::list<NM_specialsourceInt*>::const_iterator iti1;
  if ( l == NULL ) return;
  if ( deleted_row < 0 ) return;
  
  NM_specialsourceInt *ss1;
  
  for ( iti1 = l->begin(); iti1 != l->end(); iti1++ ) {
    ss1 = *iti1;
    if ( ss1->row == deleted_row ) {
      ss1->row = -1;
    } else if ( ss1->row > deleted_row ) {
      ss1->row--;
    }
  }
}

void NormalMode::showCacheAbs( int pos )
{
  NMCacheEntry *tce;
  int id;

  finishsearchmode();
  
  if ( ce != NULL ) {
    ce->setPos( lv->getYOffset() );
    ce->setXPos( lv->getXOffset() );
    ce->setActiveIsVisible( lv->isRowVisible( lv->getActiveRow() ) );
  }
  
  id = cache->initEnum();
  tce = (NMCacheEntry*)cache->getElementAt( id, pos );
  if ( tce != NULL ) {
    // use copy of dir just to be sure we don't have a freed pointer
    std::string tstr = tce->getDir();
    enterDir( tstr.c_str() );
  }
  cache->closeEnum( id );
}

/*
 * isColumnVisible
 *
 * checks if given column is currently visible
 */
bool NormalMode::isColumnVisible( const WorkerTypes::listcol_t c )
{
  int side = parentlister->getSide();
  const std::vector<WorkerTypes::listcol_t> *dis = wconfig->getVisCols( side );
  int i;
  bool found = false;
  
  if ( dis == NULL ) return false;
  for ( i = 0; i < (int)dis->size(); i++ ) {
    if ( (*dis)[i] == c ) {
      found = true;
      break;
    }
  }
  return found;
}

char *NormalMode::parseComStrExt( const char *sstr,class NM_externorder *extorder,int maxlen,NMExtList *entries[2], bool quote )
{
  char *tstr;
  
  tstr = parseComStr( sstr, extorder, maxlen, entries, quote );
  return tstr;
}

void NormalMode::deselect( const FileEntry *fe )
{
  FileEntry *tfe;
  int row;

  if ( ce == NULL ) return;
  if ( ce->dirOpened() == false ) return;

  row = 0;
  tfe = NULL;
  
  for ( Verzeichnis::verz_it tfe_it1 = ce->begin();
        tfe_it1 != ce->end();
        tfe_it1++ ) {
    tfe = *tfe_it1;
    if ( tfe->use == true ) {
      if ( tfe->equals( *fe ) == true ) break;
      row++;
    }
    tfe = NULL;
  }
  if (tfe != NULL ) {
    /* found */
    deselect( tfe, row );
  }
}

/*
 * getSpecialsourceForFE
 *
 * creates a NM_specialsource for the given fe if found!
 * can return NULL
 */
NM_specialsourceExt *NormalMode::getSpecialsourceForFE( const FileEntry *searchfe )
{
  int row;
  NM_specialsourceExt *ss2;
  FileEntry *fe;

  if ( ce == NULL ) return NULL;
  if ( ce->dirOpened() == false ) return NULL;
  if ( searchfe == NULL ) return NULL;

  row = 0;
  ss2 = NULL;

  for ( Verzeichnis::verz_it fe_it1 = ce->begin();
        fe_it1 != ce->end();
        fe_it1++ ) {
    fe = *fe_it1;
    if ( fe->use == true ) {
      if ( fe->equals( searchfe ) == true ) {
	ss2 = new NM_specialsourceExt( fe );
        break;
      }
      row++;
    }
  }

  return ss2;
}

NM_externorder::NM_externorder()
{
  separate_each_entry = false;
  recursive = false;
  inbackground = false;

  extstart = NM_EXT_START_NORMAL;

  com_str = NULL;
  view_str = NULL;

  source = NM_ALLENTRIES;
  sources = NULL;
  
  destmode = NULL;
  wpu = NULL;
  take_dirs = false;
  dontcd = false;
}

NM_externorder::~NM_externorder()
{
  //TODO: at the moment I do nothing here but perhaps it's a good
  //      idea to free com_str/viewstr/sources... (it's done by the user of
  //      this class
}

void NormalMode::lvbDoubleClicked()
{
  showCacheList();
}

int NormalMode::showCacheList()
{
  AWindow *win;
  int my_w, my_h, endmode, mw, mh;
  AGMessage *msg;
  int row, my_lastrow;
  FieldListView *cachelv;
  StringGadget *cachesg;
  char *tstr1, *tstr2;
  int caches = cache->size();
  NMCacheEntry *tce = NULL;
  int id;
  int parents, cachebegin;
  struct timeval my_lastclick, acttime;
  const char *tstr, *ctstr2;
  
  win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 523 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );
  ac1->setBorderWidth( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 524 ), 1 ),
            0, 0, AContainer::CINCWNR );

  AContainer *ac1_2 = ac1->add( new AContainer( win, 1, 2 ), 0, 1 );
  ac1_2->setMinSpace( 0 );
  ac1_2->setMaxSpace( 0 );
  ac1_2->setBorderWidth( 0 );
  
  cachelv = (FieldListView*)ac1_2->add( new FieldListView( aguix, 0, 0, 50, 50 ,0 ),
                                        0, 0, AContainer::CMIN );
  cachelv->setHBarState( 2 );
  cachelv->setVBarState( 2 );
  
  cachesg = (StringGadget*)ac1_2->add( new StringGadget( aguix, 0, 0, 50, "", 0 ),
                                       0, 1, AContainer::CINCW );
  
  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 2 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( -1 );
  ac1_1->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_1->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, AContainer::CFIX );
  Button *cancelb = (Button*)ac1_1->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, AContainer::CFIX );

  ctstr2 = getCurrentDir();
  if ( ctstr2 != NULL ) {
    tstr1 = dupstring( ctstr2 );
    for ( ;; ) {
      tstr2 = ParentDir( tstr1, NULL );
      if ( tstr2 == NULL ) break;
      
      // add tstr2 to the lv
      row = cachelv->addRow();
      cachelv->setText( row, 0 , tstr2 );
      cachelv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
      
      _freesafe( tstr1 );
      tstr1 = tstr2;
      if ( strcmp( tstr1, "/" ) == 0 ) break;
    }
    _freesafe( tstr1 );
  }
  
  parents = cachelv->getElements();

  std::string delim1( strlen( catalog.getLocale( 523 ) ), '-' );
  row = cachelv->addRow();
  cachelv->setText( row, 0, delim1 );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  row = cachelv->addRow();
  cachelv->setText( row, 0, catalog.getLocale( 523 ) );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  row = cachelv->addRow();
  cachelv->setText( row, 0, catalog.getLocale( 812 ) );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  row = cachelv->addRow();
  cachelv->setText( row, 0, delim1 );
  cachelv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
  
  cachebegin = row + 1;

  if ( caches > 0 ) {
    id = cache->initEnum();
    tce = (NMCacheEntry*)cache->getFirstElement( id );
    while ( tce != NULL ) {
      row = cachelv->addRow();
      cachelv->setText( row, 0, tce->getDir() );
      cachelv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
      tce = (NMCacheEntry*)cache->getNextElement( id );
    }
    cache->closeEnum( id );
  }

  // maximize cachelv and store sizes
  cachelv->maximizeX();
  cachelv->maximizeY();
  my_w = cachelv->getWidth();
  my_h = cachelv->getHeight();

  // now resize container to 80% screen size
  mw = aguix->getRootWindowWidth() * 80 / 100;
  mh = aguix->getRootWindowHeight() * 80 / 100;
  ac1->resize( mw, mh );
  ac1->rearrange();

  // and check whether the cachelv needs less space
  // and set min parameter to match 80% or less
  if ( my_w < cachelv->getWidth() ) {
    ac1_2->setMinWidth( my_w, 0, 0 );
  } else {
    ac1_2->setMinWidth( cachelv->getWidth(), 0, 0 );
  }
  if ( my_h < cachelv->getHeight() ) {
    ac1_2->setMinHeight( my_h, 0, 0 );
  } else {
    ac1_2->setMinHeight( cachelv->getHeight(), 0, 0 );
  }

  win->contMaximize( true );
  win->setDoTabCycling( true );
  cachesg->takeFocus();

  if ( parents > 0 ) {
    cachelv->setActiveRow( 0 );
    cachesg->setText( cachelv->getText( 0, 0 ).c_str() );
  } else if ( caches > 0 ) {
    cachelv->setActiveRow( cachebegin );
    cachesg->setText( cachelv->getText( cachebegin, 0 ).c_str() );
  }

  // catch msgs before mapping, this are mostly resize msgs
  while ( ( msg = aguix->GetMessage( win ) ) != NULL ) aguix->ReplyMessage( msg );

  win->show();

  my_lastrow = -1;
  timerclear( &my_lastclick );
  for( endmode = -1; endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == okb ) endmode = 0;
          else if ( msg->button.button == cancelb ) endmode = 1;
          break;
        case AG_FIELDLV_ONESELECT:
        case AG_FIELDLV_MULTISELECT:
          if ( msg->fieldlv.lv == cachelv ) {
            row = cachelv->getActiveRow();
            if ( cachelv->isValidRow( row ) == true ) {
              if ( ( row < parents ) ||
                   ( row >= cachebegin ) ) {
                // this is one of the parents
                cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                if ( my_lastrow == row ) {
                  gettimeofday( &acttime, NULL );
                  if ( aguix->isDoubleClick( &acttime, &my_lastclick ) == true ) {
                    endmode = 0;
                    timerclear( &my_lastclick );
                  } else {
                    my_lastclick.tv_sec = acttime.tv_sec;
                    my_lastclick.tv_usec = acttime.tv_usec;
                  }
                } else {
                  my_lastrow = row;
                  gettimeofday( &my_lastclick, NULL );
                }
              }
            }
          }
          break;
        case AG_STRINGGADGET_CONTENTCHANGE:
          if ( msg->stringgadget.sg == cachesg ) {
            // to avoid doubleclick
            timerclear( &my_lastclick );
            //TODO: U.U. merkt user ein Lag, deshalb koennte man ein Flag setzen und
            // ausserhalb messen, wenn letzte Nachricht her ist und dann suchen
            // Aber dann muss ich getMessage machen
            tstr = cachesg->getText();
            for ( row = 0; row < cachelv->getElements(); row++ ) {
              if ( strcmp( tstr, cachelv->getText( row, 0 ).c_str() ) == 0 ) {
                if ( ( ( row >= 0 ) && ( row < parents ) ) ||
                     ( ( row >= cachebegin ) && ( row < ( cachebegin + caches ) ) ) ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                }
                break;
              }
            }
          }
          break;
        case AG_STRINGGADGET_CANCEL:
          if ( msg->stringgadget.sg == cachesg ) endmode = 1;
          break;
        case AG_STRINGGADGET_OK:
          if ( msg->stringgadget.sg == cachesg ) endmode = 0;
          break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_Up:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  if ( ( row >= parents ) && ( row <= cachebegin ) ) {
                    row = parents - 1;
                  } else {
                    row--;
                  }
                  if ( row >= 0 ) {
                    cachelv->setActiveRow( row );
                    cachelv->showActive();
                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                    my_lastrow = -1;
                  }
                }
                break;
              case XK_Down:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  if ( ( row >= ( parents - 1 ) ) && ( row < cachebegin ) ) {
                    row = cachebegin;
                  } else {
                    row++;
                  }
                  if ( cachelv->isValidRow( row ) == true ) {
                    cachelv->setActiveRow( row );
                    cachelv->showActive();
                    cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                    my_lastrow = -1;
                  }
                }
                break;
              case XK_Prior:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  row -= cachelv->getMaxDisplayV() - 1;
                  if ( ( row >= parents ) && ( row < cachebegin ) ) row = parents - 1;
                  if ( row < 0 ) {
                    if ( parents > 0 ) row = 0;
                    else row = cachebegin;
                  }
                } else {
                  if ( parents > 0 ) row = 0;
                  else if ( caches > 0 ) row = cachebegin;
                }
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_Next:
                row = cachelv->getActiveRow();
                if ( cachelv->isValidRow( row ) == true ) {
                  row += cachelv->getMaxDisplayV() - 1;
                  if ( ( row >= parents ) && ( row < cachebegin ) ) row = cachebegin;
                  if ( row >= ( cachebegin + caches ) ) {
                    if ( caches > 0 ) row = cachebegin + caches - 1;
                    else row = parents - 1;
                  }
                } else {
                  if ( parents > 0 ) row = 0;
                  else if ( caches > 0 ) row = cachebegin;
                }
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_Home:
                row = -1;
                if ( parents > 0 ) row = 0;
                else if ( caches > 0 ) row = cachebegin;
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_End:
                row = -1;
                if ( caches > 0 ) row = cachebegin + caches - 1;
                else if ( parents > 0 ) row = parents - 1;
                if ( cachelv->isValidRow( row ) == true ) {
                  cachelv->setActiveRow( row );
                  cachelv->showActive();
                  cachesg->setText( cachelv->getText( row, 0 ).c_str() );
                  my_lastrow = -1;
                }
                break;
              case XK_Return:
              case XK_KP_Enter:
                if ( cancelb->getHasFocus() == false ) {
                  endmode = 0;
                }
                break;
              case XK_Escape:
                endmode = 1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage( msg );
    }
  }
  if ( endmode == 0 ) {
    // ok->take cachesg and enterdir
    enterDir( cachesg->getText() );
  }
  delete win;
  return endmode;
}

void NormalMode::showInvalidList( ArrayList *list, const char *descr )
{
  int id;
  FileEntry *tfe;
  int endmode, row;
  AWindow *win;
  FieldListView *my_lv;
  Button *okb;
  AGMessage *msg;

  if ( ( list == NULL ) || ( descr == NULL ) ) return;
  
  win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 125 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );
  ac1->setBorderWidth( 5 );

  win->addMultiLineText( descr, *ac1, 0, 0, NULL, NULL );

  my_lv = (FieldListView*)ac1->add( new FieldListView( aguix,
                                                       0,
                                                       0,
                                                       40 * aguix->getTextWidth( "x" ),
                                                       6 * aguix->getCharHeight(),
                                                       0 ),
                                    0, 1, AContainer::CMIN );
  
  my_lv->setHBarState( 2 );
  my_lv->setVBarState( 2 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 3, 1 ), 0, 2 );
  ac1_1->setMinSpace( 0 );
  ac1_1->setMaxSpace( 0 );
  ac1_1->setBorderWidth( 0 );
  
  okb = (Button*)ac1_1->add( new Button( aguix,
                                         0,
                                         0,
                                         catalog.getLocale( 11 ),
                                         1,
                                         0,
                                         0 ),
                             1, 0, AContainer::CFIX );
  win->contMaximize( true );
  win->setDoTabCycling( true );
  win->show();
  
  id = list->initEnum();
  tfe = (FileEntry*)list->getFirstElement( id );
  while ( tfe != NULL ) {
    row = my_lv->addRow();
    my_lv->setText( row, 0, tfe->name );
    my_lv->setPreColors( row, FieldListView::PRECOLOR_NOTSELORACT );
    tfe = (FileEntry*)list->getNextElement( id );
  }
  list->closeEnum( id );
    
  for( endmode = -1; endmode == -1; ) {
    msg = aguix->WaitMessage( win );
    if ( msg != NULL ) {
      switch ( msg->type ) {
        case AG_CLOSEWINDOW:
          if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
          break;
        case AG_BUTTONCLICKED:
          if ( msg->button.button == okb ) endmode = 0;
          break;
        case AG_KEYPRESSED:
          if ( win->isParent( msg->key.window, false ) == true ) {
            switch ( msg->key.key ) {
              case XK_Return:
              case XK_KP_Enter:
                endmode = 0;
                break;
              case XK_Escape:
                endmode = 1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage( msg );
    }
  }
  delete win;
  return;
}

int NormalMode::request( const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
  return req->request( title, text, buttons, flags );
}

int NormalMode::string_request( const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
  return req->string_request( title, lines, default_str, buttons, return_str, flags );
}

int NormalMode::request_choose( const char *title,
				const char *text,
				const char *choose_text,
				bool &choose_var,
				const char *buttons,
				Requester::request_flags_t flags )
{
  return req->request_choose( title, text, choose_text, choose_var, buttons, flags );
}

int NormalMode::request( CopyOpWin *cowin,
                         const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
  if ( cowin != NULL ) {
    return cowin->request( title, text, buttons, flags );
  } else {
    return req->request( title, text, buttons, flags );
  }
}

int NormalMode::string_request( CopyOpWin *cowin,
                                const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
  if ( cowin != NULL ) {
    return cowin->string_request( title, lines, default_str, buttons, return_str, flags );
  } else {
    return req->string_request( title, lines, default_str, buttons, return_str, flags );
  }
}

int NormalMode::request( DeleteOpWin *dowin,
                         const char *title,
                         const char *text,
                         const char *buttons,
                         Requester::request_flags_t flags )
{
  if ( dowin != NULL ) {
    return dowin->request( title, text, buttons, flags );
  } else {
    return req->request( title, text, buttons, flags );
  }
}

int NormalMode::string_request( DeleteOpWin *dowin,
                                const char *title,
                                const char *lines,
                                const char *default_str,
                                const char *buttons,
                                char **return_str,
                                Requester::request_flags_t flags )
{
  if ( dowin != NULL ) {
    return dowin->string_request( title, lines, default_str, buttons, return_str, flags );
  } else {
    return req->string_request( title, lines, default_str, buttons, return_str, flags );
  }
}

/*
 * tryApplyOwnerPerm will try to set owner and permission
 *   it will apply SUID/SGID only if owner or root (as root only
 *   if chown don't fail)
 * this function don't care if chmod/chown fails
 *
 * returnvalue:
 *   0 okay
 *   -1 wrong args
 *   Bit 0 set: SUID cleared
 *   Bit 1 set: SGID cleared
 *   Bit 2 set: chown failed
 *   Bit 3 set: chmod failed
 */
int NormalMode::tryApplyOwnerPerm( const char *filename,
                                   const uid_t wanted_user,
                                   const gid_t wanted_group,
                                   const mode_t wanted_mode,
                                   struct NM_copyorder *copyorder )
{
  mode_t use_mode;
  int erg = 0, chownres, chmodres;

  if ( filename == NULL ) return -1;
  
  chownres = worker_chown( filename, wanted_user, wanted_group );
  if ( chownres != 0 ) {
    erg |= 1<<2;
  }
  use_mode = wanted_mode;
  if ( ( use_mode & S_ISUID ) != 0 ) {
    // SUID bit set, check if we still want it
    // root will apply it (when chown didn't failed and)
    // and euid/egid == ower
    if ( ! ( ( geteuid() == wanted_user ) ||
             ( ( geteuid() == 0 ) && ( chownres == 0 ) ) ) ) {
      use_mode &= ~S_ISUID;
      erg |= 1<<0;
    }
  }
  if ( ( use_mode & S_ISGID ) != 0 ) {
    // SGID bit set, check if we still want it
    // root will apply it (when chown didn't failed and)
    // and euid/egid == ower
    if ( ! ( ( getegid() == wanted_group ) ||
             ( ( geteuid() == 0 ) && ( chownres == 0 ) ) ) ) {
      use_mode &= ~S_ISGID;
      erg |= 1<<1;
    }
  }
  chmodres = chmod ( filename, use_mode );
  if ( chmodres != 0 ) {
    erg |= 1<<3;
  }
  if ( ( ( erg & 3 ) != 0 ) && ( copyorder != NULL ) ) {
    // removed SUID/SGUID bit, let the user know this
    if ( copyorder->ignoreLosedAttr != true ) {
      int choose;
      char *textstr;
      std::string str2;

      textstr = (char*)_allocsafe( strlen( catalog.getLocale( 536 ) ) + strlen( filename ) + 1 );
      sprintf( textstr, catalog.getLocale( 536 ), filename );
      str2 = catalog.getLocale( 11 );
      str2 += "|";
      str2 += catalog.getLocale( 537 );
      choose = request( copyorder->cowin,
                        catalog.getLocale( 125 ),
                        textstr,
                        str2.c_str(),
                        Requester::REQUEST_CANCELWITHLEFT );
      _freesafe( textstr );
      if ( choose == 1 ) {
        copyorder->ignoreLosedAttr = true;
      }
    }
  }
  return erg;
}

void NormalMode::changeSortModeForField( int field )
{
  int side = parentlister->getSide();
  int realfield;
  const std::vector<WorkerTypes::listcol_t> *sets;
  int used_fields;
  int newsortmode = sortmode;

  sets = wconfig->getVisCols( side );
  if ( sets == NULL ) return;
  
  used_fields = sets->size();
  
  realfield = field / 2;
  if ( ( realfield >= 0 ) && ( realfield < used_fields ) ) {
    switch ( (*sets)[realfield] ) {
      case WorkerTypes::LISTCOL_NAME:
        if ( ( newsortmode & 0xff ) == SORT_NAME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_NAME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_SIZE:
        if ( ( newsortmode & 0xff ) == SORT_SIZE ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_SIZE | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_TYPE:
        if ( ( newsortmode & 0xff ) == SORT_TYPE ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_TYPE | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_OWNER:
        if ( ( newsortmode & 0xff ) == SORT_OWNER ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_OWNER | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_MOD:
        if ( ( newsortmode & 0xff ) == SORT_MODTIME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_MODTIME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_ACC:
        if ( ( newsortmode & 0xff ) == SORT_ACCTIME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_ACCTIME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_CHANGE:
        if ( ( newsortmode & 0xff ) == SORT_CHGTIME ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_CHGTIME | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_INODE:
        if ( ( newsortmode & 0xff ) == SORT_INODE ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_INODE | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_NLINK:
        if ( ( newsortmode & 0xff ) == SORT_NLINK ) {
          // toggle reverse
          newsortmode ^= SORT_REVERSE;
        } else {
          newsortmode = SORT_NLINK | ( newsortmode & ~0xff );
          newsortmode &= ~SORT_REVERSE;
        }
        break;
      case WorkerTypes::LISTCOL_BLOCKS:
        //no sort available
        break;
      case WorkerTypes::LISTCOL_PERM:
        //no sort available
        break;
      case WorkerTypes::LISTCOL_DEST:
        //no sort available
        break;
      default:
        break;
    }
    if ( newsortmode != sortmode ) {
      setSortmode( newsortmode );
    }
  }
}

NormalMode::changemod_info::changemod_info()
{
    forAll = false;
    mode = 0;
    op = CHMOD_SET;
}

NormalMode::changeown_info::changeown_info()
{
  forAll = false;
  newuid = (uid_t)-1;
  newgid = (gid_t)-1;
}

int NormalMode::switchOrder( thread_order_t neworder )
{
  if ( slave.order != neworder ) {
    reclist->lock();
    slave.order = neworder;
    reclist->signal();
    reclist->unlock();
  }
  return 0;
}

int NormalMode::waitForNoOrder( void )
{
  reclist->lock();
  while ( slave.order != THREAD_NOP ) {
    reclist->wait();
  }
  reclist->unlock();
  return 0;
}

int NormalMode::waitForNewOrder( void )
{
  reclist->lock();
  while ( slave.order == THREAD_NOP ) {
    reclist->wait();
  }
  reclist->unlock();
  return 0;
}

int NormalMode::switchStatus( thread_order_t newstatus )
{
  if ( slave.status != newstatus ) {
    slave.statusvar.lock();
    slave.status = newstatus;
    slave.statusvar.signal();
    slave.statusvar.unlock();
  }
  return 0;
}

int NormalMode::waitForStatus( thread_order_t waitstatus )
{
  slave.statusvar.lock();
  while ( slave.status != waitstatus ) {
    slave.statusvar.wait();
  }
  slave.statusvar.unlock();
  return 0;
}

WCFiletype *NormalMode::findFiletype( std::vector<unsigned int> *v )
{
  unsigned int in1, p, tp;
  const std::list<WCFiletype*> *subtype;
  std::list<WCFiletype*>::const_iterator it1;
  WCFiletype *ft;
  
  ft = NULL;

  // check tree depth
  p = v->size();
  if ( p > 0 ) {
    // we need at least the root type
    in1 = (*v)[p - 1];  // get position
    
    ft =  (WCFiletype*)wconfig->getFiletypes()->getElementAt( in1 );
    // root type, no problem if it is NULL
    
    // now go for each tree depth
    p = v->size();
    if ( p > 1 ) {
      // we need one more depth at least
      
      p--; // skip root index
      
      do {
	// now for each depth

	p--; // jump to next level
	// p is correct entry in "v"

	if ( ft == NULL ) break;

	// get position in current tree depth
	in1 = (*v)[p];

	// now find subtype
	// because of std::list we need to iterate
	subtype = ft->getSubTypeList();
	if ( subtype == NULL ) ft = NULL;  // oops, index but no subtype
	else {
	  // iterate through subtype list and find "in1" position
	  for ( it1 = subtype->begin(), tp = 0; it1 != subtype->end(); it1++, tp++ ) {
	    if ( tp == in1 ) break;
	  }
	  
	  if ( it1 != subtype->end() ) {
	    // found subtype at index "in1"
	    ft = *it1;
	  } else {
	    ft = NULL;
	  }
	}
	// stop when we processed last depth
      } while ( p > 0 );
    }
  }
  return ft;
}

int NormalMode::getSelFiles( std::list<NM_specialsourceInt*> *list, nm_getfiles_t selmode, bool unselect )
{
  int added = 0;
  int row;
  NM_specialsourceInt *ss2;
  FileEntry *fe;

  if ( ce == NULL ) return -1;
  if ( ce->dirOpened() == false ) return -1;
  if ( list == NULL ) return -1;

  if ( selmode != NM_GETFILES_ONLYACTIVE ) {
    row = 0;
    for ( Verzeichnis::verz_it fe_it1 = ce->begin();
          fe_it1 != ce->end();
          fe_it1++ ) {
      fe = *fe_it1;
      if ( fe->use == true ) {
	if ( ( strcmp( fe->name, ".." ) != 0 ) && ( fe->select == true ) ) {
	  ss2 = new NM_specialsourceInt( fe );
	  ss2->row = row;
	  ss2->cod = NULL;
	  list->push_back( ss2 );
	  added++;
	  if ( unselect == true ) deselect( fe, ss2->row );
	}
	row++;
      }
    }
  }

  // for ONLYACTIVE always take the active
  // otherwise only if not ONLYSELECT and empty list
  if ( ( selmode == NM_GETFILES_ONLYACTIVE ) ||
       ( ( selmode != NM_GETFILES_ONLYSELECT ) && ( list->size() == 0 ) ) ) {
    if ( ce->getActiveFE() != NULL ) {
      row = lv->getActiveRow();
      if ( lv->isValidRow( row ) == true ) {
        if ( lv->getData( row ) == ce->getActiveFE()->getID() ) {
          if ( strcmp( ce->getActiveFE()->name, ".." ) != 0 ) {
            ss2 = new NM_specialsourceInt( ce->getActiveFE() );
	    ss2->row = row;
	    ss2->cod = NULL;
	    list->push_back( ss2 );
	    added++;
	    if ( unselect == true ) deselect( ce->getActiveFE(), ss2->row );
	  }
	}
      }
    }
  }
  
  return added;
}

void NormalMode::freeSelFiles( std::list<NM_specialsourceExt*> *splist )
{
  std::list<NM_specialsourceExt*>::iterator it1;
  
  if ( splist == NULL ) return;
  for ( it1 = splist->begin(); it1 != splist->end(); it1++ ) {
    if ( *it1 != NULL ) delete *it1;
  }
}

NormalMode::NM_specialsourceInt::NM_specialsourceInt( FileEntry *tfe )
{
  fe = tfe;
  row = -1;
  cod = NULL;
}

NormalMode::NM_specialsourceInt::~NM_specialsourceInt()
{
}

FileEntry *NormalMode::findFE( const FileEntry *searchfe )
{
  FileEntry *fe;

  if ( ce == NULL ) return NULL;
  if ( ce->dirOpened() == false ) return NULL;

  fe = NULL;
  for ( Verzeichnis::verz_it fe_it1 = ce->begin();
        fe_it1 != ce->end();
        fe_it1++ ) {
    fe = *fe_it1;
    if ( fe->equals( searchfe ) == true ) {
      break;
    }
    fe = NULL;
  }

  return fe;
}

void NormalMode::setSSHAllow( enum ssh_allow_t nv )
{
  ssh_allow = nv;
}

NormalMode::Requests::Requests()
{
  std::map<request_t,std::list<request_choose_t> > m;
  std::list<request_choose_t> l;

  choose_strings[OKAY] = "Okay";
  choose_strings[SKIP] = "Skip";
  choose_strings[CANCEL] = "Cancel";

  l.push_back( SKIP );
  l.push_back( CANCEL );
  request_chooses[NO_SPACE_LEFT] = l;

  request_chooses[WRITE_ERROR] = l;
  
  request_chooses[SKIP_CANCEL] = l;
}

NormalMode::Requests::~Requests()
{
}

NormalMode::Requests::request_choose_t NormalMode::Requests::request( request_t reqtype,
								      AWindow *win, 
								      std::string text )
{
  std::list<request_choose_t> &l = request_chooses[reqtype];
  std::list<request_choose_t>::iterator it1;
  std::string buttons, s1;
  request_choose_t erg;
  int c, i;
  
  if ( win == NULL ) return REQUEST_ERROR;

  if ( l.size() > 0 ) {
    for ( it1 = l.begin(); it1 != l.end(); it1++ ) {
      s1 = choose_strings[*it1];
      if ( s1.length() < 1 )
	break;
      if ( it1 != l.begin() )
	buttons += "|";
      buttons += s1;
    }
    if ( it1 == l.end() ) {
      c = win->request( catalog.getLocale( 347 ), text.c_str(), buttons.c_str() );
      erg = REQUEST_ERROR;
      
      i = 0;
      for ( it1 = l.begin(); it1 != l.end(); it1++, i++ ) {
	if ( i == c ) erg = *it1;
      }
      return erg;
    }
  }
  return REQUEST_ERROR;
}

NormalMode::Requests::request_choose_t NormalMode::Requests::request_va( request_t reqtype,
									 AWindow *win,
									 const char *format, ... )
{
  va_list va;
  int n, size = 1024;
  char *tstr;
  std::string string1;
  
  if ( format != NULL ) {
    // first create str
    while ( 1 ) {
      tstr = (char*)_allocsafe( size );
      va_start( va, format );
      n = vsnprintf( tstr, size, format, va );
      va_end( va );
      if ( ( n > -1 ) && ( n < size ) ) break;
      _freesafe( tstr );
      size *= 2;
    }
    string1 = tstr;
    _freesafe( tstr );
    return request( reqtype, win, string1 );
  }
  return REQUEST_ERROR;
}

NormalMode::Requests::request_choose_t NormalMode::Requests::request( request_t reqtype,
								      AWindow *win, 
								      const char *format,
								      const char *str2,
								      int error_number )
{
  const char *myerrstr;
  char *textstr;
  std::string s1;
  
  myerrstr = strerror( error_number );
  textstr = (char*)_allocsafe( strlen( format ) +
			       strlen( str2 ) +
			       strlen( myerrstr ) + 1);
  sprintf( textstr, format, str2, myerrstr );
  s1 = textstr;
  _freesafe( textstr );
  
  return request( reqtype, win, s1 );
}

NormalMode::reg_copy_erg_t NormalMode::copyfile_reg( const char *sourcename,
						     const char *destname,
						     const char *destdirname,
						     const char *destbasename,
						     mode_t create_mode,
						     CopyOpWin *cowin,
						     loff_t bytesToCopy )
{
  int fdw, fdr;
  std::string text, button;
  ssize_t readbytes, writebytes;
  reg_copy_erg_t return_value = REG_COPY_ERROR;
  int erg;
  const char *myerrstr;

  // firstly open source file
  fdr = worker_open( sourcename, O_RDONLY, 0 );
  if ( fdr == -1 ) {
    // can't open inputfile
    if ( cowin != NULL ) cowin->stoptimer();
    
    text = AGUIXUtils::formatStringToString( catalog.getLocale( 281 ), sourcename );
    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
    erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
    
    if ( erg == 1 ) return_value = REG_COPY_CANCEL;
    else return_value = REG_COPY_SKIP;
    
    if ( cowin != NULL ) cowin->conttimer();
    return return_value;
  }

  if ( cowin != NULL ) {
    cowin->set_bytes_to_copy_curfile( bytesToCopy );
  }
  
  // secondly try to open destination file
  // two runs as I first try to open an existing file and only remove it if it doesn't
  // work. This is because we can overwrite a file but not create it
  for ( int i = 0; i < 2; i++ ) {
    // fe is a regular file or a working symlink to a regular file so we can use dmode() without checking
    // for corrupt link
    fdw = worker_open( destname, O_CREAT | O_TRUNC | O_WRONLY, create_mode );

    if ( fdw != -1 ) break;

    // open failed
    if ( cowin != NULL ) cowin->stoptimer();
    if ( errno == ENOSPC ) {
      text = AGUIXUtils::formatStringToString( catalog.getLocale( 545 ), destbasename, destdirname );
      button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
      erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

      if ( erg == 1 ) return_value = REG_COPY_CANCEL;
      else return_value = REG_COPY_SKIP;
    } else if ( errno == EACCES ) {
      if ( Datei::lfileExistsExt( destname ) != Datei::D_FE_NOFILE ) {
	/// error
	// give request
	// textstr="Can't overwrite the file|Shall I try to delete it?";
	// buttonstr="Delete file|Skip|Cancel";
	
	text = AGUIXUtils::formatStringToString( catalog.getLocale( 282 ), destname );
	button = std::string( catalog.getLocale( 277 ) ) +
	  "|" +
	  catalog.getLocale( 225 ) + 
	  "|" +
	  catalog.getLocale( 8 );
	erg = request( cowin, catalog.getLocale( 123 ), text.c_str(), button.c_str() );

	if ( erg == 1 ) return_value = REG_COPY_SKIP;
	else if ( erg == 2 ) return_value = REG_COPY_CANCEL;
	else {
	  if ( worker_remove( destname ) != 0 ) {
	    // textstr="Failed to remove this file|I will skip this file!";
	    // buttonstr="Ok|Cancel";
	    
	    text = AGUIXUtils::formatStringToString( catalog.getLocale( 138 ), destname );
	    button = std::string( catalog.getLocale( 11 ) ) + "|" + catalog.getLocale( 8 );
	    erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

	    if ( erg == 1 ) return_value = REG_COPY_CANCEL;
	    else return_value = REG_COPY_SKIP;
	  }
	}
      } else {
	// "Can't open dest file"
	text = AGUIXUtils::formatStringToString( catalog.getLocale( 198 ), destname );
	button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
	erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

	if ( erg == 1 ) return_value = REG_COPY_CANCEL;
	else return_value = REG_COPY_SKIP;
      }
    } else {
      text = AGUIXUtils::formatStringToString( catalog.getLocale( 198 ), destname );
      button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
      erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

      if ( erg == 1 ) return_value = REG_COPY_CANCEL;
      else return_value = REG_COPY_SKIP;
    }
    if ( cowin != NULL ) cowin->conttimer();

    if ( ( return_value == REG_COPY_SKIP ) || ( return_value == REG_COPY_CANCEL ) ) break;
  }

  if ( fdw == -1 ) {
    worker_close( fdr );
    return return_value;
  }

  // now copy from fdr to fdw
  do {
    readbytes = worker_read( fdr, buf, BUFSIZE );
    if ( readbytes < 0 ) {
      // error while reading
      
      if ( cowin != NULL ) cowin->stoptimer();
      
      myerrstr = strerror( errno );
      text = AGUIXUtils::formatStringToString( catalog.getLocale( 525 ), sourcename, myerrstr );
      button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
      erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );

      if ( cowin != NULL ) cowin->conttimer();

      if ( erg == 0 ) return_value = REG_COPY_SKIP;
      else return_value = REG_COPY_CANCEL;
      break;
    } else if ( readbytes == 0 ) {
      // end of file
      break;
    }

    writebytes = worker_write( fdw, buf, readbytes );
    if ( writebytes == readbytes ) {
      if ( cowin != NULL ) {
	cowin->add_curbytes_copied( writebytes );

        //TODO if I use dynamic buffer I should skip some updates
        if ( cowin->redraw() != 0 )
          return_value = REG_COPY_CANCEL;
      }
    } else {
      // something went wrong
      // let the user choose to cancel or skip this file
      // Ask to delete the incomplete destfile
      // Attention: This method also moves files so be sure to
      //   NOT delete the source!
      if( ( ( writebytes >= 0 ) && ( writebytes < readbytes ) ) ||
	  ( errno == ENOSPC ) ) {
	// ENOSPC isn't always reported so assume ENOSPC
	text = AGUIXUtils::formatStringToString( catalog.getLocale( 545 ), destbasename, destdirname );
	button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
	erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
	
	if ( erg == 0 ) return_value = REG_COPY_SKIP;
	else return_value = REG_COPY_CANCEL;
      } else {
	if ( writebytes < 0 ) {
	  myerrstr = strerror( errno );
	  text = AGUIXUtils::formatStringToString( catalog.getLocale( 526 ), destname, myerrstr );
	} else {
	  text = AGUIXUtils::formatStringToString( catalog.getLocale( 359 ), destname );
	}
	button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
	erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
	
	if ( erg == 0 ) return_value = REG_COPY_SKIP;
	else return_value = REG_COPY_CANCEL;
      }
    }
  } while ( ( return_value != REG_COPY_CANCEL ) && ( return_value != REG_COPY_SKIP ) );

  cowin->update_file_status();
  if ( cowin->redraw() != 0 ) return_value = REG_COPY_CANCEL;

  worker_close( fdr );
  if ( worker_close( fdw ) != 0 ) {
    // oops, close error
    myerrstr = strerror( errno );
    text = AGUIXUtils::formatStringToString( catalog.getLocale( 526 ), destname, myerrstr );
    button = std::string( catalog.getLocale( 225 ) ) + "|" + catalog.getLocale( 8 );
    erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
    
    if ( erg == 0 ) return_value = REG_COPY_SKIP;
    else return_value = REG_COPY_CANCEL;
  }
  
  if ( ( return_value == REG_COPY_CANCEL ) || ( return_value == REG_COPY_SKIP ) ) {
    // error while copying!
    // ask to remove the incomplete destination file
    
    if ( cowin != NULL ) cowin->stoptimer();
    
    text = AGUIXUtils::formatStringToString( catalog.getLocale( 393 ), destname );
    button = std::string( catalog.getLocale( 394 ) ) + "|" + catalog.getLocale( 395 );
    erg = request( cowin, catalog.getLocale( 123 ), text.c_str(), button.c_str() );
    if ( erg == 1 ) {
      if ( worker_unlink( destname ) != 0 ) {
	// textstr="Failed to remove the destination file|This file is probably incomplete!";
	text = AGUIXUtils::formatStringToString( catalog.getLocale( 139 ), destname );
	button = catalog.getLocale( 11 );
	erg = request( cowin, catalog.getLocale( 347 ), text.c_str(), button.c_str() );
      }
    }
    if ( cowin != NULL ) cowin->conttimer();
  } else {
    // copy complete
    return_value = REG_COPY_OK;
  }
  return return_value;
}

bool NormalMode::checkForEmptyActionList( WCFiletype *ft )
{
  List *l;
  bool res = true;

  while ( ft != NULL ) {
    l = ft->getDoubleClickActions();
    //TODO ParentOp pruefen, da so trotz Aktion leere Liste vorkommen kann
    if ( ( l != NULL ) && ( l->size() > 0 ) ) {
      res = false;
      break;
    }
    ft = ft->getParentType();
  }
  return res;
}

int NormalMode::getAndEraseLoadedFieldWidth( std::string str1 )
{
  int tw = -2;
  if ( _loaded_field_width.count( str1 ) > 0 ) {
    tw = _loaded_field_width[str1];
    _loaded_field_width.erase( str1 );
  }
  return tw;
}

int NormalMode::writeFieldWidthsToLoadMap()
{
  std::map<std::string,int>::iterator it1;
  int tw;
  std::string str1;

  if ( lv != NULL ) {
    for ( it1 = _field_map.begin(); it1 != _field_map.end(); it1++ ) {
      tw = lv->getFieldWidth( (*it1).second );
      _loaded_field_width[(*it1).first] = tw;
    }
  }
  return 0;
}

void NormalMode::setQuicksearchEnabled( bool nv )
{
  quicksearch_enabled = nv;
}

int NormalMode::enterPath( const std::string &fullname )
{
  NWC::FSEntry e( fullname );
  int erg = 0;

  if ( e.entryExists() == true ) {
    if ( e.isDir( true ) == true ) {
      erg = enterDir( e.getFullname().c_str() );
    } else {
      std::string dirname = e.getDirname();
      erg = enterDir( dirname.c_str() );
      
      std::string basename = e.getBasename();
      activateEntry( basename );
    }
  }
  return erg;
}

void NormalMode::makeRowActive( int row )
{
  if ( ce == NULL || lv == NULL )
    return;
  
  if ( row < 1 )
      row = 0;

  if ( lv->isValidRow( row ) == false )
      return;

  ce->setActiveFE( getFEForRow( row ) );
  makeListViewRowActive( row );
}

/*
 * does lv->setActiveRow if lister is active
 */
void NormalMode::makeListViewRowActive( int row )
{
    if ( parentlister->isActive() == true )
        lv->setActiveRow( row );
}

FileEntry *NormalMode::getFEForRow( int row )
{
    if ( ce == NULL ) return NULL;

    if ( ce->getSerialOfIDs() != _lv_ids_serial ) {
        request( catalog.getLocale( 347 ),
                 "Some list view IDs don't match the directory!|That's not supposed to happen and I will abort the program.|Please report this error the author!",
                 "Ok" );
        abort();
    }

    if ( lv->isValidRow( row ) == false )
        return NULL;
    
    int id = lv->getData( row );
    return ce->getEntryByID( id );
}

void NormalMode::setEntrySearchString( const std::string &str, bool update_lv )
{
    _entrysearch_matcher.setMatchString( str );
    updateDirFilter();

    if ( update_lv == true && lv != NULL ) {
        bool active_visible = lv->isRowVisible( lv->getActiveRow() );

        rebuildView();

        if ( lv->isRowVisible( lv->getActiveRow() ) == false &&
             active_visible == true ) {
            lv->centerActive();
        }
    }
}

std::string NormalMode::getEntrySearchString()
{
    return _entrysearch_matcher.getMatchString();
}

void NormalMode::setEntrySearchCaseSensitive( bool nv )
{
    _entrysearch_matcher.setMatchCaseSensitive( nv );
}

void NormalMode::updateDirFilter()
{
    if ( _entrysearch_matcher.getMatchString().empty() == true ||
         _entrysearch_matcher.getMatchString() == "*" ||
         searchmodeon == false ||
         _filtered_search_enabled == false ) {
        _dir_filter_sets.setStringFilter( std::auto_ptr<StringMatcher>() );
        m_filtered_search_active = false;
    } else {
        _dir_filter_sets.setStringFilter( std::auto_ptr<StringMatcher>( new StringMatcherFNMatch( _entrysearch_matcher ) ) );
        m_filtered_search_active = true;
    }
}

bool NormalMode::filteredSearchActive() const
{
    return m_filtered_search_active;
}

void NormalMode::setFilteredSearchEnabled( bool nv )
{
    _filtered_search_enabled = nv;
}

void NormalMode::activateEntry( const std::string &name )
{
  FileEntry *fe;
  int row;
  
  if ( ce == NULL ) return;
  if ( ce->dirOpened() == false ) return;

  for ( Verzeichnis::verz_it fe_it1 = ce->begin();
        fe_it1 != ce->end();
        fe_it1++ ) {
      fe = *fe_it1;
      if ( fe->use == true ) {
          if ( strcmp( fe->name, name.c_str() ) == 0 ) {
              // hit
              ce->setActiveFE( fe );
              row = 0;
              while(lv->isValidRow(row)==true) {
                  if ( lv->getData( row ) == fe->getID() ) {
                      makeListViewRowActive( row );
                      break;
                  }
                  row++;
              }
              lv->showActive();
              lv->redraw();
              break;
          }
      }
  }
  
  return;
}

int NormalMode::buildSpecialSourceIntList( const std::list<NM_specialsourceExt*> &inlist,
                                           std::list<NM_specialsourceInt*> &outlist )
{
    std::list<NM_specialsourceExt*>::const_iterator ite1;
    int count = 0;
    
    for ( ite1 = inlist.begin();
          ite1 != inlist.end();
          ite1++ ) {
        NM_specialsourceExt *sse = *ite1;
        
        // skip entry ..
        if ( strcmp( sse->entry()->name, ".." ) == 0 )
            continue;

        // search for internal FileEntry matching entry from external source
        // remember, external FileEntry is a copy
        FileEntry *fe = findFE( sse->entry() );

        if ( fe != NULL ) {
            // valid entry found
            NM_specialsourceInt *ss2 = new NM_specialsourceInt( fe );
            ss2->cod = NULL;
            outlist.push_back( ss2 );
            count++;
        }
    }

    return count;
}

void NormalMode::deactivateFilteredSearch()
{
    finishsearchmode();
    if ( filteredSearchActive() == true ) {
        setEntrySearchString( "" );
    }
}

void NormalMode::buildContextPopUpMenu( FileEntry *entry )
{
    freePopUpMenus();

    int descr_id = 0;

    if ( entry == NULL ) return;
    if ( entry->isCorrupt == true ) return;
    if ( strcmp( entry->name, ".." ) == 0 ) return;

    std::list<PopUpMenu::PopUpEntry> m1;
    PopUpMenu::PopUpEntry e1;

    m_current_popup_settings.reset();

    e1.type = PopUpMenu::SUBMENU;
    e1.name = catalog.getLocale( 833 );
    std::list<PopUpMenu::PopUpEntry> labelmenu = buildLabelPopUpData( entry );
    e1.submenu = &labelmenu;
    e1.id = descr_id++;

    m_current_popup_settings.label_menu_id = e1.id;

    m1.push_back( e1 );
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );

    e1.name = catalog.getLocale( 768 );
    e1.type = PopUpMenu::HEADER;
    m1.push_back( e1 );
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );
    
    m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new DoubleClickAction::DoubleClickActionDescr() );

    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 252 );
    e1.id = descr_id++;
    m1.push_back( e1 );
    
    WCFiletype *ft;
    ft = entry->filetype;
    if ( ft == NULL ) {
        if ( entry->isDir() == true ) {
            ft = wconfig->getdirtype();
        } else {
            ft = wconfig->getnotyettype();
        }
    }

    List *s_l = ft->getActionList( true, ShowAction::ShowActionDescr() );
    if ( s_l != NULL && s_l->size() > 0 ) {
        m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new ShowAction::ShowActionDescr() );

        e1.type = PopUpMenu::NORMAL;
        e1.name = catalog.getLocale( 253 );
        e1.id = descr_id++;
        m1.push_back( e1 );
    }

    List *rs_l = ft->getActionList( true, RawShowAction::RawShowActionDescr() );
    if ( rs_l != NULL && rs_l->size() > 0 ) {
        m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new RawShowAction::RawShowActionDescr() );

        e1.type = PopUpMenu::NORMAL;
        e1.name = catalog.getLocale( 254 );
        e1.id = descr_id++;
        m1.push_back( e1 );
    }

    for ( int i = 0; i < 10; i++ ) {
        List *u_l = ft->getActionList( true, UserAction::UserActionDescr( i ) );
        if ( u_l != NULL && u_l->size() > 0 ) {
            m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new UserAction::UserActionDescr( i ) );

            e1.type = PopUpMenu::NORMAL;
            //TODO not nice using index for locale but works
            e1.name = catalog.getLocale( 255 + i );
            e1.id = descr_id++;
            m1.push_back( e1 );
        }
    }

    std::list<std::string> names;

    ft->fillNamesOfCustomActions( true, names );
    for ( std::list<std::string>::const_iterator it1 = names.begin();
          it1 != names.end();
          it1++ ) {
        m_current_popup_settings.popup_descr[descr_id] = RefCount<ActionDescr>( new RunCustomAction::RunCustomActionDescr( *it1 ) );

        e1.type = PopUpMenu::NORMAL;
        //e1.name = AGUIXUtils::formatStringToString( catalog.getLocale( 779 ), it1->c_str() );
        e1.name = it1->c_str();
        e1.id = descr_id++;
        m1.push_back( e1 );
    }

    //TODO build submenu for all selected entries

    m_current_popup_settings.lv_popup_menu = new PopUpMenu( aguix, m1 );
    m_current_popup_settings.lv_popup_menu->create();
}

void NormalMode::lv_pressed( int row, bool open_under_mouse )
{
    if ( ce == NULL || lv == NULL ) return;
    
    if ( lv->isValidRow( row ) == false ) return;
    makeRowActive( row );
    
    FileEntry *fe = getFEForRow( row );
    if ( fe != NULL ) {
        buildContextPopUpMenu( fe );
        m_current_popup_settings.entry_for_popup = std::auto_ptr<FileEntry>( new FileEntry( *fe ) );
        if ( m_current_popup_settings.lv_popup_menu != NULL ) {
            if ( open_under_mouse == true ) {
                m_current_popup_settings.lv_popup_menu->show();
            } else {
                int tx, ty;
                
                aguix->getWidgetRootPosition( lv, &tx, &ty );
                tx += lv->getWidth() / 2;
                ty += lv->getHeight() / 2;
                m_current_popup_settings.lv_popup_menu->show( tx, ty, PopUpMenu::POPUP_CENTER_IN_POSITION );
                //TODO better query the position of the active row from LV and use
                // this position
            }
        }
    }
}

void NormalMode::startLVPopUpAction( AGMessage *msg )
{
    if ( msg == NULL || ce == NULL || lv == NULL ) return;
    
    FileEntry *fe = getFEForRow( lv->getActiveRow() );
    if ( fe == NULL ) return;

    if ( m_current_popup_settings.entry_for_popup.get() == NULL ) return;
    if ( m_current_popup_settings.entry_for_popup->equals( *fe ) == false ) return;
    // ok, still the same entry
    
    int id = msg->popupmenu.clicked_entry_id;
    
    if ( m_current_popup_settings.popup_descr.count( id ) > 0 ) {
        RefCount<ActionDescr> action_descr = m_current_popup_settings.popup_descr[id];

        if ( action_descr.getVal() != NULL ) {
            if ( dynamic_cast<DoubleClickAction::DoubleClickActionDescr*>( action_descr.getVal() ) != NULL ) {
                startAction( fe );
            } else {

                ActionMessage amsg( parentlister->getWorker() );

                amsg.mode=amsg.AM_MODE_SPECIAL;

                amsg.setFE( fe );

                WCFiletype *ft;
                ft = fe->filetype;
                if ( ft == NULL ) {
                    if ( fe->isDir() == true ) {
                        ft = wconfig->getdirtype();
                    } else {
                        ft = wconfig->getnotyettype();
                    }
                }

                amsg.filetype = ft;

                List *action_list = NULL;

                amsg.m_action_descr = action_descr;
                action_list = ft->getActionList( true, *( action_descr ) );

                if ( action_list != NULL ) {
                    parentlister->getWorker()->interpret( action_list, &amsg );
                }
            }
        }
    } else {
        // no action so check other menu options
        if ( msg->popupmenu.recursive_ids != NULL &&
             msg->popupmenu.recursive_ids->empty() == false ) {
            if ( msg->popupmenu.recursive_ids->back() == m_current_popup_settings.label_menu_id ) {
                std::list<int> entry_ids = *msg->popupmenu.recursive_ids;
                entry_ids.pop_back();
                handleLabelPopUp( fe, entry_ids );
            }
        }
    }
}

void NormalMode::openContextMenu()
{
    if ( ce == NULL || lv == NULL ) return;

    lv_pressed( lv->getActiveRow(), false );
}

void NormalMode::bookmarksChanged()
{
    m_bookmarks_has_been_changed = true;
}

void NormalMode::updateOnBookmarkChange()
{
    if ( m_bookmarks_has_been_changed == true ) {
        _dir_filter_sets.bookmarksChanged();
        m_dir_bookmarks_sets.bookmarksChanged();
        
        //TODO rebuildView is enough currently but update would be more
        // complete
        //update( false, true );
        rebuildView();
        
        m_bookmarks_has_been_changed = false;
    }
}


std::list<PopUpMenu::PopUpEntry> NormalMode::buildLabelPopUpData( FileEntry *entry )
{
    int descr_id = 0;
    std::list<PopUpMenu::PopUpEntry> m1;

    if ( entry == NULL ) return m1;
    if ( entry->isCorrupt == true ) return m1;
    if ( strcmp( entry->name, ".." ) == 0 ) return m1;

    std::auto_ptr<BookmarkDBEntry> db_entry = Worker::getBookmarkDBInstance().getEntry( entry->fullname );

    PopUpMenu::PopUpEntry e1;
    e1.name = catalog.getLocale( 834 );
    if ( db_entry.get() != NULL ) {
        e1.type = PopUpMenu::NORMAL;
    } else {
        e1.type = PopUpMenu::GREYED_OUT;
    }
    e1.id = descr_id++;

    m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::REMOVE_LABEL, "" );

    m1.push_back( e1 );

    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );
    
    std::list<std::string> cats = Worker::getBookmarkDBInstance().getCats();
    const std::map<std::string, WConfig::ColorDef::label_colors_t> labels = wconfig->getColorDefs().getLabelColors();
    std::map<std::string, WConfig::ColorDef::label_colors_t>::const_iterator label_it;

    for ( label_it = labels.begin(); label_it != labels.end(); ++label_it ) {
        if ( label_it->first == "" ) continue;

        if ( db_entry.get() == NULL ||
             db_entry->getCategory() != label_it->first ) {
            e1.type = PopUpMenu::NORMAL;
        } else {
            e1.type = PopUpMenu::GREYED_OUT;
        }
        e1.name = label_it->first;
        e1.id = descr_id++;

        e1.fg = label_it->second.normal_fg;
        e1.bg = label_it->second.normal_bg;

        m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_LABEL, label_it->first );

        m1.push_back( e1 );
    }
    e1.fg = e1.bg = -1;

    bool first = true;
    for ( std::list<std::string>::iterator it1 = cats.begin();
          it1 != cats.end();
          ++it1 ) {
        if ( *it1 == "" ) continue;

        if ( labels.find( *it1 ) == labels.end() ) {

            if ( first == true ) {
                e1.type = PopUpMenu::HLINE;
                m1.push_back( e1 );
                first = false;
            }

            if ( db_entry.get() == NULL ||
                 db_entry->getCategory() != *it1 ) {
                e1.type = PopUpMenu::NORMAL;
            } else {
                e1.type = PopUpMenu::GREYED_OUT;
            }
            e1.name = *it1;
            e1.id = descr_id++;

            m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_LABEL, *it1 );

            m1.push_back( e1 );
        }
    }

    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );
    
    if ( db_entry.get() == NULL ||
         db_entry->getCategory() != "" ) {
        e1.type = PopUpMenu::NORMAL;
    } else {
        e1.type = PopUpMenu::GREYED_OUT;
    }
    e1.name = catalog.getLocale( 847 );
    e1.id = descr_id++;
    
    m_current_popup_settings.label_id_to_action[e1.id] = label_popup_table_t( label_popup_table_t::ADD_LABEL, "" );
    
    m1.push_back( e1 );
    
    return m1;
}

void NormalMode::handleLabelPopUp( AGMessage *msg )
{
    if ( msg == NULL || ce == NULL || lv == NULL ) return;

    if ( msg->popupmenu.menu != m_current_popup_settings.label_popup_menu.get() ) return;
    
    FileEntry *fe = getFEForRow( lv->getActiveRow() );
    if ( fe == NULL ) return;

    if ( m_current_popup_settings.entry_for_popup.get() == NULL ) return;
    if ( m_current_popup_settings.entry_for_popup->equals( *fe ) == false ) return;
    // ok, still the same entry
    
    std::list<int> entry_ids = *msg->popupmenu.recursive_ids;
    handleLabelPopUp( fe, entry_ids );
}

void NormalMode::handleLabelPopUp( FileEntry *fe, const std::list<int> &entry_ids )
{
    if ( entry_ids.empty() == true ) return;
    if ( fe == NULL ) return;

    int id = entry_ids.back();
    
    std::map<int, label_popup_table_t>::const_iterator it1 = m_current_popup_settings.label_id_to_action.find( id );
    if ( it1 != m_current_popup_settings.label_id_to_action.end() ) {
        // found entry

        BookmarkDBProxy &bookmarks = Worker::getBookmarkDBInstance();
        //TODO read necessary?
        bookmarks.read();

        std::auto_ptr<BookmarkDBEntry> entry = bookmarks.getEntry( fe->fullname );

        if ( it1->second.type == label_popup_table_t::REMOVE_LABEL ) {
            if ( entry.get() != NULL ) {
                bookmarks.delEntry( *entry );
                bookmarks.write();
            }
        } else if ( it1->second.type == label_popup_table_t::ADD_LABEL ) {
            if ( entry.get() != NULL ) {
                BookmarkDBEntry newentry( *entry );
                newentry.setCategory( it1->second.label );
                bookmarks.updateEntry( *entry, newentry );
            } else {
                BookmarkDBEntry newentry( it1->second.label, fe->fullname, "", true );
                bookmarks.addEntry( newentry );
            }
            bookmarks.write();
        }
        
        updateOnBookmarkChange();
    }
}

void NormalMode::buildLabelPopUpMenu( FileEntry *entry )
{
    freePopUpMenus();

    if ( entry == NULL ) return;
    if ( entry->isCorrupt == true ) return;
    if ( strcmp( entry->name, ".." ) == 0 ) return;

    m_current_popup_settings.reset();

    std::list<PopUpMenu::PopUpEntry> labelmenu = buildLabelPopUpData( entry );

    m_current_popup_settings.label_menu_id = -2;

    m_current_popup_settings.label_popup_menu = std::auto_ptr<PopUpMenu>( new PopUpMenu( aguix, labelmenu ) );
    m_current_popup_settings.label_popup_menu->create();
}

void NormalMode::openLabelPopUp()
{
    if ( ce == NULL || lv == NULL ) return;

    int row = lv->getActiveRow();

    if ( lv->isValidRow( row ) == false ) return;
    
    FileEntry *fe = getFEForRow( row );
    if ( fe != NULL ) {
        buildLabelPopUpMenu( fe );
        m_current_popup_settings.entry_for_popup = std::auto_ptr<FileEntry>( new FileEntry( *fe ) );
        if ( m_current_popup_settings.label_popup_menu.get() != NULL ) {
            int tx, ty;
            
            aguix->getWidgetRootPosition( lv, &tx, &ty );
            tx += lv->getWidth() / 2;
            ty += lv->getHeight() / 2;
            m_current_popup_settings.label_popup_menu->show( tx, ty, PopUpMenu::POPUP_CENTER_IN_POSITION );
            //TODO better query the position of the active row from LV and use
            // this position
        }
    }
}

void NormalMode::freePopUpMenus()
{
    m_current_popup_settings.reset();
}

void NormalMode::setHighlightBookmarkPrefix( bool nv )
{
    _dir_filter_sets.setHighlightBookmarkPrefix( nv );
    rebuildView();
}

bool NormalMode::getHighlightBookmarkPrefix() const
{
    return _dir_filter_sets.getHighlightBookmarkPrefix();
}

void NormalMode::setBookmarkFilter( DirFilterSettings::bookmark_filter_t v )
{
    finishsearchmode();
  
    _dir_filter_sets.setBookmarkFilter( v );
    rebuildView();
    setName();
    if ( lv != NULL ) lv->showActive();
}

DirFilterSettings::bookmark_filter_t NormalMode::getBookmarkFilter() const
{
    return _dir_filter_sets.getBookmarkFilter();
}

void NormalMode::setSpecificBookmarkLabel( const std::string &l )
{
    finishsearchmode();
    
    _dir_filter_sets.setSpecificBookmarkLabel( l );
    rebuildView();
    setName();
    if ( lv != NULL ) lv->showActive();
}

const std::string &NormalMode::getSpecificBookmarkLabel() const
{
    return _dir_filter_sets.getSpecificBookmarkLabel();
}

NormalMode::nm_filter_t NormalMode::checkFilter( const char *filter ) throw( int )
{
    NM_Filter::check_t newmode = NM_Filter::INACTIVE;
    try {
        newmode = _dir_filter_sets.checkFilter( filter );
    } catch ( int i ) {
        throw i;
    }

    switch ( newmode ) {
        case NM_Filter::EXCLUDE:
            return NM_FILTER_EXCLUDE;
            break;
        case NM_Filter::INCLUDE:
            return NM_FILTER_INCLUDE;
            break;
        default:
            break;
    }
    return NM_FILTER_UNSET;
}

NormalMode::PopUpSettings::PopUpSettings() : label_menu_id( -1 ),
                                             lv_popup_menu( NULL )
{}

void NormalMode::PopUpSettings::reset()
{
    label_menu_id = -1;
    label_id_to_action.clear();
    label_popup_menu.reset();
    popup_descr.clear();
    entry_for_popup.reset();
    if ( lv_popup_menu != NULL ) {
        delete lv_popup_menu;
        lv_popup_menu = NULL;
    }
}

int NormalMode::queryLVDimensions( int &tx, int &ty,
                                   int &tw, int &th )
{
    if ( lv != NULL ) {
        aguix->getWidgetRootPosition( lv, &tx, &ty );
        tw = lv->getWidth();
        th = lv->getHeight();
        return 0;
    }
    return 1;
}

void NormalMode::updateTabs()
{
    if ( m_tab_b == NULL ) return;

    if ( lv != NULL && ce != NULL ) {
        setTabOption( m_tab_b->getSelectedOption(), ce->getDir() );
    } else {
        setTabOption( m_tab_b->getSelectedOption(), "" );
    }
}

void NormalMode::newTab()
{
    int tab = -1;

    if ( m_tab_b == NULL ) return;

    if ( ce != NULL ) {
        tab = addTabOption( ce->getDir() );
    } else {
        tab = addTabOption( "" );
    }
    if ( tab >= 0 ) {
        m_tab_b->setOption( tab );
    }
    m_tab_b->redraw();
}

void NormalMode::closeCurrentTab()
{
    if ( m_tab_b == NULL ) return;

    if ( m_tab_b->getNrOfOptions() < 2 ) return;

    int tab = m_tab_b->getSelectedOption();

    removeTabOption( tab );

    tab = m_tab_b->getSelectedOption();
    switchToTab( tab );
}

void NormalMode::switchToTab( int tab )
{
    if ( m_tab_b == NULL ) return;

    const char *dirstr = m_tab_entries.getEntry( tab ).c_str();

    if ( dirstr != NULL ) {
        m_tab_b->setOption( tab );
        if ( strlen( dirstr ) > 0 ) {

            char *dir = dupstring( dirstr );
            for(;;) {
                int e = enterDir( dir );
                if ( e == 0 ) break;  // read ok, so finish
                if ( strcmp( dir, "/" ) == 0 ) break;  // there is no dir we can display anymore, so finish
                char *tstr = ParentDir( dir, NULL );
                _freesafe( dir );
                dir = tstr;
            }
            _freesafe( dir );
        }
    }
}

void NormalMode::switchToTabDirection( int direction )
{
    if ( m_tab_b == NULL ) return;

    int tab = m_tab_b->getSelectedOption() + ( direction % m_tab_b->getNrOfOptions() );
    tab += m_tab_b->getNrOfOptions();
    tab %= m_tab_b->getNrOfOptions();
    switchToTab( tab );
}

NMCacheEntry *NormalMode::findOldestCacheEntryNotInTab()
{
    int tabs = m_tab_b->getNrOfOptions();
    NMCacheEntry *tce = (NMCacheEntry*)cache->getFirstElement();
    while ( tce != NULL ) {
        int tab = -1;
        for ( int i = 0; i < tabs; i++ ) {
            const char *dirstr = m_tab_entries.getEntry( i ).c_str();
            if ( dirstr != NULL &&
                 strcmp( tce->getDir(), dirstr ) == 0 ) {
                tab = i;
            }
        }
        if ( tab == -1 ) break;
        tce = (NMCacheEntry*)cache->getNextElement();
    }

    return tce;
}

void NormalMode::setTabOption( int pos, const std::string &path )
{
    if ( pos < 0 || pos > m_tab_b->getNrOfOptions() ) return;

    if ( pos == m_tab_b->getNrOfOptions() ) {
        addTabOption( path );
    } else {
        if ( path.length() < 2 ) {
            m_tab_b->setOption( pos, path.c_str() );
        } else {
            m_tab_b->setOption( pos, NWC::Path::basename( path ).c_str() );
        }
        m_tab_entries.updateEntry( pos, path );
    }
}

int NormalMode::addTabOption( const std::string &path )
{
    m_tab_entries.insertEntry( path );
    if ( path.length() < 2 ) {
        return m_tab_b->addOption( path.c_str() );
    } else {
        return m_tab_b->addOption( NWC::Path::basename( path ).c_str() );
    }
}

void NormalMode::removeTabOption( int pos )
{
    if ( pos < 0 || pos >= m_tab_b->getNrOfOptions() ) return;

    m_tab_entries.removeEntry( pos );
    m_tab_b->removeOption( pos );
}
