/* SPDX-License-Identifier: MIT
 *
 * Copyright (C) 2015-2018 Jason A. Donenfeld <Jason@zx2c4.com>. All Rights Reserved.
 */

#ifdef DEBUG

enum { MAXIMUM_TEST_BUFFER_LEN = 800 };

struct chacha20_testvec {
	u8 key[CHACHA20_KEY_SIZE];
	u64 nonce;
	u8 input[MAXIMUM_TEST_BUFFER_LEN];
	u8 result[MAXIMUM_TEST_BUFFER_LEN];
	size_t ilen;
};

/*
 * #!/usr/bin/env python3
 *
 * import chacha20
 * import os
 * import struct
 *
 * def encode_blob(blob):
 * 	a = ""
 * 	x = 0
 * 	for i in blob:
 * 		a += ('0x%02x' % i) + ","
 * 		x += 1
 * 		if x % 8 == 0:
 * 			a += "\n\t\t    "
 * 		else:
 * 			a += " "
 * 	if x % 8 == 0:
 * 		return a[:len(a) - 8]
 * 	return a[:len(a) - 2]
 *
 * enc = [ ]
 * dec = [ ]
 *
 * def make_vector(plen):
 * 	key = os.urandom(32)
 * 	nonce = os.urandom(8)
 * 	p = os.urandom(plen)
 * 	c = chacha20.chacha20_encrypt(p, key, nonce)
 *
 * 	out = "{\n"
 * 	out += "\t.key\t= { " + encode_blob(key) + " },\n"
 * 	out += "\t.nonce\t= " + hex(struct.unpack("<Q", nonce)[0]) + "ULL,\n"
 * 	out += "\t.input\t= { " + encode_blob(p) + " },\n"
 * 	out += "\t.ilen\t= " + str(len(p)) + ",\n"
 * 	out += "\t.result\t= { " + encode_blob(c) + " }\n"
 * 	out += "}"
 * 	enc.append(out)
 *
 * block_size = 64
 *
 * make_vector(1)
 * make_vector(63)
 * make_vector(64)
 * make_vector(127)
 * make_vector(250)
 * make_vector(256)
 * make_vector(280)
 * make_vector(511)
 * make_vector(512)
 * make_vector(513)
 * make_vector(514)
 * make_vector(769)
 *
 * print(", ".join(enc))
 */

static const struct chacha20_testvec
chacha20_vectors[] __initconst = {{
	.key	= { 0x8e, 0xee, 0x50, 0x40, 0xb2, 0x76, 0x80, 0xeb,
		    0xd8, 0x7e, 0xf2, 0x5a, 0xcf, 0x25, 0xd6, 0x6c,
		    0x9e, 0xc8, 0x3d, 0x0c, 0xf7, 0x05, 0x00, 0x40,
		    0xa0, 0x75, 0x66, 0x86, 0xdf, 0x8d, 0xac, 0x77 },
	.nonce	= 0x37d4e0484f294097ULL,
	.input	= { 0xe1 },
	.ilen	= 1,
	.result	= { 0xf1 }
}, {
	.key	= { 0x23, 0xe2, 0x3b, 0x3a, 0x22, 0xd2, 0xfb, 0x69,
		    0x92, 0xcc, 0x0a, 0xda, 0x48, 0x21, 0x33, 0x8f,
		    0x57, 0x58, 0x08, 0x17, 0xca, 0xe5, 0x56, 0x08,
		    0x71, 0x86, 0x91, 0xc7, 0xae, 0x64, 0x4d, 0xfd },
	.nonce	= 0xe0c627c7e1a4b012ULL,
	.input	= { 0x83, 0x95, 0x1c, 0xc6, 0x8a, 0x53, 0x92, 0x6a,
		    0x3f, 0x1e, 0x67, 0x17, 0x0b, 0x2b, 0x45, 0x04,
		    0xaf, 0x5a, 0x75, 0x59, 0xb7, 0xab, 0x80, 0xde,
		    0xa4, 0x8f, 0xda, 0x03, 0xfd, 0xda, 0xe1, 0x7e,
		    0x75, 0x99, 0x44, 0x6f, 0xea, 0x75, 0x26, 0x68,
		    0xcc, 0x20, 0xe0, 0x7c, 0xf1, 0x09, 0x31, 0x44,
		    0xba, 0xb9, 0xe8, 0x7f, 0xce, 0x5e, 0x89, 0x6d,
		    0x0d, 0x4d, 0x20, 0x93, 0x52, 0x91, 0x85 },
	.ilen	= 63,
	.result	= { 0xa1, 0x32, 0xec, 0xab, 0x86, 0x35, 0x99, 0x88,
		    0xc0, 0xef, 0xae, 0x69, 0x64, 0xa3, 0xba, 0xcb,
		    0xb9, 0x44, 0x24, 0x92, 0x9f, 0x33, 0xdb, 0xc0,
		    0x7f, 0xe5, 0x39, 0xa8, 0x57, 0x2a, 0x29, 0x5f,
		    0xdc, 0xc4, 0xee, 0x7f, 0xcd, 0xd9, 0x5f, 0x0b,
		    0x49, 0xc2, 0x9a, 0xa2, 0x5e, 0x5a, 0x37, 0x3d,
		    0x7a, 0x1e, 0x66, 0xc6, 0xd1, 0xb9, 0xcb, 0x98,
		    0xc0, 0xd1, 0xc8, 0xc5, 0xf9, 0x3d, 0xed }
}, {
	.key	= { 0x09, 0x5c, 0x71, 0xde, 0x41, 0x76, 0x2d, 0x48,
		    0x4f, 0x58, 0xec, 0x0e, 0x62, 0x00, 0x73, 0x04,
		    0x03, 0x47, 0x04, 0x8a, 0x4c, 0xd7, 0x52, 0x5a,
		    0x7b, 0x84, 0x36, 0xbf, 0x8a, 0xdb, 0xeb, 0xf5 },
	.nonce	= 0x45a98ae9d6bf5373ULL,
	.input	= { 0x40, 0x3d, 0x31, 0x7d, 0x63, 0xba, 0xec, 0xd8,
		    0xa6, 0x73, 0xc4, 0x8c, 0x40, 0xc3, 0xaa, 0x3b,
		    0xca, 0x03, 0x52, 0xa0, 0xe8, 0x3f, 0xb6, 0x21,
		    0x3f, 0x07, 0xb1, 0xb3, 0x14, 0x94, 0xf8, 0x36,
		    0xb3, 0x06, 0x33, 0x1e, 0xb1, 0x05, 0x94, 0x53,
		    0x1f, 0xee, 0x16, 0x8e, 0xcd, 0x0f, 0x38, 0x41,
		    0x1f, 0x4d, 0x12, 0x60, 0xec, 0x88, 0x42, 0x2a,
		    0xca, 0x3e, 0x84, 0x33, 0x13, 0x17, 0xeb, 0x04 },
	.ilen	= 64,
	.result	= { 0x48, 0x0e, 0x7f, 0x54, 0xe6, 0x3e, 0x98, 0x81,
		    0xc6, 0x1d, 0x5a, 0x1b, 0x4d, 0xaf, 0x27, 0x3c,
		    0xce, 0xd7, 0x98, 0x05, 0x92, 0xdb, 0xb9, 0x24,
		    0xef, 0xbc, 0x6a, 0x01, 0xb9, 0x61, 0xb9, 0xf9,
		    0x66, 0xee, 0xc9, 0x30, 0xf5, 0x48, 0xb1, 0xde,
		    0xcd, 0xe9, 0x4d, 0x05, 0x2f, 0xea, 0x2c, 0xdf,
		    0x32, 0xc1, 0x43, 0x50, 0x31, 0x1d, 0x0a, 0x3a,
		    0x4e, 0x24, 0xd3, 0x69, 0xc6, 0x18, 0x23, 0xdb }
}, {
	.key	= { 0x5d, 0x01, 0xdf, 0xe5, 0xbc, 0xf6, 0xaf, 0x04,
		    0x11, 0xda, 0xef, 0x43, 0x6b, 0xdf, 0xdd, 0xb8,
		    0xa6, 0xc6, 0x4a, 0xbd, 0xa4, 0xb3, 0xba, 0xf6,
		    0xce, 0x2f, 0xf8, 0x34, 0xbb, 0xdd, 0x22, 0x42 },
	.nonce	= 0x70faa08a4184e385ULL,
	.input	= { 0xb8, 0x06, 0x92, 0x20, 0xc7, 0x7c, 0xb2, 0x01,
		    0x3e, 0xf6, 0x73, 0x6c, 0x74, 0x48, 0x72, 0xec,
		    0x5a, 0x38, 0xfa, 0x36, 0x39, 0x39, 0xff, 0x3c,
		    0x81, 0x91, 0x99, 0x0b, 0x29, 0xec, 0xed, 0x4c,
		    0x53, 0x9a, 0x24, 0xce, 0x4b, 0xcf, 0x8d, 0x36,
		    0x2b, 0xf1, 0x8b, 0x0a, 0x36, 0xdc, 0xf8, 0xdc,
		    0x5a, 0x77, 0x64, 0xca, 0x5f, 0x06, 0xf8, 0xb8,
		    0xc2, 0xa3, 0x69, 0x0d, 0x2f, 0x2f, 0xa5, 0x39,
		    0x94, 0xc6, 0x83, 0xe3, 0x20, 0x48, 0x8b, 0xb7,
		    0x58, 0x26, 0xd0, 0x39, 0x2a, 0x4f, 0x7b, 0x66,
		    0xda, 0x52, 0x0d, 0xa9, 0x5b, 0xb3, 0x44, 0x57,
		    0x38, 0x86, 0x6a, 0xba, 0xa7, 0x86, 0x44, 0x82,
		    0xfb, 0xa8, 0xe8, 0x9d, 0x08, 0x1c, 0x3a, 0xce,
		    0x92, 0x02, 0x61, 0xb6, 0x3e, 0x81, 0xcb, 0x44,
		    0x12, 0x44, 0xf8, 0x2f, 0x53, 0x62, 0xbd, 0xfb,
		    0x3e, 0x9f, 0x65, 0x19, 0xae, 0xc8, 0x4f },
	.ilen	= 127,
	.result	= { 0x0e, 0x37, 0xd1, 0x08, 0x6c, 0x55, 0xc5, 0x88,
		    0x33, 0x0a, 0xbd, 0x6e, 0x24, 0xb1, 0x00, 0xdd,
		    0xb0, 0x43, 0x1f, 0x5a, 0x20, 0x06, 0xfb, 0xa5,
		    0x00, 0x7c, 0xb4, 0x86, 0x58, 0x46, 0x55, 0x6c,
		    0xa2, 0x28, 0xd7, 0x55, 0x51, 0xd5, 0xf1, 0xa8,
		    0x43, 0x06, 0xc4, 0x50, 0xd3, 0x6f, 0x44, 0x21,
		    0x9c, 0xce, 0x29, 0x89, 0xc2, 0x79, 0x20, 0x8f,
		    0x76, 0xfe, 0x3a, 0xef, 0xdd, 0x59, 0x21, 0x94,
		    0xe4, 0xe2, 0xa3, 0x3c, 0x08, 0x74, 0xe7, 0xb5,
		    0x99, 0xf3, 0x8a, 0x88, 0xad, 0xb1, 0x98, 0x31,
		    0xd2, 0x2a, 0x36, 0xda, 0xdf, 0x2c, 0x4c, 0xcd,
		    0xd6, 0xed, 0xe2, 0xca, 0x54, 0xa0, 0x72, 0x0d,
		    0xce, 0x16, 0xe3, 0xa7, 0x2d, 0x93, 0x2b, 0x07,
		    0x18, 0x3f, 0xdb, 0x14, 0xd0, 0x00, 0x26, 0xe4,
		    0x48, 0xa6, 0xe4, 0x59, 0xb7, 0xec, 0xbc, 0xee,
		    0xba, 0x5e, 0x7e, 0xae, 0xf2, 0xd8, 0x91 }
}, {
	.key	= { 0x92, 0x58, 0xaa, 0xf5, 0x07, 0x8f, 0xa3, 0xdd,
		    0xc6, 0x97, 0x32, 0x34, 0xd6, 0x96, 0xca, 0xc1,
		    0x6d, 0xf3, 0xef, 0xfe, 0x44, 0x32, 0x82, 0xa4,
		    0x52, 0x26, 0xf2, 0x69, 0xfb, 0xbf, 0x24, 0xc8 },
	.nonce	= 0xf66f76ba3ce150e4ULL,
	.input	= { 0x69, 0x3d, 0xd7, 0xa9, 0x86, 0xf7, 0x64, 0xac,
		    0x5b, 0x0d, 0xd3, 0xa8, 0x4a, 0xf8, 0xc9, 0x12,
		    0xa9, 0x2a, 0xc2, 0x28, 0x14, 0x02, 0x48, 0x7a,
		    0x64, 0xeb, 0x5b, 0x5a, 0xce, 0x79, 0xca, 0x86,
		    0xa4, 0x60, 0xe6, 0xbe, 0x56, 0x29, 0x50, 0x2d,
		    0xbd, 0x44, 0x88, 0xf8, 0x92, 0xf7, 0x60, 0xe7,
		    0xb1, 0xf4, 0x8b, 0x44, 0xbf, 0x25, 0x4d, 0xc8,
		    0x6b, 0x2e, 0xdc, 0x9e, 0xe8, 0x0f, 0x38, 0x3b,
		    0x48, 0x8c, 0x9c, 0x30, 0xd8, 0x18, 0xf5, 0xcd,
		    0x85, 0x69, 0x71, 0x99, 0xe3, 0xb0, 0x23, 0x11,
		    0x26, 0xf5, 0x84, 0x15, 0x3f, 0xb8, 0xfb, 0xf4,
		    0xee, 0x9a, 0xc2, 0x79, 0xe6, 0x92, 0x6f, 0x4d,
		    0x8a, 0x62, 0xcb, 0x38, 0x37, 0x44, 0x84, 0x09,
		    0xc6, 0xf7, 0xa1, 0x48, 0xf1, 0xa1, 0xf7, 0x1b,
		    0x87, 0x8e, 0xd0, 0xef, 0xa2, 0x1e, 0x32, 0x8e,
		    0x9b, 0xdc, 0x0e, 0xa4, 0xf1, 0x94, 0x25, 0x1b,
		    0x9b, 0x3c, 0x50, 0xdd, 0x41, 0x99, 0xd0, 0x96,
		    0x60, 0xde, 0xbd, 0xed, 0x0b, 0x9a, 0xa4, 0xcb,
		    0x2d, 0x2e, 0xd6, 0xdf, 0x97, 0xe0, 0xea, 0x87,
		    0x12, 0xe1, 0x9b, 0x30, 0xe8, 0x6f, 0x01, 0x25,
		    0x0c, 0xa4, 0xc1, 0xd4, 0x0e, 0xca, 0x07, 0x46,
		    0xd9, 0x4b, 0x5d, 0x03, 0xcb, 0x8d, 0xcf, 0x55,
		    0xe2, 0x62, 0x61, 0x10, 0x86, 0xf9, 0xf5, 0xe8,
		    0xfe, 0x2f, 0xc5, 0xbc, 0x27, 0x2a, 0x1b, 0x1b,
		    0x43, 0x4e, 0xb7, 0xd7, 0x18, 0x7b, 0xcf, 0x7a,
		    0x5b, 0xa5, 0x18, 0x72, 0xa4, 0xa7, 0x24, 0x9f,
		    0x6a, 0x12, 0xe7, 0x44, 0xb5, 0x45, 0x8c, 0xd5,
		    0x53, 0xba, 0x2c, 0xaf, 0xaf, 0x3f, 0xe7, 0x12,
		    0x45, 0xbd, 0x61, 0x68, 0xde, 0xc3, 0x38, 0xa8,
		    0x5b, 0xbe, 0xc8, 0x7b, 0xab, 0xab, 0x21, 0x75,
		    0x95, 0x8e, 0xb7, 0x98, 0xb8, 0x76, 0x69, 0xf8,
		    0xf4, 0x70 },
	.ilen	= 250,
	.result	= { 0xd6, 0xfc, 0x21, 0x55, 0x85, 0x12, 0xa2, 0xd2,
		    0x1c, 0x22, 0x96, 0xd6, 0xfe, 0x61, 0x24, 0x75,
		    0x0c, 0xaa, 0x90, 0xe9, 0x0b, 0xe1, 0x72, 0xc8,
		    0xb5, 0xcb, 0x17, 0x95, 0x9e, 0x30, 0x7c, 0x23,
		    0x69, 0x5f, 0x7a, 0x71, 0x34, 0xac, 0x47, 0x18,
		    0x0a, 0x9e, 0x39, 0x71, 0x2a, 0xff, 0x5a, 0xba,
		    0x35, 0x1b, 0x60, 0xc2, 0xe8, 0x10, 0x2d, 0xbf,
		    0x75, 0xf8, 0x2f, 0x2a, 0x0e, 0x01, 0x67, 0x4e,
		    0xaf, 0xb5, 0xf3, 0x1c, 0xa1, 0x62, 0x7e, 0x22,
		    0x2d, 0x17, 0x3c, 0x73, 0x03, 0x4f, 0xff, 0xb2,
		    0x47, 0xae, 0x68, 0x63, 0x95, 0xcf, 0x66, 0x0e,
		    0x22, 0xdf, 0x6c, 0x8b, 0xb4, 0xee, 0x3a, 0xb3,
		    0x5d, 0x0d, 0xd3, 0x33, 0x16, 0x13, 0x45, 0x59,
		    0x42, 0x74, 0x9f, 0xc5, 0xfb, 0xc6, 0x20, 0x88,
		    0x33, 0x41, 0x55, 0x28, 0x61, 0x33, 0x9f, 0x80,
		    0x57, 0x39, 0x26, 0xe4, 0x3b, 0x98, 0xcd, 0x6b,
		    0xef, 0x10, 0x33, 0x20, 0xee, 0x18, 0x3a, 0x59,
		    0x89, 0x4a, 0x1a, 0xc3, 0xdf, 0xb2, 0x5b, 0x0b,
		    0x08, 0x1f, 0xaa, 0xaa, 0x8e, 0x54, 0xd3, 0x10,
		    0xf7, 0xc2, 0xce, 0x68, 0x06, 0x06, 0x3e, 0x39,
		    0x13, 0x0f, 0x43, 0xfd, 0x3b, 0x61, 0xc7, 0xfd,
		    0x92, 0xf2, 0x0b, 0xe7, 0xa2, 0x60, 0xd8, 0xd8,
		    0x00, 0xe1, 0x18, 0x77, 0x13, 0xb8, 0xa4, 0xa8,
		    0xc0, 0x60, 0x55, 0x96, 0x2d, 0x6a, 0x69, 0x72,
		    0x05, 0xa7, 0x27, 0xe5, 0x05, 0x20, 0x22, 0xcf,
		    0xcf, 0xd5, 0x26, 0x70, 0x1d, 0x76, 0x97, 0xd6,
		    0x93, 0x71, 0xd4, 0x0a, 0xe2, 0x5e, 0x47, 0x54,
		    0x42, 0x6e, 0xb0, 0xea, 0xa9, 0x8b, 0x73, 0x2c,
		    0xc7, 0xbb, 0x2d, 0x64, 0xfd, 0xcf, 0x23, 0x33,
		    0xb1, 0x93, 0x0c, 0x32, 0x66, 0x1e, 0x26, 0x7a,
		    0x8c, 0xb6, 0xfa, 0x85, 0x44, 0x35, 0x02, 0xce,
		    0x05, 0xf6 }
}, {
	.key	= { 0x56, 0x5b, 0x3a, 0xc9, 0xa6, 0x55, 0x34, 0xa2,
		    0xb3, 0xaf, 0x33, 0xed, 0x9d, 0xed, 0x97, 0xfa,
		    0x13, 0x1b, 0x06, 0x50, 0x09, 0x02, 0xa1, 0x93,
		    0xe4, 0xa8, 0x3b, 0xd5, 0x8f, 0xc9, 0xa0, 0xc2 },
	.nonce	= 0x88663f30a2aa338ULL,
	.input	= { 0x9a, 0x2a, 0x04, 0xa0, 0x09, 0x55, 0x05, 0xfc,
		    0x17, 0x80, 0x12, 0x6a, 0x78, 0x63, 0xe6, 0x55,
		    0x13, 0xb7, 0x1e, 0x42, 0x67, 0x86, 0x01, 0x61,
		    0x42, 0x74, 0x9a, 0x08, 0x46, 0xac, 0x69, 0x3a,
		    0x10, 0x24, 0x18, 0x97, 0x3b, 0x85, 0xfb, 0x40,
		    0x5a, 0x77, 0x5b, 0xf3, 0x6b, 0x77, 0xf0, 0x1e,
		    0x41, 0xba, 0x38, 0x7d, 0xa1, 0x23, 0x9b, 0xd3,
		    0x58, 0x59, 0x58, 0x34, 0xa9, 0x03, 0xb3, 0x43,
		    0x9f, 0x5b, 0xe3, 0x1d, 0xf5, 0x9f, 0x8a, 0xdc,
		    0x99, 0x99, 0x45, 0xb0, 0x72, 0xe0, 0xa0, 0x81,
		    0xed, 0x7e, 0x2c, 0x21, 0x3d, 0xf2, 0xad, 0xfd,
		    0xde, 0x9a, 0xf0, 0x40, 0xe7, 0xce, 0x1f, 0xc2,
		    0x34, 0x2b, 0xf9, 0xc9, 0x52, 0x71, 0x0c, 0xa6,
		    0x8a, 0x64, 0x90, 0xd0, 0xcf, 0x28, 0x8d, 0xc6,
		    0x1c, 0x44, 0xd3, 0xdc, 0x24, 0x2a, 0x57, 0x79,
		    0xc9, 0xa1, 0xb2, 0x38, 0x33, 0x20, 0xa2, 0xd0,
		    0x59, 0x6e, 0xb9, 0x27, 0x06, 0xdb, 0xaf, 0x7d,
		    0xed, 0x74, 0x8d, 0x34, 0xbc, 0x3f, 0x89, 0x42,
		    0xe0, 0x28, 0x59, 0x2b, 0x8a, 0x3f, 0x42, 0xc5,
		    0x69, 0x55, 0x1d, 0x8e, 0xa2, 0xba, 0xda, 0x01,
		    0x68, 0x7f, 0x5d, 0xbd, 0x06, 0xa6, 0x95, 0xa7,
		    0xad, 0x60, 0xf4, 0x5a, 0x43, 0x62, 0x8f, 0x69,
		    0x9a, 0xc3, 0xd1, 0x89, 0xd3, 0xfe, 0xc8, 0x2d,
		    0x4d, 0xdd, 0x1c, 0x07, 0xfa, 0x3f, 0xaa, 0x7e,
		    0xfd, 0x90, 0xff, 0x6b, 0x35, 0x5e, 0x23, 0x0d,
		    0x38, 0x16, 0x00, 0xc7, 0x00, 0xa0, 0xb2, 0x37,
		    0x1e, 0x18, 0xdf, 0xa1, 0x18, 0xbc, 0x21, 0x88,
		    0x80, 0x8f, 0x8c, 0xf8, 0x0c, 0x83, 0x69, 0xd6,
		    0x89, 0x0e, 0xe9, 0xbc, 0x69, 0x11, 0x3b, 0x17,
		    0xa1, 0x44, 0x4c, 0xf4, 0x02, 0x9e, 0x3a, 0x8f,
		    0xaf, 0xbc, 0xcc, 0x4f, 0x46, 0x42, 0x7d, 0x63,
		    0x9d, 0x95, 0x8d, 0xa7, 0x9d, 0xf9, 0x73, 0xc0 },
	.ilen	= 256,
	.result	= { 0x79, 0x75, 0xce, 0x01, 0x53, 0x9b, 0x03, 0x24,
		    0x1b, 0x28, 0x10, 0x70, 0x98, 0xd8, 0x43, 0x4d,
		    0x48, 0xba, 0xa2, 0xfb, 0x44, 0x08, 0xff, 0xac,
		    0x59, 0x91, 0xb2, 0x4c, 0x8b, 0x76, 0x99, 0x79,
		    0x56, 0x9b, 0x61, 0x69, 0x3c, 0x75, 0xef, 0xbf,
		    0x4b, 0x22, 0x91, 0x58, 0xd9, 0x54, 0xaa, 0xba,
		    0xe7, 0x20, 0xf5, 0xba, 0x4a, 0xe0, 0xbc, 0x23,
		    0x3d, 0x2d, 0x39, 0x82, 0xad, 0x74, 0x01, 0x4d,
		    0x2d, 0xd8, 0x42, 0x3d, 0x94, 0x42, 0x48, 0x31,
		    0x40, 0x81, 0xc0, 0x24, 0xeb, 0xf8, 0xa3, 0xa0,
		    0x51, 0x2e, 0x70, 0x7a, 0x6e, 0xfe, 0xa2, 0xae,
		    0x49, 0x4b, 0x60, 0x76, 0x1c, 0x82, 0x5a, 0xe5,
		    0xcd, 0x0e, 0x17, 0xd4, 0x0e, 0x50, 0x22, 0x7f,
		    0xd9, 0xd8, 0x5e, 0x60, 0x45, 0xd7, 0x98, 0xa6,
		    0xc1, 0x01, 0xc8, 0x45, 0x18, 0x37, 0x9f, 0xd1,
		    0x26, 0x61, 0xd4, 0x74, 0x7f, 0x44, 0xd3, 0xad,
		    0xf0, 0x20, 0xa1, 0x75, 0x8d, 0xaa, 0xf0, 0x26,
		    0xad, 0xf7, 0x83, 0x45, 0xb5, 0x6f, 0xd0, 0xf9,
		    0xa3, 0xa9, 0x7b, 0xdf, 0xfd, 0x90, 0x30, 0xa9,
		    0x66, 0x25, 0x19, 0xd6, 0xcb, 0x16, 0xfb, 0xb1,
		    0x8b, 0x33, 0x0b, 0x78, 0x8f, 0x40, 0xe0, 0x58,
		    0x99, 0x3d, 0xd7, 0xfe, 0xd6, 0x2c, 0x84, 0xfd,
		    0x74, 0x99, 0xea, 0x0c, 0x24, 0x18, 0x2a, 0x1c,
		    0x46, 0xc0, 0xdc, 0xdf, 0xd0, 0x62, 0x03, 0x10,
		    0x68, 0x37, 0xb1, 0x3f, 0x45, 0x97, 0xf3, 0xe5,
		    0x2e, 0xd3, 0xe1, 0x9d, 0x84, 0x57, 0x1c, 0x71,
		    0x3e, 0xb9, 0x5b, 0x42, 0xc4, 0xaa, 0x84, 0xf0,
		    0x47, 0x42, 0x5d, 0xed, 0xde, 0x80, 0x3e, 0xde,
		    0xed, 0xac, 0xeb, 0x3f, 0x13, 0xaa, 0x6f, 0x8a,
		    0x10, 0xaa, 0x97, 0x64, 0x28, 0x06, 0x2c, 0xaf,
		    0xfc, 0x71, 0x55, 0x45, 0x8b, 0xd1, 0x31, 0xc7,
		    0x29, 0xdf, 0xd2, 0x93, 0x43, 0xad, 0xe9, 0x75 }
}, {
	.key	= { 0x70, 0x11, 0x81, 0x4d, 0x0f, 0xf3, 0x69, 0xe9,
		    0x16, 0x2e, 0x24, 0x94, 0x7a, 0x7b, 0xeb, 0x6c,
		    0x8d, 0xfa, 0xc1, 0x89, 0xaf, 0x79, 0x67, 0xec,
		    0xe3, 0x0c, 0x74, 0x3d, 0xeb, 0x0e, 0x7c, 0x52 },
	.nonce	= 0x8a3a75f2adc47f8fULL,
	.input	= { 0xfa, 0xc3, 0xca, 0xf1, 0xdd, 0x0c, 0xf3, 0xbb,
		    0xbe, 0x51, 0x0f, 0xea, 0x1f, 0x09, 0xda, 0x68,
		    0xfa, 0xb2, 0x0d, 0xa9, 0xc6, 0xb3, 0x2b, 0x0c,
		    0x8a, 0xd6, 0x3f, 0xd6, 0x2e, 0x9d, 0x0e, 0x01,
		    0xdf, 0x58, 0xf2, 0x02, 0x2d, 0x16, 0x54, 0x69,
		    0x0e, 0xff, 0x23, 0x01, 0x84, 0x9c, 0xa7, 0xfc,
		    0x3e, 0xd6, 0xb7, 0x32, 0x4e, 0xa9, 0x5d, 0xc1,
		    0xb7, 0x62, 0x2d, 0x52, 0xda, 0x95, 0x17, 0x56,
		    0xf2, 0xa7, 0x81, 0x16, 0x12, 0x5a, 0xab, 0x0b,
		    0x65, 0x5e, 0xad, 0xe5, 0xbc, 0xee, 0xe9, 0xd7,
		    0x03, 0xe3, 0x18, 0x50, 0xed, 0xcc, 0xef, 0xd8,
		    0x59, 0xaa, 0x83, 0x01, 0xbc, 0x61, 0x9c, 0x5f,
		    0xa3, 0x72, 0x4a, 0x47, 0xce, 0x14, 0x7c, 0x13,
		    0xb9, 0xe9, 0xc3, 0xba, 0xb0, 0xc6, 0x4d, 0xb8,
		    0xb2, 0x02, 0xc1, 0xbf, 0x6a, 0xb6, 0xcb, 0x65,
		    0x05, 0xaa, 0xdb, 0x35, 0x4e, 0x78, 0x5c, 0xbd,
		    0x9f, 0xef, 0x58, 0x75, 0x55, 0x49, 0x93, 0xb4,
		    0xd0, 0x21, 0x44, 0x8e, 0x05, 0xb8, 0xbe, 0xa9,
		    0x72, 0x33, 0x8e, 0x9b, 0xa3, 0xf7, 0x38, 0x97,
		    0xf7, 0x25, 0xb6, 0x4b, 0x44, 0xbb, 0xc6, 0x45,
		    0x55, 0xb9, 0x64, 0xbf, 0x43, 0xf6, 0x51, 0x0d,
		    0xde, 0xa3, 0xc5, 0xda, 0x45, 0x03, 0x7c, 0x7b,
		    0xf4, 0xd2, 0x1c, 0xce, 0x38, 0xa9, 0x0e, 0x2d,
		    0x90, 0x80, 0x1d, 0x5d, 0xca, 0x18, 0xdc, 0x53,
		    0x31, 0x42, 0xaa, 0x92, 0xfb, 0x59, 0xac, 0x8b,
		    0x57, 0x25, 0xa8, 0xf0, 0xc1, 0x55, 0x4a, 0x16,
		    0xdb, 0xa4, 0xf7, 0xfd, 0xe1, 0x49, 0x31, 0x6f,
		    0xb5, 0x5c, 0x69, 0x9d, 0xc2, 0x98, 0x3e, 0x9c,
		    0xf5, 0x51, 0x52, 0xdc, 0x2f, 0x0e, 0x90, 0x5f,
		    0x41, 0x88, 0x3a, 0x69, 0x1c, 0x91, 0x69, 0x97,
		    0xcb, 0xa5, 0x86, 0x52, 0xf7, 0x98, 0x64, 0x04,
		    0x47, 0x60, 0x6f, 0x32, 0x3b, 0xe2, 0x7b, 0x33,
		    0x0a, 0x79, 0x53, 0xad, 0x97, 0xa5, 0xc9, 0x1b,
		    0x8b, 0x5d, 0xfd, 0x1e, 0x51, 0x36, 0x2a, 0x7e,
		    0x0c, 0xdd, 0x0d, 0x0a, 0xc1, 0xf8, 0xb9, 0x77 },
	.ilen	= 280,
	.result	= { 0x9b, 0x42, 0x0b, 0x62, 0x0c, 0x90, 0x82, 0x01,
		    0x6d, 0x87, 0x1d, 0x33, 0xb2, 0x9d, 0x54, 0x8b,
		    0x1b, 0xc1, 0xe5, 0x3b, 0x50, 0xcf, 0x2a, 0x49,
		    0x3d, 0xcf, 0x7b, 0x70, 0xd5, 0xa2, 0x6c, 0x9b,
		    0x55, 0x72, 0x3b, 0x70, 0xbb, 0x99, 0xf3, 0xfd,
		    0x34, 0x87, 0xa6, 0x72, 0x77, 0x56, 0x36, 0x05,
		    0x5f, 0xde, 0x34, 0x9c, 0xb9, 0x26, 0x24, 0x6e,
		    0x28, 0x64, 0x30, 0x99, 0xc2, 0xfd, 0xc1, 0x6f,
		    0x27, 0x30, 0x8a, 0x1d, 0xf5, 0x68, 0xd1, 0x99,
		    0xc8, 0x79, 0xb5, 0x40, 0xbc, 0xd2, 0x5d, 0xfc,
		    0x6d, 0x57, 0x65, 0x35, 0x77, 0x9a, 0x1f, 0x3c,
		    0xba, 0x73, 0x76, 0xf3, 0xdb, 0xe9, 0x18, 0x64,
		    0xe6, 0x7e, 0xc9, 0xf3, 0xa6, 0xb2, 0xbf, 0xf0,
		    0xa1, 0xcd, 0xd5, 0x03, 0xdd, 0x1e, 0x00, 0x07,
		    0xba, 0x76, 0x7d, 0x46, 0x48, 0xf1, 0x7d, 0xb7,
		    0x78, 0xd5, 0x9a, 0xdf, 0x80, 0x18, 0xd2, 0xad,
		    0x97, 0xc7, 0xbd, 0xeb, 0x5a, 0xab, 0x17, 0x92,
		    0x4b, 0x93, 0xd3, 0x39, 0x18, 0xef, 0xbb, 0xfe,
		    0x89, 0x6d, 0x7d, 0x1d, 0xd3, 0x86, 0x4d, 0xfd,
		    0x3f, 0x67, 0xfc, 0x9b, 0xfe, 0x4e, 0x9e, 0xe7,
		    0xfb, 0x9f, 0x45, 0x8a, 0xd4, 0xd9, 0x0c, 0x30,
		    0x90, 0xb8, 0xef, 0xc7, 0x18, 0x04, 0x09, 0x67,
		    0xdf, 0x57, 0xe7, 0x30, 0x45, 0x91, 0xe3, 0xa6,
		    0x73, 0x4a, 0x83, 0x6e, 0x92, 0x1e, 0xe5, 0xad,
		    0x56, 0x1b, 0x86, 0x5c, 0xc3, 0x52, 0x58, 0x01,
		    0xe7, 0xb9, 0x2e, 0xdc, 0x96, 0xee, 0x4c, 0x65,
		    0x5a, 0xd7, 0x19, 0x6b, 0xfa, 0xa3, 0x0b, 0xf4,
		    0xec, 0xab, 0xb5, 0xb2, 0xf8, 0xdd, 0x23, 0xff,
		    0xe6, 0xa6, 0x55, 0x75, 0xcb, 0xb1, 0xa7, 0xed,
		    0x6a, 0xa8, 0x60, 0x1a, 0x61, 0x64, 0x2a, 0xbc,
		    0x73, 0xed, 0xa7, 0x54, 0x0d, 0x79, 0xc9, 0xc6,
		    0x14, 0x6c, 0x78, 0xb1, 0x14, 0x10, 0x47, 0x80,
		    0x76, 0xb7, 0x81, 0xac, 0x09, 0xf5, 0x12, 0x2d,
		    0x21, 0x9c, 0x79, 0xd9, 0x37, 0xae, 0x2a, 0x51,
		    0x41, 0x1e, 0x9e, 0x7f, 0x0d, 0x4b, 0x17, 0x87 }
}, {
	.key	= { 0x23, 0xa3, 0x7e, 0x3a, 0xe4, 0x66, 0xb7, 0x86,
		    0x7b, 0xdb, 0xf1, 0xdf, 0x76, 0x65, 0x4a, 0x70,
		    0xc4, 0x93, 0x4d, 0xf4, 0xc1, 0x0e, 0xc1, 0x69,
		    0xce, 0x43, 0xed, 0x9f, 0x48, 0x36, 0xa3, 0x03 },
	.nonce	= 0xbc7df621eb115af2ULL,
	.input	= { 0x11, 0x32, 0xa9, 0x18, 0x65, 0x15, 0x25, 0x2e,
		    0x9a, 0xbb, 0x0c, 0x25, 0x96, 0x1e, 0xbf, 0x89,
		    0x71, 0x7f, 0x18, 0x3a, 0x33, 0xf2, 0x9d, 0x85,
		    0x23, 0x4d, 0x1f, 0x91, 0x44, 0x75, 0x08, 0xee,
		    0xe0, 0x2d, 0xab, 0x1d, 0x8f, 0xab, 0x0e, 0x3f,
		    0xe7, 0x3d, 0x43, 0x3d, 0x19, 0x49, 0xfc, 0xb2,
		    0xf1, 0xc3, 0x53, 0x41, 0x0b, 0x35, 0xda, 0xd1,
		    0x8c, 0x18, 0x9e, 0xb7, 0x1b, 0xa2, 0x5a, 0xfd,
		    0x8a, 0xa4, 0xc2, 0xf4, 0x8d, 0x98, 0x01, 0x53,
		    0xb8, 0x82, 0xf5, 0x53, 0x5c, 0xfe, 0x6d, 0x7d,
		    0x63, 0x21, 0x46, 0x16, 0x29, 0x9a, 0xcf, 0x20,
		    0xb3, 0xf7, 0xd2, 0x7f, 0xb4, 0x30, 0x90, 0xbd,
		    0x95, 0xc2, 0xd2, 0xc0, 0x14, 0x84, 0xd3, 0x4a,
		    0xf0, 0x12, 0xa6, 0xa8, 0xe8, 0x55, 0x97, 0xb2,
		    0x0c, 0xee, 0x25, 0xf7, 0x94, 0xef, 0xae, 0xf7,
		    0xbb, 0xe2, 0x64, 0x7d, 0xe2, 0xdc, 0x0f, 0x6b,
		    0x7c, 0xa6, 0xe4, 0x71, 0x38, 0x44, 0x3d, 0xcd,
		    0x0d, 0x94, 0xe9, 0x52, 0x6a, 0xb0, 0x8e, 0xd5,
		    0x20, 0xe8, 0x60, 0x9c, 0xde, 0x92, 0xfc, 0x67,
		    0x4e, 0xc3, 0x02, 0x6c, 0x6e, 0xd7, 0x56, 0x0a,
		    0x00, 0x81, 0xc8, 0xd0, 0x26, 0x3c, 0x9a, 0x97,
		    0x98, 0xfa, 0x10, 0x0e, 0xad, 0xeb, 0x7e, 0xdc,
		    0x41, 0x58, 0x68, 0xf8, 0xa4, 0x70, 0x95, 0x2d,
		    0x8f, 0x00, 0xd2, 0xcf, 0x9e, 0x57, 0x9e, 0xa7,
		    0xe8, 0x11, 0x4c, 0xfc, 0x64, 0x51, 0x64, 0x11,
		    0x85, 0xef, 0x47, 0xf5, 0xd3, 0x98, 0xda, 0x48,
		    0x7e, 0x04, 0x55, 0x95, 0x4b, 0xc1, 0xc8, 0x42,
		    0x66, 0x7c, 0xc9, 0x2c, 0xb6, 0xde, 0xc9, 0x3b,
		    0x3d, 0x2b, 0x94, 0x88, 0xbf, 0x07, 0xd4, 0xe9,
		    0x88, 0x61, 0x12, 0xb6, 0x3e, 0x2c, 0x14, 0x7f,
		    0xea, 0x6d, 0x0a, 0xd9, 0x64, 0xf0, 0x3d, 0x57,
		    0x82, 0x4a, 0xab, 0x57, 0x65, 0xba, 0x34, 0xca,
		    0x22, 0x4d, 0xa8, 0xb5, 0xfe, 0xc2, 0x0a, 0xb4,
		    0x0a, 0xb2, 0x05, 0x8d, 0x28, 0xc3, 0x5d, 0xee,
		    0x40, 0xae, 0x48, 0x16, 0xeb, 0x60, 0x07, 0x59,
		    0x53, 0x7d, 0x8d, 0xf6, 0x7c, 0x15, 0x32, 0x86,
		    0x0c, 0x58, 0x86, 0x2c, 0xab, 0x92, 0xe8, 0xc8,
		    0x7b, 0x97, 0x73, 0xd2, 0x11, 0x53, 0xed, 0xab,
		    0x7d, 0xc2, 0xeb, 0x07, 0x5b, 0xcf, 0xc6, 0x5f,
		    0xcb, 0xa1, 0x88, 0x9f, 0x4f, 0x74, 0xa9, 0x26,
		    0xdc, 0x63, 0x7c, 0x5b, 0x60, 0x4b, 0x54, 0x02,
		    0xa7, 0xaf, 0x7b, 0xf7, 0x34, 0x76, 0xd3, 0xbc,
		    0x74, 0x51, 0x3c, 0x74, 0x57, 0xe4, 0x06, 0x22,
		    0x25, 0xc3, 0x5f, 0x41, 0x40, 0x87, 0xf8, 0xf9,
		    0xed, 0x26, 0x36, 0x7a, 0xb2, 0x5c, 0xde, 0xdb,
		    0x85, 0x1b, 0x70, 0x87, 0xc8, 0x0e, 0x0e, 0x9c,
		    0x1e, 0x8d, 0x86, 0x6a, 0x52, 0x3a, 0x64, 0x7c,
		    0x50, 0xa3, 0xfe, 0x57, 0x48, 0x3a, 0xf7, 0x35,
		    0x71, 0xea, 0x90, 0xd3, 0xc5, 0xf0, 0x9d, 0x08,
		    0x94, 0x81, 0x6a, 0x90, 0xa6, 0xf1, 0x70, 0x53,
		    0x3b, 0x46, 0x10, 0x5f, 0x2a, 0x90, 0x3c, 0xa7,
		    0x06, 0x54, 0xa8, 0xd5, 0x7d, 0x8d, 0x3e, 0x59,
		    0xaa, 0x19, 0xa3, 0x48, 0x40, 0xc3, 0xf4, 0xb4,
		    0x70, 0x77, 0xed, 0x05, 0x0c, 0x41, 0x34, 0x53,
		    0x7a, 0xd3, 0x1a, 0xc9, 0xbb, 0xee, 0xd3, 0xc3,
		    0xcb, 0xd6, 0xe7, 0xed, 0x56, 0x17, 0x9d, 0x47,
		    0x5f, 0x9a, 0xe6, 0x83, 0x41, 0x70, 0x1e, 0xf1,
		    0x65, 0x98, 0xbf, 0xea, 0x00, 0x3b, 0xfb, 0x0e,
		    0x91, 0x82, 0x30, 0xa0, 0xa5, 0xbb, 0xff, 0x3c,
		    0x04, 0x52, 0x8f, 0xc2, 0x16, 0xa8, 0xe2, 0x78,
		    0x55, 0xaf, 0x91, 0x20, 0xa1, 0xbe, 0xde, 0x90,
		    0xb3, 0xeb, 0xf1, 0x18, 0x84, 0x25, 0xf0, 0xd4,
		    0x42, 0x0f, 0x0b, 0xfc, 0x16, 0x47, 0x0c, 0x94,
		    0x99, 0xeb, 0x96, 0xb5, 0x21, 0x76, 0x03 },
	.ilen	= 511,
	.result	= { 0x93, 0xf6, 0xa1, 0x01, 0xf0, 0x82, 0x18, 0x8d,
		    0xf2, 0xd1, 0x29, 0x36, 0x6b, 0x1f, 0xcb, 0x7b,
		    0xa2, 0xde, 0x37, 0x8f, 0x31, 0x46, 0x9b, 0x73,
		    0xd4, 0x9e, 0x71, 0x7b, 0x49, 0x79, 0xc5, 0x2c,
		    0x29, 0x1f, 0x60, 0xfd, 0x94, 0x9c, 0x00, 0xc0,
		    0xe5, 0x49, 0xd3, 0x99, 0xe8, 0xb8, 0x7a, 0x9a,
		    0x35, 0xb3, 0x58, 0xa1, 0x2c, 0x4b, 0xef, 0x7c,
		    0x05, 0xf0, 0x07, 0x35, 0x4e, 0x02, 0x36, 0xae,
		    0x29, 0xe4, 0x7c, 0x21, 0x35, 0xba, 0x77, 0xe0,
		    0x80, 0x59, 0xd4, 0x19, 0xcd, 0xb0, 0xc6, 0xff,
		    0x4f, 0x71, 0x91, 0x04, 0x90, 0x75, 0xe7, 0xea,
		    0xd0, 0xc6, 0xa5, 0x10, 0xd2, 0x4b, 0x9d, 0x0f,
		    0x3e, 0x08, 0x33, 0xf8, 0x7d, 0xe0, 0xcf, 0xb9,
		    0x4e, 0xae, 0xd2, 0xe7, 0xe0, 0x2e, 0x98, 0x11,
		    0xdc, 0xb9, 0x95, 0x25, 0xb2, 0x97, 0x16, 0x5b,
		    0x75, 0x40, 0xa0, 0x6c, 0xcb, 0x67, 0xfb, 0x55,
		    0x35, 0x20, 0xd1, 0xf0, 0x58, 0x3c, 0x67, 0xc3,
		    0x77, 0x4e, 0xee, 0xaf, 0xbb, 0x5c, 0x15, 0x1b,
		    0x05, 0xf8, 0xd3, 0x00, 0x8a, 0x33, 0x4c, 0x8f,
		    0x01, 0x78, 0x00, 0xb0, 0x54, 0x9c, 0x63, 0x0c,
		    0x09, 0x37, 0xf0, 0x0a, 0x4a, 0xae, 0x82, 0x2c,
		    0x14, 0x99, 0xa2, 0x32, 0x3e, 0x6a, 0x90, 0x73,
		    0x70, 0x6c, 0x01, 0xb1, 0x60, 0xf3, 0x15, 0xc6,
		    0xc9, 0x05, 0x9a, 0xdc, 0xda, 0x69, 0x1b, 0x31,
		    0xe7, 0xcf, 0x53, 0x40, 0x3f, 0x42, 0xdf, 0x35,
		    0x38, 0x6a, 0xc2, 0x00, 0x11, 0xcb, 0x2c, 0x37,
		    0x81, 0xdd, 0xa8, 0x27, 0xcb, 0x85, 0x89, 0x68,
		    0x79, 0x46, 0x0c, 0xe3, 0x8e, 0xf4, 0x05, 0x9e,
		    0xfc, 0xbd, 0x49, 0xbb, 0x4c, 0xe1, 0x05, 0xdf,
		    0x8c, 0xca, 0xef, 0xbc, 0x3e, 0xb9, 0xa7, 0x02,
		    0x0b, 0xc6, 0x91, 0xe4, 0x48, 0x9d, 0x3a, 0x58,
		    0x4a, 0xf6, 0x49, 0xb0, 0xd9, 0xf0, 0xed, 0x39,
		    0xe5, 0xf2, 0xe2, 0x47, 0xd0, 0x9e, 0xfa, 0xe1,
		    0xb3, 0x7b, 0xb8, 0x1c, 0xef, 0x92, 0xf2, 0xf0,
		    0xc0, 0xd5, 0xf1, 0x89, 0xee, 0xba, 0x52, 0xea,
		    0x39, 0x08, 0x4c, 0xec, 0x92, 0x72, 0x0d, 0x92,
		    0x3e, 0x57, 0x09, 0x5b, 0x4d, 0x6b, 0xbc, 0x31,
		    0x7a, 0xcf, 0x28, 0x30, 0x68, 0x29, 0x3a, 0x99,
		    0x84, 0xa8, 0xb7, 0x36, 0xc7, 0xa7, 0x76, 0x76,
		    0x10, 0xd5, 0x6d, 0x35, 0xb0, 0xe0, 0x86, 0xa0,
		    0x12, 0x19, 0x78, 0x0b, 0xbb, 0x7b, 0x03, 0xae,
		    0x61, 0xf6, 0xe9, 0xb8, 0x7d, 0x7b, 0xf2, 0xb4,
		    0x4e, 0x1c, 0x57, 0x37, 0xa8, 0xa6, 0x9c, 0x6a,
		    0xc2, 0xae, 0xc2, 0xfa, 0x10, 0xd7, 0x77, 0x6f,
		    0x80, 0x91, 0xfe, 0xa9, 0xd2, 0x49, 0xc6, 0x13,
		    0xad, 0x18, 0x35, 0x94, 0x95, 0xe1, 0x5a, 0x18,
		    0x0e, 0x31, 0x49, 0xa5, 0x31, 0xa9, 0xb1, 0xd2,
		    0xf3, 0x36, 0x0a, 0x72, 0x46, 0x45, 0xc1, 0xd0,
		    0x22, 0x6d, 0x3d, 0xe2, 0x73, 0x74, 0xad, 0x0a,
		    0x0c, 0xfd, 0x1d, 0x67, 0xab, 0xcf, 0xe4, 0x4b,
		    0x52, 0xa2, 0xba, 0x87, 0xe8, 0x6b, 0xc3, 0xf9,
		    0xb2, 0xb0, 0x05, 0xf7, 0xe9, 0xac, 0x96, 0x3f,
		    0xbb, 0xe5, 0xae, 0xdb, 0xd3, 0x4f, 0x88, 0x3c,
		    0x29, 0x69, 0x90, 0xc6, 0x2e, 0x56, 0x89, 0x5d,
		    0xda, 0x57, 0xe2, 0x3e, 0x21, 0x72, 0x50, 0x2d,
		    0x65, 0xe9, 0xbd, 0x53, 0x4c, 0xe4, 0x3f, 0x75,
		    0xc7, 0x46, 0x0c, 0xb7, 0x35, 0x25, 0x9b, 0x29,
		    0x6e, 0xb5, 0x91, 0xa1, 0x37, 0xca, 0xa1, 0x26,
		    0x09, 0x3f, 0x7e, 0xbd, 0x8f, 0xf0, 0x2f, 0x59,
		    0x69, 0xa5, 0x74, 0xe2, 0x31, 0xe9, 0x04, 0x36,
		    0x78, 0x14, 0xf0, 0x30, 0x39, 0x6d, 0xfb, 0xae,
		    0xed, 0xb5, 0x23, 0x54, 0x28, 0x3d, 0x31, 0x23,
		    0x49, 0x43, 0xab, 0xf2, 0xf8, 0xc4, 0xe7, 0x5f,
		    0xef, 0x51, 0xa9, 0x25, 0xb9, 0x87, 0x4c }
}, {
	.key	= { 0xa6, 0xa1, 0xb9, 0x92, 0xe4, 0x20, 0x25, 0xe9,
		    0x7e, 0x51, 0x70, 0x50, 0xc9, 0xa3, 0x9c, 0x7f,
		    0x48, 0x9f, 0xca, 0x2f, 0x16, 0x8a, 0xaf, 0xdb,
		    0x25, 0xc0, 0xd3, 0x28, 0x4d, 0x19, 0xd4, 0x57 },
	.nonce	= 0xf81deedff5b091e0ULL,
	.input	= { 0x2a, 0xdb, 0xe1, 0x1d, 0x93, 0xd9, 0x63, 0x17,
		    0x09, 0x06, 0xed, 0x92, 0x67, 0x1a, 0x35, 0x00,
		    0x1a, 0x61, 0xc1, 0xce, 0xd1, 0xd6, 0x6b, 0x78,
		    0x09, 0x32, 0xad, 0x36, 0xf8, 0x2b, 0xc7, 0x6f,
		    0xc7, 0x3b, 0x3a, 0x73, 0x33, 0x00, 0xb0, 0x5c,
		    0x38, 0x8c, 0x48, 0x82, 0x85, 0x09, 0x8a, 0x4a,
		    0x0b, 0x32, 0x53, 0xed, 0x80, 0x66, 0x39, 0xa9,
		    0xa9, 0xf7, 0x11, 0x3b, 0xac, 0x3a, 0xbc, 0xbe,
		    0xda, 0x20, 0x83, 0xc4, 0x42, 0xcf, 0xf6, 0x21,
		    0xb7, 0x70, 0x33, 0x5b, 0x6c, 0xcd, 0x20, 0xd0,
		    0x48, 0x95, 0x36, 0x17, 0xd4, 0x24, 0x55, 0xc7,
		    0x77, 0xc1, 0x32, 0xe2, 0xa5, 0xd2, 0xd4, 0xc1,
		    0x6c, 0xe5, 0x77, 0xe4, 0x8c, 0xda, 0x20, 0x03,
		    0x4b, 0x75, 0x97, 0x64, 0x10, 0x40, 0xf8, 0x5c,
		    0x94, 0xf8, 0xf7, 0x0a, 0x7f, 0x43, 0x56, 0x9a,
		    0xcb, 0xe2, 0x8d, 0xe4, 0x81, 0x4c, 0x6f, 0x89,
		    0x4c, 0x0c, 0x02, 0xf6, 0x00, 0x89, 0x0e, 0xbf,
		    0x3f, 0xd5, 0x73, 0x83, 0x7e, 0x42, 0x49, 0xf6,
		    0xdd, 0x21, 0x2f, 0x46, 0x66, 0x07, 0x59, 0xc5,
		    0x19, 0xee, 0x29, 0x80, 0xc9, 0x3e, 0xa0, 0xfc,
		    0x78, 0x73, 0xcc, 0x38, 0xed, 0xe8, 0xc1, 0x95,
		    0xd7, 0x9e, 0x02, 0x03, 0x77, 0x03, 0x19, 0x2a,
		    0x0d, 0x0b, 0x50, 0x40, 0x64, 0x07, 0x91, 0x6f,
		    0x82, 0x55, 0x1e, 0x8d, 0x9d, 0xe3, 0x83, 0x75,
		    0x06, 0xba, 0x4f, 0x42, 0x7c, 0x73, 0x6b, 0x6c,
		    0x77, 0x31, 0x57, 0xfb, 0x1f, 0x54, 0xe0, 0x05,
		    0x22, 0x22, 0x05, 0x32, 0x6c, 0x56, 0x2e, 0xc7,
		    0x7a, 0x8c, 0xc9, 0xbb, 0x7c, 0x89, 0x6a, 0xb4,
		    0x2e, 0xbb, 0xad, 0xf6, 0xd2, 0xf2, 0x38, 0xf6,
		    0xcb, 0xab, 0xcc, 0x57, 0x98, 0xd4, 0x5f, 0x99,
		    0x4c, 0xa0, 0xe1, 0x62, 0x25, 0x9e, 0x31, 0x34,
		    0xcc, 0xce, 0xd9, 0x79, 0x6a, 0x13, 0xb9, 0x1d,
		    0x0b, 0x9b, 0x40, 0x73, 0x0f, 0x9e, 0xa5, 0xfd,
		    0xae, 0x63, 0x3a, 0xa0, 0x52, 0x1e, 0x29, 0x65,
		    0x0f, 0x62, 0xcc, 0xda, 0x7f, 0x3c, 0x06, 0x91,
		    0xde, 0xea, 0x77, 0xe5, 0x88, 0x6d, 0x7d, 0x26,
		    0xc4, 0xf3, 0xc3, 0xa0, 0xf5, 0xd6, 0x51, 0xda,
		    0x55, 0xcd, 0x38, 0x6f, 0x71, 0x0f, 0x5b, 0x2e,
		    0x3d, 0xab, 0x7c, 0xf1, 0xae, 0xd5, 0xb5, 0xef,
		    0x21, 0x51, 0xb6, 0xb5, 0xe9, 0x91, 0xd0, 0xcd,
		    0x30, 0x11, 0x6d, 0x62, 0x23, 0xfe, 0x5b, 0xda,
		    0xce, 0xef, 0x82, 0x68, 0x7b, 0x3e, 0xd1, 0x7e,
		    0x3b, 0xdf, 0x9f, 0xf6, 0xd1, 0x98, 0xc7, 0xfd,
		    0xdd, 0xc5, 0xc7, 0xb4, 0x79, 0x91, 0x9e, 0x43,
		    0x93, 0xf1, 0xdb, 0x13, 0xb3, 0x46, 0x6d, 0x69,
		    0xa2, 0xce, 0xab, 0xf5, 0x23, 0xbc, 0xdf, 0x99,
		    0x84, 0x5d, 0xba, 0x42, 0x84, 0xaf, 0xf6, 0x70,
		    0xaf, 0x36, 0x5f, 0xc7, 0x32, 0x56, 0x95, 0x9f,
		    0x4a, 0x81, 0x2f, 0x1f, 0xa7, 0x58, 0x9f, 0x98,
		    0x9c, 0x1d, 0x61, 0x98, 0x32, 0xcb, 0xf6, 0xba,
		    0x3e, 0xda, 0x5d, 0x66, 0x26, 0x9b, 0x5d, 0x03,
		    0x6e, 0xa2, 0x66, 0xe8, 0x9e, 0x14, 0x23, 0xae,
		    0x02, 0x24, 0xcd, 0xc5, 0xa7, 0x81, 0x02, 0x0f,
		    0x7b, 0xf5, 0x0c, 0x6e, 0x0b, 0x88, 0x12, 0x94,
		    0xaa, 0x86, 0xff, 0x02, 0xef, 0xaa, 0x36, 0xa5,
		    0x1c, 0x12, 0x23, 0x61, 0x66, 0xfe, 0x05, 0xc1,
		    0x73, 0x43, 0x4e, 0x2b, 0xc9, 0xda, 0xbb, 0x2f,
		    0x67, 0x50, 0x0a, 0x1f, 0xd2, 0x65, 0x8c, 0x75,
		    0x9c, 0xf6, 0xcb, 0xab, 0x33, 0xea, 0xa2, 0x20,
		    0x58, 0x95, 0x05, 0x9f, 0x76, 0xa4, 0x67, 0x68,
		    0xc0, 0x77, 0x57, 0x0b, 0xe9, 0x9e, 0x93, 0x72,
		    0x22, 0x77, 0x9f, 0x8d, 0xf9, 0x2c, 0xe5, 0xa0,
		    0x32, 0x6b, 0xe1, 0xb5, 0x60, 0xa7, 0xa0, 0x7a,
		    0xe2, 0x1e, 0x5c, 0x74, 0x1a, 0x1c, 0x69, 0xc1 },
	.ilen	= 512,
	.result	= { 0xf4, 0x3e, 0xcf, 0xe1, 0xf7, 0x21, 0xb2, 0x55,
		    0xc9, 0xf5, 0xd3, 0x54, 0x40, 0xd0, 0x53, 0x28,
		    0x62, 0xfd, 0xb6, 0x7b, 0x84, 0x51, 0xfd, 0x4d,
		    0x86, 0x95, 0x9a, 0xfe, 0xcf, 0x10, 0xe1, 0xec,
		    0x9f, 0x28, 0x2e, 0x24, 0x0f, 0xd4, 0x27, 0x12,
		    0x7a, 0x36, 0xd4, 0x75, 0x52, 0xa3, 0x1b, 0xb6,
		    0x18, 0xe4, 0x2a, 0x35, 0xd2, 0x30, 0xf3, 0xc7,
		    0xdd, 0xf9, 0x3a, 0xfa, 0x90, 0xfd, 0xc3, 0xe6,
		    0xe5, 0xd4, 0x0c, 0xab, 0xff, 0xd1, 0x9d, 0xfc,
		    0x7c, 0x50, 0xcc, 0x22, 0x68, 0xd0, 0x81, 0xff,
		    0x41, 0x7d, 0x28, 0x75, 0xd0, 0x22, 0xa9, 0x45,
		    0x24, 0x53, 0xf7, 0xc6, 0xed, 0x43, 0x74, 0x14,
		    0xa8, 0xbf, 0x19, 0x22, 0xfd, 0x46, 0x2e, 0x8d,
		    0x96, 0xd2, 0x5e, 0x2f, 0xea, 0x52, 0x29, 0x61,
		    0x48, 0xd6, 0x57, 0xa7, 0x78, 0xaf, 0xdb, 0x0b,
		    0x21, 0x62, 0x12, 0x7c, 0xaa, 0x40, 0xff, 0xdc,
		    0xdc, 0x8f, 0x21, 0x73, 0x08, 0x6a, 0x05, 0xdf,
		    0x3b, 0xd3, 0xd5, 0xdf, 0x63, 0xa7, 0xaa, 0xff,
		    0x67, 0xb2, 0xd7, 0xff, 0x4f, 0xba, 0xf4, 0x35,
		    0xef, 0x3f, 0xb2, 0x84, 0xe2, 0x04, 0xd1, 0xb2,
		    0x6c, 0x53, 0x5e, 0xb7, 0xdb, 0x78, 0x3f, 0x54,
		    0x63, 0x77, 0xf4, 0xda, 0x50, 0xf4, 0x21, 0xb8,
		    0xb4, 0xf1, 0x0b, 0x97, 0x23, 0x7b, 0x5f, 0x84,
		    0xea, 0x20, 0xf2, 0xa2, 0xc4, 0xb8, 0x59, 0xb3,
		    0x50, 0x67, 0xf7, 0xaf, 0xbe, 0xf1, 0x8c, 0xa0,
		    0x37, 0x44, 0x8a, 0xd0, 0x70, 0x77, 0x22, 0x24,
		    0xd2, 0xd6, 0xa1, 0x3a, 0x74, 0x61, 0xa7, 0x38,
		    0x5a, 0xc4, 0xdc, 0x42, 0x6f, 0xdc, 0x6e, 0xc8,
		    0xc7, 0x82, 0x0b, 0x16, 0x72, 0xc5, 0x5f, 0xd4,
		    0x1f, 0x63, 0x13, 0x25, 0xb6, 0xa5, 0x0e, 0xef,
		    0xa0, 0x75, 0xa4, 0x4e, 0x68, 0xaf, 0x9b, 0x6d,
		    0x6f, 0xc9, 0x84, 0x5e, 0xc6, 0x64, 0x50, 0x70,
		    0x81, 0x94, 0xd6, 0x41, 0x6b, 0x17, 0x35, 0x2c,
		    0x15, 0xd8, 0xf0, 0x42, 0x7a, 0x20, 0x42, 0xca,
		    0xe0, 0x5a, 0x24, 0xbc, 0xff, 0x90, 0x9e, 0x1d,
		    0x05, 0x3d, 0x16, 0x35, 0x7d, 0xdc, 0xc6, 0xe2,
		    0x0c, 0xf0, 0xa7, 0xe9, 0xa5, 0x09, 0x28, 0x6c,
		    0xd9, 0x80, 0x80, 0xe8, 0xca, 0x48, 0x26, 0x68,
		    0xb6, 0x84, 0xcb, 0x87, 0x76, 0x13, 0x51, 0x25,
		    0x47, 0xc4, 0x8a, 0xcd, 0x8d, 0x1a, 0x07, 0x14,
		    0x5a, 0xa7, 0x1a, 0xbb, 0x86, 0xe6, 0x76, 0xa0,
		    0x29, 0x50, 0xae, 0xcb, 0x52, 0xfe, 0x0e, 0x27,
		    0xd6, 0x48, 0x40, 0xfb, 0x3e, 0xb3, 0xdb, 0x67,
		    0x2a, 0x3a, 0x9e, 0x62, 0xfc, 0x57, 0x5a, 0x0b,
		    0x2c, 0xe5, 0x6d, 0x3d, 0x33, 0x55, 0xfc, 0xad,
		    0x5d, 0x96, 0x68, 0xd9, 0xea, 0x92, 0xc1, 0x92,
		    0x40, 0x3e, 0x72, 0xe6, 0xcc, 0x51, 0xc0, 0x40,
		    0x70, 0x39, 0x61, 0x8a, 0xcd, 0xef, 0xde, 0x1d,
		    0x44, 0x7d, 0x86, 0x66, 0x9f, 0xaa, 0x22, 0x9e,
		    0xb9, 0xd1, 0x02, 0x05, 0x6b, 0xbd, 0xda, 0x29,
		    0x60, 0x45, 0x77, 0x35, 0xeb, 0xae, 0x8b, 0xd8,
		    0x79, 0x95, 0x8d, 0x1c, 0x3d, 0xe6, 0xf1, 0x37,
		    0x70, 0x9a, 0x63, 0xab, 0xdb, 0x21, 0x4d, 0x7b,
		    0x74, 0x00, 0x02, 0x0c, 0x72, 0x9d, 0x04, 0x39,
		    0x39, 0xfc, 0x9c, 0x7d, 0x9b, 0x42, 0xf0, 0x94,
		    0x72, 0x1e, 0x88, 0x58, 0xbf, 0xc9, 0x28, 0x9f,
		    0xa0, 0x77, 0x65, 0x2e, 0x9d, 0xe9, 0x00, 0x33,
		    0xc2, 0xd1, 0x83, 0x2a, 0x27, 0xb1, 0xa4, 0x69,
		    0xc8, 0x61, 0x99, 0x09, 0xe1, 0xb3, 0xdf, 0x8a,
		    0x11, 0xd4, 0xd4, 0x26, 0xc7, 0x31, 0x83, 0x67,
		    0x1f, 0xd5, 0x00, 0x39, 0x49, 0xf1, 0xa6, 0xaf,
		    0xbc, 0x6e, 0xe9, 0xc5, 0xe0, 0xa7, 0xee, 0x0c,
		    0x7c, 0x9e, 0x08, 0x7f, 0x47, 0x9f, 0xed, 0x85,
		    0x20, 0x23, 0x14, 0x0b, 0x3e, 0x58, 0xc3, 0x59 }
}, {
	.key	= { 0x24, 0x30, 0xaa, 0x5c, 0xad, 0xee, 0xc9, 0xe5,
		    0x15, 0x8d, 0x04, 0xd4, 0x1e, 0xd1, 0xac, 0xd5,
		    0xa2, 0xc2, 0xd9, 0xd2, 0x55, 0xd9, 0x34, 0x09,
		    0x44, 0xf4, 0x29, 0x1c, 0x85, 0xd1, 0x5f, 0xeb },
	.nonce	= 0xb9f185373d1a89baULL,
	.input	= { 0xe5, 0x1a, 0xcf, 0x39, 0x07, 0x21, 0x2b, 0x01,
		    0x7a, 0x9a, 0x13, 0x54, 0x57, 0x8e, 0x45, 0x64,
		    0xa5, 0xb4, 0x0f, 0x17, 0x72, 0x31, 0x92, 0x6b,
		    0x18, 0x5e, 0x79, 0x51, 0x51, 0x56, 0xe5, 0x34,
		    0x6d, 0x70, 0x34, 0x21, 0x00, 0x09, 0x97, 0x3f,
		    0x9a, 0x56, 0xed, 0x10, 0x45, 0x20, 0xb5, 0x99,
		    0x53, 0x09, 0xad, 0x22, 0xc0, 0xbb, 0x13, 0x1f,
		    0xeb, 0xe1, 0xa2, 0x31, 0x9e, 0xab, 0xa8, 0x3c,
		    0xf5, 0xc4, 0xa6, 0xa0, 0xb2, 0xde, 0xd3, 0x58,
		    0x3a, 0x46, 0x88, 0x74, 0x0a, 0x8f, 0x1b, 0x67,
		    0x55, 0x8a, 0xe0, 0xd1, 0x32, 0x9e, 0x07, 0x32,
		    0x87, 0x4f, 0x04, 0xc1, 0xaf, 0xcb, 0x82, 0xdb,
		    0x84, 0x15, 0x52, 0xdc, 0x31, 0xfe, 0x99, 0xee,
		    0x4c, 0x8a, 0xd9, 0x2b, 0x2e, 0xe5, 0xaf, 0x4f,
		    0x95, 0x4b, 0x46, 0x2b, 0x28, 0xe9, 0xd9, 0xf8,
		    0x04, 0xc3, 0xbd, 0x8c, 0xb3, 0x2b, 0x50, 0x7d,
		    0x2a, 0x3a, 0x9d, 0xfd, 0xca, 0x79, 0x91, 0x98,
		    0xc8, 0xe4, 0xdb, 0xe0, 0xe3, 0x51, 0x67, 0xc8,
		    0x99, 0x8e, 0x26, 0x63, 0xaa, 0xf5, 0x51, 0x71,
		    0x90, 0xa2, 0x09, 0x57, 0x6d, 0x30, 0xfe, 0xa1,
		    0x8e, 0xd6, 0x20, 0x32, 0xbb, 0x84, 0x50, 0xc6,
		    0xde, 0xc7, 0x7e, 0xd0, 0x5c, 0xd0, 0x31, 0x93,
		    0x1a, 0x09, 0x18, 0x90, 0x66, 0xc7, 0xc2, 0x05,
		    0x25, 0xb9, 0x2f, 0x8c, 0xd6, 0x38, 0x11, 0x10,
		    0x4e, 0x71, 0x26, 0x41, 0x7a, 0x27, 0x6a, 0x55,
		    0xfc, 0x58, 0x2a, 0x67, 0xd6, 0x1f, 0x9f, 0x7f,
		    0x57, 0x7c, 0xda, 0x7d, 0xa7, 0x78, 0x84, 0xdc,
		    0x7a, 0xd7, 0x4d, 0xdc, 0xb6, 0x70, 0x30, 0xea,
		    0xb9, 0xff, 0x49, 0x00, 0x66, 0x15, 0x22, 0x9e,
		    0x97, 0x53, 0xd8, 0x16, 0x8c, 0xe1, 0x58, 0xb8,
		    0x6e, 0x04, 0xe0, 0x79, 0xf9, 0xf4, 0x2f, 0x75,
		    0x46, 0x06, 0xcf, 0x5b, 0x78, 0xd6, 0xc9, 0xd8,
		    0xa2, 0xfd, 0x1b, 0x9b, 0x9a, 0x82, 0x16, 0x34,
		    0xc6, 0x9c, 0x03, 0x95, 0x89, 0xc2, 0x7f, 0xe7,
		    0x1e, 0xb4, 0x45, 0x43, 0xea, 0xa5, 0x91, 0x3d,
		    0x26, 0xfe, 0x8c, 0xa8, 0x0f, 0x8a, 0xb8, 0x13,
		    0x8e, 0x24, 0xc3, 0x75, 0xa5, 0x9f, 0xb3, 0x65,
		    0x0c, 0x3e, 0x7a, 0x69, 0x5c, 0x01, 0x2c, 0x74,
		    0xb7, 0x23, 0x1b, 0x1a, 0x07, 0x61, 0x57, 0x39,
		    0xd7, 0x8d, 0xdd, 0x15, 0x68, 0x1d, 0xdd, 0x4f,
		    0x3a, 0xc6, 0x48, 0x71, 0x3f, 0x1e, 0x69, 0xb4,
		    0xb5, 0xfc, 0x4f, 0x25, 0x4a, 0xbe, 0x3f, 0x1d,
		    0xed, 0x02, 0x7b, 0x82, 0x43, 0x9c, 0x4c, 0xee,
		    0x5d, 0x16, 0x6e, 0xb6, 0xb3, 0xaa, 0x8f, 0x6b,
		    0x94, 0x98, 0xe9, 0x83, 0x71, 0xa6, 0x4f, 0xc9,
		    0xcc, 0x0e, 0xf2, 0x53, 0x85, 0x49, 0xea, 0xda,
		    0x43, 0x5c, 0xdc, 0xf4, 0x63, 0x9e, 0xa3, 0x98,
		    0x1d, 0x17, 0x10, 0x43, 0xe0, 0xea, 0x30, 0x34,
		    0x75, 0x77, 0xc2, 0x3c, 0x56, 0xdc, 0x0c, 0x1b,
		    0x57, 0xfe, 0x73, 0x02, 0xe9, 0x3e, 0x1a, 0xe3,
		    0x44, 0xdd, 0xa1, 0x70, 0xed, 0xd9, 0x8c, 0x69,
		    0xdb, 0xbb, 0xc9, 0x7d, 0x96, 0x38, 0x8b, 0x3a,
		    0x7b, 0x99, 0x25, 0xaa, 0xab, 0xe9, 0x10, 0x2a,
		    0xa7, 0xdb, 0xad, 0x0a, 0x33, 0xdc, 0x1d, 0x0a,
		    0xc4, 0x63, 0x6b, 0x0b, 0x58, 0x67, 0xb5, 0x7a,
		    0x93, 0xee, 0x5f, 0x74, 0xac, 0xa8, 0x50, 0x29,
		    0x03, 0x91, 0xf7, 0x10, 0xb6, 0xbe, 0xc7, 0xd1,
		    0xef, 0xc9, 0xa2, 0x61, 0x89, 0x6f, 0x5e, 0x04,
		    0xb0, 0xfb, 0x46, 0x9e, 0x18, 0xe7, 0x52, 0x38,
		    0xdd, 0x64, 0x76, 0xd7, 0x4f, 0xbb, 0xaa, 0x20,
		    0x36, 0x8d, 0x86, 0xf4, 0x5d, 0x8b, 0x52, 0x66,
		    0x5c, 0x17, 0x98, 0xc0, 0xc5, 0x5b, 0xc8, 0x00,
		    0xa4, 0x99, 0x3d, 0x0f, 0x0b, 0x5b, 0x41, 0x96,
		    0x48, 0xf3, 0xa5, 0x8c, 0x8e, 0xbb, 0x37, 0x3e,
		    0x13 },
	.ilen	= 513,
	.result	= { 0x60, 0x64, 0xf1, 0xf3, 0x13, 0x5c, 0xf5, 0xd4,
		    0xdc, 0xb9, 0x7b, 0xf8, 0x85, 0xf8, 0xac, 0x46,
		    0x8c, 0x3b, 0x74, 0xa3, 0x6d, 0x15, 0x36, 0x01,
		    0x42, 0x62, 0xfb, 0xef, 0x54, 0x37, 0x92, 0x4f,
		    0xb6, 0x1a, 0x62, 0xa1, 0x0a, 0xb0, 0xf0, 0x52,
		    0x26, 0x0d, 0xd5, 0x0e, 0x4e, 0x98, 0x1a, 0xd8,
		    0x82, 0x60, 0x8b, 0x50, 0x50, 0xfd, 0xbb, 0xc8,
		    0x90, 0xb7, 0x41, 0x26, 0x7f, 0x28, 0x4c, 0x42,
		    0x50, 0xc1, 0x5c, 0x93, 0x66, 0x0c, 0xbc, 0x73,
		    0x46, 0x59, 0xc3, 0x7d, 0x16, 0xfd, 0x72, 0x6d,
		    0x9b, 0x09, 0x1e, 0xd2, 0xa7, 0xca, 0x4f, 0x96,
		    0xca, 0x58, 0x12, 0x75, 0xdb, 0xbf, 0x1b, 0xfe,
		    0xa3, 0x7d, 0x17, 0xf5, 0x93, 0x52, 0x0d, 0xbe,
		    0xc3, 0x6b, 0x06, 0x82, 0x08, 0xec, 0x02, 0xa6,
		    0x9c, 0x54, 0x74, 0x2f, 0x3f, 0x6e, 0x9e, 0x37,
		    0x5f, 0x35, 0xd9, 0xd5, 0x39, 0x88, 0xd6, 0x0e,
		    0xd9, 0x64, 0x17, 0xf2, 0xb2, 0x66, 0x08, 0xc2,
		    0x0b, 0xa5, 0xd4, 0xc2, 0x01, 0xfa, 0x73, 0x4d,
		    0x69, 0x48, 0xa0, 0x9d, 0x2b, 0xde, 0x2d, 0xbd,
		    0xb7, 0xec, 0xc7, 0x55, 0x01, 0x4e, 0xfe, 0x0a,
		    0xa8, 0xee, 0xb1, 0x5c, 0xad, 0x90, 0x3b, 0x94,
		    0x41, 0x3b, 0xa3, 0x24, 0xe5, 0xc8, 0x5c, 0xb9,
		    0x5f, 0x5e, 0x8e, 0x9c, 0x87, 0x57, 0xea, 0xb9,
		    0x93, 0xbb, 0x68, 0xc1, 0xad, 0xc3, 0xcc, 0xdc,
		    0x54, 0xc3, 0xab, 0xe8, 0xee, 0x63, 0xe3, 0x78,
		    0x47, 0x52, 0x36, 0x7e, 0xf0, 0x04, 0xd7, 0xcd,
		    0x23, 0xa9, 0xd6, 0x7b, 0x6b, 0x39, 0xe6, 0xf1,
		    0xf1, 0xf4, 0xe6, 0x43, 0x43, 0x31, 0x2f, 0xe6,
		    0x15, 0x24, 0x0e, 0x02, 0x56, 0x3a, 0xa2, 0x1f,
		    0x82, 0xf6, 0xd1, 0x23, 0x96, 0x09, 0xf8, 0x7a,
		    0x55, 0x66, 0x39, 0x4c, 0x5e, 0x77, 0xff, 0x20,
		    0xf5, 0xef, 0x9e, 0x34, 0x24, 0xbc, 0x33, 0x31,
		    0x27, 0xb5, 0x0b, 0x20, 0x3e, 0x78, 0x31, 0xa3,
		    0x48, 0x9d, 0x51, 0xfe, 0x60, 0xc6, 0xb2, 0xec,
		    0x76, 0x6b, 0x6c, 0xde, 0xfc, 0x53, 0x20, 0xa3,
		    0x83, 0x0f, 0x7b, 0x82, 0xd9, 0x01, 0xf4, 0x03,
		    0xba, 0x90, 0x57, 0x72, 0x8e, 0x7b, 0xa1, 0x17,
		    0xed, 0x61, 0xe9, 0x57, 0x20, 0x37, 0x18, 0xc2,
		    0x99, 0xa8, 0x62, 0x26, 0x9d, 0x7d, 0x6b, 0x4f,
		    0xc1, 0xe5, 0x92, 0x77, 0x3f, 0x44, 0xfd, 0xcd,
		    0xa4, 0x1d, 0x59, 0x51, 0xac, 0x8b, 0x76, 0xc5,
		    0x64, 0xc7, 0x4d, 0xf6, 0x84, 0x42, 0xc5, 0x73,
		    0x4d, 0xe6, 0x6f, 0xa7, 0xda, 0x15, 0x43, 0x32,
		    0xd0, 0x2e, 0x5a, 0xee, 0x36, 0x42, 0x44, 0x77,
		    0x92, 0xa9, 0xc0, 0xdc, 0xa9, 0x23, 0x59, 0xce,
		    0x51, 0xf3, 0x67, 0x7a, 0xca, 0x06, 0xed, 0xc6,
		    0xaf, 0x29, 0x0e, 0xa5, 0xe6, 0x3b, 0xeb, 0x1e,
		    0x53, 0x48, 0x1a, 0x58, 0x13, 0x01, 0x51, 0x7f,
		    0xb6, 0x54, 0x8a, 0xab, 0x02, 0x48, 0x73, 0xac,
		    0x24, 0x6b, 0x5e, 0x5a, 0x2e, 0xf6, 0x63, 0x50,
		    0xb6, 0x32, 0x0e, 0xcf, 0x93, 0xa1, 0x48, 0x7a,
		    0xdb, 0x65, 0xab, 0x2e, 0x8a, 0x22, 0x96, 0x71,
		    0x5a, 0xb9, 0xf7, 0xe4, 0x99, 0x25, 0xa4, 0x80,
		    0xb5, 0x62, 0xe5, 0x4d, 0x2f, 0xcb, 0xce, 0x73,
		    0x61, 0x8f, 0x00, 0xa3, 0xd5, 0x40, 0xec, 0xbb,
		    0x4a, 0xa3, 0xe3, 0x95, 0x20, 0x8b, 0x6a, 0xd6,
		    0x89, 0xa0, 0x91, 0x42, 0xb4, 0xe6, 0x4e, 0x49,
		    0xfa, 0x73, 0xab, 0x1e, 0x22, 0x20, 0x30, 0x5e,
		    0xe1, 0x94, 0x62, 0x48, 0x4d, 0x16, 0x4f, 0xea,
		    0x44, 0x4c, 0x19, 0xea, 0x00, 0x29, 0x5c, 0xce,
		    0x99, 0x32, 0xc0, 0xb3, 0x51, 0xe8, 0x39, 0x7b,
		    0x1c, 0x8f, 0x24, 0xe3, 0x01, 0x5e, 0x65, 0xb2,
		    0x5e, 0x80, 0x15, 0xdf, 0x17, 0x11, 0x1b, 0x40,
		    0xe6, 0xdb, 0xe3, 0x60, 0x7b, 0x2f, 0xa8, 0xa7,
		    0xb6 }
}, {
	.key	= { 0xb2, 0xd2, 0x69, 0x85, 0x45, 0xb5, 0x4c, 0x3c,
		    0xba, 0xb1, 0x01, 0xd4, 0x3d, 0xbe, 0x75, 0x91,
		    0x98, 0x4c, 0x7b, 0x1d, 0xb1, 0xb0, 0x52, 0xf1,
		    0xdd, 0xc3, 0x58, 0x1f, 0x2e, 0x9a, 0x72, 0x4f },
	.nonce	= 0x9aba2243ec59aca0ULL,
	.input	= { 0x6b, 0xf4, 0xfa, 0xf5, 0x28, 0x04, 0xfa, 0xd3,
		    0x03, 0xab, 0x62, 0x69, 0x9f, 0x5e, 0x14, 0x83,
		    0xe9, 0xf2, 0xe5, 0x98, 0xd9, 0x7e, 0x05, 0x22,
		    0xf1, 0xf7, 0x0f, 0x76, 0x4a, 0x33, 0x9d, 0x49,
		    0xa2, 0x16, 0xcf, 0x70, 0xd2, 0x94, 0x7e, 0xe5,
		    0xcc, 0xe3, 0x01, 0x4d, 0x21, 0x14, 0x98, 0xef,
		    0xbf, 0x18, 0x0e, 0x07, 0x00, 0x8a, 0x82, 0xcb,
		    0x12, 0x65, 0xab, 0xd5, 0x2a, 0xe2, 0xe6, 0x07,
		    0xe3, 0x3a, 0x78, 0x22, 0xb0, 0xd8, 0xce, 0x96,
		    0xde, 0x47, 0x6f, 0xa8, 0xfb, 0x44, 0x00, 0x7b,
		    0x33, 0x4e, 0xb4, 0x5d, 0x99, 0x9e, 0x14, 0x49,
		    0xca, 0x90, 0x80, 0x11, 0xac, 0xa9, 0x9b, 0x6d,
		    0x01, 0x24, 0x9b, 0x05, 0x6d, 0x57, 0x35, 0x2a,
		    0xde, 0xd2, 0xb6, 0x54, 0x93, 0x55, 0x08, 0xef,
		    0xfb, 0x19, 0xd0, 0x3e, 0xf2, 0x95, 0xda, 0x75,
		    0x0e, 0x27, 0x38, 0x14, 0xcf, 0xd2, 0x4e, 0xb7,
		    0x27, 0xd1, 0xbd, 0x52, 0x08, 0x7e, 0x6c, 0xe4,
		    0xfd, 0xc5, 0xc3, 0x2a, 0x76, 0x84, 0xf8, 0x4a,
		    0x4e, 0x10, 0x90, 0xb0, 0x0c, 0x32, 0x25, 0x9f,
		    0x5f, 0x75, 0x2a, 0x24, 0x57, 0x45, 0xd5, 0x64,
		    0x32, 0xf7, 0x30, 0x5a, 0x13, 0xbd, 0x79, 0x1c,
		    0x50, 0x46, 0xf7, 0xbd, 0xdc, 0x32, 0x3c, 0x43,
		    0x1c, 0x84, 0xe6, 0x73, 0x78, 0x01, 0x94, 0xe2,
		    0x49, 0x34, 0x32, 0xd9, 0x86, 0x2c, 0xff, 0xa6,
		    0x6a, 0x46, 0x9c, 0xfc, 0xf3, 0x22, 0x31, 0xef,
		    0x60, 0xb8, 0xdc, 0x2e, 0xcc, 0x0e, 0x8c, 0x3a,
		    0xbb, 0xba, 0xb0, 0x36, 0xa0, 0x97, 0x78, 0x09,
		    0x3a, 0x87, 0xa5, 0xef, 0xc6, 0x50, 0x87, 0x02,
		    0xfa, 0xb8, 0x3e, 0x55, 0x2d, 0x78, 0xc8, 0xa8,
		    0x8a, 0xa6, 0x63, 0xba, 0x31, 0x72, 0x56, 0x39,
		    0x7d, 0xb8, 0x4a, 0xe4, 0x1c, 0xe9, 0x7b, 0x8c,
		    0x97, 0x25, 0x3f, 0x25, 0xea, 0x8a, 0x6e, 0x2a,
		    0x09, 0x83, 0x17, 0x12, 0x9b, 0x3b, 0xb8, 0xe6,
		    0xf8, 0x0f, 0x48, 0x68, 0x9a, 0xcf, 0x73, 0x95,
		    0xf1, 0x4c, 0x1b, 0x9b, 0x2e, 0x64, 0x17, 0x6c,
		    0xb1, 0xcd, 0x90, 0x7a, 0xf6, 0x90, 0xf5, 0xa1,
		    0x22, 0x9a, 0x6b, 0x16, 0xfc, 0xc8, 0x86, 0x5a,
		    0xe7, 0x68, 0x53, 0x2b, 0x31, 0xd0, 0x06, 0x63,
		    0x81, 0xdf, 0xd2, 0x9b, 0x7f, 0x60, 0x46, 0xf7,
		    0xf3, 0xef, 0xb5, 0x71, 0xa4, 0x96, 0xf7, 0xe2,
		    0xd1, 0x9b, 0x9d, 0x71, 0x0e, 0x06, 0xb5, 0x6e,
		    0x6e, 0x14, 0x4b, 0xcf, 0xdd, 0x9e, 0x53, 0x71,
		    0x54, 0xb9, 0xfa, 0x4f, 0xb4, 0x25, 0x48, 0xa4,
		    0x8a, 0xf8, 0xca, 0x1e, 0x56, 0xc6, 0xa6, 0x19,
		    0x11, 0x67, 0x9f, 0xb6, 0x59, 0xa1, 0x80, 0x97,
		    0xb5, 0xdd, 0x50, 0x95, 0x94, 0xbe, 0x8f, 0x65,
		    0x05, 0xe3, 0xa2, 0xba, 0x22, 0x36, 0x04, 0x1b,
		    0x44, 0xdf, 0xd8, 0x2c, 0x08, 0xb4, 0x6b, 0x40,
		    0xec, 0x5d, 0x7f, 0xbb, 0x94, 0x25, 0x3c, 0xf3,
		    0x73, 0x52, 0xd4, 0xc5, 0x63, 0x54, 0xde, 0x0f,
		    0x4d, 0xd6, 0xfe, 0x5e, 0xf9, 0x02, 0x84, 0x31,
		    0x28, 0xee, 0xfd, 0x0e, 0xbe, 0xaf, 0xe0, 0x75,
		    0x1c, 0xfd, 0xcc, 0xcf, 0x4e, 0x67, 0xd6, 0x86,
		    0x9c, 0x9d, 0x3c, 0x87, 0xb2, 0x89, 0x79, 0x9b,
		    0xd7, 0x8e, 0xca, 0x90, 0xb6, 0xdd, 0x4a, 0xf1,
		    0x0a, 0xed, 0xaa, 0xc1, 0x80, 0xf9, 0x96, 0x49,
		    0xce, 0xa3, 0x12, 0x63, 0xae, 0xc6, 0xcb, 0x3b,
		    0xf1, 0xe7, 0xb9, 0x1d, 0x0b, 0x0d, 0xd4, 0x1c,
		    0x48, 0x30, 0x53, 0xab, 0xba, 0x94, 0x45, 0xc4,
		    0x5c, 0xaf, 0xe0, 0xe7, 0x81, 0x61, 0x2f, 0xa1,
		    0x02, 0x12, 0x55, 0x5c, 0xde, 0x3a, 0x48, 0x7c,
		    0xe0, 0x3b, 0xab, 0x37, 0x18, 0xf1, 0x1b, 0x8b,
		    0xba, 0x6c, 0x4a, 0xef, 0x5b, 0x95, 0xdb, 0xd8,
		    0xab, 0x8d, 0xde, 0x27, 0xdf, 0x44, 0xcd, 0xb1,
		    0x8d, 0x97 },
	.ilen	= 514,
	.result	= { 0x20, 0xaf, 0x08, 0x43, 0x2c, 0x9b, 0xff, 0x20,
		    0x30, 0xc3, 0xec, 0xec, 0x81, 0x08, 0x9b, 0x8d,
		    0x6e, 0x2a, 0x48, 0xef, 0x65, 0x01, 0xaf, 0x1a,
		    0xe9, 0x46, 0x32, 0x74, 0x28, 0x50, 0xd4, 0x8a,
		    0xc9, 0xf3, 0x2c, 0x6c, 0x6b, 0xad, 0x79, 0xa3,
		    0x61, 0x2c, 0x7e, 0x41, 0xf0, 0xef, 0x74, 0x90,
		    0xc2, 0xc2, 0xfb, 0xc1, 0x41, 0x48, 0xcb, 0xa2,
		    0xc3, 0x37, 0x31, 0x63, 0x97, 0x28, 0xd8, 0x7a,
		    0x6e, 0xa7, 0x57, 0xfb, 0xb2, 0x7d, 0xfc, 0xe1,
		    0x78, 0xe9, 0x32, 0x2c, 0x74, 0xed, 0x29, 0xf7,
		    0xb7, 0x4c, 0x65, 0x04, 0x09, 0x47, 0x22, 0x2a,
		    0x16, 0x3e, 0xe4, 0xff, 0x53, 0x2a, 0x4f, 0xea,
		    0x7c, 0x31, 0x12, 0x0d, 0xc0, 0x55, 0xc8, 0x6b,
		    0x97, 0x7f, 0x2e, 0x64, 0xc1, 0x7b, 0x46, 0xb9,
		    0x9a, 0xfa, 0xcc, 0x36, 0x94, 0x8e, 0x44, 0x06,
		    0xe2, 0xa6, 0xf2, 0xef, 0x8b, 0x6a, 0x45, 0x44,
		    0x2e, 0xc8, 0x6a, 0x3c, 0x1b, 0xa5, 0xa8, 0x01,
		    0xf5, 0x27, 0x23, 0x27, 0xb0, 0xe0, 0x97, 0x69,
		    0x73, 0x08, 0xa4, 0x3e, 0x71, 0x79, 0xce, 0xa7,
		    0xce, 0x38, 0xd8, 0x1c, 0xb8, 0xf3, 0x7f, 0x03,
		    0x6a, 0x4f, 0x19, 0x05, 0xf0, 0x09, 0xe8, 0xb2,
		    0x19, 0xd4, 0x12, 0xcd, 0x46, 0x87, 0xc2, 0xc5,
		    0xd2, 0xe3, 0x04, 0x98, 0x97, 0x16, 0xda, 0xd6,
		    0x39, 0x3a, 0xfe, 0xa8, 0x77, 0x69, 0xc7, 0xe9,
		    0xb2, 0xe7, 0x65, 0x64, 0x06, 0x22, 0xe7, 0xc2,
		    0xd4, 0x26, 0x10, 0xf5, 0xd3, 0x01, 0x9f, 0x3d,
		    0x32, 0x90, 0xbf, 0xd0, 0x19, 0x80, 0xbd, 0xe5,
		    0xde, 0xe8, 0x15, 0xb3, 0xe9, 0x13, 0x2e, 0x5b,
		    0x59, 0x2f, 0x3a, 0x24, 0xb2, 0xce, 0xfa, 0x94,
		    0xe2, 0xc7, 0x69, 0x05, 0xa5, 0xc4, 0xb2, 0x30,
		    0xfb, 0x48, 0xd9, 0x75, 0x94, 0xf1, 0x85, 0xb5,
		    0xff, 0xe3, 0xe9, 0xd5, 0xa6, 0xcf, 0x12, 0x46,
		    0xc5, 0x9a, 0x70, 0xf3, 0xec, 0xbe, 0x2c, 0x24,
		    0xd2, 0xa2, 0x17, 0x67, 0x7a, 0x46, 0x8f, 0xb4,
		    0x96, 0xa4, 0xab, 0xf6, 0xab, 0xb4, 0x7b, 0x31,
		    0x6d, 0xf0, 0x37, 0x8f, 0x98, 0xda, 0x3c, 0x56,
		    0xd3, 0x5b, 0xe1, 0xd0, 0x57, 0x04, 0xf5, 0xcc,
		    0x0c, 0x20, 0x4b, 0xd2, 0x17, 0xed, 0x3c, 0x4b,
		    0x32, 0xa4, 0xd8, 0x87, 0xaf, 0xae, 0xf9, 0x45,
		    0x26, 0x9d, 0xc5, 0xe4, 0x61, 0x31, 0xf0, 0xde,
		    0x7b, 0x55, 0x1b, 0x7e, 0xc9, 0x01, 0x91, 0x07,
		    0xb0, 0x45, 0xf1, 0x1c, 0x75, 0xa9, 0xf5, 0x58,
		    0xbd, 0x4c, 0x49, 0x4e, 0xd8, 0x73, 0x37, 0x08,
		    0x92, 0x05, 0x87, 0xba, 0xa3, 0xc8, 0x14, 0xa0,
		    0xee, 0x21, 0xd7, 0xf6, 0x03, 0x37, 0xe3, 0xe0,
		    0x4f, 0xea, 0x85, 0x94, 0x1b, 0x45, 0x92, 0x0a,
		    0xa0, 0xea, 0xba, 0xc2, 0xb9, 0x88, 0x29, 0xe8,
		    0xbd, 0x81, 0xda, 0x95, 0x45, 0xf5, 0x5f, 0x20,
		    0x94, 0x80, 0x21, 0x36, 0x10, 0x6a, 0xb7, 0x90,
		    0xc0, 0xa9, 0x9e, 0x2a, 0x20, 0xa9, 0xd5, 0x61,
		    0xf4, 0x72, 0x56, 0xaf, 0x00, 0x9a, 0x33, 0xf6,
		    0x51, 0x12, 0xfc, 0x84, 0x15, 0x36, 0xc5, 0x27,
		    0x4f, 0x44, 0xdf, 0x64, 0xde, 0xd7, 0xe1, 0xe9,
		    0xc8, 0x08, 0xa3, 0x42, 0x60, 0xec, 0x1f, 0xbe,
		    0x9f, 0x7b, 0x79, 0x47, 0xff, 0xea, 0x6f, 0x4f,
		    0xc6, 0xea, 0x34, 0x54, 0x09, 0xb2, 0xa1, 0xde,
		    0xb4, 0x54, 0xb9, 0x82, 0x9e, 0x41, 0xad, 0xa6,
		    0x4f, 0x8f, 0xac, 0x3b, 0x01, 0x04, 0xbb, 0x9c,
		    0x6a, 0xac, 0x24, 0xc1, 0xa8, 0x91, 0xb7, 0x5b,
		    0x88, 0x88, 0xf3, 0x42, 0x17, 0xd4, 0xb7, 0xfc,
		    0xc8, 0x80, 0x1b, 0x54, 0x91, 0x47, 0xd8, 0x1b,
		    0xbc, 0x12, 0x7e, 0x4a, 0x03, 0x3b, 0x78, 0x83,
		    0x2c, 0x92, 0x92, 0xa4, 0x31, 0xf2, 0xe8, 0x13,
		    0x36, 0x0e, 0xc6, 0xb4, 0xed, 0x5c, 0x22, 0x71,
		    0xa0, 0xa9 }
}, {
	.key	= { 0x61, 0x93, 0x3b, 0xc8, 0x4b, 0x53, 0x4f, 0xc1,
		    0x9e, 0x74, 0x24, 0x7f, 0x00, 0xcb, 0xc8, 0x33,
		    0x42, 0x55, 0x6f, 0x77, 0xf1, 0x35, 0x52, 0xb0,
		    0x67, 0xe7, 0x1d, 0x6e, 0xf1, 0xf6, 0x1c, 0x4d },
	.nonce	= 0xa3030d135287782eULL,
	.input	= { 0x83, 0x01, 0x49, 0xa9, 0x1b, 0x38, 0x85, 0x91,
		    0x26, 0x41, 0x91, 0x20, 0xb2, 0x9e, 0x98, 0xaf,
		    0x27, 0x30, 0xe1, 0xa6, 0x44, 0x4d, 0x2a, 0xaa,
		    0xb3, 0x50, 0x7b, 0x27, 0x12, 0x86, 0x8b, 0xba,
		    0x18, 0xaf, 0xc8, 0x8c, 0x68, 0x1b, 0x51, 0x1e,
		    0x99, 0x89, 0x15, 0xcb, 0xd1, 0x5e, 0xaa, 0xe6,
		    0x7f, 0x0d, 0x78, 0x7a, 0x66, 0xfc, 0xb5, 0x09,
		    0x0d, 0x6e, 0xeb, 0xc1, 0x8b, 0x92, 0x5f, 0x94,
		    0x83, 0xb5, 0x3e, 0x5a, 0x73, 0x28, 0xca, 0xb4,
		    0x99, 0x9d, 0xe1, 0x17, 0x64, 0x12, 0x59, 0xb7,
		    0xbe, 0x50, 0xf1, 0x52, 0xd3, 0x34, 0x93, 0x3c,
		    0xa3, 0xbc, 0xc2, 0x00, 0x86, 0x7e, 0x40, 0x9e,
		    0xd6, 0x65, 0xf8, 0xf3, 0x8a, 0x60, 0x7c, 0x63,
		    0x01, 0xc7, 0x1a, 0xbd, 0xe5, 0x0c, 0x91, 0xfe,
		    0x6b, 0x6c, 0xae, 0xca, 0x9b, 0x79, 0xa6, 0x07,
		    0xa3, 0x00, 0xc0, 0x63, 0x3b, 0xfb, 0xc6, 0xa5,
		    0xeb, 0x36, 0xd5, 0xaa, 0xc5, 0x28, 0xed, 0x37,
		    0x04, 0x90, 0xd2, 0x4d, 0x08, 0x5e, 0x23, 0x8b,
		    0xe7, 0xf3, 0x93, 0xf4, 0x91, 0xc6, 0xcf, 0x13,
		    0x24, 0x0a, 0x2d, 0x6a, 0x81, 0x3b, 0xaa, 0xd9,
		    0x4c, 0xe0, 0x57, 0xe1, 0x11, 0x62, 0xf5, 0x58,
		    0x7f, 0xcd, 0x7b, 0xe6, 0xa8, 0xc3, 0x5a, 0xd5,
		    0xc3, 0xd0, 0x32, 0xcf, 0x14, 0x10, 0x4c, 0x79,
		    0x2d, 0x67, 0x4d, 0x1c, 0xe7, 0x93, 0x2e, 0x1f,
		    0x9c, 0x51, 0x31, 0xb9, 0x3c, 0x8b, 0x00, 0x72,
		    0x95, 0x16, 0xb8, 0x5e, 0xf6, 0x6c, 0xfa, 0xe6,
		    0x7c, 0xbb, 0xa6, 0xef, 0x73, 0x4b, 0xcd, 0x10,
		    0x32, 0x6b, 0x0d, 0xcb, 0x82, 0x3c, 0x1a, 0x2d,
		    0xde, 0x7a, 0xc8, 0xcc, 0x23, 0xce, 0x06, 0x6a,
		    0x92, 0x65, 0xe7, 0x00, 0xe3, 0xe5, 0xf1, 0x46,
		    0x9e, 0x6a, 0xe8, 0xfb, 0x8c, 0x2c, 0x52, 0x4b,
		    0x9a, 0x0a, 0x66, 0x26, 0xdc, 0x6f, 0x25, 0x86,
		    0x2b, 0x63, 0x6b, 0x57, 0x4f, 0x62, 0xb2, 0xe6,
		    0x5c, 0xbf, 0x55, 0xd9, 0x65, 0x88, 0x8e, 0xb5,
		    0xc1, 0xfb, 0x0e, 0x80, 0xb5, 0x76, 0x0b, 0x1f,
		    0xe6, 0x49, 0x8b, 0x60, 0xef, 0xad, 0xb6, 0x47,
		    0xc2, 0x0e, 0x4c, 0xc7, 0xf8, 0xe6, 0xcb, 0x83,
		    0x41, 0xf8, 0xd6, 0x72, 0x17, 0xfc, 0x62, 0x03,
		    0x9f, 0x42, 0x38, 0x3d, 0xe4, 0x26, 0x2b, 0x8c,
		    0xf8, 0xa4, 0x1d, 0x05, 0x7b, 0x03, 0x3f, 0x64,
		    0xec, 0x64, 0x77, 0x29, 0xdd, 0xb3, 0x02, 0xf9,
		    0xb3, 0x0c, 0x60, 0x3d, 0xcb, 0x4d, 0x61, 0x8c,
		    0xf2, 0x07, 0x2b, 0x0d, 0x07, 0x67, 0xf7, 0xd6,
		    0x7e, 0xf1, 0xe8, 0x0e, 0xbb, 0x93, 0x9b, 0x3a,
		    0xe5, 0xe1, 0x03, 0x75, 0x66, 0xe0, 0x10, 0x7e,
		    0x7b, 0x35, 0x54, 0xc1, 0xd4, 0xec, 0xfa, 0x66,
		    0x55, 0x22, 0x22, 0x6a, 0xee, 0x30, 0x7c, 0x8f,
		    0xc9, 0x5d, 0xb9, 0x1f, 0x86, 0x5c, 0x7d, 0x71,
		    0xc5, 0x2b, 0x74, 0x0a, 0x24, 0x15, 0x8f, 0x91,
		    0x7b, 0xb0, 0x6b, 0x0b, 0x02, 0x84, 0x92, 0x4a,
		    0x68, 0x22, 0x63, 0xb4, 0xf1, 0x4f, 0x94, 0x69,
		    0x25, 0x24, 0x87, 0xef, 0x12, 0x36, 0xb9, 0x75,
		    0x96, 0x42, 0x65, 0xb6, 0x6d, 0x8b, 0x5d, 0x7b,
		    0x99, 0x0c, 0xc8, 0xbb, 0x87, 0x83, 0x21, 0xe4,
		    0xb0, 0xe4, 0xa5, 0x76, 0x0c, 0xe6, 0x2d, 0x70,
		    0x8e, 0xb2, 0xf9, 0xd0, 0xa8, 0xf2, 0x93, 0xfa,
		    0x7d, 0xf4, 0x08, 0x58, 0x1d, 0x53, 0xee, 0x1c,
		    0x74, 0xdf, 0x83, 0x16, 0xbb, 0x84, 0x30, 0xe0,
		    0x06, 0xf6, 0x4b, 0x05, 0x03, 0xd8, 0x7d, 0xbc,
		    0x04, 0xc8, 0x44, 0x2a, 0x5d, 0xe1, 0xdc, 0x06,
		    0xea, 0x61, 0xb7, 0xde, 0xea, 0x27, 0x0f, 0x74,
		    0x9a, 0x6d, 0xed, 0x1f, 0x03, 0x11, 0xd3, 0xca,
		    0xf0, 0x6c, 0x3c, 0x62, 0x5c, 0x6a, 0x64, 0xa2,
		    0x1d, 0x1b, 0xec, 0x21, 0xd0, 0xaa, 0x92, 0xdb,
		    0x89, 0xac, 0x19, 0xd2, 0xd6, 0xd3, 0x52, 0xa4,
		    0x36, 0x5b, 0x66, 0xe2, 0xea, 0xa3, 0xe8, 0x6a,
		    0x1f, 0x8f, 0xaf, 0x6d, 0xfe, 0xcc, 0x5d, 0x9f,
		    0x88, 0x70, 0xe7, 0x41, 0xa1, 0xb5, 0x3e, 0x6a,
		    0x2a, 0x94, 0xc5, 0x33, 0x71, 0xb9, 0xea, 0x74,
		    0xf4, 0x46, 0x8f, 0x71, 0x0e, 0x8b, 0x84, 0x1b,
		    0xaf, 0xc8, 0x50, 0x74, 0x60, 0xe1, 0xf2, 0x44,
		    0xc1, 0xac, 0x77, 0x50, 0x93, 0x4e, 0xb1, 0x3c,
		    0xd3, 0xcf, 0x0c, 0xdf, 0xc1, 0xb9, 0x28, 0x4f,
		    0xe1, 0x6b, 0xd6, 0xeb, 0x88, 0x57, 0x1c, 0x98,
		    0xa6, 0xc7, 0x4c, 0xf6, 0xb6, 0x27, 0x9d, 0xfb,
		    0xc5, 0xc2, 0x7c, 0xf3, 0xa7, 0x29, 0x72, 0xca,
		    0x3f, 0xf3, 0x56, 0x42, 0xca, 0x50, 0xaf, 0x76,
		    0x88, 0xd1, 0x7d, 0x88, 0x91, 0x0d, 0x80, 0x05,
		    0x4f, 0xc1, 0xa4, 0xe4, 0xd5, 0x17, 0xd6, 0xa7,
		    0xd3, 0xb7, 0x85, 0x44, 0x80, 0x70, 0xee, 0x6a,
		    0x27, 0x4d, 0x52, 0xd8, 0xfb, 0x7b, 0x51, 0x6c,
		    0x44, 0x8f, 0x44, 0x4d, 0x74, 0x11, 0x65, 0xac,
		    0x04, 0x4a, 0x4e, 0xef, 0x98, 0x89, 0x20, 0x09,
		    0x34, 0x19, 0xfe, 0x38, 0x29, 0xd4, 0x53, 0xee,
		    0xf9, 0x37, 0x1f, 0xc0, 0x2a, 0x5e, 0xa4, 0x3e,
		    0x3f, 0x30, 0xd4, 0xaf, 0xa5, 0x3c, 0x77, 0xe4,
		    0xf5, 0xf7, 0xad, 0x55, 0xd8, 0xc9, 0x90, 0x00,
		    0x03, 0xa3, 0x9e, 0xb1, 0x84, 0xb0, 0xbb, 0x46,
		    0x9a, 0xdd, 0xe7, 0xdd, 0xaa, 0x8a, 0xa2, 0xd7,
		    0xa8, 0x03, 0x63, 0xf4, 0x16, 0x7c, 0x3e, 0xa0,
		    0xa8, 0x32, 0x9e, 0x3f, 0x16, 0xf0, 0xa5, 0x7a,
		    0xff, 0x44, 0xfc, 0x60, 0xb1, 0x65, 0xf4, 0xf5,
		    0xb1, 0x11, 0x7c, 0xe5, 0x0d, 0x78, 0xdc, 0x1a,
		    0xc3, 0xe0, 0x3f, 0x8c, 0x5e, 0x5a, 0x6b, 0x2f,
		    0x11, 0xbe, 0xc9, 0xff, 0xe6, 0x84, 0xbf, 0x62,
		    0x93, 0xdb, 0xe7, 0xe6, 0x30, 0x56, 0x99, 0x4e,
		    0x85 },
	.ilen	= 769,
	.result	= { 0x49, 0xbd, 0x98, 0x31, 0xb8, 0x44, 0x59, 0x02,
		    0x7e, 0x1e, 0x7a, 0x4f, 0x77, 0x71, 0xd8, 0x5d,
		    0xef, 0x8d, 0xcb, 0xa4, 0x61, 0x76, 0xab, 0xd8,
		    0x07, 0x75, 0x86, 0x51, 0x5b, 0xcf, 0xc3, 0xd8,
		    0x6c, 0x36, 0xe9, 0x1f, 0x24, 0x48, 0xe1, 0xca,
		    0xff, 0x73, 0x0d, 0x30, 0xc1, 0xff, 0x8f, 0x6c,
		    0x51, 0xf0, 0x58, 0x7f, 0x1c, 0xba, 0x94, 0x16,
		    0x59, 0x81, 0x7f, 0x04, 0xcf, 0x58, 0xa0, 0x5f,
		    0x64, 0x35, 0x55, 0x3e, 0x5d, 0x2e, 0x45, 0x55,
		    0x71, 0xd1, 0xc9, 0x43, 0x9d, 0x3a, 0xf2, 0x93,
		    0xed, 0x05, 0x25, 0x2e, 0xc0, 0x8c, 0x09, 0x0b,
		    0x06, 0x3a, 0x89, 0x26, 0xd3, 0x32, 0x94, 0xc7,
		    0x66, 0xd1, 0x13, 0xb0, 0x12, 0xd9, 0x99, 0x37,
		    0x08, 0x14, 0xe4, 0xd5, 0x32, 0x07, 0x3d, 0xcc,
		    0xa6, 0xc4, 0x25, 0x2f, 0x49, 0x60, 0xa9, 0x78,
		    0x4c, 0x1c, 0x42, 0xdb, 0x46, 0xa0, 0x6e, 0x84,
		    0x09, 0x01, 0xe8, 0x69, 0xb0, 0x6e, 0xc4, 0x3c,
		    0xd9, 0x53, 0x5a, 0x8e, 0xf0, 0xe5, 0xc6, 0x73,
		    0x7d, 0x46, 0x8c, 0x49, 0x6e, 0xfd, 0x7e, 0x7f,
		    0xb1, 0xb6, 0xd6, 0x4e, 0x94, 0xce, 0xe1, 0x55,
		    0x9b, 0x8f, 0x96, 0x46, 0x62, 0x43, 0x59, 0x9c,
		    0x8f, 0x07, 0x85, 0xad, 0xac, 0x53, 0x54, 0xda,
		    0xce, 0x99, 0xb8, 0x8d, 0x23, 0x5f, 0x15, 0xcd,
		    0x13, 0x00, 0xf8, 0xc4, 0x73, 0x74, 0xef, 0x6e,
		    0x69, 0x9d, 0x10, 0x7e, 0x32, 0xaa, 0xab, 0x72,
		    0xd9, 0xa0, 0x35, 0x96, 0xbf, 0xbe, 0xae, 0xf5,
		    0x1d, 0xbc, 0x73, 0x2b, 0x07, 0x0c, 0x1d, 0xb5,
		    0xfe, 0x9d, 0xc2, 0xda, 0x57, 0x0e, 0xcb, 0x2c,
		    0x6a, 0x47, 0xab, 0x27, 0xd5, 0x02, 0xc4, 0x4c,
		    0xc4, 0xb2, 0x1d, 0xca, 0x0d, 0x13, 0xa3, 0x2c,
		    0x93, 0x23, 0x7d, 0xc9, 0xe9, 0x6b, 0x39, 0xaa,
		    0xba, 0x24, 0xaf, 0x09, 0x42, 0x5c, 0xfe, 0x63,
		    0xa2, 0x0e, 0x15, 0x0b, 0x06, 0xf8, 0x1a, 0xfc,
		    0x9c, 0x03, 0x94, 0x90, 0x0d, 0x2b, 0xdc, 0x7a,
		    0x08, 0xf3, 0xdd, 0xc6, 0x34, 0xac, 0x2a, 0xfb,
		    0xd1, 0x5b, 0x04, 0x27, 0x4f, 0x63, 0x76, 0xbb,
		    0x03, 0x73, 0xb5, 0x9c, 0x49, 0x8b, 0xb5, 0xe7,
		    0x23, 0xb9, 0x79, 0xf6, 0x4b, 0x48, 0xe4, 0x74,
		    0x7c, 0xa4, 0xb0, 0x6b, 0xbf, 0x0c, 0x61, 0x9e,
		    0x19, 0x57, 0xaa, 0x38, 0x40, 0xbf, 0x7b, 0x33,
		    0x28, 0xeb, 0x96, 0x3d, 0xc2, 0xb6, 0x2e, 0x3e,
		    0xc7, 0xc9, 0xb5, 0x7b, 0x3e, 0x2c, 0x79, 0x94,
		    0x10, 0xb5, 0xcd, 0x5b, 0x52, 0xef, 0x57, 0x41,
		    0x98, 0xae, 0x51, 0x6b, 0x0d, 0x4c, 0x42, 0xe3,
		    0x9e, 0x62, 0x25, 0xd1, 0x71, 0x19, 0x73, 0x4f,
		    0x5e, 0x52, 0x76, 0x8e, 0x5f, 0xd7, 0xa2, 0xa9,
		    0xd2, 0x60, 0x44, 0x81, 0x87, 0xe8, 0x2c, 0xb2,
		    0xf1, 0xa7, 0x2a, 0x2b, 0x40, 0xf9, 0x0a, 0x01,
		    0x83, 0xfc, 0x8d, 0xcc, 0x83, 0x28, 0xf4, 0xad,
		    0x6e, 0xcb, 0xe1, 0x02, 0x89, 0xd7, 0x14, 0x73,
		    0xd7, 0x56, 0x1e, 0x41, 0xf3, 0x0b, 0xa4, 0xf2,
		    0x02, 0x8b, 0xc1, 0x9e, 0xe6, 0xd1, 0x89, 0xbd,
		    0x20, 0xf3, 0x1a, 0xfc, 0x8e, 0x7c, 0xde, 0xb9,
		    0x2a, 0x51, 0x67, 0xeb, 0x0e, 0x32, 0x3f, 0x88,
		    0xc1, 0x26, 0xd9, 0xb8, 0x6e, 0xb0, 0x20, 0x4c,
		    0xfa, 0x4b, 0xfa, 0x21, 0xd9, 0xd3, 0x52, 0xdd,
		    0xbc, 0x20, 0x91, 0xe7, 0x45, 0x03, 0xe7, 0x3e,
		    0x40, 0xb0, 0x76, 0xc7, 0xb5, 0xec, 0xb3, 0x67,
		    0x8f, 0xba, 0xfd, 0xf3, 0x15, 0x81, 0xdc, 0x2f,
		    0x03, 0x8d, 0xf0, 0x69, 0x9f, 0xd6, 0xda, 0x8f,
		    0x98, 0xa0, 0x04, 0xb3, 0x73, 0x7b, 0xf2, 0xbd,
		    0xf2, 0x9e, 0xc8, 0x30, 0x0f, 0x40, 0x65, 0x1f,
		    0x67, 0xda, 0x7d, 0x56, 0x53, 0x66, 0xff, 0xe9,
		    0x15, 0xe4, 0x4b, 0x61, 0x26, 0x62, 0x8f, 0x5a,
		    0xd0, 0xe7, 0xc4, 0x53, 0x5f, 0x96, 0x4d, 0x82,
		    0x50, 0x61, 0x64, 0x1b, 0xe3, 0xe3, 0xbb, 0x6b,
		    0x4f, 0x1d, 0x58, 0xf2, 0xde, 0xfd, 0x59, 0x0c,
		    0x62, 0xa9, 0xc3, 0x65, 0x45, 0xd8, 0x82, 0x45,
		    0xaa, 0x06, 0x4c, 0xc1, 0x6e, 0x95, 0x9a, 0x7a,
		    0x2b, 0x1c, 0x91, 0xc1, 0x87, 0x45, 0x47, 0x71,
		    0xca, 0xea, 0x0c, 0xa6, 0x70, 0x08, 0x2d, 0x1b,
		    0x26, 0x38, 0xaa, 0x41, 0x0b, 0xe7, 0xa1, 0xe6,
		    0x6e, 0xa4, 0x7c, 0x8d, 0xf1, 0x1a, 0xd8, 0x05,
		    0xb3, 0x69, 0x0b, 0x06, 0xc4, 0x49, 0x84, 0x2d,
		    0x08, 0xe6, 0x57, 0xe9, 0x64, 0x02, 0xd2, 0xc0,
		    0x73, 0xf1, 0x6d, 0x88, 0x07, 0x48, 0x47, 0x7c,
		    0xe4, 0x42, 0x4b, 0x5a, 0x65, 0xb4, 0xc3, 0x65,
		    0xcc, 0xd6, 0xc6, 0x5d, 0xc7, 0x21, 0xc8, 0x65,
		    0x28, 0x96, 0xd6, 0xad, 0x2b, 0xc7, 0x49, 0xbe,
		    0x98, 0xb3, 0x43, 0xb8, 0x9a, 0xd1, 0xe9, 0xa0,
		    0xed, 0x8d, 0x8c, 0x2c, 0x00, 0xa6, 0x1e, 0x3a,
		    0xb5, 0x0d, 0xeb, 0xcf, 0x12, 0xe6, 0x0e, 0x4f,
		    0x7d, 0x87, 0xca, 0x3d, 0x37, 0xaf, 0xb4, 0xa3,
		    0xdc, 0xd4, 0xf7, 0xe2, 0x7e, 0xfc, 0xa7, 0x1b,
		    0x92, 0x79, 0x63, 0x10, 0xce, 0x7b, 0x59, 0x87,
		    0x89, 0x49, 0xb7, 0x50, 0x86, 0x7c, 0xa7, 0x5f,
		    0x3f, 0x50, 0x31, 0x46, 0xd4, 0xfa, 0x6e, 0xcd,
		    0xf4, 0xbe, 0xce, 0xbe, 0xd0, 0x71, 0xef, 0x2c,
		    0xfe, 0xf1, 0x3c, 0xf0, 0xfd, 0xef, 0x96, 0x73,
		    0xa3, 0xa7, 0x23, 0x35, 0x6c, 0x27, 0xef, 0x78,
		    0xec, 0xdd, 0x46, 0xb7, 0xe4, 0xce, 0x37, 0x43,
		    0x35, 0xf2, 0x38, 0x35, 0xf5, 0xb5, 0xc2, 0x9f,
		    0xf0, 0x97, 0x52, 0x23, 0x0c, 0x74, 0x03, 0xd1,
		    0x53, 0xd9, 0xf0, 0x73, 0xb6, 0xae, 0x4c, 0xa3,
		    0xac, 0x6f, 0x4c, 0x93, 0x72, 0x47, 0x7b, 0x44,
		    0x8d, 0xf9, 0xee, 0x24, 0x08, 0x95, 0x62, 0xbe,
		    0x31 }
}};

static bool __init chacha20_selftest(void)
{
	size_t i;
	u8 offset_input[MAXIMUM_TEST_BUFFER_LEN + 1]
			__aligned(__alignof__(unsigned long));
	u8 offset_key[CHACHA20_KEY_SIZE + 1]
			__aligned(__alignof__(unsigned long));
	u8 computed_result[MAXIMUM_TEST_BUFFER_LEN + 1]
			__aligned(__alignof__(unsigned long));
	struct chacha20_ctx state;
	bool success = true;
	simd_context_t simd_context;

	simd_get(&simd_context);
	for (i = 0; i < ARRAY_SIZE(chacha20_vectors); ++i) {
		memset(computed_result, 0, sizeof(computed_result));
		memset(&state, 0, sizeof(state));
		chacha20_init(&state, chacha20_vectors[i].key,
			      chacha20_vectors[i].nonce);
		chacha20(&state, computed_result, chacha20_vectors[i].input,
			 chacha20_vectors[i].ilen, &simd_context);
		if (memcmp(computed_result, chacha20_vectors[i].result,
			   chacha20_vectors[i].ilen)) {
			pr_info("chacha20 self-test %zu: FAIL\n", i + 1);
			success = false;
		}
		memset(computed_result, 0, sizeof(computed_result));
		memset(&state, 0, sizeof(state));
		memcpy(offset_input + 1, chacha20_vectors[i].input,
		       chacha20_vectors[i].ilen);
		memcpy(offset_key + 1, chacha20_vectors[i].key,
		       CHACHA20_KEY_SIZE);
		chacha20_init(&state, offset_key + 1, chacha20_vectors[i].nonce);
		chacha20(&state, computed_result + 1, offset_input + 1,
			 chacha20_vectors[i].ilen, &simd_context);
		if (memcmp(computed_result + 1, chacha20_vectors[i].result,
			   chacha20_vectors[i].ilen)) {
			pr_info("chacha20 self-test %zu (unaligned): FAIL\n", i + 1);
			success = false;
		}
	}
	simd_put(&simd_context);
	if (success)
		pr_info("chacha20 self-tests: pass\n");
	return success;
}
#endif
