! WHIZARD 2.2.8 Nov 22 2015
! 
! Copyright (C) 1999-2015 by 
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!     
!     with contributions from
!     Fabian Bach <fabian.bach@t-online.de>
!     Bijan Chokoufe <bijan.chokoufe@desy.de>
!     Christian Speckner <cnspeckn@googlemail.com> 
!     Soyoung Shim <soyoung.shim@desy.de>
!     Florian Staub <florian.staub@cern.ch>  
!     Christian Weiss <christian.weiss@desy.de>
!     and Hans-Werner Boschmann, Felix Braam, 
!     Sebastian Schmidt, So-young Shim, Daniel Wiesler 
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by 
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module blha_config

  use kinds
  use iso_varying_string, string_t => varying_string
  use io_units
  use constants
  use string_utils
  use system_defs, only: EOF
  use diagnostics
  use md5
  use model_data
  use flavors
  use quantum_numbers
  use pdg_arrays
  use sorting
  use lexers
  use parser
  use syntax_rules
  use ifiles

  use beam_structures, only: beam_structure_t

  implicit none
  private

  public :: blha_configuration_t
  public :: blha_cfg_process_node_t
  public :: blha_flv_state_t
  public :: blha_master_t
  public :: blha_configuration_init
  public :: blha_configuration_append_processes
  public :: blha_configuration_set
  public :: blha_configuration_get_n_proc
  public :: blha_configuration_write

  integer, public, parameter :: &
       BLHA_CT_QCD=1, BLHA_CT_EW=2, BLHA_CT_QED=3, BLHA_CT_OTHER=4
  integer, public, parameter :: &
       BLHA_IRREG_CDR=1, BLHA_IRREG_DRED=2, BLHA_IRREG_THV=3, &
       BLHA_IRREG_MREG=4, BLHA_IRREG_OTHER=5
  integer, public, parameter :: &
       BLHA_MPS_ONSHELL=1, BLHA_MPS_OTHER=2
  integer, public, parameter :: &
       BLHA_MODE_GOSAM=1, BLHA_MODE_FEYNARTS = 2, BLHA_MODE_GENERIC=3, &
       BLHA_MODE_OPENLOOPS=4
  integer, public, parameter :: &
       BLHA_VERSION_1 = 1, BLHA_VERSION_2 = 2
  integer, public, parameter :: &
       BLHA_AMP_LOOP = 1, BLHA_AMP_CC = 2, BLHA_AMP_SC = 3, &
       BLHA_AMP_TREE = 4, BLHA_AMP_LOOPINDUCED = 5
  integer, public, parameter :: &
       BLHA_EW_GF = 1, BLHA_EW_MZ = 2, BLHA_EW_MSBAR = 3, &
       BLHA_EW_0 = 4, BLHA_EW_RUN = 5, BLHA_EW_DEFAULT = 6
  integer, public, parameter :: &
       BLHA_WIDTH_COMPLEX = 1, BLHA_WIDTH_FIXED = 2, &
       BLHA_WIDTH_RUNNING = 3, BLHA_WIDTH_POLE = 4, &
       BLHA_WIDTH_DEFAULT = 5 

  integer, parameter, public :: OLP_N_MASSIVE_PARTICLES = 10
  integer, dimension(OLP_N_MASSIVE_PARTICLES), public :: &
    OLP_MASSIVE_PARTICLES = [5,-5,6,-6,15,-15,23,24,-24,25]
  integer, parameter :: OLP_HEL_UNPOLARIZED = 0
  integer, parameter :: OLP_HEL_LEFT = -1
  integer, parameter :: OLP_HEL_RIGHT = 1
  integer, parameter :: OLP_HEL_LONG = 2


  type :: blha_particle_string_element_t
     integer :: pdg = 0
     integer :: hel = OLP_HEL_UNPOLARIZED
     logical :: polarized = .false. 
  contains
    generic :: init => init_default
    generic :: init => init_polarized
    procedure :: init_default => blha_particle_string_element_init_default
    procedure :: init_polarized => blha_particle_string_element_init_polarized
    generic :: write_pdg => write_pdg_unit
    generic :: write_pdg => write_pdg_character
    procedure :: write_pdg_unit => blha_particle_string_element_write_pdg_unit
    procedure :: write_pdg_character &
       => blha_particle_string_element_write_pdg_character
    generic :: write_helicity => write_helicity_unit
    generic :: write_helicity => write_helicity_character
    procedure :: write_helicity_unit &
       => blha_particle_string_element_write_helicity_unit
    procedure :: write_helicity_character &
       => blha_particle_string_element_write_helicity_character
  end type blha_particle_string_element_t

  type :: blha_cfg_process_node_t
     type(blha_particle_string_element_t), dimension(:), allocatable :: pdg_in, pdg_out
     integer, dimension(:), allocatable :: fingerprint
     integer :: nsub
     integer, dimension(:), allocatable :: ids
     integer :: amplitude_type
     type(blha_cfg_process_node_t), pointer :: next => null ()
  end type blha_cfg_process_node_t

  type :: blha_configuration_t
     type(string_t) :: name
     class(model_data_t), pointer :: model => null ()
     type(string_t) :: md5
     integer :: version = 2
     logical :: dirty = .false.
     integer :: n_proc = 0
     real(default) :: accuracy_target
     logical :: debug_unstable = .false.
     integer :: mode = BLHA_MODE_GENERIC
     logical :: polarized = .false.
     type(blha_cfg_process_node_t), pointer :: processes => null ()
     !integer, dimension(2) :: matrix_element_square_type = BLHA_MEST_SUM
     integer :: correction_type = BLHA_CT_QCD
     type(string_t) :: correction_type_other
     integer :: irreg = BLHA_IRREG_THV
     type(string_t) :: irreg_other
     integer :: massive_particle_scheme = BLHA_MPS_ONSHELL
     type(string_t) :: massive_particle_scheme_other
     type(string_t) :: model_file
     logical :: subdivide_subprocesses = .false.
     integer :: alphas_power = -1, alpha_power = -1
     integer :: ew_scheme = BLHA_EW_DEFAULT
     integer :: width_scheme = BLHA_WIDTH_DEFAULT
     integer :: openloops_phs_tolerance = 0
     logical :: openloops_top_signal = .false.
  end type blha_configuration_t

  type:: blha_flv_state_t
    integer, dimension(:), allocatable :: flavors
    integer :: flv_mult
    logical :: flv_real = .false.
  end type blha_flv_state_t

  type :: blha_master_t
    integer, dimension(4) :: blha_mode
    integer :: n_in, n_out
    logical :: compute_borns = .false.
    logical :: compute_real_trees = .false.
    logical :: compute_loops = .true.
    logical :: compute_correlations = .false.
    integer :: alpha_power, alphas_power
    type(string_t) :: basename
    type(string_t), dimension(:), allocatable :: suffix
    type(blha_configuration_t), dimension(:), allocatable :: blha_cfg
    integer :: n_files = 0
  contains
    procedure :: set_methods => blha_master_set_methods
    procedure :: allocate_config_files => blha_master_allocate_config_files
    procedure :: init => blha_master_init 
    procedure :: setup_additional_features => blha_master_setup_additional_features
    procedure :: set_gosam => blha_master_set_gosam
    procedure :: set_openloops => blha_master_set_openloops
    procedure :: reset_olp_modes => blha_master_reset_olp_modes
    procedure :: set_polarization => blha_master_set_polarization
    procedure :: generate => blha_master_generate
    procedure :: final => blha_master_final
  end type blha_master_t






contains

  subroutine blha_particle_string_element_init_default (blha_p, id)
    class(blha_particle_string_element_t), intent(out) :: blha_p
    integer, intent(in) :: id
    blha_p%pdg = id
  end subroutine blha_particle_string_element_init_default

  subroutine blha_particle_string_element_init_polarized (blha_p, id, hel)
    class(blha_particle_string_element_t), intent(out) :: blha_p
    integer, intent(in) :: id, hel
    blha_p%polarized = .true.
    blha_p%pdg = id
    blha_p%hel = hel
  end subroutine blha_particle_string_element_init_polarized

  subroutine blha_particle_string_element_write_pdg_unit (blha_p, unit)
    class(blha_particle_string_element_t), intent(in) :: blha_p
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit)
    write (u, '(I3)') blha_p%pdg
  end subroutine blha_particle_string_element_write_pdg_unit

  subroutine blha_particle_string_element_write_pdg_character (blha_p, c)
    class(blha_particle_string_element_t), intent(in) :: blha_p
    character(3), intent(inout) :: c
    write (c, '(I3)') blha_p%pdg
  end subroutine blha_particle_string_element_write_pdg_character

  subroutine blha_particle_string_element_write_helicity_unit (blha_p, unit)
    class(blha_particle_string_element_t), intent(in) :: blha_p
    integer, intent(in), optional :: unit
    integer :: u
    u = given_output_unit (unit)
    write (u, '(A1,I0,A1)') '(', blha_p%hel, ')'
  end subroutine blha_particle_string_element_write_helicity_unit

  subroutine blha_particle_string_element_write_helicity_character (blha_p, c)
    class(blha_particle_string_element_t), intent(in) :: blha_p
    character(4), intent(inout) :: c
    write (c, '(A1,I0,A1)') '(', blha_p%hel, ')'
  end subroutine blha_particle_string_element_write_helicity_character

  subroutine blha_master_set_methods (master, cmp_born, &
     cmp_real, cmp_loop, cmp_corr)
    class(blha_master_t), intent(inout) :: master
    logical, intent(in) :: cmp_born, cmp_loop, cmp_corr, cmp_real
    master%n_files = count ([cmp_born, cmp_real, cmp_loop, cmp_corr])
    master%compute_borns = cmp_born
    master%compute_real_trees = cmp_real
    master%compute_loops = cmp_loop
    master%compute_correlations = cmp_corr
  end subroutine blha_master_set_methods

  subroutine blha_master_allocate_config_files (master)
    class(blha_master_t), intent(inout) :: master
    allocate (master%blha_cfg (master%n_files))
    allocate (master%suffix (master%n_files))
  end subroutine blha_master_allocate_config_files

  subroutine blha_master_init (master, basename, model, &
       n_in, alpha_power, alphas_power, flv_born, flv_real)
    class(blha_master_t), intent(inout) :: master
    type(string_t), intent(in) :: basename
    class(model_data_t), intent(in), target :: model
    integer, intent(in) :: n_in
    integer, intent(in) :: alpha_power, alphas_power
    integer, dimension(:,:), allocatable, intent(in) :: &
         flv_born, flv_real
    integer :: n_proc_real, n_flv
    type(blha_flv_state_t), dimension(:), allocatable :: blha_flavor
    integer :: i_flv, i_file
    integer :: n_flv_born 

    if (master%n_files < 1) &
       call msg_fatal ("Attempting to generate OLP-files, but none are specified!")
    n_flv = 1; n_proc_real = 0
    n_flv_born = size (flv_born, 2)
    if (master%compute_real_trees) then
       if (allocated (flv_real)) then
         n_proc_real = size (flv_real, 2)
         n_flv = n_flv + n_proc_real
       end if
    end if
    i_file = 1          
    if (master%compute_loops) then
       if (allocated (flv_born)) then
          allocate (blha_flavor (size (flv_born, 2)))
          do i_flv = 1, size (flv_born, 2)
             allocate (blha_flavor(i_flv)%flavors (size (flv_born(:,i_flv))))
             blha_flavor(i_flv)%flavors = flv_born(:,i_flv)
             blha_flavor(i_flv)%flv_mult = 2
          end do
          master%suffix(i_file) = "_LOOP"
          call blha_init_virtual (master%blha_cfg(i_file), blha_flavor, &
               n_in, alpha_power, alphas_power, &
               basename, model, master%blha_mode(1))
          i_file = i_file + 1
        else
          call msg_fatal ("BLHA Loops requested but " &
                           // "Born flavor not existing")
        end if
    end if
    if (allocated (blha_flavor)) deallocate (blha_flavor)
    if (master%compute_correlations) then
       if (allocated (flv_born)) then
          allocate (blha_flavor (size (flv_born, 2)))
          do i_flv = 1, size (flv_born, 2)
             allocate (blha_flavor(i_flv)%flavors (size (flv_born(:,i_flv))))
             blha_flavor(i_flv)%flavors = flv_born(:,i_flv)
             blha_flavor(i_flv)%flv_mult = 3
          end do
          master%suffix(i_file) = "_SUB"
          call blha_init_subtraction (master%blha_cfg(i_file), blha_flavor, &
               n_in, alpha_power, alphas_power, &
               basename, model, master%blha_mode(2))
          i_file = i_file + 1
       else
          call msg_fatal ("BLHA Correlations requested but "&
                           // "Born flavor not existing")
       end if
    end if
    if (allocated (blha_flavor)) deallocate (blha_flavor)
    if (master%compute_real_trees) then
       if (allocated (flv_real)) then
          allocate (blha_flavor (size (flv_real, 2)))
          do i_flv = 1, size (flv_real, 2)
             allocate (blha_flavor(i_flv)%flavors (size (flv_real(:,i_flv))))
             blha_flavor(i_flv)%flavors = flv_real(:,i_flv)
             blha_flavor(i_flv)%flv_mult = 1
          end do
          master%suffix(i_file) = "_REAL"
          call blha_init_real (master%blha_cfg(i_file), blha_flavor, &
               n_in, alpha_power, alphas_power, &
               basename, model, master%blha_mode(3))
          i_file = i_file + 1
       else
          call msg_fatal ("BLHA Trees requested but "&
                           // "Real flavor not existing")
       end if
    end if
    if (allocated (blha_flavor)) deallocate (blha_flavor)
    if (master%compute_borns) then
       if (allocated (flv_born)) then
          allocate (blha_flavor (n_flv_born))
          do i_flv = 1, n_flv_born
             allocate (blha_flavor(i_flv)%flavors (size (flv_born(:,i_flv))))
             blha_flavor(i_flv)%flavors = flv_born(:,i_flv)
             blha_flavor(i_flv)%flv_mult = 1
          end do
          master%suffix(i_file) = "_BORN"
          call blha_init_born (master%blha_cfg(i_file), blha_flavor, &
               n_in, alpha_power, alphas_power, &
               basename, model, master%blha_mode(4))
       end if
    end if
  end subroutine blha_master_init

  subroutine blha_master_setup_additional_features (master, phs_tolerance, top_signal, beam_structure)
     class(blha_master_t), intent(inout) :: master
     integer, intent(in) :: phs_tolerance
     logical, intent(in), optional :: top_signal
     type(beam_structure_t), intent(in), optional :: beam_structure
     integer :: i_file
     logical :: polarized
     logical :: yorn

     yorn = .false.; if (present (top_signal)) yorn = top_signal     
     polarized = .false.
     if (present (beam_structure)) polarized = beam_structure%has_polarized_beams ()

     do i_file = 1, master%n_files
        if (phs_tolerance > 0) then
           select case (master%blha_mode(i_file))
           case (BLHA_MODE_GOSAM)
              if (polarized) &
                 call gosam_error_message () 
           case (BLHA_MODE_OPENLOOPS)
              master%blha_cfg(i_file)%openloops_phs_tolerance = phs_tolerance
              master%blha_cfg(i_file)%polarized = polarized
           end select
        end if
        master%blha_cfg(i_file)%openloops_top_signal = yorn
     end do
  contains
     subroutine gosam_error_message ()
        call msg_fatal ("You are trying to evaluate a process at NLO ", &
           [var_str ("which involves polarized beams using GoSam. "), &
            var_str ("This feature is not supported yet. "), &
            var_str ("Please use OpenLoops instead")])
     end subroutine gosam_error_message
  end subroutine blha_master_setup_additional_features

  subroutine blha_master_set_gosam (master, i)
    class(blha_master_t), intent(inout) :: master
    integer, intent(in) :: i
    master%blha_mode(i) = BLHA_MODE_GOSAM
  end subroutine blha_master_set_gosam

  subroutine blha_master_set_openloops (master, i)
    class(blha_master_t), intent(inout) :: master
    integer, intent(in) :: i
    master%blha_mode(i) = BLHA_MODE_OPENLOOPS
  end subroutine blha_master_set_openloops

  subroutine blha_master_reset_olp_modes (master, i)
    class(blha_master_t), intent(inout) :: master
    integer, intent(in) :: i
    master%blha_cfg(i)%mode = master%blha_mode(i)
  end subroutine blha_master_reset_olp_modes

  subroutine blha_master_set_polarization (master, i)
    class(blha_master_t), intent(inout) :: master
    integer, intent(in) :: i
    master%blha_cfg(i)%polarized = .true.
  end subroutine blha_master_set_polarization

  subroutine blha_init_born (blha_cfg, blha_flavor, n_in, &
        ap, asp, basename, model, blha_mode)
    type(blha_configuration_t), intent(inout) :: blha_cfg
    type(blha_flv_state_t), intent(in), dimension(:) :: blha_flavor
    integer, intent(in) :: n_in
    integer, intent(in) :: ap, asp
    type(string_t), intent(in) :: basename
    type(model_data_t), intent(in), target :: model
    integer, intent(in) :: blha_mode
    integer, dimension(:), allocatable :: amp_type
    integer :: i, ew_scheme

    allocate (amp_type (size (blha_flavor)))
    do i = 1, size (blha_flavor)
       amp_type(i) = BLHA_AMP_TREE
    end do
    call blha_configuration_init (blha_cfg, basename // "_BORN" , &
         model, blha_mode)
    call blha_configuration_append_processes (blha_cfg, n_in, &
         blha_flavor, amp_type)
    select case (blha_cfg%mode)
    case (BLHA_MODE_GOSAM)
       ew_scheme = BLHA_EW_0
    case (BLHA_MODE_OPENLOOPS)
       ew_scheme = BLHA_EW_0
    end select 
    call blha_configuration_set (blha_cfg, BLHA_VERSION_2, &
         correction_type = BLHA_CT_QCD, &
         irreg = BLHA_IRREG_CDR, alphas_power = asp, &
         alpha_power = ap, ew_scheme = ew_scheme, &
         debug = blha_mode == BLHA_MODE_GOSAM)
  end subroutine blha_init_born

  subroutine blha_init_virtual (blha_cfg, blha_flavor, n_in, &
     ap, asp, basename, model, blha_mode)
    type(blha_configuration_t), intent(inout) :: blha_cfg
    type(blha_flv_state_t), intent(in), dimension(:) :: blha_flavor
    integer, intent(in) :: n_in
    integer, intent(in) :: ap, asp
    type(string_t), intent(in) :: basename
    type(model_data_t), intent(in), target :: model
    integer, intent(in) :: blha_mode
    integer, dimension(:), allocatable :: amp_type
    integer :: i, ew_scheme

    allocate (amp_type (size (blha_flavor)*2))
    do i = 1, size (blha_flavor)
       amp_type(2*i-1) = BLHA_AMP_LOOP
       amp_type(2*i) = BLHA_AMP_CC
    end do
    call blha_configuration_init (blha_cfg, basename // "_LOOP" , &
         model, blha_mode)
    call blha_configuration_append_processes (blha_cfg, n_in, &
         blha_flavor, amp_type)
    select case (blha_cfg%mode)
    case (BLHA_MODE_GOSAM)
       ew_scheme = BLHA_EW_0
    case (BLHA_MODE_OPENLOOPS)
       ew_scheme = BLHA_EW_0
    end select 
    call blha_configuration_set (blha_cfg, BLHA_VERSION_2, &
         correction_type = BLHA_CT_QCD, &
         irreg = BLHA_IRREG_CDR, &
         alphas_power = asp, &
         alpha_power = ap, &
         ew_scheme = ew_scheme, &
         debug = blha_mode == BLHA_MODE_GOSAM)
  end subroutine blha_init_virtual

  subroutine blha_init_subtraction (blha_cfg, blha_flavor, n_in, &
     ap, asp, basename, model, blha_mode)
    type(blha_configuration_t), intent(inout) :: blha_cfg
    type(blha_flv_state_t), intent(in), dimension(:) :: blha_flavor
    integer, intent(in) :: n_in
    integer, intent(in) :: ap, asp
    type(string_t), intent(in) :: basename
    type(model_data_t), intent(in), target :: model
    integer, intent(in) :: blha_mode
    integer, dimension(:), allocatable :: amp_type
    integer :: i, ew_scheme

    allocate (amp_type (size (blha_flavor)*3))
    do i = 1, size (blha_flavor)
       amp_type(3*i-2) = BLHA_AMP_TREE
       amp_type(3*i-1) = BLHA_AMP_CC
       amp_type(3*i) = BLHA_AMP_SC
    end do
    call blha_configuration_init (blha_cfg, basename // "_SUB" , &
         model, blha_mode)
    call blha_configuration_append_processes (blha_cfg, n_in, &
         blha_flavor, amp_type)
    select case (blha_cfg%mode)
    case (BLHA_MODE_GOSAM)
       ew_scheme = BLHA_EW_0
    case (BLHA_MODE_OPENLOOPS)
       ew_scheme = BLHA_EW_0
    end select 
    call blha_configuration_set (blha_cfg, BLHA_VERSION_2, &
         correction_type = BLHA_CT_QCD, &
         irreg = BLHA_IRREG_CDR, &
         alphas_power = asp, &
         alpha_power = ap, &
         ew_scheme = ew_scheme, &
         debug = blha_mode == BLHA_MODE_GOSAM)
  end subroutine blha_init_subtraction

  subroutine blha_init_real (blha_cfg, blha_flavor, n_in, & 
     ap, asp, basename, model, blha_mode)
    type(blha_configuration_t), intent(inout) :: blha_cfg
    type(blha_flv_state_t), intent(in), dimension(:) :: blha_flavor
    integer, intent(in) :: n_in
    integer, intent(in) :: ap, asp
    type(string_t), intent(in) :: basename
    type(model_data_t), intent(in), target :: model
    integer, intent(in) :: blha_mode
    integer, dimension(:), allocatable :: amp_type
    integer :: i, ew_scheme

    allocate (amp_type (size (blha_flavor)))
    do i = 1, size (blha_flavor)
       amp_type(i) = BLHA_AMP_TREE
    end do
    call blha_configuration_init (blha_cfg, basename // "_REAL" , &
         model, blha_mode)
    call blha_configuration_append_processes (blha_cfg, n_in, &
         blha_flavor, amp_type)
    select case (blha_cfg%mode)
    case (BLHA_MODE_GOSAM)
       ew_scheme = BLHA_EW_0
    case (BLHA_MODE_OPENLOOPS)
       ew_scheme = BLHA_EW_0
    end select 
    call blha_configuration_set (blha_cfg, BLHA_VERSION_2, &
         correction_type = BLHA_CT_QCD, &
         irreg = BLHA_IRREG_CDR, &
         alphas_power = asp+1, &
         alpha_power = ap, &
         ew_scheme = ew_scheme, &
         ! debug = .true.)
         debug = blha_mode == BLHA_MODE_GOSAM)
  end subroutine blha_init_real

  subroutine blha_master_generate (master, basename)
    class(blha_master_t), intent(in) :: master
    type(string_t), intent(in) :: basename
    integer :: unit
    type(string_t) :: filename
    integer :: i_file
    do i_file = 1, master%n_files
       filename = basename // master%suffix(i_file) // ".olp"
       unit = free_unit ()
       open (unit, file = char (filename), status = 'replace', action = 'write')
       call blha_configuration_write (master%blha_cfg(i_file), unit)
       close (unit)
    end do
  end subroutine blha_master_generate

  subroutine blha_master_final (master)
    class(blha_master_t), intent(inout) :: master
    master%n_files = 0
    deallocate (master%suffix)
    deallocate (master%blha_cfg)
  end subroutine blha_master_final

  subroutine blha_configuration_init (cfg, name, model, mode)
    type(blha_configuration_t), intent(inout) :: cfg
    type(string_t), intent(in) :: name
    class(model_data_t), target, intent(in) :: model
    integer, intent(in), optional :: mode
    if (.not. associated (cfg%model)) then
       cfg%name = name
       cfg%model => model
    end if
    if (present (mode)) cfg%mode = mode
  end subroutine blha_configuration_init

  subroutine blha_configuration_get_massive_particles &
             (cfg, massive, i_massive)
    type(blha_configuration_t), intent(in) :: cfg
    logical, intent(out) :: massive
    integer, intent(out), dimension(:), allocatable :: i_massive
    integer, parameter :: max_particles = 10
    integer, dimension(max_particles) :: i_massive_tmp
    integer, dimension(max_particles) :: checked
    type(blha_cfg_process_node_t), pointer :: current_process 
    integer :: k
    integer :: n_massive
    n_massive = 0; k = 1
    checked = 0
    if (associated (cfg%processes)) then
       current_process => cfg%processes
    else
       call msg_fatal ("BLHA, massive particles: " // &
                       "No processes allocated!")
    end if
    do
       call check_pdg_list (current_process%pdg_in%pdg)
       call check_pdg_list (current_process%pdg_out%pdg)
       if (k > max_particles) &
          call msg_fatal ("BLHA, massive particles: " // &
                          "Max. number of particles exceeded!")
       if (associated (current_process%next)) then
          current_process => current_process%next
       else
          exit
       end if
    end do       
    if (n_massive > 0) then
       allocate (i_massive (n_massive))
       i_massive = i_massive_tmp (1:n_massive)
       massive = .true.
    else
       massive = .false.
    end if
  contains
    subroutine check_pdg_list (pdg_list)
       integer, dimension(:), intent(in) :: pdg_list
       integer :: i, i_pdg
       type(flavor_t) :: flv
       do i = 1, size (pdg_list)
          i_pdg = abs (pdg_list(i))
          call flv%init (i_pdg, cfg%model)
          if (flv%get_mass () > 0._default) then 
             !!! Avoid duplicates in output
             if (.not. any (checked == i_pdg)) then
                i_massive_tmp(k) = i_pdg
                checked(k) = i_pdg  
                k=k+1
                n_massive=n_massive+1
             end if
          end if
       end do
    end subroutine check_pdg_list
  end subroutine blha_configuration_get_massive_particles

  subroutine blha_configuration_append_processes (cfg, n_in, flavor, amp_type)
    type(blha_configuration_t), intent(inout) :: cfg
    integer, intent(in) :: n_in
    type(blha_flv_state_t), dimension(:), intent(in) :: flavor
    integer, dimension(:), intent(in), optional :: amp_type
    integer :: n_tot
    type(blha_cfg_process_node_t), pointer :: current_node
    integer :: i_process, i_flv
    integer, dimension(:), allocatable :: pdg_in, pdg_out
    integer, dimension(:), allocatable :: flavor_state
    integer :: proc_offset, n_proc_tot
    integer :: h1, h2
    proc_offset = 0; n_proc_tot = 0
    do i_flv = 1, size (flavor)
       n_proc_tot = n_proc_tot + flavor(i_flv)%flv_mult
    end do
    if (.not. associated (cfg%processes)) &
      allocate (cfg%processes)
    current_node => cfg%processes
    do i_flv = 1, size (flavor)
       n_tot = size (flavor(i_flv)%flavors)
       allocate (pdg_in (n_in), pdg_out (n_tot - n_in))
       allocate (flavor_state (n_tot))
       flavor_state = flavor(i_flv)%flavors
       do i_process = 1, flavor(i_flv)%flv_mult
          pdg_in = flavor_state (1:n_in)
          pdg_out = flavor_state (n_in+1:)
          if (cfg%polarized) then
             select case (cfg%mode)
             case (BLHA_MODE_OPENLOOPS)
                call allocate_and_init_pdg_and_helicities (current_node, &
                   pdg_in, pdg_out, amp_type (proc_offset+i_process))
             case (BLHA_MODE_GOSAM)
                !!! Nothing special for GoSam yet. This exception is already caught
                !!! in blha_master_setup_additional_features
             end select
          else
             call allocate_and_init_pdg (current_node, pdg_in, pdg_out, &
                amp_type (proc_offset + i_process)) 
          end if
          if (proc_offset+i_process /= n_proc_tot) then
            allocate (current_node%next)
            current_node => current_node%next
          end if
          if (i_process == flavor(i_flv)%flv_mult) &
             proc_offset = proc_offset + flavor(i_flv)%flv_mult
       end do
       deallocate (pdg_in, pdg_out)
       deallocate (flavor_state)
    end do
  contains
    subroutine allocate_and_init_pdg (node, pdg_in, pdg_out, amp_type)
      type(blha_cfg_process_node_t), intent(inout), pointer :: node
      integer, intent(in), dimension(:), allocatable :: pdg_in, pdg_out
      integer, intent(in) :: amp_type
      allocate (node%pdg_in (size (pdg_in)))
      allocate (node%pdg_out (size (pdg_out)))
      node%pdg_in%pdg = pdg_in
      node%pdg_out%pdg = pdg_out
      node%amplitude_type = amp_type  
    end subroutine allocate_and_init_pdg

    subroutine allocate_and_init_pdg_and_helicities (node, pdg_in, pdg_out, amp_type)
      type(blha_cfg_process_node_t), intent(inout), pointer :: node
      integer, intent(in), dimension(:), allocatable :: pdg_in, pdg_out
      integer, intent(in) :: amp_type
      integer :: h1, h2
      do h1 = -1, 1
         do h2 = -1, 1
            if (h1 == 0 .or. h2 == 0) cycle 
            call allocate_and_init_pdg (current_node, pdg_in, pdg_out, amp_type) 
            current_node%pdg_in(1)%polarized = .true.
            current_node%pdg_in(2)%polarized = .true.
            current_node%pdg_in(1)%hel = h1
            current_node%pdg_in(2)%hel = h2
            if (h1 + h2 /= 2) then
               allocate (current_node%next)
               current_node => current_node%next
            end if
         end do
      end do
    end subroutine allocate_and_init_pdg_and_helicities
      
  end subroutine blha_configuration_append_processes

  subroutine blha_configuration_set (cfg, &
       version, correction_type, irreg, massive_particle_scheme, &
       model_file, alphas_power, alpha_power, ew_scheme, width_scheme, &
       accuracy, debug)
    type(blha_configuration_t), intent(inout) :: cfg
    integer, optional, intent(in) :: version
    integer, optional, intent(in) :: correction_type
    integer, optional, intent(in) :: irreg
    integer, optional, intent(in) :: massive_particle_scheme
    type(string_t), optional, intent(in) :: model_file
    integer, optional, intent(in) :: alphas_power, alpha_power
    integer, optional, intent(in) :: ew_scheme
    integer, optional, intent(in) :: width_scheme
    real(default), optional, intent(in) :: accuracy
    logical, optional, intent(in) :: debug
    if (present (version)) &
       cfg%version = version
    if (present (correction_type)) &
       cfg%correction_type = correction_type
    if (present (irreg)) &
       cfg%irreg = irreg
    if (present (massive_particle_scheme)) &
       cfg%massive_particle_scheme = massive_particle_scheme
    if (present (model_file)) &
       cfg%model_file = model_file
    if (present (alphas_power)) &
       cfg%alphas_power = alphas_power
    if (present (alpha_power)) &
       cfg%alpha_power = alpha_power
    if (present (ew_scheme)) &
       cfg%ew_scheme = ew_scheme
    if (present (width_scheme)) &
       cfg%width_scheme = width_scheme
    if (present (accuracy)) &
       cfg%accuracy_target = accuracy
    if (present (debug)) &
       cfg%debug_unstable = debug
    cfg%dirty = .false.
  end subroutine blha_configuration_set

  function blha_configuration_get_n_proc (cfg) result (n_proc)
    type(blha_configuration_t), intent(in) :: cfg
    integer :: n_proc
    n_proc = cfg%n_proc
  end function blha_configuration_get_n_proc

  subroutine blha_configuration_write (cfg, unit, internal, no_version)
    type(blha_configuration_t), intent(in) :: cfg
    integer, intent(in), optional :: unit
    logical, intent(in), optional :: internal, no_version
    integer :: u
    logical :: full
    type(string_t) :: buf
    type(blha_cfg_process_node_t), pointer :: node
    integer :: i
    character(3) :: pdg_char
    character(4) :: hel_char
    character(len=25), parameter :: pad = ""
    logical :: write_process, no_v
    if (present (no_version))  no_v = no_version
    
    u = given_output_unit (unit)
    full = .true.; if (present (internal)) full = .not. internal
    if (full .and. cfg%dirty) call msg_bug ( &
       "BUG: attempted to write out a dirty BLHA configuration")
    if (full) then
       if (no_v) then
          write (u, "(A)") "# BLHA order written by WHIZARD [version]"
       else
          write (u, "(A)") "# BLHA order written by WHIZARD 2.2.8"
       end if
       write (u, "(A)")
    end if
    select case (cfg%mode)
       case (BLHA_MODE_GOSAM); buf = "GoSam"
       case (BLHA_MODE_OPENLOOPS); buf = "OpenLoops"
       case default; buf = "vanilla"
    end select
    write (u, "(A)") "# BLHA interface mode: " // char (buf)
    write (u, "(A)") "# process: " // char (cfg%name)
    write (u, "(A)") "# model: " // char (cfg%model%get_name ())
    select case (cfg%version)
       case (1); buf = "BLHA1"
       case (2); buf = "BLHA2"
    end select
    write (u, '(A25,A)') "InterfaceVersion " // pad, char (buf)
    select case (cfg%correction_type)
       case (BLHA_CT_QCD); buf = "QCD"
       case (BLHA_CT_EW); buf = "EW"
       case (BLHA_CT_QED); buf = "QED"
       case default; buf = cfg%correction_type_other
    end select
    write (u,'(A25,A)') "CorrectionType" // pad, char (buf)

    select case (cfg%mode)
    case (BLHA_MODE_OPENLOOPS)
       buf = cfg%name // '.olc'
       write (u, '(A25,A)') "Extra AnswerFile" // pad, char (buf)
    end select

    select case (cfg%irreg)
       case (BLHA_IRREG_CDR); buf = "CDR"
       case (BLHA_IRREG_DRED); buf = "DRED"
       case (BLHA_IRREG_THV); buf = "tHV"
       case (BLHA_IRREG_MREG); buf = "MassReg"
       case default; buf = cfg%irreg_other
    end select
    write (u,'(A25,A)') "IRregularisation" // pad, char (buf)
    select case (cfg%massive_particle_scheme)
       case (BLHA_MPS_ONSHELL); buf = "OnShell"
       case default; buf = cfg%massive_particle_scheme_other
    end select
    if (cfg%mode == BLHA_MODE_GOSAM) &
       write (u,'(A25,A)') "MassiveParticleScheme" // pad, char (buf)
    select case (cfg%version)
    case (1)
      if (cfg%alphas_power >= 0) write (u,'(A25,A)') &
         "AlphasPower" // pad, int2char (cfg%alphas_power)
      if (cfg%alpha_power >= 0) write (u,'(A25,A)') &
         "AlphaPower " // pad, int2char (cfg%alpha_power)
    case (2) 
      if (cfg%alphas_power >= 0) write (u,'(A25,A)') &
         "CouplingPower QCD " // pad, int2char (cfg%alphas_power)
      if (cfg%alpha_power >= 0) write (u, '(A25,A)') &
         "CouplingPower QED " // pad, int2char (cfg%alpha_power)
    end select
    select case (cfg%ew_scheme)
       case (BLHA_EW_GF); buf = "alphaGF"
       case (BLHA_EW_MZ); buf = "alphaMZ"
       case (BLHA_EW_MSBAR); buf = "alphaMSbar"
       case (BLHA_EW_0); buf = "alpha0"
       case (BLHA_EW_RUN); buf = "alphaRUN"
       case (BLHA_EW_DEFAULT); buf = "OLPDefined"
    end select
    select case (cfg%mode)
    case (BLHA_MODE_GOSAM)
!       write (u, '(A25, A)') "EWScheme " // pad, char (buf)
    case (BLHA_MODE_OPENLOOPS)
!       write (u, '(A25, A)') "ewscheme " // pad, char (buf)
    endselect
    select case (cfg%mode)
    case (BLHA_MODE_GOSAM)
       write (u, '(A25)', advance='no') "MassiveParticles " // pad
       do i = 1, size (OLP_MASSIVE_PARTICLES)
          if (OLP_MASSIVE_PARTICLES(i) > 0) &
             write (u, '(I2,1X)', advance='no') OLP_MASSIVE_PARTICLES(i)
       end do
       write (u,*) 
    case (BLHA_MODE_OPENLOOPS)
       write (u, '(A25,I1)') "extra use_cms " // pad, 0
       write (u, '(A25,I1)') "extra me_cache " // pad, 0
       if (cfg%openloops_phs_tolerance > 0) then
          write (u, '(A25,A4,I0)') "extra psp_tolerance " // pad, "10e-", &
             cfg%openloops_phs_tolerance
       end if
       if (cfg%openloops_top_signal) &
          write (u, '(A)') "extra approx top"
    end select
    if (full) then
       write (u, "(A)")
       write (u, "(A)") "# Process definitions"
       write (u, "(A)")
    end if
    if (cfg%debug_unstable) &
      write (u, '(A25,A)') "DebugUnstable " // pad, "True"
    write (u, *)
    node => cfg%processes
    do while (associated (node))
       write_process = .true.
       select case (node%amplitude_type)
         case (BLHA_AMP_LOOP); buf = "Loop"
         case (BLHA_AMP_CC); buf = "ccTree"
         case (BLHA_AMP_SC) 
            buf = "scTree"
            if (cfg%mode == BLHA_MODE_OPENLOOPS) write_process = .false. 
         case (BLHA_AMP_TREE); buf = "Tree"
         case (BLHA_AMP_LOOPINDUCED); buf = "LoopInduced"
       end select
       if (write_process) then
          write (u, '(A25, A)') "AmplitudeType " // pad, char (buf)
   
          buf = ""
          do i = 1, size (node%pdg_in)
             call node%pdg_in(i)%write_pdg (pdg_char)
             if (node%pdg_in(i)%polarized) then
                call node%pdg_in(i)%write_helicity (hel_char)
                buf = (buf // pdg_char // hel_char) // " "
             else
                buf = (buf // pdg_char) // " "
             end if
          end do
          buf = buf // "-> "
          do i = 1, size (node%pdg_out)
             call node%pdg_out(i)%write_pdg (pdg_char)
             buf = (buf // pdg_char) // " "
          end do
          write (u, "(A)") char (trim (buf))
          write (u, *)
       end if
       node => node%next
    end do

  end subroutine blha_configuration_write


end module blha_config

