! WHIZARD 2.2.1 June 3 2014
! 
! Copyright (C) 1999-2014 by 
!     Wolfgang Kilian <kilian@physik.uni-siegen.de>
!     Thorsten Ohl <ohl@physik.uni-wuerzburg.de>
!     Juergen Reuter <juergen.reuter@desy.de>
!     
!     with contributions from
!     Christian Speckner <cnspeckn@googlemail.com> 
!     and  Fabian Bach, Felix Braam, Sebastian Schmidt, Daniel Wiesler 
!
! WHIZARD is free software; you can redistribute it and/or modify it
! under the terms of the GNU General Public License as published by 
! the Free Software Foundation; either version 2, or (at your option)
! any later version.
!
! WHIZARD is distributed in the hope that it will be useful, but
! WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
! GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software
! Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This file has been stripped of most comments.  For documentation, refer
! to the source 'whizard.nw'

module phs_wood

  use kinds, only: default !NODEP!
  use iso_varying_string, string_t => varying_string !NODEP!
  use file_utils !NODEP!
  use diagnostics !NODEP!
  use unit_tests
  use os_interface
  use md5
  use constants !NODEP!
  use lorentz !NODEP!
  use variables
  use models
  use flavors
  use process_constants
  use sf_mappings
  use sf_base
  use phs_base
  use mappings
  use phs_forests
  use cascades

  implicit none
  private

  public :: phs_wood_config_t
  public :: phs_wood_t
  public :: phs_wood_test
  public :: phs_wood_vis_test
  public :: write_test_phs_file

  type, extends (phs_config_t) :: phs_wood_config_t
     character(32) :: md5sum_forest = ""
     integer :: io_unit = 0
     logical :: io_unit_keep_open = .false.
     logical :: use_equivalences = .false.
     logical :: fatal_beam_decay = .true.
     type(mapping_defaults_t) :: mapping_defaults
     type(phs_parameters_t) :: par
     type(string_t) :: run_id 
     type(cascade_set_t), allocatable :: cascade_set
     type(phs_forest_t) :: forest
     type(os_data_t) :: os_data
   contains
     procedure :: final => phs_wood_config_final
     procedure :: write => phs_wood_config_write
     procedure :: write_forest => phs_wood_config_write_forest
     procedure :: set_parameters => phs_wood_config_set_parameters
     procedure :: enable_equivalences => phs_wood_config_enable_equivalences
     procedure :: set_mapping_defaults => phs_wood_config_set_mapping_defaults
     procedure :: set_input => phs_wood_config_set_input
     procedure :: generate_phase_space => phs_wood_config_generate_phase_space
     procedure :: write_phase_space => phs_wood_config_write_phase_space
     procedure :: clear_phase_space => phs_wood_config_clear_phase_space
     procedure :: configure => phs_wood_config_configure
     procedure :: record_s_mappings => phs_wood_config_record_s_mappings
     procedure :: record_on_shell => phs_wood_config_record_on_shell
     procedure :: compute_md5sum_forest => phs_wood_config_compute_md5sum_forest
     procedure :: get_md5sum => phs_wood_config_get_md5sum
     procedure :: read_phs_file => phs_wood_read_phs_file
     procedure :: startup_message => phs_wood_config_startup_message
     procedure, nopass :: allocate_instance => phs_wood_config_allocate_instance
  end type phs_wood_config_t
  
  type, extends (phs_t) :: phs_wood_t
     real(default) :: sqrts = 0
     type(phs_forest_t) :: forest
   contains
     procedure :: write => phs_wood_write
     procedure :: write_forest => phs_wood_write_forest
     procedure :: final => phs_wood_final
     procedure :: init => phs_wood_init
     procedure :: evaluate_selected_channel => phs_wood_evaluate_selected_channel
     procedure :: evaluate_other_channels => phs_wood_evaluate_other_channels
     procedure :: inverse => phs_wood_inverse
  end type phs_wood_t
  

contains

  subroutine phs_wood_config_final (object)
    class(phs_wood_config_t), intent(inout) :: object
    logical :: opened
    if (object%io_unit /= 0) then
       inquire (unit = object%io_unit, opened = opened)
       if (opened)  close (object%io_unit)
    end if
    call object%clear_phase_space ()
    call phs_forest_final (object%forest)
  end subroutine phs_wood_config_final
  
  subroutine phs_wood_config_write (object, unit)
    class(phs_wood_config_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u
    u = output_unit (unit)
    write (u, "(1x,A)") &
         "Partonic phase-space configuration (phase-space forest):"
    call object%base_write (unit)
    write (u, "(1x,A)")    "Phase-space configuration parameters:"
    call phs_parameters_write (object%par, u)
    call object%mapping_defaults%write (u)
    write (u, "(3x,A,A,A)")  "Run ID: '", char (object%run_id), "'"
  end subroutine phs_wood_config_write
  
  subroutine phs_wood_config_write_forest (object, unit)
    class(phs_wood_config_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u
    u = output_unit (unit)
    call phs_forest_write (object%forest, u)
  end subroutine phs_wood_config_write_forest
  
  subroutine phs_wood_config_set_parameters (phs_config, par)
    class(phs_wood_config_t), intent(inout) :: phs_config
    type(phs_parameters_t), intent(in) :: par
    phs_config%par = par
  end subroutine phs_wood_config_set_parameters

  subroutine phs_wood_config_enable_equivalences (phs_config)
    class(phs_wood_config_t), intent(inout) :: phs_config
    phs_config%use_equivalences = .true.
  end subroutine phs_wood_config_enable_equivalences
  
  subroutine phs_wood_config_set_mapping_defaults (phs_config, mapping_defaults)
    class(phs_wood_config_t), intent(inout) :: phs_config
    type(mapping_defaults_t), intent(in) :: mapping_defaults
    phs_config%mapping_defaults = mapping_defaults
  end subroutine phs_wood_config_set_mapping_defaults

  subroutine phs_wood_config_set_input (phs_config, unit)
    class(phs_wood_config_t), intent(inout) :: phs_config
    integer, intent(in) :: unit
    phs_config%io_unit = unit
    rewind (unit)
  end subroutine phs_wood_config_set_input
  
  subroutine phs_wood_config_generate_phase_space (phs_config)
    class(phs_wood_config_t), intent(inout) :: phs_config
    integer :: off_shell, extra_off_shell
    call msg_message ("Phase space: generating configuration ...")
    off_shell = phs_config%par%off_shell
    allocate (phs_config%cascade_set)
    do extra_off_shell = 0, max (phs_config%n_tot - 3, 0)
       phs_config%par%off_shell = off_shell + extra_off_shell
       call cascade_set_generate (phs_config%cascade_set, &
            phs_config%model, phs_config%n_in, phs_config%n_out, &
            phs_config%flv, &
            phs_config%par, phs_config%fatal_beam_decay)
       if (cascade_set_is_valid (phs_config%cascade_set)) then
          exit
       else
          call msg_message ("Phase space: ... failed.  &
               &Increasing phs_off_shell ...")
       end if
    end do
    if (cascade_set_is_valid (phs_config%cascade_set)) then
       call msg_message ("Phase space: ... success.")
    else
       call msg_fatal ("Phase-space: generation failed")
    end if
  end subroutine phs_wood_config_generate_phase_space
    
  subroutine phs_wood_config_write_phase_space (phs_config, &
       filename_vis, unit)
    class(phs_wood_config_t), intent(in) :: phs_config
    integer, intent(in), optional :: unit
    type(string_t), intent(in), optional :: filename_vis
    type(string_t) :: setenv_tex, setenv_mp, pipe, pipe_dvi
    integer :: u, unit_tex, unit_dev, status   
    if (allocated (phs_config%cascade_set)) then
       if (present (unit)) then
          u = unit
       else
          u = phs_config%io_unit
       end if
       write (u, "(1x,A,A)") "process ", char (phs_config%id)
       write (u, "(A)")
       call cascade_set_write_process_bincode_format (phs_config%cascade_set, u)
       write (u, "(A)")
       write (u, "(3x,A,A,A32,A)") "md5sum_process    = ", &
            '"', phs_config%md5sum_process, '"'
       write (u, "(3x,A,A,A32,A)") "md5sum_model_par  = ", &
            '"', phs_config%md5sum_model_par, '"'
       write (u, "(3x,A,A,A32,A)") "md5sum_phs_config = ", &
            '"', phs_config%md5sum_phs_config, '"'
       call phs_parameters_write (phs_config%par, u)
       call cascade_set_write_file_format (phs_config%cascade_set, u)
       if (phs_config%vis_channels) then 
          unit_tex = free_unit ()
          open (unit=unit_tex, file=char(filename_vis // ".tex"), &
               action="write", status="replace")      
          call cascade_set_write_graph_format (phs_config%cascade_set, &
               filename_vis // "-graphs", phs_config%id, unit_tex)
          close (unit_tex)      
          call msg_message ("Phase space: visualizing channels in file " & 
               // char(trim(filename_vis)) // "...")
          if (phs_config%os_data%event_analysis_ps) then
             BLOCK: do
                unit_dev = free_unit ()
                open (file = "/dev/null", unit = unit_dev, &
                     action = "write", iostat = status)
                if (status /= 0) then
                   pipe = ""
                   pipe_dvi = ""
                else
                   pipe = " > /dev/null"
                   pipe_dvi = " 2>/dev/null 1>/dev/null"
                end if
                close (unit_dev)
                if (phs_config%os_data%whizard_texpath /= "") then
                   setenv_tex = "TEXINPUTS=" // &
                        phs_config%os_data%whizard_texpath // ":$TEXINPUTS "
                   setenv_mp = "MPINPUTS=" // &
                        phs_config%os_data%whizard_texpath // ":$MPINPUTS "
                else
                   setenv_tex = ""
                   setenv_mp = ""
                end if
                call os_system_call (setenv_tex // &
                     phs_config%os_data%latex // " " // &
                     filename_vis // ".tex " // pipe, status)
                if (status /= 0)  exit BLOCK
                if (phs_config%os_data%mpost /= "") then
                   call os_system_call (setenv_mp // &
                        phs_config%os_data%mpost // " " // &
                        filename_vis // "-graphs.mp" // pipe, status)
                else 
                   call msg_fatal ("Could not use MetaPOST.")
                end if
                if (status /= 0)  exit BLOCK
                call os_system_call (setenv_tex // &
                     phs_config%os_data%latex // " " // &
                     filename_vis // ".tex" // pipe, status)
                if (status /= 0)  exit BLOCK
                call os_system_call &
                     (phs_config%os_data%dvips // " -o " // filename_vis &
                     // ".ps " // filename_vis // ".dvi" // pipe_dvi, status)
                if (status /= 0)  exit BLOCK
                if (phs_config%os_data%event_analysis_pdf) then
                   call os_system_call (phs_config%os_data%ps2pdf // " " // &
                        filename_vis // ".ps", status)
                   if (status /= 0)  exit BLOCK
                end if
                exit BLOCK
             end do BLOCK
             if (status /= 0) then
                call msg_error ("Unable to compile analysis output file")
             end if
          end if
       end if
    else
       call msg_fatal ("Phase-space configuration: &
            &no phase space object generated")
    end if
  end subroutine phs_wood_config_write_phase_space
       
  subroutine phs_wood_config_clear_phase_space (phs_config)
    class(phs_wood_config_t), intent(inout) :: phs_config
    if (allocated (phs_config%cascade_set)) then
       call cascade_set_final (phs_config%cascade_set)
       deallocate (phs_config%cascade_set)
    end if
  end subroutine phs_wood_config_clear_phase_space
  
  subroutine phs_wood_config_configure (phs_config, sqrts, &
       sqrts_fixed, cm_frame, azimuthal_dependence, rebuild, ignore_mismatch)
    class(phs_wood_config_t), intent(inout) :: phs_config
    real(default), intent(in) :: sqrts
    logical, intent(in), optional :: sqrts_fixed
    logical, intent(in), optional :: cm_frame
    logical, intent(in), optional :: azimuthal_dependence
    logical, intent(in), optional :: rebuild
    logical, intent(in), optional :: ignore_mismatch
    type(string_t) :: filename, filename_vis
    logical :: variable_limits
    logical :: ok, exist, found, check, match, rebuild_phs
    integer :: g, c0, c1, n
    phs_config%sqrts = sqrts
    phs_config%par%sqrts = sqrts
    if (present (sqrts_fixed)) &
         phs_config%sqrts_fixed = sqrts_fixed
    if (present (cm_frame)) &
         phs_config%cm_frame = cm_frame
    if (present (azimuthal_dependence)) &
         phs_config%azimuthal_dependence = azimuthal_dependence
    if (present (rebuild)) then
       rebuild_phs = rebuild
    else
       rebuild_phs = .true.
    end if
    if (present (ignore_mismatch)) then
       check = .not. ignore_mismatch
       if (ignore_mismatch) &
            call msg_warning ("Reading phs file: MD5 sum check disabled")
    else
       check = .true.
    end if
    phs_config%md5sum_forest = ""
    call phs_config%compute_md5sum ()
    if (phs_config%io_unit == 0) then
       if (phs_config%run_id /= "") then
          filename = phs_config%id // "." // phs_config%run_id // ".phs"
          filename_vis = phs_config%id // "." // phs_config%run_id // "_phs"
       else
          filename = phs_config%id // ".phs"
          filename_vis = phs_config%id // "_phs"          
       end if
       if (.not. rebuild_phs) then
          if (check) then
             call phs_config%read_phs_file (exist, found, match)
             rebuild_phs = .not. (exist .and. found .and. match)
          else
             call phs_config%read_phs_file (exist, found)
             rebuild_phs = .not. (exist .and. found)
          end if
       end if
       if (rebuild_phs) then
          call phs_config%generate_phase_space ()
          phs_config%io_unit = free_unit ()
          if (phs_config%id /= "") then
             call msg_message ("Phase space: writing configuration file '" &
                  // char (filename) // "'")
             open (phs_config%io_unit, file = char (filename), &
                  status = "replace", action = "readwrite")
          else
             open (phs_config%io_unit, status = "scratch", action = "readwrite")
          end if
          call phs_config%write_phase_space (filename_vis)
          rewind (phs_config%io_unit)
       else
          call msg_message ("Phase space: keeping configuration file '" &
               // char (filename) // "'")
       end if
    end if
    if (phs_config%io_unit == 0) then
       ok = .true.
    else
       call phs_forest_read (phs_config%forest, phs_config%io_unit, &
            phs_config%id, phs_config%n_in, phs_config%n_out, &
            phs_config%model, ok)
       if (.not. phs_config%io_unit_keep_open) then
          close (phs_config%io_unit)
          phs_config%io_unit = 0
       end if
    end if
    if (ok) then
       call phs_forest_set_flavors (phs_config%forest, phs_config%flv(:,1))
       variable_limits = .not. phs_config%cm_frame
       call phs_forest_set_parameters &
            (phs_config%forest, phs_config%mapping_defaults, variable_limits)
       call phs_forest_setup_prt_combinations (phs_config%forest)
       phs_config%n_channel = phs_forest_get_n_channels (phs_config%forest)
       phs_config%n_par = phs_forest_get_n_parameters (phs_config%forest)
       allocate (phs_config%channel (phs_config%n_channel))
       if (phs_config%use_equivalences) then
          call phs_forest_set_equivalences (phs_config%forest)
          call phs_forest_get_equivalences (phs_config%forest, &
               phs_config%channel, phs_config%azimuthal_dependence)
          phs_config%provides_equivalences = .true.
       end if
       call phs_forest_set_s_mappings (phs_config%forest)
       call phs_config%record_on_shell ()
       if (phs_config%mapping_defaults%enable_s_mapping) then
          call phs_config%record_s_mappings ()
       end if
       allocate (phs_config%chain (phs_config%n_channel), source = 0)
       do g = 1, phs_forest_get_n_groves (phs_config%forest)
          call phs_forest_get_grove_bounds (phs_config%forest, g, c0, c1, n)
          phs_config%chain (c0:c1) = g
       end do
       phs_config%provides_chains = .true.
       call phs_config%compute_md5sum_forest ()
    else
       write (msg_buffer, "(A,A,A)") &
            "Phase space: process '", &
            char (phs_config%id), "' not found in configuration file"
       call msg_fatal ()
    end if
  end subroutine phs_wood_config_configure
  
  subroutine phs_wood_config_record_s_mappings (phs_config)
    class(phs_wood_config_t), intent(inout) :: phs_config
    logical :: flag
    real(default) :: mass, width
    integer :: c
    do c = 1, phs_config%n_channel
       call phs_forest_get_s_mapping (phs_config%forest, c, flag, mass, width)
       if (flag) then
          if (mass == 0) then
             call msg_fatal ("Phase space: s-channel resonance " &
                  // " has zero mass")
          end if
          if (width == 0) then
             call msg_fatal ("Phase space: s-channel resonance " &
                  // " has zero width")
          end if
          call phs_config%channel(c)%set_resonant (mass, width)
       end if
    end do
  end subroutine phs_wood_config_record_s_mappings

  subroutine phs_wood_config_record_on_shell (phs_config)
    class(phs_wood_config_t), intent(inout) :: phs_config
    logical :: flag
    real(default) :: mass
    integer :: c
    do c = 1, phs_config%n_channel
       call phs_forest_get_on_shell (phs_config%forest, c, flag, mass)
       if (flag) then
          call phs_config%channel(c)%set_on_shell (mass)
       end if
    end do
  end subroutine phs_wood_config_record_on_shell

  subroutine phs_wood_config_compute_md5sum_forest (phs_config)
    class(phs_wood_config_t), intent(inout) :: phs_config
    integer :: u
    u = free_unit ()
    open (u, status = "scratch", action = "readwrite")
    call phs_config%write_forest (u)
    rewind (u)
    phs_config%md5sum_forest = md5sum (u)
    close (u)
  end subroutine phs_wood_config_compute_md5sum_forest
  
  function phs_wood_config_get_md5sum (phs_config) result (md5sum)
    class(phs_wood_config_t), intent(in) :: phs_config
    character(32) :: md5sum
    if (phs_config%md5sum_forest /= "") then
       md5sum = phs_config%md5sum_forest
    else
       md5sum = phs_config%md5sum_phs_config
    end if
  end function phs_wood_config_get_md5sum
 
  subroutine phs_wood_read_phs_file (phs_config, exist, found, match)
    class(phs_wood_config_t), intent(inout) :: phs_config
    logical, intent(out) :: exist
    logical, intent(out) :: found
    logical, intent(out), optional :: match
    type(string_t) :: filename
    integer :: u
    filename = phs_config%id // ".phs"
    inquire (file = char (filename), exist = exist)
    if (exist) then
       u = free_unit ()
       open (u, file = char (filename), action = "read", status = "old")
       call phs_forest_read (phs_config%forest, u, &
            phs_config%id, phs_config%n_in, phs_config%n_out, &
            phs_config%model, found, &
            phs_config%md5sum_process, &
            phs_config%md5sum_model_par, &
            phs_config%md5sum_phs_config, &
            match = match)
       close (u)
    else
       found = .false.
       if (present (match))  match = .false.
    end if
  end subroutine phs_wood_read_phs_file
    
  subroutine phs_wood_config_startup_message (phs_config, unit)
    class(phs_wood_config_t), intent(in) :: phs_config
    integer, intent(in), optional :: unit
    integer :: n_groves, n_eq
    n_groves = phs_forest_get_n_groves (phs_config%forest)
    n_eq = phs_forest_get_n_equivalences (phs_config%forest)
    call phs_config%base_startup_message (unit)
    if (phs_config%n_channel == 1) then
       write (msg_buffer, "(A,2(I0,A))") &
            "Phase space: found ", phs_config%n_channel, &
            " channel, collected in ", n_groves, &
            " grove."
    else if (n_groves == 1) then
       write (msg_buffer, "(A,2(I0,A))") &
            "Phase space: found ", phs_config%n_channel, &
            " channels, collected in ", n_groves, &
            " grove." 
       else
       write (msg_buffer, "(A,2(I0,A))") &
            "Phase space: found ", phs_config%n_channel, &
            " channels, collected in ", &
            phs_forest_get_n_groves (phs_config%forest), &
            " groves."
    end if
    call msg_message (unit = unit)    
    if (phs_config%use_equivalences) then
       if (n_eq == 1) then 
          write (msg_buffer, "(A,I0,A)") &
               "Phase space: Using ", n_eq, &
               " equivalence between channels."
       else
          write (msg_buffer, "(A,I0,A)") &
               "Phase space: Using ", n_eq, &
               " equivalences between channels."          
       end if
    else
       write (msg_buffer, "(A)") &
            "Phase space: no equivalences between channels used."
    end if
    call msg_message (unit = unit)
    write (msg_buffer, "(A,2(1x,I0,1x,A))") &
         "Phase space: wood"
    call msg_message (unit = unit)
  end subroutine phs_wood_config_startup_message
    
  subroutine phs_wood_config_allocate_instance (phs)
    class(phs_t), intent(inout), pointer :: phs
    allocate (phs_wood_t :: phs)
  end subroutine phs_wood_config_allocate_instance
  
  subroutine phs_wood_write (object, unit, verbose)
    class(phs_wood_t), intent(in) :: object
    integer, intent(in), optional :: unit
    logical, intent(in), optional :: verbose
    integer :: u
    u = output_unit (unit)
    call object%base_write (u)
  end subroutine phs_wood_write
    
  subroutine phs_wood_write_forest (object, unit)
    class(phs_wood_t), intent(in) :: object
    integer, intent(in), optional :: unit
    integer :: u
    u = output_unit (unit)
    call phs_forest_write (object%forest, u)
  end subroutine phs_wood_write_forest
  
  subroutine phs_wood_final (object)
    class(phs_wood_t), intent(inout) :: object
    call phs_forest_final (object%forest)
  end subroutine phs_wood_final
  
  subroutine phs_wood_init (phs, phs_config)
    class(phs_wood_t), intent(out) :: phs
    class(phs_config_t), intent(in), target :: phs_config
    call phs%base_init (phs_config)
    select type (phs_config)
    type is (phs_wood_config_t)
       phs%forest = phs_config%forest
    end select
  end subroutine phs_wood_init
  
  subroutine phs_wood_evaluate_selected_channel (phs, c_in, r_in)
    class(phs_wood_t), intent(inout) :: phs
    integer, intent(in) :: c_in
    real(default), intent(in), dimension(:) :: r_in
    logical :: ok
    phs%q_defined = .false.
    if (phs%p_defined) then
       call phs_forest_set_prt_in (phs%forest, phs%p)
       phs%r(:,c_in) = r_in
       call phs_forest_evaluate_selected_channel (phs%forest, &
            c_in, phs%active_channel, &
            phs%sqrts_hat, phs%r, phs%f, phs%volume, ok)
       if (ok) then
          phs%q = phs_forest_get_momenta_out (phs%forest)
          phs%q_defined = .true.
       end if
    end if
  end subroutine phs_wood_evaluate_selected_channel
  
  subroutine phs_wood_evaluate_other_channels (phs, c_in)
    class(phs_wood_t), intent(inout) :: phs
    integer, intent(in) :: c_in
    if (phs%q_defined) then
       call phs_forest_evaluate_other_channels (phs%forest, &
            c_in, phs%active_channel, &
            phs%sqrts_hat, phs%r, phs%f, combine=.true.)
       phs%r_defined = .true.
    end if
  end subroutine phs_wood_evaluate_other_channels
  
  subroutine phs_wood_inverse (phs)
    class(phs_wood_t), intent(inout) :: phs
    if (phs%p_defined .and. phs%q_defined) then
       call phs_forest_set_prt_in (phs%forest, phs%p)
       call phs_forest_set_prt_out (phs%forest, phs%q)
       call phs_forest_recover_channel (phs%forest, &
            1, &
            phs%sqrts_hat, phs%r, phs%f, phs%volume)
       call phs_forest_evaluate_other_channels (phs%forest, &
            1, phs%active_channel, &
            phs%sqrts_hat, phs%r, phs%f, combine=.false.)
       phs%r_defined = .true.
    end if
  end subroutine phs_wood_inverse
  

  subroutine phs_wood_test (u, results)
    integer, intent(in) :: u
    type(test_results_t), intent(inout) :: results
    call test (phs_wood_1, "phs_wood_1", &
         "phase-space configuration", &
         u, results)
    call test (phs_wood_2, "phs_wood_2", &
         "phase-space evaluation", &
         u, results)
    call test (phs_wood_3, "phs_wood_3", &
         "phase-space generation", &
         u, results)
    call test (phs_wood_4, "phs_wood_4", &
         "nontrivial process", &
         u, results)
    call test (phs_wood_5, "phs_wood_5", &
         "equivalences", &
         u, results)
    call test (phs_wood_6, "phs_wood_6", &
         "phase-space generation", &
         u, results)
  end subroutine phs_wood_test
  
  subroutine phs_wood_vis_test (u, results)
    integer, intent(in) :: u
    type(test_results_t), intent(inout) :: results
    call test (phs_wood_vis_1, "phs_wood_vis_1", &
         "visualizing phase space channels", &
         u, results)
  end subroutine phs_wood_vis_test
  
  subroutine write_test_phs_file (u_phs, procname)
    integer, intent(in) :: u_phs
    type(string_t), intent(in), optional :: procname
    if (present (procname)) then
       write (u_phs, "(A,A)")  "process ", char (procname)
    else
       write (u_phs, "(A)")  "process testproc"
    end if
    write (u_phs, "(A,A)")  "   md5sum_process    = ", '""'
    write (u_phs, "(A,A)")  "   md5sum_model_par  = ", '""'
    write (u_phs, "(A,A)")  "   md5sum_phs_config = ", '""'
    write (u_phs, "(A)")  "   sqrts         = 1000"
    write (u_phs, "(A)")  "   m_threshold_s =   50"    
    write (u_phs, "(A)")  "   m_threshold_t =  100"    
    write (u_phs, "(A)")  "   off_shell = 2"
    write (u_phs, "(A)")  "   t_channel = 6"
    write (u_phs, "(A)")  "   keep_nonresonant = T"
    write (u_phs, "(A)")  "  grove #1"
    write (u_phs, "(A)")  "    tree 3"
  end subroutine write_test_phs_file

  subroutine phs_wood_1 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(process_constants_t) :: process_data
    class(phs_config_t), allocatable :: phs_data
    type(mapping_defaults_t) :: mapping_defaults
    real(default) :: sqrts
    integer :: u_phs, iostat
    character(32) :: buffer
    
    write (u, "(A)")  "* Test output: phs_wood_1"
    write (u, "(A)")  "*   Purpose: initialize and display &
         &phase-space configuration data"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)

    call syntax_phs_forest_init ()
    
    write (u, "(A)")  "* Initialize a process"
    write (u, "(A)")

    call init_test_process_data (var_str ("phs_wood_1"), process_data)

    write (u, "(A)")  "* Create a scratch phase-space file"
    write (u, "(A)")

    u_phs = free_unit ()
    open (u_phs, status = "scratch", action = "readwrite")
    call write_test_phs_file (u_phs, var_str ("phs_wood_1"))
    rewind (u_phs)
    do
       read (u_phs, "(A)", iostat = iostat)  buffer
       if (iostat /= 0)  exit
       write (u, "(A)") trim (buffer)
    end do

    write (u, "(A)")
    write (u, "(A)")  "* Setup phase-space configuration object"
    write (u, "(A)")

    mapping_defaults%step_mapping = .false.

    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_input (u_phs)
       call phs_data%set_mapping_defaults (mapping_defaults)
    end select

    sqrts = 1000._default
    call phs_data%configure (sqrts)
       
    call phs_data%write (u)
    write (u, "(A)")

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%write_forest (u)
    end select
    
    write (u, "(A)")
    write (u, "(A)")  "* Cleanup"

    close (u_phs)
    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_1"

  end subroutine phs_wood_1

  subroutine phs_wood_2 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(flavor_t) :: flv
    type(process_constants_t) :: process_data
    real(default) :: sqrts, E
    class(phs_config_t), allocatable, target :: phs_data
    class(phs_t), pointer :: phs => null ()
    type(vector4_t), dimension(2) :: p, q
    integer :: u_phs
    
    write (u, "(A)")  "* Test output: phs_wood_2"
    write (u, "(A)")  "*   Purpose: test simple single-channel phase space"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)
    call flavor_init (flv, 25, model)

    write (u, "(A)")  "* Initialize a process and a matching &
         &phase-space configuration"
    write (u, "(A)")

    call init_test_process_data (var_str ("phs_wood_2"), process_data)
    u_phs = free_unit ()
    open (u_phs, status = "scratch", action = "readwrite")
    call write_test_phs_file (u_phs, var_str ("phs_wood_2"))
    rewind (u_phs)

    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_input (u_phs)
    end select

    sqrts = 1000._default
    call phs_data%configure (sqrts)

    call phs_data%write (u)
    
    write (u, "(A)")
    write (u, "(A)")  "* Initialize the phase-space instance"
    write (u, "(A)")

    call phs_data%allocate_instance (phs)
    call phs%init (phs_data)
       
    call phs%write (u, verbose=.true.)
    
    write (u, "(A)")
    write (u, "(A)")  "* Set incoming momenta"
    write (u, "(A)")

    E = sqrts / 2
    p(1) = vector4_moving (E, sqrt (E**2 - flavor_get_mass (flv)**2), 3)
    p(2) = vector4_moving (E,-sqrt (E**2 - flavor_get_mass (flv)**2), 3)

    call phs%set_incoming_momenta (p)
    call phs%compute_flux ()
    call phs%write (u)
    
    write (u, "(A)")
    write (u, "(A)")  "* Compute phase-space point &
         &for x = 0.125, 0.5"
    write (u, "(A)")

    call phs%evaluate_selected_channel (1, [0.125_default, 0.5_default])
    call phs%evaluate_other_channels (1)
    call phs%write (u)
    write (u, "(A)")
    select type (phs)
    type is (phs_wood_t)
       call phs%write_forest (u)
    end select
    
    write (u, "(A)")
    write (u, "(A)")  "* Inverse kinematics"
    write (u, "(A)")

    call phs%get_outgoing_momenta (q)
    call phs%final ()
    deallocate (phs)
    
    call phs_data%allocate_instance (phs)
    call phs%init (phs_data)
       
    call phs%set_incoming_momenta (p)
    call phs%compute_flux ()
    call phs%set_outgoing_momenta (q)
    
    call phs%inverse ()
    call phs%write (u)
    write (u, "(A)")
    select type (phs)
    type is (phs_wood_t)
       call phs%write_forest (u)
    end select
    
    call phs%final ()
    deallocate (phs)
    
    close (u_phs)
    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_2"

  end subroutine phs_wood_2

  subroutine phs_wood_3 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(process_constants_t) :: process_data
    type(phs_parameters_t) :: phs_par
    class(phs_config_t), allocatable :: phs_data
    integer :: iostat
    character(80) :: buffer
   
    write (u, "(A)")  "* Test output: phs_wood_3"
    write (u, "(A)")  "*   Purpose: generate a phase-space configuration"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)

    call syntax_phs_forest_init ()
    
    write (u, "(A)")  "* Initialize a process and phase-space parameters"
    write (u, "(A)")

    call init_test_process_data (var_str ("phs_wood_3"), process_data)
    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)

    phs_par%sqrts = 1000
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%io_unit_keep_open = .true.
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Generate a scratch phase-space file"
    write (u, "(A)")

    call phs_data%configure (phs_par%sqrts)

    select type (phs_data)
    type is (phs_wood_config_t)
       rewind (phs_data%io_unit)
       do
          read (phs_data%io_unit, "(A)", iostat = iostat)  buffer
          if (iostat /= 0)  exit
          write (u, "(A)") trim (buffer)
       end do
    end select
    
    write (u, "(A)")
    write (u, "(A)")  "* Cleanup"

    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_3"

  end subroutine phs_wood_3

  subroutine phs_wood_4 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(process_constants_t) :: process_data
    type(phs_parameters_t) :: phs_par
    class(phs_config_t), allocatable, target :: phs_data
    integer :: iostat
    character(80) :: buffer
    class(phs_t), pointer :: phs => null ()
    real(default) :: E, pL
    type(vector4_t), dimension(2) :: p
    type(vector4_t), dimension(3) :: q
   
    write (u, "(A)")  "* Test output: phs_wood_4"
    write (u, "(A)")  "*   Purpose: generate a phase-space configuration"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)

    call syntax_phs_forest_init ()
    
    write (u, "(A)")  "* Initialize a process and phase-space parameters"
    write (u, "(A)")

    process_data%id = "phs_wood_4"
    process_data%model_name = "Test"
    process_data%n_in = 2
    process_data%n_out = 3
    process_data%n_flv = 1
    allocate (process_data%flv_state (process_data%n_in + process_data%n_out, &
         process_data%n_flv))
    process_data%flv_state(:,1) = [25, 25, 25, 6, -6]

    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)

    phs_par%sqrts = 1000
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%io_unit_keep_open = .true.
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Generate a scratch phase-space file"
    write (u, "(A)")

    call phs_data%configure (phs_par%sqrts)

    select type (phs_data)
    type is (phs_wood_config_t)
       rewind (phs_data%io_unit)
       do
          read (phs_data%io_unit, "(A)", iostat = iostat)  buffer
          if (iostat /= 0)  exit
          write (u, "(A)") trim (buffer)
       end do
    end select
    
    write (u, "(A)")
    write (u, "(A)")  "* Initialize the phase-space instance"
    write (u, "(A)")

    call phs_data%allocate_instance (phs)
    call phs%init (phs_data)
       
    write (u, "(A)")  "* Set incoming momenta"
    write (u, "(A)")

    select type (phs_data)
    type is (phs_wood_config_t)
       E = phs_data%sqrts / 2
       pL = sqrt (E**2 - flavor_get_mass (phs_data%flv(1,1))**2)
    end select
    p(1) = vector4_moving (E, pL, 3)
    p(2) = vector4_moving (E, -pL, 3)

    call phs%set_incoming_momenta (p)
    call phs%compute_flux ()
    
    write (u, "(A)")  "* Compute phase-space point &
         &for x = 0.1, 0.2, 0.3, 0.4, 0.5"
    write (u, "(A)")

    call phs%evaluate_selected_channel (1, &
         [0.1_default, 0.2_default, 0.3_default, 0.4_default, 0.5_default])
    call phs%evaluate_other_channels (1)
    call phs%write (u)
    
    write (u, "(A)")
    write (u, "(A)")  "* Inverse kinematics"
    write (u, "(A)")

    call phs%get_outgoing_momenta (q)
    call phs%final ()
    deallocate (phs)
    
    call phs_data%allocate_instance (phs)
    call phs%init (phs_data)
       
    call phs%set_incoming_momenta (p)
    call phs%compute_flux ()
    call phs%set_outgoing_momenta (q)
    
    call phs%inverse ()
    call phs%write (u)
    
    write (u, "(A)")
    write (u, "(A)")  "* Cleanup"

    call phs%final ()
    deallocate (phs)

    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_4"

  end subroutine phs_wood_4

  subroutine phs_wood_5 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(process_constants_t) :: process_data
    type(phs_parameters_t) :: phs_par
    class(phs_config_t), allocatable :: phs_data
   
    write (u, "(A)")  "* Test output: phs_wood_5"
    write (u, "(A)")  "*   Purpose: generate a phase-space configuration"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)

    call syntax_phs_forest_init ()
    
    write (u, "(A)")  "* Initialize a process and phase-space parameters"
    write (u, "(A)")

    call init_test_process_data (var_str ("phs_wood_5"), process_data)
    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)

    phs_par%sqrts = 1000
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       call phs_data%enable_equivalences ()
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Generate a scratch phase-space file"
    write (u, "(A)")

    call phs_data%configure (phs_par%sqrts)
    call phs_data%write (u)
    write (u, "(A)")

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%write_forest (u)
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Cleanup"

    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_5"

  end subroutine phs_wood_5

  subroutine phs_wood_6 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(process_constants_t) :: process_data
    type(phs_parameters_t) :: phs_par
    class(phs_config_t), allocatable :: phs_data
    logical :: exist, found, match
    integer :: u_phs
    character(*), parameter :: filename = "phs_wood_6_p.phs"
    type(var_list_t), pointer :: var_list
   
    write (u, "(A)")  "* Test output: phs_wood_6"
    write (u, "(A)")  "*   Purpose: generate and check  phase-space file"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)

    call syntax_phs_forest_init ()
    
    write (u, "(A)")  "* Initialize a process and phase-space parameters"
    write (u, "(A)")

    call init_test_process_data (var_str ("phs_wood_6"), process_data)
    process_data%id = "phs_wood_6_p"
    process_data%md5sum = "1234567890abcdef1234567890abcdef"
    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    
    phs_par%sqrts = 1000
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
    end select

    write (u, "(A)")  "* Remove previous phs file, if any"
    write (u, "(A)")

    inquire (file = filename, exist = exist)
    if (exist) then
       u_phs = free_unit ()
       open (u_phs, file = filename, action = "write")
       close (u_phs, status = "delete")
    end if

    write (u, "(A)")  "* Check phase-space file (should fail)"
    write (u, "(A)")
    
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%read_phs_file (exist, found, match)
       write (u, "(1x,A,L1)")  "exist = ", exist
       write (u, "(1x,A,L1)")  "found = ", found
       write (u, "(1x,A,L1)")  "match = ", match
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Generate a phase-space file"
    write (u, "(A)")

    call phs_data%configure (phs_par%sqrts)

    write (u, "(1x,A,A,A)")  "MD5 sum (process)    = '", &
         phs_data%md5sum_process, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (model par)  = '", &
         phs_data%md5sum_model_par, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (phs config) = '", &
         phs_data%md5sum_phs_config, "'"

    write (u, "(A)")
    write (u, "(A)")  "* Check MD5 sum"
    write (u, "(A)")

    call phs_data%final ()
    deallocate (phs_data)
    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    phs_par%sqrts = 1000
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%sqrts = phs_par%sqrts
       phs_data%par%sqrts = phs_par%sqrts
    end select
    call phs_data%compute_md5sum ()

    write (u, "(1x,A,A,A)")  "MD5 sum (process)    = '", &
         phs_data%md5sum_process, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (model par)  = '", &
         phs_data%md5sum_model_par, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (phs config) = '", &
         phs_data%md5sum_phs_config, "'"

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%read_phs_file (exist, found, match)
       write (u, "(1x,A,L1)")  "exist = ", exist
       write (u, "(1x,A,L1)")  "found = ", found
       write (u, "(1x,A,L1)")  "match = ", match
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Modify sqrts and check MD5 sum"
    write (u, "(A)")

    call phs_data%final ()
    deallocate (phs_data)
    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    phs_par%sqrts = 500
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%sqrts = phs_par%sqrts
       phs_data%par%sqrts = phs_par%sqrts
    end select
    call phs_data%compute_md5sum ()

    write (u, "(1x,A,A,A)")  "MD5 sum (process)    = '", &
         phs_data%md5sum_process, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (model par)  = '", &
         phs_data%md5sum_model_par, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (phs config) = '", &
         phs_data%md5sum_phs_config, "'"

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%read_phs_file (exist, found, match)
       write (u, "(1x,A,L1)")  "exist = ", exist
       write (u, "(1x,A,L1)")  "found = ", found
       write (u, "(1x,A,L1)")  "match = ", match
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Modify process and check MD5 sum"
    write (u, "(A)")

    call phs_data%final ()
    deallocate (phs_data)
    process_data%md5sum = "77777777777777777777777777777777"
    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    phs_par%sqrts = 1000
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%sqrts = phs_par%sqrts
       phs_data%par%sqrts = phs_par%sqrts
    end select
    call phs_data%compute_md5sum ()

    write (u, "(1x,A,A,A)")  "MD5 sum (process)    = '", &
         phs_data%md5sum_process, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (model par)  = '", &
         phs_data%md5sum_model_par, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (phs config) = '", &
         phs_data%md5sum_phs_config, "'"

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%read_phs_file (exist, found, match)
       write (u, "(1x,A,L1)")  "exist = ", exist
       write (u, "(1x,A,L1)")  "found = ", found
       write (u, "(1x,A,L1)")  "match = ", match
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Modify phs parameter and check MD5 sum"
    write (u, "(A)")

    call phs_data%final ()
    deallocate (phs_data)
    allocate (phs_wood_config_t :: phs_data)
    process_data%md5sum = "1234567890abcdef1234567890abcdef"
    call phs_data%init (process_data, model)
    phs_par%sqrts = 1000
    phs_par%off_shell = 17
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%sqrts = phs_par%sqrts
       phs_data%par%sqrts = phs_par%sqrts
    end select
    call phs_data%compute_md5sum ()

    write (u, "(1x,A,A,A)")  "MD5 sum (process)    = '", &
         phs_data%md5sum_process, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (model par)  = '", &
         phs_data%md5sum_model_par, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (phs config) = '", &
         phs_data%md5sum_phs_config, "'"

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%read_phs_file (exist, found, match)
       write (u, "(1x,A,L1)")  "exist = ", exist
       write (u, "(1x,A,L1)")  "found = ", found
       write (u, "(1x,A,L1)")  "match = ", match
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Modify model parameter and check MD5 sum"
    write (u, "(A)")

    call phs_data%final ()
    deallocate (phs_data)
    allocate (phs_wood_config_t :: phs_data)
    var_list => model_get_var_list_ptr (model)
    call var_list_set_real (var_list, var_str ("ms"), 100._default, &
         is_known = .true.)
    call phs_data%init (process_data, model)
    phs_par%sqrts = 1000
    phs_par%off_shell = 1
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_parameters (phs_par)
       phs_data%sqrts = phs_par%sqrts
       phs_data%par%sqrts = phs_par%sqrts
    end select
    call phs_data%compute_md5sum ()

    write (u, "(1x,A,A,A)")  "MD5 sum (process)    = '", &
         phs_data%md5sum_process, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (model par)  = '", &
         phs_data%md5sum_model_par, "'"
    write (u, "(1x,A,A,A)")  "MD5 sum (phs config) = '", &
         phs_data%md5sum_phs_config, "'"

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%read_phs_file (exist, found, match)
       write (u, "(1x,A,L1)")  "exist = ", exist
       write (u, "(1x,A,L1)")  "found = ", found
       write (u, "(1x,A,L1)")  "match = ", match
    end select

    write (u, "(A)")
    write (u, "(A)")  "* Cleanup"

    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_6"

  end subroutine phs_wood_6

  subroutine phs_wood_vis_1 (u)
    integer, intent(in) :: u
    type(os_data_t) :: os_data
    type(model_list_t) :: model_list
    type(model_t), pointer :: model
    type(process_constants_t) :: process_data
    class(phs_config_t), allocatable :: phs_data
    type(mapping_defaults_t) :: mapping_defaults
    type(string_t) :: vis_file, pdf_file, ps_file
    real(default) :: sqrts
    logical :: exist, exist_pdf, exist_ps
    integer :: u_phs, iostat, u_vis
    character(95) :: buffer
    
    write (u, "(A)")  "* Test output: phs_wood_vis_1"
    write (u, "(A)")  "*   Purpose: visualizing the &
         &phase-space configuration"
    write (u, "(A)")
    
    call os_data_init (os_data)
    call syntax_model_file_init ()
    call model_list%read_model (var_str ("Test"), &
         var_str ("Test.mdl"), os_data, model)

    call syntax_phs_forest_init ()
    
    write (u, "(A)")  "* Initialize a process"
    write (u, "(A)")

    call init_test_process_data (var_str ("phs_wood_vis_1"), process_data)

    write (u, "(A)")  "* Create a scratch phase-space file"
    write (u, "(A)")

    u_phs = free_unit ()
    open (u_phs, status = "scratch", action = "readwrite")
    call write_test_phs_file (u_phs, var_str ("phs_wood_vis_1"))
    rewind (u_phs)
    do
       read (u_phs, "(A)", iostat = iostat)  buffer
       if (iostat /= 0)  exit
       write (u, "(A)") trim (buffer)
    end do

    write (u, "(A)")
    write (u, "(A)")  "* Setup phase-space configuration object"
    write (u, "(A)")

    mapping_defaults%step_mapping = .false.

    allocate (phs_wood_config_t :: phs_data)
    call phs_data%init (process_data, model)
    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%set_input (u_phs)
       call phs_data%set_mapping_defaults (mapping_defaults)
       phs_data%os_data = os_data
       phs_data%io_unit = 0
       phs_data%io_unit_keep_open = .true.
       phs_data%vis_channels = .true.
    end select

    sqrts = 1000._default
    call phs_data%configure (sqrts)
       
    call phs_data%write (u)
    write (u, "(A)")

    select type (phs_data)
    type is (phs_wood_config_t)
       call phs_data%write_forest (u)
    end select
    
    vis_file = "phs_wood_vis_1_phs.tex"
    ps_file  = "phs_wood_vis_1_phs.ps"
    pdf_file = "phs_wood_vis_1_phs.pdf"    
    inquire (file = char (vis_file), exist = exist)
    if (exist) then
       u_vis = free_unit ()
       open (u_vis, file = char (vis_file), action = "read", status = "old")
       iostat = 0
       do while (iostat == 0)
          read (u_vis, "(A)", iostat = iostat)  buffer
          if (iostat == 0)  write (u, "(A)")  trim (buffer)
       end do
       close (u_vis)
    else
       write (u, "(A)")  "[Visualize LaTeX file is missing]"
    end if
    inquire (file = char (ps_file), exist = exist_ps)
    if (exist_ps) then
       write (u, "(A)")  "[Visualize Postscript file exists and is nonempty]"
    else
       write (u, "(A)")  "[Visualize Postscript file is missing/non-regular]"
    end if
    inquire (file = char (pdf_file), exist = exist_pdf)
    if (exist_pdf) then
       write (u, "(A)")  "[Visualize PDF file exists and is nonempty]"
    else
       write (u, "(A)")  "[Visualize PDF file is missing/non-regular]"
    end if        
    
    write (u, "(A)")
    write (u, "(A)")  "* Cleanup"

    close (u_phs)
    call phs_data%final ()
    call model_list%final ()
    call syntax_model_file_final ()
    
    write (u, "(A)")
    write (u, "(A)")  "* Test output end: phs_wood_vis_1"

  end subroutine phs_wood_vis_1


end module phs_wood
