/************************************************************************/
/* File		webpublishparse.cpp					*/
/*									*/
/* Purpose	This C++ program file contains private WebPublish	*/
/*		member functions. The private WebPublish member		*/
/*		functions that are contained in this file include the	*/
/*		functions that support the WebPublish class.		*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	0.1.0 (Prototype)					*/
/*									*/
/* Date		Sunday, July 7, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

/* Include	These are the header files used by program file.	*/

#include <getopt.h>			// 'getopt_long' function.
#include <stdlib.h>			// atoi function.
#include "webpublish.h"			// WebPublish class.

/* Local	These member variables contain the data that is needed	*/
/*		to parse the command line arguments. The structure of	*/
/*		these variables is dictated by the 'getopt_long'	*/
/*		function.						*/

static int		operation;
static char		shortoptions[]	= "hvVfa:w:s:u:p:d:e:P:D:m:F:";
static struct option	longoptions[]	=
   {
      { "help",		false,	0,		'h'		},
      { "version",	false,	0,		'v'		},
      { "verbose",	false,	0,		'V'		},
      { "force",	false,	0,		'f'		},
      { "prompt",	false,	0,		1		},

      { "account",	true,	0,		'a'		},
      { "website",	true,	0,		'w'		},
      { "server",	true,	0,		's'		},
      { "user",		true,	0,		'u'		},
      { "password",	true,	0,		'p'		},
      { "directory",	true,	0,		'd'		},
      { "extension",	true,	0,		'e'		},
      { "path",		true,	0,		'P'		},
      { "depth",	true,	0,		'D'		},
      { "mode",		true,	0,		'm'		},
      { "file",		true,	0,		'F'		},

      { "publish",
         false,	&operation,	WebPublishDoPublish	},
      { "synchronize",
         false,	&operation,	WebPublishDoSynch	},
      { "add",
         false, &operation,	WebPublishDoAdd		},
      { "change",
         false, &operation,	WebPublishDoChange	},
      { "remove",
         false, &operation,	WebPublishDoRemove	},
      { "list",
         false, &operation,	WebPublishDoList	},

      { 0,
         0,		0,		0		}
   };

/************************************************************************/
/* Function	store_parse_option					*/
/*									*/
/* Notes	This WebPublish member function is responsible for	*/
/*		storing the argument for a given command line argument	*/
/*		in its correct storage area. If there is already data	*/
/*		stored then 'itserror' is set to WEBPUBERR_REPOPT which	*/
/*		means that an option has been repeated.			*/
/*									*/
/* Accepts	This function expects a pointer to a string pointer in	*/
/*		'charpntr' and this function expects 'optarg' to point	*/
/*		to the argument to store.				*/
/*									*/
/* Returns	This function does not return a value. However,		*/
/*		'itserror' can be tested to see if any errors were	*/
/*		found. If 'itserror' is set to WEBPUBERR_NOERROR then	*/
/*		no problems were found.					*/
/************************************************************************/

status WebPublish::store_parse_option(String& item, char *optarg)
   {
      status		result		= OK;

      if (item.Length() > 0)
         {
            itserror	= WebPublishRepeatCmd;
	    result	= ERROR;
	    Report_Error();
         }
      else
         {
	    item	= optarg;
         }
      return(result);
   }

/************************************************************************/
/* Function	parse_command_line_arguments(int argc, char** argv)	*/
/*									*/
/* Purpose	This WebPublish member function will parse the command	*/
/*		line options that were passed to the WebPublish class	*/
/*		constructor.						*/
/*									*/
/* Input	'argc' and 'argv' must contain the command line options	*/
/*		to be parsed. These are the arguments that were passed	*/
/*		to the WebPublish constructor when a WebPublish object	*/
/*		is created.						*/
/*									*/
/* Returns	This function does not return a value. However,		*/
/*		'itserror' can be tested to see if any erros were	*/
/*		found. If 'itserror' is set to WEBPUBERR_NOERROR then	*/
/*		no problems were found.					*/
/************************************************************************/

status WebPublish::parse_command_line_arguments(int argc, char** argv)
   {
      status		result		= OK;
      register int	index;
      int		option;			// Switch value.
      int		flagindex	= 0;	// Cmd line flag index.

      while ((option = getopt_long(argc, argv, shortoptions, longoptions,
	 &flagindex)) != EOF && result == OK)
         {
            switch(option)
               {
                  /* Processed a long option.				*/

                  case 0:
                     if (itsoperation != (unsigned)WebPublishNoCommand)
                        {
                           itserror		= WebPublishNoAct;
			   result		= ERROR;
			   Report_Error();
                        }
                     else
                        {
                           itsoperation		= operation;
                        }
                     break;

                  case 'h':			// -h or --help
                     itshelpflag	= true;
                     break;
                  case 'v':			// -v or --version
                     itsversionflag	= true;
                     break;
                  case 'V':			// -V or --verbose
                     itsverboseflag	= true;
                     break;
                  case 'f':			// -f or --force
                     itsforceflag	= true;
                     break;
		  case 1:			// --prompt
		     itspromptflag	= true;
		     break;

                  case 'a':			// -a or --account
		     result	= store_parse_option(
				  itsaccount, optarg);
                     break;
                  case 'w':			// -w or --website
		     result	= store_parse_option(
				  itswebsite, optarg);
                     break;
                  case 's':			// -s or --server
		     result	= store_parse_option(
				  itsserver, optarg);
                     break;
                  case 'u':			// -u or --user
		     result	= store_parse_option(
				  itsuser, optarg);
                     break;
                  case 'p':			// -p or --password
		     result	= store_parse_option(
				  itspassword, optarg);
                     break;
                  case 'd':			// -d or --directory
		     result	= store_parse_option(
				  itsdirectory, optarg);
                     break;
                  case 'e':			// -e or --extension
		     result	= store_parse_option(
				  itsextension, optarg);
                     break;
                  case 'P':			// -P or --path
		     result	= store_parse_option(
				  itspath, optarg);
                     break;
                  case 'D':			// -D or --depth
		     result	= store_parse_option(
				  itsdepthtext, optarg);
		     if (result == OK)
		        {
			   itsdepth	= atoi(optarg);
			}
                     break;
                  case 'm':			// -m or --mode
		     result	= store_parse_option(
				  itsmodetext, optarg);
		     if (result == OK)
		        {
		           if (itsmodetext.Case_Compare("text") == EQUAL)
		              {
			         itsmode	= FTPText;
			      }
		           else if (itsmodetext.Case_Compare("binary")
			      == EQUAL)
		              {
			         itsmode	= FTPBinary;
			      }
		           else
		              {
			         /* Invalid file transfer mode.		*/

			         itserror	= WebPublishNotMode;
			         result		= ERROR;
			         Report_Error();
			      }
			}
                     break;
                  case 'F':			// -F or --file
		     result	= store_parse_option(
				  itsdbfile, optarg);
		     if (itsdbfile.Case_Compare("account") == EQUAL)
		        {
			   itsfileitem	= WebPublishAccounts;
			}
		     else if (itsdbfile.Case_Compare("shell") == EQUAL)
		        {
			   itsfileitem	= WebPublishShell;
			}
		     else if (itsdbfile.Case_Compare("synch") == EQUAL)
		        {
			   itsfileitem	= WebPublishSynch;
			}
		     else if (itsdbfile.Case_Compare("publish") == EQUAL)
		        {
			   itsfileitem	= WebPublishPublish;
			}
		     else if (itsdbfile.Case_Compare("mode") == EQUAL)
		        {
			   itsfileitem	= WebPublishModes;
			}
		     else
		        {
			   /* Unknown database file.			*/

			   itserror		= WebPublishBadDBFile;
			   itserrorinfo		= "Unkown database "
			   			  "file = ";
			   itserrorinfo		+= itsdbfile;
			   result		= ERROR;
			   Report_Error();
		        }
                     break;

                  default:				// Unsupported
                     itserror		= WebPublishBadCmd;
		     result		= ERROR;
		     Report_Error();
                     std::cerr << "Use -h or --help for a list of options.\n";
                     break;
               }
         }
      if (result == OK)
         {
            /* Assume that all remaining arguments are individual files	*/
            /* to be transferred.					*/

	    index		= 0;
	    itssinglecount	= argc - optind;
	    itssingles		= new String[itssinglecount];
            while (optind < argc)
               {
	          itssingles[index++]	= argv[optind++];
               }
         }
       return(result);
   }
