/************************************************************************/
/* File		datarecord.h						*/
/*									*/
/* Purpose	This C++ program file contains the definition for the	*/
/*		DataRecord class. The DataRecord class defines the	*/
/*		record structure of the records that are managed by the	*/
/*		DataBase class.						*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, June 23, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <time.h>			// struct tm.
#include <stdio.h>			// sprintf
#include <iostream>			// cerr.
#include "datarecord.h"			// DataRecord class

/************************************************************************/
/* Function	DataRecord()						*/
/*									*/
/* Purpose	This is the default constructor for the DataRecord	*/
/*		class. This constructor simply initializes an empty	*/
/*		DataRecord object. The record will still need to have	*/
/*		its element list set before it can be used to store	*/
/*		data.							*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataRecord::DataRecord()
   {
      itselementcount	= 0;
      itselements	= (DataElement*)0;
   }

/************************************************************************/
/* Function	DataRecord(const int count, const DataElementType* list)*/
/*									*/
/* Purpose	This is an alternate constructor for the DataRecord	*/
/*		class. This alternate constructor will initialize a	*/
/*		DataRecord object with the number of DataElements	*/
/*		specified in the variable 'count'. The DataElements	*/
/*		will be initialized with the list of types that are	*/
/*		specified in the array 'list'.				*/
/*									*/
/* Input	This function expects the variable 'count' to contain	*/
/*		the number of DataElements that are a part of the	*/
/*		DataRecord. The variable 'list' must point to an array	*/
/*		of DataElementType types that specify the type of each	*/
/*		DataElement in the DataRecord.				*/
/*									*/
/* Output	This function will initialize a DataRecord object with	*/
/*		an array of DataElement objects. The number of		*/
/*		DataElement objects is specified by the variable	*/
/*		'count'. Each DataElement in the DataRecord will be	*/
/*		initialized with the type that is specified in the	*/
/*		array of DataElementType types that is specified by the	*/
/*		variable 'list'.					*/
/************************************************************************/

DataRecord::DataRecord(const int count, const DataElementType* list)
   {
      itselementcount	= 0;
      itselements	= (DataElement*)0;
      set_record_elements(count, list);
   }

/************************************************************************/
/* Function	DataRecord(const DataRecord& record)			*/
/*									*/
/* Purpose	This is the copy constructor for a DataRecord object.	*/
/*		This copy constructor will make a deep copy of the	*/
/*		object that is specified by the variable 'record'.	*/
/*									*/
/* Input	This function expects the variable 'record' to be a	*/
/*		reference to the DataRecord object that is to be	*/
/*		copied.							*/
/*									*/
/* Output	The DataRecord object will be a deep copy of 'record'.	*/
/************************************************************************/

DataRecord::DataRecord(const DataRecord& record)
   {
      Copy_Record(record);
   }

/************************************************************************/
/* Function	~DataRecord()						*/
/*									*/
/* Purpose	This is the default destructor for a DataRecord object.	*/
/*		If the object is using memory from the heap, this	*/
/*		destructor will delete it.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataRecord::~DataRecord()
   {
      if (itselements != (DataElement*)0)
         {
            delete [] itselements;
	 }
   }

/************************************************************************/
/* Function	int Record_Size(void)					*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. The	*/
/*		function returns the number of bytes that are required	*/
/*		to write the object to a file.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the number of bytes that are	*/
/*		required to write the object to a file.			*/
/************************************************************************/

int DataRecord::Record_Size(void)
   {
      register int	index;
      int		size		= sizeof(int);

      for (index = 0; index < itselementcount; index++)
         {
	    size	+= itselements[index].Record_Size();
	 }
      return(size);
   }

/************************************************************************/
/* Function	status Read_From_File(File& file, const int filehandle)	*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will read a DataRecord object from a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		reading the object from the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataElement object is to be read	*/
/*		from.							*/
/*									*/
/* Output	If this function is able to read in a DataRecord	*/
/*		object from 'file' then this function will return OK	*/
/*		and the DataRecord object will contain the contents	*/
/*		that were read in from the file. Any previous data that	*/
/*		the DataRecord object contained will be lost. If this	*/
/*		function is not able to read in a DataRecord object	*/
/*		from 'file' then this function will return ERROR.	*/
/*		Errors by this function are handled by the ObjectFile	*/
/*		class.							*/
/************************************************************************/

status DataRecord::Read_From_File(File& file, const int filehandle)
   {
      status		result		= OK;
      register int	index;

      /* If the object already contains data then delete it.		*/

      if (itselementcount > 0)
         {
	    delete [] itselements;
	 }

      /* Read in the number of elements in the DataRecord.		*/

      if (file.Read_File(filehandle, (void*)&itselementcount,
         sizeof(int)) == ERROR)
         {
	    /* Error writing record to file.				*/

	    file.Set_Error_Info_String(
	       "Could not read record's element count from file");
	    itselements		= (DataElement*)0;
	    result		= ERROR;
	 }
      else
         {
	    /* Allocate the record's array of DataElement objects.	*/

            itselements	= new DataElement[itselementcount];
	 }

      /* Read in each DataElement that belongs to this record.		*/

      for (index = 0; index < itselementcount && result == OK; index++)
         {
	    result	= itselements[index].Read_From_File(file,
			  filehandle);
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_From_File(File& file, const int filehandle)*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will write a DataRecord object to a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		writing the object to the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataRecord object is to be written	*/
/*		to.							*/
/*									*/
/* Output	If this function is able to write the DataRecord	*/
/*		object to 'file' then this function will return OK. If	*/
/*		this function is not able to write a DataRecord object	*/
/*		to 'file' then this function will return ERROR. Errors	*/
/*		by this function are handled by the ObjectFile class.	*/
/************************************************************************/

status DataRecord::Write_To_File(File& file, const int filehandle)
   {
      status		result		= OK;
      register int	index;

      /* Write the number of elements that are in this record.		*/

      if (file.Write_File(filehandle, (void*)&itselementcount,
         sizeof(int)) == ERROR)
         {
	    /* Error writing record to file.				*/

	    file.Set_Error_Info_String(
	       "Could not write record's element count to file");
	    result		= ERROR;
	 }

      /* Write each element that is a part of this record.		*/

      for (index = 0; index < itselementcount && result == OK; index++)
         {
	    result	= itselements[index].Write_To_File(file,
			  filehandle);
	 }
      return(result);
   }

/************************************************************************/
/* Function	void Set_Record_Elements(const int count,		*/
/*		   const DataElementType* list)				*/
/*									*/
/* Purpose	This function will initialize the list of elements that	*/
/*		are a part of the DataRecord object. The previous list	*/
/*		of elements, if any, will be deleted.			*/
/*									*/
/* Input	This function expects the variable 'count' to contain	*/
/*		the number of elements that the record should contain.	*/
/*		The variable 'list' should point to an array of		*/
/*		DataElementType types that specify the type of each	*/
/*		element in the record.					*/
/*									*/
/* Output	The original contents of the record, if any, will be	*/
/*		deleted and the record will be initialized to contain	*/
/*		the elements that are specified with the variable	*/
/*		'count' and the list of types in the variable 'list'.	*/
/************************************************************************/

void DataRecord::Set_Record_Elements(const int count,
   const DataElementType* list)
   {
      set_record_elements(count, list);
   }

/************************************************************************/
/* Function	DataElementType Get_Element_Type(			*/
/*		   const int element) const				*/
/*									*/
/* Purpose	This function can be used to get the type of a specific	*/
/*		element in a DataRecord. If the index is out of range	*/
/*		then this function will return DataElementUndefined.	*/
/*									*/
/* Input	This function expects the variable 'element' to		*/
/*		specify which element in the array of elements that is	*/
/*		to have its type returned.				*/
/*									*/
/* Output	If the element specified by the variable 'element' is	*/
/*		not out of range then this function will return the	*/
/*		type for the specified element. If the variable		*/
/*		'element' is out of range then this function will	*/
/*		report a warning message to stderr and return		*/
/*		DataElementUndefined.					*/
/************************************************************************/

DataElementType DataRecord::Get_Element_Type(const int element) const
   {
      DataElementType	result		= DataElementString;

      if (element < 0 || element >= itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to get "
	       "element type.\n";
	    result	= DataElementUndefined;
	 }
      else
         {
	    result	= itselements[element].Get_Element_Type();
	 }
      return(result);
   }

/************************************************************************/
/* Function	void Clear_Record(void)					*/
/*									*/
/* Purpose	This function will clear all of the elements in the	*/
/*		DataRecord record. Clearing the elements sets all of	*/
/*		the element's stored values to their default values.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	All of the DataElements in the DataRecord will contain	*/
/*		their respective default values.			*/
/************************************************************************/

void DataRecord::Clear_Record(void)
   {
      register int	index;

      for (index = 0; index < itselementcount; index++)
         {
	    itselements[index].Clear_Element();
	 }
   }

/************************************************************************/
/* Function	void Copy_Record(const DataRecord& record)		*/
/*									*/
/* Purpose	This function will make a copy of a DataRecord record.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The DataRecord object will be a deep copy of the	*/
/*		DataRecord object that was passed to this function in	*/
/*		the variable 'record'.					*/
/************************************************************************/

void DataRecord::Copy_Record(const DataRecord& record)
   {
      register int	index;

      if (itselements != (DataElement*)0)
         {
	    delete [] itselements;
	 }
      itselementcount	= record.itselementcount;
      if (itselementcount == 0)
         {
	    itselements	= (DataElement*)0;
	 }
      else
         {
            itselements		= new DataElement[itselementcount];
            for (index = 0; index < itselementcount; index++)
               {
	          itselements[index].Copy_Element(record.itselements[index]);
	       }
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const int element, const String& value)	*/
/*									*/
/* Purpose	This function will set the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value set then a warning	*/
/*		is written to stderr and the element will be set to its	*/
/*		default value.						*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value set. The variable 'value' must contain	*/
/*		the value that is to be stored in the element.		*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value set to	*/
/*		the value passed in the variable 'value'. If 'element'	*/
/*		is out of range then a warning message will be written	*/
/*		to stderr and nothing will be done. If 'element' is in	*/
/*		range but the element is a different type than 'value'	*/
/*		then a warning message will be written to stderr and	*/
/*		the element will be set to its default value.		*/
/************************************************************************/

void DataRecord::Set_Data(const int element, const String& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to set "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Set_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const int element, const int value)	*/
/*									*/
/* Purpose	This function will set the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value set then a warning	*/
/*		is written to stderr and the element will be set to its	*/
/*		default value.						*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value set. The variable 'value' must contain	*/
/*		the value that is to be stored in the element.		*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value set to	*/
/*		the value passed in the variable 'value'. If 'element'	*/
/*		is out of range then a warning message will be written	*/
/*		to stderr and nothing will be done. If 'element' is in	*/
/*		range but the element is a different type than 'value'	*/
/*		then a warning message will be written to stderr and	*/
/*		the element will be set to its default value.		*/
/************************************************************************/

void DataRecord::Set_Data(const int element, const int value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to set "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Set_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const int element, const float value)	*/
/*									*/
/* Purpose	This function will set the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value set then a warning	*/
/*		is written to stderr and the element will be set to its	*/
/*		default value.						*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value set. The variable 'value' must contain	*/
/*		the value that is to be stored in the element.		*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value set to	*/
/*		the value passed in the variable 'value'. If 'element'	*/
/*		is out of range then a warning message will be written	*/
/*		to stderr and nothing will be done. If 'element' is in	*/
/*		range but the element is a different type than 'value'	*/
/*		then a warning message will be written to stderr and	*/
/*		the element will be set to its default value.		*/
/************************************************************************/

void DataRecord::Set_Data(const int element, const float value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to set "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Set_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const int element,			*/
/*		   const DataElementTimeType& value)			*/
/*									*/
/* Purpose	This function will set the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value set then a warning	*/
/*		is written to stderr and the element will be set to its	*/
/*		default value.						*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value set. The variable 'value' must contain	*/
/*		the value that is to be stored in the element.		*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value set to	*/
/*		the value passed in the variable 'value'. If 'element'	*/
/*		is out of range then a warning message will be written	*/
/*		to stderr and nothing will be done. If 'element' is in	*/
/*		range but the element is a different type than 'value'	*/
/*		then a warning message will be written to stderr and	*/
/*		the element will be set to its default value.		*/
/************************************************************************/

void DataRecord::Set_Data(const int element,
   const DataElementTimeType& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to set "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Set_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const int element,			*/
/*		   const DataElementDateType& value)			*/
/*									*/
/* Purpose	This function will set the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value set then a warning	*/
/*		is written to stderr and the element will be set to its	*/
/*		default value.						*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value set. The variable 'value' must contain	*/
/*		the value that is to be stored in the element.		*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value set to	*/
/*		the value passed in the variable 'value'. If 'element'	*/
/*		is out of range then a warning message will be written	*/
/*		to stderr and nothing will be done. If 'element' is in	*/
/*		range but the element is a different type than 'value'	*/
/*		then a warning message will be written to stderr and	*/
/*		the element will be set to its default value.		*/
/************************************************************************/

void DataRecord::Set_Data(const int element,
   const DataElementDateType& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to set "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Set_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const int element, const String& value)	*/
/*									*/
/* Purpose	This function will get the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value returned then a	*/
/*		warning is written to stderr and the returned value	*/
/*		will be set to the default value for its type.		*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value returned. The variable 'value' must	*/
/*		contain a reference to the variable that is to receive	*/
/*		the element's data.					*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value written	*/
/*		to the variable that is referenced by the variable	*/
/*		'value'. If 'element' is out of range then a warning	*/
/*		message will be written to stderr and nothing will be	*/
/*		done. If 'element' is in range but the element is a	*/
/*		different type than 'value' then a warning message will	*/
/*		be written to stderr and variable that is referenced	*/
/*		by the variable 'value' will be set to the default	*/
/*		value for its type.					*/
/************************************************************************/

void DataRecord::Get_Data(const int element, String& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to get "
	       "element value.\n";
	 }
      else
         {
	    itselements[element].Get_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const int element, const int& value)	*/
/*									*/
/* Purpose	This function will get the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value returned then a	*/
/*		warning is written to stderr and the returned value	*/
/*		will be set to the default value for its type.		*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value returned. The variable 'value' must	*/
/*		contain a reference to the variable that is to receive	*/
/*		the element's data.					*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value written	*/
/*		to the variable that is referenced by the variable	*/
/*		'value'. If 'element' is out of range then a warning	*/
/*		message will be written to stderr and nothing will be	*/
/*		done. If 'element' is in range but the element is a	*/
/*		different type than 'value' then a warning message will	*/
/*		be written to stderr and variable that is referenced	*/
/*		by the variable 'value' will be set to the default	*/
/*		value for its type.					*/
/************************************************************************/

void DataRecord::Get_Data(const int element, int& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to get "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Get_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const int element, const float& value)	*/
/*									*/
/* Purpose	This function will get the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value returned then a	*/
/*		warning is written to stderr and the returned value	*/
/*		will be set to the default value for its type.		*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value returned. The variable 'value' must	*/
/*		contain a reference to the variable that is to receive	*/
/*		the element's data.					*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value written	*/
/*		to the variable that is referenced by the variable	*/
/*		'value'. If 'element' is out of range then a warning	*/
/*		message will be written to stderr and nothing will be	*/
/*		done. If 'element' is in range but the element is a	*/
/*		different type than 'value' then a warning message will	*/
/*		be written to stderr and variable that is referenced	*/
/*		by the variable 'value' will be set to the default	*/
/*		value for its type.					*/
/************************************************************************/

void DataRecord::Get_Data(const int element, float& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to get "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Get_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const int element,			*/
/*		   const DataElementTimeType& value)			*/
/*									*/
/* Purpose	This function will get the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value returned then a	*/
/*		warning is written to stderr and the returned value	*/
/*		will be set to the default value for its type.		*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value returned. The variable 'value' must	*/
/*		contain a reference to the variable that is to receive	*/
/*		the element's data.					*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value written	*/
/*		to the variable that is referenced by the variable	*/
/*		'value'. If 'element' is out of range then a warning	*/
/*		message will be written to stderr and nothing will be	*/
/*		done. If 'element' is in range but the element is a	*/
/*		different type than 'value' then a warning message will	*/
/*		be written to stderr and variable that is referenced	*/
/*		by the variable 'value' will be set to the default	*/
/*		value for its type.					*/
/************************************************************************/

void DataRecord::Get_Data(const int element, DataElementTimeType& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to get "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Get_Data(value);
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const int element,			*/
/*		   const DataElementDateType& value)			*/
/*									*/
/* Purpose	This function will get the value of a DataRecord's	*/
/*		DataElement. The value must be the same as the		*/
/*		element's type. If the variable 'value' is not the same	*/
/*		type as the element having its value returned then a	*/
/*		warning is written to stderr and the returned value	*/
/*		will be set to the default value for its type.		*/
/*									*/
/* Input	This function expects the varaible 'element' to		*/
/*		specify which element in the array of elements is to	*/
/*		have its value returned. The variable 'value' must	*/
/*		contain a reference to the variable that is to receive	*/
/*		the element's data.					*/
/*									*/
/* Output	If the value of the variable 'element' is in range, the	*/
/*		element that it specifies will have its value written	*/
/*		to the variable that is referenced by the variable	*/
/*		'value'. If 'element' is out of range then a warning	*/
/*		message will be written to stderr and nothing will be	*/
/*		done. If 'element' is in range but the element is a	*/
/*		different type than 'value' then a warning message will	*/
/*		be written to stderr and variable that is referenced	*/
/*		by the variable 'value' will be set to the default	*/
/*		value for its type.					*/
/************************************************************************/

void DataRecord::Get_Data(const int element, DataElementDateType& value)
   {
      if (element < 0 || element > itselementcount)
         {
	    /* Out of range.						*/

	    std::cerr << "Element index out of range while attempting to get "
	       "element value.\n";
	 }
      else
	 {
	    itselements[element].Get_Data(value);
	 }
   }

/************************************************************************/
/*									*/
/*									*/
/*									*/
/************************************************************************/

void DataRecord::Get_Key(String& key)
   {
      int			number;
      float			value;
      DataElementTimeType	time;
      DataElementDateType	date;
      struct tm			tm_data;
      char			buffer[100];

      switch (Get_Element_Type(0))
         {
            case DataElementString:
	       Get_Data(0, key);
	       break;
	    case DataElementNumber:
	       Get_Data(0, number);
	       sprintf(buffer, "%d", number);
	       key	= buffer;
	       break;
	    case DataElementFloat:
	       Get_Data(0, value);
	       sprintf(buffer, "%f", value);
	       key	= buffer;
	       break;
	    case DataElementTime:
	       Get_Data(0, time);
	       tm_data.tm_sec	= time.seconds;
	       tm_data.tm_min	= time.minutes;
	       tm_data.tm_hour	= time.hours;
	       strftime(buffer, 100, "%H:%M:%S", &tm_data);
	       key	= buffer;
	       break;
	    case DataElementDate:
	       Get_Data(0, date);
	       tm_data.tm_mday	= date.day;
	       tm_data.tm_mon	= date.month;
	       tm_data.tm_year	= date.year;
	       tm_data.tm_wday	= date.dayinweek;
	       tm_data.tm_yday	= date.dayinyear;
	       tm_data.tm_isdst	= date.daylight;
	       strftime(buffer, 100, "%B %d, %Y", &tm_data);
	       key	= buffer;
	       break;
	    case DataElementUndefined:
	       key	= "";
         }
   }

/************************************************************************/
/* Function	void set_record_elements(const int count,		*/
/*		   const DataElementType* list)				*/
/*									*/
/* Purpose	This private DataRecord member function is used to	*/
/*		intialize the list of DataElement's that are a part of	*/
/*		a DataRecord.						*/
/*									*/
/* Input	This function expects the variable 'count' to contain	*/
/*		the number of elements that are to be a part of the	*/
/*		DataRecord. The variable 'list' must point to an array	*/
/*		of types that specify the type of each element in the	*/
/*		record.							*/
/*									*/
/* Output	The original contents of the DataRecord, if any, will	*/
/*		be deleted and the DataRecord will be initialized with	*/
/*		the number of elements in 'count' that contain the type	*/
/*		of data that is specified in the array pointed to by	*/
/*		the variable 'list'.					*/
/************************************************************************/

void DataRecord::set_record_elements(const int count,
   const DataElementType* list)
   {
      register int	index;

      if (itselements != (DataElement*)0)
         {
	    delete [] itselements;
	 }
      itselementcount	= count;
      if (itselementcount <= 0)
         {
	    itselements	= (DataElement*)0;
	 }
      else
         {
            itselements	= new DataElement[count];
            for (index = 0; index < count; index++)
               {
	          itselements[index].Set_Element_Type(list[index]);
	       }
	 }
   }
