/************************************************************************/
/* File		dataindex.cpp						*/
/*									*/
/* Purpose	This C++ program file contains the class implementation	*/
/*		for the DataIndexRec class, the DataIndexHeader		*/
/*		class, and the DataIndex class. The DataIndex class	*/
/*		manages ObjectFile class record numbers. Each record	*/
/*		number is stored in the DataIndex file using a unique	*/
/*		key. The DataIndex file maintains the unique keys	*/
/*		using a red/black tree structure. The DataIndexRec	*/
/*		class and the DataIndexHeader class are classes that	*/
/*		support the DataIndex class.				*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, May 26, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <string.h>			// memcpy function.
#include <stdio.h>			// sprintf function.
#include "dataindex.h"			// DataIndex class.
#ifdef DEBUG
#include <iostream>			// cout
#endif

/************************************************************************/
/* Function	DataIndex()						*/
/*									*/
/* Purpose	This is the DataIndex default constructor. This		*/
/*		constructor does not do anything.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataIndex::DataIndex()
   {
      /* The following variable is the error message array that is	*/
      /* specific to the DataIndex class. The default UtilityModule	*/
      /* error values are overwritten by this DataIndex object		*/
      /* constructor.							*/

      static char *errormessages[] =
         {
             "No error",				// NoError
	     "Could not create index file",		// BadMake
	     "Could not open index file",		// BadOpen
	     "Could not close index file",		// BadClose
	     "Index file is corrupted",			// BadIndex
	     "Could not add key to index file",		// BadAdd
	     "Could not remove key from index file",	// BadRemove
	     "Could not insert new key",		// NoInsert
	     "Could not delete old key",		// NoDelete
	     "Could not read index file record",	// BadRead
	     "Could not write index file record",	// BadWrite
	     "Key already exists in index file",	// KeyDup
	     "Key does NOT exist in index file",	// KeyNot
	     "Could not find key in tree",		// BadFind
	     "Could not find next key in tree",		// BadNext
	     "Could not find previous key in tree",	// BadPrev
	     "Could not change key's value",		// BadChange
         };

      /* Reimplement the UtilityModule error variables .		*/

      itsmodulename	= "DataIndex";
      itsmaxerror	= DataIndexErrors;
      itserrormessages	= errormessages;
      itsindex		= -1;
      itsnextindex	= -1;
      itsprevindex	= -1;
      itsstacksize	= DataIndexArrayBlock;
      itsnextstacksize	= DataIndexArrayBlock;
      itsprevstacksize	= DataIndexArrayBlock;
      itsdifferences	= new relation[DataIndexArrayBlock];
      itspointers	= new FilePointer[DataIndexArrayBlock];
      itsobjects	= new DataIndexRec[DataIndexArrayBlock];
      itsnextobjects	= new DataIndexRec[DataIndexArrayBlock];
      itsprevobjects	= new DataIndexRec[DataIndexArrayBlock];
   }

/************************************************************************/
/* Function	~DataIndex()						*/
/*									*/
/* Purpose	This is the DataIndex default destructor. The default	*/
/*		destructor does nothing.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

DataIndex::~DataIndex()
   {
      delete [] itsdifferences;
      delete [] itspointers;
      delete [] itsobjects;
      delete [] itsnextobjects;
      delete [] itsprevobjects;
   }

/************************************************************************/
/* Function	status Create_Data_File(const char* filename,		*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This DataIndex function is responsible for creating	*/
/*		an DataIndex file. The DataIndex file is physically	*/
/*		created and a header structure is written to the	*/
/*		beginning of the file. The header structure contains	*/
/*		the information that is needed to store records in the	*/
/*		file. The file is closed after the header structure is	*/
/*		written.						*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain the name of the file to create. This should	*/
/*		include the path information. The file will be created	*/
/*		using the permissions that are passed to this function	*/
/*		in the variable 'mode'. If the variable 'mode' is not	*/
/*		specified then the default value of ObjectPhysicalOS is	*/
/*		used.							*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the file and write the header structure to the		*/
/*		beginning of the file. If this function was not able to	*/
/*		create the file then this function will return ERROR.	*/
/*		All errors are reported by this function to stderr.	*/
/************************************************************************/

status DataIndex::Create_DataIndex_File(const char* filename,
   const int mode)
   {
      status		result		= OK;
      char		message[65];

      sprintf(message, "Padre Software  "
		       "DataIndex File  "
		       "Version Number  "
		       "%02d.%02d.%02d        ", DataIndexVersion,
			      DataIndexRevision, DataIndexUpdata);
      itsheader.Set_Header_Message(message);
      itsheader.Set_Locked_Index_State(false);
      itsheader.Set_Root_Node(0);
      itsheader.Set_Node_Count(0);
      if (itsindexfile.Create_ObjectFile_File(filename,
         sizeof(DataIndexHeader), (void*)&itsheader, mode) == ERROR)
	 {
	    /* Could not create index file.				*/

	    itserrorinfo	= "Could not create physical file ";
	    itserrorinfo	+= filename;
	    itserror		= DataIndexBadMake;
	    result	= ERROR;
	    Report_Error();
	 }

      return(result);
   }

/************************************************************************/
/* Function	status Create_Data_File(const String& filename,		*/
/*		   const int mode)					*/
/*									*/
/* Purpose	This DataIndex function is responsible for creating	*/
/*		an DataIndex file. The DataIndex file is physically	*/
/*		created and a header structure is written to the	*/
/*		beginning of the file. The header structure contains	*/
/*		the information that is needed to store records in the	*/
/*		file. The file is closed after the header structure is	*/
/*		written.						*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		to contain the name of the file to create. This should	*/
/*		include the path information. The file will be created	*/
/*		using the permissions that are passed to this function	*/
/*		in the variable 'mode'. If the variable 'mode' is not	*/
/*		specified then the default value of ObjectPhysicalOS is	*/
/*		used.							*/
/*									*/
/* Output	This function will return OK if it was able to create	*/
/*		the file and write the header structure to the		*/
/*		beginning of the file. If this function was not able to	*/
/*		create the file then this function will return ERROR.	*/
/*		All errors are reported by this function to stderr.	*/
/************************************************************************/

status DataIndex::Create_DataIndex_File(const String& filename,
   const int mode)
   {
      return(Create_DataIndex_File(filename.Data(), mode));
   }

/************************************************************************/
/* Function	status Open_DataIndex_File(const char* filename,	*/
/*		   const int flag)					*/
/*									*/
/* Purpose	This function is responsible for opening an DataIndex	*/
/*		object's file. To open an DataIndex object's file,	*/
/*		this function will open the physical file that was	*/
/*		previously created with Create_DataIndex_File and	*/
/*		then this function will read in the file's header.	*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the full file path of the DataIndex file to	*/
/*		open. The file will be opened with the permissions	*/
/*		passed to this function in the variable 'flag'. If the	*/
/*		variable 'flag' is not specified then the default value	*/
/*		of the constant DataIndexFlag is used.			*/
/*									*/
/* Output	If this function is able to open the file then it will	*/
/*		return OK. If this function was not able to open the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::Open_DataIndex_File(const char* filename,
   const int flag)
   {
      status		result		= OK;

      itsfilename		= filename;
      if (itsindexfile.Open_ObjectFile_File(filename, (void*)&itsheader,
         flag) == ERROR)
	 {
	    /* Could not open index file.				*/

	    itserrorinfo	= "Could not open physical file ";
	    itserrorinfo	+= filename;
	    itserror		= DataIndexBadOpen;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Cannot use corrupted index file ";
	    itserrorinfo	+= filename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Open_DataIndex_File(const String& filename,	*/
/*		   const int flag)					*/
/*									*/
/* Purpose	This function is responsible for opening an DataIndex	*/
/*		object's file. To open an DataIndex object's file,	*/
/*		this function will open the physical file that was	*/
/*		previously created with Create_DataIndex_File and	*/
/*		then this function will read in the file's header.	*/
/*									*/
/* Input	This function expects the variable 'filename' to	*/
/*		contain the full file path of the DataIndex file to	*/
/*		open. The file will be opened with the permissions	*/
/*		passed to this function in the variable 'flag'. If the	*/
/*		variable 'flag' is not specified then the default value	*/
/*		of the constant DataIndexFlag is used.			*/
/*									*/
/* Output	If this function is able to open the file then it will	*/
/*		return OK. If this function was not able to open the	*/
/*		file then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::Open_DataIndex_File(const String& filename,
   const int flag)
   {
      return(Open_DataIndex_File(filename.Data(), flag));
   }

/************************************************************************/
/* Function	status Close_DataIndex_File(void)			*/
/*									*/
/* Purpose	This function is responsible for closing an DataIndex	*/
/*		file. To close the DataIndex file, this function	*/
/*		simply closes the physical file.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to close the DataIndex file	*/
/*		then this function will return OK. If this function was	*/
/*		not able to close the DataIndex file then this		*/
/*		function will return ERROR. All errors by this function	*/
/*		are reported to stderr.					*/
/************************************************************************/

status DataIndex::Close_DataIndex_File(void)
   {
      status		result		= OK;

      /* Write the header to the index file to record the locked state	*/
      /* of the index file.						*/

      if (itsindexfile.Write_ObjectFile_Header() == ERROR)
	 {
	    /* Could not close index file.				*/

	    itserrorinfo	= "Could not close physical file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadClose;
	    result		= ERROR;
	    Report_Error();
	 }
      else if (itsindexfile.Close_ObjectFile_File() == ERROR)
	 {
	    /* Could not close index file.				*/

	    itserrorinfo	= "Could not close physical file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadClose;
	    result		= ERROR;
	    Report_Error();
	 }

      return(result);
   }

/************************************************************************/
/* Function	Find_Key(const char* key, FilePointer& value)		*/
/*									*/
/* Purpose	This function can be used to locate a key in the	*/
/*		red/black tree. If the key is found in the tree then	*/
/*		the key's value will be returned in the variable	*/
/*		'value'.						*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key to search for in the red/black tree.		*/
/*									*/
/* Output	If this function is able to locate the key in the	*/
/*		red/black tree then this function will return ERROR and	*/
/*		the variable 'value' will contain the key's value. If	*/
/*		this function is not able to locate the key in the tree	*/
/*		then the variable 'value' will contain 0. If an error	*/
/*		occurrs then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::Find_Key(const char* key, FilePointer& value)
   {
      status		result		= OK;
      relation		difference;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Make sure that the index is not already locked.		*/

      if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Attempted to find ";
	    itserrorinfo	+= key;
	    itserrorinfo	+= " in file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
            value		= 0;
            workpointer	= itsheader.Get_Root_Node();
            while (workpointer != 0)
               {
	          if (read_index_record(workobject, workpointer)
		     == ERROR)
	             {
	                /* Could not find key in tree.			*/

		        itserrorinfo	= "Attempted to find ";
		        itserrorinfo	+= key;
		        itserrorinfo	+= " in index file ";
		        itserrorinfo	+= itsfilename;
		        itserror	= DataIndexBadFind;
	                result	= ERROR;
		        Report_Error();
		        itsheader.Set_Locked_Index_State(true);
		        break;
	             }
	          difference	= workobject.Get_Record_Key().
				  Compare(key);
	          if (difference == EQUAL)
	             {
	                value		= workobject.Get_Value();
		        break;
	             }
	          else if (difference == LESS)
	             {
	                workpointer	= workobject.Get_Right_Node();
	             }
	          else
	             {
	                workpointer	= workobject.Get_Left_Node();
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	Find_Key(const String key, FilePointer& value)		*/
/*									*/
/* Purpose	This function can be used to locate a key in the	*/
/*		red/black tree. If the key is found in the tree then	*/
/*		the key's value will be returned in the variable	*/
/*		'value'.						*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key to search for in the red/black tree.		*/
/*									*/
/* Output	If this function is able to locate the key in the	*/
/*		red/black tree then this function will return ERROR and	*/
/*		the variable 'value' will contain the key's value. If	*/
/*		this function is not able to locate the key in the tree	*/
/*		then the variable 'value' will contain 0. If an error	*/
/*		occurrs then this function will return ERROR. All	*/
/*		errors by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::Find_Key(const String& key, FilePointer& value)
   {
      return(Find_Key(key.Data(), value));
   }

/************************************************************************/
/* Function	status Get_First_Key(String& key, FilePointer& value)	*/
/*									*/
/* Purpose	This function can be used to return the first key in	*/
/*		the red/black tree. If the first key is found then this	*/
/*		function will return the key and its value.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to find the first record in	*/
/*		the red/black tree then this function will return OK	*/
/*		and the variable 'key' will contain the key for the	*/
/*		first record in the tree and the variable 'value' will	*/
/*		contain the first key's value. If there is no first key	*/
/*		then the variable 'key' will contain a zero length	*/
/*		string and the variable 'value' will contain 0. If	*/
/*		there were any errors then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status DataIndex::Get_First_Key(String& key, FilePointer& value)
   {
      /* Initialize the variables used to list the keys in assending	*/
      /* order and then use the Get_Next_Key function to do the work.	*/

      itsnextindex	= -1;
      itsnextnode	= itsheader.Get_Root_Node();
      return(Get_Next_Key(key, value));
   }

/************************************************************************/
/* Function	status Get_Next_Key(String& key, FilePointer& value)	*/
/*									*/
/* Purpose	This function can be used to return the next key in	*/
/*		the red/black tree. If the next key is found then this	*/
/*		function will return the key and its value.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to find the next record in	*/
/*		the red/black tree then this function will return OK	*/
/*		and the variable 'key' will contain the key for the	*/
/*		next record in the tree and the variable 'value' will	*/
/*		contain the next key's value. If there is no next key	*/
/*		then the variable 'key' will contain a zero length	*/
/*		string and the variable 'value' will contain 0. If	*/
/*		there were any errors then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status DataIndex::Get_Next_Key(String& key, FilePointer& value)
   {
      status		result		= OK;
      DataIndexRec	workobject;

      /* Make sure that the index is not already locked.		*/

      if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Attempted to get next key in file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
            key		= "";
            value		= 0;

            /* This is the loop that will search through the nodes	*/
	    /* looking for the next key in the tree.			*/

            while (itsnextnode != 0)
               {
	          if (read_index_record(workobject, itsnextnode)
		     == ERROR)
	             {
	                itserrorinfo	= "Attempted to get next key in "
					  "file ";
		        itserrorinfo	+= itsfilename;
		        itserror	= DataIndexBadNext;
	                result	= ERROR;
		        Report_Error();
		        itsheader.Set_Locked_Index_State(true);
		        break;
	             }
	          push_nextrecord(workobject);
	          itsnextnode		= workobject.Get_Left_Node();
	       }
            if (result == OK && itsnextindex > -1)
               {
	          key		= itsnextobjects[itsnextindex].
				  Get_Record_Key();
	          value		= itsnextobjects[itsnextindex].
				  Get_Value();
	          itsnextnode	= itsnextobjects[itsnextindex].
				  Get_Right_Node();
	          itsnextindex--;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Get_Last_Key(String& key, FilePointer& value)	*/
/*									*/
/* Purpose	This function can be used to return the last key in	*/
/*		the red/black tree. If the last key is found then this	*/
/*		function will return the key and its value.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to find the last record in	*/
/*		the red/black tree then this function will return OK	*/
/*		and the variable 'key' will contain the key for the	*/
/*		last record in the tree and the variable 'value' will	*/
/*		contain the last key's value. If there is no last key	*/
/*		then the variable 'key' will contain a zero length	*/
/*		string and the variable 'value' will contain 0. If	*/
/*		there were any errors then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status DataIndex::Get_Last_Key(String& key, FilePointer& value)
   {
      /* Initialize the variables used to list the keys in descending	*/
      /* order and then use the Get_Previous_Key function to do the	*/
      /* work.								*/

      itsprevindex	= -1;
      itsprevnode	= itsheader.Get_Root_Node();
      return(Get_Previous_Key(key, value));
   }

/************************************************************************/
/* Function	status Get_Previous_Key(String& key, FilePointer& value)*/
/*									*/
/* Purpose	This function can be used to return the previous key in	*/
/*		the red/black tree. If the previous key is found then	*/
/*		this function will return the key and its value.	*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If this function is able to find the previous record in	*/
/*		the red/black tree then this function will return OK	*/
/*		and the variable 'key' will contain the key for the	*/
/*		previous record in the tree and the variable 'value'	*/
/*		will contain the previous key's value. If there is no	*/
/*		previous key then the variable 'key' will contain a	*/
/*		zero length string and the variable 'value' will	*/
/*		contain 0. If there were any errors then this function	*/
/*		will return ERROR. All errors by this function are	*/
/*		reported to stderr.					*/
/************************************************************************/

status DataIndex::Get_Previous_Key(String& key, FilePointer& value)
   {
      status		result		= OK;
      DataIndexRec	workobject;

      /* Make sure that the index is not already locked.		*/

      if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Attempted to get previous key in "
				  "file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
            key		= "";
            value		= 0;

            /* This is the loop that will search through the nodes	*/
	    /* looking for the previous key in the tree.		*/

            while (itsprevnode != 0)
               {
	          if (read_index_record(workobject, itsprevnode)
		     == ERROR)
	             {
	                itserrorinfo	= "Attempted to get next key in "
					  "file ";
		        itserrorinfo	+= itsfilename;
		        itserror	= DataIndexBadPrev;
	                result	= ERROR;
		        Report_Error();
		        itsheader.Set_Locked_Index_State(true);
		        break;
	             }
	          push_prevrecord(workobject);
	          itsprevnode		= workobject.Get_Right_Node();
	       }
            if (result == OK && itsprevindex > -1)
               {
	          key		= itsprevobjects[itsprevindex].
				  Get_Record_Key();
	          value		= itsprevobjects[itsprevindex].
				  Get_Value();
	          itsprevnode	= itsprevobjects[itsprevindex].
				  Get_Left_Node();
	          itsprevindex--;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Change_Value(const char* key,			*/
/*		   const FilePointer value)				*/
/*									*/
/* Purpose	This function can be used to change a key's value. That	*/
/*		is, to change the record number that is associated with	*/
/*		the given key.						*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key that is to have its associated value changed. The	*/
/*		variable 'value' must contain the new value that the	*/
/*		key's associated value is changed to.			*/
/*									*/
/* Output	If this function is able to change a key's value to the	*/
/*		new value then this function will return OK. If this	*/
/*		function is not able to change a key's value to the new	*/
/*		value then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::Change_Value(const char* key, const FilePointer value)
   {
      status		result		= OK;
      relation		difference;
      FilePointer	workpointer;
      DataIndexRec	workobject;

      /* Make sure that the index is not already locked.		*/

      if (itsheader.Is_Index_Locked() == true)
         {
	    /* Index is corrupted.					*/

	    itserrorinfo	= "Attempted to change value for ";
	    itserrorinfo	+= key;
	    itserrorinfo	+= " in file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadIndex;
	    result		= ERROR;
	    Report_Error();
	 }
      else
         {
            workpointer	= itsheader.Get_Root_Node();
            while (workpointer != 0)
               {
	          if (read_index_record(workobject, workpointer)
		     == ERROR)
	             {
	                /* Could not change key's value.		*/

		        itserrorinfo	= "Attempted to change value of ";
		        itserrorinfo	+= key;
		        itserrorinfo	+= " in index file ";
		        itserrorinfo	+= itsfilename;
		        itserror	= DataIndexBadChange;
	                result	= ERROR;
		        Report_Error();
		        itsheader.Set_Locked_Index_State(true);
		        break;
	             }
	          difference	= workobject.Get_Record_Key().
				  Compare(key);
	          if (difference == EQUAL)
	             {
	                workobject.Set_Value(value);
		        if (write_index_record(workobject, workpointer)
		           == ERROR)
		           {
		              /* Could not change key's value.		*/

		              itserrorinfo	= "Attempted to change "
						  "value of ";
		              itserrorinfo	+= key;
		              itserrorinfo	+= " in index file ";
		              itserrorinfo	+= itsfilename;
		              itserror	= DataIndexBadChange;
	                      result		= ERROR;
		              Report_Error();
			      itsheader.Set_Locked_Index_State(true);
		           }
		        break;
	             }
	          else if (difference == LESS)
	             {
	                workpointer	= workobject.Get_Right_Node();
	             }
	          else
	             {
	                workpointer	= workobject.Get_Left_Node();
	             }
	       }
            if (result == OK && workpointer == 0)
               {
	          /* Key does not exist.				*/

	          itserrorinfo	= "Attempted to change value of ";
	          itserrorinfo	+= key;
	          itserrorinfo	+= " in index file ";
	          itserrorinfo	+= itsfilename;
	          itserror	= DataIndexKeyNot;
	          result	= ERROR;
	          Report_Error();
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Change_Value(const String& key,			*/
/*		   const FilePointer value)				*/
/*									*/
/* Purpose	This function can be used to change a key's value. That	*/
/*		is, to change the record number that is associated with	*/
/*		the given key.						*/
/*									*/
/* Input	This function expects the variable 'key' to contain the	*/
/*		key that is to have its associated value changed. The	*/
/*		variable 'value' must contain the new value that the	*/
/*		key's associated value is changed to.			*/
/*									*/
/* Output	If this function is able to change a key's value to the	*/
/*		new value then this function will return OK. If this	*/
/*		function is not able to change a key's value to the new	*/
/*		value then this function will return ERROR. All errors	*/
/*		by this function are reported to stderr.		*/
/************************************************************************/

status DataIndex::Change_Value(const String& key, const FilePointer value)
   {
      return(Change_Value(key.Data(), value));
   }

/************************************************************************/
/* Function	void expand_stack(void)					*/
/*									*/
/* Purpose	When the stack that is used to add and remove keys from	*/
/*		the red/black tree needs to be expanded to hold more	*/
/*		records, this function is called.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will increase the size of the stack that	*/
/*		is used to add and remove keys from the tree by the	*/
/*		amount of the constant 'DataIndexArrayBlock'.		*/
/************************************************************************/

void DataIndex::expand_stack(void)
   {
      register int	index;
      int		newstacksize;
      relation*		newdifferences;
      FilePointer*	newpointers;
      DataIndexRec*	newobjects;

      /* Allocate the larger arrays.					*/

      newstacksize	= itsstacksize + DataIndexArrayBlock;
      newdifferences	= new relation[newstacksize];
      newpointers	= new FilePointer[newstacksize];
      newobjects	= new DataIndexRec[newstacksize];

      /* Copy the current arrays into the larger arrays.		*/

      memcpy(newdifferences, itsdifferences,
         sizeof(relation[itsstacksize]));
      memcpy(newpointers, itspointers,
         sizeof(FilePointer[itsstacksize]));
      for (index = 0; index < itsstacksize; index++)
         {
	    newobjects[index]	= itsobjects[index];
	 }

      /* Delete the original arrays.					*/

      delete [] itsdifferences;
      delete [] itspointers;
      delete [] itsobjects;

      /* Replace the old arrays with the larger arrays.			*/

      itsstacksize	= newstacksize;
      itsdifferences	= newdifferences;
      itspointers	= newpointers;
      itsobjects	= newobjects;
   }

/************************************************************************/
/* Function	void expand_nextstack(void)				*/
/*									*/
/* Purpose	When the stack that is used to find the next key in the	*/
/*		red/black tree needs to be expanded to hold more	*/
/*		records, this function is called.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will increase the size of the stack that	*/
/*		is used to find the next key in the tree by the amount	*/
/*		of the constant 'DataIndexArrayBlock'.			*/
/************************************************************************/

void DataIndex::expand_nextstack(void)
   {
      register int	index;
      int		newstacksize;
      DataIndexRec*	newobjects;

      /* Allocate the larger array.					*/

      newstacksize	= itsnextstacksize + DataIndexArrayBlock;
      newobjects	= new DataIndexRec[newstacksize];

      /* Copy the current array into the larger array.			*/

      for (index = 0; index < itsnextstacksize; index++)
         {
	    newobjects[index]	= itsnextobjects[index];
	 }

      /* Delete the original array.					*/

      delete [] itsnextobjects;

      /* Replace the old array with the larger array.			*/

      itsnextstacksize	= newstacksize;
      itsnextobjects	= newobjects;
   }

/************************************************************************/
/* Function	void expand_prevstack(void)				*/
/*									*/
/* Purpose	When the stack that is used to find the previous key in	*/
/*		the red/black tree needs to be expanded to hold more	*/
/*		records, this function is called.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will increase the size of the stack that	*/
/*		is used to find the previous key in the tree by the	*/
/*		amount of the constant 'DataIndexArrayBlock'.		*/
/************************************************************************/

void DataIndex::expand_prevstack(void)
   {
      register int	index;
      int		newstacksize;
      DataIndexRec*	newobjects;

      /* Allocate the larger array.					*/

      newstacksize	= itsprevstacksize + DataIndexArrayBlock;
      newobjects	= new DataIndexRec[newstacksize];

      /* Copy the current array into the larger array.			*/

      for (index = 0; index < itsprevstacksize; index++)
         {
	    newobjects[index]	= itsprevobjects[index];
	 }

      /* Delete the original array.					*/

      delete [] itsprevobjects;

      /* Replace the old array with the larger array.			*/

      itsprevstacksize	= newstacksize;
      itsprevobjects	= newobjects;
   }

/************************************************************************/
/* Function	void push_record(DataIndexRec& object,			*/
/*		   FilePointer pointer, relation difference)		*/
/*									*/
/* Purpose	This function will push a record onto the stack that is	*/
/*		used to add and remove keys from the red/black tree.	*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		a reference to a DataIndexRec object. A deep copy of	*/
/*		the object is pushed onto the stack. The variable	*/
/*		'pointer' must contain the index file record pointer	*/
/*		for the DataIndexRec object. Finally, the variable	*/
/*		'difference' must contain a relation to push onto the	*/
/*		stack.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndex::push_record(DataIndexRec& object, FilePointer pointer,
   relation difference)
   {
      if (++itsindex == itsstacksize)
         {
	    expand_stack();
	 }
      itsdifferences[itsindex]	= difference;
      itspointers[itsindex]	= pointer;
      itsobjects[itsindex]	= object;
   }

/************************************************************************/
/* Function	void push_nextrecord(DataIndexRec& object)		*/
/*									*/
/* Purpose	This function will push a record onto the stack that is	*/
/*		used to find the next key in the red/black tree.	*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		a reference to the object that is to be pushed onto the	*/
/*		stack. A deep copy of the object is pushed onto the	*/
/*		stack.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndex::push_nextrecord(DataIndexRec& object)
   {
      if (++itsnextindex == itsnextstacksize)
         {
	    expand_nextstack();
	 }
      itsnextobjects[itsnextindex]	= object;
   }

/************************************************************************/
/* Function	void push_prevrecord(DataIndexRec& object)		*/
/*									*/
/* Purpose	This function will push a record onto the stack that is	*/
/*		used to find the previous key in the red/black tree.	*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		a reference to the object that is to be pushed onto the	*/
/*		stack. A deep copy of the object is pushed onto the	*/
/*		stack.							*/
/*									*/
/* Output	None.							*/
/************************************************************************/

void DataIndex::push_prevrecord(DataIndexRec& object)
   {
      if (++itsprevindex == itsprevstacksize)
         {
	    expand_prevstack();
	 }
      itsprevobjects[itsprevindex]	= object;
   }

/************************************************************************/
/* Function	status read_index_record(DataIndexRec& object,		*/
/*		   FilePointer recordpointer)				*/
/*									*/
/* Purpose	This function is responsible for reading an index file	*/
/*		record from the index file.				*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		a reference to the object that is to receive the record	*/
/*		from the file. The variable 'recordpointer' must	*/
/*		contain the record pointer for the record in the index	*/
/*		file that is to be read in.				*/
/*									*/
/* Output	If this function was able to read in the record from	*/
/*		the index file then this function will return OK and	*/
/*		the variable referenced by 'object' will contain the	*/
/*		index file record. If this function was not able to	*/
/*		read in the index file record then this function will	*/
/*		return ERROR. All errors by this function are reported	*/
/*		to stderr.						*/
/************************************************************************/

status DataIndex::read_index_record(DataIndexRec& object,
   FilePointer recordpointer)
   {
      status		result		= OK;

      if (itsindexfile.Read_ObjectFile_Record(object, recordpointer)
         == ERROR)
         {
	    /* Could not read index record.				*/

	    itserrorinfo	= "Attempted to read record from index "
				  "file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadRead;
	    result		= ERROR;
	    Report_Error();
	 }

      return(result);
   }

/************************************************************************/
/* Function	status write_index_record(DataIndexRec& object,		*/
/*		   FilePointer recordpointer)				*/
/*									*/
/* Purpose	This function is responsible for writing an index file	*/
/*		record to the index file.				*/
/*									*/
/* Input	This function expects the variable 'object' to contain	*/
/*		a reference to the object that contains the record that	*/
/*		is to be written to the index file. The variable	*/
/*		'recordpointer' must contain the record pointer for the	*/
/*		record in the index file that is to be read in.		*/
/*									*/
/* Output	If this function was able to write the record to the	*/
/*		index file then this function will return OK and the	*/
/*		variable referenced by 'object' have been written to	*/
/*		the index file. If this function was not able to write	*/
/*		the index file record then this function will return	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status DataIndex::write_index_record(DataIndexRec& object,
   FilePointer recordpointer)
   {
      status		result		= OK;

      if (itsindexfile.Write_ObjectFile_Record(object, recordpointer)
         == ERROR)
         {
	    /* Could not read index record.				*/

	    itserrorinfo	= "Attempted to write record to index "
				  "file ";
	    itserrorinfo	+= itsfilename;
	    itserror		= DataIndexBadRead;
	    result		= ERROR;
	    Report_Error();
	 }

      return(result);
   }

#ifdef DEBUG
/************************************************************************/
/* Function	status Verify_Tree(void)				*/
/*									*/
/* Purpose	This function can be used to verify the integrity of	*/
/*		the red/black tree.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	If the red/black tree is valid then this function will	*/
/*		return OK. If the tree is not valid or if there was an	*/
/*		error validating the tree then this function will	*/
/*		ERROR. All errors by this function are reported to	*/
/*		stderr.							*/
/************************************************************************/

status DataIndex::Verify_Tree(void)
   {
      status		result		= OK;
      int		count;
      int		height;

      recursive_verify_tree(itsheader.Get_Root_Node(), result, count,
         height);
      if (count != itsheader.Get_Node_Count())
         {
	    std::cout << "Tree's nodes do not match: nodes = "
	       << itsheader.Get_Node_Count() << " count = " << count
	       << std::endl;
	    result	= ERROR;
	 }
      if (result == OK)
         {
	    std::cout << "Tree is valid: nodes = " << count << " height = "
	       << height << std::endl;
	 }
      else
         {
	    std::cout << "Tree is NOT valid!\n";
	 }
      return(result);
   }

/************************************************************************/
/* Function	void recursive_verify_tree()				*/
/*									*/
/* Purpose	This function is used to validate the integrity of the	*/
/*		red/black tree.						*/
/************************************************************************/

void DataIndex::recursive_verify_tree(FilePointer recordpointer,
   status& error, int& nodecount, int& blackheight)
   {
      int		nodes[2];
      int		blacknodes[2];
      DataIndexRec	workobject;
      DataIndexRec	workobject2;

      if (recordpointer == 0)
         {
	    nodecount	= 0;
	    blackheight	= 0;
	 }
      else if (read_index_record(workobject, recordpointer) == ERROR)
         {
	    /* Could not verify tree.					*/

	    error	= ERROR;
	 }
      else
         {
	    recursive_verify_tree(workobject.Get_Left_Node(), error,
	       nodes[0], blacknodes[0]);
	    recursive_verify_tree(workobject.Get_Right_Node(), error,
	       nodes[1], blacknodes[1]);
	    nodecount	= 1 + nodes[0] + nodes[1];
	    blackheight	= (workobject.Get_Color() == RBColorBlack)
	       + blacknodes[0];
	    if (workobject.Get_Color() != RBColorRed
	       && workobject.Get_Color() != RBColorBlack)
	       {
	          std::cout << "Node " << recordpointer
		     << " is not red or black!\n";
		  error		= ERROR;
	       }
	    if (workobject.Get_Color() == RBColorRed)
	       {
	          if (workobject.Get_Left_Node() != 0)
		     {
		        if (read_index_record(workobject2,
			   workobject.Get_Left_Node()) == ERROR)
			   {
			      /* Could not verify tree.			*/

			      error	= ERROR;
			   }
			else if (workobject2.Get_Color() == RBColorRed)
			   {
			      std::cout << "Red node " << recordpointer
			         << " has red left child!\n";
			      error	= ERROR;
			   }
		     }
	          if (workobject.Get_Right_Node() != 0)
		     {
		        if (read_index_record(workobject2,
			   workobject.Get_Right_Node()) == ERROR)
			   {
			      /* Could not verify tree.			*/

			      error	= ERROR;
			   }
			else if (workobject2.Get_Color() == RBColorRed)
			   {
			      std::cout << "Red node " << recordpointer
			         << " has red right child!\n";
			      error	= ERROR;
			   }
		     }
	       }
	    if (blacknodes[0] != blacknodes[1])
	       {
	          std::cout << "Node " << recordpointer
		     << " has two different black heights: left = "
		     << blacknodes[0] << " right = " << blacknodes[1]
		     << std::endl;
		  error		= ERROR;
	       }
	 }
   }

/************************************************************************/
/* Function	status Write_Tree(void)					*/
/*									*/
/* Purpose	This function will write out the contents of the	*/
/*		red/black tree to stdout in assending order.		*/
/************************************************************************/

status DataIndex::Write_Tree(void)
   {
      status		result		=OK;

      std::cout << "Tree's root node = " << itsheader.Get_Root_Node() << std::endl;
      if (itsheader.Get_Root_Node() == 0)
         {
	    std::cout << "The tree is empty.\n";
	 }
      else
         {
            recursive_write_tree(itsheader.Get_Root_Node(), result);
	 }
      return(result);
   }

/************************************************************************/
/* Function	void recursive_write_tree()				*/
/*									*/
/* Purpose	This function will write out the contents of the	*/
/*		red/black tree to stdout in assending order.		*/
/************************************************************************/

void DataIndex::recursive_write_tree(FilePointer node, status& result)
   {
      DataIndexRec	workobject;

      if (result == OK)
         {
            if (read_index_record(workobject, node) == ERROR)
               {
	          result	= ERROR;
	       }
	  }
      if (workobject.Get_Left_Node() != 0 && result == OK)
         {
	    recursive_write_tree(workobject.Get_Left_Node(), result);
	 }
      if (result == OK)
         {
            std::cout << "Node[" << node << "] key = "
	       << workobject.Get_Record_Key().Data()
               << " color = " << (workobject.Get_Color() == RBColorRed
	       ? "RED" : "BLACK")
	       << " left node = " << workobject.Get_Left_Node()
	       << " right node = "
	       << workobject.Get_Right_Node() << std::endl;
	 }
      if (workobject.Get_Right_Node() != 0 && result == OK)
         {
	    recursive_write_tree(workobject.Get_Right_Node(), result);
	 }
   }
#endif
