/************************************************************************/
/* File		dataelement.h						*/
/*									*/
/* Purpose	This C++ program file contains the implementation for	*/
/*		the DataElement class. The DataElement class is used to	*/
/*		store individual elements of a DataBase record. Each	*/
/*		element can contain a specific type of information.	*/
/*									*/
/* Author	This C++ program file was written by Charles Henry	*/
/*		Schoonover for Padre Software. You can contact Charles	*/
/*		Henry Schoonover at charles@padresoftware.com.		*/
/*									*/
/* Owner	The contents of this C++ program file were written for	*/
/*		Padre Software. You can contact Padre Software at	*/
/*		webmaster@padresoftware.com.				*/
/*									*/
/* Version	00.00.00 (Prototype)					*/
/*									*/
/* Date		Sunday, June 23, 2002.					*/
/*									*/
/* Copyright	(C) 2002 by Padre Software Incorporated.		*/
/*		All rights are reserved.				*/
/*									*/
/*		Padre Software has released the source code in this	*/
/*		file to the public domain under the terms of the GNU	*/
/*		General Public License. (See the file COPYING).		*/
/*									*/
/*		This program is free software; you can redistribute it	*/
/*		and/or modify it under the terms of the GNU General	*/
/*		Public License as published by the Free Software	*/
/*		Foundation; either version 2 of the License, or (at	*/
/*		your option) any later version.				*/
/************************************************************************/

#include <stdlib.h>			// abort().
#include <iostream>			// cerr.
#include "objectfile.h"			// ObjectFile class.
#include "dataelement.h"		// DataElement class.

/* Static	The following static variable is used as the default	*/
/*		value for a time data element. The default value is	*/
/*		is used when there is an error setting or getting an	*/
/*		element that is of the time type.			*/

static DataElementTimeType	defaulttime =
				   {
				      0, 0, 0
				   };

/* Static	The following static variable is used as the default	*/
/*		value for a date data element. The default value is	*/
/*		is used when there is an error setting or getting an	*/
/*		element that is of the date type.			*/

static DataElementDateType	defaultdate =
				   {
				      0, 0, 0, 0, 0, 0
				   };

/************************************************************************/
/* Function	DataElement()						*/
/*									*/
/* Purpose	This is the default constructor for a DataElement	*/
/*		object. The default constructor intializes the object	*/
/*		with an undefined type.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The new DataElement object will be initialized but the	*/
/*		object will still need to have its type set before it	*/
/*		can be used to store any data.				*/
/************************************************************************/

DataElement::DataElement()
   {
      itstype		= DataElementUndefined;
      itsdata		= (void*)0;
   }

/************************************************************************/
/* Function	DataElement(const DataElementType type)			*/
/*									*/
/* Purpose	This is an alternate constructor for a DataElement	*/
/*		object. This constructor intializes the object with the	*/
/*		type that is passed to this function in the variable	*/
/*		'type'.							*/
/*									*/
/* Input	This function expects the variable 'type' to contain	*/
/*		the type of data that the object is going to store.	*/
/*									*/
/* Output	The new DataElement object will be initialized and	*/
/*		ready to store data.					*/
/************************************************************************/

DataElement::DataElement(const DataElementType type)
   {
      itstype	= DataElementUndefined;
      itsdata	= (void*)0;
      allocate_element_data(type);
   }

/************************************************************************/
/* Function	DataElement(const DataElement& element)			*/
/*									*/
/* Purpose	This is the copy constructor for a DataElement object.	*/
/*		This copy constructor will make a deep copy of the	*/
/*		DataElement object.					*/
/*									*/
/* Input	This function expects the variable 'element' to contain	*/
/*		a pointer to the DataElement object that is to be	*/
/*		copied.							*/
/*									*/
/* Output	The new object will contain a deep copy of the object	*/
/*		that was passed to this function in the variable	*/
/*		'element'.						*/
/************************************************************************/

DataElement::DataElement(const DataElement& element)
   {
      Copy_Element(element);
   }

/************************************************************************/
/* Function	~DataElement()						*/
/*									*/
/* Purpose	This is the default destructor for a DataElement	*/
/*		object. This default destructor deletes any allocated	*/
/*		memory that is being used by the object.		*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The object will be undefined.				*/
/************************************************************************/

DataElement::~DataElement()
   {
      delete_element_data();
   }

/************************************************************************/
/* Function	int Record_Size(void)					*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. The	*/
/*		function returns the number of bytes that are required	*/
/*		to write the object to a file.				*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return the number of bytes that are	*/
/*		required to write the object to a file.			*/
/************************************************************************/

int DataElement::Record_Size(void)
   {
      int		result		= 0;

      if (itstype != DataElementUndefined)
         {
	    result	= sizeof(DataElementType);
	    switch(itstype)
	       {
	          case DataElementString:
		     result	+= sizeof(int) +
				   ((String*)itsdata)->Length() + 1;
		     break;
		  case DataElementNumber:
		     result	+= sizeof(int);
		     break;
		  case DataElementFloat:
		     result	+= sizeof(float);
		     break;
		  case DataElementTime:
		     result	+= sizeof(DataElementTimeType);
		     break;
		  case DataElementDate:
		     result	+= sizeof(DataElementDateType);
		     break;
		  default:
		     std::cerr << "Invalid enumerated data element type in "
		        "determining record size.\n";
		     result	= 0;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Read_From_File(File& file, const int filehandle)	*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will read a DataElement object from a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		reading the object from the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataElement object is to be read	*/
/*		from.							*/
/*									*/
/* Output	If this function is able to read in a DataElement	*/
/*		object from 'file' then this function will return OK	*/
/*		and the DataElement object will contain the contents	*/
/*		that were read in from the file. Any previous data,	*/
/*		including type, that the DataElement object contained	*/
/*		will be lost. If this function is not able to read in a	*/
/*		DataElement object from 'file' then this function will	*/
/*		return ERROR. Errors by this function are handled by	*/
/*		the ObjectFile class.					*/
/************************************************************************/

status DataElement::Read_From_File(File& file, const int filehandle)
   {
      status		result		= OK;

      /* Delete any data that the object might already contain and then	*/
      /* read in the data type from the file.				*/

      delete_element_data();
      if (file.Read_File(filehandle, (void*)&itstype,
         sizeof(DataElementType)) == ERROR)
	 {
	     file.Set_Error_Info_String(
	        "Could not read element's type from file");
	     result	= ERROR;
	 }
      else
         {
	    /* Read the data from the file depending on the type.	*/

            switch(itstype)
               {
	          case DataElementString:
	             itsdata	= (void*)new String;
		     result	= Read_String_From_ObjectFile(file,
				  filehandle, *(String*)itsdata);
	             break;
	          case DataElementNumber:
	             itsdata	= (void*)new int;
		     result	= Read_Integer_From_ObjectFile(file,
				  filehandle, *(int*)itsdata);
	             break;
	          case DataElementFloat:
	             itsdata	= (void*)new float;
		     result	= Read_Float_From_ObjectFile(file,
				  filehandle, *(float*)itsdata);
	             break;
	          case DataElementTime:
	             itsdata	= (void*)new DataElementTimeType;
		     if (file.Read_File(filehandle, itsdata,
			sizeof(DataElementTimeType)) == ERROR)
			{
			   file.Set_Error_Info_String(
			      "Could not read time from file");
			   result	= ERROR;
			}
		     break;
	          case DataElementDate:
	             itsdata	= (void*)new DataElementDateType;
		     if (file.Read_File(filehandle, itsdata,
			sizeof(DataElementDateType)) == ERROR)
			{
			   file.Set_Error_Info_String(
			      "Could not read date from file");
			   result	= ERROR;
			}
		     break;
		  case DataElementUndefined:
		  default:
		     file.Set_Error_Info_String(
		        "Invalid enumerated data element type read in "
			"from file");
		     itstype	= DataElementUndefined;
		     itsdata	= (void*)0;
		     result	= ERROR;
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	status Write_From_File(File& file, const int filehandle)*/
/*									*/
/* Purpose	This function is required by the ObjectFile class. This	*/
/*		function will write a DataElement object to a file.	*/
/*									*/
/* Input	This function expects the variable 'file' to contain a	*/
/*		a reference to the File object that is used when	*/
/*		writing the object to the file. The variable		*/
/*		'filehandle' must contain the filehandle that specifies	*/
/*		the file that the DataElement object is to be written	*/
/*		to.							*/
/*									*/
/* Output	If this function is able to write the DataElement	*/
/*		object to 'file' then this function will return OK. If	*/
/*		this function is not able to write a DataElement object	*/
/*		to 'file' then this function will return ERROR. Errors	*/
/*		by this function are handled by the ObjectFile class.	*/
/************************************************************************/

status DataElement::Write_To_File(File& file, const int filehandle)
   {
      status		result		= OK;

      if (itstype != DataElementUndefined)
         {
	    /* Write the DataElement object's type to the file.		*/

	    if (file.Write_File(filehandle, (void*)&itstype,
	       sizeof(DataElementType)) == ERROR)
	       {
		  file.Set_Error_Info_String(
		     "Could not write element's type from file");
		  result	= ERROR;
	       }
	    else
	       {
	          /* Write the data to the file depending on the type.	*/

	          switch(itstype)
	             {
	                case DataElementString:
			   result	= Write_String_To_ObjectFile(
					  file, filehandle,
					  *(String*)itsdata);
		           break;
		        case DataElementNumber:
			   result	= Write_Integer_To_ObjectFile(
					  file, filehandle,
					  *(int*)itsdata);
		           break;
		        case DataElementFloat:
			   result	= Write_Float_To_ObjectFile(
					  file, filehandle,
					  *(float*)itsdata);
		           break;
		        case DataElementTime:
			   if (file.Write_File(filehandle, itsdata,
			      sizeof(DataElementTimeType)) == ERROR)
			      {
			         file.Set_Error_Info_String(
				    "Could not write time to file");
				 result	= ERROR;
			      }
			   break;
		        case DataElementDate:
			   if (file.Write_File(filehandle, itsdata,
			      sizeof(DataElementDateType)) == ERROR)
			      {
			         file.Set_Error_Info_String(
				    "Could not write date to file");
				 result	= ERROR;
			      }
			   break;
			case DataElementUndefined:
			default:
		           file.Set_Error_Info_String(
		              "Invalid enumerated data element type "
			      "could not be written to file");
			   itstype	= DataElementUndefined;
			   itsdata	= (void*)0;
			   result	= ERROR;
	             }
	       }
	 }
      return(result);
   }

/************************************************************************/
/* Function	void Set_Element_Type(const DataElementType type)	*/
/*									*/
/* Purpose	This function can be used to set the type of data that	*/
/*		a DataElement object can store. If the object already	*/
/*		contains data, even if it is a different type, the data	*/
/*		will be deleted and new memory will be allocated for	*/
/*		the new type.						*/
/*									*/
/* Input	This function expects the variable 'type' to contain	*/
/*		the DataElementType that specifies the type of data	*/
/*		that the DataElement object will store.			*/
/*									*/
/* Output	If the DataElement object already contains typed data,	*/
/*		the data will be deleted. After the old data has been	*/
/*		deleted, or if the DataElement object was undefined,	*/
/*		new memory will be allocated and the object will be	*/
/*		ready to store data of the new type.			*/
/************************************************************************/

void DataElement::Set_Element_Type(const DataElementType type)
   {
      allocate_element_data(type);
   }

/************************************************************************/
/* Function	DataElementType Get_Element_Type(void) const		*/
/*									*/
/* Purpose	This function can be used to determine the type of data	*/
/*		that the DataElement object can store.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	This function will return an enumerated type that	*/
/*		specifies the type of data that the DataElement object	*/
/*		can store.						*/
/************************************************************************/

DataElementType DataElement::Get_Element_Type(void) const
   {
      return(itstype);
   }

/************************************************************************/
/* Function	void Clear_Element(void)				*/
/*									*/
/* Purpose	This function can be used to set a DataElement object	*/
/*		to its default value. The default values are:		*/
/*									*/
/*		Type			Default Value			*/
/*									*/
/*		String			"" (empty string)		*/
/*		Number			0 (zero)			*/
/*		Float			0.0 (zero point zero)		*/
/*		Time			0:0:0 (midnight)		*/
/*		Date			all zeros.			*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	The DataElement object will be set to its default	*/
/*		value. If the DataElementType is undefined then a	*/
/*		warning message will be written to stderr but the	*/
/*		DataElement object will remain unchanged.		*/
/************************************************************************/

void DataElement::Clear_Element(void)
   {
      switch(itstype)
         {
	    case DataElementString:
	       *(String*)itsdata	= "";
	       break;
	    case DataElementNumber:
	       *(int*)itsdata		= 0;
	       break;
	    case DataElementFloat:
	       *(float*)itsdata		= 0.0;
	       break;
	    case DataElementTime:
	       *(DataElementTimeType*)itsdata	= defaulttime;
	       break;
	    case DataElementDate:
	       *(DataElementDateType*)itsdata	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to clear an undefined data element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "clearing element data.\n";
	 }
   }

/************************************************************************/
/* Function	void Copy_Element(const DataElement& element)		*/
/*									*/
/* Purpose	This is the copy constructor for a DataElement object.	*/
/*		This copy constructor will make a deep copy of the	*/
/*		DataElement object.					*/
/*									*/
/* Input	This function expects the variable 'element' to contain	*/
/*		a pointer to the DataElement object that is to be	*/
/*		copied.							*/
/*									*/
/* Output	The new object will contain a deep copy of the object	*/
/*		that was passed to this function in the variable	*/
/*		'element'.						*/
/************************************************************************/

void DataElement::Copy_Element(const DataElement& element)
   {
      delete_element_data();
      itstype		= element.itstype;
      switch(itstype)
         {
	    case DataElementString:
	       itsdata	= (void*)new
			  String(((String*)element.itsdata)->Data());
	       break;
	    case DataElementNumber:
	       itsdata	= (void*)new int;
	       *(int*)itsdata	= *(int*)element.itsdata;
	       break;
	    case DataElementFloat:
	       itsdata	= (void*)new float;
	       *(float*)itsdata	= *(float*)element.itsdata;
	       break;
	    case DataElementTime:
	       itsdata	= (void*)new DataElementTimeType;
	       *(DataElementTimeType*)itsdata	= *(DataElementTimeType*)
						  element.itsdata;
	       break;
	    case DataElementDate:
	       itsdata	= (void*)new DataElementDateType;
	       *(DataElementDateType*)itsdata	= *(DataElementDateType*)
						  element.itsdata;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to copy an undefined data element.";
	       abort();
	    default:
	       std::cerr << "Invalid enumerated data element type in copy "
	          "constructor.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const String& value)			*/
/*									*/
/* Purpose	This function will store a string value in a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to string before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the string that is to be stored in the	*/
/*		DataElement object.					*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		string type then the value that is passed to this	*/
/*		function in the variable 'value' will be stored in the	*/
/*		DataElement object. If the DataElement object has not	*/
/*		been intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and the DataElement object will have its	*/
/*		specific type of data set to its respective default	*/
/*		value.							*/
/************************************************************************/

void DataElement::Set_Data(const String& value)
   {
      switch(itstype)
         {
	    case DataElementString:
	       *(String*)itsdata	= value;
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to assign a string to a "
	          "number element.\n";
	       *(int*)itsdata		= 0;
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to assign a string to a "
	          "float element.\n";
	       *(float*)itsdata		= 0.0;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to assign a string to a "
	          "time element.\n";
	       *(DataElementTimeType*)itsdata	= defaulttime;
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to assign a string to a "
	          "date element.\n";
	       *(DataElementDateType*)itsdata	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to assign a string to an "
	          "undefined element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to set element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const int& value)				*/
/*									*/
/* Purpose	This function will store a number value in a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to number before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the number that is to be stored in the	*/
/*		DataElement object.					*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		number type then the value that is passed to this	*/
/*		function in the variable 'value' will be stored in the	*/
/*		DataElement object. If the DataElement object has not	*/
/*		been intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and the DataElement object will have its	*/
/*		specific type of data set to its respective default	*/
/*		value.							*/
/************************************************************************/

void DataElement::Set_Data(const int& value)
   {
      switch(itstype)
         {
	    case DataElementNumber:
	       *(int*)itsdata		= value;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to assign a number to a "
	          "string element.\n";
	       *(String*)itsdata	= "";
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to assign a number to a "
	          "float element.\n";
	       *(float*)itsdata		= 0.0;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to assign a number to a "
	          "time element.\n";
	       *(DataElementTimeType*)itsdata	= defaulttime;
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to assign a number to a "
	          "date element.\n";
	       *(DataElementDateType*)itsdata	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to assign a number to an "
	          "undefined element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to set element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const float& value)			*/
/*									*/
/* Purpose	This function will store a float value in a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to float before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the float that is to be stored in the	*/
/*		DataElement object.					*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		float type then the value that is passed to this	*/
/*		function in the variable 'value' will be stored in the	*/
/*		DataElement object. If the DataElement object has not	*/
/*		been intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and the DataElement object will have its	*/
/*		specific type of data set to its respective default	*/
/*		value.							*/
/************************************************************************/

void DataElement::Set_Data(const float& value)
   {
      switch(itstype)
         {
	    case DataElementFloat:
	       *(float*)itsdata		= value;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to assign a float to a "
	          "string element.\n";
	       *(String*)itsdata	= "";
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to assign a float to a "
	          "number element.\n";
	       *(int*)itsdata		= 0;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to assign a float to a "
	          "time element.\n";
	       *(DataElementTimeType*)itsdata	= defaulttime;
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to assign a float to a "
	          "date element.\n";
	       *(DataElementDateType*)itsdata	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to assign a number to an "
	          "undefined element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to set element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const DataElementTimeType& value)		*/
/*									*/
/* Purpose	This function will store a time value in a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to time before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the time that is to be stored in the	*/
/*		DataElement object.					*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		time type then the value that is passed to this		*/
/*		function in the variable 'value' will be stored in the	*/
/*		DataElement object. If the DataElement object has not	*/
/*		been intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and the DataElement object will have its	*/
/*		specific type of data set to its respective default	*/
/*		value.							*/
/************************************************************************/

void DataElement::Set_Data(const DataElementTimeType& value)
   {
      switch(itstype)
         {
	    case DataElementTime:
	       *(DataElementTimeType*)itsdata	= value;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to assign a time to a "
	          "string element.\n";
	       *(String*)itsdata	= "";
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to assign a time to a "
	          "number element.\n";
	       *(int*)itsdata		= 0;
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to assign a time to a "
	          "float element.\n";
	       *(float*)itsdata		= 0.0;
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to assign a time to a "
	          "date element.\n";
	       *(DataElementDateType*)itsdata	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to assign a time to an "
	          "undefined element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to set element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Set_Data(const DataElementDateType& value)		*/
/*									*/
/* Purpose	This function will store a date value in a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to date before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the date that is to be stored in the	*/
/*		DataElement object.					*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		date type then the value that is passed to this		*/
/*		function in the variable 'value' will be stored in the	*/
/*		DataElement object. If the DataElement object has not	*/
/*		been intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and the DataElement object will have its	*/
/*		specific type of data set to its respective default	*/
/*		value.							*/
/************************************************************************/

void DataElement::Set_Data(const DataElementDateType& value)
   {
      switch(itstype)
         {
	    case DataElementDate:
	       *(DataElementDateType*)itsdata	= value;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to assign a date to a "
	          "string element.\n";
	       *(String*)itsdata	= "";
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to assign a date to a "
	          "number element.\n";
	       *(int*)itsdata		= 0;
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to assign a date to a "
	          "float element.\n";
	       *(float*)itsdata		= 0.0;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to assign a date to a "
	          "time element.\n";
	       *(DataElementTimeType*)itsdata	= defaulttime;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to assign a date to an "
	          "undefined element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to set element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const String& value)			*/
/*									*/
/* Purpose	This function will retrieve a string value from a	*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to string before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the string that is to receive the	*/
/*		DataElement object's value.				*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		string type then the value that is stored in the	*/
/*		DataElement object will be written to the variable	*/
/*		'value'. If the DataElement object has not been		*/
/*		intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and this function will set the variable	*/
/*		'value' to its default value.				*/
/*		'value' to be an empty string.				*/
/************************************************************************/

void DataElement::Get_Data(String& value) const
   {
      switch(itstype)
         {
	    case DataElementString:
	       value		= *(String*)itsdata;
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to get a string from a "
	          "number element.\n";
	       value		= "";
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to get a string from a "
	          "float element.\n";
	       value		= "";
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to get a string from a "
	          "time element.\n";
	       value		= "";
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to get a string from a "
	          "date element.\n";
	       value		= "";
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to get a string from an "
	          "undefined element.\n";
	       value		= "";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to get element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const int& value)				*/
/*									*/
/* Purpose	This function will retrieve a number value from a	*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to number before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the number that is to receive the	*/
/*		DataElement object's value.				*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		number type then the value that is stored in the	*/
/*		DataElement object will be written to the variable	*/
/*		'value'. If the DataElement object has not been		*/
/*		intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and this function will set the variable	*/
/*		'value' to its default value.				*/
/************************************************************************/

void DataElement::Get_Data(int& value) const
   {
      switch(itstype)
         {
	    case DataElementNumber:
	       value		= *(int*)itsdata;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to get a number from a "
	          "string element.\n";
	       value		= 0;
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to get a number from a "
	          "float element.\n";
	       value		= 0;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to get a number from a "
	          "time element.\n";
	       value		= 0;
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to get a number from a "
	          "date element.\n";
	       value		= 0;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to get a number from an "
	          "undefined element.\n";
	       value		= 0;
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to get element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const float& value)			*/
/*									*/
/* Purpose	This function will retrieve a float value from a	*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to float before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the float that is to receive the		*/
/*		DataElement object's value.				*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		float type then the value that is stored in the		*/
/*		DataElement object will be written to the variable	*/
/*		'value'. If the DataElement object has not been		*/
/*		intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and this function will set the variable	*/
/*		'value' to its default value.				*/
/************************************************************************/

void DataElement::Get_Data(float& value) const
   {
      switch(itstype)
         {
	    case DataElementFloat:
	       value		= *(float*)itsdata;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to get a float from a "
	          "string element.\n";
	       value		= 0.0;
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to get a float from a "
	          "number element.\n";
	       value		= 0.0;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to get a float from a "
	          "time element.\n";
	       value		= 0.0;
	       break;
	    case DataElementDate:
	       std::cerr << "Attempted to get a float from a "
	          "date element.\n";
	       value		= 0.0;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to get a float from an "
	          "undefined element.\n";
	       value		= 0.0;
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to get element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const DataElementTimeType& value)		*/
/*									*/
/* Purpose	This function will retrieve a time value from a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to time before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the time that is to receive the		*/
/*		DataElement object's value.				*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		time type then the value that is stored in the		*/
/*		DataElement object will be written to the variable	*/
/*		'value'. If the DataElement object has not been		*/
/*		intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and this function will set the variable	*/
/*		'value' to its default value.				*/
/************************************************************************/

void DataElement::Get_Data(DataElementTimeType& value) const
   {
      switch(itstype)
         {
	    case DataElementTime:
	       value	= *(DataElementTimeType*)itsdata;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to get a time from a "
	          "string element.\n";
	       value	= defaulttime;
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to get a time from a "
	          "number element.\n";
	       value	= defaulttime;
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to get a time from a "
	          "float element.\n";
	       value	= defaulttime;
	       break;
	    case DataElementDate:
	       value	= defaulttime;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to get a time from an "
	          "undefined element.\n";
	       value	= defaulttime;
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to get element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void Get_Data(const DataElementDateType& value)		*/
/*									*/
/* Purpose	This function will retrieve a date value from a		*/
/*		DataElement object. The DataElement object must have	*/
/*		its type set to date before this function is called.	*/
/*									*/
/* Input	This function expects the variable 'value' to contain	*/
/*		a reference to the date that is to receive the		*/
/*		DataElement object's value.				*/
/*									*/
/* Output	If the DataElement object's type is set to store the	*/
/*		date type then the value that is stored in the		*/
/*		DataElement object will be written to the variable	*/
/*		'value'. If the DataElement object has not been		*/
/*		intialized, or if the DataElement object is of a	*/
/*		different type, then a warning message will be written	*/
/*		to stderr and this function will set the variable	*/
/*		'value' to its default value.				*/
/************************************************************************/

void DataElement::Get_Data(DataElementDateType& value) const
   {
      switch(itstype)
         {
	    case DataElementDate:
	       value	= *(DataElementDateType*)itsdata;
	       break;
	    case DataElementString:
	       std::cerr << "Attempted to get a date from a "
	          "string element.\n";
	       value	= defaultdate;
	       break;
	    case DataElementNumber:
	       std::cerr << "Attempted to get a date from a "
	          "number element.\n";
	       value	= defaultdate;
	       break;
	    case DataElementFloat:
	       std::cerr << "Attempted to get a date from a "
	          "float element.\n";
	       value	= defaultdate;
	       break;
	    case DataElementTime:
	       std::cerr << "Attempted to get a date from a "
	          "time element.\n";
	       value	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to get a date from an "
	          "undefined element.\n";
	       value	= defaultdate;
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "attempting to get element data.\n";
	       abort();
	 }
   }

/************************************************************************/
/* Function	void delete_element_data(void)				*/
/*									*/
/* Purpose	This private DataElement member function is responsible	*/
/*		for deleting the memory that is used to store data in	*/
/*		a DataElement object.					*/
/*									*/
/* Input	None.							*/
/*									*/
/* Output	After this function has been called, the DataElement	*/
/*		object will need to be initialized with a type before	*/
/*		it can be used again.					*/
/************************************************************************/

void DataElement::delete_element_data(void)
   {
      if (itsdata != (void*)0)
         {
	    switch(itstype)
	       {
	          case DataElementString:
		     delete (String*)itsdata;
		     break;
		  case DataElementNumber:
		     delete (int*)itsdata;
		     break;
		  case DataElementFloat:
		     delete (float*)itsdata;
		     break;
		  case DataElementTime:
		     delete (DataElementTimeType*)itsdata;
		     break;
		  case DataElementDate:
		     delete (DataElementDateType*)itsdata;
		     break;
		  case DataElementUndefined:
		     std::cerr << "An undefined element contained data.\n";
		     abort();
		  default:
		     std::cerr << "Invalid enumerated data element "
		        "type while deleting element data.\n";
		     abort();
	       }
	    itstype	= DataElementUndefined;
	    itsdata	= (void*)0;
	 }
   }

/************************************************************************/
/* Function	void allocate_element_data(const DataElementType type)	*/
/*									*/
/* Purpose	This private DataElement member function is responsible	*/
/*		for intializing a DataElement object. This function	*/
/*		will set the object's type and then this function will	*/
/*		allocate the memory that is required for the object to	*/
/*		store data of the specified type.			*/
/*									*/
/* Input	This function expects the variable, 'type' to contain	*/
/*		the type of information that the DataElement object is	*/
/*		going to store.						*/
/*									*/
/* Output	If a valid type was passed to this function in the	*/
/*		variable 'type' then this function will set the		*/
/*		DataElement object's type to the specified type and	*/
/*		this function will allocated memory to store data of	*/
/*		the specified type.					*/
/************************************************************************/

void DataElement::allocate_element_data(const DataElementType type)
   {
      delete_element_data();
      itstype		= type;
      switch(type)
         {
	    case DataElementString:
	       itsdata	= (void*)new String;
	       break;
	    case DataElementNumber:
	       itsdata	= (void*)new int;
	       break;
	    case DataElementFloat:
	       itsdata	= (void*)new float;
	       break;
	    case DataElementTime:
	       itsdata	= (void*)new DataElementTimeType;
	       *(DataElementTimeType*)itsdata	= defaulttime;
	       break;
	    case DataElementDate:
	       itsdata	= (void*)new DataElementDateType;
	       *(DataElementDateType*)itsdata	= defaultdate;
	       break;
	    case DataElementUndefined:
	       std::cerr << "Attempted to allocate memory for an "
	          "undefined element.\n";
	       break;
	    default:
	       std::cerr << "Invalid enumerated data element type while "
	          "allocating element data.\n";
	      abort();
	 }
   }
