/*
    Copyright (C) 2011 Nokia Corporation and/or its subsidiary(-ies)
    Copyright (C) 2013 Company 100, Inc.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "config.h"

#if USE(COORDINATED_GRAPHICS_MULTIPROCESS)
#include "CoordinatedLayerTreeHostProxy.h"

#include "CoordinatedLayerTreeHostMessages.h"
#include "CoordinatedLayerTreeHostProxyMessages.h"
#include "WebPageProxy.h"
#include "WebProcessProxy.h"
#include <WebCore/CoordinatedGraphicsState.h>

namespace WebKit {

using namespace WebCore;

CoordinatedLayerTreeHostProxy::CoordinatedLayerTreeHostProxy(WebPageProxy& webPageProxy)
    : m_webPageProxy(webPageProxy)
    , m_scene(adoptRef(new CoordinatedGraphicsScene(this)))
{
    m_webPageProxy.process().addMessageReceiver(Messages::CoordinatedLayerTreeHostProxy::messageReceiverName(), m_webPageProxy.pageID(), *this);
}

CoordinatedLayerTreeHostProxy::~CoordinatedLayerTreeHostProxy()
{
    m_webPageProxy.process().removeMessageReceiver(Messages::CoordinatedLayerTreeHostProxy::messageReceiverName(), m_webPageProxy.pageID());
    m_scene->detach();
}

void CoordinatedLayerTreeHostProxy::updateViewport()
{
    m_webPageProxy.setViewNeedsDisplay(IntRect(IntPoint::zero(), m_webPageProxy.viewSize()));
}

void CoordinatedLayerTreeHostProxy::dispatchUpdate(std::function<void()> function)
{
    m_scene->appendUpdate(WTFMove(function));
}

void CoordinatedLayerTreeHostProxy::commitCoordinatedGraphicsState(const CoordinatedGraphicsState& graphicsState)
{
    RefPtr<CoordinatedGraphicsScene> sceneProtector(m_scene);
    dispatchUpdate([=] {
        sceneProtector->commitSceneState(graphicsState);
    });

    updateViewport();
    m_webPageProxy.didRenderFrame(graphicsState.contentsSize, graphicsState.coveredRect);
}

void CoordinatedLayerTreeHostProxy::setVisibleContentsRect(const FloatRect& rect, const FloatPoint& trajectoryVector)
{
    if (rect == m_lastSentVisibleRect && trajectoryVector == m_lastSentTrajectoryVector)
        return;

    m_webPageProxy.process().send(Messages::CoordinatedLayerTreeHost::SetVisibleContentsRect(rect, trajectoryVector), m_webPageProxy.pageID());
    m_lastSentVisibleRect = rect;
    m_lastSentTrajectoryVector = trajectoryVector;
}

void CoordinatedLayerTreeHostProxy::renderNextFrame()
{
    m_webPageProxy.process().send(Messages::CoordinatedLayerTreeHost::RenderNextFrame(), m_webPageProxy.pageID());
}

void CoordinatedLayerTreeHostProxy::commitScrollOffset(uint32_t layerID, const IntSize& offset)
{
    m_webPageProxy.process().send(Messages::CoordinatedLayerTreeHost::CommitScrollOffset(layerID, offset), m_webPageProxy.pageID());
}

}
#endif // USE(COORDINATED_GRAPHICS_MULTIPROCESS)
