/*-
 ***********************************************************************
 *
 * $Id: webjob.c,v 1.52 2004/04/09 20:47:34 mavrik Exp $
 *
 ***********************************************************************
 *
 * Copyright 2001-2004 Klayton Monroe, All Rights Reserved.
 *
 ***********************************************************************
 */
#include "all-includes.h"

/*-
 ***********************************************************************
 *
 * Global Variables.
 *
 ***********************************************************************
 */
static WEBJOB_PROPERTIES *gpsProperties;

/*-
 ***********************************************************************
 *
 * Main
 *
 ***********************************************************************
 */
int
main(int iArgumentCount, char *ppcArgumentVector[])
{
  const char          acRoutine[] = "Main()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iError = ER_OK;
  int                 iErrorCount = 0;
  int                 iLastError = XER_OK;
  WEBJOB_PROPERTIES  *psProperties = NULL;

  /*-
   *********************************************************************
   *
   * Punch in and go to work.
   *
   *********************************************************************
   */
  iError = WebJobBootStrap(acLocalError);
  if (iError != ER_OK)
  {
    fprintf(stderr, "%s: %s\n", acRoutine, acLocalError);
    iLastError = XER_BootStrap; iErrorCount++;
    return WebJobShutdown(psProperties, iLastError, iErrorCount);
  }
  psProperties = WebJobGetPropertiesReference();

  /*-
   *********************************************************************
   *
   * Process command line arguments.
   *
   *********************************************************************
   */
  iError = WebJobProcessArguments(psProperties, iArgumentCount, ppcArgumentVector);
  if (iError != ER_OK)
  {
    fprintf(stderr, "%s: %s\n", acRoutine, acLocalError);
    iLastError = XER_ProcessArguments; iErrorCount++;
    return WebJobShutdown(psProperties, iLastError, iErrorCount);
  }

  /*-
   *********************************************************************
   *
   * Read the properties file.
   *
   *********************************************************************
   */
  iError = PropertiesReadFile(psProperties->pcConfigFile, psProperties, acLocalError);
  if (iError != ER_OK)
  {
    fprintf(stderr, "%s: %s\n", acRoutine, acLocalError);
    iLastError = XER_ReadProperties; iErrorCount++;
    return WebJobShutdown(psProperties, iLastError, iErrorCount);
  }

  /*-
   *********************************************************************
   *
   * Check dependencies.
   *
   *********************************************************************
   */
  iError = WebJobCheckDependencies(psProperties, acLocalError);
  if (iError != ER_OK)
  {
    fprintf(stderr, "%s: %s\n", acRoutine, acLocalError);
    iLastError = XER_CheckDependencies; iErrorCount++;
    return WebJobShutdown(psProperties, iLastError, iErrorCount);
  }

  /*-
   *********************************************************************
   *
   * Prepare for GET, RUN, and PUT stages.
   *
   *********************************************************************
   */
  iError = WebJobDoConfigure(psProperties, acLocalError);
  if (iError != ER_OK)
  {
    fprintf(stderr, "%s: %s\n", acRoutine, acLocalError);
    iLastError = XER_Configure; iErrorCount++;
    return WebJobShutdown(psProperties, iLastError, iErrorCount);
  }

  /*-
   *********************************************************************
   *
   * Do GET stage (i.e. download the specified program).
   *
   *********************************************************************
   */
  iError = WebJobDoGetStage(psProperties, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(psProperties->acGetError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    fprintf(stderr, "%s\n", psProperties->acGetError);
    iLastError = XER_GetStage; iErrorCount++;
    if (psProperties->psPutURL == NULL)
    {
      return WebJobShutdown(psProperties, iLastError, iErrorCount);
    }
    psProperties->iRunStageDisabled = 1;
  }

  /*-
   *********************************************************************
   *
   * Do RUN stage (i.e. execute the specified program).
   *
   *********************************************************************
   */
  iError = WebJobDoRunStage(psProperties, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(psProperties->acRunError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
#ifdef WIN32
    fprintf(stderr, "%s\n", psProperties->acRunError);
#else
    if (psProperties->sWEBJOB.iKidPid == 0)
    {
      fprintf(psProperties->sWEBJOB.pStdErr, "%s\n", psProperties->acRunError);
      return XER_Abort;
    }
    else
    {
      fprintf(stderr, "%s\n", psProperties->acRunError);
    }
#endif
    iLastError = XER_RunStage; iErrorCount++;
    if (psProperties->psPutURL == NULL)
    {
      return WebJobShutdown(psProperties, iLastError, iErrorCount);
    }
  }

  /*-
   *********************************************************************
   *
   * Do PUT stage (i.e. conditionally upload output).
   *
   *********************************************************************
   */
  iError = WebJobDoPutStage(psProperties, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(psProperties->acPutError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    fprintf(stderr, "%s\n", psProperties->acPutError);
    iLastError = XER_PutStage; iErrorCount++;
    if (psProperties->psPutURL == NULL)
    {
      return WebJobShutdown(psProperties, iLastError, iErrorCount);
    }
  }

  /*-
   *********************************************************************
   *
   * Shutdown and go home.
   *
   *********************************************************************
   */
  return WebJobShutdown(psProperties, iLastError, iErrorCount);
}


/*-
 ***********************************************************************
 *
 * BuildCommandLine
 *
 ***********************************************************************
 */
char *
BuildCommandLine(int iArgumentCount, char *ppcArgumentVector[])
{
  char               *pcCommandLine;
  int                 i;
  int                 iIndex;
  int                 iLength;

  for (i = iLength = 0; i < iArgumentCount; i++)
  {
    iLength += ((i > 0) ? 1 : 0) + strlen(ppcArgumentVector[i]);
  }
  pcCommandLine = malloc(iLength + 1);
  if (pcCommandLine == NULL)
  {
    return NULL;
  }
  for (i = iIndex = 0; i < iArgumentCount; i++)
  {
    if (iIndex > iLength)
    {
      free(pcCommandLine);
      return NULL;
    }
    iIndex += snprintf(&pcCommandLine[iIndex], iLength + 1 - iIndex, "%s%s", (i > 0) ? " " : "", ppcArgumentVector[i]);
  }
  return pcCommandLine;
}


#ifdef WIN32
/*-
 ***********************************************************************
 *
 * FormatWin32Error
 *
 ***********************************************************************
 */
void
FormatWin32Error(DWORD dwError, char **ppcMessage)
{
  static char         acMessage[MESSAGE_SIZE];
  char               *pc;
  int                 i;
  int                 j;
  LPVOID              lpvMessage;

  acMessage[0] = 0;

  *ppcMessage = acMessage;

  FormatMessage(
    FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
    NULL,
    dwError,
    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* default language */
    (LPTSTR) & lpvMessage,
    0,
    NULL
    );

  /*-
   *********************************************************************
   *
   * Replace linefeeds with spaces. Eliminate carriage returns.
   *
   *********************************************************************
   */
  for (i = 0, j = 0, pc = (char *) lpvMessage; (i < (int) strlen(lpvMessage)) && (i < (MESSAGE_SIZE)); i++, j++)
  {
    if (pc[i] == '\n')
    {
      acMessage[j] = ' ';
    }
    else if (pc[i] == '\r')
    {
      j--;
      continue;
    }
    else
    {
      acMessage[j] = pc[i];
    }
  }
  acMessage[j] = 0;

  LocalFree(lpvMessage);
}
#endif


/*-
 ***********************************************************************
 *
 * GetHostname
 *
 ***********************************************************************
 */
char *
GetHostname(void)
{
#define MAX_HOSTNAME_LENGTH 256
  static char         acHostname[MAX_HOSTNAME_LENGTH] = "NA";
#ifdef UNIX
  struct utsname      utsName;

  memset(&utsName, 0, sizeof(struct utsname));
  if (uname(&utsName) != -1)
  {
    snprintf(acHostname, MAX_HOSTNAME_LENGTH, "%s", (utsName.nodename[0]) ? utsName.nodename : "NA");
  }
#endif
#ifdef WIN32
  char                acTempname[MAX_HOSTNAME_LENGTH];
  DWORD               dwTempNameLength = sizeof(acTempname);

  if (GetComputerName(acTempname, &dwTempNameLength) == TRUE)
  {
    snprintf(acHostname, MAX_HOSTNAME_LENGTH, "%s", acTempname);
  }
#endif
  return acHostname;
}


/*-
 ***********************************************************************
 *
 * GetMyVersion
 *
 ***********************************************************************
 */
char *
GetMyVersion(void)
{
#define MAX_VERSION_LENGTH 256
  static char         acMyVersion[MAX_VERSION_LENGTH] = "NA";

#ifdef USE_SSL
  snprintf(acMyVersion, MAX_VERSION_LENGTH, "%s %s ssl %d bit",
#else
  snprintf(acMyVersion, MAX_VERSION_LENGTH, "%s %s %d bit",
#endif
    PROGRAM_NAME,
    VERSION,
    (int) (sizeof(&GetMyVersion) * 8)
    );
  return acMyVersion;
}


/*-
 ***********************************************************************
 *
 * GetSystemOS
 *
 ***********************************************************************
 */
char *
GetSystemOS(void)
{
#define MAX_SYSTEMOS_LENGTH 256
  static char         acSystemOS[MAX_SYSTEMOS_LENGTH] = "NA";
#ifdef UNIX
  struct utsname      utsName;

  memset(&utsName, 0, sizeof(struct utsname));
  if (uname(&utsName) != -1)
  {
#ifdef WEBJOB_AIX
    snprintf(acSystemOS, MAX_SYSTEMOS_LENGTH, "%s %s %s.%s",
      utsName.machine,
      utsName.sysname,
      utsName.version,
      utsName.release
      );
#else
    snprintf(acSystemOS, MAX_SYSTEMOS_LENGTH, "%s %s %s",
      utsName.machine,
      utsName.sysname,
      utsName.release
      );
#endif
  }
#endif
#ifdef WIN32
  char                acOS[16];
  char                acPlatform[16];
  OSVERSIONINFO       osvi;
  SYSTEM_INFO         si;

  memset(&si, 0, sizeof(SYSTEM_INFO));
  GetSystemInfo(&si);
  switch (si.wProcessorArchitecture)
  {
  case PROCESSOR_ARCHITECTURE_INTEL:
    strncpy(acPlatform, "INTEL", sizeof(acPlatform));
    break;
  case PROCESSOR_ARCHITECTURE_MIPS:
    strncpy(acPlatform, "MIPS", sizeof(acPlatform));
    break;
  case PROCESSOR_ARCHITECTURE_ALPHA:
    strncpy(acPlatform, "ALPHA", sizeof(acPlatform));
    break;
  case PROCESSOR_ARCHITECTURE_PPC:
    strncpy(acPlatform, "PPC", sizeof(acPlatform));
    break;
  default:
    strncpy(acPlatform, "NA", sizeof(acPlatform));
    break;
  }

  memset(&osvi, 0, sizeof(OSVERSIONINFO));
  osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  if (GetVersionEx(&osvi) == TRUE)
  {
    switch (osvi.dwPlatformId)
    {
    case VER_PLATFORM_WIN32s:
      strncpy(acOS, "Windows 3.1", sizeof(acOS));
      break;
    case VER_PLATFORM_WIN32_WINDOWS:
      strncpy(acOS, "Windows 98", sizeof(acOS));
      break;
    case VER_PLATFORM_WIN32_NT:
      strncpy(acOS, "Windows NT", sizeof(acOS));
      break;
    default:
      strncpy(acOS, "NA", sizeof(acOS));
      break;
    }

    snprintf(acSystemOS, MAX_SYSTEMOS_LENGTH, "%s %s %u.%u Build %u %s",
      acPlatform,
      acOS,
      osvi.dwMajorVersion,
      osvi.dwMinorVersion,
      osvi.dwBuildNumber,
      osvi.szCSDVersion
      );
  }
#endif
  return acSystemOS;
}


/*-
 ***********************************************************************
 *
 * HTTPFinalizeCredentials
 *
 ***********************************************************************
 */
int
HTTPFinalizeCredentials(HTTP_URL *psURL, char *pcUser, char *pcPass, char *pcError)
{
  const char          acRoutine[] = "HTTPFinalizeCredentials()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iError;

  psURL->iAuthType = HTTP_AUTH_TYPE_BASIC;

  if (psURL->pcUser[0] == 0 || strcasecmp(psURL->pcUser, WEBJOB_USER_TOKEN) == 0)
  {
    if (pcUser[0])
    {
      iError = HTTPSetURLUser(psURL, pcUser, acLocalError);
      if (iError != ER_OK)
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
        return ER;
      }
    }
    else
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing username for basic authentication", acRoutine);
      return ER;
    }
  }

  if (psURL->pcPass[0] == 0 || strcasecmp(psURL->pcPass, WEBJOB_PASS_TOKEN) == 0)
  {
    if (pcPass[0])
    {
      iError = HTTPSetURLPass(psURL, pcPass, acLocalError);
      if (iError != ER_OK)
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
        return ER;
      }
    }
    else
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing password for basic authentication", acRoutine);
      return ER;
    }
  }
  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * MakeRandomExtension
 *
 ***********************************************************************
 */
char *
MakeRandomExtension(void)
{
#define MAX_EXTENSION_LENGTH 9
  static char         acExtension[MAX_EXTENSION_LENGTH];
  unsigned char       a;
  unsigned char       b;
  unsigned char       c;
  unsigned char       d;
  long                lValue;

#ifdef WIN32
  lValue = (GetTickCount() << 16) | rand();
#else
  lValue = random();
#endif
  a = (unsigned char) ((lValue >> 24) & 0xff);
  b = (unsigned char) ((lValue >> 16) & 0xff);
  c = (unsigned char) ((lValue >>  8) & 0xff);
  d = (unsigned char) ((lValue >>  0) & 0xff);
  snprintf(acExtension, MAX_EXTENSION_LENGTH, "%02x%02x%02x%02x", a, b, c, d);
  return acExtension;
}


#ifdef USE_SSL
/*-
 ***********************************************************************
 *
 * SSLCheckDependencies
 *
 ***********************************************************************
 */
int
SSLCheckDependencies(SSL_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "SSLCheckDependencies()";

  if (psProperties->iUseCertificate)
  {
    if (psProperties->pcPublicCertFile == NULL || psProperties->pcPublicCertFile[0] == 0)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing SSLPublicCertFile.", acRoutine);
      return ER;
    }

    if (psProperties->pcPrivateKeyFile == NULL || psProperties->pcPrivateKeyFile[0] == 0)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing SSLPrivateKeyFile.", acRoutine);
      return ER;
    }
  }

  if (psProperties->iVerifyPeerCert)
  {
    if (psProperties->pcBundledCAsFile == NULL || psProperties->pcBundledCAsFile[0] == 0)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing SSLBundledCAsFile.", acRoutine);
      return ER;
    }

    if (psProperties->pcExpectedPeerCN == NULL || psProperties->pcExpectedPeerCN[0] == 0)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing SSLExpectedPeerCN.", acRoutine);
      return ER;
    }
  }

  return ER_OK;
}
#endif


/*-
 ***********************************************************************
 *
 * SeedRandom
 *
 ***********************************************************************
 */
int
SeedRandom(unsigned long ulTime1, unsigned long ulTime2, char *pcError)
{
  const char          acRoutine[] = "SeedRandom()";

  if (
       ulTime1 == (unsigned long) ~0 ||
       ulTime2 == (unsigned long) ~0 ||
       ulTime1 == (unsigned long)  0 ||
       ulTime2 == (unsigned long)  0
     )
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Seed values are blatantly abnormal!: Time1 = [%08lx], Time2 = [%08lx]", acRoutine, ulTime1, ulTime2);
    return ER;
  }

#ifdef WIN32
  srand((unsigned long) (0xE97482AD ^ ((ulTime1 << 16) | (ulTime1 >> 16)) ^ ulTime2 ^ GetTickCount()));
#else
  srandom((unsigned long) (0xE97482AD ^ ((ulTime1 << 16) | (ulTime1 >> 16)) ^ ulTime2 ^ getpid()));
#endif
  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobBootStrap
 *
 ***********************************************************************
 */
int
WebJobBootStrap(char *pcError)
{
  const char          acRoutine[] = "WebJobBootStrap()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  WEBJOB_PROPERTIES  *psProperties;

#ifdef WIN32
  /*-
   *********************************************************************
   *
   * Suppress critical-error-handler message boxes.
   *
   *********************************************************************
   */
  SetErrorMode(SEM_FAILCRITICALERRORS);
#endif

#ifdef USE_SSL
  /*-
   *********************************************************************
   *
   * Initialize SSL library, error strings, and random seed.
   *
   *********************************************************************
   */
  SSLBoot();
#endif

  /*-
   *********************************************************************
   *
   * Allocate and initialize a new properties structure.
   *
   *********************************************************************
   */
  psProperties = WebJobNewProperties(acLocalError);
  if (psProperties == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }
  WebJobSetPropertiesReference(psProperties);

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobCheckDependencies
 *
 ***********************************************************************
 */
int
WebJobCheckDependencies(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobCheckDependencies()";
#ifdef USE_SSL
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iError;
#endif

  /*-
   *********************************************************************
   *
   * The specified request must not include any path separators.
   *
   *********************************************************************
   */
  if (
       (strstr(psProperties->sWEBJOB.pcRequest,  "/") != NULL) ||
       (strstr(psProperties->sWEBJOB.pcRequest, "\\") != NULL)
     )
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Program = [%s], Argument must not include any path separators.", acRoutine, psProperties->sWEBJOB.pcRequest);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * A GetURL is required.
   *
   *********************************************************************
   */
  if (psProperties->psGetURL == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Missing URLGetURL key/value pair.", acRoutine);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * A ClientId is required.
   *
   *********************************************************************
   */
  if (psProperties->acClientId[0] == 0)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Missing ClientId key/value pair.", acRoutine);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Timeout range = [WEBJOB_MIN_TIME_LIMIT,WEBJOB_MAX_TIME_LIMIT]
   *
   *********************************************************************
   */
  if (psProperties->iGetTimeLimit < WEBJOB_MIN_TIME_LIMIT || psProperties->iGetTimeLimit > WEBJOB_MAX_TIME_LIMIT)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: GetTimeLimit = [%d], Value must be in the range [%d,%d].",
      acRoutine,
      psProperties->iGetTimeLimit,
      WEBJOB_MIN_TIME_LIMIT,
      WEBJOB_MAX_TIME_LIMIT
      );
    return ER;
  }
  if (psProperties->iRunTimeLimit < WEBJOB_MIN_TIME_LIMIT || psProperties->iRunTimeLimit > WEBJOB_MAX_TIME_LIMIT)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: RunTimeLimit = [%d], Value must be in the range [%d,%d].",
      acRoutine,
      psProperties->iRunTimeLimit,
      WEBJOB_MIN_TIME_LIMIT,
      WEBJOB_MAX_TIME_LIMIT
      );
    return ER;
  }
  if (psProperties->iPutTimeLimit < WEBJOB_MIN_TIME_LIMIT || psProperties->iPutTimeLimit > WEBJOB_MAX_TIME_LIMIT)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: PutTimeLimit = [%d], Value must be in the range [%d,%d].",
      acRoutine,
      psProperties->iPutTimeLimit,
      WEBJOB_MIN_TIME_LIMIT,
      WEBJOB_MAX_TIME_LIMIT
      );
    return ER;
  }

#ifdef USE_SSL
  /*-
   *********************************************************************
   *
   * Check SSL dependencies.
   *
   *********************************************************************
   */
  iError = SSLCheckDependencies(psProperties->psSSLProperties, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }
#endif

  /*-
   *********************************************************************
   *
   * GetHookCommandLine and GetHookStatus are required when GetHook is
   * enabled.
   *
   *********************************************************************
   */
  if (psProperties->psGetHook->iActive)
  {
    if (psProperties->psGetHook->pcOriginalCommandLine == NULL || psProperties->psGetHook->pcOriginalCommandLine[0] == 0)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing GetHookCommandLine key/value pair.", acRoutine);
      return ER;
    }

    if (psProperties->psGetHook->iTargetStatus == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Missing GetHookStatus key/value pair.", acRoutine);
      return ER;
    }
  }

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobDoConfigure
 *
 ***********************************************************************
 */
int
WebJobDoConfigure(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoConfigure()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  char               *pcTemp;
  int                 iError;
  int                 iLength;
#ifdef UNIX
  struct sigaction    signalAlarm;
#endif

  /*-
   *********************************************************************
   *
   * Change to the TempDirectory.
   *
   *********************************************************************
   */
  iError = chdir((const char *) psProperties->acTempDirectory);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: chdir(): Directory = [%s], %s", acRoutine, psProperties->acTempDirectory, strerror(errno));
    return ER;
  }

#ifdef UNIX
  /*-
   *********************************************************************
   *
   * Establish a timeout signal handler, if required.
   *
   *********************************************************************
   */
  if (psProperties->iGetTimeLimit || psProperties->iRunTimeLimit || psProperties->iPutTimeLimit)
  {
    signalAlarm.sa_handler = WebJobTimeoutHandler;
    sigemptyset(&signalAlarm.sa_mask);
    signalAlarm.sa_flags = 0;
    iError = sigaction(SIGALRM, &signalAlarm, NULL);
    if (iError == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: sigaction(): WebJobTimeoutHandler, %s", acRoutine, strerror(errno));
      return ER;
    }
  }
#endif

#ifdef USE_SSL
  /*-
   *********************************************************************
   *
   * Initialize SSL Context.
   *
   *********************************************************************
   */
  if (
       (psProperties->psGetURL != NULL && psProperties->psGetURL->iScheme == HTTP_SCHEME_HTTPS) ||
       (psProperties->psPutURL != NULL && psProperties->psPutURL->iScheme == HTTP_SCHEME_HTTPS)
     )
  {
    if (SSLInitializeCTX(psProperties->psSSLProperties, acLocalError) == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }
    if (psProperties->psGetURL != NULL && psProperties->psGetURL->iScheme == HTTP_SCHEME_HTTPS)
    {
      psProperties->psGetURL->psSSLProperties = psProperties->psSSLProperties;
    }
    if (psProperties->psPutURL != NULL && psProperties->psPutURL->iScheme == HTTP_SCHEME_HTTPS)
    {
      psProperties->psPutURL->psSSLProperties = psProperties->psSSLProperties;
    }
  }
#endif

  /*-
   *********************************************************************
   *
   * Seed the random number generator.
   *
   *********************************************************************
   */
  WebJobGetEpoch(&psProperties->tvSRGEpoch);
  iError = SeedRandom((unsigned long) psProperties->tvJobEpoch.tv_usec, (unsigned long) psProperties->tvSRGEpoch.tv_usec, acLocalError);
  if (iError == -1)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

#ifdef UNIX
  /*-
   *********************************************************************
   *
   * Prepare pipe handles.
   *
   *********************************************************************
   */
  iError = pipe(psProperties->sWEBJOB.iPipe);
  if (iError == -1)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: pipe(): %s", acRoutine, strerror(errno));
    return ER;
  }
#endif

  /*-
   *********************************************************************
   *
   * Prepare output file handles.
   *
   *********************************************************************
   */
  iError = WebJobPrepareHandles(psProperties, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Finalize GetHook configuration, if necessary.
   *
   *********************************************************************
   */
  if (psProperties->psGetHook->iActive)
  {
    iLength = strlen(psProperties->sWEBJOB.pcRequest) + strlen(psProperties->psGetHook->pcSuffix);
    pcTemp = malloc(iLength + 1);
    if (pcTemp == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: malloc(): %s", acRoutine, strerror(errno));
      return ER;
    }
    snprintf(pcTemp, iLength + 1, "%s%s", psProperties->sWEBJOB.pcRequest, psProperties->psGetHook->pcSuffix);

    iError = HookSetValue(psProperties->psGetHook, HOOK_VALUE_MODIFIED_FILENAME, pcTemp, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      free(pcTemp);
      return ER;
    }
    free(pcTemp);

    psProperties->sWEBJOB.pcRequest = psProperties->psGetHook->pcModifiedFilename;

    pcTemp = HookExpandCommandLine(psProperties->psGetHook, acLocalError);
    if (pcTemp == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }

    iError = HookSetValue(psProperties->psGetHook, HOOK_VALUE_EXPANDED_COMMAND_LINE, pcTemp, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      free(pcTemp);
      return ER;
    }
    free(pcTemp);
  }

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobDoGetHook
 *
 ***********************************************************************
 */
int
WebJobDoGetHook(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoGetHook()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iError;
#ifdef UNIX
  struct stat         statEntry;
#endif

  iError = HookRunSystemCommand(psProperties->psGetHook, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

#ifdef UNIX
  if (stat(psProperties->sWEBJOB.pcCommand, &statEntry) == -1)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: stat(): %s", acRoutine, strerror(errno));
    return ER;
  }
  if ((statEntry.st_mode & S_IXUSR) != S_IXUSR)
  {
    statEntry.st_mode |= S_IXUSR;
    if (chmod(psProperties->sWEBJOB.pcCommand, statEntry.st_mode) == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: chmod(): %s", acRoutine, strerror(errno));
      return ER;
    }
  }
#endif

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobDoGetRequest
 *
 ***********************************************************************
 */
int
WebJobDoGetRequest(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoGetRequest()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  char                acQuery[WEBJOB_MAX_QUERY_LENGTH];
  char               *apcEscaped[5];
  int                 i;
  int                 iEscaped;
  int                 iError;
  int                 iFileDescriptor;
  int                 iOpenFlags;
  HTTP_URL           *psURL;
  HTTP_RESPONSE_HDR   sResponseHeader;

  /*-
   *********************************************************************
   *
   * If the URL is not defined, abort.
   *
   *********************************************************************
   */
  psURL = psProperties->psGetURL;
  if (psURL == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Undefined URL.", acRoutine);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Finalize the user's HTTP credentials, if necessary.
   *
   *********************************************************************
   */
  if (psProperties->iURLAuthType == HTTP_AUTH_TYPE_BASIC)
  {
    iError = HTTPFinalizeCredentials(psURL, psProperties->acURLUsername, psProperties->acURLPassword, acLocalError);
    if (iError == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }
  }

  /*-
   *********************************************************************
   *
   * Set the Path/Query string.
   *
   *********************************************************************
   */
  iEscaped = 0;
  apcEscaped[iEscaped] = HTTPEscape(GetMyVersion(), acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(GetSystemOS(), acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(psProperties->acClientId, acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(psProperties->sWEBJOB.pcRequest, acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }

  snprintf(acQuery, WEBJOB_MAX_QUERY_LENGTH, "VERSION=%s&SYSTEM=%s&CLIENTID=%s&FILENAME=%s",
    apcEscaped[0],
    apcEscaped[1],
    apcEscaped[2],
    apcEscaped[3]
    );

  for (i = 0; i < iEscaped; i++)
  {
    HTTPFreeData(apcEscaped[i]);
  }

  iError = HTTPSetURLQuery(psURL, acQuery, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Set the request method.
   *
   *********************************************************************
   */
  iError = HTTPSetURLMeth(psURL, "GET", acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Set the download limit.
   *
   *********************************************************************
   */
  HTTPSetURLDownloadLimit(psURL, (K_UINT32) psProperties->iURLDownloadLimit);

  /*-
   *********************************************************************
   *
   * Check the overwrite flag, and do the appropriate open.
   *
   *********************************************************************
   */
  iOpenFlags = O_CREAT | O_TRUNC | O_EXCL | O_WRONLY;
#ifdef WIN32
  iOpenFlags |= O_BINARY;
#endif
  if (psProperties->iOverwriteExecutable)
  {
    iOpenFlags ^= O_EXCL;
  }

  iFileDescriptor = open(psProperties->sWEBJOB.pcRequest, iOpenFlags, 0700);
  if (iFileDescriptor == ER)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: open(): Directory = [%s], File = [%s], %s", acRoutine, psProperties->acTempDirectory, psProperties->sWEBJOB.pcRequest, strerror(errno));
    if (errno == EEXIST && !psProperties->iOverwriteExecutable)
    {
      psProperties->iUnlinkExecutable = 0;
    }
    return ER;
  }

  psProperties->sWEBJOB.pRequestFile = fdopen(iFileDescriptor, "wb");
  if (psProperties->sWEBJOB.pRequestFile == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: fdopen(): Directory = [%s], File = [%s], %s", acRoutine, psProperties->acTempDirectory, psProperties->sWEBJOB.pcRequest, strerror(errno));
    WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pRequestFile);
    unlink(psProperties->sWEBJOB.pcRequest);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Submit the request, check the response code, and close the file.
   *
   *********************************************************************
   */
  iError = HTTPSubmitRequest(psURL, HTTP_IGNORE_INPUT, NULL, HTTP_STREAM_OUTPUT, psProperties->sWEBJOB.pRequestFile, &sResponseHeader, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pRequestFile);
    unlink(psProperties->sWEBJOB.pcRequest);
    return ER;
  }

  if (sResponseHeader.iStatusCode < 200 || sResponseHeader.iStatusCode > 299)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Status = [%d], Reason = [%s]", acRoutine, sResponseHeader.iStatusCode, sResponseHeader.acReasonPhrase);
    WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pRequestFile);
    unlink(psProperties->sWEBJOB.pcRequest);
    return ER;
  }

  WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pRequestFile);

  /*-
   *********************************************************************
   *
   * Record the job id.
   *
   *********************************************************************
   */
  strncpy(psProperties->acJobId, sResponseHeader.acJobId, HTTP_JOB_ID_SIZE);

  /*-
   *********************************************************************
   *
   * Execute the GetHook as required.
   *
   *********************************************************************
   */
  if (psProperties->psGetHook->iActive)
  {
    iError = WebJobDoGetHook(psProperties, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      unlink(psProperties->sWEBJOB.pcRequest);
      return ER;
    }
  }

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobDoGetStage
 *
 ***********************************************************************
 */
int
WebJobDoGetStage(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoGetStage()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iTimeout;
#ifdef UNIX
  int                 iError;
#endif
#ifdef WIN32
  char               *pcLocalError;
  DWORD               dwStatus;
  DWORD               dwThreadID;
  HANDLE              hThread;
  WEBJOB_THREAD       sThreadArguments;
#endif

  /*-
   *********************************************************************
   *
   * Update the RunStage.
   *
   *********************************************************************
   */
  psProperties->iRunStage = WEBJOB_GET_STAGE;

  /*-
   *********************************************************************
   *
   * Record the time.
   *
   *********************************************************************
   */
  WebJobGetEpoch(&psProperties->tvGetEpoch);

  /*-
   *********************************************************************
   *
   * Set the timeout value -- zero means no timeout.
   *
   *********************************************************************
   */
#ifdef WIN32
  iTimeout = (psProperties->iGetTimeLimit == 0) ? INFINITE : psProperties->iGetTimeLimit * 1000;
#else
  iTimeout = psProperties->iGetTimeLimit;
#endif

#ifdef UNIX
  /*-
   *********************************************************************
   *
   * Execute the GET request. Abort on timeout.
   *
   *********************************************************************
   */
  alarm(iTimeout);
  if (setjmp(psProperties->sGetEnvironment) == 0)
  {
    iError = WebJobDoGetRequest(psProperties, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }
  }
  else
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: setjmp(): GET stage timed out after %d seconds.", acRoutine, psProperties->iGetTimeLimit);
    return ER;
  }
  alarm(0);
#endif

#ifdef WIN32
  /*-
   *********************************************************************
   *
   * Create a worker thread and execute the GET request.
   *
   *********************************************************************
   */
  sThreadArguments.psProperties = psProperties;
  sThreadArguments.pcError = acLocalError;
  sThreadArguments.piRoutine = WebJobDoGetRequest;
  hThread = (HANDLE) _beginthreadex(
    NULL,
    0,
    (void *) WebJobThreadWrapper,
    &sThreadArguments,
    0,
    (void *) &dwThreadID
    );
  if (hThread == INVALID_HANDLE_VALUE)
  {
    FormatWin32Error(GetLastError(), &pcLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: _beginthreadex(): %s", acRoutine, pcLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Wait for the thread to complete. Abort on timeout.
   *
   *********************************************************************
   */
  dwStatus = WaitForSingleObject(hThread, iTimeout);
  if (dwStatus == WAIT_TIMEOUT)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: WaitForSingleObject(): GET stage timed out after %d seconds.", acRoutine, psProperties->iGetTimeLimit);
    TerminateThread(hThread, ER);
    CloseHandle(hThread);
    return ER;
  }
  else
  {
    if (!GetExitCodeThread(hThread, &dwStatus))
    {
      FormatWin32Error(GetLastError(), &pcLocalError);
      snprintf(pcError, MESSAGE_SIZE, "%s: GetExitCodeThread(): %s", acRoutine, pcLocalError);
      CloseHandle(hThread);
      return ER;
    }
    if (dwStatus != (DWORD) ER_OK) /* The GET failed. */
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      CloseHandle(hThread);
      return ER;
    }
  }
  CloseHandle(hThread);
#endif

  return ER_OK;
}


#ifdef UNIX
/*-
 ***********************************************************************
 *
 * WebJobDoKidStage
 *
 ***********************************************************************
 */
int
WebJobDoKidStage(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoKidStage()";
  int                 iError;

  /*-
   *********************************************************************
   *
   * Set stdin to be the read-side of the provided pipe.
   *
   *********************************************************************
   */
  WEBJOB_SAFE_FCLOSE(stdin);
  iError = dup(psProperties->sWEBJOB.iPipe[WEBJOB_READER_HANDLE]);
  if (iError == -1)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: dup(): stdin, %s", acRoutine, strerror(errno));
    return ER;
  }
  close(psProperties->sWEBJOB.iPipe[WEBJOB_READER_HANDLE]);
  close(psProperties->sWEBJOB.iPipe[WEBJOB_WRITER_HANDLE]);

  /*-
   *********************************************************************
   *
   * If PutURL was specified, use the provided handles as std{out|err}.
   *
   *********************************************************************
   */
  if (psProperties->psPutURL != NULL)
  {
    WEBJOB_SAFE_FCLOSE(stdout);
    iError = dup(fileno(psProperties->sWEBJOB.pStdOut));
    if (iError == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: dup(): stdout, %s", acRoutine, strerror(errno));
      return ER;
    }
    WEBJOB_SAFE_FCLOSE(stderr);
    iError = dup(fileno(psProperties->sWEBJOB.pStdErr));
    if (iError == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: dup(): stderr, %s", acRoutine, strerror(errno));
      return ER;
    }
  }

  /*-
   *********************************************************************
   *
   * Execute the specified command line.
   *
   *********************************************************************
   */
  iError = execv(psProperties->sWEBJOB.pcCommand, psProperties->sWEBJOB.ppcArguments);
  if (iError == -1)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: execv(): %s, %s", acRoutine, psProperties->sWEBJOB.pcCommand, strerror(errno));
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Execution beyond execv() should not happen.
   *
   *********************************************************************
   */
  snprintf(pcError, MESSAGE_SIZE, "%s: Execution has gone beyond execv(). This should not happen.", acRoutine);
  return ER;
}
#endif


/*-
 ***********************************************************************
 *
 * WebJobDoPutRequest
 *
 ***********************************************************************
 */
int
WebJobDoPutRequest(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoPutRequest()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  char                acQuery[WEBJOB_MAX_QUERY_LENGTH];
  char                aacStreams[3][7];
  char               *apcEscaped[6];
  int                 i;
  int                 iEscaped;
  int                 iError;
  int                 iLimit;
  HTTP_URL           *psURL;
  HTTP_STREAM_LIST    asStreamList[3];
  HTTP_RESPONSE_HDR   sResponseHeader;

  /*-
   *********************************************************************
   *
   * If the URL is not defined, abort.
   *
   *********************************************************************
   */
  psURL = psProperties->psPutURL;
  if (psURL == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Undefined URL.", acRoutine);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Finalize the user's HTTP credentials, if necessary.
   *
   *********************************************************************
   */
  if (psProperties->iURLAuthType == HTTP_AUTH_TYPE_BASIC)
  {
    iError = HTTPFinalizeCredentials(psURL, psProperties->acURLUsername, psProperties->acURLPassword, acLocalError);
    if (iError == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }
  }

  /*-
   *********************************************************************
   *
   * Build the HTTP_STREAM_LIST list.
   *
   *********************************************************************
   */
  strncpy(aacStreams[0], "stdout", sizeof(aacStreams[0]));
  strncpy(aacStreams[1], "stderr", sizeof(aacStreams[1]));
  strncpy(aacStreams[2], "stdenv", sizeof(aacStreams[2]));
  asStreamList[0].pFile = psProperties->sWEBJOB.pStdOut;
  asStreamList[1].pFile = psProperties->sWEBJOB.pStdErr;
  asStreamList[2].pFile = psProperties->sWEBJOB.pStdEnv;
  for (i = 0, iLimit = 3; i < iLimit; i++)
  {
    if (fseek(asStreamList[i].pFile, 0, SEEK_END) == ER)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: fseek(): Stream = [%s]: %s", acRoutine, aacStreams[i], strerror(errno));
      return ER;
    }
    asStreamList[i].ui32Size = (K_UINT32) ftell(asStreamList[i].pFile);
    if (asStreamList[i].ui32Size == (K_UINT32) ER)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: fseek(): Stream = [%s]: %s", acRoutine, aacStreams[i], strerror(errno));
      return ER;
    }
    rewind(asStreamList[i].pFile);
    asStreamList[i].psNext = (i < iLimit - 1) ? &asStreamList[i + 1] : NULL;
  }

  /*-
   *********************************************************************
   *
   * Set the query string.
   *
   *********************************************************************
   */
  iEscaped = 0;
  apcEscaped[iEscaped] = HTTPEscape(GetMyVersion(), acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(GetSystemOS(), acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(psProperties->acClientId, acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(psProperties->sWEBJOB.pcRequest, acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }
  apcEscaped[iEscaped] = HTTPEscape(psProperties->acRunType, acLocalError);
  if (apcEscaped[iEscaped++] == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    for (i = 0; i < iEscaped; i++)
    {
      HTTPFreeData(apcEscaped[i]);
    }
    return ER;
  }

  snprintf(acQuery, WEBJOB_MAX_QUERY_LENGTH, "VERSION=%s&SYSTEM=%s&CLIENTID=%s&FILENAME=%s&RUNTYPE=%s&STDOUT_LENGTH=%u&STDERR_LENGTH=%u&STDENV_LENGTH=%u",
    apcEscaped[0],
    apcEscaped[1],
    apcEscaped[2],
    apcEscaped[3],
    apcEscaped[4],
    asStreamList[0].ui32Size,
    asStreamList[1].ui32Size,
    asStreamList[2].ui32Size
    );

  for (i = 0; i < iEscaped; i++)
  {
    HTTPFreeData(apcEscaped[i]);
  }

  iError = HTTPSetURLQuery(psURL, acQuery, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Set the request method.
   *
   *********************************************************************
   */
  iError = HTTPSetURLMeth(psURL, "PUT", acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Set the job id.
   *
   *********************************************************************
   */
  if (psProperties->acJobId[0])
  {
    iError = HTTPSetURLJobId(psURL, psProperties->acJobId, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }
  }

  /*-
   *********************************************************************
   *
   * Submit the request.
   *
   *********************************************************************
   */
  iError = HTTPSubmitRequest(psURL, HTTP_STREAM_INPUT, asStreamList, HTTP_IGNORE_OUTPUT, NULL, &sResponseHeader, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Check the HTTP response code.
   *
   *********************************************************************
   */
  if (sResponseHeader.iStatusCode < 200 || sResponseHeader.iStatusCode > 299)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Status = [%d], Reason = [%s]", acRoutine, sResponseHeader.iStatusCode, sResponseHeader.acReasonPhrase);
    return ER;
  }

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobDoPutStage
 *
 ***********************************************************************
 */
int
WebJobDoPutStage(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoPutStage()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 i;
  int                 iTimeout;
#ifdef UNIX
  int                 iError;
#endif
#ifdef WIN32
  char               *pcLocalError;
  DWORD               dwStatus;
  DWORD               dwThreadID;
  HANDLE              hThread;
  WEBJOB_THREAD       sThreadArguments;
#endif

  /*-
   *********************************************************************
   *
   * If there's no PutURL, there's nothing to upload.
   *
   *********************************************************************
   */
  if (psProperties->psPutURL == NULL)
  {
    return ER_OK;
  }

  /*-
   *********************************************************************
   *
   * Update the RunStage.
   *
   *********************************************************************
   */
  psProperties->iRunStage = WEBJOB_PUT_STAGE;

  /*-
   *********************************************************************
   *
   * Record the time.
   *
   *********************************************************************
   */
  WebJobGetEpoch(&psProperties->tvPutEpoch);

  /*-
   *********************************************************************
   *
   * Set the timeout value -- zero means no timeout.
   *
   *********************************************************************
   */
#ifdef WIN32
  iTimeout = (psProperties->iPutTimeLimit == 0) ? INFINITE : psProperties->iPutTimeLimit * 1000;
#else
  iTimeout = psProperties->iPutTimeLimit;
#endif

  /*-
   *********************************************************************
   *
   * Hash output streams.
   *
   *********************************************************************
   */
  rewind(psProperties->sWEBJOB.pStdOut);
  psProperties->piHashStream(psProperties->sWEBJOB.pStdOut, psProperties->pcStdOutHash);

  rewind(psProperties->sWEBJOB.pStdErr);
  psProperties->piHashStream(psProperties->sWEBJOB.pStdErr, psProperties->pcStdErrHash);

  /*-
   *********************************************************************
   *
   * Write out WebJob's environment data.
   *
   *********************************************************************
   */
  fprintf(psProperties->sWEBJOB.pStdEnv, "Version%s%s\n",
    WEBJOB_SEPARATOR_S,
    GetMyVersion()
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "Hostname%s%s\n",
    WEBJOB_SEPARATOR_S,
    GetHostname()
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "SystemOS%s%s\n",
    WEBJOB_SEPARATOR_S,
    GetSystemOS()
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "ClientId%s%s\n",
    WEBJOB_SEPARATOR_S,
    psProperties->acClientId
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "GetRequest%s%s\n",
    WEBJOB_SEPARATOR_S,
    psProperties->sWEBJOB.pcRequest
    );
  if (psProperties->psGetHook->iActive)
  {
    fprintf(psProperties->sWEBJOB.pStdEnv, "GetHookCommandLine%s%s\n",
      WEBJOB_SEPARATOR_S,
      (psProperties->psGetHook->pcExpandedCommandLine) ? psProperties->psGetHook->pcExpandedCommandLine : "NONE"
      );
    fprintf(psProperties->sWEBJOB.pStdEnv, "GetHookStatus%s%d/%d (actual/target)\n",
      WEBJOB_SEPARATOR_S,
      psProperties->psGetHook->iActualStatus,
      psProperties->psGetHook->iTargetStatus
      );
  }
  fprintf(psProperties->sWEBJOB.pStdEnv, "Command%s%s\n",
    WEBJOB_SEPARATOR_S,
    psProperties->sWEBJOB.pcCommand
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "CommandLine%s%s\n",
    WEBJOB_SEPARATOR_S,
    (psProperties->sWEBJOB.pcCommandLine) ? psProperties->sWEBJOB.pcCommandLine : "NA"
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "Jid%s%s\n",
    WEBJOB_SEPARATOR_S,
    (psProperties->acJobId[0]) ? psProperties->acJobId : "NA"
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "Pid%s%d\n",
    WEBJOB_SEPARATOR_S,
    (int) getpid()
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "KidPid%s%d\n",
    WEBJOB_SEPARATOR_S,
    psProperties->sWEBJOB.iKidPid
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "KidStatus%s%d\n",
    WEBJOB_SEPARATOR_S,
    psProperties->sWEBJOB.iKidStatus
    );
#ifdef UNIX
  fprintf(psProperties->sWEBJOB.pStdEnv, "KidSignal%s%d\n",
    WEBJOB_SEPARATOR_S,
    psProperties->sWEBJOB.iKidSignal
    );
#endif
  fprintf(psProperties->sWEBJOB.pStdEnv, "KidReason%s%s\n",
    WEBJOB_SEPARATOR_S,
    psProperties->sWEBJOB.acKidReason
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "JobEpoch%s%s (%ld.%06ld)\n",
    WEBJOB_SEPARATOR_S,
    WebJobFormatTime((time_t)psProperties->tvJobEpoch.tv_sec),
    (long) psProperties->tvJobEpoch.tv_sec,
    (long) psProperties->tvJobEpoch.tv_usec
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "GetEpoch%s%s (%ld.%06ld)\n",
    WEBJOB_SEPARATOR_S,
    WebJobFormatTime((time_t)psProperties->tvGetEpoch.tv_sec),
    (long) psProperties->tvGetEpoch.tv_sec,
    (long) psProperties->tvGetEpoch.tv_usec
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "RunEpoch%s%s (%ld.%06ld)\n",
    WEBJOB_SEPARATOR_S,
    WebJobFormatTime((time_t)psProperties->tvRunEpoch.tv_sec),
    (long) psProperties->tvRunEpoch.tv_sec,
    (long) psProperties->tvRunEpoch.tv_usec
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "PutEpoch%s%s (%ld.%06ld)\n",
    WEBJOB_SEPARATOR_S,
    WebJobFormatTime((time_t)psProperties->tvPutEpoch.tv_sec),
    (long) psProperties->tvPutEpoch.tv_sec,
    (long) psProperties->tvPutEpoch.tv_usec
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "HashType%s%s\n",
    WEBJOB_SEPARATOR_S,
    psProperties->acHashType
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "StdOutHash%s", WEBJOB_SEPARATOR_S);
  for (i = 0; i < psProperties->iHashSize; i++)
  {
    fprintf(psProperties->sWEBJOB.pStdEnv, "%02x", psProperties->pcStdOutHash[i]);
  }
  fprintf(psProperties->sWEBJOB.pStdEnv, "\n");
  fprintf(psProperties->sWEBJOB.pStdEnv, "StdErrHash%s", WEBJOB_SEPARATOR_S);
  for (i = 0; i < psProperties->iHashSize; i++)
  {
    fprintf(psProperties->sWEBJOB.pStdEnv, "%02x", psProperties->pcStdErrHash[i]);
  }
  fprintf(psProperties->sWEBJOB.pStdEnv, "\n");
  fprintf(psProperties->sWEBJOB.pStdEnv, "GetError%s%s\n",
    WEBJOB_SEPARATOR_S,
    (psProperties->acGetError[0]) ? psProperties->acGetError : "NA"
    );
  fprintf(psProperties->sWEBJOB.pStdEnv, "RunError%s%s\n",
    WEBJOB_SEPARATOR_S,
    (psProperties->acRunError[0]) ? psProperties->acRunError : "NA"
    );

#ifdef UNIX
  /*-
   *********************************************************************
   *
   * Execute the PUT request. Abort on timeout.
   *
   *********************************************************************
   */
  alarm(iTimeout);
  if (setjmp(psProperties->sPutEnvironment) == 0)
  {
    iError = WebJobDoPutRequest(psProperties, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      return ER;
    }
  }
  else
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: setjmp(): PUT stage timed out after %d seconds.", acRoutine, psProperties->iPutTimeLimit);
    return ER;
  }
  alarm(0);
#endif

#ifdef WIN32
  /*-
   *********************************************************************
   *
   * Create a worker thread and execute the PUT request.
   *
   *********************************************************************
   */
  sThreadArguments.psProperties = psProperties;
  sThreadArguments.pcError = acLocalError;
  sThreadArguments.piRoutine = WebJobDoPutRequest;
  hThread = (HANDLE) _beginthreadex(
    NULL,
    0,
    (void *) WebJobThreadWrapper,
    &sThreadArguments,
    0,
    (void *) &dwThreadID
    );
  if (hThread == INVALID_HANDLE_VALUE)
  {
    FormatWin32Error(GetLastError(), &pcLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: _beginthreadex(): %s", acRoutine, pcLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Wait for the thread to complete. Abort on timeout.
   *
   *********************************************************************
   */
  dwStatus = WaitForSingleObject(hThread, iTimeout);
  if (dwStatus == WAIT_TIMEOUT)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: WaitForSingleObject(): PUT stage timed out after %d seconds.", acRoutine, psProperties->iPutTimeLimit);
    TerminateThread(hThread, ER);
    CloseHandle(hThread);
    return ER;
  }
  else
  {
    if (!GetExitCodeThread(hThread, &dwStatus))
    {
      FormatWin32Error(GetLastError(), &pcLocalError);
      snprintf(pcError, MESSAGE_SIZE, "%s: GetExitCodeThread(): %s", acRoutine, pcLocalError);
      CloseHandle(hThread);
      return ER;
    }
    if (dwStatus != (DWORD) ER_OK) /* The PUT failed. */
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
      CloseHandle(hThread);
      return ER;
    }
  }
  CloseHandle(hThread);
#endif

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobDoRunStage
 *
 ***********************************************************************
 */
int
WebJobDoRunStage(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobDoRunStage()";
  int                 iTimeout;
#ifdef UNIX
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iPid;
  int                 iStatus;
#endif
#ifdef WIN32
  BOOL                bStatus;
  char               *pcLocalError;
  DWORD               dwStatus;
  int                 i;
  PROCESS_INFORMATION sProcessInformation;
  STARTUPINFO         sStartupInformation;
#endif

  /*-
   *********************************************************************
   *
   * Update the RunStage.
   *
   *********************************************************************
   */
  psProperties->iRunStage = WEBJOB_RUN_STAGE;

  /*-
   *********************************************************************
   *
   * Record the time.
   *
   *********************************************************************
   */
  WebJobGetEpoch(&psProperties->tvRunEpoch);

  /*-
   *********************************************************************
   *
   * Return and error, if this stage has been disabled.
   *
   *********************************************************************
   */
  if (psProperties->iRunStageDisabled == 1)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: This stage has been disabled.", acRoutine);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Set the timeout value -- zero means no timeout.
   *
   *********************************************************************
   */
#ifdef WIN32
  iTimeout = (psProperties->iRunTimeLimit == 0) ? INFINITE : psProperties->iRunTimeLimit * 1000;
#else
  iTimeout = psProperties->iRunTimeLimit;
#endif

#ifdef UNIX
  /*-
   *********************************************************************
   *
   * Create a new process and execute the specified command line.
   *
   *********************************************************************
   */
  switch ((psProperties->sWEBJOB.iKidPid = fork()))
  {
  case -1:
    snprintf(pcError, MESSAGE_SIZE, "%s: fork(): %s", acRoutine, strerror(errno));
    return ER;
    break;

  case 0:
    WebJobDoKidStage(psProperties, acLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
    break;
  }

  /*-
   *********************************************************************
   *
   * Close the stdin pipe. If this is not done, programs that expect
   * something on stdin are likely to block. If RunTimeLimit is zero,
   * then they will probably block forever.
   *
   * NOTE: This pipe was established so that future versions could
   * feed data to the child process. To implement that capability,
   * one could add some logic here (e.g. write loop).
   *
   *********************************************************************
   */
  close(psProperties->sWEBJOB.iPipe[WEBJOB_READER_HANDLE]);
/* FIXME Add loop here when writing data to stdin. */
  close(psProperties->sWEBJOB.iPipe[WEBJOB_WRITER_HANDLE]);

  /*-
   *********************************************************************
   *
   * Wait for the process to finish, but terminate it on timeout.
   *
   *********************************************************************
   */
  alarm(iTimeout);
  while ((iPid = wait(&iStatus)) != psProperties->sWEBJOB.iKidPid)
  {
    if (iPid == -1)
    {
      if (errno == EINTR)
      {
        fprintf(stderr, "Main(): %s: wait(): RUN stage timed out after %d seconds.\n", acRoutine, psProperties->iRunTimeLimit);
        if (kill(psProperties->sWEBJOB.iKidPid, SIGKILL) != ER_OK)
        {
          snprintf(pcError, MESSAGE_SIZE, "%s: kill(): KidPid = [%d], %s", acRoutine, psProperties->sWEBJOB.iKidPid, strerror(errno));
          return ER;
        }
      }
      else
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: wait(): %s", acRoutine, strerror(errno));
        return ER;
      }
    }
  }
  alarm(0);

  /*-
   *********************************************************************
   *
   * Determine the kid's exit status.
   *
   *********************************************************************
   */
  psProperties->sWEBJOB.iKidStatus = WEXITSTATUS(iStatus);
  psProperties->sWEBJOB.iKidSignal = WTERMSIG(iStatus);
  if (WIFEXITED(iStatus))
  {
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid exited cleanly.");
  }
  else if (WIFSIGNALED(iStatus))
  {
#ifndef WCOREDUMP
#define WCOREDUMP(w) ((w)&0x80)
#endif
#ifdef WEBJOB_CYGWIN
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid exited due to a signal.");
#else
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid exited due to a signal. A core file %s created.", WCOREDUMP(iStatus) ? "was" : "was not");
#endif
  }
  else if (WIFSTOPPED(iStatus))
  {
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid was stopped. This is not normal behavior.");
  }
  else
  {
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid status could not be determined. This should not happen.");
  }
#endif

#ifdef WIN32
  /*-
   *********************************************************************
   *
   * Create a new process and execute the specified command line.
   *
   *********************************************************************
   */
  GetStartupInfo(&sStartupInformation);
  if (psProperties->psPutURL != NULL)
  {
    sStartupInformation.hStdInput = GetStdHandle(STD_INPUT_HANDLE);
    sStartupInformation.hStdOutput = (HANDLE) _get_osfhandle(_fileno(psProperties->sWEBJOB.pStdOut));
    if (sStartupInformation.hStdOutput == (HANDLE) -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: _get_osfhandle(): stdout, %s", acRoutine, strerror(errno));
      return ER;
    }
    sStartupInformation.hStdError = (HANDLE) _get_osfhandle(_fileno(psProperties->sWEBJOB.pStdErr));
    if (sStartupInformation.hStdError == (HANDLE) -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: _get_osfhandle(): stderr, %s", acRoutine, strerror(errno));
      return ER;
    }
  }
  else
  {
    sStartupInformation.hStdInput = GetStdHandle(STD_INPUT_HANDLE);
    sStartupInformation.hStdOutput = GetStdHandle(STD_OUTPUT_HANDLE);
    sStartupInformation.hStdError = GetStdHandle(STD_ERROR_HANDLE);
  }
  sStartupInformation.dwFlags = STARTF_USESTDHANDLES;

  bStatus = CreateProcess(
    psProperties->sWEBJOB.pcCommand,
    psProperties->sWEBJOB.pcCommandLine,
    0,
    0,
    TRUE,
    0,
    NULL,
    NULL,
    &sStartupInformation,
    &sProcessInformation
    );
  if (!bStatus)
  {
    FormatWin32Error(GetLastError(), &pcLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: CreateProcess(): %s", acRoutine, pcLocalError);
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Wait for the process to finish, but terminate it on timeout.
   *
   *********************************************************************
   */
  for (i = 0; i < 2 && (dwStatus = WaitForSingleObject(sProcessInformation.hProcess, iTimeout)) == WAIT_TIMEOUT; i++)
  {
    fprintf(stderr, "Main(): %s: WaitForSingleObject(): RUN stage timed out after %d seconds.", acRoutine, psProperties->iRunTimeLimit);
    TerminateProcess(sProcessInformation.hProcess, XER_Abort); /* NOTE: This will not kill grandkids. */
  }
  if (dwStatus == WAIT_FAILED)
  {
    FormatWin32Error(GetLastError(), &pcLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: WaitForSingleObject(): %s", acRoutine, pcLocalError);
    TerminateProcess(sProcessInformation.hProcess, XER_Abort); /* NOTE: This will not kill grandkids. */
    return ER;
  }

  /*-
   *********************************************************************
   *
   * Determine the kid's exit status.
   *
   *********************************************************************
   */
  if (i == 0)
  {
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid exited cleanly.");
  }
  else
  {
    snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "The kid was terminated. Any grandkids are now orphans.");
  }
  bStatus = GetExitCodeProcess(sProcessInformation.hProcess, &dwStatus);
  if (!bStatus)
  {
    FormatWin32Error(GetLastError(), &pcLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: GetExitCodeProcess(): %s", acRoutine, pcLocalError);
    return ER;
  }
  psProperties->sWEBJOB.iKidPid = (int) sProcessInformation.dwProcessId;
  psProperties->sWEBJOB.iKidStatus = (int) dwStatus;

  CloseHandle(sProcessInformation.hProcess);
#endif

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobFormatTime
 *
 ***********************************************************************
 */
char *
WebJobFormatTime(time_t time)
{
  static char         acDateTimeZone[64];

  if (time < 0 || strftime(acDateTimeZone, 64, "%Y-%m-%d %H:%M:%S %Z", localtime(&time)) == 0)
  {
    strncpy(acDateTimeZone, "NA", sizeof(acDateTimeZone));
  }
  return acDateTimeZone;
}


/*-
 ***********************************************************************
 *
 * WebJobFreeProperties
 *
 ***********************************************************************
 */
void
WebJobFreeProperties(WEBJOB_PROPERTIES *psProperties)
{
  if (psProperties != NULL)
  {
    if (psProperties->pcStdOutHash != NULL)
    {
      free(psProperties->pcStdOutHash);
    }
    if (psProperties->pcStdErrHash != NULL)
    {
      free(psProperties->pcStdErrHash);
    }
    if (psProperties->sWEBJOB.pcCommandLine != NULL)
    {
      free(psProperties->sWEBJOB.pcCommandLine);
    }
    HTTPFreeURL(psProperties->psGetURL);
    HTTPFreeURL(psProperties->psPutURL);
#ifdef USE_SSL
    SSLFreeProperties(psProperties->psSSLProperties);
#endif
    HookFreeHook(psProperties->psGetHook);
    free(psProperties);
  }
}


/*-
 ***********************************************************************
 *
 * WebJobGetEpoch
 *
 ***********************************************************************
 */
long
WebJobGetEpoch(struct timeval *tvEpoch)
{
  struct timeval      sTimeValue;
#ifdef WIN32
  FILETIME            sFileTime;
  SYSTEMTIME          sSystemTime;
  unsigned __int64    ui64Time;
#endif

  sTimeValue.tv_sec = sTimeValue.tv_usec = -1;

#ifdef WIN32
  GetSystemTime(&sSystemTime);
  if (SystemTimeToFileTime(&sSystemTime, &sFileTime))
  {
    ui64Time = ((unsigned __int64) (sFileTime.dwHighDateTime) << 32) +
                (unsigned __int64) (sFileTime.dwLowDateTime) -
                (unsigned __int64) (UNIX_EPOCH_IN_NT_TIME);
    sTimeValue.tv_sec = (long) (ui64Time / N_100ns_UNITS_IN_1s);
    sTimeValue.tv_usec = (long) ((ui64Time - ((unsigned __int64) sTimeValue.tv_sec * N_100ns_UNITS_IN_1s)) / N_100ns_UNITS_IN_1us);
  }
#else
  gettimeofday(&sTimeValue, NULL);
#endif
  if (tvEpoch != NULL)
  {
    tvEpoch->tv_sec = sTimeValue.tv_sec;
    tvEpoch->tv_usec = sTimeValue.tv_usec;
  }
  return sTimeValue.tv_sec;
}


/*-
 ***********************************************************************
 *
 * WebJobGetHandle
 *
 ***********************************************************************
 */
FILE *
WebJobGetHandle(char *pcFilename, char *pcError)
{
  const char          acRoutine[] = "WebJobGetHandle()";
  FILE               *pFile;
#ifdef WIN32
  char               *pcLocalError;
  HANDLE              hFile;
  int                 iFile;
  SECURITY_ATTRIBUTES sSecurityAttributes;

  sSecurityAttributes.nLength= sizeof(SECURITY_ATTRIBUTES);
  sSecurityAttributes.lpSecurityDescriptor = NULL;
  sSecurityAttributes.bInheritHandle = TRUE;

  pFile = NULL;

  hFile = CreateFile
    (
    pcFilename,
    GENERIC_READ | GENERIC_WRITE,
    0,
    &sSecurityAttributes,
    CREATE_ALWAYS,
    FILE_ATTRIBUTE_NORMAL,
    NULL
    );
  if (hFile == INVALID_HANDLE_VALUE)
  {
    FormatWin32Error(GetLastError(), &pcLocalError);
    snprintf(pcError, MESSAGE_SIZE, "%s: CreateFile(): File = [%s], %s", acRoutine, pcFilename, pcLocalError);
    return NULL;
  }

  iFile = _open_osfhandle((long) hFile, 0);
  if (iFile == ER)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: open_osfhandle(): File = [%s], Handle association failed.", acRoutine, pcFilename);
    return NULL;
  }

  pFile = fdopen(iFile, "wb+");
  if (pFile == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: fdopen(): File = [%s], %s", acRoutine, pcFilename, strerror(errno));
    return NULL;
  }
#else
  pFile = fopen(pcFilename, "wb+");
  if (pFile == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: fopen(): File = [%s], %s", acRoutine, pcFilename, strerror(errno));
    return NULL;
  }
#endif
  return pFile;
}


/*-
 ***********************************************************************
 *
 * WebJobGetPropertiesReference
 *
 ***********************************************************************
 */
WEBJOB_PROPERTIES *
WebJobGetPropertiesReference(void)
{
  return gpsProperties;
}


/*-
 ***********************************************************************
 *
 * WebJobHashSum
 *
 ***********************************************************************
 */
void
WebJobHashSum(char *pcFilename, char *pcHashType)
{
  const char          acRoutine[] = "WebJobHashSum()";
  int                 i;
  int                 iHashSize;
  int               (*piHashStream)();
  FILE               *pFile;
  struct stat         statEntry;
  unsigned char      *pucHash;

  /*-
   *********************************************************************
   *
   * Determine hash type and set various hash related parameters.
   *
   *********************************************************************
   */
  if (strcasecmp(pcHashType, WEBJOB_HASHTYPE_MD5) == 0)
  {
    iHashSize = MD5_HASH_SIZE;
    piHashStream = MD5HashStream;
  }
  else if (strcasecmp(pcHashType, WEBJOB_HASHTYPE_SHA1) == 0)
  {
    iHashSize = SHA1_HASH_SIZE;
    piHashStream = SHA1HashStream;
  }
  else
  {
    fprintf(stderr, "%s: HashType (i.e. digest) must be [%s|%s].\n", acRoutine, WEBJOB_HASHTYPE_MD5, WEBJOB_HASHTYPE_SHA1);
    exit(XER_Abort);
  }

  /*-
   *********************************************************************
   *
   * Prepare input stream. Abort, if a directory was specified.
   *
   *********************************************************************
   */
  if (strcmp(pcFilename, "-") == 0)
  {
#ifdef WIN32 /* Prevent CRLF mappings. */
    if (_setmode(_fileno(stdin), _O_BINARY) == -1)
    {
      fprintf(stderr, "%s: _setmode(): %s\n", acRoutine, strerror(errno));
      exit(XER_Abort);
    }
#endif
    pFile = stdin;
  }
  else
  {
    if (stat(pcFilename, &statEntry) == -1)
    {
      fprintf(stderr, "%s: stat(): %s\n", acRoutine, strerror(errno));
      exit(XER_Abort);
    }
    if ((statEntry.st_mode & S_IFMT) == S_IFDIR)
    {
      fprintf(stderr, "%s: Directory hashing is not supported.\n", acRoutine);
      exit(XER_Abort);
    }
    if ((pFile = fopen(pcFilename, "rb")) == NULL)
    {
      fprintf(stderr, "%s: fopen(): %s\n", acRoutine, strerror(errno));
      exit(XER_Abort);
    }
  }

  /*-
   *********************************************************************
   *
   * Initialize memory, compute hash, and write it to stdout.
   *
   *********************************************************************
   */
  if ((pucHash = calloc(iHashSize, 1)) == NULL)
  {
    fprintf(stderr, "%s: calloc(): %s\n", acRoutine, strerror(errno));
    WEBJOB_SAFE_FCLOSE(pFile);
    exit(XER_Abort);
  }
  piHashStream(pFile, pucHash);
  WEBJOB_SAFE_FCLOSE(pFile);
  for (i = 0; i < iHashSize; i++)
  {
    fprintf(stdout, "%02x", pucHash[i]);
  }
  fprintf(stdout, "\n");
  free(pucHash);

  exit(XER_OK);
}


/*-
 ***********************************************************************
 *
 * WebJobNewProperties
 *
 ***********************************************************************
 */
WEBJOB_PROPERTIES *
WebJobNewProperties(char *pcError)
{
  const char          acRoutine[] = "WebJobNewProperties()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  WEBJOB_PROPERTIES  *psProperties;

  /*-
   *********************************************************************
   *
   * Allocate and clear memory for the properties structure.
   *
   *********************************************************************
   */
  psProperties = (WEBJOB_PROPERTIES *) calloc(sizeof(WEBJOB_PROPERTIES), 1);
  if (psProperties == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: calloc(): %s", acRoutine, strerror(errno));
    return NULL;
  }

  /*-
   *********************************************************************
   *
   * Initialize variables that require a non-zero value.
   *
   *********************************************************************
   */
  WebJobGetEpoch(&psProperties->tvJobEpoch);
  psProperties->iRunMode = WEBJOB_RUNMODE;
  strncpy(psProperties->acTempDirectory, WEBJOB_TEMP_DIRECTORY, sizeof(psProperties->acTempDirectory));

  /*-
   *********************************************************************
   *
   * Initialize RunType variable.
   *
   *********************************************************************
   */
  strncpy(psProperties->acRunType, WEBJOB_RUNTYPE_SNAPSHOT, sizeof(psProperties->acRunType));

  /*-
   *********************************************************************
   *
   * Initialize Hash properties.
   *
   *********************************************************************
   */
  strncpy(psProperties->acHashType, WEBJOB_HASHTYPE_MD5, sizeof(psProperties->acHashType));
  psProperties->iHashSize = MD5_HASH_SIZE;
  psProperties->piHashStream = MD5HashStream;
  psProperties->pcStdOutHash = calloc(psProperties->iHashSize, 1);
  if (psProperties->pcStdOutHash == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: calloc(): %s", acRoutine, strerror(errno));
    WebJobFreeProperties(psProperties);
    return NULL;
  }
  psProperties->pcStdErrHash = calloc(psProperties->iHashSize, 1);
  if (psProperties->pcStdErrHash == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: calloc(): %s", acRoutine, strerror(errno));
    WebJobFreeProperties(psProperties);
    return NULL;
  }

#ifdef USE_SSL
  /*-
   *********************************************************************
   *
   * Initialize SSL properties.
   *
   *********************************************************************
   */
  psProperties->psSSLProperties = SSLNewProperties(acLocalError);
  if (psProperties->psSSLProperties == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    WebJobFreeProperties(psProperties);
    return NULL;
  }
#endif

  /*-
   *********************************************************************
   *
   * Initialize GetHook properties.
   *
   *********************************************************************
   */
  psProperties->psGetHook = HookNewHook(acLocalError);
  if (psProperties->psGetHook == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    WebJobFreeProperties(psProperties);
    return NULL;
  }

  /*-
   *********************************************************************
   *
   * Initialize Kid properties.
   *
   *********************************************************************
   */
  psProperties->sWEBJOB.iKidPid = -1;
  psProperties->sWEBJOB.iKidStatus = -1;
  psProperties->sWEBJOB.iKidSignal = -1;
  snprintf(psProperties->sWEBJOB.acKidReason, MESSAGE_SIZE, "NA");

  return psProperties;
}


/*-
 ***********************************************************************
 *
 * WebJobPrepareHandles
 *
 ***********************************************************************
 */
int
WebJobPrepareHandles(WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "WebJobPrepareHandles()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  char                aacExtensions[3][4];
  char               *pcNonce;
  char               *apcFilenames[3];
  FILE              **appFiles[3];
  int                 i;

  /*-
   *********************************************************************
   *
   * If PutURL is not specified, output goes to the terminal.
   *
   *********************************************************************
   */
  if (psProperties->psPutURL == NULL)
  {
    psProperties->sWEBJOB.pStdOut = stdout;
    psProperties->sWEBJOB.pStdErr = stderr;
    return ER_OK;
  }

  appFiles[0] = &psProperties->sWEBJOB.pStdOut;
  appFiles[1] = &psProperties->sWEBJOB.pStdErr;
  appFiles[2] = &psProperties->sWEBJOB.pStdEnv;

  apcFilenames[0] = psProperties->acStdOutFile;
  apcFilenames[1] = psProperties->acStdErrFile;
  apcFilenames[2] = psProperties->acStdEnvFile;

  strncpy(aacExtensions[0], "out", sizeof(aacExtensions[0]));
  strncpy(aacExtensions[1], "err", sizeof(aacExtensions[1]));
  strncpy(aacExtensions[2], "env", sizeof(aacExtensions[2]));

  pcNonce = MakeRandomExtension();

  for (i = 0; i < 3; i++)
  {
    snprintf(apcFilenames[i], WEBJOB_MAX_PATHNAME_LENGTH, "%s_%ld_%s.%s",
      PROGRAM_NAME,
      (long) psProperties->tvJobEpoch.tv_sec,
      pcNonce,
      aacExtensions[i]
      );
    *appFiles[i] = WebJobGetHandle(apcFilenames[i], acLocalError);
    if (*appFiles[i] == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Directory = [%s]: %s", acRoutine, psProperties->acTempDirectory, acLocalError);
      return ER;
    }
#ifdef UNIX
    if (psProperties->iUnlinkOutput)
    {
      unlink(apcFilenames[i]); /* Minimize file system availability by unlinking now. */
    }
#endif
  }
  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobProcessArguments
 *
 ***********************************************************************
 */
int
WebJobProcessArguments(WEBJOB_PROPERTIES *psProperties, int iArgumentCount, char *ppcArgumentVector[])
{
  int                 iComplete = 0;

  if (iArgumentCount < 2)
  {
    WebJobUsage();
  }

  iArgumentCount -= 2;

  if (strcmp(ppcArgumentVector[1], "-e") == 0 || strcmp(ppcArgumentVector[1], "--execute") == 0)
  {
    if (iArgumentCount >= 3)
    {
      if (strcmp(ppcArgumentVector[2], "-f") == 0 || strcmp(ppcArgumentVector[2], "--file") == 0)
      {
        psProperties->pcConfigFile = ppcArgumentVector[3];
        iArgumentCount -= 2;
        psProperties->sWEBJOB.pcRequest = ppcArgumentVector[4];
        psProperties->sWEBJOB.pcCommand = ppcArgumentVector[4];
        psProperties->sWEBJOB.pcCommandLine = BuildCommandLine(iArgumentCount, &ppcArgumentVector[4]);
        psProperties->sWEBJOB.ppcArguments = &ppcArgumentVector[4];
        psProperties->sWEBJOB.iArgumentCount = iArgumentCount;
        iComplete = 1;
      }
    }
  }
  else if (strcmp(ppcArgumentVector[1], "-h") == 0 || strcmp(ppcArgumentVector[1], "--hashsum") == 0)
  {
    if (iArgumentCount == 3)
    {
      if (strcmp(ppcArgumentVector[2], "-t") == 0 || strcmp(ppcArgumentVector[2], "--type") == 0)
      {
        WebJobHashSum(ppcArgumentVector[4], ppcArgumentVector[3]);
      }
    }
  }
  else if (strcmp(ppcArgumentVector[1], "-v") == 0 || strcmp(ppcArgumentVector[1], "--version") == 0)
  {
    WebJobVersion();
  }

  if (!iComplete)
  {
    WebJobUsage();
  }

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * WebJobSetPropertiesReference
 *
 ***********************************************************************
 */
void
WebJobSetPropertiesReference(WEBJOB_PROPERTIES *psProperties)
{
  gpsProperties = psProperties;
}


/*-
 ***********************************************************************
 *
 * WebJobShutdown
 *
 ***********************************************************************
 */
int
WebJobShutdown(WEBJOB_PROPERTIES *psProperties, int iLastError, int iErrorCount)
{
  int                 iFinalError = iLastError;

  /*-
   *********************************************************************
   *
   * Return, if the properties structure has not been defined.
   *
   *********************************************************************
   */
  if (psProperties == NULL)
  {
    return iFinalError;
  }

  /*-
   *********************************************************************
   *
   * Make final cleanup adjustments.  If the error occurred in the
   * configure stage, then the job never got the point where output
   * files would be useful, so they should be deleted.  If there
   * were multiple errors, set the external error to XER_MultiStage.
   *
   *********************************************************************
   */
  if (iLastError != XER_OK)
  {
    if (iLastError == XER_Configure)
    {
      psProperties->iUnlinkOutput = 1;
    }
    if (iErrorCount > 1)
    {
      iFinalError = XER_MultiStage;
    }
  }

  /*-
   *********************************************************************
   *
   * Close/Unlink files as required.
   *
   *********************************************************************
   */
  if (psProperties->iUnlinkExecutable)
  {
    unlink(psProperties->sWEBJOB.pcCommand);
    if (psProperties->psGetHook->iActive)
    {
      unlink(psProperties->sWEBJOB.pcRequest);
    }
  }
  WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pStdOut);
  WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pStdErr);
  WEBJOB_SAFE_FCLOSE(psProperties->sWEBJOB.pStdEnv);
  if (psProperties->iUnlinkOutput)
  {
    unlink(psProperties->acStdOutFile);
    unlink(psProperties->acStdErrFile);
    unlink(psProperties->acStdEnvFile);
  }

  return iFinalError;
}


#ifdef WIN32
/*-
 ***********************************************************************
 *
 * WebJobThreadWrapper
 *
 ***********************************************************************
 */
DWORD WINAPI
WebJobThreadWrapper(LPVOID lpArgument)
{
  const char          acRoutine[] = "WebJobThreadWrapper()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  int                 iError;
  WEBJOB_THREAD      *psThreadArguments;

  psThreadArguments = (WEBJOB_THREAD *) lpArgument;

  iError = psThreadArguments->piRoutine(psThreadArguments->psProperties, acLocalError);
  if (iError != ER_OK)
  {
    snprintf(psThreadArguments->pcError, MESSAGE_SIZE, "%s: %s", acRoutine, acLocalError);
    return ER;
  }
  return ER_OK;
}
#endif


/*-
 ***********************************************************************
 *
 * WebJobTimeoutHandler
 *
 ***********************************************************************
 */
#ifdef UNIX
void
WebJobTimeoutHandler()
{
  WEBJOB_PROPERTIES  *psProperties;

  psProperties = WebJobGetPropertiesReference();
  switch (psProperties->iRunStage)
  {
  case WEBJOB_GET_STAGE:
    longjmp(psProperties->sGetEnvironment, ER);
    break;
  case WEBJOB_RUN_STAGE:
    break;
  case WEBJOB_PUT_STAGE:
    longjmp(psProperties->sPutEnvironment, ER);
    break;
  default:
    break;
  }
  return;
}
#endif


/*-
 ***********************************************************************
 *
 * WebJobUsage
 *
 ***********************************************************************
 */
void
WebJobUsage(void)
{
  fprintf(stderr, "\n");
  fprintf(stderr, "Usage: webjob {-e|--execute} {-f|--file} config program [options]\n");
  fprintf(stderr, "       webjob {-h|--hashsum} {-t|--type} digest file\n");
  fprintf(stderr, "       webjob {-v|--version}\n");
  fprintf(stderr, "\n");
  exit(XER_Usage);
}


/*-
 ***********************************************************************
 *
 * WebJobVersion
 *
 ***********************************************************************
 */
void
WebJobVersion(void)
{
  fprintf(stdout, "%s\n", GetMyVersion());
  exit(XER_OK);
}
