/*-
 ***********************************************************************
 *
 * $Id: properties.c,v 1.19 2004/04/09 17:22:26 mavrik Exp $
 *
 ***********************************************************************
 *
 * Copyright 2001-2004 Klayton Monroe, All Rights Reserved.
 *
 ***********************************************************************
 */

#include "all-includes.h"

/*-
 ***********************************************************************
 *
 * Defines
 *
 ***********************************************************************
 */
#define PROPERTIES_MAX_RECURSION_LEVEL 3
#define PROPERTIES_MAX_LINE_LENGTH 1024
#define PROPERTIES_COMMENT_C '#'
#define PROPERTIES_COMMENT_S "#"
#define PROPERTIES_SEPARATOR_C '='
#define PROPERTIES_SEPARATOR_S "="


/*-
 ***********************************************************************
 *
 * Macros
 *
 ***********************************************************************
 */
#define DUPLICATE_ERROR(b) \
  if ((b) == TRUE) \
  { \
    snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Duplicate controls aren't allowed.", acRoutine, pcControl); \
    return ER_DuplicateControl; \
  }

#define EVALUATE_TWOSTATE(pc, s1, s2, result) \
  if (strcasecmp(pc, s1) == 0) \
  {  \
    result = TRUE; \
  } \
  else if (strcasecmp(pc, s2) == 0) \
  {  \
    result = FALSE; \
  }  \
  else \
  { \
    snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be [%s|%s].", acRoutine, pcControl, pc, s1, s2); \
    return ER_Bool; \
  }


/*-
 ***********************************************************************
 *
 * PropertiesReadFile
 *
 ***********************************************************************
 */
int
PropertiesReadFile(char *pcFilename, WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "PropertiesReadFile()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  char                acLine[PROPERTIES_MAX_LINE_LENGTH];
  char               *pc;
  int                 iError;
  int                 iLength;
  int                 iLineNumber;
  int                 iSaveLength;
  FILE               *pFile;
#ifdef UNIX
  struct stat         statEntry;
#endif

  /*-
   *********************************************************************
   *
   * Check recursion level. Abort, if the level is too high.
   *
   *********************************************************************
   */
  if (psProperties->iImportRecursionLevel > PROPERTIES_MAX_RECURSION_LEVEL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s], Imports may not exceed %d levels of recursion.", acRoutine, pcFilename, PROPERTIES_MAX_RECURSION_LEVEL);
    return ER_MaxIndirection;
  }

  if (strcmp(pcFilename, "-") == 0)
  {
    pFile = stdin;
  }
  else
  {
#ifdef UNIX
    iError = lstat(pcFilename, &statEntry);
    if (iError == -1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s]: %s", acRoutine, pcFilename, strerror(errno));
      return ER_lstat;
    }

    if (!S_ISREG(statEntry.st_mode))
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s]: A regular file is required.", acRoutine, pcFilename);
      return ER_BadValue;
    }
#endif

    if ((pFile = fopen(pcFilename, "r")) == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s]: %s", acRoutine, pcFilename, strerror(errno));
      return ER_fopen;
    }
  }

  for (acLine[0] = 0, iLineNumber = 1; fgets(acLine, PROPERTIES_MAX_LINE_LENGTH, pFile) != NULL; acLine[0] = 0, iLineNumber++)
  {
    iLength = iSaveLength = strlen(acLine);

    /*-
     *******************************************************************
     *
     * Ignore full line comments.
     *
     *******************************************************************
     */
    if (acLine[0] == PROPERTIES_COMMENT_C)
    {
      continue;
    }

    /*-
     *******************************************************************
     *
     * Scan backwards over EOL characters.
     *
     *******************************************************************
     */
    while (iLength && ((acLine[iLength - 1] == '\r') || (acLine[iLength - 1] == '\n')))
    {
      iLength--;
    }

    /*-
     *******************************************************************
     *
     * Its an error if no EOL was found. The exception to this is EOF.
     *
     *******************************************************************
     */
    if (iLength == iSaveLength && !feof(pFile))
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s], Line = [%d]: Length exceeds %d bytes.", acRoutine, pcFilename, iLineNumber, PROPERTIES_MAX_LINE_LENGTH - 1);
      if (pFile != stdin)
      {
        fclose(pFile);
      }
      return ER_Length;
    }

    /*-
     *******************************************************************
     *
     * Terminate the line excluding any EOL characters.
     *
     *******************************************************************
     */
    acLine[iLength] = 0;

    /*-
     *******************************************************************
     *
     * Look for the first imbedded comment and mark it.
     *
     *******************************************************************
     */
    if ((pc = strstr(acLine, PROPERTIES_COMMENT_S)) != NULL)
    {
      *pc = 0;
      iLength = strlen(acLine);
    }

    /*-
     *******************************************************************
     *
     * Burn any trailing white space off line.
     *
     *******************************************************************
     */
    while (isspace((int) acLine[iLength - 1]))
    {
      acLine[iLength--] = 0;
    }

    /*-
     *******************************************************************
     *
     * If there's anything left over, process it.
     *
     *******************************************************************
     */
    if (iLength)
    {
      iError = PropertiesReadLine(acLine, psProperties, acLocalError);
      if (iError != ER_OK)
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s], Line = [%d]: %s", acRoutine, pcFilename, iLineNumber, acLocalError);
        if (pFile != stdin)
        {
          fclose(pFile);
        }
        return iError;
      }
    }
  }
  if (ferror(pFile))
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: File = [%s], Line = [%d]: %s", acRoutine, pcFilename, iLineNumber, strerror(errno));
    if (pFile != stdin)
    {
      fclose(pFile);
    }
    return ER_fgets;
  }
  if (pFile != stdin)
  {
    fclose(pFile);
  }

  return ER_OK;
}


/*-
 ***********************************************************************
 *
 * PropertiesReadLine
 *
 ***********************************************************************
 */
int
PropertiesReadLine(char *pcLine, WEBJOB_PROPERTIES *psProperties, char *pcError)
{
  const char          acRoutine[] = "PropertiesReadLine()";
  char                acLocalError[MESSAGE_SIZE] = { 0 };
  char               *pc;
  char               *pcControl;
  char               *pcEnd;
  unsigned char      *pcTempHash;
  int                 iError;
  int                 iRunMode;
  int                 iValue;
  unsigned int        iLength;

  /*-
   *********************************************************************
   *
   * Process one line of input from the config file. It is assumed that
   * the input string has already been stripped of comments and EOLs.
   * The string is expected to have the following form: "control:value"
   *
   *********************************************************************
   */

  /*-
   *********************************************************************
   *
   * Look for the first separator, and mark it to isolate the control.
   *
   *********************************************************************
   */
  if ((pc = strstr(pcLine, PROPERTIES_SEPARATOR_S)) == NULL)
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Line does not contain a control/value separator (i.e. '%s').", acRoutine, PROPERTIES_SEPARATOR_S);
    return ER_Jibberish;
  }
  *pc++ = 0;
  pcControl = pcLine;

  /*-
   *********************************************************************
   *
   * Burn any leading white space off value.
   *
   *********************************************************************
   */
  while (isspace((int) *pc))
  {
    pc++;
  }

  /*-
   *********************************************************************
   *
   * Burn any trailing white space off value.
   *
   *********************************************************************
   */
  iLength = strlen(pc);
  pcEnd = &pc[iLength - 1];
  while (isspace((int) *pcEnd))
  {
    *pcEnd-- = 0;
    iLength--;
  }

  /*-
   *********************************************************************
   *
   * Burn any leading white space off control.
   *
   *********************************************************************
   */
  while (isspace((int) *pcControl))
  {
    pcControl++;
  }

  /*-
   *********************************************************************
   *
   * Burn any trailing white space off control.
   *
   *********************************************************************
   */
  pcEnd = &pcControl[strlen(pcControl) - 1];
  while (isspace((int) *pcEnd))
  {
    *pcEnd-- = 0;
  }

  /*-
   *********************************************************************
   *
   * At this point pc should be pointing at a control value. Sift
   * through the various controls and do the appropriate things. If
   * the control is unrecognized, complain about it.
   *
   *********************************************************************
   */

  iRunMode = psProperties->iRunMode;

  if (strcasecmp(pcControl, KEY_HashType) == 0 && RUN_MODE_IS_SET(MODES_HashType, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.HashTypeFound);
    if (strcasecmp(pc, WEBJOB_HASHTYPE_MD5) == 0)
    {
      strncpy(psProperties->acHashType, WEBJOB_HASHTYPE_MD5, WEBJOB_HASHTYPE_SIZE);
      psProperties->iHashSize = MD5_HASH_SIZE;
      psProperties->piHashStream = MD5HashStream;
    }
    else if (strcasecmp(pc, WEBJOB_HASHTYPE_SHA1) == 0)
    {
      strncpy(psProperties->acHashType, WEBJOB_HASHTYPE_SHA1, WEBJOB_HASHTYPE_SIZE);
      psProperties->iHashSize = SHA1_HASH_SIZE;
      psProperties->piHashStream = SHA1HashStream;
    }
    else
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value must be [%s|%s].", acRoutine, pcControl, WEBJOB_HASHTYPE_MD5, WEBJOB_HASHTYPE_SHA1);
      return ER_BadValue;
    }
    pcTempHash = realloc(psProperties->pcStdOutHash, psProperties->iHashSize);
    if (pcTempHash == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: realloc(): %s", acRoutine, strerror(errno));
      return ER;
    }
    psProperties->pcStdOutHash = pcTempHash;
    memset(psProperties->pcStdOutHash, 0, psProperties->iHashSize);
    pcTempHash = realloc(psProperties->pcStdErrHash, psProperties->iHashSize);
    if (pcTempHash == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: realloc(): %s", acRoutine, strerror(errno));
      return ER;
    }
    psProperties->pcStdErrHash = pcTempHash;
    memset(psProperties->pcStdErrHash, 0, psProperties->iHashSize);
    psProperties->sFound.HashTypeFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_Import) == 0 && RUN_MODE_IS_SET(MODES_Import, iRunMode))
  {
    psProperties->iImportRecursionLevel++;
    iError = PropertiesReadFile(pc, psProperties, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->iImportRecursionLevel--;
  }

  else if (strcasecmp(pcControl, KEY_OverwriteExecutable) == 0 && RUN_MODE_IS_SET(MODES_OverwriteExecutable, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.OverwriteExecutableFound);
    EVALUATE_TWOSTATE(pc, "y", "n", psProperties->iOverwriteExecutable);
    psProperties->sFound.OverwriteExecutableFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_RunType) == 0 && RUN_MODE_IS_SET(MODES_RunType, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.RunTypeFound);
    if (strcasecmp(pc, WEBJOB_RUNTYPE_LINKTEST) == 0)
    {
      strncpy(psProperties->acRunType, WEBJOB_RUNTYPE_LINKTEST, WEBJOB_RUNTYPE_SIZE);
    }
    else if (strcasecmp(pc, WEBJOB_RUNTYPE_SNAPSHOT) == 0)
    {
      strncpy(psProperties->acRunType, WEBJOB_RUNTYPE_SNAPSHOT, WEBJOB_RUNTYPE_SIZE);
    }
    else
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value must be [%s|%s].", acRoutine, pcControl, WEBJOB_RUNTYPE_LINKTEST, WEBJOB_RUNTYPE_SNAPSHOT);
      return ER_BadValue;
    }
    psProperties->sFound.RunTypeFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_TempDirectory) == 0 && RUN_MODE_IS_SET(MODES_TempDirectory, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.TempDirectoryFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PATHNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    strncpy(psProperties->acTempDirectory, pc, WEBJOB_MAX_PATHNAME_LENGTH);
    psProperties->sFound.TempDirectoryFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_UnlinkExecutable) == 0 && RUN_MODE_IS_SET(MODES_UnlinkExecutable, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.UnlinkExecutableFound);
    EVALUATE_TWOSTATE(pc, "y", "n", psProperties->iUnlinkExecutable);
    psProperties->sFound.UnlinkExecutableFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_UnlinkOutput) == 0 && RUN_MODE_IS_SET(MODES_UnlinkOutput, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.UnlinkOutputFound);
    EVALUATE_TWOSTATE(pc, "y", "n", psProperties->iUnlinkOutput);
    psProperties->sFound.UnlinkOutputFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_URLAuthType) == 0 && RUN_MODE_IS_SET(MODES_URLAuthType, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.URLAuthTypeFound);
    if (strcasecmp(pc, "basic") == 0)
    {
      psProperties->iURLAuthType = HTTP_AUTH_TYPE_BASIC;
    }
    else if (strcasecmp(pc, "none") == 0)
    {
      psProperties->iURLAuthType = HTTP_AUTH_TYPE_NONE;
    }
    else
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value must be [basic|none].", acRoutine, pcControl);
      return ER_BadValue;
    }
    psProperties->sFound.URLAuthTypeFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_URLGetURL) == 0 && RUN_MODE_IS_SET(MODES_URLGetURL, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.URLGetURLFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PATHNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    psProperties->psGetURL = HTTPParseURL(pc, acLocalError);
    if (psProperties->psGetURL == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return ER_HTTPParseURL;
    }
    psProperties->sFound.URLGetURLFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_URLPutURL) == 0 && RUN_MODE_IS_SET(MODES_URLPutURL, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.URLPutURLFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PATHNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    psProperties->psPutURL = HTTPParseURL(pc, acLocalError);
    if (psProperties->psPutURL == NULL)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return ER_HTTPParseURL;
    }
    psProperties->sFound.URLPutURLFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_URLDownloadLimit) == 0 && RUN_MODE_IS_SET(MODES_URLDownloadLimit, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.URLDownloadLimitFound);
    while (iLength > 0)
    {
      if (!isdigit((int) pc[iLength - 1]))
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be 1-%d.", acRoutine, pcControl, pc, WEBJOB_MAX_CONTENT_LENGTH);
        return ER_BadValue;
      }
      iLength--;
    }
    iValue = atoi(pc);
    if (iValue < 1 || iValue > WEBJOB_MAX_CONTENT_LENGTH)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be 1-%d.", acRoutine, pcControl, pc, WEBJOB_MAX_CONTENT_LENGTH);
      return ER_BadValue;
    }
    else
    {
      psProperties->iURLDownloadLimit = iValue;
    }
    psProperties->sFound.URLDownloadLimitFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_URLUsername) == 0 && RUN_MODE_IS_SET(MODES_URLUsername, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.URLUsernameFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_USERNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    strncpy(psProperties->acURLUsername, pc, WEBJOB_MAX_USERNAME_LENGTH);
    psProperties->sFound.URLUsernameFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_URLPassword) == 0 && RUN_MODE_IS_SET(MODES_URLPassword, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.URLPasswordFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PASSWORD_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    strncpy(psProperties->acURLPassword, pc, WEBJOB_MAX_PASSWORD_LENGTH);
    psProperties->sFound.URLPasswordFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_GetTimeLimit) == 0 && RUN_MODE_IS_SET(MODES_GetTimeLimit, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.GetTimeLimitFound);
    while (iLength > 0)
    {
      if (!isdigit((int) pc[iLength - 1]))
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be %d-%d.", acRoutine, pcControl, pc, WEBJOB_MIN_TIME_LIMIT, WEBJOB_MAX_TIME_LIMIT);
        return ER_BadValue;
      }
      iLength--;
    }
    iValue = atoi(pc);
    if (iValue < WEBJOB_MIN_TIME_LIMIT || iValue > WEBJOB_MAX_TIME_LIMIT)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be %d-%d.", acRoutine, pcControl, pc, WEBJOB_MIN_TIME_LIMIT, WEBJOB_MAX_TIME_LIMIT);
      return ER_BadValue;
    }
    else
    {
      psProperties->iGetTimeLimit = iValue;
    }
    psProperties->sFound.GetTimeLimitFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_PutTimeLimit) == 0 && RUN_MODE_IS_SET(MODES_PutTimeLimit, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.PutTimeLimitFound);
    while (iLength > 0)
    {
      if (!isdigit((int) pc[iLength - 1]))
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be %d-%d.", acRoutine, pcControl, pc, WEBJOB_MIN_TIME_LIMIT, WEBJOB_MAX_TIME_LIMIT);
        return ER_BadValue;
      }
      iLength--;
    }
    iValue = atoi(pc);
    if (iValue < WEBJOB_MIN_TIME_LIMIT || iValue > WEBJOB_MAX_TIME_LIMIT)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be %d-%d.", acRoutine, pcControl, pc, WEBJOB_MIN_TIME_LIMIT, WEBJOB_MAX_TIME_LIMIT);
      return ER_BadValue;
    }
    else
    {
      psProperties->iPutTimeLimit = iValue;
    }
    psProperties->sFound.PutTimeLimitFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_RunTimeLimit) == 0 && RUN_MODE_IS_SET(MODES_RunTimeLimit, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.RunTimeLimitFound);
    while (iLength > 0)
    {
      if (!isdigit((int) pc[iLength - 1]))
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be %d-%d.", acRoutine, pcControl, pc, WEBJOB_MIN_TIME_LIMIT, WEBJOB_MAX_TIME_LIMIT);
        return ER_BadValue;
      }
      iLength--;
    }
    iValue = atoi(pc);
    if (iValue < WEBJOB_MIN_TIME_LIMIT || iValue > WEBJOB_MAX_TIME_LIMIT)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be %d-%d.", acRoutine, pcControl, pc, WEBJOB_MIN_TIME_LIMIT, WEBJOB_MAX_TIME_LIMIT);
      return ER_BadValue;
    }
    else
    {
      psProperties->iRunTimeLimit = iValue;
    }
    psProperties->sFound.RunTimeLimitFound = TRUE;
  }

#ifdef USE_SSL
  else if (strcasecmp(pcControl, KEY_SSLBundledCAsFile) == 0 && RUN_MODE_IS_SET(MODES_SSLBundledCAsFile, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLBundledCAsFileFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PATHNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    iError = SSLSetBundledCAsFile(psProperties->psSSLProperties, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.SSLBundledCAsFileFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLPassPhrase) == 0 && RUN_MODE_IS_SET(MODES_SSLPassPhrase, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLPassPhraseFound);
    iError = SSLSetPassPhrase(psProperties->psSSLProperties, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.SSLPassPhraseFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLPublicCertFile) == 0 && RUN_MODE_IS_SET(MODES_SSLPublicCertFile, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLPublicCertFileFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PATHNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    iError = SSLSetPublicCertFile(psProperties->psSSLProperties, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.SSLPublicCertFileFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLMaxChainLength) == 0 && RUN_MODE_IS_SET(MODES_SSLMaxChainLength, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLMaxChainLengthFound);
    while (iLength > 0)
    {
      if (!isdigit((int) pc[iLength - 1]))
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be 1-10.", acRoutine, pcControl, pc);
        return ER_BadValue;
      }
      iLength--;
    }
    iValue = atoi(pc);
    if (iValue < 1 || iValue > 10 /*SSL_MAX_CHAIN_LENGTH*/)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%d], Value must be 1-10.", acRoutine, pcControl, iValue);
      return ER_BadValue;
    }
    else
    {
      psProperties->psSSLProperties->iMaxChainLength = iValue;
    }
    psProperties->sFound.SSLMaxChainLengthFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLPrivateKeyFile) == 0 && RUN_MODE_IS_SET(MODES_SSLPrivateKeyFile, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLPrivateKeyFileFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_PATHNAME_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    iError = SSLSetPrivateKeyFile(psProperties->psSSLProperties, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.SSLPrivateKeyFileFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLExpectedPeerCN) == 0 && RUN_MODE_IS_SET(MODES_SSLExpectedPeerCN, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLExpectedPeerCNFound);
    iError = SSLSetExpectedPeerCN(psProperties->psSSLProperties, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.SSLExpectedPeerCNFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLUseCertificate) == 0 && RUN_MODE_IS_SET(MODES_SSLUseCertificate, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLUseCertificateFound);
    EVALUATE_TWOSTATE(pc, "y", "n", psProperties->psSSLProperties->iUseCertificate);
    psProperties->sFound.SSLUseCertificateFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_SSLVerifyPeerCert) == 0 && RUN_MODE_IS_SET(MODES_SSLVerifyPeerCert, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.SSLVerifyPeerCertFound);
    EVALUATE_TWOSTATE(pc, "y", "n", psProperties->psSSLProperties->iVerifyPeerCert);
    psProperties->sFound.SSLVerifyPeerCertFound = TRUE;
  }
#endif

  else if (strcasecmp(pcControl, KEY_GetHookEnable) == 0 && RUN_MODE_IS_SET(MODES_GetHookEnable, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.GetHookEnableFound);
    EVALUATE_TWOSTATE(pc, "y", "n", psProperties->psGetHook->iActive);
    iError = HookSetValue(psProperties->psGetHook, HOOK_VALUE_ORIGINAL_FILENAME, psProperties->sWEBJOB.pcCommand, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return ER;
    }
    psProperties->sFound.GetHookEnableFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_GetHookCommandLine) == 0 && RUN_MODE_IS_SET(MODES_GetHookCommandLine, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.GetHookCommandLineFound);
    iError = HookSetValue(psProperties->psGetHook, HOOK_VALUE_ORIGINAL_COMMAND_LINE, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.GetHookCommandLineFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_GetHookStatus) == 0 && RUN_MODE_IS_SET(MODES_GetHookStatus, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.GetHookStatusFound);
    while (iLength > 0)
    {
      if (!isdigit((int) pc[iLength - 1]))
      {
        snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%s], Value must be 0-%d.", acRoutine, pcControl, pc, XER_MaxExternalErrorCode);
        return ER_BadValue;
      }
      iLength--;
    }
    iValue = atoi(pc);
    if (iValue < 0 || iValue > XER_MaxExternalErrorCode)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Value = [%d], Value must be 0-%d.", acRoutine, pcControl, iValue, XER_MaxExternalErrorCode);
      return ER_BadValue;
    }
    else
    {
      psProperties->psGetHook->iTargetStatus = iValue;
    }
    psProperties->sFound.GetHookStatusFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_GetHookSuffix) == 0 && RUN_MODE_IS_SET(MODES_GetHookSuffix, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.GetHookSuffixFound);
    iError = HookSetValue(psProperties->psGetHook, HOOK_VALUE_SUFFIX, pc, acLocalError);
    if (iError != ER_OK)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s]: %s", acRoutine, pcControl, acLocalError);
      return iError;
    }
    psProperties->sFound.GetHookSuffixFound = TRUE;
  }

  else if (strcasecmp(pcControl, KEY_ClientId) == 0 && RUN_MODE_IS_SET(MODES_ClientId, iRunMode))
  {
    DUPLICATE_ERROR(psProperties->sFound.ClientIdFound);
    if (iLength < 1 || iLength > WEBJOB_MAX_CLIENTID_LENGTH - 1)
    {
      snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], Invalid length [%d].", acRoutine, pcControl, iLength);
      return ER_Length;
    }
    strncpy(psProperties->acClientId, pc, WEBJOB_MAX_CLIENTID_LENGTH);
    psProperties->sFound.ClientIdFound = TRUE;
  }

  else
  {
    snprintf(pcError, MESSAGE_SIZE, "%s: Control = [%s], The specified control is not valid in this mode of operation.", acRoutine, pcLine);
    return ER_Jibberish;
  }

  return ER_OK;
}
