/***************************************************************************
                          snmpdialogimp.cpp  -  description
                             -------------------
    begin                : Fri Dec 14 2001
    copyright            : (C) 2001 by Ori Pessach
    email                : mail@oripessach.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <qlabel.h>
#include <qlineedit.h>
#include <qtabwidget.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qbuttongroup.h>
#include <qmessagebox.h>
#include <qvalidator.h>
#include <stdlib.h>
#include <unistd.h>
#include <fstream>
#include <iostream>
#include "sstream"
#include "snmpdialogimp.h"
#include "passworddlgimp.h"
#include "wap11config.h"
#include "wap11exception.h"
#include "macvalidator.h"
#include "trapsdialogimp.h"

using namespace std;

SNMPDialogImp::SNMPDialogImp(): WEPKeysValid(NONE_VALID)
{
  etherstats=new EthernetStatsImp(w11c);
  wirelessstats=new WirelessStatsImp(w11c);
  changepassword=new ChangePassImp(w11c);
  macauth=new MacAuthImp(w11c);
  trapsdialog=new TrapsDialogImp();
  EditKey1->setValidator(new KeyValidator(*this, KEY1_VALID));
  EditKey2->setValidator(new KeyValidator(*this, KEY2_VALID));
  EditKey3->setValidator(new KeyValidator(*this, KEY3_VALID));
  EditKey4->setValidator(new KeyValidator(*this, KEY4_VALID));
  PreferredBSSInfo->setValidator(new MACValidator(0));
  RemoteMacAddressInfo->setValidator(new MACValidator(0));
}

SNMPDialogImp::~SNMPDialogImp()
{
}

/** Invoked when the Cancel button is pressed */
void SNMPDialogImp::slotCancelButton()
{
  done(0);
}

/** Invoked when the Close button is clicked */
void SNMPDialogImp::slotCloseButton()
{
  SaveOptFile();
  done(0);
}

void SNMPDialogImp::slotAccessPointModeCheck()
{
  SetAccessPointModeFieldEnables();
}

void SNMPDialogImp::slotApplyButton()
{
  if( WEPKeysValid!=ALL_VALID && ComboWEP->currentItem()!=1 )
  {
    // Don't upload invalid WEP keys:
    TabWidget->setCurrentPage(4);  // take the user to the right page
    QMessageBox::warning( this, "WAP11 SNMP Configuration",
                          "The WEP keys are incomplete. You must correct this \n"
                          "before you can upload your changes to the WAP11.", "OK");
    return;
  }

  QMessageBox mb( "WAP11 SNMP Configuration",
                  "Upload configuration to the WAP11?",
                  QMessageBox::Information,
                  QMessageBox::Yes | QMessageBox::Default,
                  QMessageBox::Cancel | QMessageBox::Escape,
                  0 );
  mb.setButtonText( QMessageBox::Yes, "Upload" );
  mb.setButtonText( QMessageBox::Cancel, "Cancel" );
  switch( mb.exec() )
  {
    case QMessageBox::Cancel:
      break;
    case QMessageBox::Yes:
      UpdateW11C();
      try
      {
        w11c.Upload();
      }
      catch( WAP11Exception &e )
      {
        ReportException(e);
      }
      if(w11c.Get())
      {
        UpdateUI();
        ApplyButton->setEnabled(false);
      }
      else
      {
        DisableUI();
      }
      break;
  }
}

void SNMPDialogImp::SetAccessPointModeFieldEnables()
{
  if (AccessPointBtn->isOn())
  {
    OperationModeText->setText("Access Point");
    PreferredBSSText->setEnabled(false);
    PreferredBSSInfo->setEnabled(false);
    RemoteMacAddressText->setEnabled(false);
    RemoteMacAddressInfo->setEnabled(false);
    WirelessBridgeGrp->setEnabled(false);
  }
  else if (AccessPointClientBtn->isOn())
  {
    OperationModeText->setText("Access Point Client");
    PreferredBSSText->setEnabled(true);
    PreferredBSSInfo->setEnabled(true);
    RemoteMacAddressText->setEnabled(false);
    RemoteMacAddressInfo->setEnabled(false);
    WirelessBridgeGrp->setEnabled(false);
  }
  else
  {
    PreferredBSSText->setEnabled(false);
    PreferredBSSInfo->setEnabled(false);
    WirelessBridgeGrp->setEnabled(true);

    if (PointToPointBtn->isOn())
    {
      OperationModeText->setText("Wireless Bridge -- Point to Point");
      RemoteMacAddressText->setEnabled(true);
      RemoteMacAddressInfo->setEnabled(true);
    }
    else
    {
      OperationModeText->setText("Wireless Bridge -- Point to Multi-Point");
      RemoteMacAddressText->setEnabled(false);
      RemoteMacAddressInfo->setEnabled(false);
    }
  }
}

/** Final initialization stage - prompt the user for password. */
void SNMPDialogImp::polish()
{
  setMaximumSize(size());
  setMinimumSize(size());
  ReadOptFile();
  Connect();
}

/** Prompt the user for the password and IP address, and try to connect to the AP.
If successful, download info from the AP, and set the UI. If not, disable most of 
the UI. */
void SNMPDialogImp::Connect()
{
  pdi.AddAddresses(addr_vec);
  if(pdi.exec()==1)
  {
    w11c.SetPassword(pdi.GetPassword());
    w11c.SetIPAddress(pdi.GetIPAddress());
    try
    {
      if(w11c.Get())
      {
        EnableUI();
        UpdateUI();
      }
      else
      {
        DisableUI();
      }
      int found_index=-1;
      for(int i=0; i<int(addr_vec.size()); ++i)
        if( addr_vec[i]==pdi.GetIPAddress() )
        {
          found_index=i;
          break;
        }
      if(found_index==-1)
      {
        addr_vec.push_back(pdi.GetIPAddress());
        found_index=addr_vec.size()-1;
      }
      // Put the selected item on top:
      std::string t=addr_vec[0];
      addr_vec[0]=addr_vec[found_index];
      addr_vec[found_index]=t;
    }
    catch(WAP11SNMPAuthError &e)
    {
      // We need to do this, in case this was called to reconnect:
      DisableUI();
      ReportException(e);
    }
    catch(WAP11Exception &e)
    {
      DisableUI();
      ReportException(e);
    }
  }
  else
  {
    DisableUI();
  }
  ApplyButton->setEnabled(false);
}

/** Disable most of the UI. For use when the app isn't connected to the AP. */
void SNMPDialogImp::DisableUI()
{
  // Disable all the tabs except the first and last ones:
  for( int i=1; i<6; ++i)
  {
   TabWidget->setCurrentPage(i);
   TabWidget->setTabEnabled(TabWidget->currentPage(), false);
  }
  WorkingStatusText->setText("Couldn't connect to access point");
  RegulatoryDomainText->setText("");
  FirmwareVersionText->setText("");
  EthernetAddressText->setText("");
  ESSIDText->setText("");
  OperationModeText->setText("");
  ChannelText->setText("");
  WEPText->setText("");

  ResetButton->setEnabled(false);
  RestoreDefaultsButton->setEnabled(false);

  EtherStatsButton->setEnabled(false);
  WirelessStatsButton->setEnabled(false);

  TabWidget->setCurrentPage(0);

  WEPKeysValid=NONE_VALID; // This will force the WEP Keys fields to be erased
  EditPassphrase->setText("");
}

/** Updates the UI's fields with data from our WAP11Config instance. w11c must
be valid for this to work (after calling Get() successfully). */
void SNMPDialogImp::UpdateUI()
{
  std::string domain=w11c.DecodeDomain(w11c.GetRegDomain());

  RegulatoryDomainText->setText(domain.c_str());
  std::string essid;
  w11c.GetVariable("OperationalSettingsGRP.operESSID.0", essid);
  ESSIDText->setText(essid.c_str());
  EditESSID->setText(essid.c_str());

  char s[16];
  int channel_id;
  w11c.GetVariable("OperationalSettingsGRP.operChannelID.0", channel_id);
  sprintf(s, "%d", channel_id);
  ChannelText->setText(s);
  WAP11Config::bytevector_t ethernet_address;
  w11c.GetVariable("operBridgingLevel.operEthernetAddress.0", ethernet_address);
  char ea[19];
  sprintf(ea, "%02X %02X %02X %02X %02X %02X",
          (unsigned char)ethernet_address[0],
          (unsigned char)ethernet_address[1],
          (unsigned char)ethernet_address[2],
          (unsigned char)ethernet_address[3],
          (unsigned char)ethernet_address[4],
          (unsigned char)ethernet_address[5]);
  EthernetAddressText->setText(ea);

  FirmwareVersionText->setText(w11c.GetFirmwareVersion().c_str());

  ComboChannel->setCurrentItem(channel_id-1);
  int auth_type;
  w11c.GetVariable("OperationalSettingsGRP.operAuthenticationType.0", auth_type);
  switch( auth_type )
  {
  case 1: // open system
    RBOpenSystem->setChecked(true);
    break;
  case 2: // shared key
    RBSharedKey->setChecked(true);
    break;
  case 3: // both types
    RBBoth->setChecked(true);
    break;
  }

  std::string tstr;
  int tint;
  w11c.GetVariable("OperationalSettingsGRP.operAccessPointName.0", tstr);
  EditAPName->setText(tstr.c_str());
  w11c.GetVariable("OperationalSettingsGRP.operRTSThreshold.0", tint);
  SpinRTSThresh->setValue(tint);
  w11c.GetVariable("OperationalSettingsGRP.operFragmentationThreshold.0", tint);
  SpinFragThresh->setValue(tint);
  w11c.GetVariable("OperationalSettingsGRP.operAutoRateFallBack.0", tint);
  CheckAutoRateFallback->setChecked(tint==1);
  w11c.GetVariable("sysCtrlGrp.sysTrapSwitch.0", tint);
  CheckEnableTraps->setChecked(tint==1);
  w11c.GetVariable("OperationalSettingsGRP.operPreambleType.0", tint);
  switch(tint)
  {
    case 1: // short preamble
      RBShortPreamble->setChecked(true);
      break;
    case 2: // long preamble
      RBLongPreamble->setChecked(true);
      break;
  }
  w11c.GetVariable("OperationalSettingsGRP.operSSIDBroadcasting.0", tint);
  DisableSSIDBroadcastBtn->setChecked(tint == 2);
  w11c.GetVariable("operFiltering.operIpFilter.0", tint);
  IpFilteringBtn->setChecked(tint==1);
  w11c.GetVariable("OperationalSettingsGRP.operAntennaSettings.0", tint);
  // Set antenna settings from this field
  ReceiveAntennaCombo->setCurrentItem((tint&3) - 1);
  TransmitAntennaCombo->setCurrentItem(((tint >> 8) & 3) - 1);
  w11c.GetVariable("operBridgingLevel.operDHCP.0)", tint);
  CheckDHCPEnable->setChecked(tint==1);
  w11c.GetVariable("operBridgingLevel.operPrimaryPort.0", tint);
  RBWireless->setChecked(tint==2);
  RBEthernet->setChecked(tint==1);

  if(!CheckDHCPEnable->isChecked())
  {
    ButtonGroup3_2->setEnabled(false);
  }
  else
  {
    ButtonGroup3_2->setEnabled(true);
  }

  w11c.GetVariable("operBridgingLevel.operIPAddress.0", tstr);
  EditIPAddress->setText(tstr.c_str());
  w11c.GetVariable("operBridgingLevel.operIPMask.0", tstr);
  EditIPMask->setText(tstr.c_str());

  w11c.GetVariable("operBridgingLevel.operGateway.0", tstr);
  if(tstr=="0.0.0.0")
    EditGateway->setText("");
  else
    EditGateway->setText(tstr.c_str());

  w11c.GetVariable("OperationalSettingsGRP.operBasicRates.0", basic_rates);
  CheckRate1->setChecked((basic_rates[0]&0x80)!=0);
  CheckRate2->setChecked((basic_rates[1]&0x80)!=0);
  CheckRate5_5->setChecked((basic_rates[2]&0x80)!=0);
  CheckRate11->setChecked((basic_rates[3]&0x80)!=0);

  int wep;
  w11c.GetVariable("privacyGRP.privacyWEPEnable.0", tint);
  wep=tint-1;
  ComboWEP->setCurrentItem(wep);
  char *wepmodes[]={ (char*)"64bit", (char*)"Disabled", (char*)"128bit"};
  WEPText->setText(wepmodes[wep]);
  if(wep==1)
  {
    // WEP is disabled
    EditPassphrase->setEnabled(false);
    DoneButton->setEnabled(false);
    ButtonGroup4->setEnabled(false);
  }
  else
  {
    EditPassphrase->setEnabled(true);
    DoneButton->setEnabled(true);
    ButtonGroup4->setEnabled(true);

    // Don't overwrite the WEP keys fields, if they were previously valid,
    // because the AP doesn't divulge the WEP keys via SNMP:
    if(WEPKeysValid!=ALL_VALID)
    {
      EditKey1->setText(w11c.GetWEPKey(1).c_str());
      EditKey2->setText(w11c.GetWEPKey(2).c_str());
      EditKey3->setText(w11c.GetWEPKey(3).c_str());
      EditKey4->setText(w11c.GetWEPKey(4).c_str());
    }
    w11c.GetVariable("privacyGRP.privacyDefaultWEPKeyID.0", tint);
    ComboDefaultKey->setCurrentItem(tint-1);
  }

  int mac_authorization_enabled;
  w11c.GetVariable("AuthorizedMacAddressesGRP.AuthorizationMacEnable.0", mac_authorization_enabled);
  AuthorizedMACButton->setEnabled(mac_authorization_enabled==1);
  CheckEnableMACAuthorization->setChecked(mac_authorization_enabled==1);
  
  // AP Mode stuff (4/14/2002 G. Oliver)
  int aptype;
  w11c.GetVariable("operBridgeOperationalMode.bridgeOperationalMode.0", aptype);

  AccessPointBtn->setChecked(aptype == 2);
  AccessPointClientBtn->setChecked(aptype == 3);
  WirelessBridgeBtn->setChecked(aptype == 1 || aptype == 4);
  PointToPointBtn->setChecked(aptype == 4);
  PointToMultiPointBtn->setChecked(aptype == 1);

  WAP11Config::bytevector_t tvec;
  w11c.GetVariable("operBridgeOperationalMode.bridgeRemoteBridgeBSSID.0", tvec);
  RemoteMacAddressInfo->setText(VectorToStr(tvec).c_str());
  PreferredBSSInfo->setText(VectorToStr(tvec).c_str());

  // Set the field and button enables as needed
  SetAccessPointModeFieldEnables();
}

/** Invoked when the Reconnect button is clicked(). */
void SNMPDialogImp::slotReconnectButton()
{
  Connect();
}

/** Enables the tabs, buttons, and entry fields. */
void SNMPDialogImp::EnableUI()
{
  for( int i=1; i<6; ++i)
  {
   TabWidget->setCurrentPage(i);
   TabWidget->setTabEnabled(TabWidget->currentPage(), true);
  }
  TabWidget->setCurrentPage(0);
  WorkingStatusText->setText("Connected to access point");

  ResetButton->setEnabled(true);
  RestoreDefaultsButton->setEnabled(true);

  ComboChannel->clear();
  for(int i=1; i<=11; ++i)
  {
    char s[16];
    sprintf(s, "Channel %d", i);
    ComboChannel->insertItem(s, -1);
  }

  EtherStatsButton->setEnabled(true);
  WirelessStatsButton->setEnabled(true);

  ComboWEP->clear();
  ComboWEP->insertItem("Enable 64bit", -1);
  ComboWEP->insertItem("Disable", -1);
  ComboWEP->insertItem("Enable 128bit", -1);

  ComboDefaultKey->clear();
  ComboDefaultKey->insertItem("Key 1", -1);
  ComboDefaultKey->insertItem("Key 2", -1);
  ComboDefaultKey->insertItem("Key 3", -1);
  ComboDefaultKey->insertItem("Key 4", -1);
}

/** Update the UI state when the DHCP checkbox is toggled */
void SNMPDialogImp::slotDHCPToggled()
{
  if(!CheckDHCPEnable->isChecked())
  {
    ButtonGroup3_2->setEnabled(false);
  }
  else
  {
    ButtonGroup3_2->setEnabled(true);
  }

  EnableApplyButton();
}

/** Enables the dialog's Apply button after the data was changed. */
void SNMPDialogImp::EnableApplyButton()
{
  ApplyButton->setEnabled(true);
}

/** Called when data in the dialog was modified */
void SNMPDialogImp::slotDataChanged()
{
  EnableApplyButton();
}

/** No descriptions */
void SNMPDialogImp::slotWEPChanged()
{
  const char *wepmodes[]={ "64bit", "Disabled", "128bit"};

  WEPText->setText(wepmodes[ComboWEP->currentItem()]);
  w11c.SetVariable("privacyGRP.privacyWEPEnable.0", ComboWEP->currentItem()+1);

  if(ComboWEP->currentItem()==1)
  {
    EditPassphrase->setEnabled(false);
    DoneButton->setEnabled(false);
    ButtonGroup4->setEnabled(false);
  }
  else
  {
    EditPassphrase->setEnabled(true);
    DoneButton->setEnabled(true);
    ButtonGroup4->setEnabled(true);
    EditKey1->setText(w11c.GetWEPKey(1).c_str());
    EditKey2->setText(w11c.GetWEPKey(2).c_str());
    EditKey3->setText(w11c.GetWEPKey(3).c_str());
    EditKey4->setText(w11c.GetWEPKey(4).c_str());
  }
}
/** Update the WEP keys from the passphrase: */
void SNMPDialogImp::slotEnteredPassphrase()
{
  w11c.SetPassphrase(EditPassphrase->text(), ComboWEP->currentItem()+1);
  EditKey1->setText(w11c.GetWEPKey(1).c_str());
  EditKey2->setText(w11c.GetWEPKey(2).c_str());
  EditKey3->setText(w11c.GetWEPKey(3).c_str());
  EditKey4->setText(w11c.GetWEPKey(4).c_str());
  WEPKeysValid=ALL_VALID;
}

/** Copy fields from the UI to our WAP11Config instance. */
void SNMPDialogImp::UpdateW11C()
{
  w11c.SetVariable("OperationalSettingsGRP.operESSID.0", QStostr(EditESSID->text().left(32)));
  w11c.SetVariable("OperationalSettingsGRP.operESSIDLength.0", min((uint)32, EditESSID->text().length()));
  w11c.SetVariable("OperationalSettingsGRP.operChannelID.0", ComboChannel->currentItem()+1);
  w11c.SetVariable("OperationalSettingsGRP.operAccessPointName.0", QStostr(EditAPName->text()));

  w11c.SetVariable("OperationalSettingsGRP.operRTSThreshold.0", SpinRTSThresh->value());
  w11c.SetVariable("OperationalSettingsGRP.operFragmentationThreshold.0", SpinFragThresh->value());
  w11c.SetVariable("OperationalSettingsGRP.operPreambleType.0", RBShortPreamble->isChecked() ? 1 : 2);
  w11c.SetVariable("OperationalSettingsGRP.operAuthenticationType.0",
                   RBOpenSystem->isChecked()?1:(RBSharedKey->isChecked()?2:3));

  // Update the basic_rates vector:
  basic_rates[0]=(basic_rates[0]&0x7f)|(CheckRate1->isChecked()?0x80:0x00);
  basic_rates[1]=(basic_rates[1]&0x7f)|(CheckRate2->isChecked()?0x80:0x00);
  basic_rates[2]=(basic_rates[2]&0x7f)|(CheckRate5_5->isChecked()?0x80:0x00);
  basic_rates[3]=(basic_rates[3]&0x7f)|(CheckRate11->isChecked()?0x80:0x00);
  w11c.SetVariable("OperationalSettingsGRP.operBasicRates.0", basic_rates);
  w11c.SetVariable("OperationalSettingsGRP.operAutoRateFallBack.0", CheckAutoRateFallback->isChecked()? 1 : 2);
  w11c.SetVariable("sysCtrlGRP.sysTrapSwitch.0", CheckEnableTraps->isChecked()? 1 : 2);
  w11c.SetVariable("OperationalSettingsGRP.operSSIDBroadcasting.0", DisableSSIDBroadcastBtn->isOn() ? 2 : 1);
  w11c.SetVariable("operFiltering.operIpFilter.0", IpFilteringBtn->isOn() ? 1 : 2);
  w11c.SetVariable("OperationalSettingsGRP.operAntennaSettings.0",
      ((ReceiveAntennaCombo->currentItem() + 1)) +
      ((TransmitAntennaCombo->currentItem() + 1) << 8));
  w11c.SetVariable("operBridgingLevel.operIPAddress.0", IPtoV(QStostr(EditIPAddress->text())));
  w11c.SetVariable("operBridgingLevel.operIPMask.0", IPtoV(QStostr(EditIPMask->text())));
  w11c.SetVariable("operBridgingLevel.operGateway.0", IPtoV(QStostr(EditGateway->text())));
  w11c.SetVariable("operBridgingLevel.operDHCP.0", CheckDHCPEnable->isChecked() ? 1 : 2 );
  w11c.SetVariable("operBridgingLevel.operPrimaryPort.0", RBEthernet->isChecked() ? 1 : 2);
  w11c.SetVariable("privacyGRP.privacyDefaultWEPKeyID.0", ComboDefaultKey->currentItem()+1);
  w11c.SetVariable("privacyGRP.privacyWEPEnable.0", ComboWEP->currentItem()+1);

  if( WEPKeysValid )
  {
    w11c.SetVariable("privacyGRP.defaultWEPKey1.0", StrToVector(QStostr(EditKey1->text())));
    w11c.SetVariable("privacyGRP.defaultWEPKey2.0", StrToVector(QStostr(EditKey2->text())));
    w11c.SetVariable("privacyGRP.defaultWEPKey3.0", StrToVector(QStostr(EditKey3->text())));
    w11c.SetVariable("privacyGRP.defaultWEPKey4.0", StrToVector(QStostr(EditKey4->text())));
  }

  if (AccessPointBtn->isOn())
  {
    w11c.SetVariable("operBridgeOperationalMode.bridgeOperationalMode.0", 2);
  }
  else if (AccessPointClientBtn->isOn())
  {
    w11c.SetVariable("operBridgeOperationalMode.bridgeOperationalMode.0", 3);
    w11c.SetVariable("operBridgeOperationalMode.bridgeRemoteBridgeBSSID.0", StrToVector(QStostr(PreferredBSSInfo->text())));
  }
  else
  {
    if (PointToPointBtn->isOn())
    {
      w11c.SetVariable("operBridgeOperationalMode.bridgeOperationalMode.0", 4);
    }
    else
    {
      w11c.SetVariable("operBridgeOperationalMode.bridgeOperationalMode.0", 1);
    }
    w11c.SetVariable("operBridgeOperationalMode.bridgeRemoteBridgeBSSID.0", StrToVector(QStostr(RemoteMacAddressInfo->text())));
  }
}

/** Convert a QString to an std::string. */
std::string SNMPDialogImp::QStostr(const QString &s)
{
  return std::string((const char *)s, 0, s.length());
}
/** Convert an IP address in dot notation to an unsigned 32 bit int. */
uint32_t SNMPDialogImp::BinIP(const std::string &ip)
{
  uint32_t ipn=0;
  std::string n;
  for(int i=0; i<(int)ip.length(); ++i)
  {
    if(ip[i]=='.')
    {
      ipn=(ipn<<8) | atoi(n.c_str());
      n="";
    }
    else
    {
      n=n+ip[i];
    }
  }
  ipn=(ipn<<8) | atoi(n.c_str());
  return ipn;
}

/** Convert a string IP address in dot notation to a vector of 4 octets. Weird. */
WAP11Config::bytevector_t SNMPDialogImp::IPtoV( const std::string &s)
{
  WAP11Config::bytevector_t v(4);

  uint32_t ipa=BinIP(s);

  v[0]=(uint32_t)(ipa&0xff000000)>>24;
  v[1]=(uint32_t)(ipa&0xff0000)>>16;
  v[2]=(uint32_t)(ipa&0xff00)>>8;
  v[3]=(uint32_t)(ipa&0xff);

  return v;
}

/** Issues a reset command */
void SNMPDialogImp::slotResetButton()
{
  QMessageBox mb( "WAP11 SNMP Configuration",
                  "Reset the WAP11?",
                  QMessageBox::Information,
                  QMessageBox::Yes | QMessageBox::Default,
                  QMessageBox::Cancel | QMessageBox::Escape,
                  0 );
  mb.setButtonText( QMessageBox::Yes, "Reset" );
  mb.setButtonText( QMessageBox::Cancel, "Cancel" );
  switch( mb.exec() )
  {
    case QMessageBox::Cancel:
      break;
    case QMessageBox::Yes:
      w11c.Reset();
      if(w11c.Get())
      {
        UpdateUI();
      }
      else
      {
        DisableUI();
      }
      break;
  }
}
/** Invoked when the Ethernet Statistics button is clicked */
void SNMPDialogImp::slotEthernetStatsButton()
{
  etherstats->Reload();
  etherstats->show();
}

/** Invoked when the Wireless Statistics button is clicked*/
void SNMPDialogImp::slotWirelessStatsButton()
{
  wirelessstats->Reload();
  wirelessstats->show();
}

/** Invoked when the Password Setting button is clicked */
void SNMPDialogImp::slotPasswordSettingButton()
{
  changepassword->Clear();

  if(changepassword->exec()==QDialog::Accepted)
  {
    w11c.UpdatePassword(QStostr(changepassword->GetNewPassword()));
  }
}
/** Converts a string containing hex octets into a byte vector. */
WAP11Config::bytevector_t SNMPDialogImp::StrToVector(const std::string &s)
{
  WAP11Config::bytevector_t v;
  int n=0;    	
  int ndigit=0;

  for(int i=0; i<(int)s.length(); ++i)
  {
    if(isxdigit(s[i]))
    {
      n<<=4;
      if(isdigit(s[i]))
      n+=s[i]-'0';
      else
      n+=toupper(s[i])-'A'+10;			
      ndigit++;
    }
    else
    {
      if(ndigit!=0)
        ndigit=2; // force the current number out
    }

    if(ndigit>1)
    {
      v.push_back((char)n);
      n=0;
      ndigit=0;
    }
  }
  return v;
}

/** Format a byte vector to a string of hexadecimal numbers. */
std::string SNMPDialogImp::VectorToStr(const WAP11Config::bytevector_t &v)
{
  std::stringstream ss;

  for(int i=0; i<(int)v.size(); ++i)
  {
    ss.fill('0'); ss.width(2);
    ss << hex << (int)v[i] << " ";
  }
  std::string s=ss.str();
  return s;
}


/** Return the required length of WEP keys, in non-space hex digits. */
int SNMPDialogImp::GetWEPKeyLength() const
{
  if(ComboWEP->currentItem()==0)
  {
    return 10;
  }
  else
  {
    return 26;
  }
}

/** Mark a WEP key field as containing a valid key. This is called by KeyValidator. */
void SNMPDialogImp::SetKeyValid(SNMPDialogImp::KeyField f)
{
  WEPKeysValid |= f;
}

/** Mark a WEP key field as containing an invalid key. */
void SNMPDialogImp::SetKeyInvalid(SNMPDialogImp::KeyField f)
{
  WEPKeysValid &= ~f;
}

/** Opens the options file, ~/.wap11gui. If the file isn't found, ask the user whether
to create one. Once the file is open, read it and return. */
void SNMPDialogImp::ReadOptFile()
{
  string filename;
  filename += getenv("HOME");
  filename += "/.wap11gui";
  fstream f( filename.c_str(), ios::in);
  if( !f )
  {
    if(QMessageBox::warning( this, "WAP11 SNMP Configuration",
                             "The configuration file, ~/.wap11gui, wasn't found. \n"
                             "Would you like to create it now?",
                             "Yes", "No") == 1)
    {
      QMessageBox::warning( this, "WAP11 SNMP Configuration",
                                  "OK, proceeding without a configuration file.\n"
                                  "Some (few) configuration changes won't be saved.");
      return;
    }
    else
    {
      f.open( filename.c_str(), ios::app | ios::in | ios::out);
    }
  }

  std::string l;
  f >> l; // Skip the first line for now - It's the file format version number.
  f >> l;
  while( f )
  {
    addr_vec.push_back(l);
    f >> l;
  }
  f.close();
}

/** Displays information about an exception */
void SNMPDialogImp::ReportException(WAP11Exception &e)
{
  std::stringstream sline;
  sline << "An exception was thrown:\n" << e.report()
        << "\nPossible cause: " << e.cause();
  QMessageBox::critical( this, "WAP11 SNMP Configuration", QString(sline.str().c_str()));
}

/** Saves the current options to ~/.wap11gui */
void SNMPDialogImp::SaveOptFile()
{
  string filename;
  filename += getenv("HOME");
  filename += "/.wap11gui";
  fstream f( filename.c_str(), ios::out | ios::trunc);

  f << "000100" << endl;

  for( int i=0; i<int(addr_vec.size()); ++i)
  {
    f << addr_vec[i] << endl;
  }
  f.close();
}

/** Invoked when the Restore Defaults button is pressed. */
void SNMPDialogImp::slotRestoreDefaultsButton()
{
  if(QMessageBox::information( this, "WAP11 SNMP Configuration",
                               "Restore Access Point to factory defaults?",
                               "Yes", "No", QString::null, 1) == 0)
  {
    w11c.RestoreDefaults();
    w11c.Reset();
    if(w11c.Get())
    {
      UpdateUI();
      ApplyButton->setEnabled(false);
    }
    else
    {
      DisableUI();
    }
  }
}
/** Invoked when clicking the Authrized MAC pushbutton */
void SNMPDialogImp::slotAuthorizedMAC()
{
  macauth->show();
}

/** Triggered when the user clicks the Enable MAC Authorization checkbox. */
void SNMPDialogImp::slotEnableMACAuthorization()
{
  AuthorizedMACButton->setEnabled(CheckEnableMACAuthorization->isChecked());
  int mac_auth_enabled=CheckEnableMACAuthorization->isChecked()==1 ? 1 : 2;
  w11c.SetVariable("AuthorizedMacAddressesGRP.AuthorizationMacEnable.0", mac_auth_enabled);
  ApplyButton->setEnabled(true);
}

/** Invoked when the Show Traps button is clicked. */
void SNMPDialogImp::slotTrapButton()
{
  trapsdialog->show();
}

/** Invoked when the Help button is clicked. */
void SNMPDialogImp::slotHelpButton()
{
  if(!fork())
  {
    // Run assistant to show our help file. There must be a better way of
    // doing this...
    system("assistant "+QString(DATADIR)+"/wap11gui.html");
    exit(0);
  }
}
