/*
 *  Copyright (c) 2016, Facebook, Inc.
 *  All rights reserved.
 *
 *  This source code is licensed under the BSD-style license found in the
 *  LICENSE file in the root directory of this source tree. An additional grant
 *  of patent rights can be found in the PATENTS file in the same directory.
 *
 */

#pragma once

#include <folly/MoveWrapper.h>
#include <gmock/gmock.h>
#include <wangle/channel/Handler.h>

namespace wangle {

template <class Rin, class Rout = Rin, class Win = Rout, class Wout = Rin>
class MockHandler : public Handler<Rin, Rout, Win, Wout> {
 public:
  typedef typename Handler<Rin, Rout, Win, Wout>::Context Context;

  MockHandler() = default;
  MockHandler(MockHandler&&) = default;

  MOCK_METHOD2_T(read_, void(Context*, Rin&));
  MOCK_METHOD1_T(readEOF, void(Context*));
  MOCK_METHOD2_T(readException, void(Context*, folly::exception_wrapper));

  MOCK_METHOD2_T(write_, void(Context*, Win&));
  MOCK_METHOD1_T(close_, void(Context*));
  MOCK_METHOD2_T(writeException_, void(Context*, folly::exception_wrapper));

  MOCK_METHOD1_T(attachPipeline, void(Context*));
  MOCK_METHOD1_T(attachTransport, void(Context*));
  MOCK_METHOD1_T(detachPipeline, void(Context*));
  MOCK_METHOD1_T(detachTransport, void(Context*));

  void read(Context* ctx, Rin msg) override {
    read_(ctx, msg);
  }

  folly::Future<folly::Unit> write(Context* ctx, Win msg) override {
    return folly::makeFutureWith([&](){
      write_(ctx, msg);
    });
  }

  folly::Future<folly::Unit> close(Context* ctx) override {
    return folly::makeFutureWith([&](){
      close_(ctx);
    });
  }

  folly::Future<folly::Unit> writeException(
    Context* ctx, folly::exception_wrapper ex) override {
    return folly::makeFutureWith([&](){
        writeException_(ctx, ex);
    });
  }
};

template <class R, class W = R>
using MockHandlerAdapter = MockHandler<R, R, W, W>;

class MockBytesToBytesHandler : public BytesToBytesHandler {
 public:
  folly::MoveWrapper<folly::Future<folly::Unit>> defaultFuture() {
    return folly::MoveWrapper<folly::Future<folly::Unit>>();
  }

  MOCK_METHOD1(transportActive, void(Context*));
  MOCK_METHOD1(transportInactive, void(Context*));
  MOCK_METHOD2(read, void(Context*, folly::IOBufQueue&));
  MOCK_METHOD1(readEOF, void(Context*));
  MOCK_METHOD2(readException, void(Context*, folly::exception_wrapper));
  MOCK_METHOD2(write,
               folly::MoveWrapper<folly::Future<folly::Unit>>(
                 Context*,
                 std::shared_ptr<folly::IOBuf>));
  MOCK_METHOD1(mockClose,
               folly::MoveWrapper<folly::Future<folly::Unit>>(Context*));
  MOCK_METHOD2(mockWriteException,
               folly::MoveWrapper<folly::Future<folly::Unit>>(
                   Context*, folly::exception_wrapper));

  folly::Future<folly::Unit> write(Context* ctx,
                                   std::unique_ptr<folly::IOBuf> buf) override {
    std::shared_ptr<folly::IOBuf> sbuf(buf.release());
    return write(ctx, sbuf).move();
  }

  folly::Future<folly::Unit> close(Context* ctx) override {
    return mockClose(ctx).move();
  }

  folly::Future<folly::Unit> writeException(
      Context* ctx, folly::exception_wrapper ex) override {
    return mockWriteException(ctx, ex).move();
  }
};

} // namespace wangle
