// This is brl/bbas/volm/volm_candidate_list.h
#ifndef volm_candidate_list_h_
#define volm_candidate_list_h_
//:
// \file
// \brief A class to create candidate list from probability map generated by volm_matcher
//
// \author Yi Dong
// \date April 17, 2013
// \verbatim
// Modifications
//  <none yet>
// \endverbatim

#include <iostream>
#include <functional>
#include <map>
#include <vgl/vgl_polygon.h>
#include <vil/algo/vil_find_4con_boundary.h>
#ifdef _MSC_VER
#  include <vcl_msvc_warnings.h>
#endif
#include <volm/volm_io.h>
#include <volm/volm_tile.h>
#include <volm/volm_camera_space.h>
#include <vil/vil_image_view.h>
#include <vgl/vgl_polygon.hxx>
#include <vgl/algo/vgl_convex_hull_2d.h>

typedef std::multimap<unsigned, vgl_point_2d<int>, std::greater<unsigned> > mymap;

class volm_candidate_list
{
  public:
    //: default constructor
    volm_candidate_list() = default;
    //: constructor
    volm_candidate_list(vil_image_view<vxl_byte> const& image,
                        unsigned threshold);
    //: destructor
    ~volm_candidate_list() = default;

    //: access
    vgl_polygon<int> cand_poly() { return poly_; }
    unsigned threshold()         { return thres_; }

    //: number of sheets in candidate list
    unsigned num_of_regions() { return n_sheet_; }

    //: best score for each sheet
    bool region_score(std::vector<unsigned>& scores);

    //: return the top pixel which has highest scores
    bool top_locations(std::vector<std::vector<vgl_point_2d<int> > >& top_locs, std::vector<std::vector<unsigned> >& top_loc_scores);

    //: for each sheet, return top number of pixels which have high scores
    bool top_locations(std::vector<std::vector<vgl_point_2d<int> > >& top_locs, std::vector<std::vector<unsigned> >& top_loc_scores, unsigned const& size);

    //: return the top number of pixels which have highest scores, for sheet sh_idx
    bool top_locations(std::vector<vgl_point_2d<int> >& top_locs, std::vector<unsigned>& top_loc_scores, unsigned const& size, unsigned const& sh_idx);

    //: return the top number of locs which have highest scores, for sheet sh_idx
    bool top_locations(std::vector<vgl_point_2d<double> >& top_locs, std::vector<unsigned>& top_loc_scores, volm_tile& tile, unsigned const& size, unsigned const& sh_idx);

    //: create an image with candidate region highlighted
    bool candidate_list_image(vil_image_view<vxl_byte>& image);

    //: transfer the region coordinates from image to global, returned points vector are the polygon for kml
    bool img_to_golbal(unsigned const& sh_idx, volm_tile& tile, std::vector<vgl_point_2d<double> >& region_global);

    //: find the region where the given location is in
    bool find(unsigned const& i, unsigned const& j, unsigned& sh_idx, unsigned& loc_score);
    bool find(vgl_point_2d<unsigned> const& loc, unsigned& sh_idx, unsigned& loc_score)
    {
      return this->find(loc.x(), loc.y(), sh_idx, loc_score);
    }
    bool find(double const& lon, double const& lat, volm_tile& tile, unsigned& sh_idx, unsigned& loc_score)
    {
      unsigned u, v;
      if (!tile.global_to_img(lon, lat, u, v))
        return false;
      return this->find(u, v, sh_idx, loc_score);
    }
    //: write the header of the kml
    static void open_kml_document(std::ofstream& str, std::string const& name, float const& threshold);
    //: close the document
    static void close_kml_document(std::ofstream& str);
    //: write a region into kml
    static void write_kml_regions(std::ofstream& str,
                                  std::vector<vgl_point_2d<double> >& region,
                                  std::vector<vgl_point_2d<double> >& top_locs,
                                  std::vector<cam_angles>& camera,
                                  std::vector<double>& right_fov,
                                  float const& likelihood,
                                  unsigned const& rank);
    static void write_kml_regions(std::ofstream& str,
                                  std::vector<vgl_point_2d<double> >& region,
                                  vgl_point_2d<double>& top_loc,
                                  cam_angles const& camera,
                                  double const& right_fov,
                                  float const& likelihood,
                                  unsigned const& rank);
    //: write a candidate region associated with its favored heading direction (represented as a line)
    static void write_kml_regions(std::ofstream& str,
                                  std::vector<vgl_point_2d<double> >& region,
                                  vgl_point_2d<double>& top_loc,
                                  std::vector<vgl_point_2d<double> >& heading,
                                  std::vector<vgl_point_2d<double> >& viewing,
                                  std::vector<vgl_point_2d<double> >& landmarks,
                                  std::vector<unsigned char>& landmark_types,
                                  float const& likelihood,
                                  unsigned const& rank);

    //: generate a circular region given a circular center and radius
    static bool generate_pin_point_circle(vgl_point_2d<double> const& center, double const& radius, std::vector<vgl_point_2d<double> >& circle);
    //: generate a heading directional line and camera viewing volume given the camera center and heading direction
    static bool generate_heading_direction(vgl_point_2d<double> const& center, float const& heading_angle, float const& length, float const& right_fov,
                                           std::vector<vgl_point_2d<double> >& heading_line,
                                           std::vector<vgl_point_2d<double> >& viewing);
    //: method to check whether a given point is inside the candidate region
    static bool inside_candidate_region(vgl_polygon<double> const& cand_poly, double const& lon, double const& lat);
    static bool inside_candidate_region(vgl_polygon<double> const& cand_poly, vgl_point_2d<double> const& pt);
    static bool inside_candidate_region(vgl_polygon<double> const& cand_poly_in, vgl_polygon<double> const& cand_poly_out, double const& lon, double const& lat);
    static bool inside_candidate_region(vgl_polygon<double> const& cand_poly_in, vgl_polygon<double> const& cand_poly_out, vgl_point_2d<double> const& pt);

  private:
    unsigned thres_;
    unsigned n_sheet_;
    vil_image_view<vxl_byte> image_;
    //: pixels that have values larger than thres_
    mymap pt_map_;
    //: candidate list polygon for current image
    vgl_polygon<int> poly_;
    //: a expanded sheet vector used to check whether a given points is inside these sheets
    std::vector<vgl_polygon<double> > hull_poly_;
    //: create a expand polygon sheet given a sheet inside the polygon
    bool create_expand_polygon(std::vector<vgl_point_2d<int> > const& sheet);
    //: check whether the pixel is inside the candidate poly
    bool contains(unsigned const& sheet_id, unsigned const& i, unsigned const& j);

};


#endif //volm_candidate_list_h_
