/* =========================================================================
 Copyright (c) 2010-2014, Institute for Microelectronics,
 Institute for Analysis and Scientific Computing,
 TU Wien.
 Portions of this software are copyright by UChicago Argonne, LLC.

 -----------------
 ViennaCL - The Vienna Computing Library
 -----------------

 Project Head:    Karl Rupp                   rupp@iue.tuwien.ac.at

 (A list of authors and contributors can be found in the PDF manual)

 License:         MIT (X11), see file LICENSE in the base directory
 ============================================================================= */

/** \file tests/src/fft_1d.cpp  Tests the one-dimensional FFT routines.
*   \test Tests the one-dimensional FFT routines.
**/

#include <iostream>
#include <vector>
#include <cmath>
#include <complex>
#include <algorithm>

//#define VIENNACL_BUILD_INFO
#include "viennacl/linalg/host_based/fft_operations.hpp"

#ifdef VIENNACL_WITH_OPENCL
#include "viennacl/linalg/opencl/fft_operations.hpp"
#include "viennacl/linalg/opencl/kernels/fft.hpp"
#endif

#ifdef VIENNACL_WITH_CUDA
#include "viennacl/linalg/cuda/fft_operations.hpp"
#endif
#include "viennacl/linalg/fft_operations.hpp"
#include "viennacl/fft.hpp"

typedef float ScalarType;

const ScalarType EPS = ScalarType(0.06f); //use smaller values in double precision

typedef ScalarType (*test_function_ptr)(std::vector<ScalarType>&, std::vector<ScalarType>&,
    unsigned int, unsigned int, unsigned int);
typedef void (*input_function_ptr)(std::vector<ScalarType>&, std::vector<ScalarType>&,
    unsigned int&, unsigned int&, unsigned int&, const std::string&);

struct testData
{
  float input[2048];
  float output[2048];
  float result_multiply[1082];
  unsigned int batch_num;
  unsigned int row_num;
  unsigned int col_num;
};

static testData cufft = { { 0.958304f, 0.461109f, 0.275940f, 0.336814f, 0.867370f, 0.336998f, 0.425562f, 0.901161f, 0.572644f, 0.490057f, 0.995818f, 0.490712f, 0.246360f, 0.936560f, 0.603682f, 0.325047f, 0.782761f, 0.813179f,
    0.613340f, 0.306680f, 0.727441f, 0.560519f, 0.991118f, 0.646667f, 0.489724f, 0.491574f, 0.459562f, 0.186469f, 0.011165f, 0.834209f, 0.807055f, 0.969468f, 0.295318f, 0.082995f, 0.306282f, 0.162688f, 0.419993f,
    0.731844f, 0.063849f, 0.992637f, 0.221901f, 0.059667f, 0.483349f, 0.468260f, 0.996227f, 0.087031f, 0.793308f, 0.778988f, 0.900210f, 0.406648f, 0.085668f, 0.627650f, 0.967166f, 0.076786f, 0.274318f, 0.456890f,
    0.568360f, 0.733880f, 0.643359f, 0.579524f, 0.568088f, 0.450414f, 0.548993f, 0.863406f, 0.533409f, 0.855275f, 0.026094f, 0.953402f, 0.587119f, 0.089943f, 0.946039f, 0.809020f, 0.149611f, 0.429388f, 0.277280f,
    0.145838f, 0.516419f, 0.070588f, 0.924826f, 0.416628f, 0.477235f, 0.010494f, 0.044279f, 0.444401f, 0.087280f, 0.318596f, 0.901292f, 0.655639f, 0.052476f, 0.544651f, 0.235164f, 0.620564f, 0.995065f, 0.784156f,
    0.483970f, 0.528475f, 0.639431f, 0.510064f, 0.481877f, 0.226550f, 0.600007f, 0.427916f, 0.035570f, 0.749618f, 0.857304f, 0.312850f, 0.895456f, 0.373722f, 0.383438f, 0.820281f, 0.790351f, 0.860673f, 0.830775f,
    0.834629f, 0.305074f, 0.918055f, 0.153226f, 0.206366f, 0.573694f, 0.205702f, 0.751017f, 0.808858f, 0.826266f, 0.746083f, 0.593014f, 0.310236f, 0.274557f, 0.232445f, 0.820300f, 0.756434f, 0.458995f, 0.420307f,
    0.184350f, 0.494565f, 0.169925f, 0.041654f, 0.807415f, 0.065381f, 0.415376f, 0.190853f, 0.885662f, 0.205727f, 0.051526f, 0.716437f, 0.040356f, 0.356600f, 0.634492f, 0.193582f, 0.562966f, 0.208186f, 0.399283f,
    0.313984f, 0.017044f, 0.225549f, 0.060066f, 0.610058f, 0.535785f, 0.334624f, 0.842503f, 0.356085f, 0.091058f, 0.301498f, 0.776392f, 0.275408f, 0.796063f, 0.946317f, 0.317062f, 0.603478f, 0.011698f, 0.732438f,
    0.794331f, 0.897360f, 0.938164f, 0.845857f, 0.613797f, 0.978520f, 0.202457f, 0.248289f, 0.172102f, 0.765423f, 0.456475f, 0.571385f, 0.079407f, 0.473519f, 0.796934f, 0.139474f, 0.083577f, 0.332719f, 0.474097f,
    0.926079f, 0.688803f, 0.565155f, 0.227578f, 0.465195f, 0.840563f, 0.023641f, 0.411512f, 0.157625f, 0.627119f, 0.423210f, 0.890063f, 0.421451f, 0.320570f, 0.828227f, 0.267308f, 0.934368f, 0.806747f, 0.469765f,
    0.182657f, 0.978849f, 0.235188f, 0.639132f, 0.550235f, 0.314595f, 0.112651f, 0.347169f, 0.454069f, 0.196227f, 0.679888f, 0.928166f, 0.122307f, 0.368691f, 0.493321f, 0.349885f, 0.833886f, 0.333885f, 0.373526f,
    0.245398f, 0.491509f, 0.000645f, 0.668608f, 0.381572f, 0.422096f, 0.989179f, 0.209799f, 0.689403f, 0.923546f, 0.016546f, 0.159168f, 0.106203f, 0.995395f, 0.394356f, 0.745335f, 0.545630f, 0.708952f, 0.857986f,
    0.892798f, 0.163021f, 0.054213f, 0.572686f, 0.091187f, 0.176520f, 0.941377f, 0.584509f, 0.526405f, 0.775264f, 0.918393f, 0.899930f, 0.020662f, 0.409903f, 0.900576f, 0.689271f, 0.791475f, 0.322671f, 0.678449f,
    0.001273f, 0.012075f, 0.601995f, 0.017819f, 0.171243f, 0.708198f, 0.013214f, 0.565599f, 0.453533f, 0.558844f, 0.274551f, 0.311519f, 0.451643f, 0.437572f, 0.365733f, 0.024329f, 0.528759f, 0.542253f, 0.965706f,
    0.113268f, 0.068658f, 0.740970f, 0.031661f, 0.968588f, 0.761632f, 0.441564f, 0.869164f, 0.450903f, 0.233038f, 0.191835f, 0.129352f, 0.234312f, 0.203910f, 0.731348f, 0.252131f, 0.375153f, 0.439546f, 0.265345f,
    0.940752f, 0.893080f, 0.824190f, 0.215303f, 0.204599f, 0.275832f, 0.652876f, 0.570332f, 0.300161f, 0.181635f, 0.112585f, 0.265867f, 0.294902f, 0.181242f, 0.006837f, 0.326563f, 0.149830f, 0.768470f, 0.768127f,
    0.018994f, 0.219373f, 0.001165f, 0.210829f, 0.348725f, 0.235477f, 0.414739f, 0.080073f, 0.487608f, 0.789891f, 0.519619f, 0.752953f, 0.730644f, 0.412699f, 0.577143f, 0.945947f, 0.617298f, 0.852975f, 0.598823f,
    0.187630f, 0.153136f, 0.780457f, 0.300214f, 0.419003f, 0.075360f, 0.481457f, 0.425841f, 0.401923f, 0.631287f, 0.194311f, 0.170050f, 0.650281f, 0.413683f, 0.171215f, 0.861109f, 0.762409f, 0.406692f, 0.275848f,
    0.842482f, 0.894300f, 0.065739f, 0.362101f, 0.647253f, 0.796383f, 0.774801f, 0.224395f, 0.742330f, 0.392099f, 0.077370f, 0.341152f, 0.579729f, 0.230506f, 0.121610f, 0.879943f, 0.649510f, 0.196969f, 0.361399f,
    0.075350f, 0.598892f, 0.992686f, 0.269661f, 0.768942f, 0.642967f, 0.683344f, 0.940157f, 0.504076f, 0.445753f, 0.346849f, 0.779924f, 0.288235f, 0.241148f, 0.845664f, 0.650336f, 0.888401f, 0.642047f, 0.425137f,
    0.112796f, 0.384377f, 0.817236f, 0.190167f, 0.725529f, 0.396964f, 0.420673f, 0.847139f, 0.276907f, 0.070182f, 0.044108f, 0.638306f, 0.145533f, 0.643000f, 0.630993f, 0.415194f, 0.411943f, 0.273959f, 0.098538f,
    0.352100f, 0.778036f, 0.544291f, 0.698949f, 0.557960f, 0.832526f, 0.940097f, 0.403624f, 0.482862f, 0.828498f, 0.045671f, 0.907999f, 0.941294f, 0.430047f, 0.725235f, 0.131461f, 0.155577f, 0.122199f, 0.552134f,
    0.002715f, 0.399106f, 0.622316f, 0.046824f, 0.037412f, 0.767849f, 0.689824f, 0.668405f, 0.183042f, 0.101767f, 0.942364f, 0.281580f, 0.453866f, 0.720400f, 0.825871f, 0.152815f, 0.278360f, 0.658397f, 0.092912f,
    0.681984f, 0.141259f, 0.921410f, 0.727654f, 0.049258f, 0.862704f, 0.157702f, 0.774493f, 0.994165f, 0.313278f, 0.896691f, 0.546299f, 0.315994f, 0.295797f, 0.168615f, 0.362817f, 0.333209f, 0.936464f, 0.052641f,
    0.001613f, 0.119506f, 0.154408f, 0.943977f, 0.401087f, 0.608274f, 0.664377f, 0.226958f, 0.761089f, 0.942737f, 0.885355f, 0.854001f, 0.624720f, 0.026614f, 0.775411f, 0.352375f, 0.075872f, 0.638115f, 0.510076f,
    0.850365f, 0.632280f, 0.823354f, 0.747056f, 0.178579f, 0.139348f, 0.042853f, 0.347194f, 0.502165f, 0.376062f, 0.283658f, 0.554806f, 0.377675f, 0.403164f, 0.709214f, 0.321652f, 0.804251f, 0.317489f, 0.986029f,
    0.031209f, 0.078578f, 0.928766f, 0.916564f, 0.932579f, 0.553487f, 0.943178f, 0.707990f, 0.905861f, 0.019050f, 0.346105f, 0.415938f, 0.869414f, 0.978385f, 0.239292f, 0.616470f, 0.156964f, 0.378640f, 0.659323f,
    0.504159f, 0.880805f, 0.035385f, 0.787817f, 0.435612f, 0.413060f, 0.190981f, 0.144826f, 0.734712f, 0.995232f, 0.462315f, 0.720742f, 0.026441f, 0.540893f, 0.649508f, 0.943005f, 0.473472f, 0.202994f, 0.886182f,
    0.181462f, 0.108856f, 0.905232f, 0.527568f, 0.524793f, 0.774646f, 0.505953f, 0.764086f, 0.391116f, 0.662917f, 0.142726f, 0.050439f, 0.167076f, 0.023531f, 0.085824f, 0.954893f, 0.459143f, 0.498884f, 0.145874f,
    0.603969f, 0.233596f, 0.141106f, 0.066283f, 0.954338f, 0.167547f, 0.607176f, 0.603846f, 0.110551f, 0.080648f, 0.806840f, 0.996734f, 0.262110f, 0.915696f, 0.901965f, 0.789678f, 0.440489f, 0.676611f, 0.295631f,
    0.204575f, 0.067728f, 0.958548f, 0.347301f, 0.118167f, 0.125624f, 0.370832f, 0.203991f, 0.080517f, 0.829974f, 0.702875f, 0.226390f, 0.433943f, 0.936471f, 0.367496f, 0.500226f, 0.890809f, 0.535043f, 0.107402f,
    0.494654f, 0.645594f, 0.188051f, 0.301494f, 0.642328f, 0.450161f, 0.217190f, 0.544293f, 0.239839f, 0.657679f, 0.220905f, 0.535470f, 0.862254f, 0.288632f, 0.494018f, 0.209555f, 0.406799f, 0.619642f, 0.580386f,
    0.610790f, 0.700159f, 0.410360f, 0.313665f, 0.926549f, 0.844303f, 0.250136f, 0.294045f, 0.344530f, 0.140944f, 0.829088f, 0.451932f, 0.635599f, 0.474683f, 0.639983f, 0.937093f, 0.117011f, 0.090144f, 0.154282f,
    0.661304f, 0.329983f, 0.811962f, 0.882209f, 0.865453f, 0.674215f, 0.170841f, 0.359471f, 0.883770f, 0.577641f, 0.979113f, 0.464156f, 0.188431f, 0.679272f, 0.874516f, 0.502096f, 0.605821f, 0.718820f, 0.752232f,
    0.899866f, 0.063349f, 0.893176f, 0.728954f, 0.515281f, 0.528775f, 0.203637f, 0.155264f, 0.465868f, 0.320648f, 0.245408f, 0.620150f, 0.981952f, 0.575391f, 0.432112f, 0.864160f, 0.440844f, 0.106327f, 0.035002f,
    0.800315f, 0.990097f, 0.612642f, 0.779428f, 0.454253f, 0.801073f, 0.458700f, 0.328769f, 0.303169f, 0.064521f, 0.047589f, 0.055401f, 0.964387f, 0.110938f, 0.948578f, 0.693341f, 0.626220f, 0.477353f, 0.896978f,
    0.781484f, 0.943220f, 0.217626f, 0.026892f, 0.563370f, 0.199578f, 0.602283f, 0.995482f, 0.063738f, 0.043127f, 0.101809f, 0.098740f, 0.843442f, 0.091906f, 0.711382f, 0.622870f, 0.546159f, 0.512455f, 0.081569f,
    0.874928f, 0.815624f, 0.146090f, 0.922517f, 0.871025f, 0.110477f, 0.033456f, 0.819603f, 0.803818f, 0.659676f, 0.296956f, 0.700796f, 0.441160f, 0.240176f, 0.918423f, 0.468052f, 0.803547f, 0.118000f, 0.070335f,
    0.799029f, 0.181738f, 0.113463f, 0.900838f, 0.280478f, 0.956905f, 0.992744f, 0.991860f, 0.579774f, 0.538903f, 0.504315f, 0.661344f, 0.413831f, 0.319939f, 0.807433f, 0.336349f, 0.190965f, 0.917910f, 0.369804f,
    0.010568f, 0.721728f, 0.029480f, 0.307524f, 0.422525f, 0.470640f, 0.547700f, 0.340947f, 0.938692f, 0.351247f, 0.458947f, 0.009027f, 0.150276f, 0.640686f, 0.122490f, 0.051114f, 0.921164f, 0.079395f, 0.043857f,
    0.913024f, 0.659169f, 0.582760f, 0.417339f, 0.320513f, 0.996592f, 0.737279f, 0.127946f, 0.332940f, 0.928244f, 0.045856f, 0.702745f, 0.938812f, 0.767584f, 0.732225f, 0.246335f, 0.190109f, 0.202865f, 0.794036f,
    0.531056f, 0.141557f, 0.145282f, 0.990004f, 0.150584f, 0.295558f, 0.630690f, 0.273074f, 0.346672f, 0.551854f, 0.352469f, 0.390529f, 0.464878f, 0.011638f, 0.973290f, 0.882217f, 0.332150f, 0.969881f, 0.619495f,
    0.460096f, 0.302822f, 0.547739f, 0.505952f, 0.005566f, 0.486551f, 0.273536f, 0.737791f, 0.732886f, 0.463645f, 0.940656f, 0.526922f, 0.994702f, 0.082212f, 0.672204f, 0.984706f, 0.232796f, 0.967762f, 0.615395f,
    0.505870f, 0.314434f, 0.167249f, 0.858339f, 0.704963f, 0.632126f, 0.869977f, 0.678253f, 0.514343f, 0.202127f, 0.648134f, 0.133839f, 0.662223f, 0.950956f, 0.681578f, 0.168175f, 0.956522f, 0.168128f, 0.441711f,
    0.694313f, 0.901015f, 0.905357f, 0.634969f, 0.427936f, 0.900059f, 0.717181f, 0.100141f, 0.884764f, 0.949978f, 0.067903f, 0.500160f, 0.455848f, 0.382337f, 0.667409f, 0.314187f, 0.087300f, 0.299535f, 0.184164f,
    0.765553f, 0.813878f, 0.386291f, 0.413687f, 0.947717f, 0.048513f, 0.364643f, 0.629295f, 0.216688f, 0.321165f, 0.797423f, 0.658400f, 0.015479f, 0.698438f, 0.563757f, 0.650447f, 0.126374f, 0.463815f, 0.367629f,
    0.226515f, 0.348580f, 0.317606f, 0.294418f, 0.848740f, 0.773454f, 0.676755f, 0.516148f, 0.087641f, 0.764055f, 0.815683f, 0.271805f, 0.529608f, 0.629562f, 0.658096f, 0.943296f, 0.577279f, 0.706609f, 0.307939f,
    0.206574f, 0.923298f, 0.629104f, 0.003997f, 0.581698f, 0.644583f, 0.702435f, 0.145454f, 0.295030f, 0.828810f, 0.609270f, 0.662659f, 0.055325f, 0.957850f, 0.980265f, 0.349742f, 0.806589f, 0.753720f, 0.026497f,
    0.322737f, 0.841361f, 0.790552f, 0.138421f, 0.113167f, 0.320160f, 0.767982f, 0.771262f, 0.263456f, 0.345261f, 0.477872f, 0.571395f, 0.551835f, 0.401169f, 0.200499f, 0.555832f, 0.982867f, 0.845082f, 0.258267f,
    0.128321f, 0.140112f, 0.087077f, 0.737591f, 0.802772f, 0.142401f, 0.695441f, 0.783037f, 0.492144f, 0.502030f, 0.536757f, 0.518641f, 0.824767f, 0.378118f, 0.309193f, 0.963188f, 0.491285f, 0.629354f, 0.731170f,
    0.262547f, 0.892810f, 0.076431f, 0.740419f, 0.464205f, 0.628266f, 0.141588f, 0.664704f, 0.184098f, 0.124455f, 0.509786f, 0.442366f, 0.252776f, 0.649899f, 0.529442f, 0.990367f, 0.452670f, 0.671844f, 0.685808f,
    0.235707f, 0.163988f, 0.187838f, 0.772464f, 0.682628f, 0.012605f, 0.150583f, 0.991822f, 0.975793f, 0.641867f, 0.621175f, 0.706963f, 0.904414f, 0.513985f, 0.783395f, 0.644833f, 0.978190f, 0.411661f, 0.786421f,
    0.642894f, 0.595759f, 0.910875f, 0.152681f, 0.038125f, 0.163651f, 0.802580f, 0.567567f, 0.154018f, 0.255250f, 0.239411f, 0.839826f, 0.490957f, 0.403399f, 0.027663f, 0.263422f, 0.086027f, 0.040268f, 0.414004f,
    0.077849f, 0.016061f, 0.055872f, 0.699024f, 0.723024f, 0.960286f, 0.213009f, 0.506419f, 0.605119f, 0.191199f, 0.918080f, 0.391540f, 0.834093f, 0.513840f, 0.302416f, 0.986774f, 0.551965f, 0.466067f, 0.789354f,
    0.119532f, 0.620085f, 0.044604f, 0.358943f, 0.459911f, 0.535561f, 0.762341f, 0.487575f, 0.798983f, 0.848368f, 0.527843f, 0.212987f, 0.926217f, 0.543904f, 0.268859f, 0.625241f, 0.266929f, 0.229145f, 0.838250f,
    0.773348f, 0.834265f, 0.029449f, 0.691428f, 0.225805f, 0.863542f, 0.205268f, 0.528220f, 0.850316f, 0.757233f, 0.994287f, 0.639670f, 0.876765f, 0.614373f, 0.684274f, 0.235708f, 0.074284f, 0.219835f, 0.998049f }

, { 265.477570f, 269.832336f, -6.281282f, 6.641087f, 3.085260f, 1.468255f, 2.062925f, -2.219217f, 1.562588f, -3.774542f, 6.210458f, 0.434553f, 1.473519f, 2.805579f, 4.309592f, -6.777161f, 8.050759f, 9.202207f,
    7.204707f, 1.510179f, -3.346544f, 7.724257f, -6.132596f, -0.804892f, 10.332303f, 1.871409f, 6.655151f, 6.117392f, 0.058095f, 1.776042f, -7.512165f, -5.886488f, -1.495663f, 2.262748f, -5.563531f, 1.066007f,
    6.442278f, 6.128870f, -1.809378f, 8.028464f, 0.756860f, 1.558473f, 1.673254f, -6.895615f, 12.980582f, 0.405215f, 3.323415f, -10.885802f, 2.434934f, -11.920554f, -5.119056f, 10.620436f, 6.239692f, 4.452574f,
    3.156869f, -0.077278f, -2.156140f, -1.040241f, -9.775122f, 2.219220f, -12.363783f, 1.288218f, -14.918893f, 10.101790f, -12.714343f, -3.123183f, -0.255403f, 1.242015f, -4.219859f, -17.194458f, -10.563678f,
    2.555880f, 4.999885f, 3.586536f, -10.149136f, 4.025347f, 0.428679f, 6.697313f, 10.831069f, -0.116510f, 2.438632f, -3.571252f, -3.973325f, -5.350132f, -0.271570f, 1.012172f, 4.074557f, 0.354435f, 2.181108f,
    -4.750814f, -4.397014f, 2.142635f, 2.512309f, -4.840276f, -1.161433f, -0.229460f, -0.299458f, 8.449678f, -2.547060f, -1.783970f, -6.826090f, -0.846037f, 0.670122f, 6.612278f, -1.412156f, -2.586074f, -4.247308f,
    -4.597376f, 9.777460f, -4.189824f, -7.108118f, -8.265704f, -3.751435f, -7.162501f, -13.290262f, -6.848680f, 0.945900f, 1.850814f, -3.624468f, -9.603675f, -0.564678f, -9.611258f, 8.376052f, -6.259708f,
    2.352061f, -5.865232f, 1.854362f, -2.240050f, 4.082589f, -0.556343f, 4.420088f, -1.741045f, -14.269746f, 2.970055f, 1.608672f, -1.208900f, 4.790323f, 11.093012f, -3.646715f, 1.764042f, -7.718101f, 2.059876f,
    11.904912f, -11.603116f, 4.337718f, -11.306424f, 3.660642f, -18.249920f, -7.870895f, 1.343078f, 11.121312f, 2.216587f, 5.987393f, -11.675453f, 4.162141f, 1.912616f, -4.533420f, 6.587620f, -5.524618f, 2.657326f,
    10.622898f, -5.009583f, 1.617495f, -0.525515f, 5.893703f, -5.034750f, -2.382767f, 1.612615f, -9.371872f, 2.275909f, -1.878605f, 9.425722f, 5.232674f, 13.369369f, -9.039780f, -1.054359f, -0.749580f, 0.900475f,
    -0.161160f, 10.457182f, -3.567031f, 0.464825f, -6.887940f, 7.110065f, -0.925568f, -16.104229f, 0.471995f, 15.622788f, 16.791073f, 1.084570f, 4.955261f, -8.817974f, -7.960642f, -0.091447f, -5.776711f,
    -5.647820f, 6.576977f, 8.536584f, 7.660462f, 0.310385f, -5.664694f, -10.246677f, 8.412231f, -6.491307f, 8.721404f, -6.858874f, 4.924000f, 3.159826f, -4.125678f, -10.793512f, -5.338314f, 13.494216f, 0.745811f,
    -8.351656f, -8.713416f, 13.407349f, 6.728643f, 5.501867f, 9.015443f, -1.355654f, -1.802503f, 2.711152f, 0.538907f, 2.499354f, 2.797177f, 2.083851f, -18.628952f, 4.641853f, -8.573132f, -13.695271f, 5.413444f,
    17.726368f, 1.950253f, 1.199775f, -4.107238f, 5.644404f, -4.926291f, -1.029564f, 3.401692f, 8.401718f, -1.034914f, -10.340586f, 3.595437f, -1.154990f, -2.022038f, -5.326516f, -3.312632f, 6.429633f, 12.320015f,
    1.882305f, 2.054628f, -3.387951f, 6.073052f, -3.242687f, 11.003525f, -16.639301f, 3.781174f, -5.578990f, 2.937887f, -14.167067f, -7.436025f, 7.093699f, -11.742753f, 6.827620f, 2.792532f, -9.028412f, 1.830292f,
    -3.971087f, -12.250819f, 10.465918f, -1.506019f, 1.784756f, 2.388722f, 12.569431f, -8.722069f, -2.483387f, -7.659946f, 3.766811f, 14.462860f, 12.776262f, -3.837689f, 2.085792f, 1.504001f, -6.520282f,
    -1.497645f, 7.017129f, -7.610742f, 3.933441f, -13.370028f, -1.317183f, 2.969180f, -6.065273f, 8.595788f, -10.468719f, -5.199576f, -9.632074f, -6.260214f, 0.702841f, -7.144930f, 1.083202f, -0.883020f, 8.446010f,
    7.601700f, 10.909828f, -2.010260f, 1.885705f, 8.600731f, -1.400577f, -3.471998f, 7.790930f, 3.981818f, -17.912056f, 4.267098f, -13.615426f, 3.924356f, -1.294306f, 10.221536f, -2.901402f, -7.499428f, 15.400040f,
    -4.611712f, -14.926388f, -5.235963f, -1.453979f, 5.220313f, 9.470819f, -3.829745f, -3.853885f, -0.788608f, 0.735015f, 9.340893f, -4.756156f, -6.862822f, -1.345537f, 2.914357f, -5.555783f, -2.692354f, 8.617622f,
    -10.697104f, 5.892055f, -5.553278f, -4.795091f, 2.440234f, -2.453033f, 1.435685f, 11.711970f, -4.916733f, 3.158844f, -1.219957f, 2.205355f, 3.384010f, -5.187429f, -4.046430f, 3.098966f, 6.820691f, 7.405268f,
    -0.561806f, 6.074167f, -17.312483f, 11.065754f, -7.499033f, -5.045103f, -4.386439f, -13.929318f, -3.763134f, 23.476391f, -4.654320f, 6.644573f, 13.334518f, 2.323262f, 5.986541f, 1.038785f, 0.239536f, 2.581632f,
    -9.944656f, 7.397114f, 2.688263f, 3.492923f, 5.850791f, -6.295029f, -4.654384f, -0.805569f, -11.567688f, 2.664076f, 7.328889f, 0.371476f, 1.737476f, 7.815259f, -8.381383f, 0.258981f, -12.501060f, -13.630202f,
    -6.179264f, -4.475775f, -0.155309f, 4.546054f, -1.753647f, -1.123977f, 5.034809f, 14.970467f, 14.184572f, 4.749614f, -5.818598f, -3.182054f, -0.404507f, 3.913354f, -0.227936f, -3.070438f, 12.106720f, 2.122710f,
    7.091056f, -5.634785f, 5.198673f, 12.353919f, 1.250807f, -8.201432f, 11.753622f, -6.412996f, -6.253985f, 8.011473f, 2.156845f, 5.885045f, 5.365874f, 1.816244f, -5.474942f, 4.105309f, 5.850098f, 2.473696f,
    5.011164f, -1.526222f, 8.342945f, 5.315174f, 7.416545f, -2.822525f, 0.346685f, -2.924274f, 1.112553f, -0.503677f, 0.516686f, 4.851023f, 9.122018f, 1.172228f, -3.376639f, -6.246335f, -0.426485f, 1.845142f,
    -2.632698f, 11.501021f, -0.917811f, -8.319624f, 3.211722f, 4.358172f, 3.193742f, -14.062444f, -5.221861f, 13.642365f, -7.814400f, -6.491995f, 4.156569f, -2.149630f, -2.843954f, -16.651978f, 7.619571f,
    0.810895f, -4.320256f, 7.998853f, -1.880560f, -3.041227f, -4.070188f, -11.805959f, 10.856344f, 4.191437f, -9.150784f, 2.602691f, 4.226574f, 0.046750f, 5.757174f, 5.328087f, -3.412838f, -12.486424f, 6.758978f,
    -1.470311f, 8.924647f, -12.747158f, 2.585845f, 5.419098f, -31.219212f, -1.047764f, 6.095867f, -9.825105f, 7.301771f, 1.553394f, 6.000603f, -1.302688f, 13.775797f, -0.649277f, 5.028845f, -2.596515f, 0.491332f,
    12.632045f, 2.790866f, -7.891503f, -5.451321f, -3.392651f, -9.673877f, -3.290911f, 13.527504f, -11.143648f, 3.892815f, -1.893179f, -3.406136f, -7.291657f, -0.434956f, -6.295663f, 1.377115f, 2.899463f,
    -6.847254f, -9.183233f, 10.961942f, 6.106332f, -4.477934f, 5.992685f, -2.653695f, 15.539237f, 10.964013f, 4.901660f, -14.975603f, 9.104028f, -5.906822f, -3.241549f, 3.361080f, -6.894635f, 6.491853f, 5.214055f,
    0.340057f, 2.177623f, 1.056908f, -2.923114f, 4.674820f, -3.936303f, -0.633176f, 7.579105f, -3.919437f, -5.723557f, -3.880847f, -17.357109f, -3.414813f, -4.291742f, 6.680182f, 0.958921f, -6.038884f, 0.704217f,
    -10.186163f, 4.163349f, 1.357400f, -10.409632f, -10.184418f, 6.144989f, 5.110682f, 0.341639f, 11.131057f, -2.737936f, 3.913830f, 5.665910f, -4.887468f, 5.003384f, 6.047396f, 1.503230f, 3.353575f, 11.895478f,
    -6.220557f, -1.031719f, -4.302063f, 5.577667f, -4.735453f, -2.292084f, 8.972246f, 7.935566f, 5.568385f, -7.007401f, -6.149975f, -10.354392f, 9.258462f, -3.268816f, 3.613171f, 2.616825f, 16.003550f, -8.214693f,
    7.831873f, 0.995927f, 10.503004f, -5.655463f, 3.644843f, -4.465456f, -8.002244f, -2.565113f, -2.600720f, -5.484006f, -0.363766f, 0.070986f, -5.594088f, -10.072073f, -1.774687f, -10.519798f, -2.028671f,
    -4.186376f, 0.303182f, -4.194547f, 0.270646f, -0.882947f, -2.831423f, -4.126069f, 7.186472f, -9.687106f, 3.371431f, 3.556659f, 7.810393f, -7.622321f, -7.306807f, -2.662662f, -2.740419f, 4.787480f, 8.398676f,
    6.113845f, 1.488211f, -3.844419f, 7.639075f, 2.676285f, 1.182032f, 0.884404f, -9.074320f, -7.663038f, 3.937954f, 2.262076f, 7.295219f, -16.556778f, 12.949368f, 0.563417f, 14.295797f, -1.404750f, -3.194478f,
    4.275634f, -2.468198f, 0.510086f, -12.130469f, 3.762059f, 2.322736f, -10.868699f, 9.053523f, 6.708532f, 12.808792f, -7.033282f, -0.608936f, 4.410141f, -4.141525f, 4.666715f, -3.000190f, 7.712261f, -2.209930f,
    1.093906f, 4.038775f, 8.606831f, 5.001119f, -15.903023f, -3.019452f, -1.279128f, 6.380682f, 4.203551f, 4.478116f, 4.190514f, 6.592153f, -2.888382f, -5.868249f, -7.497109f, 0.929182f, -3.906974f, 4.348999f,
    -0.347211f, 8.671771f, -2.623824f, 0.155382f, -7.474522f, 5.543261f, 5.935426f, 5.886332f, 7.671038f, -7.461238f, -3.193762f, -11.434480f, -5.872171f, -1.772791f, 16.444311f, -9.685337f, -0.277690f, -0.501585f,
    -10.675951f, -4.861526f, 0.841680f, 5.829590f, -7.265580f, -12.346306f, -0.249522f, 1.142665f, 3.487215f, 7.083692f, -0.138202f, 8.132135f, -15.994522f, 0.304689f, 5.225499f, 12.283710f, 4.544263f, 12.301310f,
    -1.083265f, 0.378518f, 6.743744f, -4.522845f, -10.498566f, -3.964694f, 5.075862f, -1.209075f, -1.787682f, -7.453143f, 6.949930f, 3.735006f, 13.962359f, 1.928422f, -10.875157f, 3.627008f, 1.026874f, 2.640098f,
    -1.046755f, 3.916083f, -8.637132f, 8.298273f, 1.417309f, -5.951509f, -1.118170f, 1.175777f, -9.153865f, -0.550695f, 3.265286f, 1.415947f, -0.756589f, -2.151538f, 0.700658f, 11.877617f, -2.858113f, 4.637248f,
    1.053962f, 7.946632f, 1.835391f, 6.006594f, 3.047868f, -0.471520f, -2.714739f, 5.087853f, -8.113625f, -8.125319f, -3.303083f, -0.938537f, 3.727656f, 1.331126f, -1.707616f, -7.306983f, 13.670181f, -8.116289f,
    -9.757053f, 4.673117f, 6.393153f, 1.799502f, 4.252683f, -17.653854f, 2.420948f, -0.723117f, 10.039388f, 5.518513f, 2.108525f, 10.296576f, -2.387065f, -0.399110f, 8.740544f, -6.839513f, 2.546276f, 8.157343f,
    2.318166f, -3.482059f, -10.623308f, 4.874808f, 12.790012f, -8.910692f, -7.046453f, 13.941223f, 1.587916f, 11.420963f, 2.959938f, -5.464751f, 9.321755f, 14.676608f, -1.246317f, -0.927082f, 3.381493f, 3.240629f,
    3.559814f, 1.974107f, -1.010994f, 8.360539f, 5.746264f, 2.605454f, -5.423461f, 5.785841f, -0.774929f, -2.581969f, -8.543077f, 8.406773f, 9.918266f, -6.216476f, -3.095661f, 6.168717f, -2.091675f, -2.742159f,
    -13.083153f, 4.522453f, -6.669912f, -0.631084f, 1.620041f, -2.083591f, 0.284647f, -0.090018f, -1.208263f, 11.204156f, 3.373402f, 6.731969f, 8.679746f, 8.748439f, -5.438663f, 0.466950f, -16.815813f, -6.352481f,
    -7.468130f, -3.543617f, 6.347380f, 5.306674f, -2.543913f, -5.140027f, 18.110134f, 7.471231f, -3.369584f, -8.125237f, 5.303941f, -2.925134f, 15.064285f, 5.908778f, 0.636853f, -3.660820f, -0.348082f, 3.415162f,
    2.244710f, 9.080246f, 7.254852f, 3.709892f, -2.594475f, 7.926545f, 0.981707f, -2.477806f, 9.948220f, -10.027470f, 3.807550f, 10.708315f, 3.617714f, 1.767245f, -5.184971f, 1.901516f, -4.621387f, 5.659268f,
    4.934263f, 1.754709f, 5.811354f, 1.234453f, -5.173310f, 11.304674f, -1.408020f, 7.570985f, 1.345029f, -3.183870f, 12.314240f, -3.696803f, 5.861785f, 3.403288f, 7.776047f, 5.046828f, 10.318241f, -10.058457f,
    2.794643f, -1.061254f, -11.462266f, -2.510397f, 3.270353f, -6.299072f, 5.659431f, -16.603630f, -15.604513f, 2.949822f, 5.908284f, -5.418210f, -10.451139f, 10.080765f, 5.056001f, -1.353296f, -4.699068f,
    -12.561559f, -5.754490f, -2.471263f, 0.208467f, -7.408519f, 11.554119f, -1.388891f, -5.601897f, -2.659493f, 1.580809f, 0.322458f, -4.299407f, 1.154414f, -0.764665f, -1.145788f, 1.655733f, 1.426026f, -4.156519f,
    -3.944903f, -3.024625f, 2.491175f, 6.715263f, -7.088682f, 1.401598f, 4.549637f, 5.069774f, -4.393368f, -9.946341f, 5.109366f, 1.282338f, 6.385411f, 3.256195f, 0.534509f, -1.537598f, -1.677263f, -1.773552f,
    15.278774f, 8.816522f, 5.903562f, -1.866181f, -10.501234f, -8.962178f, 0.945561f, 1.597136f, -11.574831f, 5.227357f, 8.923567f, 1.936574f, -5.744070f, 2.136705f, -2.614378f, -0.282759f, -2.434345f, -5.261054f,
    -0.482948f, 8.388806f, -3.460652f, -9.370162f, 3.458206f, -0.777334f, 5.153907f, -6.082856f, 0.237597f, -8.613085f, 7.132904f, 7.697973f, -6.099593f, -5.066557f, -0.069035f, 5.150362f, -6.758209f, 11.261834f,
    -4.868246f, -7.708077f, 11.477669f, -0.151071f, 7.849120f, -1.503046f, -11.457049f, 7.445291f, 11.615274f, -6.274791f, 8.157793f, 3.690086f, 7.467462f, -21.036251f, 5.465898f, 3.441448f, -0.353419f, 4.974216f,
    3.295279f, -10.771730f, -11.184229f, -0.892533f, -1.759250f, 0.037325f, -1.171556f, 5.174034f, -6.533063f, -6.249487f, -4.895426f, 13.560618f, -3.513312f, -9.578554f, -5.341642f, 4.542255f, 12.050463f,
    -0.747673f, -9.440533f, 4.305959f, -0.205878f, -4.718102f, 3.672420f, -3.581017f, 5.125288f, 0.004067f, 0.830603f, 2.777692f, -7.051358f, -4.132073f, 8.430955f, 9.535617f, 0.537199f, 12.308285f, -5.285636f,
    -6.372055f, -6.904351f, 7.345806f, -7.216985f, 2.346654f, -7.386206f, 3.097062f, -3.024791f, 1.134970f, -1.140893f, 11.343332f, 6.804462f, -2.135912f, -2.440479f, -3.396046f, 3.287263f, -17.274080f, 1.932364f,
    -3.286715f, -0.789017f, -7.885235f, 1.039572f, 6.475173f, 1.453122f, 5.971737f, 2.896573f, 3.617877f, 5.185029f, 3.208504f, 3.945697f, 5.349731f, 5.185720f, -7.324204f, -9.754431f, -2.823229f, 2.253184f,
    1.543716f, 7.993743f, 7.687342f, 3.734392f, -2.223834f, -1.891607f, 3.539880f, 6.230686f, 3.998879f, 12.081067f, 0.269139f }, { 129.986f, 380.996f, -3.97007f, -0.283082f, 2.18126f, 2.31325f, 2.87777f, 0.914613f,
    2.74455f, -1.39571f, 5.97125f, 3.48028f, -2.26458f, 2.07122f, 4.80452f, -2.69043f, -1.18122f, 13.7498f, 3.95579f, 3.13579f, -6.76401f, 3.74314f, -5.55763f, -4.76349f, 4.14004f, 5.99557f, 1.91775f, 4.0523f,
    -1.48094f, 0.0682929f, -0.355969f, -12.0335f, -0.629493f, 0.544098f, -1.87744f, -0.578621f, -1.77967f, 7.28883f, -8.08488f, -1.28345f, 0.0749586f, 0.390986f, 4.03771f, -2.54947f, 12.8963f, 1.5334f, 11.1164f,
    -6.04689f, 7.03942f, -9.74084f, -7.10446f, -2.30314f, 5.69292f, 4.7855f, 0.901294f, 1.42114f, -0.462052f, -2.17358f, -7.575f, -4.23716f, -7.60395f, -4.837f, -16.9123f, -7.33525f, -4.11076f, -12.5402f, -1.1908f,
    -0.211093f, -0.931038f, -10.4747f, -12.0614f, -6.12826f, -0.791978f, 2.68348f, -3.4012f, -0.363981f, -0.251372f, 3.48888f, 10.0654f, 4.40478f, 1.20128f, -1.67874f, 2.20167f, -2.00265f, -0.346177f, 0.00182125f,
    3.43998f, 2.99089f, 2.70199f, 0.938639f, -2.36366f, -2.22476f, 6.29544f, -2.84635f, -0.440835f, -0.72484f, -4.50136f, 5.25024f, -0.823211f, -1.43669f, -3.73367f, -3.42862f, -4.93285f, 0.737534f, -0.401594f,
    -2.65884f, -2.08514f, -5.70406f, 7.18588f, 6.41373f, 1.49616f, -12.6506f, 2.86143f, -9.08148f, 2.23295f, -14.2905f, -0.237009f, 0.478794f, -0.10384f, -6.25513f, 7.35006f, -7.67496f, 11.5911f, 1.07705f, 3.21441f,
    -2.74847f, 1.02982f, -0.183984f, 3.76978f, 2.63184f, 2.76057f, 1.05866f, -4.09951f, -6.50979f, 0.323709f, -0.138415f, 3.14251f, 9.26986f, -1.85143f, 0.0367542f, -7.2594f, 0.236532f, 8.92631f, 7.93126f, 4.20692f,
    1.09055f, 5.8555f, -10.8708f, -4.71066f, -0.882503f, 3.74458f, 4.37696f, 2.73544f, 1.15145f, -0.916804f, 2.65403f, -4.63331f, 2.01256f, -5.60074f, 0.271571f, 2.47768f, 2.74662f, 1.40054f, 0.0374654f, 9.45623f,
    1.56933f, -1.72866f, -0.926649f, -1.77659f, -6.83769f, -9.9505f, 5.80136f, -6.39947f, 16.9688f, -4.51688f, -9.49277f, -0.375336f, -0.00380501f, -8.0319f, 1.67635f, -1.89385f, -1.82597f, -3.9137f, -2.69698f,
    1.5085f, -12.9631f, -5.15855f, 1.46275f, 6.9562f, 16.0641f, 8.39672f, -3.27336f, -1.76913f, -3.72406f, -4.72217f, -4.88392f, 1.36093f, 4.5496f, 4.67266f, 3.43663f, -0.723462f, -11.5076f, 8.07298f, 4.88632f,
    8.74001f, 6.31557f, 2.48805f, 4.8623f, 9.81164f, -6.00993f, -9.88009f, -0.23821f, 3.03776f, -4.36074f, -5.63619f, -1.51468f, 1.97565f, 3.81857f, 7.38776f, 7.44613f, -1.22004f, -0.332974f, -0.608632f, 1.42154f,
    1.63676f, 2.67163f, -8.0975f, -2.83765f, -4.20494f, -6.73688f, -3.14441f, 13.9176f, -0.363598f, 2.43557f, -4.75296f, -1.64735f, -4.53262f, -1.03236f, -0.350847f, 1.69855f, 3.04772f, -10.7011f, 3.31f, 1.10092f,
    3.13655f, -5.51112f, -4.00568f, 5.20034f, -0.410065f, 7.15755f, 0.785396f, 0.0537459f, 7.61241f, 0.497163f, 18.6922f, -0.228374f, 8.4933f, -1.72091f, 5.86783f, 0.911529f, -11.5862f, 1.26298f, -11.4972f, 1.61484f,
    1.90608f, -6.12176f, 2.41268f, 1.05388f, -2.01051f, -1.91137f, -1.09014f, 1.24406f, -4.34959f, 8.19262f, -4.19246f, -3.78248f, -4.08747f, -2.28613f, 1.65584f, 10.8801f, -1.19625f, -1.97847f, 7.11222f, -2.08322f,
    -0.651417f, 0.691991f, -5.76387f, 2.6736f, -11.9468f, -11.4588f, 6.5828f, -0.0975016f, 6.31548f, -2.71723f, 0.248467f, -2.52034f, -1.61016f, -1.11184f, -5.49854f, -1.00926f, -4.04368f, 2.78042f, -8.24637f,
    10.0462f, -3.3495f, 0.0272491f, 2.13832f, 1.45815f, -6.0442f, -0.117796f, 7.64746f, -9.02063f, 2.30795f, -1.99263f, 1.42505f, 0.813187f, 1.87241f, -0.455971f, -4.75642f, 3.90544f, 7.9214f, -15.0129f, 0.219512f,
    -1.17625f, -1.99064f, 1.11163f, -0.428027f, -2.24576f, -0.385921f, 0.241693f, 8.31154f, 5.05915f, -2.55293f, -5.86655f, 4.42222f, -2.85655f, -9.70569f, 2.42678f, -11.6291f, -5.48721f, -2.42573f, -3.91337f,
    2.28817f, 1.52885f, -4.47634f, 4.11765f, -1.89137f, -2.12914f, -1.40589f, 0.448802f, 3.14426f, -2.61721f, -2.70329f, -2.10434f, 1.55371f, 4.23124f, -5.11478f, 4.80219f, -10.0933f, -0.27526f, -1.80596f, -10.9568f,
    4.75546f, -2.50403f, -21.1319f, 12.1983f, -5.09718f, 4.10382f, 8.98766f, 6.95308f, 0.108795f, 2.12269f, -0.456216f, 1.55186f, -7.71841f, -7.85117f, 1.05806f, 2.79819f, 2.5888f, -1.83416f, -1.9878f, -5.10279f,
    -5.16787f, -8.17648f, 4.45839f, 5.247f, -2.30598f, 8.22339f, -3.82585f, -2.79163f, -5.82117f, -14.2338f, 2.29488f, -6.30491f, -4.13972f, 2.81849f, -0.64808f, -1.46719f, -5.1864f, 3.62387f, 10.6889f, 6.57899f,
    -2.9584f, -4.61845f, -3.48532f, 1.30357f, 0.152372f, -0.866223f, -0.820935f, 7.82142f, 4.65515f, 3.7395f, -1.84895f, 9.95369f, 2.76212f, -3.03585f, 3.41619f, 3.50653f, -9.2264f, 2.82923f, -1.77609f, 5.31678f,
    2.75978f, 6.55651f, -4.19212f, -0.98664f, 4.73382f, 2.31663f, 5.98676f, 3.33117f, -0.266892f, 8.33637f, 1.41411f, 0.782792f, 1.65696f, -0.165927f, 0.204041f, 0.442659f, 0.0943977f, 3.04306f, -0.558821f, 7.04819f,
    1.84579f, -6.56583f, -0.265839f, 0.294336f, -5.71942f, 10.0968f, 5.57689f, -4.43719f, 1.98647f, 4.09009f, 10.1477f, -1.81167f, -9.78905f, -2.29369f, 4.87793f, -8.11732f, 3.13043f, -1.35944f, 0.17256f, -14.8142f,
    5.09514f, 8.20314f, -8.52594f, -1.36807f, -0.0663385f, -2.25566f, 0.786712f, -4.17846f, 2.54224f, 5.13816f, -8.70639f, 1.95562f, 0.00123056f, 0.505176f, -4.14064f, 6.25734f, 6.22632f, -7.08408f, 4.82421f,
    0.557163f, 18.8097f, -1.28813f, -2.33852f, 7.00614f, -19.4754f, -1.48543f, 8.18892f, -5.47046f, -0.437244f, 4.77723f, 4.16852f, 4.43823f, 9.24475f, 10.9318f, 4.22051f, -1.0417f, -0.472855f, 1.78131f, 4.93181f,
    -1.33841f, -1.08768f, -2.82216f, -4.12423f, -5.4794f, 13.357f, 5.10118f, 2.77472f, 2.52186f, 6.10837f, -5.67357f, 0.481126f, -0.230659f, -1.37853f, 3.95514f, -1.3028f, -12.354f, 6.01584f, 13.5203f, -4.17055f,
    5.34324f, -7.38182f, 4.27444f, 4.73656f, 14.9886f, -9.1959f, -7.05349f, 0.300222f, -2.74537f, 5.69203f, -2.85127f, 5.53356f, 4.82228f, -0.680696f, 1.8637f, 0.994826f, -1.00557f, 3.56908f, 2.86457f, -4.13409f,
    7.25024f, -2.67357f, -4.22884f, 9.17446f, -11.909f, -1.18817f, -5.66395f, 0.506258f, 6.11451f, -1.17249f, -0.52958f, -11.4173f, -1.6051f, 8.77613f, -4.4114f, -9.84814f, -4.6727f, 1.77239f, 3.52158f, 1.72679f,
    0.170665f, 0.520583f, 1.03873f, -5.19716f, -4.2376f, 2.02668f, 3.70715f, -6.6953f, 3.7607f, -1.30752f, -1.11876f, -5.60813f, -3.73592f, 0.598287f, -3.25929f, 4.54057f, 5.78375f, 6.10293f, 3.92766f, -3.4159f,
    -11.9325f, 11.4263f, 5.35757f, 1.70056f, 3.65801f, 13.2567f, -0.827006f, 1.53475f, 0.734179f, 12.0318f, -1.77333f, 0.991669f, -0.0697898f, -2.44423f, -2.58361f, 4.34218f, -2.60009f, -0.271753f, -0.0324587f,
    7.00469f, -9.60941f, 4.61009f, -4.75373f, 0.432733f, -4.81469f, 2.10741f, -0.300533f, 0.340767f, -0.51913f, 1.79663f, -3.06269f, 5.33901f, -2.79993f, 0.982021f, 2.74447f, 6.82054f, -3.28769f, -1.61668f, -7.7261f,
    -3.15607f, 0.0280036f, -0.727121f, 4.69776f, 3.1534f, -1.51843f, 2.79205f, 6.98322f, 0.207652f, 0.733686f, -1.93788f, -14.7617f, 0.319872f, 1.72376f, 4.847f, -4.67609f, 10.4815f, 6.31936f, 9.75321f, 5.89311f,
    -6.05108f, -0.25719f, -0.334788f, -0.162808f, -4.35938f, -7.44151f, 9.59143f, -1.70051f, 2.18118f, 13.7537f, 9.83745f, -2.55368f, -4.11645f, 1.04716f, -6.96156f, -1.35933f, -2.84579f, 3.01436f, -2.45778f,
    -1.18956f, -3.18635f, 6.76823f, 15.5577f, -7.66941f, -2.63607f, -1.34232f, 2.63488f, 8.40573f, 0.0916492f, 4.52721f, 1.79087f, 0.435343f, -0.329896f, -5.37431f, 2.65094f, -0.38257f, 4.47029f, 2.16143f, 6.01458f,
    6.36001f, 0.863243f, -3.27858f, -4.55619f, 4.64411f, 1.12844f, 11.2013f, -4.36472f, -5.8786f, -6.46629f, -9.94903f, -5.56824f, 4.86892f, -0.611693f, -0.478832f, 10.268f, -1.07518f, -1.33773f, -4.51816f, 8.59175f,
    -1.38692f, -5.66973f, -11.1935f, -2.39624f, 3.80299f, 1.54531f, 0.160418f, 7.77356f, -7.38784f, -5.01838f, 3.45054f, 0.586959f, 0.819402f, 1.35935f, 1.10435f, -0.300533f, 5.72274f, 3.32177f, -10.2856f, -4.7665f,
    0.740501f, 1.46547f, -1.20367f, -7.09428f, 4.5797f, -8.71596f, 16.1338f, 0.576886f, -1.13694f, 2.14729f, 3.75708f, 2.05245f, 0.0934739f, 6.55473f, -4.32525f, 0.691357f, 7.96173f, -1.88711f, -5.30568f, 0.782579f,
    -0.997458f, -1.03345f, 2.50898f, 0.842222f, 1.18969f, -1.27392f, -1.8623f, 14.6263f, 8.94356f, 2.12057f, 3.11009f, 2.79378f, 6.18107f, 1.51058f, 3.18305f, 0.532379f, -2.35057f, 8.41918f, 3.12077f, -2.96987f,
    -1.30736f, -0.78726f, 2.66269f, 1.13086f, 0.0373011f, -10.9261f, 2.4317f, 6.39164f, -10.9453f, -1.2927f, 4.39029f, -0.622832f, 0.308811f, -11.6071f, -0.611353f, -9.28488f, -0.152956f, 0.345669f, 0.409432f,
    10.9745f, 4.60774f, -3.88036f, 4.92708f, -4.72975f, -6.00009f, 5.71764f, 2.75283f, 8.7017f, -6.76912f, -8.76458f, 4.01225f, -2.95672f, -13.455f, 9.81695f, 4.59692f, 1.57076f, 2.87962f, -9.28958f, 4.49972f,
    2.25864f, 1.95582f, -1.42701f, 3.20809f, -1.28735f, 3.09596f, 0.889561f, 0.408291f, 2.58842f, 6.11793f, 3.53875f, -0.906801f, 0.821566f, 5.62228f, 0.559726f, -8.39445f, 2.00925f, 14.8275f, -1.92274f, -3.30679f,
    4.43713f, 1.97538f, 6.35036f, -1.40702f, 6.15805f, 1.51216f, -1.21364f, 0.894706f, -2.10993f, -0.830135f, 0.00979264f, -1.20926f, 4.20967f, 13.3004f, -6.83275f, 8.53555f, 8.135f, 1.07865f, 2.95925f, -5.20937f,
    -0.187827f, -10.8885f, -7.76209f, 0.929479f, 4.90771f, 1.00403f, -12.7767f, 0.329121f, 3.23136f, 4.49664f, -11.3418f, -0.494169f, -14.9013f, -0.264519f, 0.712126f, 2.71705f, -2.22813f, -3.54013f, 1.66662f,
    4.20079f, -2.64403f, 11.2774f, 2.92048f, -1.4892f, 6.08052f, 8.39862f, -5.14395f, -0.563785f, -6.02677f, -2.09821f, 6.01019f, 5.77891f, 1.70736f, 0.0767038f, 3.88811f, 0.604618f, 2.69989f, 6.20201f, -4.78162f,
    4.06705f, 1.9463f, 0.199162f, 7.41908f, 1.56353f, 1.35898f, 6.46925f, -2.28687f, 8.05841f, -5.88661f, 2.00999f, 1.23097f, 5.48799f, -1.45249f, 6.64112f, -3.25255f, -2.87315f, 3.03764f, -3.95294f, -4.66014f,
    0.834003f, -7.18403f, 0.578795f, 10.4675f, -14.0537f, 0.800218f, 5.62106f, 3.71011f, -8.94611f, 1.86481f, 12.8365f, 2.53917f, -3.66892f, -2.67947f, -4.36692f, -2.41286f, -1.3622f, -6.75062f, -4.26334f, 3.03971f,
    -4.58649f, -0.85322f, -0.554695f, 2.88676f, -3.36693f, 0.807288f, -0.442844f, -2.72055f, 0.462768f, 3.85134f, -0.303492f, -2.89321f, -2.38425f, -4.84597f, 4.26324f, -5.79798f, 0.126814f, -1.10827f, 2.03034f,
    4.29721f, -11.027f, 0.903352f, 2.1019f, 1.19084f, 5.20464f, 0.911797f, -0.634072f, 0.649507f, -1.28787f, 7.56632f, 21.5773f, 1.76417f, 0.275578f, -0.690949f, -2.17012f, -0.584699f, 1.9371f, -5.28359f, -7.30523f,
    6.03441f, 5.90809f, -4.03059f, -2.01048f, -1.12271f, -2.3029f, 0.706211f, -2.74198f, -4.58646f, 7.84273f, 4.67321f, -8.42747f, 1.60195f, 2.88343f, 4.89778f, 3.35113f, 5.5216f, -3.84896f, -4.10694f, 5.83121f,
    -0.492365f, -1.69187f, -2.31354f, 2.59504f, -9.02737f, -1.54543f, 5.05637f, -8.90233f, 5.29709f, 7.64282f, 5.73727f, 0.819292f, -3.27733f, -0.931131f, 13.2557f, 3.08186f, -0.452835f, 1.27494f, 27.9334f,
    -13.5775f, 1.37064f, 5.60423f, -4.7486f, 3.19695f, 10.1322f, -2.955f, -6.33889f, -11.5158f, -0.753568f, -1.36815f, -3.83566f, 2.62839f, -4.99663f, -6.68998f, -2.40585f, -0.284143f, 2.61676f, -9.6816f, -1.98212f,
    -0.663865f, 3.51293f, 9.94129f, -6.37191f, -1.69426f, 1.23716f, -0.18475f, 0.460128f, -0.160183f, 2.12157f, 0.400682f, -0.141855f, 0.09102f, -1.94148f, -7.98672f, 6.06496f, 10.9528f, -7.17593f, 6.55822f,
    4.83945f, -6.07097f, -8.83041f, -2.88269f, -4.41804f, -0.976727f, -8.99799f, -1.02083f, -2.30565f, -1.85866f, -7.1702f, 0.648441f, 1.07018f, 2.34714f, 0.696387f, -2.8689f, 10.9284f, -11.566f, 4.33227f,
    -0.986674f, 1.26298f, -4.33022f, -2.559f, 6.56284f, -3.34309f, 2.51411f, -0.0558391f, 1.62945f, 1.86506f, 6.69936f, 3.13421f, 4.57929f, 5.23939f, -3.8932f, -7.84384f, -4.44025f, -0.12247f, 2.73134f, -1.5903f,
    13.7692f, 4.33856f, 1.85166f, -3.5844f, 0.880429f, 1.17157f, 1.40541f, 2.38723f, 12.1167f }, 1, 1, 541 };

static testData real_to_complex_data = { { 0.786421f, 0.642894f, 0.595759f, 0.910875f, 0.152681f, 0.038125f, 0.163651f, 0.802580f, 0.567567f, 0.154018f, 0.255250f, 0.239411f, 0.839826f, 0.490957f, 0.403399f, 0.027663f,
    0.263422f, 0.086027f, 0.040268f, 0.414004f, 0.077849f, 0.016061f, 0.055872f, 0.699024f, 0.723024f, 0.960286f, 0.213009f, 0.506419f, 0.605119f, 0.191199f, 0.918080f, 0.391540f, 0.834093f, 0.513840f, 0.302416f,
    0.986774f, 0.551965f, 0.466067f, 0.789354f, 0.119532f, 0.620085f, 0.044604f, 0.358943f, 0.459911f, 0.535561f, 0.762341f, 0.487575f, 0.798983f, 0.848368f, 0.527843f, 0.212987f, 0.926217f, 0.543904f, 0.268859f,
    0.625241f, 0.266929f, 0.229145f, 0.838250f, 0.773348f, 0.834265f, 0.029449f, 0.691428f, 0.225805f, 0.863542f, 0.205268f, 0.528220f, 0.850316f, 0.757233f, 0.994287f, 0.639670f, 0.876765f, 0.614373f, 0.684274f,
    0.235708f, 0.074284f, 0.219835f }

, { 0.786421f, 0, 0.642894f, 0, 0.595759f, 0, 0.910875f, 0, 0.152681f, 0, 0.038125f, 0, 0.163651f, 0, 0.80258f, 0, 0.567567f, 0, 0.154018f, 0, 0.25525f, 0, 0.239411f, 0, 0.839826f, 0, 0.490957f, 0, 0.403399f, 0,
    0.027663f, 0, 0.263422f, 0, 0.086027f, 0, 0.040268f, 0, 0.414004f, 0, 0.077849f, 0, 0.016061f, 0, 0.055872f, 0, 0.699024f, 0, 0.723024f, 0, 0.960286f, 0, 0.213009f, 0, 0.506419f, 0, 0.605119f, 0, 0.191199f,
    0, 0.91808f, 0, 0.39154f, 0, 0.834093f, 0, 0.51384f, 0, 0.302416f, 0, 0.986774f, 0, 0.551965f, 0, 0.466067f, 0 }, { 0 }, 1, 1, 38 };

static testData complex_to_real_data = { { 0.786421f, 0.642894f, 0.595759f, 0.910875f, 0.152681f, 0.038125f, 0.163651f, 0.802580f, 0.567567f, 0.154018f, 0.255250f, 0.239411f, 0.839826f, 0.490957f, 0.403399f, 0.027663f,
    0.263422f, 0.086027f, 0.040268f, 0.414004f, 0.077849f, 0.016061f, 0.055872f, 0.699024f, 0.723024f, 0.960286f, 0.213009f, 0.506419f, 0.605119f, 0.191199f, 0.918080f, 0.391540f, 0.834093f, 0.513840f, 0.302416f,
    0.986774f, 0.551965f, 0.466067f, 0.789354f, 0.119532f, 0.620085f, 0.044604f, 0.358943f, 0.459911f, 0.535561f, 0.762341f, 0.487575f, 0.798983f, 0.848368f, 0.527843f, 0.212987f, 0.926217f, 0.543904f, 0.268859f,
    0.625241f, 0.266929f, 0.229145f, 0.838250f, 0.773348f, 0.834265f, 0.029449f, 0.691428f, 0.225805f, 0.863542f, 0.205268f, 0.528220f, 0.850316f, 0.757233f, 0.994287f, 0.639670f, 0.876765f, 0.614373f, 0.684274f,
    0.235708f, 0.074284f, 0.219835f }

, { 0.786421f, 0.595759f, 0.152681f, 0.163651f, 0.567567f, 0.25525f, 0.839826f, 0.403399f, 0.263422f, 0.040268f, 0.077849f, 0.055872f, 0.723024f, 0.213009f, 0.605119f, 0.91808f, 0.834093f, 0.302416f, 0.551965f,
    0.789354f, 0.620085f, 0.358943f, 0.535561f, 0.487575f, 0.848368f, 0.212987f, 0.543904f, 0.625241f, 0.229145f, 0.773348f, 0.029449f, 0.225805f, 0.205268f, 0.850316f, 0.994287f, 0.876765f, 0.684274f, 0.074284f, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }, { 0 }, 1, 1, 38 };

static testData batch_radix = { { 0.649409f, 0.899044f, 0.563938f, 0.942445f, 0.431988f, 0.316933f, 0.615358f, 0.056567f, 0.744165f, 0.497660f, 0.166176f, 0.124102f, 0.261771f, 0.997355f, 0.955453f, 0.478070f, 0.916259f,
    0.448302f, 0.401829f, 0.105239f, 0.504787f, 0.295963f, 0.756050f, 0.803842f, 0.379557f, 0.282510f, 0.668136f, 0.638129f, 0.546768f, 0.065371f, 0.216688f, 0.196177f, 0.964416f, 0.780626f, 0.138622f, 0.396404f,
    0.097559f, 0.753980f, 0.452971f, 0.841724f, 0.251640f, 0.619148f, 0.965826f, 0.513411f, 0.616503f, 0.921279f, 0.991482f, 0.532763f, 0.369581f, 0.393311f, 0.638002f, 0.874368f, 0.689274f, 0.394052f, 0.678210f,
    0.068831f, 0.676562f, 0.346346f, 0.706960f, 0.223330f, 0.411717f, 0.923648f, 0.419507f, 0.376133f, 0.704274f, 0.558130f, 0.772537f, 0.801833f, 0.312109f, 0.225508f, 0.643557f, 0.563749f, 0.844656f, 0.609383f,
    0.077161f, 0.461159f, 0.530662f, 0.068642f, 0.993922f, 0.900243f, 0.461954f, 0.631924f, 0.774611f, 0.151228f, 0.025975f, 0.452821f, 0.220059f, 0.702537f, 0.799167f, 0.927019f, 0.925868f, 0.210884f, 0.850667f,
    0.345375f, 0.587016f, 0.554941f, 0.903505f, 0.359553f, 0.356774f, 0.215614f, 0.585062f, 0.000331f, 0.779363f, 0.429718f, 0.609715f, 0.856524f, 0.890878f, 0.140377f, 0.925166f, 0.884800f, 0.040620f, 0.387120f,
    0.516723f, 0.815232f, 0.538348f, 0.542699f, 0.268053f, 0.758407f, 0.245236f, 0.067219f, 0.685426f, 0.171104f, 0.278103f, 0.536094f, 0.516479f, 0.865119f, 0.091035f, 0.419983f, 0.224673f, 0.447809f, 0.635597f,
    0.809734f, 0.448140f, 0.414961f, 0.239453f, 0.057855f, 0.271485f, 0.130330f, 0.198233f, 0.196651f, 0.015130f, 0.238853f, 0.583771f, 0.531853f, 0.054085f, 0.122119f, 0.074552f, 0.322137f, 0.880526f, 0.319788f,
    0.389357f, 0.565953f, 0.490892f, 0.667459f, 0.102046f, 0.007370f, 0.532579f, 0.193081f, 0.427354f, 0.757251f, 0.640890f, 0.062951f, 0.566986f, 0.089031f, 0.477911f, 0.806438f, 0.146886f, 0.749396f, 0.936768f,
    0.345118f, 0.946047f, 0.951898f, 0.583971f, 0.529818f, 0.483751f, 0.638056f, 0.651937f, 0.558303f, 0.960193f, 0.532463f, 0.878091f, 0.349550f, 0.098416f, 0.368983f, 0.017009f, 0.200462f, 0.376353f, 0.549588f,
    0.393544f, 0.803706f, 0.306839f, 0.034434f, 0.866657f, 0.873824f, 0.123465f, 0.344569f, 0.680263f, 0.270350f, 0.093964f, 0.617031f, 0.615469f, 0.040011f, 0.568929f, 0.199440f, 0.569829f, 0.052680f, 0.837496f,
    0.221766f, 0.610984f, 0.797690f, 0.754229f, 0.489075f, 0.147240f, 0.852645f, 0.858057f, 0.164249f, 0.053107f, 0.234410f, 0.713837f, 0.446651f, 0.038117f, 0.020676f, 0.481085f, 0.904774f, 0.894500f, 0.604550f,
    0.249342f, 0.574763f, 0.874900f, 0.343307f, 0.191794f, 0.490369f, 0.383318f, 0.760723f, 0.689809f, 0.953147f, 0.813403f, 0.527305f, 0.174912f, 0.424386f, 0.324995f, 0.929141f, 0.913461f, 0.472234f, 0.781786f,
    0.771518f, 0.636483f, 0.834894f, 0.005928f, 0.350320f, 0.281545f, 0.044045f, 0.370995f, 0.762630f, 0.948819f, 0.265496f, 0.367179f, 0.198161f, 0.840258f, 0.242079f, 0.541468f, 0.032052f, 0.732448f, 0.924785f,
    0.792774f, 0.422257f, 0.877932f, 0.606177f, 0.949562f, 0.052844f, 0.030564f, 0.274557f, 0.981985f, 0.944025f, 0.746791f, 0.763771f, 0.715543f, 0.383274f, 0.598665f, 0.721472f, 0.733594f, 0.880210f, 0.765517f,
    0.104590f, 0.642839f, 0.714335f, 0.370085f, 0.010019f, 0.912497f, 0.210343f, 0.252098f, 0.453964f, 0.242395f, 0.984547f, 0.378750f, 0.035170f, 0.406804f, 0.256682f, 0.641347f, 0.356366f, 0.309526f, 0.671911f,
    0.630923f, 0.291511f, 0.615936f, 0.377715f, 0.055282f, 0.331479f, 0.760989f, 0.653947f, 0.052950f, 0.494583f, 0.534157f, 0.818467f, 0.599173f, 0.176996f, 0.532802f, 0.969258f, 0.187015f, 0.445299f, 0.179602f,
    0.439113f, 0.899263f, 0.421997f, 0.423660f, 0.278013f, 0.457167f, 0.830464f, 0.534694f, 0.098514f, 0.186830f, 0.844220f, 0.770425f, 0.817753f, 0.135731f, 0.386360f, 0.195468f, 0.191013f, 0.717839f, 0.956458f,
    0.844961f, 0.770789f, 0.451041f, 0.379118f, 0.589256f, 0.050214f, 0.556114f, 0.122058f, 0.019472f, 0.743129f, 0.567357f, 0.199074f, 0.182242f, 0.466620f, 0.621071f, 0.605902f, 0.744633f, 0.078238f, 0.436366f,
    0.279327f, 0.176751f, 0.623196f, 0.123547f, 0.947176f, 0.440949f, 0.259278f, 0.333536f, 0.636418f, 0.450291f, 0.051375f, 0.592875f, 0.295252f, 0.822164f, 0.043916f, 0.674370f, 0.411420f, 0.094130f, 0.230484f,
    0.533478f, 0.113602f, 0.973613f, 0.100835f, 0.312676f, 0.155855f, 0.567455f, 0.933747f, 0.761757f, 0.312088f, 0.011985f, 0.198123f, 0.591414f, 0.188736f, 0.821319f, 0.714961f, 0.135912f, 0.262268f, 0.974239f,
    0.469448f, 0.898686f, 0.424530f, 0.520823f, 0.491561f, 0.719782f, 0.342987f, 0.535477f, 0.394151f, 0.754407f, 0.629607f, 0.624635f, 0.287885f, 0.743209f, 0.598248f, 0.388721f, 0.055885f, 0.754103f, 0.956176f,
    0.989632f, 0.515860f, 0.268263f, 0.001616f, 0.713984f, 0.859678f, 0.190352f, 0.535303f, 0.574639f, 0.326264f, 0.797571f, 0.548877f, 0.795712f, 0.696256f, 0.973407f, 0.316536f, 0.187817f, 0.693189f, 0.659523f,
    0.723294f, 0.087340f, 0.413930f, 0.352901f, 0.711975f },

{ 2.260692f, 2.214990f, 1.103298f, 0.633531f, -0.097898f, 0.216965f, -0.668457f, 0.530692f, 2.127566f, 2.097188f, 0.128426f, 0.289582f, -0.115693f, 0.892843f, 0.836362f, -1.288971f, 2.578926f, 1.653346f, -0.287130f,
    0.506559f, 0.263167f, -0.164816f, 1.110074f, -0.201882f, 1.811149f, 1.182188f, 0.274740f, -0.234309f, 0.041501f, -0.486424f, -0.609163f, 0.668587f, 1.653569f, 2.772734f, 0.421537f, 0.340995f, 0.470381f,
    0.296477f, 1.312177f, -0.287703f, 2.825451f, 2.586601f, -0.384214f, -0.276476f, -1.089164f, 0.494253f, -0.345512f, -0.327787f, 2.375066f, 1.730562f, 0.485843f, 0.039468f, -0.257356f, -0.155836f, -1.125231f,
    -0.040948f, 2.214746f, 1.869457f, 0.112042f, -0.864755f, -0.038188f, 0.670530f, 0.417648f, -0.289850f, 2.432477f, 2.149220f, 0.630248f, 0.203641f, -0.399711f, -0.581945f, 0.154081f, 0.461601f, 2.446402f,
    2.039428f, -0.125090f, 1.457502f, 0.304236f, -0.683377f, 0.753078f, -0.376020f, 1.482599f, 1.938510f, -0.115331f, -0.375449f, -0.506742f, 0.230979f, 0.987288f, 0.733655f, 3.162718f, 2.038219f, -0.395558f,
    0.242793f, 0.136950f, 0.506569f, 0.292557f, 0.920495f, 2.624704f, 1.005217f, 0.104339f, 0.781811f, 0.352429f, -0.285447f, 0.532547f, -0.063367f, 2.466379f, 2.268821f, -0.562194f, -0.878533f, 0.603383f,
    1.213826f, -0.068709f, 0.821982f, 1.568360f, 2.183557f, 0.724150f, -0.236288f, 0.001192f, 0.963721f, -0.226809f, 0.349937f, 1.571043f, 1.992300f, 0.285058f, -0.881083f, 0.832768f, 0.080146f, 0.052837f,
    -0.506948f, 1.547863f, 1.730359f, 0.528411f, -0.363296f, -0.202237f, -0.004820f, -0.975347f, 0.428993f, 1.068618f, 1.097687f, -0.078848f, 0.277016f, -0.495389f, -0.359321f, 0.591557f, -0.494061f, 1.398520f,
    1.329997f, -1.070257f, 0.117136f, 0.470703f, -0.446183f, -0.582626f, -0.512473f, 1.552870f, 1.625162f, -0.791568f, 0.799685f, 0.494070f, 0.095919f, 0.708194f, 0.149071f, 1.832673f, 1.707815f, -0.497386f,
    -1.163587f, 0.404930f, 0.030962f, 0.823344f, -0.323388f, 2.950538f, 2.464890f, 0.666639f, -0.646995f, 0.090941f, -0.715018f, 0.038955f, 0.277596f, 2.588637f, 1.809299f, -0.062674f, -0.653024f, 0.471418f,
    0.006407f, -0.389635f, 1.070531f, 1.093745f, 1.588191f, 0.138620f, -0.672758f, -0.272639f, 0.420147f, -0.891688f, -0.533730f, 1.764349f, 2.105774f, -0.086068f, 0.573974f, 1.329491f, 0.182576f, 0.458857f,
    0.632974f, 2.591723f, 0.513897f, 0.023314f, 0.255898f, -0.221128f, -0.328514f, 0.067966f, -0.281236f, 2.370509f, 2.303658f, 0.788570f, 0.048873f, -0.854063f, 0.997011f, 0.138918f, -0.158784f, 1.286146f,
    1.606511f, -0.443132f, -0.019017f, -1.103698f, -1.096339f, 0.473114f, 0.446486f, 2.210536f, 2.012988f, 0.103994f, 0.203694f, 1.328264f, -0.117275f, -0.064794f, 0.318792f, 2.061442f, 2.665561f, 0.098675f,
    -0.281479f, 0.331999f, -0.089505f, -0.958845f, 0.748314f, 2.656725f, 3.007787f, -0.819451f, -0.119355f, -0.443163f, 0.393532f, -0.094132f, 0.434601f, 1.607287f, 1.422490f, -0.586518f, 0.254964f, -0.853440f,
    0.803409f, -0.143616f, -1.079584f, 2.481354f, 1.397078f, -0.856994f, 0.058299f, -0.664060f, -0.936652f, 0.508418f, 0.273919f, 2.650833f, 1.355836f, 0.174832f, -0.477955f, 0.833841f, -0.405633f, -0.488408f,
    1.216781f, 3.042985f, 2.812542f, 0.308742f, 0.412624f, 0.352072f, -0.157944f, 0.224142f, 0.708877f, 2.512035f, 1.709153f, 0.185326f, -0.229557f, 0.240832f, 1.479937f, -0.003816f, 0.561306f, 1.785740f, 1.684024f,
    1.088896f, -0.647552f, 0.524044f, 0.705756f, 0.251307f, -0.900854f, 1.988600f, 1.576470f, 0.162133f, -0.425653f, -0.555941f, 0.280715f, 0.032423f, -0.404806f, 1.485157f, 1.857724f, -0.308158f, -0.278565f,
    1.268692f, 0.205600f, 0.018051f, -0.273900f, 1.853147f, 2.410020f, -0.266948f, -0.562949f, 0.280771f, 1.165430f, 0.269657f, 0.261367f, 1.959691f, 1.969587f, -0.652525f, -0.280996f, -0.753168f, -0.535335f,
    0.164409f, 0.603197f, 1.627680f, 2.146847f, 0.222129f, -0.770339f, 0.982558f, -0.314313f, -0.693590f, -0.668141f, 2.209309f, 2.297378f, -0.072153f, -0.846574f, -0.128451f, -0.373773f, -1.226832f, -0.312978f,
    1.732199f, 1.114475f, 0.492768f, -0.681672f, -0.514743f, 0.472211f, 0.646799f, -0.704158f, 1.984312f, 1.430086f, -0.235815f, 0.203677f, -0.130562f, -0.340371f, -0.888966f, 0.573087f, 1.074784f, 2.466067f,
    0.046561f, 0.573908f, 0.160618f, -0.701120f, -0.574956f, 0.153929f, 2.539700f, 0.801963f, -0.488041f, 0.088954f, 0.005211f, -0.611381f, -0.255705f, -0.074036f, 1.913897f, 0.600776f, -0.921736f, -0.091154f,
    0.221589f, 0.061861f, -0.837230f, 0.350451f, 1.932611f, 1.632694f, 0.678822f, 0.565281f, -0.773732f, 0.631046f, 0.432119f, 0.905967f, 2.830155f, 1.871207f, -0.315181f, 1.008287f, 0.760960f, 0.497611f, 0.009342f,
    -0.517261f, 2.530488f, 1.858306f, -0.301273f, 0.132035f, -0.417889f, -0.086882f, 0.271966f, 0.062784f, 2.510667f, 1.898194f, -0.122014f, 0.242895f, -0.483956f, -1.210653f, 0.593842f, 0.221106f, 2.162231f,
    1.912457f, -0.258038f, -0.421728f, 1.245000f, 0.838619f, 0.809335f, -0.265906f, 3.141329f, 1.887934f, 0.011268f, -0.194156f, -0.400627f, 0.157108f, -0.453416f, -0.545829f, 1.287580f, 2.542388f, 0.111796f,
    -0.027363f, -0.737266f, -0.328150f, 0.089158f, 0.585881f }, { 0 }, 55, 1, 4

};

static testData radix2_data = { { 0.141603f, 0.606969f, 0.016301f, 0.242887f, 0.137232f, 0.804177f, 0.156679f, 0.400944f, 0.129790f, 0.108809f, 0.998924f, 0.218257f, 0.512932f, 0.839112f, 0.612640f, 0.296032f, 0.637552f,
    0.524287f, 0.493583f, 0.972775f, 0.292517f, 0.771358f, 0.526745f, 0.769914f, 0.400229f, 0.891529f, 0.283315f, 0.352458f, 0.807725f, 0.919026f, 0.069755f, 0.949327f, 0.525995f, 0.086056f, 0.192214f, 0.663227f,
    0.890233f, 0.348893f, 0.064171f, 0.020023f, 0.457702f, 0.063096f, 0.238280f, 0.970634f, 0.902208f, 0.850920f, 0.266666f, 0.539760f, 0.375207f, 0.760249f, 0.512535f, 0.667724f, 0.531606f, 0.039280f, 0.437638f,
    0.931835f, 0.930810f, 0.720952f, 0.284293f, 0.738534f, 0.639979f, 0.354049f, 0.687861f, 0.165974f, 0.440105f, 0.880075f, 0.829201f, 0.330337f, 0.228968f, 0.893372f, 0.350360f, 0.686670f, 0.956468f, 0.588640f,
    0.657304f, 0.858676f, 0.439560f, 0.923970f, 0.398437f, 0.814767f, 0.684219f, 0.910972f, 0.482491f, 0.215825f, 0.950252f, 0.920128f, 0.147660f, 0.881062f, 0.641081f, 0.431953f, 0.619596f, 0.281059f, 0.786002f,
    0.307458f, 0.447034f, 0.226107f, 0.187533f, 0.276235f, 0.556444f, 0.416501f, 0.169607f, 0.906804f, 0.103171f, 0.126075f, 0.495444f, 0.760475f, 0.984752f, 0.935004f, 0.684445f, 0.383188f, 0.749771f, 0.368664f,
    0.294160f, 0.232262f, 0.584489f, 0.244413f, 0.152390f, 0.732149f, 0.125475f, 0.793470f, 0.164102f, 0.745071f, 0.074530f, 0.950104f, 0.052529f, 0.521563f, 0.176211f, 0.240062f, 0.797798f, 0.732654f, 0.656564f,
    0.967405f, 0.639458f, 0.759735f, 0.093480f, 0.134902f, 0.520210f, 0.078232f, 0.069906f, 0.204655f, 0.461420f, 0.819677f, 0.573319f, 0.755581f, 0.051939f, 0.157807f, 0.999994f, 0.204329f, 0.889956f, 0.125468f,
    0.997799f, 0.054058f, 0.870540f, 0.072329f, 0.004162f, 0.923069f, 0.593892f, 0.180372f, 0.163132f, 0.391690f, 0.913027f, 0.819695f, 0.359095f, 0.552485f, 0.579430f, 0.452576f, 0.687387f, 0.099640f, 0.530808f,
    0.757294f, 0.304295f, 0.992228f, 0.576971f, 0.877614f, 0.747809f, 0.628910f, 0.035421f, 0.747803f, 0.833239f, 0.925377f, 0.873271f, 0.831038f, 0.979434f, 0.743811f, 0.903366f, 0.983596f, 0.666880f, 0.497259f,
    0.163968f, 0.830012f, 0.888949f, 0.076995f, 0.649707f, 0.248044f, 0.629480f, 0.229137f, 0.700620f, 0.316867f, 0.328777f, 0.231428f, 0.074161f, 0.633072f, 0.223656f, 0.651132f, 0.510686f, 0.971466f, 0.280042f,
    0.546107f, 0.719269f, 0.113281f, 0.471483f, 0.592540f, 0.944318f, 0.450918f, 0.336351f, 0.847684f, 0.434513f, 0.003231f, 0.344943f, 0.598481f, 0.833243f, 0.233892f, 0.675476f, 0.482950f, 0.481936f, 0.304956f,
    0.712087f, 0.182556f, 0.621823f, 0.040864f, 0.413984f, 0.695984f, 0.673936f, 0.637640f, 0.347116f, 0.184622f, 0.609106f, 0.627158f, 0.730729f, 0.328374f, 0.740438f, 0.202213f, 0.920914f, 0.684757f, 0.653130f,
    0.257265f, 0.532441f, 0.087644f, 0.260497f, 0.877384f, 0.686125f, 0.093740f, 0.111276f, 0.361601f, 0.576690f, 0.593211f, 0.666557f, 0.288778f, 0.775767f, 0.288379f, 0.329642f, 0.189751f, 0.984363f, 0.003579f,
    0.827391f, 0.331479f, 0.188201f, 0.436497f, 0.958637f, 0.918930f, 0.764871f, 0.699075f, 0.121143f, 0.685786f, 0.383832f, 0.774274f, 0.943051f, 0.916273f, 0.861917f, 0.203548f, 0.793657f, 0.548042f, 0.297288f,
    0.904932f, 0.909643f, 0.873979f, 0.498144f, 0.576200f, 0.162757f, 0.273911f, 0.864579f, 0.492399f, 0.463662f, 0.848942f, 0.495977f, 0.291053f, 0.180421f, 0.684178f, 0.727550f, 0.139058f, 0.603109f, 0.492422f,
    0.838134f, 0.724252f, 0.178208f, 0.221966f, 0.498525f, 0.121259f, 0.138238f, 0.360443f, 0.324807f, 0.931895f, 0.908485f, 0.622095f, 0.836828f, 0.818128f, 0.496074f, 0.334972f, 0.394327f, 0.658831f, 0.608883f,
    0.258906f, 0.151230f, 0.072545f, 0.107848f, 0.647207f, 0.363598f, 0.288270f, 0.331386f, 0.091149f, 0.427328f, 0.934495f, 0.583570f, 0.265461f, 0.658747f, 0.761778f, 0.487427f, 0.157272f, 0.883037f, 0.625665f,
    0.517715f, 0.207844f, 0.557561f, 0.426199f, 0.829939f, 0.394388f, 0.244327f, 0.326013f, 0.729360f, 0.638654f, 0.984845f, 0.338243f, 0.897560f, 0.136075f, 0.410788f, 0.005409f, 0.783282f, 0.774386f, 0.293678f,
    0.114668f, 0.865535f, 0.721006f, 0.049162f, 0.449105f, 0.986467f, 0.707909f, 0.210883f, 0.473894f, 0.865181f, 0.093920f, 0.099559f, 0.382896f, 0.301763f, 0.657120f, 0.809095f, 0.131702f, 0.051508f, 0.053422f,
    0.457716f, 0.780868f, 0.692076f, 0.442560f, 0.119111f, 0.589637f, 0.578635f, 0.529899f, 0.595045f, 0.361917f, 0.304285f, 0.888723f, 0.476585f, 0.169820f, 0.609729f, 0.525747f, 0.618925f, 0.596196f, 0.233656f,
    0.829808f, 0.070090f, 0.098837f, 0.923728f, 0.169649f, 0.481733f, 0.225491f, 0.826769f, 0.290829f, 0.357193f, 0.878278f, 0.344251f, 0.814909f, 0.659146f, 0.036327f, 0.257469f, 0.778257f, 0.625964f, 0.836104f,
    0.308157f, 0.221009f, 0.198021f, 0.612442f, 0.109733f, 0.674605f, 0.782262f, 0.719462f, 0.200352f, 0.401188f, 0.315658f, 0.434009f, 0.230996f, 0.385748f, 0.532846f, 0.154724f, 0.555398f, 0.014579f, 0.380215f,
    0.382167f, 0.305408f, 0.737408f, 0.260445f, 0.649659f, 0.552316f, 0.919591f, 0.685986f, 0.809785f, 0.697848f, 0.311950f, 0.645889f, 0.006005f, 0.532960f, 0.843910f, 0.618447f, 0.642693f, 0.518515f, 0.400709f,
    0.362154f, 0.718867f, 0.801897f, 0.677812f, 0.152876f, 0.032893f, 0.063561f, 0.685722f, 0.187616f, 0.618958f, 0.700301f, 0.567831f, 0.001125f, 0.005709f, 0.305239f, 0.261570f, 0.655368f, 0.857555f, 0.181161f,
    0.341354f, 0.667341f, 0.879009f, 0.653305f, 0.313230f, 0.885014f, 0.186265f, 0.157139f, 0.503461f, 0.828957f, 0.675654f, 0.904170f, 0.191112f, 0.394521f, 0.706067f, 0.868924f, 0.547397f, 0.738959f, 0.932485f,
    0.233119f, 0.926576f, 0.551443f, 0.933420f, 0.494407f, 0.552568f, 0.939129f, 0.799646f, 0.814139f, 0.594497f, 0.657201f, 0.995300f, 0.935852f, 0.324541f, 0.874309f, 0.589157f, 0.637771f, 0.759324f, 0.775421f,
    0.794910f, 0.262785f, 0.604379f, 0.470564f, 0.166955f, 0.795490f, 0.865085f, 0.873021f, 0.664414f, 0.412483f, 0.611981f, 0.596899f, 0.645602f, 0.538557f, 0.148342f, 0.579022f, 0.032963f, 0.700910f, 0.518151f,
    0.832609f, 0.515049f, 0.112648f, 0.489810f, 0.510349f, 0.048500f, 0.814351f, 0.384658f, 0.637656f, 0.452122f, 0.143982f, 0.413078f, 0.247033f, 0.406767f, 0.017457f, 0.717597f, 0.573721f, 0.812947f, 0.582682f,
    0.446743f, 0.477361f, 0.995165f, 0.058723f, 0.074260f, 0.640766f, 0.597280f, 0.222602f, 0.219788f, 0.630243f, 0.923513f, 0.737939f, 0.462852f, 0.438562f, 0.850586f, 0.952662f, 0.948911f, 0.899086f, 0.767014f,
    0.333569f, 0.536743f, 0.219136f, 0.477551f, 0.949820f, 0.466169f, 0.884318f, 0.967277f, 0.183765f, 0.458039f, 0.780224f, 0.766448f, 0.904782f, 0.257585f, 0.761612f, 0.963505f, 0.331846f, 0.402379f, 0.560785f,
    0.554448f, 0.622167f, 0.191028f, 0.477961f, 0.360105f, 0.653880f, 0.916523f, 0.210692f, 0.606542f, 0.865434f, 0.109778f, 0.373556f, 0.199003f, 0.646520f, 0.592692f, 0.676554f, 0.596341f, 0.058860f, 0.560872f,
    0.563617f, 0.242626f, 0.018911f, 0.343841f, 0.009073f, 0.923692f, 0.601427f, 0.770686f, 0.887197f, 0.933273f, 0.173065f, 0.447982f, 0.487721f, 0.795231f, 0.639009f, 0.965682f, 0.155336f, 0.292889f, 0.882204f,
    0.366028f, 0.899431f, 0.747638f, 0.475806f, 0.272987f, 0.946640f, 0.122326f, 0.865679f, 0.623194f, 0.718666f, 0.924540f, 0.184066f, 0.282284f, 0.167165f, 0.202977f, 0.626125f, 0.176239f, 0.126669f, 0.227552f,
    0.946925f, 0.013866f, 0.160824f, 0.119989f, 0.461848f, 0.648545f, 0.915221f, 0.100857f, 0.614227f, 0.070557f, 0.393746f, 0.496431f, 0.436585f, 0.293177f, 0.244069f, 0.912391f, 0.566164f, 0.190709f, 0.034716f,
    0.431844f, 0.813904f, 0.753383f, 0.356383f, 0.997970f, 0.035666f, 0.523548f, 0.200947f, 0.661792f, 0.699787f, 0.327616f, 0.889343f, 0.646712f, 0.341482f, 0.050168f, 0.766701f, 0.803330f, 0.698713f, 0.681922f,
    0.904187f, 0.312940f, 0.752479f, 0.297933f, 0.809371f, 0.189064f, 0.591111f, 0.053439f, 0.101454f, 0.157275f, 0.244149f, 0.136171f, 0.589119f, 0.058052f, 0.889553f, 0.945502f, 0.056022f, 0.925220f, 0.469050f,
    0.256969f, 0.587011f, 0.168837f, 0.584585f, 0.476355f, 0.815549f, 0.926068f, 0.526523f, 0.582250f, 0.729398f, 0.225236f, 0.264172f, 0.633585f, 0.538175f, 0.016651f, 0.931518f, 0.347546f, 0.205714f, 0.522629f,
    0.400985f, 0.307168f, 0.679904f, 0.645134f, 0.443339f, 0.269022f, 0.703186f, 0.332892f, 0.214524f, 0.759208f, 0.258112f, 0.683574f, 0.016177f, 0.845123f, 0.852411f, 0.600763f, 0.321478f, 0.667960f, 0.526830f,
    0.848000f, 0.250210f, 0.256228f, 0.073236f, 0.514382f, 0.889813f, 0.611411f, 0.531033f, 0.821331f, 0.958957f, 0.736747f, 0.343959f, 0.359942f, 0.043915f, 0.023863f, 0.005076f, 0.487254f, 0.292886f, 0.708262f,
    0.820146f, 0.507410f, 0.467471f, 0.078258f, 0.190984f, 0.483648f, 0.923381f, 0.043395f, 0.084411f, 0.244858f, 0.711355f, 0.611241f, 0.092858f, 0.961565f, 0.867469f, 0.166094f, 0.475947f, 0.757282f, 0.777505f,
    0.006980f, 0.578613f, 0.736462f, 0.743727f, 0.922572f, 0.096404f, 0.787642f, 0.946435f, 0.101480f, 0.274897f, 0.239321f, 0.809743f, 0.095043f, 0.746730f, 0.277214f, 0.173301f, 0.937714f, 0.760862f, 0.096681f,
    0.981109f, 0.845273f, 0.341540f, 0.692463f, 0.456514f, 0.434398f, 0.654029f, 0.323983f, 0.600492f, 0.129976f, 0.081265f, 0.377997f, 0.136956f, 0.659878f, 0.114459f, 0.880683f, 0.582450f, 0.210863f, 0.668326f,
    0.528885f, 0.312343f, 0.943222f, 0.768206f, 0.122086f, 0.038265f, 0.514936f, 0.399300f, 0.211565f, 0.452650f, 0.160162f, 0.308247f, 0.433758f, 0.005435f, 0.649787f, 0.126222f, 0.461949f, 0.084185f, 0.780250f,
    0.785932f, 0.684677f, 0.910227f, 0.867197f, 0.062674f, 0.047183f, 0.527075f, 0.177133f, 0.927866f, 0.109525f, 0.387996f, 0.596191f, 0.638409f, 0.700340f, 0.539413f, 0.406615f, 0.822426f, 0.577678f, 0.921551f,
    0.221726f, 0.789244f, 0.374201f, 0.381888f, 0.097491f, 0.807959f, 0.387323f, 0.747277f, 0.934181f, 0.849272f, 0.831462f, 0.714432f, 0.635204f, 0.516139f, 0.624658f, 0.502401f, 0.578813f, 0.671841f, 0.029476f,
    0.755946f, 0.599707f, 0.139001f, 0.143942f, 0.195898f, 0.777410f, 0.844281f, 0.735311f, 0.184025f, 0.666707f, 0.312990f, 0.105576f, 0.888433f, 0.102233f, 0.479777f, 0.270321f, 0.199724f, 0.287736f, 0.657643f,
    0.947001f, 0.221918f, 0.506915f, 0.778463f, 0.936349f, 0.142119f, 0.294601f, 0.561007f, 0.644520f, 0.873414f, 0.232848f, 0.673996f, 0.629359f, 0.832555f, 0.812997f, 0.773301f, 0.028453f, 0.590407f, 0.617582f,
    0.763764f, 0.774432f, 0.284289f, 0.076753f, 0.880009f, 0.172722f, 0.178987f, 0.359786f, 0.443043f, 0.378710f, 0.647522f, 0.100686f, 0.325711f, 0.869440f, 0.607600f, 0.104174f, 0.805789f, 0.749719f, 0.398775f,
    0.366796f, 0.394239f, 0.272189f, 0.599644f, 0.068235f, 0.901549f, 0.432199f, 0.881232f, 0.674850f, 0.460652f, 0.471639f, 0.292432f, 0.224415f, 0.246071f, 0.576721f, 0.301169f, 0.126080f, 0.749443f, 0.480155f,
    0.485866f, 0.192486f, 0.858866f, 0.133388f, 0.293171f, 0.184577f, 0.002828f, 0.900772f, 0.288752f, 0.808617f, 0.650491f, 0.687527f, 0.175413f, 0.044729f, 0.959716f, 0.775058f, 0.112964f, 0.861265f, 0.207257f,
    0.994196f, 0.536115f, 0.667908f, 0.465835f, 0.828546f, 0.892324f, 0.711906f, 0.405267f, 0.193493f, 0.837986f, 0.154711f, 0.673648f, 0.323852f, 0.347196f, 0.532514f, 0.457240f, 0.640368f, 0.717092f, 0.460067f,
    0.541140f, 0.005843f, 0.268684f, 0.191630f, 0.693370f, 0.444097f, 0.236360f, 0.653087f, 0.219155f, 0.349324f, 0.514352f, 0.426412f, 0.343520f, 0.050466f, 0.094320f, 0.809355f, 0.879013f, 0.986644f, 0.521261f,
    0.284280f, 0.180136f, 0.359247f, 0.438990f, 0.853785f, 0.683098f, 0.786187f, 0.386299f, 0.140338f, 0.426555f, 0.103390f, 0.600405f, 0.967694f, 0.109233f, 0.869090f, 0.159324f, 0.802604f, 0.313187f, 0.395684f,
    0.455690f, 0.532342f, 0.745008f, 0.970042f, 0.958753f, 0.088528f, 0.020508f, 0.053073f, 0.897883f, 0.899521f, 0.039717f, 0.419144f, 0.183801f, 0.219853f, 0.778391f, 0.622791f, 0.073638f, 0.461489f, 0.408978f,
    0.459936f, 0.601827f, 0.835533f, 0.563327f, 0.202232f, 0.803227f, 0.672560f, 0.071322f, 0.962551f, 0.475164f, 0.384509f, 0.358235f, 0.930854f, 0.916851f, 0.103244f, 0.900896f, 0.875604f, 0.191772f, 0.921405f,
    0.928678f, 0.089655f, 0.820926f, 0.968395f, 0.508799f, 0.004727f, 0.188248f, 0.287189f, 0.627518f, 0.261886f, 0.748678f, 0.036496f, 0.721822f, 0.350505f, 0.872028f, 0.285149f, 0.552738f, 0.675255f, 0.957709f,
    0.624060f, 0.637806f, 0.432873f, 0.008569f, 0.996042f, 0.363727f, 0.925420f, 0.099285f, 0.264624f, 0.801024f, 0.291057f, 0.186029f, 0.729702f, 0.380712f, 0.006954f, 0.698096f, 0.889511f, 0.011681f, 0.886344f,
    0.176700f, 0.639199f, 0.148230f, 0.925379f, 0.675694f, 0.870053f, 0.275884f, 0.547723f, 0.155202f, 0.828622f, 0.222978f, 0.112911f, 0.452681f, 0.860784f, 0.545784f, 0.461250f, 0.856826f, 0.909512f, 0.386669f,
    0.956111f, 0.174136f, 0.187693f, 0.247168f, 0.360164f, 0.917395f, 0.627880f, 0.367118f, 0.615491f, 0.517391f, 0.378799f, 0.501835f, 0.694091f, 0.017998f, 0.650066f, 0.619470f, 0.693692f, 0.520118f, 0.895354f,
    0.241415f, 0.675320f, 0.723975f, 0.464393f, 0.788231f, 0.176656f, 0.325177f, 0.334016f, 0.637906f, 0.182003f, 0.243528f, 0.024575f, 0.138114f, 0.417663f, 0.212269f, 0.385282f, 0.777828f, 0.129663f, 0.013161f,
    0.144946f, 0.745154f, 0.530552f, 0.523745f, 0.246990f, 0.224643f, 0.541743f, 0.897055f, 0.844113f, 0.235435f, 0.417174f, 0.739467f, 0.476850f, 0.092494f, 0.463442f, 0.941243f, 0.880725f, 0.640098f, 0.266420f,
    0.214741f, 0.278005f, 0.448423f, 0.458269f, 0.302580f, 0.586537f, 0.875932f, 0.514849f, 0.971818f, 0.653760f, 0.644512f, 0.984980f, 0.798706f, 0.389667f, 0.515532f, 0.322451f, 0.636656f, 0.740175f, 0.864194f,
    0.533712f, 0.584288f, 0.099629f, 0.950885f, 0.323755f, 0.576479f, 0.043379f, 0.787197f, 0.517722f, 0.924104f, 0.427295f, 0.784142f, 0.138845f, 0.705300f, 0.232565f, 0.597113f, 0.007880f, 0.819102f, 0.473045f,
    0.522729f, 0.790920f, 0.126805f, 0.167241f, 0.775899f, 0.925511f, 0.556908f, 0.291431f, 0.247962f, 0.193564f, 0.031606f, 0.112157f, 0.727276f, 0.615894f, 0.211786f, 0.678161f, 0.939649f, 0.788265f, 0.721540f,
    0.726846f, 0.305987f, 0.645644f, 0.154141f, 0.090130f, 0.784489f, 0.859441f, 0.322695f, 0.381603f, 0.867321f, 0.141796f, 0.854648f, 0.390050f, 0.932716f, 0.981453f, 0.557291f, 0.708616f, 0.906964f, 0.114199f,
    0.000047f, 0.154927f, 0.307763f, 0.031653f, 0.267083f, 0.035039f, 0.647548f, 0.478869f, 0.713200f, 0.587197f, 0.267134f, 0.434740f, 0.314043f, 0.573122f, 0.080384f, 0.468185f, 0.663252f, 0.864873f, 0.327626f,
    0.985946f, 0.246476f, 0.194948f, 0.127743f, 0.101124f, 0.584998f, 0.060459f, 0.082577f, 0.142289f, 0.769074f, 0.989541f, 0.256489f, 0.769121f, 0.144468f, 0.564252f, 0.800775f, 0.411551f, 0.599291f, 0.448322f,
    0.890420f, 0.312491f, 0.035519f, 0.157555f, 0.747231f, 0.349562f, 0.730677f, 0.827615f, 0.817747f, 0.393928f, 0.692488f, 0.145373f, 0.379874f, 0.938963f, 0.340321f, 0.507617f, 0.040087f, 0.925318f, 0.568076f,
    0.122664f, 0.067608f, 0.337150f, 0.112205f, 0.324096f, 0.106272f, 0.256673f, 0.888348f, 0.907046f, 0.668225f, 0.487639f, 0.355369f, 0.558645f, 0.800129f, 0.390888f, 0.716199f, 0.547360f, 0.740451f, 0.446876f,
    0.374975f, 0.558198f, 0.840804f, 0.067462f, 0.703571f, 0.220679f, 0.006426f, 0.043891f, 0.728296f, 0.046513f, 0.969210f, 0.296372f, 0.169177f, 0.036818f, 0.633522f, 0.281382f, 0.360914f, 0.739794f, 0.538055f,
    0.249262f, 0.646840f, 0.206280f, 0.736901f, 0.002209f, 0.764925f, 0.537030f, 0.393097f, 0.481124f, 0.084390f, 0.133548f, 0.928000f, 0.459365f, 0.691745f, 0.768804f, 0.526827f, 0.395316f, 0.989483f, 0.533253f,
    0.439207f, 0.717779f, 0.579765f, 0.408417f, 0.014150f, 0.748942f, 0.445235f, 0.647672f, 0.030324f, 0.806149f, 0.387466f, 0.568379f, 0.055411f, 0.034307f, 0.774659f, 0.792311f, 0.036516f, 0.539584f, 0.329342f,
    0.429613f, 0.020708f, 0.413732f, 0.563161f, 0.948708f, 0.873097f, 0.254906f, 0.717512f, 0.399924f, 0.650222f, 0.706995f, 0.933176f, 0.089430f, 0.424774f, 0.512941f, 0.497847f, 0.438924f, 0.261883f, 0.943081f,
    0.086596f, 0.292207f, 0.749230f, 0.474063f, 0.860587f, 0.804641f, 0.508369f, 0.635246f, 0.596952f, 0.544885f, 0.174830f, 0.926294f, 0.974499f, 0.195538f, 0.340026f, 0.537660f, 0.144246f, 0.213122f, 0.792566f,
    0.861759f, 0.613046f, 0.442789f, 0.568754f, 0.546222f, 0.532218f, 0.993528f, 0.059163f, 0.030065f, 0.432452f, 0.321047f, 0.973147f, 0.519048f, 0.613254f, 0.722377f, 0.993110f, 0.473841f, 0.527017f, 0.501480f,
    0.109087f, 0.123969f, 0.046365f, 0.283917f, 0.050263f, 0.020864f, 0.479455f, 0.390289f, 0.558524f, 0.623701f, 0.603411f, 0.351090f, 0.485460f, 0.216457f, 0.793878f, 0.054214f, 0.762679f, 0.326097f, 0.047742f,
    0.821842f, 0.356162f, 0.480193f, 0.142889f, 0.329309f, 0.999241f, 0.756143f, 0.051685f, 0.992352f, 0.229984f, 0.578702f, 0.493831f, 0.339071f, 0.702672f, 0.540197f, 0.622988f, 0.752935f, 0.561060f, 0.102443f,
    0.143224f, 0.119584f, 0.726144f, 0.746635f, 0.470674f, 0.211604f, 0.963092f, 0.264553f, 0.265818f, 0.725771f, 0.590649f, 0.313560f, 0.547613f, 0.946811f, 0.793753f, 0.690502f, 0.276120f, 0.792995f, 0.446645f,
    0.327805f, 0.785346f, 0.676628f, 0.906507f, 0.279178f, 0.015699f, 0.609179f, 0.819374f, 0.638687f, 0.362115f, 0.380434f, 0.741130f, 0.505339f, 0.500019f, 0.467274f, 0.251974f, 0.970693f, 0.678878f, 0.215066f,
    0.235245f, 0.944697f, 0.940837f, 0.825895f, 0.258257f, 0.488450f, 0.772706f, 0.052010f, 0.178952f, 0.048826f, 0.845005f, 0.625596f, 0.376631f, 0.630351f, 0.302225f, 0.283138f, 0.909529f, 0.317924f, 0.892318f,
    0.728903f, 0.956611f, 0.254432f, 0.109337f, 0.697741f, 0.759771f, 0.609356f, 0.165015f, 0.011745f, 0.580048f, 0.843894f, 0.226811f, 0.815294f, 0.788590f, 0.167648f, 0.641188f, 0.046847f, 0.656098f, 0.413894f,
    0.098858f, 0.835050f, 0.462720f, 0.943863f, 0.460646f, 0.839351f, 0.574214f, 0.762871f, 0.122489f, 0.483742f, 0.080795f, 0.014807f, 0.212645f, 0.037406f, 0.269239f, 0.321982f, 0.735147f, 0.029011f, 0.931338f,
    0.900163f, 0.040756f, 0.511386f, 0.744056f, 0.267567f, 0.326680f, 0.532647f, 0.435216f, 0.967868f, 0.579494f, 0.091314f, 0.381762f, 0.678351f, 0.926364f, 0.844482f, 0.622214f, 0.387011f, 0.683833f, 0.196428f,
    0.149882f, 0.806322f, 0.680170f, 0.230677f, 0.821129f, 0.892815f, 0.268084f, 0.090368f, 0.214797f, 0.003231f, 0.119379f, 0.146135f, 0.903394f, 0.160135f, 0.657522f, 0.647450f, 0.427702f, 0.984202f, 0.180096f,
    0.862918f, 0.952070f, 0.759590f, 0.954232f, 0.333832f, 0.437942f, 0.880596f, 0.178314f, 0.060156f, 0.267607f, 0.862147f, 0.256583f, 0.417489f, 0.668469f, 0.936754f, 0.648167f, 0.489597f, 0.829569f, 0.916250f,
    0.579965f, 0.044366f, 0.919482f, 0.699344f, 0.190501f, 0.822875f, 0.859479f, 0.848023f, 0.470325f, 0.287181f, 0.832225f, 0.650421f, 0.150099f, 0.784295f, 0.410011f, 0.104331f, 0.118126f, 0.847953f, 0.984927f,
    0.296440f, 0.908109f, 0.252535f, 0.158587f, 0.164692f, 0.670024f, 0.827056f, 0.101446f, 0.318191f, 0.316653f, 0.931014f, 0.234441f, 0.896618f, 0.975380f, 0.153923f, 0.595963f, 0.165882f, 0.976798f, 0.455442f,
    0.013905f, 0.447123f, 0.742623f, 0.846130f, 0.097544f, 0.892721f, 0.630424f, 0.507555f, 0.997052f, 0.748551f, 0.355508f, 0.981980f, 0.044991f, 0.263617f, 0.234514f, 0.203578f, 0.428309f, 0.904539f, 0.030634f,
    0.529754f, 0.222729f, 0.347287f, 0.460768f, 0.457171f, 0.243905f, 0.436149f, 0.611093f, 0.839868f, 0.602030f, 0.587891f, 0.295309f, 0.615935f, 0.035014f, 0.037932f, 0.462065f, 0.132558f, 0.930653f, 0.092489f,
    0.640113f, 0.927706f, 0.841040f, 0.995620f, 0.909686f, 0.886031f, 0.259237f, 0.144200f, 0.089609f, 0.687545f, 0.048739f, 0.120243f, 0.217300f, 0.271468f, 0.467529f, 0.678068f, 0.728638f, 0.711434f, 0.114216f,
    0.339732f, 0.551302f, 0.716247f, 0.927623f, 0.846611f, 0.332182f, 0.962637f, 0.884543f, 0.794247f, 0.095194f, 0.815197f, 0.886736f, 0.735307f, 0.742903f, 0.727776f, 0.730928f, 0.652588f, 0.613807f, 0.990165f,
    0.796788f, 0.703416f, 0.677710f, 0.845527f, 0.823659f, 0.895010f, 0.116995f, 0.291188f, 0.573077f, 0.845633f, 0.002623f, 0.687294f, 0.185365f, 0.553925f, 0.403541f, 0.112988f, 0.400536f, 0.735723f, 0.075625f,
    0.285080f, 0.529970f, 0.170819f, 0.100276f, 0.416706f, 0.906126f, 0.843179f, 0.144482f, 0.637054f, 0.495767f, 0.758290f, 0.627218f, 0.292555f, 0.461706f, 0.304929f, 0.138082f, 0.285365f, 0.199938f, 0.255076f,
    0.576553f, 0.773016f, 0.100709f, 0.579176f, 0.460309f, 0.286074f, 0.133100f, 0.863850f, 0.399062f, 0.533636f, 0.599573f, 0.474687f, 0.818716f, 0.129542f, 0.645507f, 0.918992f, 0.546248f, 0.551633f, 0.762171f,
    0.690731f, 0.188687f, 0.257939f, 0.449020f, 0.815905f, 0.550494f, 0.910726f, 0.120834f, 0.688576f, 0.196091f, 0.320772f, 0.943652f, 0.772644f, 0.093788f, 0.044362f, 0.351819f, 0.554097f, 0.330436f, 0.484919f,
    0.417947f, 0.729499f, 0.018556f, 0.017520f, 0.204186f, 0.837272f, 0.147062f, 0.849692f, 0.756264f, 0.693311f, 0.401325f, 0.518436f, 0.384041f, 0.590012f, 0.776374f, 0.833062f, 0.405917f, 0.326868f, 0.743788f,
    0.526751f, 0.015444f, 0.939879f, 0.847523f, 0.959096f, 0.712522f, 0.941311f, 0.003458f, 0.064342f, 0.495408f, 0.333894f, 0.549261f, 0.913355f, 0.063393f, 0.567817f, 0.930875f, 0.267578f, 0.405089f, 0.077937f,
    0.117271f, 0.161353f, 0.771248f, 0.518596f, 0.679788f, 0.155289f, 0.108608f, 0.456162f, 0.988351f, 0.514525f, 0.783031f, 0.732138f, 0.041276f, 0.798474f, 0.672017f, 0.888799f, 0.757571f, 0.384539f, 0.830110f,
    0.761029f, 0.448881f, 0.325518f, 0.094923f, 0.998142f, 0.238873f, 0.158316f, 0.565959f, 0.169748f, 0.425894f, 0.971047f, 0.247685f, 0.543165f, 0.132400f, 0.018933f, 0.061761f, 0.812189f, 0.174222f, 0.170368f,
    0.268351f, 0.162573f, 0.684893f, 0.051382f, 0.894711f, 0.726169f, 0.849856f, 0.566728f, 0.614968f, 0.607427f, 0.951267f, 0.445079f, 0.368455f, 0.400148f, 0.770597f, 0.463378f, 0.398290f, 0.009470f, 0.621694f,
    0.964248f, 0.179218f, 0.047588f, 0.935296f, 0.426904f, 0.590753f, 0.067696f, 0.445836f, 0.652514f, 0.879885f, 0.620058f, 0.822882f, 0.148236f, 0.782631f, 0.507775f, 0.199617f, 0.677342f, 0.233944f, 0.049473f,
    0.244069f, 0.848912f, 0.656900f, 0.195336f, 0.293991f, 0.025355f, 0.595484f, 0.064588f, 0.488733f, 0.993774f, 0.074058f, 0.110427f, 0.958022f, 0.253276f, 0.158015f, 0.893318f, 0.680180f, 0.748768f, 0.961014f,
    0.126016f, 0.401282f, 0.840899f, 0.746074f, 0.224164f, 0.989134f, 0.528705f, 0.731938f, 0.188752f, 0.206047f, 0.965882f, 0.238225f, 0.450117f, 0.814794f, 0.895125f, 0.645453f, 0.108785f, 0.920479f, 0.240937f,
    0.173373f, 0.409213f, 0.234711f, 0.247431f, 0.519640f, 0.192734f, 0.500707f, 0.677655f, 0.086052f, 0.180886f, 0.426423f, 0.047066f, 0.306902f, 0.827705f, 0.887964f, 0.052977f, 0.051869f, 0.877099f, 0.581682f,
    0.783807f, 0.065851f, 0.787729f, 0.749690f, 0.304075f, 0.237846f, 0.564484f, 0.199200f, 0.883299f, 0.673269f, 0.119679f, 0.124236f, 0.846642f, 0.528892f, 0.358948f, 0.094073f, 0.048532f, 0.551681f, 0.594779f,
    0.726187f, 0.637733f, 0.775665f, 0.152610f, 0.684799f, 0.082568f, 0.980315f, 0.572763f, 0.135545f, 0.032184f, 0.449862f, 0.717227f, 0.815991f, 0.515712f, 0.504956f, 0.565681f, 0.819788f, 0.742802f, 0.130165f,
    0.018988f, 0.626101f, 0.803434f, 0.138667f, 0.750337f, 0.650077f, 0.667559f, 0.109285f, 0.744149f, 0.716091f, 0.660966f, 0.338929f, 0.442278f, 0.298698f, 0.114594f, 0.594888f, 0.983497f },

{ 515.048767f, 509.571808f, -6.025575f, 1.976360f, -12.670959f, -9.175626f, -0.766247f, -11.706539f, 12.517479f, 10.781857f, 11.190938f, -0.434605f, 2.178318f, 15.858614f, -0.522566f, 9.152844f, 3.933234f, 10.338278f,
    2.840910f, -8.104322f, -3.067332f, -3.096888f, -8.073547f, 3.066285f, 11.469490f, -9.437840f, -12.122418f, 2.904269f, -1.174334f, -12.492306f, 11.157325f, 10.773306f, -12.018282f, -5.633274f, 7.095572f,
    -9.960210f, -2.861669f, -16.231514f, -2.655921f, 0.322942f, -16.791550f, 19.061102f, 3.997539f, 19.717102f, -7.353047f, 9.617476f, 4.365694f, 5.909649f, 13.628489f, 4.445456f, 5.534916f, -0.822911f, 11.957522f,
    2.179608f, -9.589991f, -3.559615f, -11.369873f, 17.532536f, 5.574258f, -10.768557f, -13.829884f, -3.805798f, -2.561594f, -10.415363f, 7.594780f, -4.754166f, 0.142756f, 21.957130f, -4.737068f, 9.965620f,
    -6.293073f, 0.391588f, -1.720892f, 15.044313f, 0.434195f, -5.595783f, 7.994608f, -3.596308f, -3.993686f, -4.649220f, -9.486031f, -5.085734f, 8.039619f, -8.908957f, -8.929087f, -8.566183f, -11.722345f,
    -17.764816f, -1.926482f, -13.922682f, -4.735356f, 13.092692f, 1.044323f, -10.396870f, 3.985605f, -3.230143f, -1.187390f, -12.668790f, -15.539341f, 19.582123f, -19.511932f, 14.810565f, -4.506903f, -2.112122f,
    -16.815390f, -1.905779f, 3.228497f, -4.443562f, -4.560777f, -2.472111f, -2.745204f, -10.282161f, -1.817272f, -17.771423f, -0.329398f, -8.397812f, -10.730522f, -5.064418f, 12.503286f, -2.305092f, 13.005966f,
    5.268792f, -8.314013f, 0.363772f, -3.879151f, -6.875761f, -17.450581f, -7.554523f, -15.338352f, -10.422470f, 12.072240f, -20.546263f, 9.326822f, 0.686930f, -5.918107f, 0.776415f, -5.101874f, 5.605002f,
    12.400445f, -1.458924f, 0.452107f, -1.434625f, -3.758493f, 5.974786f, -3.521808f, 4.859629f, -1.772305f, -5.005835f, -8.106359f, 16.355247f, -4.350267f, 0.430454f, -3.324364f, 9.284277f, -13.325912f, 3.294477f,
    -5.169349f, -3.726660f, -4.219793f, 7.727378f, 4.232205f, 1.977085f, -0.759858f, 7.997559f, -3.339279f, -12.425816f, -3.001029f, 1.886047f, 4.714069f, 17.864540f, -0.399463f, 15.768858f, -9.103693f,
    -15.997008f, -0.716268f, 2.524358f, 8.396572f, 4.108875f, -18.629244f, 2.540874f, 2.406905f, 15.305651f, 0.155551f, 18.186625f, -6.095364f, 0.402065f, 0.652466f, 6.783611f, -6.041066f, -7.790985f, 5.695851f,
    -6.481966f, -1.716996f, -5.055265f, 2.188220f, -2.809714f, -2.347547f, -5.907421f, 12.175780f, 1.677606f, -4.946924f, 8.056129f, 3.921293f, -2.131018f, -1.100370f, 2.968524f, 5.082561f, 0.143363f, 6.365215f,
    20.361702f, -16.948793f, 16.252245f, -17.972675f, -7.232704f, 8.847612f, -8.373466f, 1.850200f, 8.483900f, 10.998673f, 2.353814f, 6.066281f, 13.821304f, -9.252668f, -16.196436f, 1.825291f, -5.688068f,
    -2.804713f, 0.479691f, 12.135989f, -12.773344f, -0.429102f, -0.563016f, 4.734875f, 7.503360f, 6.396416f, 10.442451f, -2.247446f, 5.417533f, -21.838764f, -10.599825f, 3.791265f, -4.036291f, -4.146761f,
    20.269602f, -4.955556f, 0.616998f, -4.497226f, 19.282755f, 4.967245f, -14.033916f, 11.768723f, 1.883647f, -4.028033f, 11.923183f, 16.875851f, -8.564652f, -13.737465f, -9.576836f, -4.100986f, 8.573600f,
    -7.842994f, -3.896870f, 1.351258f, 13.081131f, -1.583919f, -10.190879f, 1.837659f, 9.438413f, 2.634683f, -1.997445f, -1.110130f, 16.859087f, 1.276522f, 8.753054f, 3.640892f, -16.944193f, 15.952255f, 5.028552f,
    0.440116f, -1.579856f, 8.353903f, 2.781380f, -5.940563f, 11.541710f, -9.288488f, 20.935101f, 5.759439f, -0.220917f, -8.796144f, 1.149761f, 15.423843f, 21.292835f, -1.072023f, 6.425335f, 16.068684f, 3.861220f,
    -7.170454f, -0.771846f, -10.235249f, -12.476522f, 7.519372f, -1.497815f, 7.548597f, 5.579000f, -13.576853f, -0.564930f, 4.355659f, -2.752280f, -1.280876f, -4.614651f, 3.479094f, 7.984504f, 5.871964f,
    -4.242005f, 15.179379f, 8.271146f, -2.489961f, -0.808152f, -9.176818f, 9.768644f, 3.199901f, -9.651007f, 1.068393f, -1.486685f, 7.585000f, -6.609848f, 4.503484f, -1.401268f, 1.196620f, -10.785708f, -11.879837f,
    -2.207071f, -2.126915f, -7.474844f, 4.355451f, -9.121566f, 16.523109f, 9.421859f, 0.021920f, 2.213993f, 1.971704f, 2.810292f, 5.038353f, 8.134138f, -11.729722f, 0.031108f, 0.112832f, 9.015647f, -7.045502f,
    10.502078f, -5.380768f, -6.279679f, 12.452871f, 3.337539f, 0.703993f, 4.444684f, 7.070218f, 1.712532f, 2.107954f, -10.317196f, 2.802624f, -16.954304f, -3.414618f, 5.843038f, -0.661985f, 12.012882f, -15.146476f,
    2.740827f, 3.680496f, -4.196952f, -14.678842f, 1.042201f, 0.107127f, -3.106902f, -1.590019f, 14.988041f, -5.154358f, 10.775966f, 3.436428f, 12.813007f, -3.957654f, 2.809988f, 11.328360f, 8.332334f, 9.991679f,
    -1.134552f, 6.474350f, -3.385157f, 21.300571f, 8.172812f, 8.915218f, -8.158613f, -19.225155f, -7.042034f, 8.722211f, 13.713666f, 8.575713f, -12.844994f, -8.669450f, -18.399305f, -9.773986f, 13.571878f,
    -9.779143f, -2.579859f, -12.759364f, -1.610966f, -2.139341f, 3.470644f, 16.714096f, -13.877205f, 4.066833f, 8.148497f, 9.285519f, 0.351547f, -7.003099f, -20.549627f, 0.941593f, 3.569152f, 1.931990f, 8.236025f,
    -3.337073f, -10.818482f, 1.965147f, 10.478598f, -9.620447f, -2.932661f, 1.353977f, 15.860100f, 3.752488f, -6.279715f, 3.324231f, 5.782442f, -22.392855f, 11.335500f, -0.089033f, 6.247707f, 9.711730f, 16.349276f,
    2.134033f, -1.448327f, -8.367188f, -4.871138f, 9.041019f, 26.180059f, -0.392080f, 0.322027f, -3.176687f, 6.473042f, 4.522851f, -8.236105f, -9.842262f, -3.618795f, 12.359117f, -21.706318f, -3.971055f,
    -19.499207f, 8.423275f, 10.034572f, 5.112028f, 11.600456f, -10.517344f, -1.387632f, -7.965416f, -9.217453f, 19.806374f, 8.228178f, 8.678469f, 16.642324f, -8.668985f, -7.956187f, -12.193502f, -22.152256f,
    12.675932f, -10.010987f, 3.361073f, -9.124073f, -15.260284f, 12.485716f, 12.958562f, -13.619774f, 8.417912f, 7.142402f, -7.522203f, -5.120552f, -2.965110f, 7.369909f, 9.897572f, 1.877501f, 4.178047f, 7.615440f,
    -11.148639f, -1.939979f, 0.303061f, -1.165344f, -2.296861f, 3.843472f, -5.638946f, -4.638227f, -11.896723f, -4.375220f, 21.049767f, 5.572012f, -11.566093f, -14.979873f, 8.458713f, 2.555419f, 4.753624f,
    6.625913f, 2.092041f, -1.668102f, -4.941959f, -8.351908f, 6.862656f, -6.874657f, -3.990863f, -7.999241f, -5.796680f, -3.485245f, -6.588869f, -2.157906f, 10.735359f, -11.858204f, -13.345760f, -2.434890f,
    2.745536f, -11.191664f, -5.932686f, -15.910933f, -1.385349f, 10.336471f, -0.275891f, 13.118786f, 3.545843f, 0.034237f, -4.556586f, 7.665683f, 6.126884f, 5.637043f, -5.865258f, -9.532327f, 10.780339f, 5.754178f,
    11.372564f, 3.193031f, 4.766507f, 4.548008f, -6.513889f, -5.217194f, -6.272741f, 5.128036f, 4.444829f, -9.063091f, -6.944401f, -16.705223f, -2.947636f, -3.461692f, -1.469410f, 13.365856f, -3.905179f, 9.879644f,
    5.201013f, 8.818437f, -11.379171f, 1.009672f, -0.004489f, 11.221715f, -6.115437f, 1.231313f, -16.635225f, -4.214061f, -4.791252f, -0.756395f, -8.591458f, 8.215440f, -0.641980f, -8.756137f, 14.324884f,
    0.764855f, 5.238774f, 16.714542f, -13.665570f, -9.534633f, 4.255009f, 10.484324f, 3.252735f, 4.013517f, 5.326964f, -1.725973f, 18.955566f, 17.491409f, -16.448341f, 11.678350f, -4.048868f, -0.795780f,
    -8.198077f, 9.193855f, -1.241781f, -7.551803f, -4.107594f, -15.316603f, -16.145117f, 7.170187f, -6.878034f, -5.917693f, -6.220591f, -3.035267f, -0.326719f, -7.775177f, -12.650038f, 10.240766f, -6.091671f,
    -24.977558f, 15.135868f, 3.370985f, -3.407732f, 1.606174f, 2.550914f, 1.215901f, -6.926246f, -11.390208f, 6.043550f, 1.209374f, -0.994154f, 8.529198f, -1.346966f, -2.605700f, -13.413307f, -5.744444f, 6.395872f,
    1.455137f, -9.873875f, -13.147046f, -9.714964f, -1.872904f, -11.741944f, 16.275520f, 5.703047f, -11.004894f, -15.399282f, -6.914268f, -5.537142f, -10.834431f, 16.318268f, -0.692072f, -1.181313f, -13.461952f,
    5.832011f, 1.393280f, -7.852055f, 8.244344f, -10.189405f, -14.926023f, -11.939293f, -1.356410f, 4.320812f, 0.311119f, -11.102482f, 5.607613f, 0.844352f, 21.080612f, -16.090446f, -12.560851f, 3.387702f,
    8.601639f, 14.178577f, 14.181447f, 5.150722f, 28.425617f, 9.512111f, 19.152340f, -18.184969f, 10.964733f, -5.295418f, 4.289782f, -15.871321f, 11.804911f, -17.260010f, -3.559654f, -5.911961f, 0.693298f,
    -3.158373f, -4.101953f, 6.210736f, -13.232613f, -14.046407f, 4.712640f, 5.089438f, 0.984821f, -5.189132f, 4.341965f, -5.283492f, -8.892274f, -12.795950f, -5.703321f, -5.590813f, 14.547821f, -0.166976f,
    -12.182299f, -5.312140f, 4.056026f, 15.746872f, 2.516945f, -2.716485f, 4.657195f, -6.109494f, 6.009631f, 3.124388f, -2.068037f, 11.140948f, -0.977704f, 12.281496f, 10.984852f, -8.746916f, 2.691792f, -2.632487f,
    25.072203f, 3.158863f, -1.394684f, -5.639797f, 8.868370f, -1.889892f, 10.844110f, -12.571439f, -7.753999f, -11.389962f, -2.728120f, 11.100060f, 3.180962f, -1.128060f, 3.134414f, 8.599652f, 1.675784f,
    14.772390f, 0.356016f, 2.995774f, 5.691812f, 13.569259f, 2.352538f, -12.837963f, -8.354870f, 1.482718f, -4.874431f, 1.231462f, 3.563626f, 1.739208f, 7.927014f, -3.344223f, 12.044106f, 13.560950f, -3.681587f,
    6.645124f, -4.575503f, 0.544202f, -14.823096f, 7.400783f, 7.862655f, -7.419325f, -8.787663f, -12.688564f, -4.171362f, -2.570014f, 7.053131f, 10.187328f, -0.966878f, -13.058752f, -28.651665f, 20.217552f,
    3.294838f, -9.435959f, 0.053485f, -4.159319f, 1.540250f, -1.403959f, 1.752246f, 2.166913f, 32.557487f, -16.182793f, 0.468291f, 26.276562f, 7.585394f, -6.858788f, -6.279219f, -5.742243f, -4.157490f, 11.292061f,
    3.674446f, 6.041906f, 5.812798f, 15.734926f, 3.461552f, 9.869953f, -15.834528f, -13.067029f, 2.823362f, 6.615796f, 0.441933f, -2.316038f, 4.804338f, -4.009411f, -13.475328f, -6.535377f, 3.879934f, -8.403879f,
    -12.259583f, -9.268803f, -2.067641f, 4.031195f, -3.109630f, -5.626819f, 0.461871f, 14.558437f, -0.483303f, -3.541791f, 3.190638f, -3.777116f, -3.223243f, -10.118240f, -8.797583f, 2.869958f, 7.309965f,
    -0.993078f, 8.576652f, -5.088981f, -10.338631f, 3.029601f, 4.252801f, 0.320264f, 1.923784f, -5.335409f, 13.375669f, 9.985000f, 7.478029f, -1.651823f, 6.190308f, -2.079115f, 6.018621f, 5.189635f, 2.401905f,
    4.431230f, 7.668587f, -19.173193f, 18.266272f, -18.360521f, -12.935114f, -2.582757f, 11.280110f, -6.859839f, -4.549446f, 8.364001f, 6.789788f, -21.657852f, -4.271647f, 11.158220f, -2.436972f, -17.469044f,
    -4.208299f, -10.329798f, 2.156145f, -12.774984f, 3.212620f, 3.037886f, -16.827484f, -9.845140f, 3.818673f, 17.203400f, 9.832920f, 5.934397f, -5.083592f, 3.475554f, -5.561733f, 0.776478f, 10.148176f, 4.492512f,
    1.910677f, -2.131890f, 9.113445f, 0.599067f, -0.242363f, -26.337231f, -3.263350f, 0.687130f, -7.282282f, 3.079349f, -9.499436f, -16.636003f, -2.671493f, -0.777328f, -6.914995f, 10.500852f, -2.502230f,
    -0.719137f, 1.376765f, -5.181725f, -8.208481f, -7.845020f, -7.573354f, -10.313904f, -11.906903f, 13.861981f, 3.580997f, 6.031216f, 6.077519f, 15.331321f, 0.357388f, -14.012940f, -4.717149f, -2.833182f,
    -0.775877f, 8.371817f, -6.383162f, -1.499406f, 8.377569f, -0.794452f, -4.240912f, -9.828636f, -10.524963f, -2.372848f, 11.295761f, -0.639428f, 3.662898f, -2.714319f, 12.051702f, 8.003572f, 0.826499f, 2.833596f,
    -13.668466f, 4.278912f, -16.705139f, -0.779707f, -11.095739f, 2.261742f, 11.068474f, 0.047263f, 5.635440f, -12.313726f, -13.106979f, 3.873217f, -16.232141f, -13.041805f, -23.492167f, 2.983342f, 3.558033f,
    -6.645183f, 9.957177f, -0.073426f, -13.229465f, -1.064726f, -14.242373f, 5.536783f, -12.919047f, -2.120899f, -8.116956f, -27.900700f, -1.066976f, -6.575535f, -13.384243f, -6.560703f, -16.250937f, -6.646844f,
    -6.691673f, 9.241119f, 0.225810f, 18.391247f, 7.395995f, -1.287890f, 19.054008f, -5.715661f, -5.048567f, 2.344202f, -6.588252f, 11.256475f, -6.049391f, 4.045815f, 3.998632f, -20.701256f, -10.341751f,
    -0.544581f, 0.017565f, -6.092354f, 12.733155f, 24.700790f, 7.476691f, 0.762488f, 13.174433f, -15.142201f, 6.910282f, 5.366340f, 5.669908f, 0.414176f, 1.985034f, -13.373807f, 5.568107f, 9.497774f, 3.204105f,
    -20.152243f, 4.725494f, -7.348037f, 3.916818f, -1.296995f, -12.213487f, 8.221561f, 1.754011f, -4.121859f, 7.804862f, -8.287487f, -10.479917f, 8.574441f, -12.742554f, -3.363132f, -1.422361f, 4.681566f,
    -9.709182f, 9.568750f, 13.742260f, 7.489975f, -2.485489f, 2.935714f, 0.463926f, 9.267967f, -1.602762f, 0.442492f, 12.158168f, 9.718264f, 12.200609f, -1.031067f, -10.341888f, 7.128188f, -2.149584f, 20.747372f,
    7.306457f, 14.035366f, 2.416203f, -7.104143f, 3.253518f, -2.852659f, 10.821512f, 21.560673f, 1.266267f, -26.595943f, 4.416551f, -14.230844f, -16.486519f, -11.734299f, -15.852049f, 21.519890f, -2.928228f,
    13.661053f, -12.267429f, -0.805571f, 5.517809f, 4.549609f, -8.689898f, -19.775799f, -7.026294f, -4.584817f, -2.394832f, 16.531902f, -0.179000f, 6.626202f, 10.940154f, 4.219591f, -4.187131f, -6.284671f,
    -6.340464f, -21.896942f, 10.202515f, -4.255267f, 8.263470f, -14.898003f, -1.105536f, 0.053600f, 14.112831f, 11.235374f, 5.930917f, -13.466118f, -3.213318f, -2.325147f, -1.289462f, 3.162030f, 2.084447f,
    12.846682f, 5.964324f, 6.997753f, -5.677891f, 0.698119f, -17.831364f, 12.404846f, 14.786201f, 2.941057f, 6.952736f, -11.061385f, 8.499249f, -14.642521f, 8.342108f, -3.968000f, 12.133200f, 13.020239f,
    -0.954907f, 4.582722f, 9.773862f, -6.279247f, -11.643168f, 8.018866f, -8.449734f, 6.491271f, 17.338291f, -11.755928f, 1.245089f, 12.383610f, -0.589201f, -4.756158f, 12.553598f, 2.921804f, 16.465017f, 5.548570f,
    11.316789f, 4.846970f, 6.763668f, 6.779863f, 7.530643f, 0.748198f, 6.279439f, -11.895365f, 7.570137f, 3.220422f, 11.458266f, -3.896332f, 8.362011f, 0.568521f, 8.239796f, -9.562937f, -21.990992f, -10.020023f,
    -4.209702f, 2.995760f, -13.396329f, -19.800282f, 10.039207f, -22.055046f, -11.676407f, 13.996929f, -10.405979f, 2.745614f, 13.814919f, -18.389786f, -7.177250f, -3.519151f, 6.955134f, 0.179758f, -3.178502f,
    11.562584f, 3.169524f, 1.324303f, 0.706274f, -3.197940f, 5.881256f, 16.020313f, -14.099644f, -4.321495f, -9.312614f, -8.507051f, 1.362962f, 2.656377f, -20.505608f, 5.264695f, 3.884532f, -1.868845f, -7.470690f,
    0.516871f, -5.116251f, -0.018183f, -3.804091f, 4.298658f, -15.950325f, -1.237135f, 11.154274f, 18.794483f, 5.255928f, 16.106724f, 0.031791f, 4.005330f, -9.762729f, 2.104908f, 13.291064f, 8.587060f, -12.818718f,
    -3.692799f, -13.215702f, -5.158340f, -5.368900f, -3.357782f, 12.069958f, -5.589484f, -6.900913f, -0.417992f, -3.883848f, 16.113585f, -1.390243f, -12.959792f, 2.083207f, -6.470685f, 6.437883f, 8.247831f,
    14.140997f, -8.639792f, 7.746149f, 11.594313f, -0.459053f, -8.772566f, -3.576654f, 16.389486f, 8.276430f, -18.157822f, -4.824405f, -4.951690f, 9.329051f, -6.752484f, 0.145874f, 0.203534f, -14.930539f,
    -13.559370f, 4.466868f, 5.400437f, -5.050427f, -2.518449f, 1.542810f, 5.127673f, 12.431385f, 22.776434f, -3.599356f, 1.836551f, -5.402380f, 3.359248f, -1.252080f, 7.232835f, 4.270503f, 8.558823f, -10.387476f,
    17.368423f, -0.429510f, 4.682260f, -3.140907f, 11.846124f, -2.622700f, -3.957729f, -2.092712f, 3.858035f, 0.180265f, 6.661385f, 4.869834f, -4.012877f, -8.355537f, 9.306880f, -12.514816f, 1.212422f, 15.257447f,
    3.051450f, -3.608595f, -3.036387f, 12.523347f, -0.201355f, -16.926735f, -11.184448f, -21.146818f, 2.629846f, 10.265030f, 2.356806f, 2.752682f, 1.548798f, 2.598051f, -0.304085f, 2.925230f, 3.365304f, -1.405644f,
    -0.083739f, 1.121074f, -5.275834f, 2.525005f, 3.628125f, 9.485579f, -2.274559f, -2.371589f, -8.196259f, 27.864498f, -14.216352f, -0.420671f, -9.927025f, 11.932823f, -11.762934f, -3.733866f, 3.182370f,
    6.895154f, 2.145975f, -15.098987f, 12.519497f, -13.984479f, -3.102676f, 3.400337f, -2.670371f, -0.793257f, -6.244378f, -2.535935f, 9.626478f, 10.397112f, -13.632895f, -1.087554f, 4.150095f, -0.546089f,
    -5.741489f, 0.171951f, 1.830555f, 2.776227f, -4.710307f, -9.585217f, -2.327785f, -15.861672f, 17.749914f, -1.046235f, -2.694619f, 0.112400f, 2.776806f, -4.486475f, -2.213065f, 1.368459f, -7.420490f, -6.157038f,
    3.865466f, -5.947114f, -1.802527f, -5.598177f, -10.722310f, 0.380622f, -16.703381f, -2.963318f, 3.594397f, -8.908564f, -0.834254f, -3.341107f, -2.036427f, -13.430461f, -6.711573f, -2.595264f, 7.802629f,
    -5.589416f, 6.207128f, 1.140080f, 14.632993f, -2.062725f, -11.813453f, -6.940236f, -9.387502f, -1.363446f, -5.311069f, 2.440516f, -1.333333f, 13.449287f, 3.478777f, 4.298302f, 25.182278f, -1.115195f,
    10.501851f, -3.698123f, 10.065781f, -14.919846f, -25.568735f, -5.928421f, 27.399586f, 0.705564f, 9.641000f, -7.905223f, 4.611671f, 4.699477f, -16.398651f, 14.653256f, 0.943585f, -0.268378f, 8.109201f,
    9.045616f, 14.659944f, -2.183798f, 11.168876f, 5.741721f, 13.466798f, 8.682743f, -5.284838f, 0.457775f, 4.759321f, 1.980847f, 3.960200f, -0.028601f, 2.821799f, 14.713168f, -4.675743f, 4.085332f, 4.237683f,
    -5.416162f, -0.609955f, -17.136200f, -0.027121f, -10.048325f, 1.293943f, -9.221786f, 3.908639f, 1.617599f, -1.322240f, 6.409557f, 11.352152f, -6.112571f, -4.714913f, 4.270394f, 10.397964f, 4.804108f,
    -1.476283f, -10.391253f, 1.377846f, -3.146652f, -5.834240f, -8.282748f, -8.327962f, 9.801763f, 6.365228f, -1.971733f, -9.823537f, -3.147905f, -13.541558f, -17.178219f, 9.590825f, 7.256647f, 7.251298f,
    -7.386436f, 6.582705f, -11.270920f, 8.052238f, -3.623482f, -12.901483f, 2.393135f, -9.297455f, 1.521010f, 1.391591f, -15.919291f, 17.672735f, 6.691560f, -10.993683f, -9.250460f, 0.768708f, 6.869545f, 9.356607f,
    -6.635333f, 11.168291f, -26.011236f, -8.380869f, 2.570391f, -1.260646f, 17.574766f, 10.677329f, 23.141562f, 7.379807f, 5.853816f, -2.358614f, -6.440468f, -15.789014f, 13.311720f, -17.724472f, 1.813706f,
    5.900202f, 2.293748f, -3.016102f, 9.619631f, -1.717712f, 13.706371f, 12.267218f, -13.949120f, 8.129120f, 11.246664f, 6.903168f, 3.954795f, 12.456059f, -11.152524f, -5.994295f, -9.981610f, -0.275137f,
    -1.853362f, 9.898179f, 6.661303f, 5.177972f, 5.561171f, 14.197105f, 3.106709f, 16.163628f, -10.829681f, -1.928774f, 0.743435f, -7.824144f, 3.189177f, -1.422755f, -10.652393f, 0.886980f, 10.458999f, 2.344340f,
    -8.690491f, 16.425613f, -5.902361f, 14.494605f, -11.412006f, 4.058489f, 10.352727f, 2.720972f, 8.097337f, -12.473856f, -4.056056f, -3.889349f, -5.336528f, 1.877571f, 7.826597f, -7.789737f, -18.712654f,
    -15.234234f, 1.983776f, 12.424959f, -10.959620f, 3.710872f, 6.692961f, 1.688269f, -7.328160f, 6.973325f, 9.278824f, 1.205807f, -0.475508f, -0.545236f, -4.758235f, -3.902059f, 4.420048f, 3.724769f, 5.872193f,
    2.484561f, -13.028065f, -7.551498f, -7.084496f, 3.319983f, 3.721687f, -6.944470f, 5.286321f, -13.649103f, 18.305599f, 8.798578f, 11.707449f, 12.794806f, 1.162092f, -1.867021f, 9.861185f, -9.866682f, -1.943836f,
    -14.769156f, 0.063251f, -0.786058f, -9.130911f, 5.531267f, -13.237419f, 1.257298f, -5.333574f, -15.388750f, -8.316410f, 14.561322f, -7.998179f, -10.949454f, -5.151754f, 5.638205f, -0.089617f, 1.181876f,
    0.497534f, -6.051322f, 5.464210f, -4.689215f, 9.773981f, 13.662493f, -6.477097f, -0.228551f, 2.186498f, 12.210072f, -5.967360f, -9.676385f, -1.880383f, 17.335300f, 2.846212f, -5.893623f, 14.143561f, 5.893929f,
    16.556017f, -5.811418f, 11.956698f, 3.931551f, -2.157955f, -1.801676f, -7.480111f, 10.153038f, -12.993446f, -7.482516f, -2.688586f, 3.294284f, 6.534150f, 2.358585f, 1.670267f, 4.352794f, 2.101273f, -4.415938f,
    0.316740f, 2.899897f, 5.119410f, -3.741017f, -6.059538f, -9.930637f, -11.416687f, 1.670432f, 5.637441f, -6.764245f, 13.265562f, 1.916031f, -16.351128f, -10.126958f, 4.351907f, 7.798675f, -6.252160f, 4.121194f,
    -0.866847f, -14.491129f, 2.121340f, 4.618024f, -12.830911f, 4.129667f, 8.896847f, -8.216023f, -5.027400f, 0.328928f, -4.365849f, 1.227777f, -1.651028f, -0.475597f, -7.541091f, -1.100914f, 4.087202f, 8.477075f,
    12.797399f, 3.398986f, -6.992332f, 13.904685f, -3.648373f, -1.634019f, 2.695182f, 5.682346f, 5.896292f, -2.068223f, 4.144853f, -8.292144f, -6.298479f, -8.600998f, -5.556418f, -16.858938f, -6.226583f,
    -5.462989f, 0.516673f, 4.439364f, -4.019507f, -3.133490f, -2.514023f, 0.054914f, 1.883955f, 1.751921f, 0.096201f, 20.065672f, 4.586689f, 3.564039f, 2.059176f, 6.628250f, -9.554339f, -8.037239f, 12.284419f,
    12.115036f, 0.331345f, 6.271217f, -3.708438f, -5.629647f, 16.332115f, 5.162773f, -21.082886f, 10.216719f, 12.954202f, 13.404097f, 4.200478f, 7.896248f, 9.476270f, -4.312825f, 1.988040f, -3.185658f, 2.899301f,
    5.254571f, 10.982389f, 7.264908f, 1.590444f, 6.149549f, 1.174282f, -14.595101f, 2.739235f, -2.508319f, 2.358624f, 3.342335f, -12.831897f, 15.683235f, 16.191208f, 6.735896f, 8.710923f, 6.148141f, 1.439414f,
    -5.648272f, 6.757202f, 1.045871f, -8.508298f, -8.782381f, -25.359898f, 2.179063f, 8.410498f, 17.268856f, 0.495325f, -20.521408f, 4.621348f, -11.704700f, -6.043191f, -0.582783f, -20.475950f, -5.715844f,
    12.834491f, -2.652349f, 5.033823f, 2.340654f, 6.700262f, -14.302177f, -0.270791f, 28.023287f, -9.549105f, -12.030479f, 6.946150f, -3.588696f, 14.061947f, -2.013532f, -11.873772f, 1.353410f, -2.344524f,
    6.375873f, 1.953676f, -4.420763f, 13.521021f, -2.871043f, -3.070899f, -5.358984f, -6.811697f, -11.715629f, 3.568241f, -17.493025f, -1.836295f, -10.034955f, -1.147791f, -4.455373f, -1.408509f, -3.553368f,
    6.539475f, -10.623294f, 6.806676f, 1.261220f, -7.462727f, 2.584891f, 13.039749f, -6.293489f, -1.605095f, 6.507571f, 1.241756f, 5.570013f, -3.365613f, -17.752159f, -8.953832f, 18.427387f, -6.731126f, 7.232294f,
    -15.489716f, -2.359913f, -8.552725f, -5.620982f, -8.639449f, -2.389172f, 18.426708f, 3.283449f, 10.888176f, 4.929100f, -19.198694f, 26.070389f, 7.224378f, -3.783723f, 0.712699f, -1.850544f, 10.137056f,
    -18.182098f, -15.975182f, -4.552053f, -6.702194f, 10.006070f, -5.322373f, -3.597058f, -13.348425f, -12.041620f, 13.060863f, -3.499395f, 0.765266f, -12.264727f, -5.723804f, 4.514238f, 0.850732f, -1.103984f,
    0.847270f, 10.538741f, 4.593394f, -1.553360f, -11.345526f, -5.263173f, -1.113987f, 12.679811f, 5.925968f, 1.971790f, 13.940191f, 10.292322f, 0.355375f, 19.611076f, -8.424827f, -11.687172f, -18.891113f,
    2.393855f, -9.945542f, -11.726484f, -13.368456f, -6.568118f, 1.182465f, -0.551943f, -7.256127f, -3.490950f, 5.375882f, -6.649831f, 13.135878f, -3.311618f, 10.354960f, -21.770927f, -4.258286f, 3.138621f,
    4.169222f, -8.726085f, 9.294991f, -0.686998f, 2.246574f, 16.332043f, -1.335098f, 2.663138f, 12.019674f, 0.919391f, -6.907223f, -5.967362f, -6.726013f, 1.313580f, 1.611054f, -16.476580f, -4.702639f, -12.260868f,
    -8.682915f, 11.797264f, -7.284854f, -5.695921f, -1.863089f, -2.952110f, -2.177166f, -5.166656f, -5.099727f, 5.169176f, -0.862654f, -0.949327f, -4.687287f, -2.501458f, 4.392480f, -10.156492f, 9.719406f,
    1.226393f, 0.898899f, 0.251033f, 7.561624f, -14.763690f, 11.299557f, 6.078887f, 16.584248f, 6.437543f, 10.429852f, 9.103986f, -19.822737f, 2.100945f, 3.008302f, 5.546509f, -14.715752f, 2.785495f, 4.938319f,
    5.929581f, -1.960434f, 13.707758f, -11.475433f, -4.025057f, -3.106235f, -2.305663f, 16.823288f, -1.604130f, -8.821331f, 11.362504f, -3.675623f, -3.909241f, 4.646163f, 13.585918f, 2.180281f, -1.459380f,
    -10.670804f, -13.885897f, -0.344814f, 4.373363f, -19.020521f, -19.740833f, -7.813461f, -0.856105f, -1.683899f, 2.182599f, -1.466677f, 8.562784f, 12.568027f, 1.212156f, 10.073949f, 2.288739f, -5.109566f,
    -5.142433f, -1.628014f, 1.020448f, -5.334638f, 10.375928f, -26.571148f, -8.112627f, 11.550373f, 1.862268f, -8.713111f, -4.731730f, -12.546883f, 1.562598f, 1.329782f, -0.381532f, 3.314681f, 16.218182f,
    5.001511f, -11.350945f, 16.440815f, 23.605942f, 5.438505f, -2.492342f, 8.220360f, 0.528545f, -7.632623f, 11.436621f, 9.841532f, 4.928565f, -6.163303f, -1.671910f, -18.164196f, 4.653586f, 10.978483f,
    -16.434076f, 4.563122f, -20.772842f, -2.826509f, -38.059719f, 16.312782f, 19.107128f, -0.058033f, -7.757068f, -12.087646f, -11.139393f, -1.858025f, 3.601321f, 13.269556f, -4.567130f, -3.610200f, 7.646561f,
    -7.227961f, -20.045303f, 11.720284f, 23.644505f, -0.153076f, -6.708123f, -0.401558f, 5.044775f, -9.012714f, -5.910140f, 2.517138f, -1.950141f, 9.340166f, -1.576881f, -0.909173f, 3.749623f, -20.140862f,
    -4.877320f, 8.054358f, 3.703845f, -4.364082f, 13.542617f, -9.864304f, 4.844968f, 8.012321f, -16.738960f, -2.056262f, -4.877666f, -11.611341f, 2.055205f, -4.764958f, -13.686033f, -12.985853f, -0.046583f,
    14.221796f, -3.664927f, 6.171565f, 7.462372f, 5.963862f, -3.698985f, 1.523634f, -7.164260f, 4.689973f, -5.104511f, 6.056950f, -9.853289f, 6.343823f, 10.585016f, 0.859324f, -26.888433f, -12.641256f, -15.337730f,
    -5.413608f, -16.642197f, 12.340921f, 2.494469f, -4.610116f, 2.209176f, 6.676560f, -0.715086f, 19.455254f }, { 0 }, 1, 1, 1024 };

void set_values_struct(std::vector<ScalarType>& input, std::vector<ScalarType>& output,
    unsigned int& rows, unsigned int& cols, unsigned int& batch_size, testData& data);

void set_values_struct(std::vector<ScalarType>& input, std::vector<ScalarType>& output,
    unsigned int& rows, unsigned int& cols, unsigned int& batch_size, testData& data)
{
  unsigned int size = data.col_num * data.batch_num * 2;
  input.resize(size);
  output.resize(size);
  rows = data.row_num;
  cols = data.col_num;
  batch_size = data.batch_num;
  for (unsigned int i = 0; i < size; i++)
  {
    input[i] = data.input[i];
    output[i] = data.output[i];
  }

}

void set_values_struct(std::vector<ScalarType>& input, std::vector<ScalarType>&input2,
    std::vector<ScalarType>& output, unsigned int& rows, unsigned int& cols,
    unsigned int& batch_size, testData& data);

void set_values_struct(std::vector<ScalarType>& input, std::vector<ScalarType>&input2,
    std::vector<ScalarType>& output, unsigned int& rows, unsigned int& cols,
    unsigned int& batch_size, testData& data)
{
  unsigned int size = data.col_num * data.batch_num * 2;
  input.resize(size);
  input2.resize(size);
  output.resize(size);
  rows = data.row_num;
  cols = data.col_num;
  batch_size = data.batch_num;
  for (unsigned int i = 0; i < size; i++)
  {

    input[i] = data.input[i];
    input2[i] = data.output[i];
    output[i] = data.result_multiply[i];
  }
}

void read_vectors_pair(std::vector<ScalarType>& input, std::vector<ScalarType>& output,
    unsigned int& rows, unsigned int& cols, unsigned int& batch_size, const std::string& log_tag);

void read_vectors_pair(std::vector<ScalarType>& input, std::vector<ScalarType>& output,
    unsigned int& rows, unsigned int& cols, unsigned int& batch_size, const std::string& log_tag)
{

  if (log_tag == "fft::direct" || log_tag == "fft::convolve::1" || log_tag == "fft::bluestein::1"
      || log_tag == "fft::fft_reverse_direct")
    set_values_struct(input, output, rows, cols, batch_size, cufft);

  if (log_tag == "fft:real_to_complex")
    set_values_struct(input, output, rows, cols, batch_size, real_to_complex_data);

  if (log_tag == "fft:complex_to_real")
    set_values_struct(input, output, rows, cols, batch_size, complex_to_real_data);

  if (log_tag == "fft::batch::direct" || log_tag == "fft::batch::radix2")
    set_values_struct(input, output, rows, cols, batch_size, batch_radix);

  if (log_tag == "fft::radix2" || log_tag == "fft::convolve::2" || log_tag == "fft::bluestein::2"
      || log_tag == "fft::fft_ifft_radix2" || log_tag == "fft::ifft_fft_radix2")
    set_values_struct(input, output, rows, cols, batch_size, radix2_data);

}

void read_vectors_three(std::vector<ScalarType>& input, std::vector<ScalarType>&input2,
    std::vector<ScalarType>& output, unsigned int& rows, unsigned int& cols,
    unsigned int& batch_size, const std::string& log_tag);

void read_vectors_three(std::vector<ScalarType>& input, std::vector<ScalarType>&input2,
    std::vector<ScalarType>& output, unsigned int& rows, unsigned int& cols,
    unsigned int& batch_size, const std::string& log_tag)
{
  if (log_tag == "fft::multiplt::complex")
    set_values_struct(input, input2, output, rows, cols, batch_size, cufft);

}

template<typename ScalarType>
ScalarType diff(std::vector<ScalarType>& vec, std::vector<ScalarType>& ref)
{
  ScalarType df = 0.0;
  ScalarType norm_ref = 0;

  for (std::size_t i = 0; i < vec.size(); i++)
  {
    df = df + pow(vec[i] - ref[i], 2);
    norm_ref += ref[i] * ref[i];
  }

  return sqrt(df / norm_ref);
}

template<typename ScalarType>
ScalarType diff_max(std::vector<ScalarType>& vec, std::vector<ScalarType>& ref)
{
  ScalarType df = 0.0;
  ScalarType mx = 0.0;
  ScalarType norm_max = 0;

  for (std::size_t i = 0; i < vec.size(); i++)
  {
    df = std::max<ScalarType>(std::fabs(ScalarType(vec[i] - ref[i])), df);
    mx = std::max<ScalarType>(std::fabs(ScalarType(vec[i])), mx);

    if (mx > 0)
    {
      if (norm_max < df / mx)
        norm_max = df / mx;
    }
  }

  return norm_max;
}

template<class SCALARTYPE, unsigned int ALIGNMENT>
void copy_to_vector(std::complex<SCALARTYPE> * input_complex,
    viennacl::vector<SCALARTYPE, ALIGNMENT>& in, unsigned int size)
{
  for (unsigned int i = 0; i < size; i += 1)
  {
    in(i * 2) = (SCALARTYPE) std::real(input_complex[i]);
    in(i * 2 + 1) = std::imag(input_complex[i]);
  }
}
template<class SCALARTYPE>
void copy_to_vector(std::complex<SCALARTYPE> * input_complex, SCALARTYPE * in, unsigned int size)
{
#ifdef VIENNACL_WITH_OPENMP
#pragma omp parallel for if (size > VIENNACL_OPENMP_VECTOR_MIN_SIZE)
#endif
  for (unsigned int i = 0; i < size; i += 1)
  {
    in[i * 2] = (SCALARTYPE) std::real(input_complex[i]);
    in[i * 2 + 1] = std::imag(input_complex[i]);
  }
}
template<class SCALARTYPE>
void copy_to_complex_array(std::complex<SCALARTYPE> * input_complex, const SCALARTYPE * in, int size)
{
#ifdef VIENNACL_WITH_OPENMP
#pragma omp parallel for if (size > VIENNACL_OPENMP_VECTOR_MIN_SIZE)
#endif
  for (int i = 0; i < size * 2; i += 2)
  { //change array to complex array
    input_complex[i / 2] = std::complex<SCALARTYPE>(in[i], in[i + 1]);
  }
}

void convolve_ref(std::vector<ScalarType>& in1, std::vector<ScalarType>& in2, std::vector<ScalarType>& out);

void convolve_ref(std::vector<ScalarType>& in1, std::vector<ScalarType>& in2, std::vector<ScalarType>& out)
{
  out.resize(in1.size());
  unsigned int data_size = static_cast<unsigned int>(in1.size()) >> 1;

  for (unsigned int n = 0; n < data_size; n++)
  {
    std::complex<ScalarType> el;
    for (unsigned int k = 0; k < data_size; k++)
    {
      int offset = int(n) - int(k);
      if (offset < 0)
        offset += data_size;
      std::complex<ScalarType> m1(in1[2 * k], in1[2 * k + 1]);
      std::complex<ScalarType> m2(in2[2 * std::size_t(offset)], in2[2 * std::size_t(offset) + 1]);
      el = el + m1 * m2;
    }
    out[2 * n] = el.real();
    out[2 * n + 1] = el.imag();
  }
}


ScalarType fft(std::vector<ScalarType>& in, std::vector<ScalarType>& out, unsigned int /*row*/,
    unsigned int /*col*/, unsigned int batch_size);

ScalarType fft(std::vector<ScalarType>& in, std::vector<ScalarType>& out, unsigned int /*row*/,
    unsigned int /*col*/, unsigned int batch_size)
{
  viennacl::vector<ScalarType> input(in.size());
  viennacl::vector<ScalarType> output(in.size());

  std::vector<ScalarType> res(in.size());

  viennacl::fast_copy(in, input);

  viennacl::fft(input, output, batch_size);

  viennacl::backend::finish();
  viennacl::fast_copy(output, res);

  return diff_max(res, out);
}

ScalarType direct(std::vector<ScalarType>& in, std::vector<ScalarType>& out, unsigned int /*row*/,
    unsigned int /*col*/, unsigned int batch_num);

ScalarType direct(std::vector<ScalarType>& in, std::vector<ScalarType>& out, unsigned int /*row*/,
    unsigned int /*col*/, unsigned int batch_num)
{
  viennacl::vector<ScalarType> input(in.size());
  viennacl::vector<ScalarType> output(in.size());

  std::vector<ScalarType> res(in.size());

  viennacl::fast_copy(in, input);

  unsigned int size = (static_cast<unsigned int>(input.size()) >> 1) / batch_num;

  viennacl::linalg::direct(input, output, size, size, batch_num);
  viennacl::backend::finish();
  viennacl::fast_copy(output, res);

  return diff_max(res, out);
}

ScalarType bluestein(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int batch_size);

ScalarType bluestein(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int batch_size)
{
  viennacl::vector<ScalarType> input(in.size());
  viennacl::vector<ScalarType> output(in.size());

  std::vector<ScalarType> res(in.size());

  viennacl::fast_copy(in, input);

  viennacl::linalg::bluestein(input, output, batch_size);

  viennacl::backend::finish();
  viennacl::fast_copy(output, res);

  return diff_max(res, out);
}

ScalarType radix2(std::vector<ScalarType>& in, std::vector<ScalarType>& out, unsigned int /*row*/,
    unsigned int /*col*/, unsigned int batch_num);

ScalarType radix2(std::vector<ScalarType>& in, std::vector<ScalarType>& out, unsigned int /*row*/,
    unsigned int /*col*/, unsigned int batch_num)
{
  viennacl::vector<ScalarType> input(in.size());
  viennacl::vector<ScalarType> output(in.size());

  std::vector<ScalarType> res(in.size());

  viennacl::fast_copy(in, input);

  unsigned int size = (static_cast<unsigned int>(input.size()) >> 1) / batch_num;
  viennacl::linalg::radix2(input, size, size, batch_num);

  viennacl::backend::finish();
  viennacl::fast_copy(input, res);

  return diff_max(res, out);
}

ScalarType fft_ifft_radix2(std::vector<ScalarType>& in, std::vector<ScalarType>& /*out*/,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int batch_num);

ScalarType fft_ifft_radix2(std::vector<ScalarType>& in, std::vector<ScalarType>& /*out*/,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int batch_num)
{
  viennacl::vector<ScalarType> input(in.size());

  std::vector<ScalarType> res(in.size());

  viennacl::fast_copy(in, input);

  unsigned int size = (static_cast<unsigned int>(input.size()) >> 1) / batch_num;
  viennacl::linalg::radix2(input, size, size, batch_num);

  viennacl::inplace_ifft(input);

  viennacl::backend::finish();
  viennacl::fast_copy(input, res);

  return diff_max(res, in);
}

ScalarType ifft_fft_radix2(std::vector<ScalarType>& in, std::vector<ScalarType>& /*out*/,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int batch_num);

ScalarType ifft_fft_radix2(std::vector<ScalarType>& in, std::vector<ScalarType>& /*out*/,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int batch_num)
{
  viennacl::vector<ScalarType> input(in.size());

  std::vector<ScalarType> res(in.size());

  viennacl::fast_copy(in, input);

  viennacl::inplace_ifft(input);

  unsigned int size = (static_cast<unsigned int>(input.size()) >> 1) / batch_num;
  viennacl::linalg::radix2(input, size, size, batch_num);

  viennacl::backend::finish();
  viennacl::fast_copy(input, res);

  return diff_max(res, in);
}

ScalarType fft_reverse_direct(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_num*/);

ScalarType fft_reverse_direct(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_num*/)
{
  viennacl::vector<ScalarType> input(in.size());

  viennacl::fast_copy(in, input);

  viennacl::vector<ScalarType> tmp(out.size());
  viennacl::fast_copy(in, tmp);
  viennacl::linalg::reverse(tmp);
  viennacl::linalg::reverse(tmp);

  viennacl::fast_copy(tmp, out);

  viennacl::backend::finish();

  return diff_max(out, in);
}

ScalarType real_to_complex(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_num*/);

ScalarType real_to_complex(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_num*/)
{

  viennacl::vector<ScalarType> input(in.size());
  viennacl::vector<ScalarType> output(in.size());

  std::vector<ScalarType> res(in.size());
  viennacl::fast_copy(in, input);

  viennacl::linalg::real_to_complex(input, output, input.size() / 2);

  viennacl::backend::finish();
  viennacl::fast_copy(output, res);

  return diff_max(res, out);

}


ScalarType complex_to_real(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_num*/);

ScalarType complex_to_real(std::vector<ScalarType>& in, std::vector<ScalarType>& out,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_num*/)
{

  viennacl::vector<ScalarType> input(in.size());
  viennacl::vector<ScalarType> output(in.size());

  std::vector<ScalarType> res(in.size());
  viennacl::fast_copy(in, input);

  viennacl::linalg::complex_to_real(input, output, input.size() / 2);

  viennacl::backend::finish();
  viennacl::fast_copy(output, res);

  return diff_max(res, out);

}

ScalarType multiply_complex(std::vector<ScalarType>& in, std::vector<ScalarType>& in2,
    std::vector<ScalarType>& out);

ScalarType multiply_complex(std::vector<ScalarType>& in, std::vector<ScalarType>& in2,
    std::vector<ScalarType>& out)
{

  std::vector<ScalarType> res(out.size());

  viennacl::vector<ScalarType> input1(in.size());
  viennacl::vector<ScalarType> input2(out.size());
  viennacl::vector<ScalarType> output(out.size());

  viennacl::fast_copy(in, input1);
  viennacl::fast_copy(in2, input2);
  viennacl::linalg::multiply_complex(input1, input2, output);

  std::cout << std::endl;

  viennacl::vector<ScalarType> tmp(out.size());

  viennacl::fast_copy(output, res);

  return diff_max(res, out);
}

ScalarType convolve(std::vector<ScalarType>& in1, std::vector<ScalarType>& in2,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_size*/);

ScalarType convolve(std::vector<ScalarType>& in1, std::vector<ScalarType>& in2,
    unsigned int /*row*/, unsigned int /*col*/, unsigned int /*batch_size*/)
{
  //if (in1.size() > 2048) return -1;
  viennacl::vector<ScalarType> input1(in1.size());
  viennacl::vector<ScalarType> input2(in2.size());
  viennacl::vector<ScalarType> output(in1.size());

  viennacl::fast_copy(in1, input1);
  viennacl::fast_copy(in2, input2);

  viennacl::linalg::convolve(input1, input2, output);

  viennacl::backend::finish();
  std::vector<ScalarType> res(in1.size());
  viennacl::fast_copy(output, res);

  std::vector<ScalarType> ref(in1.size());
  convolve_ref(in1, in2, ref);

  return diff_max(res, ref);
}

int test_correctness(const std::string& log_tag, input_function_ptr input_function,
    test_function_ptr func);

int test_correctness(const std::string& log_tag, input_function_ptr input_function,
    test_function_ptr func)
{

  std::vector<ScalarType> input;
  std::vector<ScalarType> output;

  std::cout << std::endl;
  std::cout << "*****************" << log_tag << "***************************\n";

  unsigned int batch_size;
  unsigned int rows_num, cols_num;

  input_function(input, output, rows_num, cols_num, batch_size, log_tag);
  ScalarType df = func(input, output, rows_num, cols_num, batch_size);
  printf("%7s ROWS=%6d COLS=%6d; BATCH=%3d; DIFF=%3.15f;\n", ((fabs(df) < EPS) ? "[Ok]" : "[Fail]"),
      rows_num, cols_num, batch_size, df);
  std::cout << std::endl;

  if (df > EPS)
    return EXIT_FAILURE;

  return EXIT_SUCCESS;
}

int testcorrectnes_multiply(const std::string& log_tag);

int testcorrectnes_multiply(const std::string& log_tag)
{
  std::vector<ScalarType> input;
  std::vector<ScalarType> input2;
  std::vector<ScalarType> output;
  unsigned int batch_size;
  unsigned int rows_num, cols_num;
  std::cout << std::endl;
  std::cout << "*****************" << log_tag << "***************************\n";
  read_vectors_three(input, input2, output, rows_num, cols_num, batch_size, log_tag);
  ScalarType df = multiply_complex(input, input2, output);
  printf("%7s ROWS=%6d COLS=%6d; BATCH=%3d; DIFF=%3.15f;\n", ((fabs(df) < EPS) ? "[Ok]" : "[Fail]"),
      rows_num, cols_num, batch_size, df);
  std::cout << std::endl;

  if (df > EPS)
    return EXIT_FAILURE;

  return EXIT_SUCCESS;
}

int main()
{
  std::cout << "*" << std::endl;
  std::cout << "* ViennaCL test: FFT" << std::endl;
  std::cout << "*" << std::endl;

  //1D FFT tests
  if (test_correctness("fft::direct", read_vectors_pair, &direct) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::batch::direct", read_vectors_pair, &direct) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::radix2", read_vectors_pair, &radix2) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::fft_ifft_radix2", read_vectors_pair, &fft_ifft_radix2) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::ifft_fft_radix2", read_vectors_pair, &ifft_fft_radix2) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::batch::radix2", read_vectors_pair, &radix2) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::convolve::1", read_vectors_pair, &convolve) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::convolve::2", read_vectors_pair, &convolve) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::bluestein::1", read_vectors_pair, &bluestein) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::bluestein::2", read_vectors_pair, &bluestein) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (testcorrectnes_multiply("fft::multiplt::complex") == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft:real_to_complex", read_vectors_pair, &real_to_complex) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft:complex_to_real", read_vectors_pair, &complex_to_real) == EXIT_FAILURE)
    return EXIT_FAILURE;

  if (test_correctness("fft::fft_reverse_direct", read_vectors_pair,
      &fft_reverse_direct) == EXIT_FAILURE)
    return EXIT_FAILURE;

  std::cout << std::endl;
  std::cout << "------- Test completed --------" << std::endl;
  std::cout << std::endl;

  return EXIT_SUCCESS;
}
