/*****************************************************************************
*
* Copyright (c) 2000 - 2009, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtvdfFileFormat.C                           //
// ************************************************************************* //

#include <avtvdfFileFormat.h>
#include <avtGhostData.h>
#include <avtLogicalSelection.h>
#include <avtDatabaseMetaData.h>
#include <avtIntervalTree.h>

#include <vtkCellData.h>
#include <vtkFloatArray.h>
#include <vtkUnsignedCharArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

#include <InvalidVariableException.h>
#include <InvalidDBTypeException.h>
#include <DBOptionsAttributes.h>
#include <Expression.h>
#include <string>

#include <vapor/MyBase.h>
#include <vapor/WaveletBlock3DRegionReader.h>
#include <vapor/DataMgrWB.h>
#include <vapor/DataMgrLayered.h>

#include <snprintf.h> //used by VisIt for Windows workaround 

using     std::string;
using     std::vector;
using     std::map;

void debug_callback(const char *msg) { printf("debug: %s\n",msg); }
void err_callback(const char *msg, int n) { printf("err{%d}: %s\n",n,msg); }

// ****************************************************************************
//  Method: avtvdfFileFormat constructor
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Wed Feb 24 15:32:35 PST 2010
//
// ****************************************************************************

avtvdfFileFormat::avtvdfFileFormat(const char *filename, 
	DBOptionsAttributes *readOpts) : avtMTMDFileFormat(filename)
{
	size_t maxts;
	int buffer_size = readOpts->GetInt("Cache size [MB]");
	double heightMult_layered = readOpts->GetDouble(
		"Layered data resampling factor");
	multiDom = readOpts->GetBool("MultiDomain");

	if (heightMult_layered <= 0.f || buffer_size <= 0)
		EXCEPTION1(InvalidDBTypeException, "Invalid Option, must be >0."); 

	vdc_md = new VAPoR::MetadataVDC(string(filename));
	vdfiobase = new VAPoR::WaveletBlock3DRegionReader(string(filename));
	if (vdc_md == NULL) printf ("NULL VDC!\n");

	// instantiate the correct data manager sub-type.
	isLayered = (vdc_md->GetGridType().compare("layered") == 0);
	if (isLayered) {
		VAPoR::DataMgrLayered *dmLayered_inst = new VAPoR::DataMgrLayered(string(filename),
			buffer_size );
		data_mgr = dmLayered_inst;
	} else {
		VAPoR::DataMgrWB *dmWB_inst = new VAPoR::DataMgrWB(*vdc_md, buffer_size );
		data_mgr = dmWB_inst;
	}
	if (data_mgr == NULL) printf("NULL Datamanager!\n");

//	MyBase::SetErrMsgCB(err_callback);
//	MyBase::SetDiagMsgCB(debug_callback);

	num_levels = vdc_md->GetNumTransforms() + 1; 
	maxts = vdc_md->GetNumTimeSteps(),
	tsteps = new int[maxts];
	ntsteps = 0;

	levels3D = levelsXY = levelsXZ = levelsYZ = -1;
	Names3D = (vector<string>) (vdc_md->GetVariables3D());
	xyNames = (vector<string>) (vdc_md->GetVariables2DXY());
	xzNames = (vector<string>) (vdc_md->GetVariables2DXZ());
	yzNames = (vector<string>) (vdc_md->GetVariables2DYZ());

	// search through all possible variables, refinement levels, 
	//   and timesteps to eliminate timesteps with no valid data, and 
	//   find the max valid refinement level for each variable (for
	//   scalar var metadta) and variable type (for mesh metadata).
	for (int t=0; t<maxts; t++) {
		bool thisTimeIsValid = false;

		for (int l=0; l<num_levels; l++) {

			for (int n=0; n<Names3D.size(); n++){
				if (vdfiobase->VariableExists((size_t)t, Names3D[n].c_str(), l) !=0) {
					varTypes[Names3D[n]] = 0;
					levels3D = (l > levels3D)? l: levels3D;
					refLevel[Names3D[n]] = levels3D;
					thisTimeIsValid = true;
				}
			}
			for (int n=0; n<xyNames.size(); n++){
				if (vdfiobase->VariableExists((size_t)t, xyNames[n].c_str(), l) !=0) {
					varTypes[xyNames[n]] = 1;
					levelsXY = (l > levelsXY)? l: levelsXY;
					refLevel[xyNames[n]] = levelsXY;
					thisTimeIsValid = true;
				}
			}
			for (int n=0; n<xzNames.size(); n++){
				if (vdfiobase->VariableExists((size_t)t, xzNames[n].c_str(), l) !=0) {
					varTypes[xzNames[n]] = 2;
					levelsXZ = (l > levelsXZ)? l: levelsXZ;
					refLevel[xzNames[n]] = levelsXZ;
					thisTimeIsValid = true;
				}
			}
			for (int n=0; n<yzNames.size(); n++){
				if (vdfiobase->VariableExists((size_t)t, yzNames[n].c_str(), l) !=0) {
					varTypes[yzNames[n]] = 3;
					levelsYZ = (l > levelsYZ)? l: levelsYZ;
					refLevel[yzNames[n]] = levelsYZ;
					thisTimeIsValid = true;
				}
			}
		}

		if (thisTimeIsValid) {
			tsteps[ntsteps] = t;
			ntsteps++;
		}

	}

	if (ntsteps == 0 || data_mgr == NULL)
		EXCEPTION1(InvalidDBTypeException, "No valid data."); 

}

// ****************************************************************************
//  Method: avtvdfFileFormat destructor
//
//  Programmer: dlagreca 
//  Creation:   Wed Feb 24 14:12:30 PST 2010
//
// ****************************************************************************

avtvdfFileFormat::~avtvdfFileFormat()
{
	// clean up memory
	delete   data_mgr; 
	delete   vdfiobase; 
	delete   vdc_md; 
	delete[] tsteps;
}

// ****************************************************************************
//  Method: avtvdfFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Wed Feb 24 15:32:35 PST 2010
//
// ****************************************************************************

void
avtvdfFileFormat::FreeUpResources(void)
{
	//data_mgr->Clear();
}


// ****************************************************************************
//  Method: avtvdfFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Wed Feb 24 15:32:35 PST 2010
//
// ****************************************************************************

void
avtvdfFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md, int timeState)
{
	//clear any old error messages before we start
	vdfiobase->SetErrCode(0);

	size_t ldim[3], bdim[num_levels][3];
	char refNames[num_levels][20];
	for (int l=0; l<num_levels; l++) {
		getVarDims(l, ldim, bdim[l]);
		SNPRINTF(refNames[l], 20, "%dx%dx%d", (int)ldim[0], (int)ldim[1], (int)ldim[2]);
	}

	double *extents = NULL;

	char Mname3D[num_levels][50], 
         XYMname[num_levels][50], 
         XZMname[num_levels][50], 
         YZMname[num_levels][50];
	// meshes describe the positions of datapoints in 3D space, add one for
	// each type of Vapor data variable and refinement level, for which at
	// least one valid variable exists to use it.

	for (int j=0; j<=levels3D; j++) {
		SNPRINTF(Mname3D[j], 50, "3D_%s", refNames[j]);
		AddMeshToMetaData(md, Mname3D[j], AVT_RECTILINEAR_MESH, extents, 
			multiDom? bdim[j][0]*bdim[j][1]*bdim[j][2]: 1, 0,3,3);
		md->SetContainsGhostZones(string(Mname3D[j]),AVT_CREATED_GHOSTS);
		md->SetContainsOriginalCells(string(Mname3D[j]),true);
		refLevel[Mname3D[j]] = j;
	}

	for (int j=0; j<=levelsXY; j++) {
		SNPRINTF(XYMname[j], 50, "XY_%s", refNames[j]);
		AddMeshToMetaData(md, XYMname[j], AVT_RECTILINEAR_MESH, extents, 
			multiDom? bdim[j][0]*bdim[j][1]: 1, 0,2,2);
		refLevel[XYMname[j]] = j;
	}

	for (int j=0; j<=levelsXZ; j++) {
		SNPRINTF(XZMname[j], 50, "XZ_%s", refNames[j]);
		AddMeshToMetaData(md, XZMname[j], AVT_RECTILINEAR_MESH, extents, 
			multiDom? bdim[j][0]*bdim[j][2]: 1, 0,2,2);
		refLevel[XZMname[j]] = j;
	}

	for (int j=0; j<=levelsYZ; j++) {
		SNPRINTF(YZMname[j], 50, "YZ_%s", refNames[j]);
		AddMeshToMetaData(md, YZMname[j], AVT_RECTILINEAR_MESH, extents, 
			multiDom? bdim[j][1]*bdim[j][2]: 1, 0,2,2);
		refLevel[YZMname[j]] = j;
	}

	char vname[100];
	// create a visit variable for each valid refinement level for each
	// valid Vapor variable. Name it <MeshType>/XxYxZ/<Vapor variable name>
	// the '/' char tells visit to create a submenu.  Attach to the correct 
	// mesh based on dimensional type and refinement level 

	for (int i=0; i<Names3D.size(); i++) {
		for (int l=0; l<=refLevel[Names3D[i]]; l++) {
			SNPRINTF(vname, 100, "3D/%s/%s", refNames[l], Names3D[i].c_str());
			AddScalarVarToMetaData(md, vname, Mname3D[l], AVT_ZONECENT);
			varTypes[vname] = 0;
			refLevel[vname] = l;
		}
	}
	for (int i=0; i<xyNames.size(); i++) {
		for (int l=0; l<=refLevel[xyNames[i]]; l++) {
			SNPRINTF(vname, 100, "2D/%s/%s", refNames[l], xyNames[i].c_str());
			AddScalarVarToMetaData(md, vname, XYMname[l], AVT_ZONECENT);
			varTypes[vname] = 1;
			refLevel[vname] = l;
		}
	}
	for (int i=0; i<xzNames.size(); i++) {
		for (int l=0; l<=refLevel[xzNames[i]]; l++) {
			SNPRINTF(vname, 100, "2D/%s/%s", refNames[l], xzNames[i].c_str());
			AddScalarVarToMetaData(md, vname, XZMname[l], AVT_ZONECENT);
			varTypes[vname] = 2;
			refLevel[vname] = l;
		}
	}
	for (int i=0; i<yzNames.size(); i++) {
		for (int l=0; l<=refLevel[yzNames[i]]; l++) {
			SNPRINTF(vname, 100, "2D/%s/%s", refNames[l], yzNames[i].c_str());
			AddScalarVarToMetaData(md, vname, YZMname[l], AVT_ZONECENT);
			varTypes[vname] = 3;
			refLevel[vname] = l;
		}
	}

}


// ****************************************************************************
//  Method: avtvdfFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain      The index of the domain.  If there are NDomains, this
//                  value is guaranteed to be between 0 and NDomains-1,
//                  regardless of block origin.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Wed Feb 24 15:32:35 PST 2010
//
// ****************************************************************************

vtkDataSet *
avtvdfFileFormat::GetMesh(int timestate, int domain, const char *meshname)
{
	//clear any old error messages before we start
	vdfiobase->SetErrCode(0);

	// Build the 3D mesh, and refactor appropriately for 2D meshes 
	size_t dims_st[3], bdims[3];
	const size_t *block_size = vdc_md->GetBlockSize();
	int dim_start[3];

	getVarDims(string(meshname), dims_st, bdims);

	size_t coordsB[3], dims_real[3];
	if (multiDom){
		getCoordsFromIndex(coordsB, bdims, domain);
		getVoxelsInBlock(dims_real, dims_st, bdims, coordsB);
		for (int i=0; i<3; i++) {
			dims_real[i] ++;
			dim_start[i] = (block_size[i])*(coordsB[i])-(coordsB[i]==0?0:1);
		}
	} 
	else {
		for (int i=0; i<3; i++) {
			dim_start[i] = 0;
			dims_real[i] = dims_st[i] + 1;
		}
	}

	const vector<double> glExtents = vdc_md->GetExtents(),
	      tsExtents = vdc_md->GetTSExtents((size_t)tsteps[timestate]),
	      uExtents = tsExtents.empty()? glExtents: tsExtents;

	// find user delta value for each dimension 
	float dx = ((float)uExtents[3] - (float)uExtents[0])/(float)dims_st[0], 
	      dy = ((float)uExtents[4] - (float)uExtents[1])/(float)dims_st[1],
	      dz = ((float)uExtents[5] - (float)uExtents[2])/(float)dims_st[2];

	// these arrays map data coords to user coords 
	vtkFloatArray *coordsX = vtkFloatArray::New();
	vtkFloatArray *coordsY = vtkFloatArray::New();
	vtkFloatArray *coordsZ = vtkFloatArray::New();

	coordsX->SetNumberOfTuples(dims_real[0]);
	coordsY->SetNumberOfTuples(dims_real[1]);
	coordsZ->SetNumberOfTuples(dims_real[2]);

	// compute user coords for each data coord 
	for (int i=(dim_start[0]), j=0; j<dims_real[0]; i++, j++) { 
		coordsX->SetValue(j, (float)i*dx + uExtents[0]);
	}
	for (int i=(dim_start[1]), j=0; j<dims_real[1]; i++, j++) { 
		coordsY->SetValue(j, (float)i*dy + uExtents[1]);
	}
	for (int i=(dim_start[2]), j=0; j<dims_real[2]; i++, j++) { 
		coordsZ->SetValue(j, (float)i*dz + uExtents[2]);
	}

	// adjust array dimensions if var is only 2D 
	if (!strncmp(meshname, "3D", 2)) {
		//already done!

	} else if (!strncmp(meshname, "XY", 2)) {
		dims_real[2] = 1;
		coordsZ->SetNumberOfTuples(1);
		coordsZ->SetValue(0, 0.f);

	} else if (!strncmp(meshname, "XZ", 2)) {
		dims_real[1] = 1;
		coordsY->SetNumberOfTuples(1);
		coordsY->SetValue(0, 0.f);

	} else if (!strncmp(meshname, "YZ", 2)) {
		dims_real[0] = 1;
		coordsX->SetNumberOfTuples(1);
		coordsX->SetValue(0, 0.f);

	} else {
		// No mesh name that we recognize. 
		coordsX->Delete(); coordsY->Delete(); coordsZ->Delete();
		EXCEPTION1(InvalidVariableException, meshname); 
	}

	// apply coordinate maps and return rgrid
	vtkRectilinearGrid *rgrid = vtkRectilinearGrid::New();
	rgrid->SetDimensions(dims_real[0], dims_real[1], dims_real[2]);
	rgrid->SetXCoordinates(coordsX);
	rgrid->SetYCoordinates(coordsY);
	rgrid->SetZCoordinates(coordsZ);

	if (multiDom) { //only add ghost data for multi-domain
		// Now that you have your mesh, figure out which cells need 
		// to be removed. 
		int nCells = rgrid->GetNumberOfCells(); 

		// Now that we have the blanks array, create avtGhostZones. 
		unsigned char realVal = 0, ghost = 0; 
		size_t cell[3], cellMax[3]; 
		for(int i=0; i<3; i++) cellMax[i] = dims_real[i]-1;
		avtGhostData::AddGhostZoneType(ghost, DUPLICATED_ZONE_INTERNAL_TO_PROBLEM); 

		vtkUnsignedCharArray *ghostCells = vtkUnsignedCharArray::New(); 
		ghostCells->SetName("avtGhostZones"); 
		ghostCells->Allocate(nCells); 

		for(int i = 0; i < nCells; ++i) 
		{ 
			getCoordsFromIndex(cell, cellMax, i);
			bool isGhost = ((cell[0] == 0) && (coordsB[0] != 0)) || 
				 ((cell[0] == cellMax[0]-1) && (coordsB[0] != bdims[0]-1 )) ||  
				 ((cell[1] == 0) && (coordsB[1] != 0)) || 
				 ((cell[1] == cellMax[1]-1) && (coordsB[1] != bdims[1]-1 )) ||  
				 ((cell[2] == 0) && (coordsB[2] != 0)) || 
				 ((cell[2] == cellMax[2]-1) && (coordsB[2] != bdims[2]-1 )); 

			if (isGhost) {
				ghostCells->InsertNextValue(ghost);
			} else {
				ghostCells->InsertNextValue(realVal); 
			}
		} 
		rgrid->GetCellData()->AddArray(ghostCells); 
		rgrid->SetUpdateGhostLevel(0); 
		ghostCells->Delete();

		// Clean up 
		coordsX->Delete();
		coordsY->Delete();
		coordsZ->Delete();
	}

	return rgrid;
}


// ****************************************************************************
//  Method: avtvdfFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Wed Feb 24 15:32:35 PST 2010
//
// ****************************************************************************

vtkDataArray *
avtvdfFileFormat::GetVar(int timestate_in, int domain, const char *varname)
{
	//clear any old error messages before we start
	vdfiobase->SetErrCode(0);

	// Get actual timestep/varname used in vdf 
	string varstring(varname);
	char *varname_clean = cleanVarname(varname);
	size_t timestate = (size_t) tsteps[timestate_in];

	// Need to add lod value for GetRegion call for VAPOR 2.0.2 api.
	// Setting the value to zero, not sure if correct.
	int lod = 0; 

	// Verifty that data for this var at this time and refinement exists 
	if (!vdfiobase->VariableExists(timestate, varname_clean, refLevel[varstring])) { 
		stringstream mesg;
		mesg << "Data for " << varname_clean << " does not exist in timestep " 
		     << timestate << " at refinement level " << refLevel[varstring] << "].";

		EXCEPTION1(InvalidVariableException, mesg.str().c_str());
	}

	// Get blocksize and data bounds to calculate padding 
	datasize_t ds;
	size_t dmin[3], dmax[3], bdims[3], udims[3];
	const size_t *bmsize = vdc_md->GetBlockSize();
	getVarDims(varstring, udims, bdims);

	// figure out which data we need to request from data_mgr 
	if (multiDom) {
		getCoordsFromIndex(dmin, bdims, domain);
		getDatasizeFromCoords(&ds, udims, bdims, dmin, dmax);
	}
	else {
		for (int i=0; i<3; i++) {
			dmin[i] = 0;
			dmax[i]=bdims[i]-1;

			ds.frontPad[i] = 0;
			ds.padded[i] = bmsize[i]*bdims[i];
			ds.backPad[i] = ds.padded[i] - udims[i];
			ds.real[i] = ds.padded[i] - ds.backPad[i];
		}
	} // else

	// get pointer to actual data (may be padded), thow execption on error
	float *data_floats = data_mgr->GetRegion(timestate, varname_clean, 
		refLevel[varstring], lod, dmin, dmax);

	if (data_floats == NULL) { 
		EXCEPTION1(InvalidVariableException, "Data manager returned NULL pointer.");
	}

	// set up extents for return vector based on data type
	int which_dim;
	switch(varTypes[varstring]) {
		case 0: which_dim = -1; break;
		case 1: which_dim =  2; break;
		case 2: which_dim =  1; break;
		case 3: which_dim =  0; break;
		default:
			EXCEPTION1(InvalidVariableException, "Variable type undefined.");
			break;
	}
	if (which_dim != -1) {
		ds.frontPad[which_dim] = 0;
		ds.backPad[which_dim] = 0;
		ds.padded[which_dim] = 1;
		ds.real[which_dim] = 1;
	}

	// create return vector, filling with only valid data 
	int idx = 0, numdata = ds.real[0] * ds.real[1] * ds.real[2];
	vtkFloatArray *rv = vtkFloatArray::New();
	rv->SetNumberOfValues(numdata);

	for (int z=ds.frontPad[2]; z< (ds.padded[2] - ds.backPad[2]); z++) {
		for (int y=ds.frontPad[1]; y< (ds.padded[1] - ds.backPad[1]); y++) {
			for (int x=ds.frontPad[0]; x< (ds.padded[0] - ds.backPad[0]); x++) {

				rv->SetValue(idx, data_floats[ z*ds.padded[1]*ds.padded[0] + 
				                               y*ds.padded[0] + 
					                           x ]);

				idx++;
			}
		}
	}

	delete varname_clean;
	return rv;
}


// ****************************************************************************
//  Method: avtvdfFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Note: 
//      although vapor data sets often contain vector data, there is no 
//      standard representation. VisIt allows for the creation of vecotor
//      variables based on multiple scalar variables, so no 'vector type'
//      variables are reported to VisIt
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Wed Feb 24 15:32:35 PST 2010
//
// ****************************************************************************

vtkDataArray *
avtvdfFileFormat::GetVectorVar(int timestate, int domain,const char *varname)
{
	return 0;
}


// ****************************************************************************
//  Method: avtvdfFileFormat::GetTimes
//
//  Purpose:
//      Sets the time numbers to match the VDF user-time
//
//  Arguments:
//      times    double precision float vector to be filled with user time for 
//               each valid timestep.
//
//  Note:
//      I am unsure why, but reporting the vapor TSUserTime breaks streamlines
//      for WRF-based Vapor datasets.
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Thu Feb 11 14:12:30 PST 2010
//
// ****************************************************************************

void 
avtvdfFileFormat::GetTimes(vector<double> &times) 
{
	for (int i=0; i<ntsteps; i++) 
		times.push_back(tsteps[i]);
		//times.push_back(data_mgr->GetTSUserTime((size_t)tsteps[i])[0]);
}


// ****************************************************************************
//  Method: avtvdfFileFormat::GetCycles
//
//  Purpose:
//      Sets the cycle numbers to match the timestate numbers for valid time-
//      states. This allows the correct 'timestep' to be shown even though
//      the invalid timesteps are not counted.
//
//  Arguments:
//      cycles   integer vector to be filled with correct cycle number for each
//               timestep
//
//  Programmer: dlagreca -- generated by xml2avt
//  Creation:   Thu Feb 11 14:12:30 PST 2010
//
// ****************************************************************************

void 
avtvdfFileFormat::GetCycles(vector<int> &cycles) 
{
	for (int i=0; i<ntsteps; i++) 
		cycles.push_back(tsteps[i]);
}


// ********************************************************
// Method: avtvdfFileFormat::getVoxelsInBlock 
//
// Purpose: 
//   Given a certian data block, return the dimensions (in voxels) of that
//   block. Need to repeat the data on forward block edges, only if that block
//   is adjcent to another block (i.e. not the end of the dataset), therefore
//   add one voxel to dimension if this block is not an end-block.
//
// Arguments:
//   udims: 3-element array of total valid data dimensions (voxels)
//	 bdims: 3-element array of total blocks in data
//	 dmin : 3-element array of current block coords
//	 numel: 3-element array to place results in
//
// Returns: nothing
//   numel is filled.
//
// Note:
//   The original functionality of this method has been integrated into
//   getDatasizeFromCoords(), so this method simply calls that it,
//   returning the appropriate component. This is done for consistancy. 
//
// Programmer: dlagreca -- generated by xml2avt
// Creation:   Thu Feb 11 14:12:30 PST 2010
//
// Modifications:
//
// ********************************************************

void 
avtvdfFileFormat::getVoxelsInBlock(size_t *numel, size_t *udims, 
	size_t *bdims, size_t* coord) 
{

	size_t mn[3]={coord[0], coord[1], coord[2]}, mx[3];
	datasize_t dss;
	getDatasizeFromCoords(&dss, udims, bdims, mn, mx);
	for (int i=0; i<3; i++) numel[i] = dss.real[i];

}


// ********************************************************
// Method: avtvdfFileFormat::getCoordsFromIndex
//
// Purpose: 
//   Given a block index in a dataset, find the 'x,y,z' coordinates of said block 
//
// Arguments:
//   coords: 3-element array containing the x,y,z coordinates of the block at index
//   bdims: total block dimensions of dataset
//   index: index of block in the dataset
//
// Returns: nothing
//   coors is filled.
//
// Programmer: dlagreca -- generated by xml2avt
// Creation:   Thu Feb 11 14:12:30 PST 2010
//
// Modifications:
//
// ********************************************************

void 
avtvdfFileFormat::getCoordsFromIndex(size_t *coords, size_t* bdims, int index)
{

	coords[2] = index/(bdims[0] * bdims[1]);
	index -= coords[2] * (bdims[0] * bdims[1]);
	coords[1] = index/bdims[0];
	index -= coords[1] * bdims[0];
	coords[0] = index;

}


// ********************************************************
// Method: avtvdfFileFormat::getDataSizeFromCoords
//
// Purpose: 
//   given the coordinates of a block, return the nuber of elements in each dimension
//   that should be pulled from the data manager, including ghost zones for non-
//   boundary edges:
//
// Arguments:
//   datasize: return structure for padding information 
//   udims : 3-element array of total blocks in data
//   bdims : 3-element array of total blocks in data
//   dmin  : 3-element array of starting data block coords
//   dmax  : 3-element array of ending data block coords
//
// Returns: nothing
//   datasize is filled.
//
// Programmer: dlagreca -- generated by xml2avt
// Creation:   Thu Feb 11 14:12:30 PST 2010
//
// Modifications:
//
// ********************************************************

void 
avtvdfFileFormat::getDatasizeFromCoords(datasize_t *datasize, 
     size_t *udims, size_t *bdims, size_t *dmin, size_t* dmax) 
{
	const size_t *bmsize = vdc_md->GetBlockSize();

	for (int i=0; i<3; i++) {
		dmax[i] = dmin[i];
		if (dmin[i] == bdims[i]-1) { //last block
			int p = bdims[i]*bmsize[i] - udims[i];
			if (dmin[i] > 0) {		//other blocks before it
				dmin[i]--;
				datasize->padded[i]   = bmsize[i]*2;
				datasize->frontPad[i] = bmsize[i]-1;
				datasize->backPad[i]  = p;

			} else {				//only one block in this dim
				datasize->padded[i]   = bmsize[i];
				datasize->frontPad[i] = 0;
				datasize->backPad[i]  = p;
			}

		} else if (dmin[i] == 0) {
			dmax[i]++;
			datasize->padded[i]   = bmsize[i]*2;
			datasize->frontPad[i] = 0;
			datasize->backPad[i]  = bmsize[i]-1;

		} else {
			dmax[i]++;
			dmin[i]--;
			datasize->padded[i]   = bmsize[i]*3;
			datasize->frontPad[i] = bmsize[i]-1;
			datasize->backPad[i]  = bmsize[i]-1; 
		}
		datasize->real[i] = datasize->padded[i] - datasize->backPad[i]
			- datasize->frontPad[i];
	}

}


// ********************************************************
// Method: avtvdfFileFormat::cleanVarname
//
// Purpose: 
//   Removes the leading characters that were added to the varname
//   to specifiy its correct mesh. Only the clean varname can be used to 
//   interface with the data_mgr. 
//
// Arguments:
//	 varname: c-string variable name, contianing unwanted chars
//
// Returns: 
//   pointer to new c-string. Be sure to call delete() when you're done
//   with it
//
// Programmer: dlagreca -- generated by xml2avt
// Creation:   Thu Feb 11 14:12:30 PST 2010
//
// Modifications:
//
// ********************************************************

char* 
avtvdfFileFormat::cleanVarname(const char *varname)
{
	// remove refinement tag from varname - find clean length
	int length = strlen(varname), start = length;
	while (varname[start--] != '/');
	length = length-start-2;

	// copy applicaple part of varname to _clean 
	char *varname_clean = new char[length];
	for (int i=0; i<length; i++) varname_clean[i] = varname[start+i+2];
	varname_clean[length] = '\0';
	
	return varname_clean;
}


// ********************************************************
// Method: avtvdfFileFormat::getVarDims
//
// Purpose: 
//   This function will fill the two arrays passed in (udims and bdims) with 
//   the size of data at the specified refinement level in voxels, and blocks,
//   respectively. If the first arguement is a string, its refinement level
//   is first decided based on the refLevel map produced in the ctor.
//
// Arguments:
//	 level: refinement level to retrieve blocksize for. 
//   - OR -
//   varstring: index into refLevel map to choose refinement level.
//   udims: 3-element array of total valid data dimensions (voxels)
//	 bdims: 3-element array of total blocks in data
//
// Returns: nothing
//   udims and bdims are filled.
//
// Programmer: dlagreca -- generated by xml2avt
// Creation:   Thu Feb 11 14:12:30 PST 2010
//
// Modifications:
//
// ********************************************************

inline void 
avtvdfFileFormat::getVarDims(string varstring, size_t *udims, size_t *bdims) 
{
	getVarDims(refLevel[varstring], udims, bdims);
}

void 
avtvdfFileFormat::getVarDims(int level, size_t *udims, size_t *bdims) 
{
	vdfiobase->GetDim(udims, level);
	vdfiobase->GetDimBlk(bdims, level);
}


// ********************************************************
// Method: avtvdfFileFormat::GetAuxiliaryData
//
// Purpose: 
//   used by visit to access extents for each domain. This
//   allows visit to request only those data blocks that 
//   are requred for rendering.
//
// Arguments:
//	 var: variable for wich to return extent. 
//
// Returns: 
//   itree with spatial extents for each domain. 
//
// Programmer: dlagreca -- generated by xml2avt
// Creation:   Thu Feb 11 14:12:30 PST 2010
//
// Modifications:
//
// ********************************************************

void * 
avtvdfFileFormat::GetAuxiliaryData(const char *var, 
	int timestate, int domain, const char *type, void *, 
	DestructorFunction &df) 
{ 
	void *retval = 0; 
	if(strcmp(type, AUXILIARY_DATA_SPATIAL_EXTENTS) == 0) 
	{ 
		if (!multiDom) return retval;
		//printf("Returning spatial extents for %s @ %d in domain %d, ", var, timestate, domain);
		size_t dims[3], bdims[3], bcoords[3];
		float delta[3];
		const size_t *bsize = vdc_md->GetBlockSize();
		const vector<double> glExt = vdc_md->GetExtents(),
			  tsExt = vdc_md->GetTSExtents((size_t)tsteps[timestate]),
			  uExt = tsExt.empty()? glExt: tsExt;

		// Read the number of domains for the mesh. 
		// Read the spatial extents for each domain of the 
		// mesh. This information should be in a single 
		// and should be available without having to 
		// read the real data. The expected format for 
		// the data in the spatialextents array is to 
		// repeat the following pattern for each domain: 
		// xmin, xmax, ymin, ymax, zmin, zmax. 
		getVarDims(string(var), dims, bdims);

		int ndoms = multiDom? bdims[0]*bdims[1]*bdims[2]: 1;
		double *spatialextents = new double[ndoms * 6]; 

		for (int i=0; i<3; i++)
				delta[i] = ((float)uExt[i+3] - (float)uExt[i])/(float)dims[i];

		//READ ndoms*6 DOUBLE VALUES INTO spatialextents ARRAY. 
		for (int curDom = 0; curDom < ndoms; curDom++) {
			getCoordsFromIndex(bcoords, bdims, curDom);

			//Spatial dims go from (block_start)*(voxels/block)*(units/voxel) 
			//  to (block_start + 1)*(voxels/block)*(units/voxel)
			for (int i=0; i<3; i++) {
				spatialextents[(6*curDom) + 2*i] = (float)(bcoords[i]*bsize[i])*delta[i];
				spatialextents[(6*curDom) + 2*i+1] = (float)((bcoords[i]+1)*bsize[i])*delta[i];
			}
		}

		// Create an interval tree 
		avtIntervalTree *itree = new avtIntervalTree(ndoms, 3); 
		double *extents = spatialextents; 
		for(int dom = 0; dom < ndoms; ++dom) 
		{ 
			itree->AddElement(dom, extents); 
			extents += 6; 
		} 
		itree->Calculate(true); 
		// Delete temporary array. 
		delete [] spatialextents; 
		// Set return values 
		retval = (void *)itree; 
		df = avtIntervalTree::Destruct; 
	} 

	return retval;
}
