//
// Copyright 2016 Pixar
//
// Licensed under the Apache License, Version 2.0 (the "Apache License")
// with the following modification; you may not use this file except in
// compliance with the Apache License and the following modification to it:
// Section 6. Trademarks. is deleted and replaced with:
//
// 6. Trademarks. This License does not grant permission to use the trade
//    names, trademarks, service marks, or product names of the Licensor
//    and its affiliates, except as required to comply with Section 4(c) of
//    the License and to reproduce the content of the NOTICE file.
//
// You may obtain a copy of the Apache License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the Apache License with the above modification is
// distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. See the Apache License for the specific
// language governing permissions and limitations under the Apache License.
//
#ifndef PXRUSDMAYA_ROUND_TRIP_UTIL_H
#define PXRUSDMAYA_ROUND_TRIP_UTIL_H

/// \file usdMaya/roundTripUtil.h

#include "usdMaya/api.h"

#include "pxr/pxr.h"

#include "pxr/usd/usd/attribute.h"
#include "pxr/usd/usdGeom/primvar.h"


PXR_NAMESPACE_OPEN_SCOPE


/// This holds some utility functions for dealing with data from Maya to Usd
/// and back.
struct UsdMayaRoundTripUtil
{
    /// Sometimes primvars like displayColor and displayOpacity aren't *really*
    /// authored by the user and instead are generated based on shaders, etc.
    /// This function encapsulates that logic so we can make this determination
    /// when importing a USD file. This just checks that there is an actual
    /// authored opinion and that it was not Maya generated.
    /// \sa IsAttributeMayaGenerated
    PXRUSDMAYA_API
    static bool IsAttributeUserAuthored(const UsdAttribute& attr);

    PXRUSDMAYA_API
    static bool IsAttributeMayaGenerated(const UsdAttribute& attr);

    /// Annotate that an attribute was generated by Maya.
    PXRUSDMAYA_API
    static void MarkAttributeAsMayaGenerated(const UsdAttribute& attr);

    /// Primvars in USD are not clamped but they may have originally have been
    /// clamped.
    PXRUSDMAYA_API
    static bool IsPrimvarClamped(const UsdGeomPrimvar& primvar);

    PXRUSDMAYA_API
    static void MarkPrimvarAsClamped(const UsdGeomPrimvar& primvar);

    /// If Maya attribute arrays are used, we may need to store this encoding.
    /// This is especially true for shading where connections are made between
    /// attributes inside the array. Returns true if this was marked as an
    /// array attribute and will set \p index. Returns false otherwise.
    PXRUSDMAYA_API
    static bool GetAttributeArray(
            const UsdAttribute& attr,
            unsigned int* index);

    PXRUSDMAYA_API
    static void MarkAttributeAsArray(
            const UsdAttribute& attr,
            const unsigned int index);
};


PXR_NAMESPACE_CLOSE_SCOPE


#endif
