/*
 Uptime Client v5.0 beta
 
 $Id: stats-lnx.c,v 1.36 2003/05/26 19:54:04 carstenklapp Exp $

 Logs system uptime and statistics with Uptimes Project servers

 Copyright (C) 1999-2002 Martijn Broenland, Alex C. de Haas, Carsten Klapp

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 Carsten Klapp <carstenklapp@users.sourceforge.net>
 Alex C. de Haas <alex@uptimes.net>
 Martijn Broenland <tgm@uptimes.net>
 */

/**
 * @filename    stats-lnx.c
 *
 * @desc        Retrieve stats for the Linux platform
 */

#if defined PLATFORM_LINUX

/*@unused@*/ static const char rcsid[] =
    "@(#)$Id: stats-lnx.c,v 1.36 2003/05/26 19:54:04 carstenklapp Exp $";

/* System includes */
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <string.h>
#include <sys/utsname.h>

#include "locale.h"     /* gettext */

/* My includes */
#include "upclient.h"
#include "options.h"
#include "stats.h"
#include "uplog.h"      /* wrapper for <syslog.h> */
#include "compat/sysexits.h"    /* sendmail-style sysexits constants */

#if !defined TICKS_PER_SECOND
#   define TICKS_PER_SECOND 60;
#endif /* !defined TICKS_PER_SECOND */

/**
 * Sanity check
 */

#if defined NR_LINUX_UPTIME_WRAPAROUNDS
#   if NR_LINUX_UPTIME_WRAPAROUNDS < 1
#      error "Define your uptime wraparounds properly, pal."
#   endif /* < 1 */
#endif /* NR_LINUX_UPTIME_WRAPAROUNDS */

/**
 * @desc    Verbose level 3 logging of calulations
 */
void
logcalc(char *whatwascalculateddesc, char *value)
{
#if defined DEBUG
    uplog(LOG_DEBUG, _("%s calculated: %s"), whatwascalculateddesc, value);
#endif /* DEBUG */
}

/**
 * @desc    Get cpu once
 */
void
initCPU(char *cpu)
{
    static int initialized = 0;

    if (!initialized) {
       /**
        * utsname.machine doesn't provide useful info on some Linux, so use
        * /proc/cpuinfo
        */
       /**
        * struct utsname uts;
                      * strncpy(cpu, uts.machine, CPU_SIZE - 1); *//* _UTSNAME_LENGTH */

        char   temp_string[BUFSIZ];
        char   machine_string[BUFSIZ] = "unknown";
        char   cpu_string[BUFSIZ] = "";
        FILE  *fp;

        initialized = 1;

        if ((fp = fopen("/proc/cpuinfo", "r")) != NULL) {
            while (fscanf(fp, "%s :", temp_string) != EOF) {
               /* Look for model name & vendor_id on some linux. Probably most
                  use cpu & machine now. */
                if (!(strcmp(temp_string, "model name")) ||
                    (!strcmp(temp_string, "cpu"))) {
                    fscanf(fp, "%s", cpu_string);
                }
            }
            fclose(fp);
        }
        if ((fp = fopen("/proc/cpuinfo", "r")) != NULL) {
            while (fscanf(fp, "%s :", temp_string) != EOF) {
                if (!(strcmp(temp_string, "vendor_id")) ||
                    !(strcmp(temp_string, "machine"))) {
                    fscanf(fp, "%s", machine_string);
                }
            }
            fclose(fp);
        }
#if defined DEBUG
        uplog(LOG_DEBUG, "initCPU() cpu_string='%s'", cpu_string);
        uplog(LOG_DEBUG, "initCPU() machine_string='%s'", machine_string);
#endif /* DEBUG */

        if (cfg_sendcpudetail) {
            sprintf(cpu, "%s %s", machine_string, cpu_string);
        }
        else
            strncpy(cpu, machine_string, CPU_SIZE - 1);

        logcalc(_("CPU"), cpu);
    }
}

/**
 * @desc    Get os & osversion once
 */
void
initOS(char *osname, char *osversion)
{
    static int initialized = 0;

    if (!initialized) {
        struct utsname uts;

        initialized = 1;

        uname(&uts);

        if (cfg_sendosname) {
            strncpy(osname, uts.sysname, OS_SIZE - 1);
            logcalc(_("OS"), osname);
        }

        if (cfg_sendosversion) {
            strncpy(osversion, uts.release, OSVERSION_SIZE - 1);
            logcalc(_("OS version"), osversion);
        }
    }
}

/**
 * @desc    Get laod and idle percentages
 */
void
getLoadIdle(double *UsagePercent, double *IdlePercent)
{
    FILE  *fp;
    unsigned long int user, nice, system, idle;
    static unsigned long int prevuser, prevnice, prevsystem, previdle;

    double totaldiff;

    char   temp_string[BUFSIZ];

    fp = fopen("/proc/stat", "r");
    if (!fp) {
        uplog(LOG_ERR, _("%s %s: unable to open %s: %s"), _("FATAL ERROR:"),
              "getstats()", "/proc/stat", strerror(errno));
        exit(EX_OSFILE);
    }
    while (fscanf(fp, "%s ", temp_string) != EOF) {
        if (!(strcmp(temp_string, "cpu"))) {
           /* user nice system idle */
            fscanf(fp, "%ld %ld %ld %ld", &user, &nice, &system, &idle);
        }
    }
    fclose(fp);

    totaldiff =
        user - prevuser + nice - prevnice + system - prevsystem + idle -
        previdle;

    *UsagePercent =
        (double)(100.0 *
                 (user - prevuser + nice - prevnice + system -
                  prevsystem) / totaldiff);
    *IdlePercent = (double)(100.0 * ((idle - previdle) / totaldiff));

   /* Store static values for the next report */
    prevuser = user;
    prevnice = nice;
    prevsystem = system;
    previdle = idle;
}

/**
 * @desc    Get load average
 */
void
getLoadavg(double *loadavg)
{
    FILE  *fp;

    fp = fopen("/proc/loadavg", "r");
    if (fp == NULL) {
        uplog(LOG_ERR, _("%s %s: unable to open %s: %s"), _("FATAL ERROR:"),
              "getLoadavg()", "/proc/loadavg", strerror(errno));
        exit(EX_OSFILE);
    }
    if (fscanf(fp, "%*f %*f %lf %*s", loadavg) != 1) {
        uplog(LOG_ERR, _("%s %s: bogus format in %s"), _("FATAL ERROR:"),
              "getLoadavg()", "/proc/loadavg");
        exit(EX_OSFILE);
    }
    fclose(fp);
}

void
getUptime(unsigned long *uptimeminutes)
{
    FILE  *fp;
    double up = 0, idle = 0;

    fp = fopen("/proc/uptime", "r");
    if (!fp) {
        uplog(LOG_ERR, _("%s %s: unable to open %s: %s"), _("FATAL ERROR:"),
              "getUptime()", "/proc/uptime", strerror(errno));
        exit(EX_OSFILE);
    }
    if (fscanf(fp, "%lf %lf", &up, &idle) != 2) {
        uplog(LOG_ERR, _("%s %s: bogus format in %s"), _("FATAL ERROR:"),
              "getUptime()", "/proc/uptime");
        exit(EX_OSFILE);
    }
    fclose(fp);

    *uptimeminutes = (unsigned long int)(up / 60);      /* SECONDS_PER_MINUTE */
    if (up == 0) {
        up = 1;
    }
#if defined NR_LINUX_UPTIME_WRAPAROUNDS
   /* Make up for those miserable uptime wraparounds */
    *uptime += ((ULONG_MAX / 100) / 60) * NR_LINUX_UPTIME_WRAPAROUNDS;
#endif /* NR_LINUX_UPTIME_WRAPAROUNDS */
}

/**
 * @desc    Get statistics
 */
void
getstats(unsigned long *uptimeminutes, double *UsagePercent,
         double *IdlePercent, char *osname, char *osversion, char *cpu,
         double *loadavg)
{
    getUptime(&*uptimeminutes);

    if (cfg_sendloadavg) {
        getLoadavg(&*loadavg);
    }

    if (cfg_SendUsage || cfg_SendIdle) {
        getLoadIdle(&*UsagePercent, &*IdlePercent);
    }

    if (cfg_sendosname || cfg_sendosversion)
        initOS(&*osname, &*osversion);
    if (cfg_sendcpu)
        initCPU(&*cpu);
}
#endif /* PLATFORM_LINUX */
