/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import * as lsp from 'vscode-languageserver';
export class CodeActionKind {
    constructor(value) {
        this.value = value;
    }
    equals(other) {
        return this.value === other.value;
    }
    /**
     * Checks if `other` is a sub-kind of this `CodeActionKind`.
     *
     * The kind `"refactor.extract"` for example contains `"refactor.extract"` and ``"refactor.extract.function"`,
     * but not `"unicorn.refactor.extract"`, or `"refactor.extractAll"` or `refactor`.
     *
     * @param other Kind to check.
     */
    contains(other) {
        return this.equals(other) || this.value === '' || other.value.startsWith(this.value + CodeActionKind.sep);
    }
    /**
     * Checks if this code action kind intersects `other`.
     *
     * The kind `"refactor.extract"` for example intersects `refactor`, `"refactor.extract"` and ``"refactor.extract.function"`,
     * but not `"unicorn.refactor.extract"`, or `"refactor.extractAll"`.
     *
     * @param other Kind to check.
     */
    intersects(other) {
        return this.contains(other) || other.contains(this);
    }
    /**
     * Create a new kind by appending a more specific selector to the current kind.
     *
     * Does not modify the current kind.
     */
    append(part) {
        return new CodeActionKind(this.value + CodeActionKind.sep + part);
    }
}
CodeActionKind.sep = '.';
CodeActionKind.Empty = new CodeActionKind(lsp.CodeActionKind.Empty);
CodeActionKind.QuickFix = new CodeActionKind(lsp.CodeActionKind.QuickFix);
CodeActionKind.Refactor = new CodeActionKind(lsp.CodeActionKind.Refactor);
CodeActionKind.Source = new CodeActionKind(lsp.CodeActionKind.Source);
CodeActionKind.SourceAddMissingImportsTs = CodeActionKind.Source.append('addMissingImports').append('ts');
CodeActionKind.SourceRemoveUnusedTs = CodeActionKind.Source.append('removeUnused').append('ts');
CodeActionKind.SourceOrganizeImports = new CodeActionKind(lsp.CodeActionKind.SourceOrganizeImports);
CodeActionKind.SourceOrganizeImportsTs = CodeActionKind.SourceOrganizeImports.append('ts');
CodeActionKind.SourceFixAll = new CodeActionKind(lsp.CodeActionKind.SourceFixAll);
CodeActionKind.SourceFixAllTs = CodeActionKind.SourceFixAll.append('ts');
//# sourceMappingURL=types.js.map