/*
Copyright © 2020 banst

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/
package cmd

import (
	"fmt"

	"github.com/spf13/cobra"
)

var validCompletionArgs = []string{"bash", "zsh", "fish", "powershell"}

func init() {
	RootCmd.AddCommand(completionCmd)
}

// completionCmd represents the completion command
var completionCmd = &cobra.Command{
	Use:                   fmt.Sprintf("completion %s", validCompletionArgs),
	Short:                 "Generate completion script",
	DisableFlagsInUseLine: true,
	Example: `
Bash:

$ source <(tug completion bash)

# To load completions for each session, execute once:
Linux:
  $ tug completion bash > /etc/bash_completion.d/tug
MacOS:
  $ tug completion bash > /usr/local/etc/bash_completion.d/tug

Zsh:

$ source <(tug completion zsh)

# To load completions for each session, execute once:
$ tug completion zsh > "${fpath[1]}/_tug"

Fish:

$ tug completion fish | source

# To load completions for each session, execute once:
$ tug completion fish > ~/.config/fish/completions/tug.fish
`,
	ValidArgs: validCompletionArgs,
	Args:      cobra.ExactValidArgs(1),
	RunE:      completion,
}

func completion(cmd *cobra.Command, args []string) error {
	switch args[0] {
	case "bash":
		cmd.Root().GenBashCompletion(cmd.OutOrStdout())
	case "zsh":
		cmd.Root().GenZshCompletion(cmd.OutOrStdout())
	case "fish":
		cmd.Root().GenFishCompletion(cmd.OutOrStdout(), true)
	case "powershell":
		cmd.Root().GenPowerShellCompletion(cmd.OutOrStdout())
	default:
		return fmt.Errorf("%s is not a supported shell", args[0])
	}
	return nil
}
