import Foundation

/// A target of a project.
public struct Target: Codable, Equatable {
    /// The name of the target. Also, the product name if not specified with ``productName``.
    public let name: String

    /// The platform the target product is built for.
    public let platform: Platform

    /// The type of build product this target will output.
    public let product: Product

    /// The built product name. If nil, it will be equal to ``name``.
    public let productName: String?

    /// The product bundle identifier.
    public let bundleId: String

    /// The minimum deployment target your product will support.
    public let deploymentTarget: DeploymentTarget?

    /// The Info.plist representation.
    public let infoPlist: InfoPlist?

    /// The source files of the target.
    /// Note: any playgrounds matched by the globs used in this property will be automatically added.
    public let sources: SourceFilesList?

    /// The resource files of target.
    /// Note: localizable files, `*.lproj`, are supported.
    public let resources: ResourceFileElements?

    /// The build phase copy files actions for the target.
    public let copyFiles: [CopyFilesAction]?

    /// The headers for the target.
    public let headers: Headers?

    /// The entitlements representation
    public let entitlements: Entitlements?

    /// The build phase scripts actions for the target.
    public let scripts: [TargetScript]

    /// The target's dependencies.
    public let dependencies: [TargetDependency]

    /// The target's settings.
    public let settings: Settings?

    /// The Core Data models.
    public let coreDataModels: [CoreDataModel]

    /// The environment variables. Used by autogenerated schemes for the target.
    public let environmentVariables: [String: EnvironmentVariable]

    /// The launch arguments. Used by autogenerated schemes for the target.
    public let launchArguments: [LaunchArgument]

    /// The additional files for the target. For project's additional files, see ``Project/additionalFiles``.
    public let additionalFiles: [FileElement]

    /// The build rules used for transformation of source files during compilation.
    public let buildRules: [BuildRule]

    /// Specifies whether if the target can merge or not the dynamic dependencies as part of its binary
    public let mergedBinaryType: MergedBinaryType

    /// Specifies whether if the target can be merged as part of another binary or not
    public let mergeable: Bool

    public init(
        name: String,
        platform: Platform,
        product: Product,
        productName: String? = nil,
        bundleId: String,
        deploymentTarget: DeploymentTarget? = nil,
        infoPlist: InfoPlist? = .default,
        sources: SourceFilesList? = nil,
        resources: ResourceFileElements? = nil,
        copyFiles: [CopyFilesAction]? = nil,
        headers: Headers? = nil,
        entitlements: Entitlements? = nil,
        scripts: [TargetScript] = [],
        dependencies: [TargetDependency] = [],
        settings: Settings? = nil,
        coreDataModels: [CoreDataModel] = [],
        environmentVariables: [String: EnvironmentVariable] = [:],
        launchArguments: [LaunchArgument] = [],
        additionalFiles: [FileElement] = [],
        buildRules: [BuildRule] = [],
        mergedBinaryType: MergedBinaryType = .disabled,
        mergeable: Bool = false
    ) {
        self.name = name
        self.platform = platform
        self.bundleId = bundleId
        self.productName = productName
        self.product = product
        self.infoPlist = infoPlist
        self.entitlements = entitlements
        self.dependencies = dependencies
        self.settings = settings
        self.sources = sources
        self.resources = resources
        self.copyFiles = copyFiles
        self.headers = headers
        self.scripts = scripts
        self.coreDataModels = coreDataModels
        self.environmentVariables = environmentVariables
        self.launchArguments = launchArguments
        self.deploymentTarget = deploymentTarget
        self.additionalFiles = additionalFiles
        self.buildRules = buildRules
        self.mergedBinaryType = mergedBinaryType
        self.mergeable = mergeable
    }

    @available(*, deprecated, message: "please use environmentVariables instead")
    public init(
        name: String,
        platform: Platform,
        product: Product,
        productName: String? = nil,
        bundleId: String,
        deploymentTarget: DeploymentTarget? = nil,
        infoPlist: InfoPlist? = .default,
        sources: SourceFilesList? = nil,
        resources: ResourceFileElements? = nil,
        copyFiles: [CopyFilesAction]? = nil,
        headers: Headers? = nil,
        entitlements: Entitlements? = nil,
        scripts: [TargetScript] = [],
        dependencies: [TargetDependency] = [],
        settings: Settings? = nil,
        coreDataModels: [CoreDataModel] = [],
        environment: [String: String],
        launchArguments: [LaunchArgument] = [],
        additionalFiles: [FileElement] = [],
        buildRules: [BuildRule] = [],
        mergedBinaryType: MergedBinaryType = .disabled,
        mergeable: Bool = false
    ) {
        self.name = name
        self.platform = platform
        self.bundleId = bundleId
        self.productName = productName
        self.product = product
        self.infoPlist = infoPlist
        self.entitlements = entitlements
        self.dependencies = dependencies
        self.settings = settings
        self.sources = sources
        self.resources = resources
        self.copyFiles = copyFiles
        self.headers = headers
        self.scripts = scripts
        self.coreDataModels = coreDataModels
        environmentVariables = environment.mapValues { value in
            EnvironmentVariable(value: value, isEnabled: true)
        }
        self.launchArguments = launchArguments
        self.deploymentTarget = deploymentTarget
        self.additionalFiles = additionalFiles
        self.buildRules = buildRules
        self.mergedBinaryType = mergedBinaryType
        self.mergeable = mergeable
    }
}
