//----------------------------------------------------------------------------
//
// TSDuck - The MPEG Transport Stream Toolkit
// Copyright (c) 2005-2020, Thierry Lelegard
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
//    this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
// THE POSSIBILITY OF SUCH DAMAGE.
//
//----------------------------------------------------------------------------

#include "tsInputSwitcherArgs.h"
#include "tsArgsWithPlugins.h"
TSDUCK_SOURCE;

#if defined(TS_NEED_STATIC_CONST_DEFINITIONS)
constexpr size_t ts::InputSwitcherArgs::DEFAULT_MAX_INPUT_PACKETS;
constexpr size_t ts::InputSwitcherArgs::MIN_INPUT_PACKETS;
constexpr size_t ts::InputSwitcherArgs::DEFAULT_MAX_OUTPUT_PACKETS;
constexpr size_t ts::InputSwitcherArgs::MIN_OUTPUT_PACKETS;
constexpr size_t ts::InputSwitcherArgs::DEFAULT_BUFFERED_PACKETS;
constexpr size_t ts::InputSwitcherArgs::MIN_BUFFERED_PACKETS;
constexpr ts::MilliSecond ts::InputSwitcherArgs::DEFAULT_RECEIVE_TIMEOUT;
#endif


//----------------------------------------------------------------------------
// Constructors.
//----------------------------------------------------------------------------

ts::InputSwitcherArgs::InputSwitcherArgs() :
    appName(),
    fastSwitch(false),
    delayedSwitch(false),
    terminate(false),
    monitor(false),
    reusePort(false),
    firstInput(0),
    primaryInput(NPOS),
    cycleCount(1),
    bufferedPackets(0),
    maxInputPackets(0),
    maxOutputPackets(0),
    sockBuffer(0),
    remoteServer(),
    allowedRemote(),
    receiveTimeout(0),
    inputs(),
    output()
{
}

ts::InputSwitcherArgs::InputSwitcherArgs(const InputSwitcherArgs& other) :
    appName(other.appName),
    fastSwitch(other.fastSwitch),
    delayedSwitch(other.delayedSwitch),
    terminate(other.terminate),
    monitor(other.monitor),
    reusePort(other.reusePort),
    firstInput(std::min(other.firstInput, std::max<size_t>(other.inputs.size(), 1) - 1)),
    primaryInput(other.primaryInput),
    cycleCount(other.cycleCount),
    bufferedPackets(std::max(other.bufferedPackets, MIN_BUFFERED_PACKETS)),
    maxInputPackets(std::max(other.maxInputPackets, MIN_INPUT_PACKETS)),
    maxOutputPackets(std::max(other.maxOutputPackets, MIN_OUTPUT_PACKETS)),
    sockBuffer(other.sockBuffer),
    remoteServer(other.remoteServer),
    allowedRemote(other.allowedRemote),
    receiveTimeout(other.receiveTimeout),
    inputs(other.inputs),
    output(other.output)
{
    if (inputs.empty()) {
        // If no input plugin is used, used only standard input.
        inputs.push_back(PluginOptions(u"file"));
    }
    if (output.name.empty()) {
        output.set(u"file");
    }
    if (receiveTimeout <= 0 && primaryInput != NPOS) {
        receiveTimeout = DEFAULT_RECEIVE_TIMEOUT;
    }
}


//----------------------------------------------------------------------------
// Define command line options in an Args.
//----------------------------------------------------------------------------

void ts::InputSwitcherArgs::defineArgs(Args& args) const
{
    args.option(u"allow", 'a', Args::STRING);
    args.help(u"allow",
              u"Specify an IP address or host name which is allowed to send remote commands. "
              u"Several --allow options are allowed. By default, all remote commands are accepted.");

    args.option(u"buffer-packets", 'b', Args::POSITIVE);
    args.help(u"buffer-packets",
              u"Specify the size in TS packets of each input plugin buffer. "
              u"The default is " + UString::Decimal(DEFAULT_BUFFERED_PACKETS) + u" packets.");

    args.option(u"cycle", 'c', Args::POSITIVE);
    args.help(u"cycle",
              u"Specify how many times to repeat the cycle through all input plugins in sequence. "
              u"By default, all input plugins are executed in sequence only once (--cycle 1). "
              u"The options --cycle, --infinite and --terminate are mutually exclusive.");

    args.option(u"delayed-switch", 'd');
    args.help(u"delayed-switch",
              u"Perform delayed input switching. When switching from one input plugin to another one, "
              u"the second plugin is started first. Packets from the first plugin continue to be "
              u"output while the second plugin is starting. Then, after the second plugin starts to "
              u"receive packets, the switch occurs: packets are now fetched from the second plugin. "
              u"Finally, after the switch, the first plugin is stopped.");

    args.option(u"fast-switch", 'f');
    args.help(u"fast-switch",
              u"Perform fast input switching. All input plugins are started at once and they "
              u"continuously receive packets in parallel. Packets are dropped, except for the "
              u"current input plugin. This option is typically used when all inputs are live "
              u"streams on distinct devices (not the same DVB tuner for instance).\n\n"
              u"By default, only one input plugin is started at a time. When switching, "
              u"the current input is first stopped and then the next one is started.");

    args.option(u"first-input", 0, Args::UNSIGNED);
    args.help(u"first-input",
              u"Specify the index of the first input plugin to start. "
              u"By default, the first plugin (index 0) is used.");

    args.option(u"infinite", 'i');
    args.help(u"infinite", u"Infinitely repeat the cycle through all input plugins in sequence.");

    args.option(u"max-input-packets", 0, Args::POSITIVE);
    args.help(u"max-input-packets",
              u"Specify the maximum number of TS packets to read at a time. "
              u"This value may impact the switch response time. "
              u"The default is " + UString::Decimal(DEFAULT_MAX_INPUT_PACKETS) + u" packets. "
              u"The actual value is never more than half the --buffer-packets value.");

    args.option(u"max-output-packets", 0, Args::POSITIVE);
    args.help(u"max-output-packets",
              u"Specify the maximum number of TS packets to write at a time. "
              u"The default is " + UString::Decimal(DEFAULT_MAX_OUTPUT_PACKETS) + u" packets.");

    args.option(u"monitor", 'm');
    args.help(u"monitor",
              u"Continuously monitor the system resources which are used by tsswitch. "
              u"This includes CPU load, virtual memory usage. Useful to verify the "
              u"stability of the application.");

    args.option(u"primary-input", 'p', Args::UNSIGNED);
    args.help(u"primary-input",
              u"Specify the index of the input plugin which is considered as primary "
              u"or preferred. This input plugin is always started, never stopped, even "
              u"without --fast-switch. When no packet is received on this plugin, the "
              u"normal switching rules apply. However, as soon as packets are back on "
              u"the primary input, the reception is immediately switched back to it. "
              u"By default, there is no primary input, all input plugins are equal.");

    args.option(u"no-reuse-port");
    args.help(u"no-reuse-port",
              u"Disable the reuse port socket option for the remote control. "
              u"Do not use unless completely necessary.");

    args.option(u"receive-timeout", 0, Args::UNSIGNED);
    args.help(u"receive-timeout",
              u"Specify a receive timeout in milliseconds. "
              u"When the current input plugin has received no packet within "
              u"this timeout, automatically switch to the next plugin. "
              u"By default, without --primary-input, there is no automatic switch "
              u"when the current input plugin is waiting for packets. With "
              u"--primary-input, the default is " + UString::Decimal(DEFAULT_RECEIVE_TIMEOUT) + u" ms.");

    args.option(u"remote", 'r', Args::STRING);
    args.help(u"remote", u"[address:]port",
              u"Specify the local UDP port which is used to receive remote commands. "
              u"If an optional address is specified, it must be a local IP address of the system. "
              u"By default, there is no remote control.");

    args.option(u"terminate", 't');
    args.help(u"terminate", u"Terminate execution when the current input plugin terminates.");

    args.option(u"udp-buffer-size", 0, Args::UNSIGNED);
    args.help(u"udp-buffer-size",
              u"Specifies the UDP socket receive buffer size (socket option).");
}


//----------------------------------------------------------------------------
// Load arguments from command line.
//----------------------------------------------------------------------------

bool ts::InputSwitcherArgs::loadArgs(DuckContext& duck, Args& args)
{
    appName = args.appName();
    fastSwitch = args.present(u"fast-switch");
    delayedSwitch = args.present(u"delayed-switch");
    terminate = args.present(u"terminate");
    cycleCount = args.intValue<size_t>(u"cycle", args.present(u"infinite") ? 0 : 1);
    monitor = args.present(u"monitor");
    bufferedPackets = args.intValue<size_t>(u"buffer-packets", DEFAULT_BUFFERED_PACKETS);
    maxInputPackets = std::min(args.intValue<size_t>(u"max-input-packets", DEFAULT_MAX_INPUT_PACKETS), bufferedPackets / 2);
    maxOutputPackets = args.intValue<size_t>(u"max-output-packets", DEFAULT_MAX_OUTPUT_PACKETS);
    const UString remoteName(args.value(u"remote"));
    reusePort = !args.present(u"no-reuse-port");
    sockBuffer = args.intValue<size_t>(u"udp-buffer-size");
    firstInput = args.intValue<size_t>(u"first-input", 0);
    primaryInput = args.intValue<size_t>(u"primary-input", NPOS);
    receiveTimeout = args.intValue<MilliSecond>(u"receive-timeout", primaryInput >= inputs.size() ? 0 : DEFAULT_RECEIVE_TIMEOUT);

    // Check conflicting modes.
    if (args.present(u"cycle") + args.present(u"infinite") + args.present(u"terminate") > 1) {
        args.error(u"options --cycle, --infinite and --terminate are mutually exclusive");
    }
    if (fastSwitch && delayedSwitch) {
        args.error(u"options --delayed-switch and --fast-switch are mutually exclusive");
    }

    // Resolve remote control name.
    if (!remoteName.empty() && remoteServer.resolve(remoteName, args) && !remoteServer.hasPort()) {
        args.error(u"missing UDP port number in --remote");
    }

    // Resolve all allowed remote.
    UStringVector remotes;
    args.getValues(remotes, u"allow");
    allowedRemote.clear();
    for (auto it = remotes.begin(); it != remotes.end(); ++it) {
        const IPAddress addr(*it, args);
        if (addr.hasAddress()) {
            allowedRemote.insert(addr);
        }
    }

    // Load all plugin descriptions. Default output is the standard output file.
    ArgsWithPlugins* pargs = dynamic_cast<ArgsWithPlugins*>(&args);
    if (pargs != nullptr) {
        pargs->getPlugins(inputs, PluginType::INPUT);
        pargs->getPlugin(output, PluginType::OUTPUT, u"file");
    }
    else {
        inputs.clear();
        output.set(u"file");
    }
    if (inputs.empty()) {
        // If no input plugin is used, used only standard input.
        inputs.push_back(PluginOptions(u"file"));
    }

    // Check validity of input indexes.
    if (firstInput >= inputs.size()) {
        args.error(u"invalid input index for --first-input %d", {firstInput});
    }

    if (primaryInput != NPOS && primaryInput >= inputs.size()) {
        args.error(u"invalid input index for --primary-input %d", {primaryInput});
    }

    return args.valid();
}
