//----------------------------------------------------------------------------
//
// TSDuck - The MPEG Transport Stream Toolkit
// Copyright (c) 2005-2020, Thierry Lelegard
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
//    this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
// THE POSSIBILITY OF SUCH DAMAGE.
//
//----------------------------------------------------------------------------

#include "tsRRT.h"
#include "tsBinaryTable.h"
#include "tsTablesDisplay.h"
#include "tsPSIRepository.h"
#include "tsPSIBuffer.h"
#include "tsDuckContext.h"
#include "tsxmlElement.h"
TSDUCK_SOURCE;

#define MY_XML_NAME u"RRT"
#define MY_CLASS ts::RRT
#define MY_TID ts::TID_RRT
#define MY_STD ts::Standards::ATSC

TS_REGISTER_TABLE(MY_CLASS, {MY_TID}, MY_STD, MY_XML_NAME, MY_CLASS::DisplaySection);


//----------------------------------------------------------------------------
// Constructors
//----------------------------------------------------------------------------

ts::RRT::RRT(uint8_t vers, uint8_t reg) :
    AbstractLongTable(MY_TID, MY_XML_NAME, MY_STD, vers, true), // RRT is always "current"
    rating_region(reg),
    protocol_version(0),
    rating_region_name(),
    dimensions(),
    descs(this)
{
}

ts::RRT::RRT(const RRT& other) :
    AbstractLongTable(other),
    rating_region(other.rating_region),
    protocol_version(other.protocol_version),
    rating_region_name(other.rating_region_name),
    dimensions(other.dimensions),
    descs(this, other.descs)
{
}

ts::RRT::RRT(DuckContext& duck, const BinaryTable& table) :
    RRT()
{
    deserialize(duck, table);
}


ts::RRT::Dimension::Dimension() :
    graduated_scale(false),
    dimension_name(),
    values()
{
}

ts::RRT::RatingValue::RatingValue() :
    abbrev_rating_value(),
    rating_value()
{
}


//----------------------------------------------------------------------------
// Get the table id extension.
//----------------------------------------------------------------------------

uint16_t ts::RRT::tableIdExtension() const
{
    return 0xFF00 | rating_region;
}


//----------------------------------------------------------------------------
// Clear the content of the table.
//----------------------------------------------------------------------------

void ts::RRT::clearContent()
{
    rating_region = 0;
    protocol_version = 0;
    rating_region_name.clear();
    dimensions.clear();
    descs.clear();
}


//----------------------------------------------------------------------------
// Deserialization
//----------------------------------------------------------------------------

void ts::RRT::deserializePayload(PSIBuffer& buf, const Section& section)
{
    rating_region = uint8_t(section.tableIdExtension());
    protocol_version = buf.getUInt8();
    buf.getMultipleStringWithLength(rating_region_name);

    // Loop on all dimensions.
    size_t dim_count = buf.getUInt8();
    while (!buf.error() && dim_count-- > 0) {
        Dimension dim;
        buf.getMultipleStringWithLength(dim.dimension_name);
        buf.skipBits(3);
        dim.graduated_scale = buf.getBit() != 0;
        size_t val_count = buf.getBits<size_t>(4);

        // Loop on all values.
        while (val_count-- > 0) {
            RatingValue val;
            buf.getMultipleStringWithLength(val.abbrev_rating_value);
            buf.getMultipleStringWithLength(val.rating_value);
            dim.values.push_back(val);
        }

        dimensions.push_back(dim);
    }

    // Get global descriptor list (with 10-bit length field).
    buf.getDescriptorListWithLength(descs, 10);
}


//----------------------------------------------------------------------------
// Serialization
//----------------------------------------------------------------------------

void ts::RRT::serializePayload(BinaryTable& table, PSIBuffer& buf) const
{
    // An RRT is not allowed to use more than one section, see A/65, section 6.4.

    if (dimensions.size() > 255) {
        // Too many dimensions, invalid.
        buf.setUserError();
        return;
    }

    buf.putUInt8(protocol_version);
    buf.putMultipleStringWithLength(rating_region_name);
    buf.putUInt8(uint8_t(dimensions.size()));

    // Loop on dimensions definitions.
    for (auto dim = dimensions.begin(); !buf.error() && dim != dimensions.end(); ++dim) {
        if (dim->values.size() > 15) {
            // Too many value, invalid.
            buf.setUserError();
            return;
        }
        buf.putMultipleStringWithLength(dim->dimension_name);
        buf.putBits(0xFF, 3);
        buf.putBit(dim->graduated_scale);
        buf.putBits(dim->values.size(), 4);
        for (auto val = dim->values.begin(); !buf.error() && val != dim->values.end(); ++val) {
            buf.putMultipleStringWithLength(val->abbrev_rating_value);
            buf.putMultipleStringWithLength(val->rating_value);
        }
    }

    // Insert common descriptor list (with leading 10-bit length field)
    buf.putPartialDescriptorListWithLength(descs, 0, NPOS, 10);
}


//----------------------------------------------------------------------------
// A static method to display an RRT section.
//----------------------------------------------------------------------------

void ts::RRT::DisplaySection(TablesDisplay& disp, const ts::Section& section, PSIBuffer& buf, const UString& margin)
{
    disp << margin << UString::Format(u"Rating region: 0x%X (%<d)", {uint8_t(section.tableIdExtension())}) << std::endl;

    if (buf.remainingReadBytes() < 2) {
        buf.setUserError();
    }
    else {
        disp << margin << UString::Format(u"Protocol version: %d", {buf.getUInt8()}) << std::endl;
        disp.displayATSCMultipleString(buf, 1, margin, u"Rating region name: ");
    }

    // Display all dimensions.
    const size_t dim_count = buf.error() ? 0 : buf.getUInt8();
    disp << margin << "Number of dimensions: " << dim_count << std::endl;
    for (size_t dim_index = 0; !buf.error() && dim_index < dim_count; ++dim_index) {
        disp << margin << "- Dimension " << dim_index << std::endl;
        disp.displayATSCMultipleString(buf, 1, margin + u"  ", u"Dimension name: ");
        buf.skipBits(3);
        disp << margin << UString::Format(u"  Graduated scale: %s", {buf.getBit() != 0});
        size_t val_count = buf.getBits<size_t>(4);
        disp << ", number of rating values: " << val_count << std::endl;

        // Display all values.
        while (val_count-- > 0) {
            disp.displayATSCMultipleString(buf, 1, margin + u"  ", u"- Abbreviated rating value: ");
            disp.displayATSCMultipleString(buf, 1, margin + u"  ", u"  Rating value: ");
        }
    }

    // Common descriptors.
    disp.displayDescriptorListWithLength(section, buf, margin, u"Descriptors", UString(), 10);
    disp.displayExtraData(buf, margin);
}


//----------------------------------------------------------------------------
// XML serialization
//----------------------------------------------------------------------------

void ts::RRT::buildXML(DuckContext& duck, xml::Element* root) const
{
    root->setIntAttribute(u"version", version);
    root->setIntAttribute(u"protocol_version", protocol_version);
    root->setIntAttribute(u"rating_region", rating_region, true);
    rating_region_name.toXML(duck, root, u"rating_region_name", true);

    for (auto dim = dimensions.begin(); dim != dimensions.end(); ++dim) {
        xml::Element* xdim = root->addElement(u"dimension");
        xdim->setBoolAttribute(u"graduated_scale", dim->graduated_scale);
        dim->dimension_name.toXML(duck, xdim, u"dimension_name", true);
        for (auto val = dim->values.begin(); val != dim->values.end(); ++val) {
            xml::Element* xval = xdim->addElement(u"value");
            val->abbrev_rating_value.toXML(duck, xval, u"abbrev_rating_value", true);
            val->rating_value.toXML(duck, xval, u"rating_value", true);
        }
    }

    descs.toXML(duck, root);
}


//----------------------------------------------------------------------------
// XML deserialization
//----------------------------------------------------------------------------

bool ts::RRT::analyzeXML(DuckContext& duck, const xml::Element* element)
{
    xml::ElementVector xdim;
    bool ok =
        element->getIntAttribute<uint8_t>(version, u"version", false, 0, 0, 31) &&
        element->getIntAttribute<uint8_t>(protocol_version, u"protocol_version", false, 0) &&
        element->getIntAttribute<uint8_t>(rating_region, u"rating_region", true) &&
        rating_region_name.fromXML(duck, element, u"rating_region_name", false) &&
        descs.fromXML(duck, xdim, element, u"rating_region_name,dimension");

    for (size_t idim = 0; ok && idim < xdim.size(); ++idim) {
        // The extracted non-descriptor children can be <rating_region_name> or <dimension>.
        // The optional <rating_region_name> has already been separately processed.
        // Process <dimension> only.
        if (xdim[idim]->name().similar(u"dimension")) {
            Dimension dim;
            xml::ElementVector xval;
            ok = xdim[idim]->getBoolAttribute(dim.graduated_scale, u"graduated_scale", true) &&
                 dim.dimension_name.fromXML(duck, xdim[idim], u"dimension_name", false) &&
                 xdim[idim]->getChildren(xval, u"value", 0, 15);
            for (size_t ival = 0; ok && ival < xval.size(); ++ival) {
                RatingValue val;
                ok = val.abbrev_rating_value.fromXML(duck, xval[ival], u"abbrev_rating_value", false) &&
                     val.rating_value.fromXML(duck, xval[ival], u"rating_value", false);
                if (ok) {
                    dim.values.push_back(val);
                }
            }
            if (ok) {
                dimensions.push_back(dim);
            }
        }
    }
    return ok;
}
