//----------------------------------------------------------------------------
//
// TSDuck - The MPEG Transport Stream Toolkit
// Copyright (c) 2005-2020, Thierry Lelegard
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice,
//    this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
// THE POSSIBILITY OF SUCH DAMAGE.
//
//----------------------------------------------------------------------------

#include "tsxmlNode.h"
#include "tsxmlComment.h"
#include "tsxmlDeclaration.h"
#include "tsxmlDocument.h"
#include "tsxmlElement.h"
#include "tsxmlText.h"
#include "tsxmlUnknown.h"
#include "tsTextFormatter.h"
#include "tsNullReport.h"
TSDUCK_SOURCE;


//----------------------------------------------------------------------------
// Constructors and destructors.
//----------------------------------------------------------------------------

ts::xml::Node::Node(Report& report, size_t line) :
    RingNode(),
    _report(report),
    _value(),
    _parent(nullptr),
    _firstChild(nullptr),
    _inputLineNum(line)
{
}

ts::xml::Node::Node(Node* parent, const UString& value, bool last) :
    Node(parent == nullptr ? *static_cast<Report*>(&NULLREP) : parent->_report, 0)
{
    setValue(value);
    reparent(parent, last);
}

ts::xml::Node::~Node()
{
    clear();
    reparent(nullptr);
}


//----------------------------------------------------------------------------
// Simple virtual methods
//----------------------------------------------------------------------------

void ts::xml::Node::printClose(TextFormatter& output, size_t levels) const
{
}

bool ts::xml::Node::stickyOutput() const
{
    return false;
}


//----------------------------------------------------------------------------
// Clear the content of the node.
//----------------------------------------------------------------------------

void ts::xml::Node::clear()
{
    // Free all our children nodes.
    while (_firstChild != nullptr) {
        // The child will cleanly remove itself from the list of children.
        delete _firstChild;
    }

    // Clear other fields.
    _value.clear();
    _inputLineNum = 0;
}


//----------------------------------------------------------------------------
// Attach the node to a new parent.
//----------------------------------------------------------------------------

void ts::xml::Node::reparent(Node* newParent, bool last)
{
    // If the parent does not change (including zero), nothing to do.
    if (newParent == _parent) {
        return;
    }

    // Detach from our parent.
    if (_parent != nullptr) {
        // If we are the first child, make the parent point to the next child.
        // Unless we are alone in the ring of children, in which case the parent has no more children.
        if (_parent->_firstChild == this) {
            _parent->_firstChild = ringAlone() ? nullptr : ringNext<Node>();
        }
        // Remove ourselves from our parent's children.
        ringRemove();
    }

    // Set new parent.
    _parent = newParent;

    // Insert inside new parent structure.
    if (_parent != nullptr) {
        if (_parent->_firstChild == nullptr) {
            // We become the only child of the parent.
            _parent->_firstChild = this;
        }
        else {
            // Insert in the ring of children, "before the first child", meaning at end of list.
            ringInsertBefore(_parent->_firstChild);
            if (!last) {
                // If we need to be added as first child, simply adjust the pointer to the first child.
                _parent->_firstChild = this;
            }
        }
    }
}


//----------------------------------------------------------------------------
// Get the document into which the node is located.
//----------------------------------------------------------------------------

ts::xml::Document* ts::xml::Node::document()
{
    Node* node = this;
    while (node->_parent != nullptr) {
        node = node->_parent;
    }
    return dynamic_cast<Document*>(node);
}


//----------------------------------------------------------------------------
// Get the depth of an XML element.
//----------------------------------------------------------------------------

size_t ts::xml::Node::depth() const
{
    size_t count = 0;
    const Node* node = _parent;
    while (node != nullptr) {
        node = node->_parent;
        count++;
        // Fool-proof check.
        assert(count < 1024);
    }
    return count;
}


//----------------------------------------------------------------------------
// Get the current XML parsing and formatting tweaks for this node.
//----------------------------------------------------------------------------

const ts::xml::Tweaks ts::xml::Node::defaultTweaks;

const ts::xml::Tweaks& ts::xml::Node::tweaks() const
{
    const ts::xml::Document* const doc = document();
    return doc != nullptr ? doc->tweaks() : defaultTweaks;
}


//----------------------------------------------------------------------------
// Get the next or previous sibling node.
//----------------------------------------------------------------------------

ts::xml::Node* ts::xml::Node::nextSibling()
{
    // When the ring points to the first child, this is the end of the list.
    Node* next = ringNext<Node>();
    return next == this || (_parent != nullptr && next == _parent->_firstChild) ? nullptr : next;
}

ts::xml::Node* ts::xml::Node::previousSibling()
{
    Node* prev = ringPrevious<Node>();
    return prev == this || (_parent != nullptr && this == _parent->_firstChild) ? nullptr : prev;
}


//----------------------------------------------------------------------------
// Find the next sibling element.
//----------------------------------------------------------------------------

ts::xml::Element* ts::xml::Node::nextSiblingElement()
{
    for (Node* child = nextSibling(); child != nullptr; child = child->nextSibling()) {
        Element* elem = dynamic_cast<Element*>(child);
        if (elem != nullptr) {
            return elem;
        }
    }
    return nullptr;
}


//----------------------------------------------------------------------------
// Find the first child element by name, case-insensitive.
//----------------------------------------------------------------------------

ts::xml::Element* ts::xml::Node::firstChildElement()
{
    // Loop on all children.
    for (Node* child = firstChild(); child != nullptr; child = child->nextSibling()) {
        Element* elem = dynamic_cast<Element*>(child);
        if (elem != nullptr) {
            return elem;
        }
    }
    return nullptr;
}


//----------------------------------------------------------------------------
// Parse children nodes and add them to the node.
//----------------------------------------------------------------------------

bool ts::xml::Node::parseChildren(TextParser& parser)
{
    bool result = true;
    Node* node;

    // Loop on each token we find.
    // Exit loop either at end of document or before a "</" sequence.
    while ((node = identifyNextNode(parser)) != nullptr) {

        // Read the complete node.
        if (node->parseNode(parser, this)) {
            // The child node is fine, insert it.
            node->reparent(this);
        }
        else {
            // Error, we expect the child's parser to have displayed the error message.
            delete node;
            result = false;
        }
    }

    return result;
}


//----------------------------------------------------------------------------
// Build a debug string for the node.
//----------------------------------------------------------------------------

ts::UString ts::xml::Node::debug() const
{
    return UString::Format(u"%s, line %d, children: %d, value '%s'", {typeName(), lineNumber(), childrenCount(), value()});
}


//----------------------------------------------------------------------------
// Identify the next token in the document.
//----------------------------------------------------------------------------

ts::xml::Node* ts::xml::Node::identifyNextNode(TextParser& parser)
{
    // Save the current state in case we realize that the leading spaces are part of the token.
    const TextParser::Position previous(parser.position());

    // Skip all white spaces until next token.
    parser.skipWhiteSpace();

    // Stop at end of document or before "</".
    if (parser.eof() || parser.match(u"</", false)) {
        return nullptr;
    }

    // Check each expected token.
    if (parser.match(u"<?", true)) {
        return new Declaration(_report, parser.lineNumber());
    }
    else if (parser.match(u"<!--", true)) {
        return new Comment(_report, parser.lineNumber());
    }
    else if (parser.match(u"<![CDATA[", true, CASE_INSENSITIVE)) {
        return new Text(_report, parser.lineNumber(), true);
    }
    else if (parser.match(u"<!", true)) {
        // Should be a DTD, we ignore it.
        return new Unknown(_report, parser.lineNumber());
    }
    else if (parser.match(u"<", true)) {
        return new Element(_report, parser.lineNumber());
    }
    else {
        // This must be a text node. Revert skipped spaces, they are part of the text.
        parser.seek(previous);
        return new Text(_report, parser.lineNumber(), false);
    }
}
