#!/usr/bin/env python3

import json
import sys
import urllib.request

import yaml

from tortoisehg.util import hgversion


def job_name(name, pyver, hgver):
    return f'{name}-py{pyver}-hg-{hgver}'


def docker_image_name(pyver, hgver, prebuilt_images):
    if f'hg-{hgver}' in prebuilt_images:
        suffix = f':hg-{hgver}'
    else:
        suffix = ''
    return f'octobus/ci-py{pyver}-hgext3rd' + suffix


def gen_install_apt_dependencies(pyver, extra=[]):
    if pyver == 2:
        python_prefix = 'python'
    elif pyver == 3:
        python_prefix = 'python3'
    return [
        'apt-get update',
        'apt-get -y install '
        + ' '.join(
            [f'{python_prefix}-pyqt5', f'{python_prefix}-pyqt5.qsci'] + extra
        ),
    ]


def gen_update_to_mercurial_rev(pyver, hgver):
    return [
        'hg -R /ci/repos/mercurial pull',
        f'hg -R /ci/repos/mercurial update {hgver}',
        'hg -R /ci/repos/mercurial summary',
    ]


def gen_ensure_mercurial_build(pyver, hgver, prebuilt_images):
    if f'hg-{hgver}' in prebuilt_images:
        build = []
    else:
        build = gen_update_to_mercurial_rev(pyver, hgver) + [
            f'make -C /ci/repos/mercurial local PYTHON=python{pyver}',
        ]
    return build + ['/ci/repos/mercurial/hg version --debug']


def thg_make_command(target, pyver):
    return f'make {target} PYTHON=python{pyver} HGPATH=/ci/repos/mercurial'


def gen_gitlab_ci_yml():
    with urllib.request.urlopen(
        'https://index.docker.io/v1/repositories/octobus/ci-py3-hgext3rd/tags'
    ) as r:
        prebuilt_images = [t['name'] for t in json.load(r)]

    tests = {
        job_name('tests', pyver, hgver): {
            'image': docker_image_name(pyver, hgver, prebuilt_images),
            'script': [
                *gen_install_apt_dependencies(pyver, extra=['xvfb']),
                f'pip{pyver} install --user nose mock',
                *gen_ensure_mercurial_build(pyver, hgver, prebuilt_images),
                thg_make_command('local', pyver),
                'xvfb-run ' + thg_make_command('tests', pyver),
            ],
            'allow_failure': hgver == 'default',
        }
        for pyver in [2, 3]
        for hgver in hgversion.testedwith.decode().split()
        + ['stable', 'default']
    }
    pytype = {
        job_name('pytype', 3, hgver): {
            'image': docker_image_name(3, hgver, prebuilt_images),
            'script': [
                *gen_install_apt_dependencies(3),
                *gen_update_to_mercurial_rev(3, hgver),
                thg_make_command('local', 3),
                thg_make_command('pytype', 3),
            ],
            'allow_failure': True,
        }
        for hgver in ['stable', 'default']
    }
    return {**tests, **pytype}


if __name__ == '__main__':
    toplevel = gen_gitlab_ci_yml()
    sys.stdout.write(
        "# This file was generated by contrib/generate_gitlab_ci_yml.py.\n\n"
    )
    yaml.dump(toplevel, sys.stdout, sort_keys=False)
    sys.stdout.flush()
