import os, tempfile
from nose.tools import *

from mercurial import hg, ui
from tortoisehg.util import hglib

import helpers

def setup():
    global _tmpdir
    _tmpdir = helpers.mktmpdir(__name__)

    hgcli = helpers.HgClient(os.path.join(_tmpdir, 'unapplied'))
    hgcli.init()
    hgcli.qinit('-c')
    hgcli.qnew('foo')
    hgcli.qnew('bar')
    hgcli.qnew('baz')
    hgcli.qnew('qux')
    hgcli.qnew('quux')
    hgcli.qpop('-a')
    hgcli.qguard('bar', '+guard')
    # series file may contain comment or empty line
    hgcli.fwrite('.hg/patches/series',
                 '# comment\n' + hgcli.fread('.hg/patches/series'))
    hgcli.commit('--mq', '-m', 'patches')

def openrepo(name):
    dest = tempfile.mkdtemp('', name + '-clone', _tmpdir)
    hgcli = helpers.HgClient(dest)
    hgcli.qclone(os.path.join(_tmpdir, name), '.')
    return hg.repository(ui.ui(), dest), hgcli


def test_movefirst():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, None, ['baz'])
    assert_equal(['baz', 'foo', 'bar', 'qux', 'quux'], repo.mq.series)
    assert_equal(['baz', '# comment', 'foo', 'bar #+guard', 'qux', 'quux'],
                 repo.mq.fullseries)

def test_movelast():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, 'quux', ['baz'])
    assert_equal(['foo', 'bar', 'qux', 'quux', 'baz'], repo.mq.series)
    assert_equal(['# comment', 'foo', 'bar #+guard', 'qux', 'quux', 'baz'],
                 repo.mq.fullseries)

def test_moveintersect():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, 'baz', ['bar', 'quux'])
    assert_equal(['foo', 'baz', 'bar', 'quux', 'qux'], repo.mq.series)
    assert_equal(['# comment', 'foo', 'baz', 'bar #+guard', 'quux', 'qux'],
                 repo.mq.fullseries)


def test_moveafterapplied():
    repo, hgcli = openrepo('unapplied')
    hgcli.qpush('baz')
    hglib.movemqpatches(repo, 'baz', ['quux'])
    assert_equal(['foo', 'bar', 'baz', 'quux', 'qux'], repo.mq.series)

def test_moveguarded():
    repo, hgcli = openrepo('unapplied')
    hgcli.qpush('baz')
    hglib.movemqpatches(repo, 'baz', ['bar'])
    assert_equal(['foo', 'baz', 'bar', 'qux', 'quux'], repo.mq.series)

@raises(ValueError)
def test_movebeforeapplied():
    repo, hgcli = openrepo('unapplied')
    hgcli.qpush('baz')
    hglib.movemqpatches(repo, 'bar', ['quux'])

@raises(ValueError)
def test_moveapplied():
    repo, hgcli = openrepo('unapplied')
    hgcli.qpush('baz')
    hglib.movemqpatches(repo, 'baz', ['foo'])


@raises(ValueError)
def test_unknownpatch():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, 'baz', ['unknown'])

@raises(ValueError)
def test_unknowninsertpos():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, 'unknown', ['foo'])

@raises(ValueError)
def test_selfinsertpos():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, 'foo', ['foo'])


def test_save():
    repo, hgcli = openrepo('unapplied')
    hglib.movemqpatches(repo, 'baz', ['bar', 'quux'])
    assert_equal('# comment\nfoo\nbaz\nbar #+guard\nquux\nqux\n',
                 hgcli.fread('.hg/patches/series'))

def test_locked():
    repo, hgcli = openrepo('unapplied')
    hgcli.ftouch('.hg/wlock')
    r = hglib.movemqpatches(repo, 'baz', ['quux'])
    assert_equals(False, r)
