import os, sys, unittest
from nose.plugins.skip import SkipTest
from PyQt4.QtCore import QModelIndex, QString
from PyQt4.QtGui import QApplication
from tortoisehg.hgqt import modeltest, thgrepo
from tortoisehg.hgqt.manifestmodel import ManifestModel

import helpers

def setup():
    # necessary for style().standardIcon()
    if QApplication.type() != QApplication.GuiClient:
        raise SkipTest

    global _tmpdir
    _tmpdir = helpers.mktmpdir(__name__)

def alldata(model, parent=QModelIndex()):
    return [model.data(model.index(r, 0, parent))
            for r in xrange(model.rowCount(parent))]


class ManifestModelTest(unittest.TestCase):
    @classmethod
    def setUpClass(cls):
        hg = helpers.HgClient(os.path.join(_tmpdir, cls.__name__))
        hg.init()
        hg.ftouch('foo', 'bar', 'baz/bax', 'baz/box')
        hg.addremove()
        hg.commit('-m', 'foobar')

        hg.fwrite('bar', 'hello\n')
        hg.remove('baz/box')
        hg.ftouch('zzz')
        hg.addremove()
        hg.commit('-m', 'remove baz/box, add zzz, modify bar')

        hg.clone('-U', '.', 'sub')
        hg.fappend('.hgsub', 'sub = sub\n')
        hg.commit('-Am', 'add empty sub')

        hg.update('-R', 'sub', '1')
        hg.commit('-Am', 'update sub')

        cls.repo = thgrepo.repository(path=hg.path)

    @classmethod
    def tearDownClass(cls):
        del cls.repo

    def new_model(self, rev):
        m = ManifestModel(self.repo, rev)
        # install common assertions triggered by model signals
        modeltest.ModelTest(m, m)
        return m

    def test_data(self):
        m = self.new_model(0)
        self.assertEqual('bar', m.data(m.index(1, 0)))
        self.assertEqual('baz', m.data(m.index(0, 0)))
        self.assertEqual('foo', m.data(m.index(2, 0)))

    def test_data_subdir(self):
        m = self.new_model(0)
        self.assertEqual('bax', m.data(m.index(0, 0, m.index(0, 0))))
        self.assertEqual('box', m.data(m.index(1, 0, m.index(0, 0))))

    def test_data_inexistent(self):
        m = self.new_model(0)
        self.assertEqual(None, m.data(QModelIndex()))
        self.assertEqual(None, m.data(m.index(0, 0, m.index(1, 0))))

    def test_data_subrepo(self):
        m = self.new_model(3)
        self.assertEqual(['baz', 'sub', '.hgsub'],  alldata(m)[:3])
        self.assertEqual(['baz', 'bar', 'foo', 'zzz'],
                         alldata(m, m.indexFromPath('sub')))

    def test_data_empty_subrepo(self):
        m = self.new_model(2)
        self.assertEqual(['baz', 'sub', '.hgsub'],  alldata(m)[:3])

    def test_isdir(self):
        m = self.new_model(3)
        self.assertTrue(m.isDir(m.indexFromPath('')))
        self.assertTrue(m.isDir(m.indexFromPath('baz')))
        self.assertFalse(m.isDir(m.indexFromPath('foo')))
        self.assertTrue(m.isDir(m.indexFromPath('sub')))
        self.assertFalse(m.isDir(m.indexFromPath('sub/foo')))
        self.assertTrue(m.isDir(m.indexFromPath('sub/baz')))

    def test_rowcount(self):
        m = self.new_model(0)
        self.assertEqual(3, m.rowCount())

    def test_rowcount_subdirs(self):
        m = self.new_model(0)
        self.assertEqual(2, m.rowCount(m.index(0, 0)))

    def test_rowcount_invalid(self):
        m = self.new_model(0)
        self.assertEqual(0, m.rowCount(m.index(1, 0)))

    def test_pathfromindex(self):
        m = self.new_model(0)
        self.assertEqual('', m.filePath(QModelIndex()))
        self.assertEqual('bar', m.filePath(m.index(1, 0)))
        self.assertEqual('baz', m.filePath(m.index(0, 0)))
        self.assertEqual('baz/bax', m.filePath(m.index(0, 0, m.index(0, 0))))

    def test_indexfrompath(self):
        m = self.new_model(0)
        self.assertEqual(QModelIndex(), m.indexFromPath(''))
        self.assertEqual(m.index(1, 0), m.indexFromPath('bar'))
        self.assertEqual(m.index(0, 0), m.indexFromPath('baz'))
        self.assertEqual(m.index(0, 0, m.index(0, 0)),
                         m.indexFromPath('baz/bax'))

    def test_indexfrompath_qstr(self):
        m = self.new_model(0)
        self.assertEqual(m.index(1, 0), m.indexFromPath(QString('bar')))

    def test_subrepotype(self):
        m = self.new_model(2)
        self.assertEqual('hg', m.subrepoType(m.indexFromPath('sub')))
        self.assertEqual(None, m.subrepoType(m.indexFromPath('foo')))

    def test_removed_should_be_listed(self):
        m = self.new_model(1)
        m.setStatusFilter('MARC')
        self.assertTrue(m.indexFromPath('baz/box').isValid())

    def test_status_role(self):
        m = self.new_model(0)
        self.assertEqual('A', m.data(m.indexFromPath('foo'),
                                     role=ManifestModel.StatusRole))

        m = self.new_model(1)
        m.setStatusFilter('MARC')
        self.assertEqual('C', m.data(m.indexFromPath('foo'),
                                     role=ManifestModel.StatusRole))
        self.assertEqual('R', m.data(m.indexFromPath('baz/box'),
                                     role=ManifestModel.StatusRole))

        m = self.new_model(2)
        self.assertEqual('S', m.data(m.indexFromPath('sub'),
                                     role=ManifestModel.StatusRole))

    def test_status_role_invalid(self):
        m = self.new_model(0)
        self.assertEqual(None, m.data(QModelIndex(),
                                      role=ManifestModel.StatusRole))

    def test_status_filter_modified(self):
        m = self.new_model(1)
        m.setStatusFilter('M')
        self.assertNotEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertEqual(QModelIndex(), m.indexFromPath('zzz'))  # added
        self.assertEqual(QModelIndex(), m.indexFromPath('baz/box'))  # removed
        self.assertEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

    def test_status_filter_added(self):
        m = self.new_model(1)
        m.setStatusFilter('A')
        self.assertEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertNotEqual(QModelIndex(), m.indexFromPath('zzz'))  # added
        self.assertEqual(QModelIndex(), m.indexFromPath('baz/box'))  # removed
        self.assertEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

    def test_status_filter_removed(self):
        m = self.new_model(1)
        m.setStatusFilter('R')
        self.assertEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertEqual(QModelIndex(), m.indexFromPath('zzz'))  # added
        self.assertNotEqual(QModelIndex(), m.indexFromPath('baz/box'))  # removed
        self.assertEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

    def test_status_filter_clean(self):
        m = self.new_model(1)
        m.setStatusFilter('C')
        self.assertEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertEqual(QModelIndex(), m.indexFromPath('zzz'))  # added
        self.assertEqual(QModelIndex(), m.indexFromPath('baz/box'))  # removed
        self.assertNotEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

    def test_status_filter_change(self):
        m = self.new_model(1)
        m.setStatusFilter('C')
        self.assertEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertNotEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

        m.setStatusFilter('M')
        self.assertNotEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

    def test_status_filter_subrepo(self):
        m = self.new_model(3)
        m.setStatusFilter('S')
        self.assertTrue(m.indexFromPath('sub').isValid())
        self.assertEqual(1, m.rowCount())
        self.assertEqual(0, m.rowCount(m.indexFromPath('sub')))

    def test_status_filter_multi(self):
        m = self.new_model(1)
        m.setStatusFilter('MC')
        self.assertNotEqual(QModelIndex(), m.indexFromPath('bar'))  # modified
        self.assertEqual(QModelIndex(), m.indexFromPath('zzz'))  # added
        self.assertEqual(QModelIndex(), m.indexFromPath('baz/box'))  # removed
        self.assertNotEqual(QModelIndex(), m.indexFromPath('foo'))  # clean

    def test_name_filter(self):
        m = self.new_model(0)
        m.setNameFilter('ax')
        self.assertFalse(m.indexFromPath('bar').isValid())
        self.assertTrue(m.indexFromPath('baz/bax').isValid())
        self.assertFalse(m.indexFromPath('baz/box').isValid())
        self.assertFalse(m.indexFromPath('foo').isValid())

    def test_name_filter_glob(self):
        m = self.new_model(0)
        m.setNameFilter('b*x')
        self.assertFalse(m.indexFromPath('bar').isValid())
        self.assertTrue(m.indexFromPath('baz/bax').isValid())
        self.assertTrue(m.indexFromPath('baz/box').isValid())
        self.assertFalse(m.indexFromPath('foo').isValid())


_aloha_ja = u'\u3042\u308d\u306f\u30fc'

class ManifestModelEucjpTest(unittest.TestCase):
    @classmethod
    def setUpClass(cls):
        # TODO: make this compatible with binary-unsafe filesystem
        if os.name != 'posix' or sys.platform == 'darwin':
            raise SkipTest
        cls.encodingpatch = helpers.patchencoding('euc-jp')

        # include non-ascii char in repo path to test concatenation
        hg = helpers.HgClient(os.path.join(
            _tmpdir, cls.__name__ + _aloha_ja.encode('euc-jp')))
        hg.init()
        hg.ftouch(_aloha_ja.encode('euc-jp'))
        hg.ftouch(_aloha_ja.encode('euc-jp') + '.txt')
        hg.addremove()
        hg.commit('-m', 'add aloha')
        cls.repo = thgrepo.repository(path=hg.path)

    @classmethod
    def tearDownClass(cls):
        del cls.repo
        cls.encodingpatch.restore()

    def new_model(self):
        return ManifestModel(self.repo, rev=0)

    def test_data(self):
        m = self.new_model()
        self.assertEqual(_aloha_ja, m.data(m.index(0, 0)))

    def test_pathfromindex(self):
        m = self.new_model()
        self.assertEqual(_aloha_ja, m.filePath(m.index(0, 0)))

    def test_indexfrompath(self):
        m = self.new_model()
        self.assertEqual(m.index(0, 0), m.indexFromPath(_aloha_ja))

    def test_fileicon_path_concat(self):
        m = self.new_model()
        m.fileIcon(m.indexFromPath(_aloha_ja + '.txt'))  # no unicode error
