/*
 * Copyright (c) 2004-2007, INRIA
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 
 * 	- Redistributions of source code must retain the above copyright
 * 	notice, this list of conditions and the following disclaimer.  
 * 	- Redistributions in binary form must reproduce the above copyright
 * 	notice, this list of conditions and the following disclaimer in the
 * 	documentation and/or other materials provided with the distribution.
 * 	- Neither the name of the INRIA nor the names of its
 * 	contributors may be used to endorse or promote products derived from
 * 	this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package xml;

import tom.library.xml.*;
import tom.library.adt.tnode.*;
import tom.library.adt.tnode.types.*;
import java.io.*;
import tom.library.sl.*;

public class HTMLToLaTeX {
  
  %include{ adt/tnode/TNode.tom }
  %include{ sl.tom }
    
  public static void main (String args[]) {
    HTMLToLaTeX html2latex = new HTMLToLaTeX();
    if (args.length == 2) {
      html2latex.run(args[0],args[1]);
    } else {
      System.out.println("Usage : java HTMLToLaTeX file.html output.tex\n"+
                         "\tfile.html must be a valid XHTML file");
    }
  }

  private static FileWriter out;
  
  private static void write(String s) {
    try {
      out.write(s);
    } catch (IOException e) {
      System.out.println("Error in write");
    }
  }

  private void run(String filename, String output){
    try {
      out = new FileWriter(output);
      XmlTools xtools = new XmlTools();
      TNode term = xtools.convertXMLToTNode(filename);
      toLaTeX(term);
      out.flush();
      out.close();
    } catch (IOException e) {
      System.out.println("Cannot open file "+output);
    }
  }
    
  private static boolean firstLine, firstCol;

  private static void toLaTeX(TNode subject) {
    Strategy rule = `ToLaTeX();
    try {
      System.out.println("subject = " + subject);
      `TopDownCollect(ToLaTeX()).visitLight(subject);
    } catch (VisitFailure e) {
      System.out.println("reduction failed on: " + subject);
    }
  }
  private static void toLaTeX(TNodeList subject) {
    Strategy rule = `ToLaTeX();
    try {
      System.out.println("subject = " + subject);
      `TopDownCollect(ToLaTeX()).visitLight(subject);
    } catch (VisitFailure e) {
      System.out.println("reduction failed on: " + subject);
    }
  }

  %strategy ToLaTeX() extends `Identity() {
    visit TNode {
      <html><head>h*</head> <body>b*</body> </html> -> {
        write("\\documentclass{article}\n");
        toLaTeX(`h);
        write("\\begin{document}\n\\maketitle\n");
        toLaTeX(`b);
        write("\\end{document}\n");
        throw new VisitFailure();
      }
      <title>#TEXT(title)</title> -> {
        write("\\title{"+`title+"}\n\\author{}\n"+
            "\\date{Auto-generated by HTMLToLaTeX}\n");
        throw new VisitFailure();
      }
      <p>data*</p> -> { toLaTeX(`data); throw new VisitFailure(); }
      <pre>data*</pre> -> { toLaTeX(`data); throw new VisitFailure(); }
      <blockquote>data*</blockquote> -> { toLaTeX(`data); throw new VisitFailure(); }
      <u>data*</u> -> { toLaTeX(`data); throw new VisitFailure(); }
      <s>data*</s> -> { toLaTeX(`data); throw new VisitFailure(); }
      <strike>data*</strike> -> { toLaTeX(`data); throw new VisitFailure(); }
      <font>data*</font> -> { toLaTeX(`data); throw new VisitFailure(); }
      <basefont>data*</basefont> -> { toLaTeX(`data); throw new VisitFailure(); }
      <h1>h*</h1> -> {
        write("\\section{");
        toLaTeX(`h);
        write("}\n");
        throw new VisitFailure();
      }
      <h2>h*</h2> -> {
        write("\\subsection{");
        toLaTeX(`h);
        write("}\n");
        throw new VisitFailure();
      }
      <h3>h*</h3> -> {
        write("\\subsubsection{");
        toLaTeX(`h);
        write("}\n");
        throw new VisitFailure();
      }
      <h4>h*</h4> -> {
        write("\\subsubsubsection{");
        toLaTeX(`h);
        write("}\n");
        throw new VisitFailure();
      }
      <h5>h*</h5> -> {
        write("\\subsubsubsubsection{");
        toLaTeX(`h);
        write("}\n");
        throw new VisitFailure();
      }
      <h6>h*</h6> -> {
        write("\\subsubsubsubsubsection{");
        toLaTeX(`h);
        write("}\n");
        throw new VisitFailure();
      }
      <ul>data*</ul> -> {
        write("\\begin{itemize}\n");
        toLaTeX(`data);
        write("\\end{itemize}\n");
        throw new VisitFailure();
      }
      <ol>data*</ol> -> {
        write("\\begin{enumerate}\n");
        toLaTeX(`data);
        write("\\end{enumerate}\n");
        throw new VisitFailure();
      }
      <dl>data*</dl> -> {
        write("\\begin{description}\n");
        toLaTeX(`data);
        write("\\end{description}\n");
        throw new VisitFailure();
      }
      <a href=hrefurl>data*</a> -> {
        toLaTeX(`data);
        //write(" (URL :"+hrefurl+") ");
        throw new VisitFailure();
      }
      <img alt=text/> -> {
        write("img : "+`text);
        throw new VisitFailure();
      }
      
      <(li|dd|dt)>data*</(li|dd|dt)> -> { write("\\item "); toLaTeX(`data); write("\n"); throw new VisitFailure(); }

      <code>data*</code> -> {
        write("{\\tt ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <tt>data*</tt> -> {
        write("{\\tt ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <i>data*</i> -> {
        write("{\\it ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <b>data*</b> -> {
        write("{\\bf ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <big>data*</big> -> {
        write("{\\big ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <small>data*</small> -> {
        write("{\\small ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <em>data*</em> -> {
        write("{\\em ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <br/> -> {
        write("\n\n");
        throw new VisitFailure();
      }
      <center>data*</center> -> {
        write("\\begin{center}\n");
        toLaTeX(`data);
        write("\\end{center}\n");
        throw new VisitFailure();
      }
      <meta content=c/> -> {
        write("% meta : content="+`c);
        throw new VisitFailure();
      }
      <link/> -> { throw new VisitFailure(); }
      <style/> -> { throw new VisitFailure(); }
      <table>data*</table> -> {
        write("\\begin{tabular}{");
        write(columnSpec(tableMaxCol(0,0,`data)));
        write("}\n");
        firstLine = true;
        toLaTeX(`data);
        write("\\end{tabular}\n");
        throw new VisitFailure();
      }
      <tr>data*</tr> -> {
        if (!firstLine) {
          write(" \\\\\n");
        } else {
          firstLine = false;
        }
        firstCol = true;
        toLaTeX(`data);
        throw new VisitFailure();
      }
      <td>data*</td> -> {
        if (!firstCol) {
          write(" & ");
        } else {
          firstCol = false;
        }
        toLaTeX(`data);
        throw new VisitFailure();
      }
      <th>data*</th> -> {
        if (!firstCol) {
          write(" & ");
        } else {
          firstCol = false;
        }
        write("{\\bf ");
        toLaTeX(`data);
        write("}");
        throw new VisitFailure();
      }
      <div id=id>data*</div> -> {
        write("\\label{"+`id+"}\n");
        toLaTeX(`data);
        throw new VisitFailure();
      }
      <div>data*</div> -> {
        toLaTeX(`data);
        throw new VisitFailure();
      }
      <hr/> -> { throw new VisitFailure(); }
      <script/> -> { throw new VisitFailure(); }

      #TEXT(text) -> {
        write(`text);
        throw new VisitFailure();
      }
      #COMMENT(text) -> {
        write("% "+`text);
        throw new VisitFailure();
      }
    } // end visit
  }

  private static String columnSpec(int nbCol) {
    if (nbCol == 0){
      return "";
    } else {
      return "l"+columnSpec(nbCol-1);
    }
  }

  %strategy ComputeWidth() extends `Identity() {
    visit TNode {
      <tr>data*</tr> -> {
        tableMaxCol(max,currentMax,`data);
        max = (max<currentMax ? currentMax : max);
        currentMax = 0;
        throw new VisitFailure();
      }
      <td/> -> { currentMax++; throw new VisitFailure(); }
      <th/> -> { currentMax++; throw new VisitFailure(); }
    }
  }

  private static int max, currentMax;
  private static int tableMaxCol(int vmax, int vcurrentMax, TNodeList subject) {
    max=vmax;
    currentMax=vcurrentMax;
    Strategy rule = `ComputeWidth();
    try {
      System.out.println("subject          = " + subject);
      `TopDownCollect(ComputeWidth()).visitLight(subject);
    } catch (VisitFailure e) {
      System.out.println("reduction failed on: " + subject);
    }
    return max;
  }

}
