/*
 * Copyright (c) 2004-2007, INRIA
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 
 * 	- Redistributions of source code must retain the above copyright
 * 	notice, this list of conditions and the following disclaimer.  
 * 	- Redistributions in binary form must reproduce the above copyright
 * 	notice, this list of conditions and the following disclaimer in the
 * 	documentation and/or other materials provided with the distribution.
 * 	- Neither the name of the INRIA nor the names of its
 * 	contributors may be used to endorse or promote products derived from
 * 	this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package mgs;

import java.util.*;
import aterm.*;
import aterm.pure.*;
import mgs.term.*;
import mgs.term.types.*;

public class BoulderDash {
  private HashMap space;
  private HashSet marked;
  private HashMap newSpace;
  private static int SIZE = 20;

  %include { term/term.tom }

  %op Bead beadRock(n:Bead, s:Bead, e:Bead, w:Bead) {
    is_fsym(t) { (t!=null) && t.isbead() && t.getvalue() == rock && !marked.contains(t) }
    get_slot(n,t) { getNorthBead(t) }
    get_slot(s,t) { getSouthBead(t) }
    get_slot(e,t) { getEastBead(t) }
    get_slot(w,t) { getWestBead(t) }
  }

  %op Bead empty() {
    is_fsym(t) { t==null }
    make { null }
  }


  private static int undefined = 0;
  private static int rock      = 1;
  private static int ground    = 2;

  private Position getNorthPosition(Position p) {
    return `pos(p.getx(),p.gety()+1);
  }

  private Position getSouthPosition(Position p) {
    return `pos(p.getx(),p.gety()-1);
  }

  private Position getEastPosition(Position p) {
    return `pos(p.getx()+1,p.gety());
  }

  private Position getWestPosition(Position p) {
    return `pos(p.getx()-1,p.gety());
  }

  private Bead getNorthBead(Bead b) {
    return (Bead) space.get(getNorthPosition(b.getpos()));
  }

  private Bead getSouthBead(Bead b) {
    return (Bead) space.get(getSouthPosition(b.getpos()));
  }

  private Bead getEastBead(Bead b) {
    return (Bead) space.get(getEastPosition(b.getpos()));
  }

  private Bead getWestBead(Bead b) {
    return (Bead) space.get(getWestPosition(b.getpos()));
  }

  public final static void main(String[] args) {
    BoulderDash test = new BoulderDash();
    test.run();
  }

  public String toString() {
    String s = "";
    Iterator it = space.values().iterator();
    while(it.hasNext()) {
      Bead b = (Bead) it.next();
      s += b + "\n";
    }
    return s;
  }

  public String toMatrix(HashMap space) {
    int xmax=0;
    int ymax=0;

    Iterator it = space.values().iterator();
    while(it.hasNext()) {
      Bead b = (Bead) it.next();
      %match(Bead b) {
        bead(pos(x,y),_) -> {
          if(`x>xmax) xmax = `x;
          if(`y>ymax) ymax = `y;
        }
      }
    }

    int[][] array = new int[xmax+1][ymax+1];
    it = space.values().iterator();
    while(it.hasNext()) {
      Bead b = (Bead) it.next();
      %match(Bead b) {
        bead(pos(x,y),value) -> {
          array[`x][`y] = `value;
        }
      }
    }

    String s = "";
    for(int y=ymax+1 ; y<SIZE ; y++) {
      s += "\n";
    }
    for(int y=ymax ; y>=0 ; y--) {
      String line = "";
      for(int x=0 ; x<=xmax ; x++) {
        if(array[x][y] == undefined) {
          line += " ";
        } else
          if(array[x][y] == ground) {
            line += "X";
          } else if(array[x][y] == rock) {
            line += "O";
          } 
      }
      s += line + "\n";
    }
    return s;
  }


  public void run() {
    space = new HashMap();
    marked = new HashSet();
    /*
       setGround(space,20);
       putBead(space,10,5,rock);
       putBead(space,10,6,rock);
       putBead(space,10,7,rock);
       putBead(space,10,8,rock);
       putBead(space,10,9,rock);
     */
    setRock(space);
    boolean fire = true;
    while(fire) {
      System.out.println(toMatrix(space));
      fire = oneStep();
    }

    //System.out.println("\nstable space = \n" + this);
  }

  public boolean oneStep() {
    boolean fire = false;
    newSpace = new HashMap();
    Iterator it = space.values().iterator();
    while(it.hasNext()) {
      Bead b = (Bead) it.next();
      boolean f = gravity(newSpace,b);
      fire = fire || f ;
    }
    space=newSpace;
    marked.clear();
    return fire;
  }

  // return true if fire a rule
  public boolean gravity(HashMap newSpace, Bead b) {
    %match(Bead b) {
      beadRock[s=empty()] -> {
        Bead newBead = `bead(getSouthPosition(b.getpos()),b.getvalue());
        marked.add(b);
        putBead(newSpace,newBead);
        return true;
      }

      beadRock[s=s@beadRock[e=empty()],e=empty()] -> {
        Bead newBead = `bead(getEastPosition(getSouthPosition(b.getpos())),b.getvalue());
        putBead(newSpace,newBead);
        putBead(newSpace,`s);
        marked.add(b);
        marked.add(`s);
        return true;
      }

      beadRock[s=s@beadRock[w=empty()],w=empty()] -> {
        Bead newBead = `bead(getWestPosition(getSouthPosition(b.getpos())),b.getvalue());
        putBead(newSpace,newBead);
        putBead(newSpace,`s);
        marked.add(b);
        marked.add(`s);
        return true;
      }


      bead[pos=p] -> {
        newSpace.put(`p,b);
        return false;
      }
    }
    return false;
  }

  private void putBead(HashMap space, int x, int y, int beadType) {
    Position p = `pos(x,y);
    space.put(p, `bead(p,beadType));
  }

  private void putBead(HashMap space, Bead b) {
    space.put(b.getpos(),b);
  }

  private void removeBead(HashMap space, Bead b) {
    space.remove(b.getpos());
  }


  public void setGround(HashMap space, int size) {
    for(int i=0 ; i<size ; i++) {
      putBead(space,i,0,ground);
    }
  }

  public void setRock(HashMap space) {
    for(int i=0 ; i<SIZE ; i++) {
      putBead(space,i,0,ground);
      putBead(space,i,SIZE-1,ground);
      putBead(space,0,i,ground);
      putBead(space,SIZE-1,i,ground);
    }
    for(int i=0 ; i<SIZE/2-1 ; i++) {
      putBead(space,i,SIZE/2,ground);
    }
    for(int i=SIZE/2+1 ; i<SIZE ; i++) {
      putBead(space,i,SIZE/2,ground);
    }

    for(int j=SIZE/2+1 ; j<SIZE-1 ; j++) {
      for(int i=1 ; i<SIZE-1 ; i++) {
        putBead(space,i,j,rock);
      }
    }

  }


}

