/*
 *  This file is part of timespan.
 *
 *  Timespan is the legal property of its authors.  Refer to the file
 *  AUTHORS in the top-level directory of the source distribution for
 *  details.
 *
 *  Timespan is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Timespan is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with timespan; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

//prevent multiple inclusions of this header from stomping over each other
#ifndef TIMESPAN_HPP
#define TIMESPAN_HPP

/* STANDARD LIBRARY HEADERS */
#include <iostream> //cout, endl
#include <iomanip>  //output formatting
#include <cstring>  //strcmp(), strcpy(), strlen(), etc.
#include <cstdlib>  //atoi(), exit(), etc.
#include <ctime>	 //struct timespec, time(), localtime(), etc.

using namespace std; //get access to anything within std
/* END STANDARD LIBRARY HEADERS */

//if config.hpp exists, the Makefile has g++ define HAVE_CONFIG_H at compile
//time.  if HAVE_CONFIG_H is defined, include it.
#ifdef HAVE_CONFIG_H
#  include "config.hpp"
#endif

//work out a useable 64-bit integer type--we need 64 bits to get around the
//limits imposed by a conventional 16- or 32-bit int type
#ifdef HAVE_STDINT_H
#  include <stdint.h>
#endif
#ifdef HAVE_INTTYPES_H
#  include <inttypes.h>
#endif
#ifdef HAVE_LIBGEN_H
#  include <libgen.h>
#endif

#if !defined HAVE_UINT_LEAST64_T && defined UINT_LEAST64_T
	typedef UINT_LEAST64_T uint_least64_t;
#endif

#ifndef HAVE_IOSTREAM_64BIT_IO
	ostream &operator << (ostream &os, uint_least64_t v);
	istream &operator >> (istream &os, uint_least64_t &v);
#endif

#ifndef HAVE_BASENAME
	char *basename (char *s);
#endif

/* STRUCTURE DEFINITIONS */
struct Date
{ //this struct will hold the values of the dates entered at the command line
	unsigned int imonth, //integer representation for the month
					 day,
					 year;
}; //declare variables of this type using 'Date varname;'

struct Flag
{ //this struct will hold all the flags necessary for program functionality
	bool wantdays,	//want number of days as an answer
	wantdate,		//want a date as amswer
	wantlongdate,	//use words for months?
	wanthours,		//want number of hours as an answer
	wantminutes,	//want number of minutes as an answer
	wantseconds,	//want number of seconds as an answer
	extraline,		//want extra line at end of output?
	usin,			//dates on command line in US form?
	usout;			//dates output in US form?
}; //declare variables of this type using 'Flag varname;'
/* END STRUCTURE DEFINITIONS */

/* CONSTANTS */
extern const char *progname; //basename of command used to invoke timespan
extern const char * const months[]; //full, properly capitalized month names
extern const char * const lmonths[]; //lowercase month names (used in strcmp())
extern const char * const lsmonths[]; //lowercase short names (used in strcmp())
extern const unsigned int monthlength[]; //month legnths in normal years
extern const unsigned int monthlengthleap[]; //month lengths in leap years
extern const unsigned int hours_in_day;
extern const unsigned int minutes_in_hour;
extern const unsigned int minutes_in_day;
extern const unsigned int seconds_in_minute;
extern const unsigned int seconds_in_hour;
extern const unsigned int seconds_in_day;
/* END CONSTANTS */

/* FUNCTION PROTOTYPES */
bool isleap(unsigned int);
Date calcdate(uint_least64_t);
Date get_todays_date();
Date get_date(char *[], const int, const Flag);
uint_least64_t calc_days_between(Date, Date);
uint_least64_t calcdays(Date, bool = false);
uint_least64_t calc_hours_between(Date, Date);
uint_least64_t calc_minutes_between(Date, Date);
uint_least64_t calc_seconds_between(Date, Date);
uint_least64_t calchours(Date);
uint_least64_t calcminutes(Date);
uint_least64_t calcseconds(Date);
void print_date(Date, Flag);
void usage();
void version();
/* END FUNCTION PROTOTYPES */

#endif

