#!/usr/bin/env python
#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
# Copyright 2009 Rob Oakes	<LyX-Devel@oak-tree>

from timedrive.backupsettings.settings import Settings
import duplicity_interface
from __init__ import *

def CombineBackups(stringTime, folder):
	duplicity_interface.log.setup()
	
	# Create a Date Range to Test duplicity_interface filtering functions
	backupTime1 = time_String2Int(str(stringTime)) - 100
	backupTime2 = backupTime1 + 100
	settings = Settings()
	
	# Create a blank index so all entries in the manifest are retrieved
	index = ()
	
	# Retrieve options from the current settings
	if settings.AmazonS3Backup == True:
		S3_AccessId = settings.AmazonS3_AccessId
		S3_SecretKey = settings.AmazonS3_SecretKey
	else:
		S3_AccessId = None
		S3_SecretKey = None
	
	if settings.AmazonS3_Location == "European":
		S3_BucketType = settings.AmazonS3_Location
	else:
		S3_BucketType = None
	
	# Retrieve archive url from settings
	au = ArchiveUrl_FromSettings(folder, settings)
	
	# If the host is availabe, run the test procedure
	if au.available:
		
		# Connection and collection status objects
		col_stats, filelist_backend, filelist_archive = duplicity_interface.get_collection_status(str(au.archiveUrl),
			str(au.gnuPassphrase), True, None, S3_AccessId, S3_SecretKey, S3_BucketType)
		
		# Retrive backup sets within the specified date range
		backup_setlist, backup_chain = duplicity_interface.get_backup_sets(col_stats, 
			backupTime1, backupTime2)
		num_vols = 0
		for backupSet in backup_setlist:
			num_vols += len(backupSet)
		cur_vol = [0]
		
		# Combine Snapshots : Locate and Download .tar files -> Extract diff files -> Compare signatures to available diff versions -> discard previous diffs that aren't needed to maintain chain continuity -> re-compress/re-encrypt to new .tar archives -> Upload to storage -> Modify manifest and other files to reflect the changes
		
		def get_fileobj_iter(backup_set):
			"""Get the file object iterator from the backup_set"""
			manifest = backup_set.get_manifest()
			volumes = manifest.get_containing_volumes(index)
			for vol_num in volumes:
				yield restore_get_enc_fileobj(backup_set.backend, 
					backup_set.volume_name_dict[vol_num],
					manifest.volume_info_dict[vol_num])
				
				cur_vol[0] += 1
				print "Processed volume " + str(cur_vol[0]) + " of " + str(num_vols)
		
		fileobj_iters = map(get_fileobj_iter, backup_setlist)
		tarfiles = map(duplicity_interface.patchdir.TarFile_FromFileobjs, fileobj_iters)
		
		# Retrieve ROPaths : A duplicity ro_path contains information about
		# files contained in an archive and the associated meta data (hashes, etc.)
		rop_iter = duplicity_interface.patchdir.tarfiles2rop_iter(tarfiles, index)
		
		for obj in rop_iter:
			print obj.__class__, ':', obj.get_ropath()
			
		
		dest_dir = duplicity_interface.path.Path("/home/roakes/Downloads")
		duplicity_interface.patchdir.Write_ROPaths(dest_dir, rop_iter)
		
		# Clean Up duplicity_interface and shutdown the log
		duplicity_interface.cleanup_globals()
		if S3_AccessId != None and S3_SecretKey != None:
			duplicity_interface.unset_AmazonS3()
		filelist_backend.close()
		duplicity_interface.log.shutdown()


def restore_get_enc_fileobj(backend, filename, volume_info):
	"""
	Returns a plaintext fileobj from encrypted filename on backend
	
	If volume_info is set, the hash of the file will be checked,
	assuming the hash is available.  If globals.sign_key is set, 
	a fatal error will be raised if the file is not signed by the key.
	"""
	
	parseresults = duplicity_interface.file_naming.parse(filename)
	tdp = duplicity_interface.dup_temp.new_tempduppath(parseresults)
	backend.get(filename, tdp)
	restore_check_hash(volume_info, tdp)
	
	fileobj = tdp.filtered_open_with_delete("rb")
	if parseresults.encrypted and duplicity_interface.globals.gpg_profile.sign_key:
		restore_add_sig_check(fileobjc)
	return fileobj


def restore_add_sig_check(fileobj):
	"""
	Requires signature when closing fileobj matches sig in gpg_profile
	
	@rtype: void
	@return: void
	"""
	assert (isinstance(fileobj, duplicity_interface.dup_temp.FileobjHooked) and
		isinstance(fileobj.fileobj, duplicity_interface.gpg.GPGFile)), fileobj
	def check_signature():
		"""Thunk run when closing volume file"""
		actual_sig = fileobj.fileobj.get_signature()
		if actual_sig != duplicity_interface.globals.gpg_profile.sign_key:
			print "Not signed by proper key."
	fileobj.addhook(check_signature)


def restore_check_hash(volume_info, vol_path):
	"""
	Check the hash of vol_path path against data in volume_info
	
	@rtype: void
	@return: void
	"""
	hash_pair = volume_info.get_best_hash()
	if hash_pair:
		calculated_hash = duplicity_interface.gpg.get_hash(hash_pair[0], vol_path)
		if calculated_hash != hash_pair[1]:
			print "Invalid data - " + hash_pair[0]
			print "Calculated hash: " + calculated_hash
			print "Manifest hash: " + hash_pair[1] 