#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
import os
from timedrive import utils

def ArchiveUrl_FromSettings(include, settings):
	"""
	Creates a valid archive Url from the backupsettings.IncludeList
	
	@type settings: Settings Object
	@param settings:
	"""
	au = ArchiveUrl()
	au.available = True
	folderName = utils.determine_folderName(include)
	if include.LocationType == include.LOCATION_LOCAL:
		folderName = os.path.basename(os.path.normpath(str(folderName)))

	# Create archiveUrl and gnuPassphrase from General Settings
	if (include.BackupType == include.GENERALSETTINGS):

		# Local Backup Folder
		if settings.LocalBackup:
			dir = os.path.normpath(str(settings.LocalBackupDirectory))
			au.archiveUrl = buildLocalUrl(dir, folderName)
			au.available = os.path.isdir(dir)
			
		# Remote Folder (ssh, ftp, webdav)
		elif settings.RemoteBackup:
			au.archiveUrl = buildRemoteUrl(settings.RemoteProtocol, settings.RemoteUsername, settings.RemotePassword, settings.RemoteHost, settings.RemotePort, settings.RemotePath, folderName)
			au.available = utils.server_available(settings.RemoteHost, settings.RemotePort)
			
		# Amazon S3
		elif settings.AmazonS3Backup:
			au.archiveUrl = buildAmazonS3Url(os.path.normpath(str(settings.AmazonS3_Bucket)), folderName)
	
	# Create archiveUrl and GnuPassphras from Advanced Folder Settings
	elif (include.BackupType == include.LOCAL)|(include.BackupType == include.REMOTE):
				
		if include.BackupType == include.LOCAL:
			au.archiveUrl = buildLocalUrl(include.LocalBackupDirectory)
			au.available = os.path.isdir(include.LocalBackupDirectory)

		elif include.BackupType == include.REMOTE:
			au.archiveUrl = buildRemoteUrl(include.RemoteProtocol, include.RemoteUsername, include.RemotePassword, include.RemoteHost, include.RemotePort, include.RemotePath)
			au.available = utils.server_available(include.RemoteHost, include.RemotePort)
	
	# Retrieve the gnuPassphrase
	if include.EncryptionKeyEnabled == False:
		au.gnuPassphrase = settings.Options_EncryptionKey
	else:
		au.gnuPassphrase = include.EncryptionKey

	return au

def buildRemoteUrl(protocol, username, password, host, port, path, includeName = None):
	"""
	Build and returns a remote url
	
	@type protocol: String
	@param protocol:
	@type username: String
	@param username:
	@type password: String
	@param password:
	@type host: String
	@param host:
	@type port: String
	@param port:
	@type path: String
	@param path:
	@type includeName: String
	@param includeName: default=None:
	
	@rtype: String
	@return:
	"""
	url = protocol + '://' + username + ':' + password + '@' + str(host) + ':' + str(port) + '/' + path
	if includeName is not None:
		url += '/' + includeName
	return str(url).replace(' ', '-')
	
def buildLocalUrl(backupDirectory, includeName = None):
	"""
	Build and returns local url
	
	@type backupDirectory: String
	@param backupDirectory:
	@type includeName: String
	@param includeName: default = None:
	
	@rtype: String
	@return:
	"""
	url = 'file://' + backupDirectory
	if includeName is not None:
		url += '/' + includeName
	return url
	
def buildAmazonS3Url(bucket, includeName = None):
	"""
	Build and returns amazonS3 url
	
	@type bucket: String
	@param bucket:
	@type includeName: String
	@param includeName: default = None:
	
	@rtype: String
	@return:
	"""
	url = "s3+http://" + bucket
	if includeName is not None:
		url += '/' + includeName
	return str(url).replace(' ', '-')

class ArchiveUrl(object):
	""" ArchiveUrl class: definition of an archive url"""
	
	def __init__(self):
		"""
		Initialization function for ArchiveUrl
		"""
		self._archiveUrl = None
		self._gnuPassphrase = None
		self._hostAvailable = False

	def _get_archiveUrl(self):
		"""
		archive url
		
		@rtype: String
		"""
		return self._archiveUrl

	def _set_archiveUrl(self, value):
		"""
		archive url
		
		@type value: String
		@param value:
		"""
		self._archiveUrl = value

	def _get_gnuPassphrase(self):
		"""
		gnu passphrase (the encryption key to the archive)
		
		@rtype: String
		"""
		return self._gnuPassphrase

	def _set_gnuPassphrase(self, value):
		"""
		gnu passphrase (the encryption key to the archive)
		
		@type value: String
		@param value:
		"""
		self._gnuPassphrase = value

	def _get_available(self):
		"""
		the available status of the arichive url 
		
		@rtype: Boolean
		"""
		return self._hostAvailable

	def _set_available(self, value):
		"""
		the available status of the arichive url 
		
		@type value: Boolean
		@param value:
		"""
		self._hostAvailable = value

	archiveUrl = property(_get_archiveUrl, _set_archiveUrl)
	gnuPassphrase = property(_get_gnuPassphrase, _set_gnuPassphrase)
	available = property(_get_available, _set_available)
