#!/usr/bin/env python
#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
# Copyright 2009 Rob Oakes	<LyX-Devel@oak-tree>

import os
import time
import re
import sys
import socket
import log
from timedrive.backupsettings import includeitem
from timedrive.backupsettings import globals
import subprocess

MANUAL = 0
DAY = 1
WEEK = 2
MONTH = 3

from duplicity import dup_time

def handle_error(err_message):
	"""
	handles the error produced by duplicity and tries to make a proper response
	
	@type err_message: String
	@param err_message: the error message from duplicity
	
	@rtype: int
	@return: returns -1 if the error can't be process. return 0 if backendException
		return 1 to 99 if fatal error occured
	"""
	if "Fatal Error" in err_message:
		print "found Fatal Error"
		if " --allow-source-mismatch" in err_message:
			return 1
		else:
			return 99
	elif "BackendException" in err_message:
		print "found BackendException"
		return 0
	else:
		return -1

def find_file(file_name):
	"""
	Using the file_name, search in the common places. Return the 
	path for the file or None if the file couldn't be found.
	
	@type file_name: String
	@param file_name: the filename to search
	
	@rtype: String
	@return: if found the full path with filename, if not found returns None
	"""
	# Ordered by priority, keep global paths before local paths
	
	current_dir = os.path.abspath(os.path.dirname(__file__))
	common_paths = [os.path.join(sys.prefix, 'local', 'share', 'pixmaps'),
			os.path.join(sys.prefix, 'share', 'time-drive'),
		os.path.join(sys.prefix, "share", "pixmaps"),
		"/usr/local/bin", "/usr/bin", 
		os.path.abspath(os.path.join(current_dir, "../..")),
		os.path.abspath(os.path.join(current_dir, "../../ui")),
		os.path.abspath(os.path.join(current_dir, "../../ui", "Resources"))]
	
	for path in common_paths:
		filename = os.path.join(path, file_name)
		if os.access(filename, os.F_OK):
			return filename

	return None

def time_String2Int(time_string):
	"""
	Convenience funciton for converting raw time strings to integer format.
	
	@type time_string: String
	@param time_string: time as string
	
	@rtype: Int
	@return: return the time
	"""
	time_int = dup_time.stringtotime(time_string)
	return time_int


def time_GetCurrentTime():
	"""Convenience function which can retrieve the current time"""
	intNow = time.time()
	strNow = time_Int2String(intNow)
	return strNow


def time_Int2String(time_int):
	"""Convenience function for converting time in integer format to raw string."""
	time_string = dup_time.timetostring(time_int)
	return time_string


def time_String2Pretty(time_string):
	"""Convenience funciton for converting raw time strings to a more readable format."""
	time_pretty = dup_time.stringtopretty(time_string)
	return time_pretty


def time_Int2Pretty(time_int):
	"""Convenience function for converting time integers to a more readable format."""
	time_pretty = dup_time.timetopretty(time_int)
	return time_pretty


def verify_ExcludePattern(exclude_pattern):
	"""
	Verifies the specified exclude_pattern or path to ensure that it doesn't contain
	illegal characters.
	
	@rtype: String
	@return: returns the pattern, if an error occured return None
	"""
	pattern = str(exclude_pattern[0].toUtf8()).strip()
	
	if len(pattern) == 0:
		raise Exception(_("Exclude pattern contains no characters"))
		return None
	
	if pattern.find(':') >=0:
		raise Exception(_("Exclude pattern can\'t contain \':\' char !"))
		return None
		
	return pattern

def setup_cron(backup_frequency):
	"""Method that creates a listing in the users crontab file so that backups can run 
	unattended.  If using Mac OS X, crontab must be enabled and the user must have a valid
	cron file before running."""
	
	programName = find_file('timedrive-backup')
	os.system("crontab -l | grep -v timedrive-backup | crontab - ")
	
	cron_line = ''
	
	if backup_frequency == MANUAL:	# Manual Backups
		cron_line = ''
	elif backup_frequency == DAY:
		cron_line = 'echo "@daily {cmd}"'
	elif backup_frequency == WEEK:
		cron_line = 'echo "@weekly {cmd}"'
	elif backup_frequency == MONTH:
		cron_line = 'echo "@monthly {cmd}"'
	
	if len(cron_line) > 0:
		cmd = programName + ' > /dev/null 2>&1'
		cmd = 'nice -n 19 ' + cmd
		cron_line = cron_line.replace('{cmd}', cmd)
		os.system("(crontab -l; %s) | crontab -" % cron_line)

def server_available(host, port):
	"""
	check if a server on a given port is online
	
	@type host: String
	@param host: a valid hostname or ip addres
	@type port: Int
	@param port: a valid port number
	
	@rtype: Boolean
	@return: if the server on the port is available returns true else false
	"""
	serverSocket = socket.socket()
	serverSocket.settimeout(0.25)
	try:
		serverSocket.connect((str(host), int(port)))
		return True
	except socket.error:
		return False
	
def findFolder(IncludeList, includeName, includeType = None):
	"""
	Locates a folder in the includes list
	
	@type IncludeList: List() of type IncludeItem Object
	@param IncludeList:
	@type includeName: String
	@param includeName: the name of the folder or location
			(field FolderName in IncludeItem Object)
	@type includeType: int
	@param includeType: backuptype defined in the IncludeItem Object
	
	@rtype: IncludeItem Object
	@return: if the given name is found it returns a includeItem else it returns None
	"""
	for f in IncludeList:
		folderName = determine_folderName(f)
		if includeName == folderName and (includeType is None or includeType == f.BackupType):
			return f
	return None

def getPortProtocol(proto):
	"""
	Convert protocol name to port
	
	@type proto: String
	@param proto: is the name of the protocol
	
	@rtype: int
	@return: returns the port of a protocol
	"""
	if proto == "ssh":
		return 22
	elif proto == "ftp":
		return 21
	elif proto == "webdav":
		return 80
	elif proto == "webdavs":
		return 443
	elif proto == "rsync":
		return 873
	else:
		log.Error("undefined protocol: " + str(proto))
		return 00

def backupTypeToText(includeItem):
	"""
	convert folder backup type to text
	Note: when text change method textToBackupType
	
	@type includeItem: IncludeItem Object
	@param includeItem:
	"""
	if includeItem.BackupType == includeItem.GENERALSETTINGS:
		return _("General Settings")
	elif includeItem.BackupType == includeItem.LOCAL:
		return _("Folder: Local")
	elif includeItem.BackupType == includeItem.REMOTE:
		return _("Folder: Remote")
		

def textToBackupType(text):
	"""
	convert text to folder backup type
	Note: when text change method backupTypeToText
	
	@type text: String
	@param text:
	"""
	if text == _("General Settings"):
		return includeitem.IncludeItem.GENERALSETTINGS
	elif text == _("Folder: Local"):
		return includeitem.IncludeItem.LOCAL
	elif text == _("Folder: Remote"):
		return includeitem.IncludeItem.REMOTE
	
def locationTypeToText(includeItem):
	"""
	convert folder location type to text
	Note: when text change method textToLocationType
	
	@type includeItem: IncludeItem Object
	@param includeItem:
	"""
	if includeItem.LocationType == includeItem.LOCATION_LOCAL:
		return _("Local")
	elif includeItem.LocationType == includeItem.LOCATION_REMOTE:
		return _("Remote")
		

def textToLocationType(text):
	"""
	convert text to folder location type
	Note: when text change method locationTypeToText
	
	@type text: String
	@param text:
	"""
	if text == _("Local"):
		return includeitem.IncludeItem.LOCATION_LOCAL
	elif text == _("Remote"):
		return includeitem.IncludeItem.LOCATION_REMOTE
	
def determine_folderName(include):
	"""
	Get the folder name
	
	@type include: IncludeItem Object
	@param include: can be of type IncludeLocalItem or IncludeRemoteItem
	
	@rtype: String
	"""
	if include.LocationType == include.LOCATION_LOCAL:
		return str(include.FolderName)
	elif include.LocationType == include.LOCATION_REMOTE:
		return str(include.RemoteName)

def mount_remote(include):
	"""
	Get the folder name
	
	@type include: IncludeRemoteItem Object
	@param include:
	
	@rtype: String
	@return: the string where it's mounted, if none mount is in error state
	"""
	mountpoint = str(globals.cache_dir+"/"+include.RemoteName)
	if not os.path.exists(mountpoint):
		try:
			os.makedirs(mountpoint)
		except:
			pass

	if include.LocationRemoteProtocol == "ssh":
		program = which("sshfs")
		if program is None:
			return program
		cmd = 'echo %s | sshfs -p %s "%s@%s:%s" "%s" -o workaround=rename -o password_stdin' % (include.LocationRemotePassword,
													include.LocationRemotePort,
													include.LocationRemoteUsername,
													include.LocationRemoteHost,
													include.LocationRemotePath,
													mountpoint)
		pipe = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
		output_out, output_err = pipe.communicate()
		if output_out:
			print "Info Output %s" % output_out
			log.Info("Info Output %s" % output_out)
		if output_err:
			print "Error Output %s" %output_err
			log.Error("Error Output %s" % output_err)
			return None
		return mountpoint
		
	else:
		print "only ssh is supported at the moment"
		
def umount_remote(include):
	"""
	umount the remote folder, no remove of folder
	
	@type include: IncludeRemoteItem Object
	@param include:
	
	@rtype: Boolean
	@return: state of the umount response, false is returned if 
		fusermount isn't found
	"""

	mountpoint = globals.cache_dir+"/"+include.RemoteName
	program = which('fusermount')
	if program is None:
		return False
		
	os.system ('%s -u "%s"' % (program, mountpoint))
	return True

def which(program):
	"""
	This mimics the behavior of the UNIX 'which' command.

	@type program: String
	@param program: can be in the form '/bin/ls' or 'ls'
	
	@rtype: String
	@return: not found returns None, else returns the location with path
	"""
	def is_exe(fpath):
		return os.path.exists(fpath) and os.access(fpath, os.X_OK)

	fpath, fname = os.path.split(program)
	if fpath:
		if is_exe(program):
			return program
	else:
		for path in os.environ["PATH"].split(os.pathsep):
			exe_file = os.path.join(path, program)
			if is_exe(exe_file):
				return exe_file

	return None

