#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup
# Requires Duplicity Version 0.6.02 released July 08, 2009
#
# Copyright 2009 Rob Oakes	<LyX-Devel@oak-tree>

from PyQt4 import QtCore
from PyQt4 import QtGui

from timedrive.advancedsettingsdialog import AdvancedSettingsDialog
from timedrive.remoteconnectionsettingsdialog import RemoteConnectionSettingsDialog
from timedrive.backupsettings.settings import Settings
from timedrive.backupsettings import includelocalitem
from timedrive.backupsettings import includeremoteitem
from timedrive.backupsettings import excludeitem
import timedrive.backupsettings as backupsettings
import timedrive.utils as utils
from timedrive.utils import gui_utils
import timedrive.settingsdialog as settingsdialog
import os

class GeneralSettingsDialog(settingsdialog.SettingsDialog):
	"""Settings Dialog Class: Includes the methods for changing the global
	backup settings."""
	
	def __init__(self, parent, backupsettings):
		"""
		Initialization function for GeneralSettingsDialog
		
		@type parent: SettingsDialog Object
		@param parent:
		@type backupsettings: Settings Object
		@param backupsettings:
		"""
		super(GeneralSettingsDialog, self).__init__(parent)
		gui_utils.loadUi(self, "uiSettings.ui")
		self.ui = self
		
		self.ui.lblTstResult.setVisible(False)
		self.ui.treeInclude.setColumnWidth(0, 125)
		self.ui.treeInclude.setColumnWidth(1, 225)
		
		self.LoadBackupSettings(backupsettings)
		
		self.ui.AdvancedSettingsDialog = AdvancedSettingsDialog(self)
		self.ui.RemoteConnectionSettingsDialog = RemoteConnectionSettingsDialog(self)
		
		# Create Menus for Add Folders and Advanced Options
		self.includeMenu = QtGui.QMenu(self)
		self.includeMenu.addAction(self.ui.actionAddLocalFolder)
		self.includeMenu.addAction(self.ui.actionAddRemoteFolder)
		self.ui.btnAddFolders.setMenu(self.includeMenu)
		
		self.advancedMenu = QtGui.QMenu(self)
		self.advancedMenu.addAction(self.ui.actionAdvancedFolderOptions)
		self.advancedMenu.addAction(self.ui.actionManageRemoteFolder)
		self.ui.buttonFolderOptions.setMenu(self.advancedMenu)
		
		# Events and Methods
		QtCore.QObject.connect(self.ui.btnOk, QtCore.SIGNAL("clicked()"), 
			self.btnOk_Pressed)
		QtCore.QObject.connect(self.ui.btnCancel, QtCore.SIGNAL("clicked()"), 
			self.btnCancel_Pressed)
		QtCore.QObject.connect(self.ui.actionAddLocalFolder, QtCore.SIGNAL("triggered()"), 
			self.btnIncludeAddLocal_Pressed)
		QtCore.QObject.connect(self.ui.actionAddRemoteFolder, QtCore.SIGNAL("triggered()"), 
			self.btnIncludeAddRemote_Pressed)
		QtCore.QObject.connect(self.ui.btnIncludeRemove, QtCore.SIGNAL("clicked()"),
			self.btnIncludeRemove_Pressed)
		QtCore.QObject.connect(self.ui.buttonSelectLocalFolder, QtCore.SIGNAL("pressed()"),
			self.buttonSelectLocalFolder_Pressed)
		QtCore.QObject.connect(self.ui.buttonTestConnection, QtCore.SIGNAL("pressed()"),
			self.buttonTestConnection_Pressed)
		QtCore.QObject.connect(self.ui.btnExcludeAdd, QtCore.SIGNAL("clicked()"),
			self.btnExcludeAdd_Pressed)
		QtCore.QObject.connect(self.ui.btnExcludeRemove, QtCore.SIGNAL("clicked()"),
			self.btnExcludeRemove_Pressed)
		QtCore.QObject.connect(self.ui.actionManageRemoteFolder, QtCore.SIGNAL("triggered()"),
			self.btnIncludeManageRemote_Pressed)
		QtCore.QObject.connect(self.ui.actionAdvancedFolderOptions, QtCore.SIGNAL("triggered()"),
			self.buttonFolderOptions_Pressed)
		QtCore.QObject.connect(self.ui.buttonSelectDefaultRestoreLocation, QtCore.SIGNAL("pressed()"),
			self.buttonSelectDefaultRestoreLocation_Pressed)
		QtCore.QObject.connect(self.ui.comboRemoteProtocol, 
			QtCore.SIGNAL("activated(const QString &)"),
			self.comboRemoteProtocol_currentIndexChanged)
		QtCore.QObject.connect(self.ui.tabSettings, QtCore.SIGNAL("currentChanged ( int )"), self.tabSettings_currentChanged)
		
		# Signals and Slots Related to Threads
		
		# Events and Methods - General Tab
		QtCore.QObject.connect(self.ui.radioLocalBackup, QtCore.SIGNAL("released()"),
			self.GeneralSettings_Update)
		QtCore.QObject.connect(self.ui.radioRemoteBackup, QtCore.SIGNAL("released()"),
			self.GeneralSettings_Update)
		QtCore.QObject.connect(self.ui.radioAmazonS3Backup, QtCore.SIGNAL("released()"),
			self.GeneralSettings_Update)
		
		# Events and Methods - Auto Remove Tab
		QtCore.QObject.connect(self.ui.checkOlderThan, QtCore.SIGNAL("released()"), 
			self.AutoRemoveSettings_Update)
		QtCore.QObject.connect(self.ui.checkFreeSpace, QtCore.SIGNAL("released()"),
			self.AutoRemoveSettings_Update)
		
		# Events and Methods - Options Tab
		QtCore.QObject.connect(self.ui.checkEncryptFiles, QtCore.SIGNAL("released()"),
			self.Options_Update)
		QtCore.QObject.connect(self.ui.checkFullBackupFrequency, QtCore.SIGNAL("released()"),
			self.Options_Update)

		# Events and Methods - Advanced Tab
		QtCore.QObject.connect(self.ui.checkDefaultRestoreLocation, QtCore.SIGNAL("released()"),
			self.Advanced_Update)
	
	
	def LoadBackupSettings(self, backupsettings):
		"""
		Loads Settings into the fields
		
		@type backupsettings: Settings Object
		@param backupsettings:
		"""
		self.backupsettings = backupsettings
		
		# Load General Options
		self.ui.radioLocalBackup.setChecked(backupsettings.LocalBackup)
		self.ui.txtLocalFolder.setText(backupsettings.LocalBackupDirectory)
		self.ui.radioRemoteBackup.setChecked(backupsettings.RemoteBackup)
		self.ui.comboRemoteProtocol.setCurrentIndex(backupsettings.RemoteProtocolInt)
		self.ui.txtRemotePath.setText(backupsettings.RemotePath)
		self.ui.txtRemoteHost.setText(backupsettings.RemoteHost)
		self.ui.txtRemotePort.setText(str(backupsettings.RemotePort))
		self.ui.txtUsername.setText(backupsettings.RemoteUsername)
		self.ui.txtPassword.setText(backupsettings.RemotePassword)
		self.ui.radioAmazonS3Backup.setChecked(backupsettings.AmazonS3Backup)
		self.ui.txtAmazonS3_Bucket.setText(backupsettings.AmazonS3_Bucket)
		self.ui.txtAmazonS3_AccessId.setText(backupsettings.AmazonS3_AccessId)
		self.ui.txtAmazonS3_SecretKey.setText(backupsettings.AmazonS3_SecretKey)
		self.ui.comboAmazonS3_BucketType.setCurrentIndex(backupsettings.AmazonS3_LocationInt)
		self.GeneralSettings_Update()
		
		# Load Auto-remove Options
		self.ui.checkOlderThan.setChecked(backupsettings.AutoRemove_OlderThan)
		self.ui.spinOlderThan.setValue(backupsettings.AutoRemove_OlderThan_Duration)
		self.ui.comboOlderThan.setCurrentIndex(backupsettings.AutoRemove_OlderThan_Units)
		self.ui.checkFreeSpace.setChecked(backupsettings.AutoRemove_FreeSpace)
		self.ui.spinFreeSpace.setValue(backupsettings.AutoRemove_FreeSpace_Size)
		self.ui.comboFreeSpace.setCurrentIndex(backupsettings.AutoRemove_FreeSpace_Units)
		self.ui.checkSmartRemove.setChecked(backupsettings.AutoRemove_SmartRemove)
		self.ui.checkPreserveNamedSnapshots.setChecked(backupsettings.AutoRemove_NamedSnapshots)
		self.AutoRemoveSettings_Update()
		
		# Load Backup Folder List (Include)
		self.loadIncludeList()
		
		# Load Exclude List
		excludeItemList = []
		trashIcon = QtGui.QIcon(QtGui.QPixmap(":/PNG/Resources/Remove-Snapshot.png"))
		self.ui.treeExclude.clear()
		for item in backupsettings.ExcludeList:
			excludeItem = QtGui.QTreeWidgetItem([QtCore.QString(item.Item)])
			excludeItem.setIcon(0, trashIcon)
			excludeItemList.append(excludeItem)
		self.ui.treeExclude.insertTopLevelItems(0, excludeItemList)
		
		# Load "Options"
		self.ui.comboSchedule.setCurrentIndex(backupsettings.Schedule)
		self.ui.checkEncryptFiles.setChecked(backupsettings.Options_EncryptFilesEnabled)
		self.ui.txtEncryptionKey.setText(backupsettings.Options_EncryptionKey)
		self.ui.checkEnableNotifications.setChecked(backupsettings.Options_EnableNotifications)
		self.ui.checkExcludeHidden.setChecked(backupsettings.Options_ExcludeHidden)
		self.ui.checkFullBackupFrequency.setChecked(backupsettings.Options_FullBackupFrequencyEnabled)
		self.ui.spinFullBackupFrequency.setValue(backupsettings.Options_FullBackupFrequency)
		self.Options_Update()

		# Load "Advanced"
		self.ui.checkDefaultRestoreLocation.setChecked(backupsettings.Advanced_DefaultRestoreEnabled)
		self.ui.txtDefaultRestoreLocation.setText(backupsettings.Advanced_DefaultRestoreLocation)
		self.Advanced_Update()
	
	def GeneralSettings_Update(self):
		"""Enable or disable the general settings controls based on the 
		options selected.  If Local Backup Directory is selected then remote 
		options should be disabled.  Likewise if the remote or options are selected."""
		
		def setLocalOptions(option):
			"""Enable/Disable for local backup"""
			self.ui.txtLocalFolder.setEnabled(option)
			self.ui.buttonSelectLocalFolder.setEnabled(option)
		
		def setRemoteOptions(option):
			"""Enable/Disable for remote backup"""
			self.ui.lblRemoteServer.setEnabled(option)
			self.ui.txtRemoteHost.setEnabled(option)
			self.ui.lblRemotePort.setEnabled(option)
			self.ui.txtRemotePort.setEnabled(option)
			self.ui.lblRemotePath.setEnabled(option)
			self.ui.txtRemotePath.setEnabled(option)
			self.ui.lblUsername.setEnabled(option)
			self.ui.txtUsername.setEnabled(option)
			self.ui.lblPassword.setEnabled(option)
			self.ui.txtPassword.setEnabled(option)
			self.ui.comboRemoteProtocol.setEnabled(option)
			self.ui.buttonTestConnection.setEnabled(option)
			if self.ui.txtRemotePort.text() == "":
				self.ui.txtRemotePort.setText(str(utils.getPortProtocol(self.ui.comboRemoteProtocol.currentText())))
		
		def setAmazonS3Options(option):
			"""Enable/Disable for Amazon S3"""
			self.ui.txtAmazonS3_Bucket.setEnabled(option)
			self.ui.lblAmazonS3_AccessId.setEnabled(option)
			self.ui.lblAmazonS3_BucketType.setEnabled(option)
			self.ui.txtAmazonS3_AccessId.setEnabled(option)
			self.ui.lblAmazonS3_SecretKey.setEnabled(option)
			self.ui.txtAmazonS3_SecretKey.setEnabled(option)
			self.ui.comboAmazonS3_BucketType.setEnabled(option)
			
		# Enable Local Backup
		if self.ui.radioLocalBackup.isChecked():
			setLocalOptions(True)
			setRemoteOptions(False)
			setAmazonS3Options(False)
		
		# Enable Remote Backup
		if self.ui.radioRemoteBackup.isChecked():
			setLocalOptions(False)
			setRemoteOptions(True)
			setAmazonS3Options(False)

		if self.ui.radioAmazonS3Backup.isChecked():
			setLocalOptions(False)
			setRemoteOptions(False)
			setAmazonS3Options(True)
	
	def AutoRemoveSettings_Update(self):
		"""Enable or disable the auto-remove settings based on the options selected."""
		
		def setOlderThan(option):
			"""Enable/Disable for older than options"""
			self.ui.spinOlderThan.setEnabled(option)
			self.ui.comboOlderThan.setEnabled(option)
		
		def setFreeSpace(option):
			"""Enable/Disable for free space options"""
			self.ui.spinFreeSpace.setEnabled(option)
			self.ui.comboFreeSpace.setEnabled(option)
		
		if self.ui.checkOlderThan.isChecked():
			setOlderThan(True)
		else:
			setOlderThan(False)
		
		if self.ui.checkFreeSpace.isChecked():
			setFreeSpace(True)
		else:
			setFreeSpace(False)
	
	
	def Options_Update(self):
		"""Enable or disable the options controls based on the values selected."""
		if self.ui.checkEncryptFiles.isChecked():
			self.ui.txtEncryptionKey.setEnabled(True)
		else:
			self.ui.txtEncryptionKey.setEnabled(False)
		
		if self.ui.checkFullBackupFrequency.isChecked():
			self.ui.lblFullBackupFrequency_Days.setEnabled(True)
			self.ui.spinFullBackupFrequency.setEnabled(True)
		else:
			self.ui.lblFullBackupFrequency_Days.setEnabled(False)
			self.ui.spinFullBackupFrequency.setEnabled(False)
	
	def Advanced_Update(self):
		"""Enable or disable the options controls based on the values selected."""
		self.ui.txtDefaultRestoreLocation.setEnabled(self.ui.checkDefaultRestoreLocation.isChecked())
		self.ui.buttonSelectDefaultRestoreLocation.setEnabled(self.ui.checkDefaultRestoreLocation.isChecked())
	
	def UpdateSettings(self):
		"""Reads settings and preferences from the settings dialog and writes them
		to platform appropriate storage.  (Registry on Windows, Xml Files on Mac OS X,
		and .ini files on Linux/Unix."""
		settings = Settings()

		# Local Backup Settings
		settings.LocalBackup = self.ui.radioLocalBackup.isChecked()
		settings.LocalBackupDirectory = self.ui.txtLocalFolder.text()
		
		# Remote Backup Settings
		settings.RemoteBackup = self.ui.radioRemoteBackup.isChecked()
		settings.RemoteProtocolInt = self.ui.comboRemoteProtocol.currentIndex()
		settings.RemoteProtocol = self.ui.comboRemoteProtocol.currentText()
		settings.RemotePath = self.ui.txtRemotePath.text()
		if self.ui.txtRemotePort.text() == "":		
			settings.RemotePort = 0
		else:
			settings.RemotePort = int(self.ui.txtRemotePort.text())
		settings.RemoteHost = self.ui.txtRemoteHost.text()
		settings.RemoteUsername = self.ui.txtUsername.text()
		settings.RemotePassword = self.ui.txtPassword.text()

		# Amazon S3
		settings.AmazonS3Backup = self.ui.radioAmazonS3Backup.isChecked()
		settings.AmazonS3_Bucket = self.ui.txtAmazonS3_Bucket.text()
		settings.AmazonS3_AccessId = self.ui.txtAmazonS3_AccessId.text()
		settings.AmazonS3_SecretKey = self.ui.txtAmazonS3_SecretKey.text()
		settings.AmazonS3_LocationInt = self.ui.comboAmazonS3_BucketType.currentIndex()
		settings.AmazonS3_Location = self.ui.comboAmazonS3_BucketType.currentText()
		
		# Autoremove Settings - Older Than
		settings.AutoRemove_OlderThan = self.ui.checkOlderThan.isChecked()
		settings.AutoRemove_OlderThan_Duration = self.ui.spinOlderThan.value()
		settings.AutoRemove_OlderThan_Units = self.ui.comboOlderThan.currentIndex()
		
		# Autoremove Settings - Free Space
		settings.AutoRemove_FreeSpace = self.ui.checkFreeSpace.isChecked()
		settings.AutoRemove_FreeSpace_Size = self.ui.spinFreeSpace.value()
		settings.AutoRemove_FreeSpace_Units = self.ui.comboFreeSpace.currentIndex()
		
		# Autoremove Settings - Smart Remove
		settings.AutoRemove_SmartRemove = self.ui.checkSmartRemove.isChecked()
		settings.AutoRemove_NamedSnapshots = self.ui.checkPreserveNamedSnapshots.isChecked()
		
		# Backup Schedule
		if settings.Schedule != self.ui.comboSchedule.currentIndex():
			utils.setup_cron( self.ui.comboSchedule.currentIndex())
		settings.Schedule = self.ui.comboSchedule.currentIndex()
		
		# Archive Options
		settings.Options_EncryptFilesEnabled = self.ui.checkEncryptFiles.isChecked()
		settings.Options_EncryptionKey = self.ui.txtEncryptionKey.text()
		settings.Options_EnableNotifications = self.ui.checkEnableNotifications.isChecked()
		settings.Options_ExcludeHidden = self.ui.checkExcludeHidden.isChecked()
		settings.Options_FullBackupFrequencyEnabled = self.ui.checkFullBackupFrequency.isChecked()
		settings.Options_FullBackupFrequency = self.ui.spinFullBackupFrequency.value()

		# advanced
		settings.Advanced_DefaultRestoreEnabled = self.ui.checkDefaultRestoreLocation.isChecked()
		settings.Advanced_DefaultRestoreLocation = self.ui.txtDefaultRestoreLocation.text()

		backupsettings.writeSettings(settings)
	
	
	def btnOk_Pressed(self):
		"""
		Closes only if all data input is correct the settings dialog, also checks data input and stores 
		"""
		if (self.ui.radioRemoteBackup.isChecked() and gui_utils.checkRemoteSettings(self, str(self.ui.txtRemoteHost.text())))or (self.ui.radioLocalBackup.isChecked() and os.path.exists(str(self.ui.txtLocalFolder.text()))) or (self.ui.radioAmazonS3Backup.isChecked()):
			self.UpdateSettings()			
			self.close()
	
	
	def btnCancel_Pressed(self):
		self.close()
		
	def closeEvent(self, event):
		self.ui.lblTstResult.setVisible(False)
	
	
	def btnIncludeAddLocal_Pressed(self):
		folderName = QtGui.QFileDialog.getExistingDirectory()
		
		if folderName != "":
			f = includelocalitem.IncludeLocalItem()
			f.set_default_values()
			f.FolderName = folderName
			self.backupsettings.IncludeList.append(f)
			self.loadIncludeList()
	
	def btnIncludeManageRemote_Pressed(self):
		if self.ui.treeInclude.currentItem() != None:
			folderName = self.ui.treeInclude.currentItem().text(1)
			btype = self.ui.treeInclude.currentItem().text(2)
			if folderName != None:
				folder = utils.findFolder(self.backupsettings.IncludeList, folderName, utils.textToBackupType(btype))
				if folder.LocationType == folder.LOCATION_REMOTE:
					self.ui.RemoteConnectionSettingsDialog.IncludeRemoteItem = folder
					self.ui.RemoteConnectionSettingsDialog.show()
				else:
					QtGui.QMessageBox.warning(self, _("Error"), _("Please select a remote folder"))
		else:
			QtGui.QMessageBox.warning(self, _("Error"), _("No folder was selected!"))
		
		
	def btnIncludeAddRemote_Pressed(self):
		self.ui.RemoteConnectionSettingsDialog.IncludeRemoteItem = None
		self.ui.RemoteConnectionSettingsDialog.show()
	
	def btnIncludeRemove_Pressed(self):
		removeFolder = self.ui.treeInclude.currentItem()
		curIndex = self.ui.treeInclude.indexOfTopLevelItem(removeFolder)
		
		if removeFolder != None:
			folder = utils.findFolder(self.backupsettings.IncludeList, removeFolder.text(1), utils.textToBackupType(removeFolder.text(2)))
			self.backupsettings.IncludeList.remove(folder)
			toDelete = self.ui.treeInclude.takeTopLevelItem(curIndex)
			del curIndex, removeFolder, toDelete

	
	def buttonSelectLocalFolder_Pressed(self):
		folderName = QtGui.QFileDialog.getExistingDirectory()
			
		if folderName != "":
			self.ui.txtLocalFolder.setText(folderName)

	def buttonSelectDefaultRestoreLocation_Pressed(self):
		folderName = QtGui.QFileDialog.getExistingDirectory()
			
		if folderName != "":
			self.ui.txtDefaultRestoreLocation.setText(folderName)
			
	def buttonTestConnection_Pressed(self):
		archiveProtocol = self.ui.comboRemoteProtocol.currentText()
		archiveHost = self.ui.txtRemoteHost.text()
		archivePort = self.ui.txtRemotePort.text()
		archiveUsername = self.ui.txtUsername.text()
		archivePassword = self.ui.txtPassword.text()
		archiveFolder = self.ui.txtRemotePath.text()
		self.ui.buttonTestConnection.setEnabled(False)
		self.doTestConnection(archiveProtocol, archiveUsername, archivePassword, archiveHost, archivePort, archiveFolder)	
	
	def ShowRemoteResults(self, tstResult):
		"""
		Updates the settings view based on the results of the connection test.
		
		@type tstResult: Boolean
		@param tstResult: result from the test connection url check
		"""
		self.ui.lblTstResult.setVisible(True)
		
		if tstResult == True:
			imageSuccess = QtGui.QPixmap(':/SVG/Resources/Dialog-Ok.svg')	
			self.ui.lblTstResult.setPixmap(imageSuccess)
		elif tstResult == False:
			imageFailure = QtGui.QPixmap(':/SVG/Resources/Dialog-Cancel.svg')
			self.ui.lblTstResult.setPixmap(imageFailure)
		
		# Re-enable buttonTestConnection
		self.GeneralSettings_Update()	
	
	def AddToExcludeList(self, e):
		"""
		Adds an excludeItem to the global exclude list.
		and show it on the exclude ui
		
		@type e: ExcludeItem Object
		@param e:	
		"""
		
		excludeItem = self.getExcludeTreeItem(e)
		self.ui.treeExclude.addTopLevelItem(excludeItem)
		self.backupsettings.ExcludeList.append(e)
	
	
	def btnExcludeAdd_Pressed(self):
		excludePattern = QtGui.QInputDialog.getText(self, _("Exclude Pattern"), "", QtGui.QLineEdit.Normal)
		if not excludePattern[1]:
			return
		
		e = self.validateExcludeItem(excludePattern, excludeitem.ExcludeItem.REGEXP)
		if not e == None:
			self.AddToExcludeList(e)
			
			
	def btnExcludeRemove_Pressed(self):
		removePattern = self.ui.treeExclude.currentItem()
		curIndex = self.ui.treeExclude.indexOfTopLevelItem(removePattern)
		
		if removePattern != None:
			for e in self.backupsettings.ExcludeList:
				if removePattern.text(0) == e.Item:
					self.backupsettings.ExcludeList.remove(e)
					continue

			toDelete = self.ui.treeExclude.takeTopLevelItem(curIndex)
			del curIndex, removePattern, toDelete
	
	
	def buttonFolderOptions_Pressed(self):
		if self.ui.treeInclude.currentItem() != None:
			folderName = self.ui.treeInclude.currentItem().text(1)
			btype = self.ui.treeInclude.currentItem().text(2)
			if folderName != None:
				folder = utils.findFolder(self.backupsettings.IncludeList, folderName, utils.textToBackupType(btype))
				self.ui.AdvancedSettingsDialog.LoadPreferences(folder)
				self.ui.AdvancedSettingsDialog.show()
		else:
			QtGui.QMessageBox.warning(self, _("Error"), _("No folder was selected!"))

	def comboRemoteProtocol_currentIndexChanged(self, index):
		self.ui.txtRemotePort.setText(str(utils.getPortProtocol(self.ui.comboRemoteProtocol.currentText())))
		

	def tabSettings_currentChanged(self, index):
		if self.ui.radioRemoteBackup.isChecked():
			gui_utils.checkRemoteSettings(self, str(self.ui.txtRemoteHost.text()))
		
	def loadIncludeList(self):
		""" Load Backup Include List """
		includeItemList = []
		folderIcon = QtGui.QIcon(QtGui.QPixmap(":/SVG/Resources/Folder.svg"))
		self.ui.treeInclude.clear()
		
		for item in self.backupsettings.IncludeList:
			includeItem = QtGui.QTreeWidgetItem([QtCore.QString(utils.locationTypeToText(item)),
							     QtCore.QString(utils.determine_folderName(item)),
							     QtCore.QString(utils.backupTypeToText(item))])
			includeItem.setIcon(0,folderIcon)
			includeItemList.append(includeItem)
		self.ui.treeInclude.insertTopLevelItems(0, includeItemList)