#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
from PyQt4 import QtCore, QtGui
from socket import gethostname
import os

from timedrive.utils import duplicity_interface
from duplicity import dup_time
from timedrive.background import backgroundtask
import timedrive.utils as utils
from timedrive.utils import archiveurl

class LoadArchive(backgroundtask.BackgroundTask):
	"""Background class that retrieves remote file lists and notifies the user of its progress."""
	
	def __init__(self, parent = None):
		"""
		Initialize LoadArchive class
		
		@type parent: BackgroundTask Object
		@param parent:
		"""
		super(LoadArchive, self).__init__(_("Load Archive"), parent)
		
		self.modelFileList = QtGui.QStandardItemModel()
		self.From_Settings = None
		self.From_Url = None
		self.From_Timeline = None
		self.archiveUrl = None
		self.passphrase = None
		self.restoreTime = None
		self.S3_AccessId = None
		self.S3_SecretKey = None
	
	def FromSettings(self, settings, restore_time = None):
		"""
		Load a remote archive from the settings
		
		@type settings: Settings Object
		@param settings:
		@type restore_time:
		@param restore_time: default=None
		"""
		self.From_Settings = True
		self.From_Url = False
		self.From_Timeline = False
		self.settings = settings
		self.restoreTime = restore_time
		self.S3_AccessId = None
		self.S3_SecretKey = None
		self.modelFileList.clear()
		self.start()
		
	def FromUrl(self, archiveUrl, gnu_passphrase,
		S3_AccessId = None, S3_SecretKey = None):
		"""
		Load a remote archive from a valid Url
		
		@type archiveUrl: String
		@param archiveUrl:
		@type gnu_passphrase: String
		@param gnu_passphrase:
		@type S3_AccessId: String
		@param S3_AccessId: default=None
		@type S3_SecretKey: String
		@param S3_SecretKey: default=None
		"""
		
		self.From_Settings = False
		self.From_Url = True
		self.From_Timeline = False
		self.archiveUrl = archiveUrl
		self.passphrase = gnu_passphrase
		self.settings = None
		self.S3_AccessId = S3_AccessId
		self.S3_SecretKey = S3_SecretKey
		self.modelFileList.clear()
		self.start()
		
	def FromTimeline(self, settings, restore_time):
		"""
		Loads the archive based on existing settings
		and a valid restore_time
		
		@type settings: Settings Object
		@param settings:
		@type restore_time:
		@param restore_time:
		"""
		
		self.From_Settings = False
		self.From_Url = False
		self.From_Timeline = True
		self.settings = settings
		self.restoreTime = restore_time
		self.S3_AccessId = None
		self.S3_SecretKey = None
		self.modelFileList.clear()
		self.start()

	def GetFileList(self, archive_url, gnu_passphrase, 
		restore_time = None, sync = True, S3_AccessId = None,
		S3_SecretKey = None, S3_BucketType = None):
		"""
		Convenience method for retrieving a hierarchical file 
		structure for all files in the specified archive.
		The list is generated by examining the signature only.
		Returns a QStandardItem with all associated files as children.
		"""
		
		# Set Amazon S3 values from settings
		if self.settings != None:
			if self.settings.AmazonS3Backup == True:
				S3_AccessId = self.settings.AmazonS3_AccessId
				S3_SecretKey = self.settings.AmazonS3_SecretKey
			if self.settings.AmazonS3_Location == "European":
				S3_BucketType = self.settings.AmazonS3_Location
		
		# Retrieve the Duplicity file list structure
		path_iter = duplicity_interface.get_file_list(archive_url, gnu_passphrase, 
			sync, restore_time, S3_AccessId, S3_SecretKey, S3_BucketType)
		
		# If nothing returned, create blank QStandardItem
		if path_iter == None:
			itemArchiveUrl = QtGui.QStandardItem(QtCore.QString(archive_url))
			itemPassPhrase = QtGui.QStandardItem(QtCore.QString(gnu_passphrase))
			return [QtGui.QStandardItem(""), itemArchiveUrl, itemPassPhrase]
		
		# Parse the Path_Iter and return a QStandardItem with hierarchy
		itemFileList = self.ParseDirectoryStructure(path_iter, archive_url, 
			gnu_passphrase, restore_time)
		return itemFileList

	def ParseDirectoryStructure(self, path_iter, archive_url, gnu_passphrase, restore_time = None):
		"""
		Parses the directory structure of supplied path_iter
	
		@type path_iter: PathIteration object
		@param path_iter (duplicity class)
	
		@rtype: QStandardItemModel (PyQt class)
		@return: modelFileList (QStandardItemModel) that contains data regarding
		the files in the currently selected archive.
		"""
		modelFileList = QtGui.QStandardItemModel()
		if restore_time != None:
			itemFileList = QtGui.QStandardItem(restore_time)
		else:
			itemFileList = QtGui.QStandardItem(_('Most Recent'))
		modelFileList.appendRow([itemFileList])
	
		dirItem = modelFileList.item(0, 0)
		itemArchiveUrl = QtGui.QStandardItem(QtCore.QString(archive_url))
		itemPassPhrase = QtGui.QStandardItem(QtCore.QString(gnu_passphrase))
	
		# Create a treeModel with Filenames and Basic Attributes
		for path in path_iter:
			if path.difftype != "deleted":
				relPath = path.get_relative_path()
				fileName = os.path.basename(relPath)
				dirName = os.path.dirname(relPath)
			
				try:
					curDir = modelFileList.data(dirItem.index().sibling(dirItem.row(), 2)).toString()
				except:
					curDir = ''
			
				# Exclude the Root of the Directory Structure
				if fileName != '.':			
					itemName = QtGui.QStandardItem(QtCore.QString(fileName))
					itemDateTime = QtGui.QStandardItem(QtCore.QString(dup_time.timetopretty(path.getmtime())))
					itemRelPath = QtGui.QStandardItem(QtCore.QString(relPath))
				
					# Directory Handling
					if path.isdir():
						# Root Directory
						if (curDir == ''):
							dirItem.appendRow([itemName, itemDateTime, itemRelPath])
							dirItem = itemName	
						# Add a subdirectory to the current directory
						elif (curDir==dirName):
							dirItem.appendRow([itemName, itemDateTime, itemRelPath])
							dirItem = itemName
						# Find the correct directory and add subdirectory
						elif (curDir!=dirName):
							try:
								while (curDir != dirName):
									dirItem = dirItem.parent()
									curDir = modelFileList.data(dirItem.index().sibling(dirItem.row(),
										2)).toString()
								dirItem.appendRow([itemName, itemDateTime, itemRelPath])
								dirItem = itemName
							
							# Go to the root directory and add a subdirectory
							except:
								dirItem = modelFileList.item(0, 0)
								dirItem.appendRow([itemName, itemDateTime, itemRelPath])
								dirItem = itemName
				
					# File Handling	
					else:
						# Root Directory
						if (curDir == dirName):
							dirItem.appendRow([itemName, itemDateTime, itemRelPath])
						else:
							try:
								while (curDir != dirName):
									dirItem = dirItem.parent()
									curDir = modelFileList.data(dirItem.index().sibling(dirItem.row(), 2)).toString()
								dirItem.appendRow([itemName, itemDateTime, itemRelPath])
							
							# Go to the root directory and add file
							except:
								dirItem = modelFileList.item(0, 0)
								dirItem.appendRow([itemName, itemDateTime, itemRelPath])
	
		return [modelFileList.takeItem(0, 0), itemArchiveUrl, itemPassPhrase]
			
	def run(self):
		"""Primary method of the thread.  Checks to see if it needs to create a file list from the local settings or a supplied Url from the advanced restore options dialog.  Returns a list of QStandardItemModels for all of the folders in the backup.  If loading a remote archive, the list will only contain one entry."""
		
		self.toggleControls(False)
		self.default_message = _("Loading File List ... ")
		self.modelFileList.setHorizontalHeaderItem(0, QtGui.QStandardItem(_('File Name')))
		self.modelFileList.setHorizontalHeaderItem(1, QtGui.QStandardItem(_('Date Modified')))
		self.modelFileList.setHorizontalHeaderItem(2, QtGui.QStandardItem(_('Relative Path')))
		
		if (self.From_Settings == True)|(self.From_Timeline == True):
		# Load from existing backup settings
			numItems = len(self.settings.IncludeList)
			self.setupTaskProgress(numItems)
			machineName = gethostname()
			progress = 0
			itemDirectoryList = QtGui.QStandardItem(_("Time Vault Backup List - Local Settings"))
			itemNumDirectories = QtGui.QStandardItem(numItems)
			self.modelFileList.appendRow([itemDirectoryList, itemNumDirectories])
		
			for folder in self.settings.IncludeList:
				if self.stop:
					continue
				folderName = utils.determine_folderName(folder)
				if folder.LocationType == folder.LOCATION_LOCAL:
					folderName = os.path.basename(os.path.normpath(str(folderName)))
				
				# Create archiveUrl and gnuPassphrase from General Settings
				au = archiveurl.ArchiveUrl_FromSettings(folder, self.settings)
				if au.available:
					if self.From_Settings == True:
						itemFiles = self.GetFileList(str(au.archiveUrl), au.gnuPassphrase, self.restoreTime, True)
					elif self.From_Timeline == True:
						itemFiles = self.GetFileList(str(au.archiveUrl), au.gnuPassphrase, self.restoreTime, False)
					self.modelFileList.appendRow(itemFiles)
			
				else:
					self._error_available(self.task_name, folder)
				
				progress = progress + 1
				self.updateTaskProgress(progress)
		
		# Load from a Url
		elif self.From_Url == True:
			itemDirectoryList = QtGui.QStandardItem(_("Time Vault Backup List - Remote Archive"))
			itemNumDirectories = QtGui.QStandardItem(1)
			self.modelFileList.appendRow([itemDirectoryList, itemNumDirectories])
			itemFiles = self.GetFileList(str(self.archiveUrl), str(self.passphrase), 
				None, True, self.S3_AccessId, self.S3_SecretKey)
			self.modelFileList.appendRow(itemFiles)
		
		# Update MainWindow, RestoreDialog and notify user
		self.emit(QtCore.SIGNAL("ReturnFileList(QStandardItemModel*)"), self.modelFileList)
		self.toggleControls(True)
		self.taskComplete(self.task_name, _("File List Loaded"))
		
		# Reset Values
		self.settings = None
		self.archiveUrl = None
		self.passphrase = None
		self.From_Settings = None
		self.From_Url = None
		self.From_Timeline = None
		self.restoreTime = None
		self.S3_AccessId = None
		self.S3_SecretKey = None
