#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup.
#
from PyQt4 import QtCore
import os
import timedrive.utils as utils
import timedrive.backupsettings as backupsettings

from timedrive.background import backgroundtask
from timedrive.backupsettings import globals 
from timedrive.utils import duplicity_interface
from timedrive.utils import log
from timedrive.utils import archiveurl
from timedrive.utils import validate

class BackupJob(backgroundtask.BackgroundTask):
	"""
	Class that executes backup jobs and updates user on progress.  Designed to run
	as a background thread so that the main GUI instance remains responsive.
	"""
	
	def __init__(self, parent = None):
		"""
		Initialization function for BackupJob
		
		@type parent: BackgroundTask Object
		@param parent:
		"""
		super(BackupJob, self).__init__(_("Backup Job"), parent)
		self._Validate = validate.Validate()
		self.exiting = False
	
	def StartBackup(self, settings):
		"""
		Start a backup job, either from the GUI or from the command line.
		
		@type settings: Settings Object
		@param settings:
		"""
		self.settings = settings
		self.start()

	def isValidExclude(self, exclude, mountpoint = None):
		"""
		check if exclude type is know and validate the exclude
		
		@type exclude: ExcludeItem Object
		@param exclude: the exclude of type ExcludeItem
		
		@type mountpoint: String
		@param mountpoint: in this case it's added for remote folders to define there mount points,
					because it could change on a different computer. 
		
		@rtype: Boolean
		"""
		if mountpoint is None:
			mountpoint = ""

		if exclude.Type == exclude.FOLDER:
			if os.path.exists(mountpoint + exclude.Item):
				return True
			else:
				log.Error(str(mountpoint + exclude.Item) + " could not be found, ignoring")

		elif exclude.Type == exclude.FILE:
			if os.path.isfile(mountpoint + exclude.Item):
				return True
			else:
				log.Error(str(mountpoint + exclude.Item) + " could not be found, ignoring")

		elif exclude.Type == exclude.REGEXP:
			if self._Validate.check(self._Validate.REGEX, exclude.Item):
				return True
			else:
				log.Error(str(exclude.Item) + " invalid REGEXP, ignoring")
		else:
			raise Exception(_("Invalid exclude type"))
	
		return False
			
	def process_duplicityError(self, task, result_err, folder):
		"""
		Error processing of duplicity
		
		@type task: String
		@param task: the name of task where the error occured
		@type result_err: String
		@param result_err: message that contains the error
		@type folder: IncludeItem Object
		@param folder: the includeitem for whitch the error occured
		"""
		if self.settings.Mode == globals.GuiMode:
			handle = utils.handle_error(result_err)
			folderName = utils.determine_folderName(folder)
			if handle == 1:
				self.questionPrompt(_("%s %s Computer name changed") % (str(task), str(folderName)))
				duplicity_interface.allow_source_mismatch = True
			elif handle == 99:
				self.taskError(task, _("Unknow Fatal error %s") % str(folderName),result_err)
			elif handle == 0:
				self.taskError(task, _("Backend Error occurd with folder %s") % str(folderName),result_err)
			else:
				self.taskError(task, _("Unknow Error occured %s") % str(folderName),result_err)
				log.Error(result_err)
		elif self.settings.Mode == globals.BatchMode:
			self.settings.ErrorOccurred = True
			backupsettings.writeSettings(self.settings)
			log.Error(result_err)
			
	def buildExcludeList(self, excludeItemList, mountpoint = None):
		"""
		Builds an valid Exclude list based on ExcludeItems

		
		@type excludeItemList: list (ExcludeItem Object)
		@param exclude: the list of type ExcludeItem
		
		@type mountpoint: String
		@param mountpoint: default : none in this case it's added for remote folders to define there mount points,
					because it could change on a different computer. 
		
		@rtype: excludeList []
		@return: returns a list to pass on to duplicity's interface.
		"""
		excludelist = []
		if mountpoint is None:
			mountpoint = ""
			
		for e in excludeItemList:
			if self.isValidExclude(e, mountpoint):
				excludelist.append((mountpoint + e.Item, e.Type))
		
		return excludelist

	def doBackupJob(self, include, archiveUrl, gnuPassphrase):
		"""
		Does the backup job for a single folder, archive url already build
		
		@type folder: IncludeItem Object
		@param folder:
		@type archiveUrl: String
		@param archiveUrl:
		@type gnuPassphrase: String
		@param gnuPassphrase:
		"""
		#reset fatal error options
		duplicity_interface.allow_source_mismatch = False
	
		combinedExcludeList = []
		
		#folderName = os.path.basename(str(include.FolderName))
		
		if include.LocationType == include.LOCATION_LOCAL:
			folderName = utils.determine_folderName(include)
		elif include.LocationType == include.LOCATION_REMOTE:
			folderName = utils.mount_remote(include)
			if folderName is None:
				self._error(_("The mount failed: %s, this could be that the command isn't found") % include.RemoteName)
				return


		if self.settings.Options_ExcludeHidden == True:
			combinedExcludeList.append(globals.EXCLUDEHIDDEN)
		
		if self.settings.Options_FullBackupFrequencyEnabled == True:
			FullBackupInterval = self.settings.Options_FullBackupFrequency
		else:
			FullBackupInterval = None
		
		if include.LocationType == include.LOCATION_REMOTE:
			combinedExcludeList.extend(self.buildExcludeList(self.settings.ExcludeList, folderName))
			combinedExcludeList.extend(self.buildExcludeList(include.ExcludeList, folderName))
		else:
			combinedExcludeList.extend(self.buildExcludeList(self.settings.ExcludeList))
			combinedExcludeList.extend(self.buildExcludeList(include.ExcludeList))
		
		if self.settings.Options_EncryptFilesEnabled == False:
			gnuPassphrase = None
			

		# Start the Backup Job
		try:
			self._restart = True

			while self._restart:
				self._restart = False
				if self.settings.AmazonS3Backup == True:
					if self.settings.AmazonS3_Location == "European":
						S3_BucketType = self.settings.AmazonS3_Location
					else:
						S3_BucketType = None
						
					result_out, result_err = duplicity_interface.start_backup(folderName, 
						archiveUrl, gnuPassphrase, combinedExcludeList, FullBackupInterval,
						self.settings.AmazonS3_AccessId, self.settings.AmazonS3_SecretKey,
						S3_BucketType)
				else:
					result_out, result_err = duplicity_interface.start_backup(folderName, archiveUrl, gnuPassphrase, combinedExcludeList, FullBackupInterval)
				
				if result_out:
					log.Info(_("%s Backup Successful using %s \n %s") % (folderName, utils.backupTypeToText(include), result_out))
				
				if result_err and result_err != "":
					self.process_duplicityError("BackupJob", result_err, include)
			
		except:
			self._error(_("Failed: %s") % folderName)
			
		if include.LocationType == include.LOCATION_REMOTE:
			if not utils.umount_remote(include):
				self._error(_("Error occured during the unmount process of location: %s, please check command fusermount exists") % folderName)

	def _error(self, msg):
		"""
		Returns a given message to the log and emits and error
		
		@type msg: String
		@param msg: a message to the user
		"""
		log.Error(msg)
		self.taskError(self.task_name, "error",  QtCore.QString(msg))

	def run(self):
		"""
		The run it self, to process each folder in the folder list.
		"""
		self.default_message = _("Backup Running ... ")
		
		numItems = len(self.settings.IncludeList)
		self.setupTaskProgress(numItems)
		progress = 0	

		for include in self.settings.IncludeList:
			if self.stop:
				continue
			
			progress += 1

			au = archiveurl.ArchiveUrl_FromSettings(include, self.settings)
			if au.available:					
				self.doBackupJob(include, str(au.archiveUrl), str(au.gnuPassphrase))
			else:
				self._error_available(self.task_name, include)				
			
			self.updateTaskProgress(progress)
			
		
		self.updateTaskProgress(numItems)
		self.taskComplete(self.task_name, _("Backup Completed"))
