#!/usr/bin/env python
#
# Time Drive - based on duplicity - Encrypted bandwidth efficient backup
# Requires Duplicity Version 0.6.04 released August 01, 2009.
#
# Copyright 2009 Rob Oakes	<LyX-Devel@oak-tree>

from PyQt4 import QtCore, QtGui

import timedrive.utils as utils
from timedrive.utils import gui_utils
from timedrive.backupsettings import excludeitem
import timedrive.settingsdialog as settingsdialog
import re

class AdvancedSettingsDialog(settingsdialog.SettingsDialog):
	"""Set advanced settings for archives included in the backup job description."""
	
	def __init__(self, parent):
		"""
		Initialization function for AdvancedSettingsDialog
		"""
		super(AdvancedSettingsDialog, self).__init__(parent)
		gui_utils.loadUi(self, "uiAdvancedFolderSettings.ui")
		self.ui = self
		self.folder = None
		self.LoadSnapshots = self.parent().parent().LoadSnapshots
		self.ManageBackups = self.parent().parent().ManageBackups
		
		# Set Up the Snapshots List
		self.modelSnapshotsList = self.setupSnapshotsList()
		self.ui.treeSnapshotHistory.setModel(self.modelSnapshotsList)
		self.setupSnapshotsList_SetViewParameters()
		
		# Set Initial State of UI
		self.ui.lblTstResult.setVisible(False)
		
		# Events and Methods
		QtCore.QObject.connect(self.ui.radioGeneralSettings, QtCore.SIGNAL("released()"),
			self.ArchiveSettings_Update)
		QtCore.QObject.connect(self.ui.radioLocalBackup, QtCore.SIGNAL("released()"),
			self.ArchiveSettings_Update)
		QtCore.QObject.connect(self.ui.radioRemoteBackup, QtCore.SIGNAL("released()"),
			self.ArchiveSettings_Update)
		QtCore.QObject.connect(self.ui.btnExcludeAddPattern, QtCore.SIGNAL("clicked()"),
			self.btnExcludeAddPatern_Pressed)
		QtCore.QObject.connect(self.ui.btnExcludeAddFolder, QtCore.SIGNAL("clicked()"),
			self.btnExcludeAddFolder_Pressed)
		QtCore.QObject.connect(self.ui.btnExcludeAddFile, QtCore.SIGNAL("clicked()"),
			self.btnExcludeAddFile_Pressed)
		QtCore.QObject.connect(self.ui.btnExcludeRemove, QtCore.SIGNAL("clicked()"),
			self.btnExcludeRemove_Pressed)
		QtCore.QObject.connect(self.ui.buttonSelectLocalFolder, QtCore.SIGNAL("clicked()"),
			self.buttonSelectLocalFolder_Pressed)
		QtCore.QObject.connect(self.ui.buttonTestConnection, QtCore.SIGNAL("clicked()"),
			self.buttonTestConnection_Pressed)
		QtCore.QObject.connect(self.ui.btnRunAutoRemove,
			QtCore.SIGNAL("clicked()"), self.manageBackups_RunAutoRemove)
		QtCore.QObject.connect(self.ui.btnOk, 
			QtCore.SIGNAL("clicked()"), self.btnOk_Pressed)
		QtCore.QObject.connect(self.ui.btnCancel, 
			QtCore.SIGNAL("clicked()"), self.btnCancel_Pressed)
			
		QtCore.QObject.connect(self.ui.checkEncryptFiles,
			QtCore.SIGNAL("toggled(bool)"), 
			self.checkEncryptFiles_Toggled)
		QtCore.QObject.connect(self.ui.btnLoadHistory, QtCore.SIGNAL("clicked()"),
			self.LoadFolderSnapshots)
		QtCore.QObject.connect(self.ui.btnDeleteSnapshot, QtCore.SIGNAL("clicked()"),
			self.manageBackups_DeleteSet)
		QtCore.QObject.connect(self.ui.btnCleanupBackup, QtCore.SIGNAL("clicked()"),
			self.manageBackups_Cleanup)
		
		QtCore.QObject.connect(self.ui.tabWidget, 
			QtCore.SIGNAL("currentChanged ( int )"),
			self.tabWidget_currentChanged)
		QtCore.QObject.connect(self.ui.comboRemoteProtocol, 
			QtCore.SIGNAL("activated(const QString &)"),
			self.comboRemoteProtocol_currentIndexChanged)
		
		QtCore.QObject.connect(self.ui.treeSnapshotHistory, 
			QtCore.SIGNAL("activated(QModelIndex)"), self.ManageSnapshots_Update)
		QtCore.QObject.connect(self.ui.treeSnapshotHistory, 
			QtCore.SIGNAL("clicked(QModelIndex)"), self.ManageSnapshots_Update)
		
		# Signals and Slots Related to Threads		
		QtCore.QObject.connect(self.LoadSnapshots,  
			QtCore.SIGNAL("ReturnSnapshotList_Folder(QStandardItemModel*)"), 
			self.LoadFolderSnapshots_SetModelList)
		QtCore.QObject.connect(self.LoadSnapshots,
			QtCore.SIGNAL("toggleControls(bool)"), 
			self.LoadFolderSnapshots_ToggleControls)
		
		QtCore.QObject.connect(self.parent().parent().ui.windowRestore.LoadArchive,
			QtCore.SIGNAL("toggleControls(bool)"), 
			self.LoadFolderSnapshots_ToggleControls)
			
		QtCore.QObject.connect(self.ManageBackups, QtCore.SIGNAL("toggleControls(bool)"),
			self.LoadFolderSnapshots_ToggleControls)
		QtCore.QObject.connect(self.ManageBackups, QtCore.SIGNAL("finished()"),
			self.manageBackups_ResetModel)
		
		# Run Setup Events
		self.ArchiveSettings_Update()
		self.ManageSnapshots_Update()
	
	
	def LoadPreferences(self, folder):
		"""
		Sets the dialog box control values from a valid folder data list.
		
		@type folder: IncludeItem Object
		@param folder:
		"""
		folderName = utils.determine_folderName(folder)
		
		self.ui.lblFolderName.setText(_("Advanced Settings - ") + folderName)
		self.folder = folder
		
		if (folder.BackupType == folder.GENERALSETTINGS):
			self.ui.radioGeneralSettings.setChecked(True)
		elif folder.BackupType == folder.LOCAL:
			self.ui.radioLocalBackup.setChecked(True)
		elif folder.BackupType == folder.REMOTE:
			self.ui.radioRemoteBackup.setChecked(True)
		
		# Load Folder Exclude List
		excludeItemList = []
		trashIcon = QtGui.QIcon(QtGui.QPixmap(":/PNG/Resources/Remove-Snapshot.png"))
		self.ui.treeExclude.clear()
		for item in folder.ExcludeList:
			excludeItem = QtGui.QTreeWidgetItem([QtCore.QString(item.Item)])
			excludeItem.setIcon(0, trashIcon)
			excludeItemList.append(excludeItem)
		self.ui.treeExclude.insertTopLevelItems(0, excludeItemList)
		
		# Load Archive Settings
		
		self.ui.txtLocalFolder.setText(folder.LocalBackupDirectory)
		self.ui.comboRemoteProtocol.setCurrentIndex(folder.RemoteProtocolInt)
		self.ui.txtRemotePath.setText(folder.RemotePath)
		self.ui.txtRemotePort.setText(str(folder.RemotePort))
		self.ui.txtRemoteHost.setText(folder.RemoteHost)
		self.ui.txtUsername.setText(folder.RemoteUsername)
		self.ui.txtPassword.setText(folder.RemotePassword)
		self.ui.checkEncryptFiles.setChecked(folder.EncryptionKeyEnabled)
		self.ui.txtEncryptionKey.setText(folder.EncryptionKey)
		self.ArchiveSettings_Update()
	
	
	def AddToExcludeList(self, e):
		"""
		Adds an excludeItem to the global exclude list.
		and show it on the exclude ui
		
		@type e: ExcludeItem Object
		@param e:			
		"""		

		excludeItem = self.getExcludeTreeItem(e)
		self.ui.treeExclude.addTopLevelItem(excludeItem)
		self.folder.ExcludeList.append(e)
	
	
	def btnExcludeAddPatern_Pressed(self):
		excludePattern = QtGui.QInputDialog.getText(self, _("Exclude Pattern"), "", QtGui.QLineEdit.Normal)
		if not excludePattern[1]:
			return
		e = self.validateExcludeItem(excludePattern, excludeitem.ExcludeItem.REGEXP)
		if not e == None:
			self.AddToExcludeList(e)	
	
	def btnExcludeAddFolder_Pressed(self):
		mountpoint = ""
		if self.folder.LocationType == self.folder.LOCATION_LOCAL:
			excludePattern = QtGui.QFileDialog.getExistingDirectory(self, "", self.folder.FolderName)
		elif self.folder.LocationType == self.folder.LOCATION_REMOTE:
			#mount
			mountpoint = utils.mount_remote(self.folder)
			excludePattern = QtGui.QFileDialog.getExistingDirectory(self, "", mountpoint)
			#umount
			utils.umount_remote(self.folder)

		if excludePattern != "":
			excludePattern = re.sub ( mountpoint, '', str(excludePattern))
			e = excludeitem.ExcludeItem(excludePattern, excludeitem.ExcludeItem.FOLDER)
			self.AddToExcludeList(e)
	
	
	def btnExcludeAddFile_Pressed(self):
		if self.folder.LocationType == self.folder.LOCATION_LOCAL:
			excludePatterns = QtGui.QFileDialog.getOpenFileNames(self, "", self.folder.FolderName)
		elif self.folder.LocationType == self.folder.LOCATION_REMOTE:
			#mount
			mountpoint = utils.mount_remote(self.folder)
			excludePatterns = QtGui.QFileDialog.getOpenFileNames(self, "", mountpoint)
			#umount
			utils.umount_remote(self.folder)		
		
		if excludePatterns != []:
			for excludePattern in excludePatterns:
				excludePattern = re.sub ( mountpoint, '', str(excludePattern))
				e = excludeitem.ExcludeItem(excludePattern, excludeitem.ExcludeItem.FILE)
				self.AddToExcludeList(e)
	
	
	def btnExcludeRemove_Pressed(self):
		removePattern = self.ui.treeExclude.currentItem()
		curIndex = self.ui.treeExclude.indexOfTopLevelItem(removePattern)
	
		if removePattern != None:
			for e in self.folder.ExcludeList:
				if removePattern.text(0) == e.Item:
					self.folder.ExcludeList.remove(e)
					continue
			
			toDelete = self.ui.treeExclude.takeTopLevelItem(curIndex)
			del curIndex, removePattern, toDelete
	
	
	def btnOk_Pressed(self):
		self.UpdateFolderSettings()
		self.close()
	
	
	def btnCancel_Pressed(self):
		self.close()
	
	
	def closeEvent(self, event):
		self.folder = None
		self.ui.lblTstResult.setVisible(False)
		self.ui.btnDeleteSnapshot.setEnabled(False)
		self.modelSnapshotsList.clear()
		self.setupSnapshotsList(self.modelSnapshotsList)
		self.setupSnapshotsList_SetViewParameters()
	
	
	def UpdateFolderSettings(self):
		"""Updates the folder specific advanced settings."""
		
		btype = self.folder.BackupType
		
		if self.ui.radioGeneralSettings.isChecked():
			self.folder.BackupType = self.folder.GENERALSETTINGS
		elif self.ui.radioLocalBackup.isChecked():
			self.folder.BackupType = self.folder.LOCAL
		elif self.ui.radioRemoteBackup.isChecked():
			self.folder.BackupType = self.folder.REMOTE
		
		self.folder.LocalBackupDirectory = self.ui.txtLocalFolder.text()
		self.folder.RemoteProtocolInt = self.ui.comboRemoteProtocol.currentIndex()
		self.folder.RemoteProtocol = self.ui.comboRemoteProtocol.currentText()
		self.folder.RemoteUsername = self.ui.txtUsername.text()
		self.folder.RemotePassword = self.ui.txtPassword.text()
		self.folder.RemotePath = self.ui.txtRemotePath.text()
		if self.ui.txtRemotePort.text() == "":		
			self.folder.RemotePort = 0
		else:
			self.folder.RemotePort = int(self.ui.txtRemotePort.text())
		self.folder.RemoteHost = self.ui.txtRemoteHost.text()
		self.folder.EncryptionKeyEnabled = self.ui.checkEncryptFiles.isChecked()
		self.folder.EncryptionKey = self.ui.txtEncryptionKey.text()
		
		if btype != self.folder.BackupType:
			self.parent().loadIncludeList()
	
	
	def buttonSelectLocalFolder_Pressed(self):
		folderName = QtGui.QFileDialog.getExistingDirectory(self)
		
		if folderName != "":
			self.ui.txtLocalFolder.setText(folderName)
	
	
	def buttonTestConnection_Pressed(self):
		archiveProtocol = self.ui.comboRemoteProtocol.currentText()
		archiveHost = self.ui.txtRemoteHost.text()
		archivePort = self.ui.txtRemotePort.text()
		archiveUsername = self.ui.txtUsername.text()
		archivePassword = self.ui.txtPassword.text()
		archiveFolder = self.ui.txtRemotePath.text()
		self.ui.buttonTestConnection.setEnabled(False)
		self.doTestConnection(archiveProtocol, archiveUsername, archivePassword, archiveHost, archivePort, archiveFolder)
	
	
	def ShowRemoteResults(self, tstResult):
		"""
		Updates the settings view based on the results of the connection test.
		
		@type tstResult: Boolean
		@param tstResult: result from the test connection url check
		"""
		self.ui.lblTstResult.setVisible(True)
		
		if tstResult == True:
			imageSuccess = QtGui.QPixmap(':/SVG/Resources/Dialog-Ok.svg')
			self.ui.lblTstResult.setPixmap(imageSuccess)
		elif tstResult == False:
			imageFailure = QtGui.QPixmap(':/SVG/Resources/Dialog-Cancel.svg')
			self.ui.lblTstResult.setPixmap(imageFailure)
		
		# Re-Enable buttonTestConnection
		self.ArchiveSettings_Update()
	
	
	def ArchiveSettings_Update(self):
		"""Method to enable/disable Archive Settings based on the current radio
		option selected.  When using global settings, local and remote should be
		disabled.  Likewise for when local or remote is enabled."""
		
		def LocalSettings(option):
			self.ui.txtLocalFolder.setEnabled(option)
			self.ui.buttonSelectLocalFolder.setEnabled(option)
		
		def RemoteSettings(option):
			self.ui.lblRemoteServer.setEnabled(option)
			self.ui.comboRemoteProtocol.setEnabled(option)
			self.ui.txtRemoteHost.setEnabled(option)
			self.ui.lblRemotePort.setEnabled(option)
			self.ui.txtRemotePort.setEnabled(option)
			self.ui.lblRemotePath.setEnabled(option)
			self.ui.txtRemotePath.setEnabled(option)
			self.ui.buttonTestConnection.setEnabled(option)
			self.ui.lblUsername.setEnabled(option)
			self.ui.txtUsername.setEnabled(option)
			self.ui.lblPassword.setEnabled(option)
			self.ui.txtPassword.setEnabled(option)
			if self.ui.txtRemotePort.text() == "" or int(self.ui.txtRemotePort.text()) == 0:
				self.ui.txtRemotePort.setText(str(utils.getPortProtocol(self.ui.comboRemoteProtocol.currentText())))
		
		if self.ui.radioGeneralSettings.isChecked():
			LocalSettings(False)
			RemoteSettings(False)
		
		if self.ui.radioLocalBackup.isChecked():
			LocalSettings(True)
			RemoteSettings(False)
		
		if self.ui.radioRemoteBackup.isChecked():
			LocalSettings(False)
			RemoteSettings(True)
		
		self.checkEncryptFiles_Toggled()
	
	
	def checkEncryptFiles_Toggled(self):
		"""Enable/Disable the encryption options."""
		
		if self.ui.checkEncryptFiles.isChecked():
			self.ui.txtEncryptionKey.setEnabled(True)
		else:
			self.ui.txtEncryptionKey.setEnabled(False)
	
	
	def LoadFolderSnapshots(self):
		self.UpdateFolderSettings()
		settings = self.parent().backupsettings
		self.LoadSnapshots.FolderList(settings, self.folder)
		
		# Reset the Model and Ui Elements
		self.manageBackups_ResetModel()
		self.ui.btnDeleteSnapshot.setEnabled(False)
	
	
	def LoadFolderSnapshots_SetModelList(self, transferModel):
		"""Transfers the snapshots to the Snapshot model."""
		
		receiveModel = self.modelSnapshotsList
		iconFullBackup = QtGui.QIcon(QtGui.QPixmap(":/PNG/Resources/Full-Backup-Emblem.png"))
		currentFullBackup = receiveModel.invisibleRootItem()
		i = 0
		
		while transferModel.rowCount() > 0:
			backupType = transferModel.data(transferModel.index(0, 3)).toString()
			moveRow = transferModel.takeRow(0)
			
			if backupType == "full":
				receiveModel.appendRow(moveRow)
				currentFullBackup = receiveModel.itemFromIndex(receiveModel.index(i, 0))
				currentFullBackup.setIcon(iconFullBackup)
				i = i+1
				
			elif backupType == "inc":
				currentFullBackup.appendRow(moveRow)
		
		self.ui.treeSnapshotHistory.setColumnHidden(1, True)
		self.ui.treeSnapshotHistory.setColumnHidden(2, True)
	
	
	def LoadFolderSnapshots_ToggleControls(self, option):
		self.ui.btnLoadHistory.setEnabled(option)
		self.ui.btnRunAutoRemove.setEnabled(option)
		self.ui.btnCleanupBackup.setEnabled(option)
	
	
	def setupSnapshotsList(self, currentModel = None):
		"""Sets the Horizontal Headers of the snapshotsModel, or creates
		a new one as necessary."""
		if currentModel == None:
			snapshotsList = QtGui.QStandardItemModel()
		else:
			snapshotsList = currentModel
		
		snapshotsList.setHorizontalHeaderItem(0, QtGui.QStandardItem(_("Date")))
		snapshotsList.setHorizontalHeaderItem(1, QtGui.QStandardItem(_("Timestamp")))
		snapshotsList.setHorizontalHeaderItem(2, QtGui.QStandardItem(_("Int")))
		snapshotsList.setHorizontalHeaderItem(3, QtGui.QStandardItem(_("Backup Type")))
		
		return snapshotsList
	
	
	def setupSnapshotsList_SetViewParameters(self):
		"""Hides the appropriate snapshotsModel Columns and sets the width
		of the others."""
		self.ui.treeSnapshotHistory.setColumnHidden(1, True)
		self.ui.treeSnapshotHistory.setColumnHidden(2, True)
		self.ui.treeSnapshotHistory.setColumnWidth(0, 350)
	
	
	def manageBackups_DeleteSet(self):
		"""Deletes the currently deleted snapshot."""
		curSelection = self.ui.treeSnapshotHistory.currentIndex()
		selectedBackup = self.modelSnapshotsList.data(curSelection.sibling(curSelection.row(), 1)).toString()
		self.ManageBackups.DeleteBackupSet(selectedBackup, self.folder)
		self.ui.btnDeleteSnapshot.setEnabled(False)
		self.parent().parent().ui.windowRestore.changed = True
	
	
	def manageBackups_Cleanup(self):
		"""Runs the duplicity cleanup operation on the specified folder."""
		self.ManageBackups.CleanupBackup(self.folder)
		self.ui.btnCleanupBackup.setEnabled(False)
		self.parent().parent().ui.windowRestore.changed = True
	
	
	def manageBackups_ResetModel(self):
		self.modelSnapshotsList.clear()
		self.setupSnapshotsList(self.modelSnapshotsList)
		self.setupSnapshotsList_SetViewParameters()
	
	
	def manageBackups_RunAutoRemove(self):
		curSelection = self.ui.treeSnapshotHistory.currentIndex()
		selectedBackup = self.modelSnapshotsList.data(curSelection.sibling(curSelection.row(), 1)).toString()
		print "manageBackups_RunAutoRemove", ":", selectedBackup
		utils.manage_snapshots.CombineBackups(selectedBackup, self.folder)
	
	
	def ManageSnapshots_Update(self, currentIndex = None):
		"""Checks to see if a valid backup set is selected, if so, the 
		delete button is enabled.  If not, it is disabled."""
		
		try:
			currentItem = self.modelSnapshotsList.itemFromIndex(currentIndex)
			if currentItem != None:
				if (currentItem.index().isValid()) & (currentItem.parent() == None):
					self.ui.btnDeleteSnapshot.setEnabled(True)
				else:
					self.ui.btnDeleteSnapshot.setEnabled(False)
		except:
			self.ui.btnDeleteSnapshot.setEnabled(False)

	def comboRemoteProtocol_currentIndexChanged(self, index):
		self.ui.txtRemotePort.setText(str(utils.getPortProtocol(self.ui.comboRemoteProtocol.currentText())))
		

	def tabWidget_currentChanged(self, index):
		if self.ui.radioRemoteBackup.isChecked():
			gui_utils.checkRemoteSettings(self, str(self.ui.txtRemoteHost.text()))
