/*
    Copyright (C) 2013 Nicola L.C. Talbot
    www.dickimaw-books.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package com.dickimawbooks.texparserlib.latex.stix;

import java.io.IOException;

import com.dickimawbooks.texparserlib.*;
import com.dickimawbooks.texparserlib.latex.*;
import com.dickimawbooks.texparserlib.primitives.Undefined;
import com.dickimawbooks.texparserlib.generic.*;

public class StixSty extends LaTeXSty
{
   public StixSty(KeyValList options, LaTeXParserListener listener, 
     boolean loadParentOptions)
    throws IOException
   {
      super(options, "stix", listener, loadParentOptions);
   }

   public void addDefinitions()
   {
      LaTeXParserListener listener = getListener();

      for (int i = 0; i < GREEK.length; i++)
      {
         registerControlSequence(listener.createGreekSymbol(
           (String)GREEK[i][0], 
           ((Integer)GREEK[i][1]).intValue()));
      }

      for (int i = 0; i < ALPHABETICS.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)ALPHABETICS[i][0], 
           ((Integer)ALPHABETICS[i][1]).intValue()));
      }

      for (int i = 0; i < ORDINARY.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)ORDINARY[i][0], 
           ((Integer)ORDINARY[i][1]).intValue()));
      }

      for (int i = 0; i < BINARY.length; i++)
      {
         registerControlSequence(listener.createBinarySymbol(
           (String)BINARY[i][0], 
           ((Integer)BINARY[i][1]).intValue()));
      }

      for (int i = 0; i < PUNCTUATION.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)PUNCTUATION[i][0], 
           ((Integer)PUNCTUATION[i][1]).intValue()));
      }

      for (int i = 0; i < SMALL_INTEGRALS.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)SMALL_INTEGRALS[i][0], 
           ((Integer)SMALL_INTEGRALS[i][1]).intValue()));
      }

      for (int i = 0; i < INTEGRALS.length; i++)
      {
         registerControlSequence(listener.createBigOperator(
           (String)INTEGRALS[i][0], 
           ((Integer)INTEGRALS[i][1]).intValue()));
      }

      for (int i = 0; i < BIG_OPERATORS.length; i++)
      {
         registerControlSequence(listener.createBigOperator(
           (String)BIG_OPERATORS[i][0], 
           ((Integer)BIG_OPERATORS[i][1]).intValue()));
      }

      for (int i = 0; i < DELIMITERS.length; i++)
      {
         registerControlSequence(listener.createBigOperator(
           (String)DELIMITERS[i][0], 
           ((Integer)DELIMITERS[i][1]).intValue()));
      }

      for (int i = 0; i < OTHER_BRACES.length; i++)
      {
         registerControlSequence(listener.createMathSymbol(
           (String)OTHER_BRACES[i][0], 
           ((Integer)OTHER_BRACES[i][1]).intValue()));
      }

      for (int i = 0; i < ACCENTS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)ACCENTS[i][0], ((Integer)ACCENTS[i][1]).intValue()));
      }

      for (int i = 0; i < WIDE_ACCENTS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)WIDE_ACCENTS[i][0],
           ((Integer)WIDE_ACCENTS[i][1]).intValue()));
      }

      for (int i = 0; i < WIDE_BRACKETS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)WIDE_BRACKETS[i][0],
           ((Integer)WIDE_BRACKETS[i][1]).intValue()));
      }

      for (int i = 0; i < RADICALS.length; i++)
      {
         registerControlSequence(new MathAccent(
           (String)RADICALS[i][0],
           ((Integer)RADICALS[i][1]).intValue()));
      }
   }

   private static final Object[][] GREEK = new Object[][]
   {
      // Greek lower case
      new Object[]{"alpha", new Integer(0x1D6FC)},
      new Object[]{"beta", new Integer(0x1D6FD)},
      new Object[]{"gamma", new Integer(0x1D6FE)},
      new Object[]{"delta", new Integer(0x1D6FF)},
      new Object[]{"varepsilon", new Integer(0x1D700)},
      new Object[]{"zeta", new Integer(0x1D701)},
      new Object[]{"eta", new Integer(0x1D702)},
      new Object[]{"theta", new Integer(0x1D703)},
      new Object[]{"iota", new Integer(0x1D704)},
      new Object[]{"kappa", new Integer(0x1D705)},
      new Object[]{"lambda", new Integer(0x1D706)},
      new Object[]{"mu", new Integer(0x1D707)},
      new Object[]{"nu", new Integer(0x1D708)},
      new Object[]{"xi", new Integer(0x1D709)},
      new Object[]{"omicron", new Integer(0x1D70A)},
      new Object[]{"pi", new Integer(0x1D70B)},
      new Object[]{"rho", new Integer(0x1D70C)},
      new Object[]{"varsigma", new Integer(0x1D70D)},
      new Object[]{"sigma", new Integer(0x1D70E)},
      new Object[]{"tau", new Integer(0x1D70F)},
      new Object[]{"upsilon", new Integer(0x1D710)},
      new Object[]{"varphi", new Integer(0x1D711)},
      new Object[]{"chi", new Integer(0x1D712)},
      new Object[]{"psi", new Integer(0x1D713)},
      new Object[]{"omega", new Integer(0x1D714)},
      new Object[]{"epsilon", new Integer(0x1D716)},
      new Object[]{"vartheta", new Integer(0x1D717)},
      new Object[]{"varkappa", new Integer(0x1D718)},
      new Object[]{"phi", new Integer(0x1D719)},
      new Object[]{"varrho", new Integer(0x1D71A)},
      new Object[]{"varpi", new Integer(0x1D71B)},
      // Greek upper case
      new Object[]{"Alpha", new Integer(0x1D6E2)},
      new Object[]{"Beta", new Integer(0x1D6E3)},
      new Object[]{"Gamma", new Integer(0x1D6E4)},
      new Object[]{"Delta", new Integer(0x1D6E5)},
      new Object[]{"Epsilon", new Integer(0x1D6E6)},
      new Object[]{"Zeta", new Integer(0x1D6E7)},
      new Object[]{"Eta", new Integer(0x1D6E8)},
      new Object[]{"Theta", new Integer(0x1D6E9)},
      new Object[]{"Iota", new Integer(0x1D6EA)},
      new Object[]{"Kappa", new Integer(0x1D6EB)},
      new Object[]{"Lambda", new Integer(0x1D6EC)},
      new Object[]{"Mu", new Integer(0x1D6ED)},
      new Object[]{"Nu", new Integer(0x1D6EE)},
      new Object[]{"Xi", new Integer(0x1D6EF)},
      new Object[]{"Omicron", new Integer(0x1D6F0)},
      new Object[]{"Pi", new Integer(0x1D6F1)},
      new Object[]{"Rho", new Integer(0x1D6F2)},
      new Object[]{"Theta", new Integer(0x1D6F3)},
      new Object[]{"Sigma", new Integer(0x1D6F4)},
      new Object[]{"Tau", new Integer(0x1D6F5)},
      new Object[]{"Upsilon", new Integer(0x1D6F6)},
      new Object[]{"Phi", new Integer(0x1D6F7)},
      new Object[]{"Chi", new Integer(0x1D6F8)},
      new Object[]{"Psi", new Integer(0x1D6F9)},
      new Object[]{"Omega", new Integer(0x1D6FA)},
      new Object[]{"nabla", new Integer(0x1D6FB)},
   };

   private static final Object[][] ALPHABETICS = new Object[][]
   {
      // other alphabetics
      new Object[]{"nabla", new Integer(0x2207)},
      new Object[]{"partial", new Integer(0x2202)},
      new Object[]{"imath", new Integer(0x1D6A4)},
      new Object[]{"jmath", new Integer(0x1D6A5)},
   };

   private static final Object[][] ORDINARY = new Object[][]
   {
      // ordinary symbols
      new Object[]{"mathdollar", new Integer(0x0024)},
      new Object[]{"mathsterling", new Integer(0x00A3)},
      new Object[]{"mathsection", new Integer(0x00A7)},
      new Object[]{"neg", new Integer(0x00AC)},
      new Object[]{"lnot", new Integer(0x00AC)},
      new Object[]{"mathparagraph", new Integer(0x00B6)},
      new Object[]{"eth", new Integer(0x00F0)},
      new Object[]{"Zbar", new Integer(0x01B5)},
      new Object[]{"digamma", new Integer(0x03DD)},
      new Object[]{"backepsilon", new Integer(0x03F6)},
      new Object[]{"upbackepsilon", new Integer(0x03F6)},
      new Object[]{"enleadertwodots", new Integer(0x2025)},
      new Object[]{"mathellipsis", new Integer(0x2026)},
      new Object[]{"prime", new Integer(0x2032)},
      new Object[]{"dprime", new Integer(0x2033)},
      new Object[]{"trprime", new Integer(0x2034)},
      new Object[]{"backprime", new Integer(0x2035)},
      new Object[]{"backdprime", new Integer(0x2036)},
      new Object[]{"backtrprime", new Integer(0x2037)},
      new Object[]{"caretinsert", new Integer(0x2038)},
      new Object[]{"Exclam", new Integer(0x203C)},
      new Object[]{"hyphenbullet", new Integer(0x2043)},
      new Object[]{"Question", new Integer(0x2047)},
      new Object[]{"qprime", new Integer(0x2057)},
      new Object[]{"enclosecircle", new Integer(0x20DD)},
      new Object[]{"enclosesquare", new Integer(0x20DE)},
      new Object[]{"enclosediamond", new Integer(0x20DF)},
      new Object[]{"enclosetriangle", new Integer(0x20E4)},
      new Object[]{"Eulerconst", new Integer(0x2107)},
      new Object[]{"hbar", new Integer(0x210F)},
      new Object[]{"hslash", new Integer(0x210F)},
      new Object[]{"Im", new Integer(0x2111)},
      new Object[]{"ell", new Integer(0x2113)},
      new Object[]{"wp", new Integer(0x2118)},
      new Object[]{"Re", new Integer(0x211C)},
      new Object[]{"mho", new Integer(0x2127)},
      new Object[]{"turnediota", new Integer(0x2129)},
      new Object[]{"Angstrom", new Integer(0x212B)},
      new Object[]{"Finv", new Integer(0x2132)},
      new Object[]{"aleph", new Integer(0x2135)},
      new Object[]{"beth", new Integer(0x2136)},
      new Object[]{"gimel", new Integer(0x2137)},
      new Object[]{"daleth", new Integer(0x2138)},
      new Object[]{"Game", new Integer(0x2141)},
      new Object[]{"sansLturned", new Integer(0x2142)},
      new Object[]{"sansLmirroed", new Integer(0x2143)},
      new Object[]{"Yup", new Integer(0x2144)},
      new Object[]{"PropertyLine", new Integer(0x214A)},
      new Object[]{"updownarrowbar", new Integer(0x21A8)},
      new Object[]{"linefeed", new Integer(0x21B4)},
      new Object[]{"carriagereturn", new Integer(0x21B5)},
      new Object[]{"barovernorthwestarrow", new Integer(0x21B8)},
      new Object[]{"barleftarrowrightarrowbar", new Integer(0x21B9)},
      new Object[]{"acwopencirclearrow", new Integer(0x21BA)},
      new Object[]{"cwopencirclearrow", new Integer(0x21BB)},
      new Object[]{"nHuparrow", new Integer(0x21DE)},
      new Object[]{"nHdownarrow", new Integer(0x21DF)},
      new Object[]{"leftdasharrow", new Integer(0x21E0)},
      new Object[]{"updasharrow", new Integer(0x21E1)},
      new Object[]{"rightdasharrow", new Integer(0x21E2)},
      new Object[]{"downdasharrow", new Integer(0x21E3)},
      new Object[]{"leftwhitearrow", new Integer(0x21E6)},
      new Object[]{"upwhitearrow", new Integer(0x21E7)},
      new Object[]{"rightwhitearrow", new Integer(0x21E8)},
      new Object[]{"downwhitearrow", new Integer(0x21E9)},
      new Object[]{"whitearrowupfrombar", new Integer(0x21EA)},
      new Object[]{"forall", new Integer(0x2200)},
      new Object[]{"complement", new Integer(0x2201)},
      new Object[]{"exists", new Integer(0x2203)},
      new Object[]{"nexists", new Integer(0x2204)},
      new Object[]{"varnothing", new Integer(0x2205)},
      new Object[]{"emptyset", new Integer(0x2205)},
      new Object[]{"increment", new Integer(0x2206)},
      new Object[]{"QED", new Integer(0x220E)},
      new Object[]{"infty", new Integer(0x221E)},
      new Object[]{"rightangle", new Integer(0x221F)},
      new Object[]{"angle", new Integer(0x2220)},
      new Object[]{"measuredangle", new Integer(0x2221)},
      new Object[]{"sphericalangle", new Integer(0x2222)},
      new Object[]{"therefore", new Integer(0x2234)},
      new Object[]{"because", new Integer(0x2235)},
      new Object[]{"sinewave", new Integer(0x223F)},
      new Object[]{"top", new Integer(0x22A4)},
      new Object[]{"bot", new Integer(0x22A5)},
      new Object[]{"hermitmatrix", new Integer(0x22B9)},
      new Object[]{"measuredrightangle", new Integer(0x22BE)},
      new Object[]{"varlrtriangle", new Integer(0x22BF)},
      new Object[]{"cdots", new Integer(0x22EF)},
      new Object[]{"diameter", new Integer(0x2300)},
      new Object[]{"house", new Integer(0x2302)},
      new Object[]{"invnot", new Integer(0x2310)},
      new Object[]{"sqlozenge", new Integer(0x2311)},
      new Object[]{"profline", new Integer(0x2312)},
      new Object[]{"profsurf", new Integer(0x2313)},
      new Object[]{"viewdata", new Integer(0x2317)},
      new Object[]{"turnednot", new Integer(0x2319)},
      new Object[]{"varhexagonlrbonds", new Integer(0x232C)},
      new Object[]{"conictaper", new Integer(0x2332)},
      new Object[]{"topbot", new Integer(0x2336)},
      new Object[]{"APLnotbackslash", new Integer(0x2340)},
      new Object[]{"APLboxquestion", new Integer(0x2370)},
      new Object[]{"rangledownzigzagarrow", new Integer(0x237C)},
      new Object[]{"hexagon", new Integer(0x2394)},
      new Object[]{"bbrktbrk", new Integer(0x23B6)},
      new Object[]{"varcarriagereturn", new Integer(0x23CE)},
      new Object[]{"obrbrak", new Integer(0x23E0)},
      new Object[]{"ubrbrak", new Integer(0x23E1)},
      new Object[]{"trapezium", new Integer(0x23E2)},
      new Object[]{"benzenr", new Integer(0x23E3)},
      new Object[]{"strns", new Integer(0x23E4)},
      new Object[]{"fltns", new Integer(0x23E5)},
      new Object[]{"accurrent", new Integer(0x23E6)},
      new Object[]{"elinters", new Integer(0x23E7)},
      new Object[]{"mathvisiblespace", new Integer(0x2423)},
      new Object[]{"circledR", new Integer(0x24C7)},
      new Object[]{"circledS", new Integer(0x24C8)},
      new Object[]{"mdlgblksquare", new Integer(0x25A0)},
      new Object[]{"mdlgwhtsquare", new Integer(0x25A1)},
      new Object[]{"square", new Integer(0x25A1)},
      new Object[]{"Box", new Integer(0x25A1)},
      new Object[]{"squoval", new Integer(0x25A2)},
      new Object[]{"blackinwhitesquare", new Integer(0x25A3)},
      new Object[]{"squarehfill", new Integer(0x25A4)},
      new Object[]{"squarevfill", new Integer(0x25A5)},
      new Object[]{"squarehvfill", new Integer(0x25A6)},
      new Object[]{"squarenwsefill", new Integer(0x25A7)},
      new Object[]{"squareneswfill", new Integer(0x25A8)},
      new Object[]{"squarecrossfill", new Integer(0x25A9)},
      new Object[]{"smblksquare", new Integer(0x25AA)},
      new Object[]{"smwhtsquare", new Integer(0x25AB)},
      new Object[]{"hrectangleblack", new Integer(0x25AC)},
      new Object[]{"hrectangle", new Integer(0x25AD)},
      new Object[]{"vrectangleblack", new Integer(0x25AE)},
      new Object[]{"vrectangle", new Integer(0x25AF)},
      new Object[]{"parallelogramblack", new Integer(0x25B0)},
      new Object[]{"parallelogram", new Integer(0x25B1)},
      new Object[]{"bigblacktriangleup", new Integer(0x25B2)},
      new Object[]{"blacktriangle", new Integer(0x25B4)},
      new Object[]{"blacktriangleright", new Integer(0x25B6)},
      new Object[]{"smallblacktriangleright", new Integer(0x25B8)},
      new Object[]{"smalltriangleright", new Integer(0x25B9)},
      new Object[]{"blackpointerright", new Integer(0x25BA)},
      new Object[]{"whitepointerright", new Integer(0x25BB)},
      new Object[]{"bigblacktriangledown", new Integer(0x25BC)},
      new Object[]{"bigtriangledown", new Integer(0x25BD)},
      new Object[]{"blacktriangledown", new Integer(0x25BE)},
      new Object[]{"triangledown", new Integer(0x25BF)},
      new Object[]{"blacktriangleleft", new Integer(0x25C0)},
      new Object[]{"smallblacktriangleleft", new Integer(0x25C2)},
      new Object[]{"smalltriangleleft", new Integer(0x25C3)},
      new Object[]{"blackpointerleft", new Integer(0x25C4)},
      new Object[]{"whitepointerleft", new Integer(0x25C5)},
      new Object[]{"mdlgblkdiamond", new Integer(0x25C6)},
      new Object[]{"mdlgwhtdiamond", new Integer(0x25C7)},
      new Object[]{"blackinwhitediamond", new Integer(0x25C8)},
      new Object[]{"fisheye", new Integer(0x25C9)},
      new Object[]{"mdlgwhtlozenge", new Integer(0x25CA)},
      new Object[]{"lozenge", new Integer(0x25CA)},
      new Object[]{"Diamond", new Integer(0x25CA)},
      new Object[]{"dottedcircle", new Integer(0x25CC)},
      new Object[]{"circlevertfill", new Integer(0x25CD)},
      new Object[]{"bullseye", new Integer(0x25CE)},
      new Object[]{"mdlgblkcircle", new Integer(0x25CF)},
      new Object[]{"circlelefthalfblack", new Integer(0x25D0)},
      new Object[]{"circlerighthalfblack", new Integer(0x25D1)},
      new Object[]{"circlebottomhalfblack", new Integer(0x25D2)},
      new Object[]{"circletophalfblack", new Integer(0x25D3)},
      new Object[]{"circleurquadblack", new Integer(0x25D4)},
      new Object[]{"blackcircleulquadwhite", new Integer(0x25D5)},
      new Object[]{"blacklefthalfcircle", new Integer(0x25D6)},
      new Object[]{"blackrighthalfcircle", new Integer(0x25D7)},
      new Object[]{"inversebullet", new Integer(0x25D8)},
      new Object[]{"inversewhitecircle", new Integer(0x25D9)},
      new Object[]{"invwhiteupperhalfcircle", new Integer(0x25DA)},
      new Object[]{"invwhitelowerhalfcircle", new Integer(0x25DB)},
      new Object[]{"ularc", new Integer(0x25DC)},
      new Object[]{"urarc", new Integer(0x25DD)},
      new Object[]{"lrarc", new Integer(0x25DE)},
      new Object[]{"llarc", new Integer(0x25DF)},
      new Object[]{"topsemicircle", new Integer(0x25E0)},
      new Object[]{"botsemicircle", new Integer(0x25E1)},
      new Object[]{"lrblacktriangle", new Integer(0x25E2)},
      new Object[]{"llblacktriangle", new Integer(0x25E3)},
      new Object[]{"ulblacktriangle", new Integer(0x25E4)},
      new Object[]{"urblacktriangle", new Integer(0x25E5)},
      new Object[]{"circ", new Integer(0x25E6)},
      new Object[]{"smwhtcircle", new Integer(0x25E6)},
      new Object[]{"squareleftblack", new Integer(0x25E7)},
      new Object[]{"squarerightblack", new Integer(0x25E8)},
      new Object[]{"squareulblack", new Integer(0x25E9)},
      new Object[]{"squarelrblack", new Integer(0x25EA)},
      new Object[]{"trianglecdot", new Integer(0x25EC)},
      new Object[]{"triangleleftblack", new Integer(0x25ED)},
      new Object[]{"trianglerightblack", new Integer(0x25EE)},
      new Object[]{"lgwhtcircle", new Integer(0x25EF)},
      new Object[]{"squareulquad", new Integer(0x25F0)},
      new Object[]{"squarellquad", new Integer(0x25F1)},
      new Object[]{"squarelrquad", new Integer(0x25F2)},
      new Object[]{"squareurquad", new Integer(0x25F3)},
      new Object[]{"circleulquad", new Integer(0x25F4)},
      new Object[]{"circlellquad", new Integer(0x25F5)},
      new Object[]{"circlelrquad", new Integer(0x25F6)},
      new Object[]{"circleurquad", new Integer(0x25F7)},
      new Object[]{"ultriangle", new Integer(0x25F8)},
      new Object[]{"urtriangle", new Integer(0x25F9)},
      new Object[]{"lltriangle", new Integer(0x25FA)},
      new Object[]{"mdwhtsquare", new Integer(0x25FB)},
      new Object[]{"mdblksquare", new Integer(0x25FC)},
      new Object[]{"mdsmwhtsquare", new Integer(0x25FD)},
      new Object[]{"mdsmblksquare", new Integer(0x25FE)},
      new Object[]{"lrtriangle", new Integer(0x25FF)},
      new Object[]{"bigstar", new Integer(0x2605)},
      new Object[]{"bigwhitestar", new Integer(0x2606)},
      new Object[]{"astrosun", new Integer(0x2609)},
      new Object[]{"danger", new Integer(0x2621)},
      new Object[]{"blacksmiley", new Integer(0x263B)},
      new Object[]{"sun", new Integer(0x263C)},
      new Object[]{"rightmoon", new Integer(0x263D)},
      new Object[]{"leftmoon", new Integer(0x263E)},
      new Object[]{"female", new Integer(0x2640)},
      new Object[]{"male", new Integer(0x2642)},
      new Object[]{"spadesuit", new Integer(0x2660)},
      new Object[]{"heartsuit", new Integer(0x2661)},
      new Object[]{"diamondsuit", new Integer(0x2662)},
      new Object[]{"clubsuit", new Integer(0x2663)},
      new Object[]{"varspadesuit", new Integer(0x2664)},
      new Object[]{"varheartsuit", new Integer(0x2665)},
      new Object[]{"vardiamondsuit", new Integer(0x2666)},
      new Object[]{"varclubsuit", new Integer(0x2667)},
      new Object[]{"quarternote", new Integer(0x2669)},
      new Object[]{"eighthnote", new Integer(0x266A)},
      new Object[]{"twonotes", new Integer(0x266B)},
      new Object[]{"flat", new Integer(0x266D)},
      new Object[]{"natural", new Integer(0x266E)},
      new Object[]{"sharp", new Integer(0x266F)},
      new Object[]{"acidfree", new Integer(0x267E)},
      new Object[]{"dicei", new Integer(0x2680)},
      new Object[]{"diceii", new Integer(0x2681)},
      new Object[]{"diceiii", new Integer(0x2682)},
      new Object[]{"diceiv", new Integer(0x2683)},
      new Object[]{"dicev", new Integer(0x2684)},
      new Object[]{"dicevi", new Integer(0x2685)},
      new Object[]{"circledrightdot", new Integer(0x2686)},
      new Object[]{"circledtwodots", new Integer(0x2687)},
      new Object[]{"blackcircledrightdot", new Integer(0x2688)},
      new Object[]{"blackcircledtwodots", new Integer(0x2689)},
      new Object[]{"Hermaphrodite", new Integer(0x26A5)},
      new Object[]{"mdwhtcircle", new Integer(0x26AA)},
      new Object[]{"mdblkcircle", new Integer(0x26AB)},
      new Object[]{"mdsmwhtcircle", new Integer(0x26AC)},
      new Object[]{"neuter", new Integer(0x26B2)},
      new Object[]{"checkmark", new Integer(0x2713)},
      new Object[]{"maltese", new Integer(0x2720)},
      new Object[]{"circledstar", new Integer(0x272A)},
      new Object[]{"varstar", new Integer(0x2736)},
      new Object[]{"dingasterisk", new Integer(0x273D)},
      new Object[]{"draftingarrow", new Integer(0x279B)},
      new Object[]{"threedangle", new Integer(0x27C0)},
      new Object[]{"whiteinwhitetriangle", new Integer(0x27C1)},
      new Object[]{"subsetcirc", new Integer(0x27C3)},
      new Object[]{"supsetcirc", new Integer(0x27C4)},
      new Object[]{"diagup", new Integer(0x27CB)},
      new Object[]{"diagdown", new Integer(0x27CD)},
      new Object[]{"diamondcdot", new Integer(0x27D0)},
      new Object[]{"rdiagovfdiag", new Integer(0x292B)},
      new Object[]{"fdiagovrdiag", new Integer(0x292C)},
      new Object[]{"seovnearrow", new Integer(0x292D)},
      new Object[]{"neovsearrow", new Integer(0x292E)},
      new Object[]{"fdiagovnearrow", new Integer(0x292F)},
      new Object[]{"rdiagovsearrow", new Integer(0x2930)},
      new Object[]{"neovnwarrow", new Integer(0x2931)},
      new Object[]{"nwovnearrow", new Integer(0x2932)},
      new Object[]{"uprightcurvearrow", new Integer(0x2934)},
      new Object[]{"downrightcurvedarrow", new Integer(0x2935)},
      new Object[]{"mdsmblkcircle", new Integer(0x2981)},
      new Object[]{"fourvdots", new Integer(0x2999)},
      new Object[]{"vzigzag", new Integer(0x299A)},
      new Object[]{"measuredangleleft", new Integer(0x299B)},
      new Object[]{"rightanglesqr", new Integer(0x299C)},
      new Object[]{"rightanglemdot", new Integer(0x299D)},
      new Object[]{"angles", new Integer(0x299E)},
      new Object[]{"angdnr", new Integer(0x299F)},
      new Object[]{"gtlpar", new Integer(0x29A0)},
      new Object[]{"sphericalangleup", new Integer(0x29A1)},
      new Object[]{"turnangle", new Integer(0x29A2)},
      new Object[]{"revangle", new Integer(0x29A3)},
      new Object[]{"angleubar", new Integer(0x29A4)},
      new Object[]{"revangleubar", new Integer(0x29A5)},
      new Object[]{"wideangledown", new Integer(0x29A6)},
      new Object[]{"wideangleup", new Integer(0x29A7)},
      new Object[]{"measanglerutone", new Integer(0x29A8)},
      new Object[]{"measanglelutonw", new Integer(0x29A9)},
      new Object[]{"measanglerdtose", new Integer(0x29AA)},
      new Object[]{"measangleldtosw", new Integer(0x29AB)},
      new Object[]{"measangleurtone", new Integer(0x29AC)},
      new Object[]{"measangleultonw", new Integer(0x29AD)},
      new Object[]{"measangledrtose", new Integer(0x29AE)},
      new Object[]{"measangledltosw", new Integer(0x29AF)},
      new Object[]{"revemptyset", new Integer(0x29B0)},
      new Object[]{"emptysetobar", new Integer(0x29B1)},
      new Object[]{"emptysetocirc", new Integer(0x29B2)},
      new Object[]{"emptysetoarr", new Integer(0x29B3)},
      new Object[]{"emptysetoarrl", new Integer(0x29B4)},
      new Object[]{"obot", new Integer(0x29BA)},
      new Object[]{"olcross", new Integer(0x29BB)},
      new Object[]{"odotslashdot", new Integer(0x29BC)},
      new Object[]{"uparrowoncircle", new Integer(0x29BD)},
      new Object[]{"circledwhitebullet", new Integer(0x29BE)},
      new Object[]{"circledbullet", new Integer(0x29BF)},
      new Object[]{"cirscir", new Integer(0x29C2)},
      new Object[]{"cirE", new Integer(0x29C3)},
      new Object[]{"boxonbox", new Integer(0x29C9)},
      new Object[]{"triangleodot", new Integer(0x29CA)},
      new Object[]{"triangleubar", new Integer(0x29CB)},
      new Object[]{"triangles", new Integer(0x29CC)},
      new Object[]{"iinfin", new Integer(0x29DC)},
      new Object[]{"tieinfty", new Integer(0x29DD)},
      new Object[]{"nvinfty", new Integer(0x29DE)},
      new Object[]{"laplac", new Integer(0x29E0)},
      new Object[]{"thermod", new Integer(0x29E7)},
      new Object[]{"downtriangleleftblack", new Integer(0x29E8)},
      new Object[]{"downtrianglerightblack", new Integer(0x29E9)},
      new Object[]{"blackdiamonddownarrow", new Integer(0x29EA)},
      new Object[]{"blacklozenge", new Integer(0x29EB)},
      new Object[]{"circledownarrow", new Integer(0x29EC)},
      new Object[]{"blackcircledownarrow", new Integer(0x29ED)},
      new Object[]{"errbarsquare", new Integer(0x29EE)},
      new Object[]{"errbarblacksquare", new Integer(0x29EF)},
      new Object[]{"errbardiamond", new Integer(0x29F0)},
      new Object[]{"errbarblackdiamond", new Integer(0x29F1)},
      new Object[]{"errbarcircle", new Integer(0x29F2)},
      new Object[]{"errbarblackcircle", new Integer(0x29F3)},
      new Object[]{"perps", new Integer(0x2AE1)},
      new Object[]{"topcir", new Integer(0x2AF1)},
      new Object[]{"squaretopblack", new Integer(0x2B12)},
      new Object[]{"squarebotblack", new Integer(0x2B13)},
      new Object[]{"squareurblack", new Integer(0x2B14)},
      new Object[]{"squarellblack", new Integer(0x2B15)},
      new Object[]{"diamondleftblack", new Integer(0x2B16)},
      new Object[]{"diamondrightblack", new Integer(0x2B17)},
      new Object[]{"diamondtopblack", new Integer(0x2B18)},
      new Object[]{"diamondbotblack", new Integer(0x2B19)},
      new Object[]{"dottedsquare", new Integer(0x2B1A)},
      new Object[]{"lgblksquare", new Integer(0x2B1B)},
      new Object[]{"lgwhtsquare", new Integer(0x2B1C)},
      new Object[]{"vysmblksquare", new Integer(0x2B1D)},
      new Object[]{"vysmwhtsquare", new Integer(0x2B1E)},
      new Object[]{"pentagonblack", new Integer(0x2B1F)},
      new Object[]{"pentagon", new Integer(0x2B20)},
      new Object[]{"varhexagon", new Integer(0x2B21)},
      new Object[]{"varhexagonblack", new Integer(0x2B22)},
      new Object[]{"hexagonblack", new Integer(0x2B23)},
      new Object[]{"lgblkcircle", new Integer(0x2B24)},
      new Object[]{"mdblkdiamond", new Integer(0x2B25)},
      new Object[]{"mdwhtdiamond", new Integer(0x2B26)},
      new Object[]{"mdblklozenge", new Integer(0x2B27)},
      new Object[]{"mdwhtlozenge", new Integer(0x2B28)},
      new Object[]{"smblkdiamond", new Integer(0x2B29)},
      new Object[]{"smblklozenge", new Integer(0x2B2A)},
      new Object[]{"smwhtlozenge", new Integer(0x2B2B)},
      new Object[]{"blkhorzoval", new Integer(0x2B2C)},
      new Object[]{"whthorzoval", new Integer(0x2B2D)},
      new Object[]{"blkvertoval", new Integer(0x2B2E)},
      new Object[]{"whtvertoval", new Integer(0x2B2F)},
      new Object[]{"medwhitestar", new Integer(0x2B50)},
      new Object[]{"medblackstar", new Integer(0x2B51)},
      new Object[]{"smwhitestar", new Integer(0x2B52)},
      new Object[]{"rightpentagonblack", new Integer(0x2B53)},
      new Object[]{"rightpentagon", new Integer(0x2B54)},
      new Object[]{"postalmark", new Integer(0x3012)},
      new Object[]{"hzigzag", new Integer(0x3030)},
      new Object[]{"Bbbk", new Integer(0x1D55C)},
   };

   private static final Object[][] BINARY = new Object[][]
   {
      // binary operators
      new Object[]{"pm", new Integer(0x00B1)},
      new Object[]{"cdotp", new Integer(0x00B7)},
      new Object[]{"centerdot", new Integer(0x00B7)},
      new Object[]{"times", new Integer(0x00D7)},
      new Object[]{"div", new Integer(0x00F7)},
      new Object[]{"dagger", new Integer(0x2020)},
      new Object[]{"ddagger", new Integer(0x2021)},
      new Object[]{"smblkcircle", new Integer(0x2022)},
      new Object[]{"fracslash", new Integer(0x2044)},
      new Object[]{"upand", new Integer(0x214B)},
      new Object[]{"mp", new Integer(0x2213)},
      new Object[]{"dotplus", new Integer(0x2214)},
      new Object[]{"smallsetminus", new Integer(0x2216)},
      new Object[]{"ast", new Integer(0x2217)},
      new Object[]{"vysmwhtcircle", new Integer(0x2218)},
      new Object[]{"vysmblkcircle", new Integer(0x2219)},
      new Object[]{"wedge", new Integer(0x2227)},
      new Object[]{"land", new Integer(0x2227)},
      new Object[]{"vee", new Integer(0x2228)},
      new Object[]{"lor", new Integer(0x2228)},
      new Object[]{"cap", new Integer(0x2229)},
      new Object[]{"cup", new Integer(0x222A)},
      new Object[]{"dotminus", new Integer(0x2238)},
      new Object[]{"invlazys", new Integer(0x223E)},
      new Object[]{"wr", new Integer(0x2240)},
      new Object[]{"cupleftarrow", new Integer(0x228C)},
      new Object[]{"cupdot", new Integer(0x228D)},
      new Object[]{"uplus", new Integer(0x228E)},
      new Object[]{"sqcap", new Integer(0x2293)},
      new Object[]{"sqcup", new Integer(0x2294)},
      new Object[]{"oplus", new Integer(0x2295)},
      new Object[]{"ominus", new Integer(0x2296)},
      new Object[]{"otimes", new Integer(0x2297)},
      new Object[]{"oslash", new Integer(0x2298)},
      new Object[]{"odot", new Integer(0x2299)},
      new Object[]{"circledcirc", new Integer(0x229A)},
      new Object[]{"circledast", new Integer(0x229B)},
      new Object[]{"circledequal", new Integer(0x229C)},
      new Object[]{"circleddash", new Integer(0x229D)},
      new Object[]{"boxplus", new Integer(0x229E)},
      new Object[]{"boxminus", new Integer(0x229F)},
      new Object[]{"boxtimes", new Integer(0x22A0)},
      new Object[]{"boxdot", new Integer(0x22A1)},
      new Object[]{"intercal", new Integer(0x22BA)},
      new Object[]{"veebar", new Integer(0x22BB)},
      new Object[]{"barwedge", new Integer(0x22BC)},
      new Object[]{"barvee", new Integer(0x22BD)},
      new Object[]{"diamond", new Integer(0x22C4)},
      new Object[]{"smwhtdiamond", new Integer(0x22C4)},
      new Object[]{"cdot", new Integer(0x22C5)},
      new Object[]{"star", new Integer(0x22C6)},
      new Object[]{"divideontimes", new Integer(0x22C7)},
      new Object[]{"ltimes", new Integer(0x22C9)},
      new Object[]{"rtimes", new Integer(0x22CA)},
      new Object[]{"leftthreetimes", new Integer(0x22CB)},
      new Object[]{"rightthreetimes", new Integer(0x22CC)},
      new Object[]{"curlyvee", new Integer(0x22CE)},
      new Object[]{"curlywedge", new Integer(0x22CF)},
      new Object[]{"Cap", new Integer(0x22D2)},
      new Object[]{"doublecap", new Integer(0x22D2)},
      new Object[]{"Cup", new Integer(0x22D3)},
      new Object[]{"doublecup", new Integer(0x22D3)},
      new Object[]{"varbarwedge", new Integer(0x2305)},
      new Object[]{"vardoublebarwedge", new Integer(0x2306)},
      new Object[]{"obar", new Integer(0x233D)},
      new Object[]{"triangle", new Integer(0x25B3)},
      new Object[]{"bigtriangleup", new Integer(0x25B3)},
      new Object[]{"lhd", new Integer(0x22B2)},
      new Object[]{"rhd", new Integer(0x22B3)},
      new Object[]{"unlhd", new Integer(0x22B4)},
      new Object[]{"unrhd", new Integer(0x22B5)},
      new Object[]{"mdlgwhtcircle", new Integer(0x25CB)},
      new Object[]{"boxbar", new Integer(0x25EB)},
      new Object[]{"veedot", new Integer(0x27C7)},
      new Object[]{"wedgedot", new Integer(0x27D1)},
      new Object[]{"lozengeminus", new Integer(0x27E0)},
      new Object[]{"concavediamond", new Integer(0x27E1)},
      new Object[]{"concavediamondtickleft", new Integer(0x27E2)},
      new Object[]{"concavediamondtickright", new Integer(0x27E3)},
      new Object[]{"whitesquaretickleft", new Integer(0x27E4)},
      new Object[]{"whitesquaretickright", new Integer(0x27E5)},
      new Object[]{"typecolon", new Integer(0x2982)},
      new Object[]{"circlehbar", new Integer(0x29B5)},
      new Object[]{"circledvert", new Integer(0x29B6)},
      new Object[]{"circledparallel", new Integer(0x29B7)},
      new Object[]{"obslash", new Integer(0x29B8)},
      new Object[]{"operp", new Integer(0x29B9)},
      new Object[]{"olessthan", new Integer(0x29C0)},
      new Object[]{"ogreaterthan", new Integer(0x29C1)},
      new Object[]{"boxdiag", new Integer(0x29C4)},
      new Object[]{"boxbslash", new Integer(0x29C5)},
      new Object[]{"boxast", new Integer(0x29C6)},
      new Object[]{"boxcircle", new Integer(0x29C7)},
      new Object[]{"boxbox", new Integer(0x29C8)},
      new Object[]{"triangleserifs", new Integer(0x29CD)},
      new Object[]{"hourglass", new Integer(0x29D6)},
      new Object[]{"blackhourglass", new Integer(0x29D7)},
      new Object[]{"shuffle", new Integer(0x29E2)},
      new Object[]{"mdlgblklozenge", new Integer(0x29EB)},
      new Object[]{"setminus", new Integer(0x29F5)},
      new Object[]{"dsol", new Integer(0x29F6)},
      new Object[]{"rsolbar", new Integer(0x29F7)},
      new Object[]{"doubleplus", new Integer(0x29FA)},
      new Object[]{"tripleplus", new Integer(0x29FB)},
      new Object[]{"tplus", new Integer(0x29FE)},
      new Object[]{"tminus", new Integer(0x29FF)},
      new Object[]{"ringplus", new Integer(0x2A22)},
      new Object[]{"plushat", new Integer(0x2A23)},
      new Object[]{"simplus", new Integer(0x2A24)},
      new Object[]{"plusdot", new Integer(0x2A25)},
      new Object[]{"plussim", new Integer(0x2A26)},
      new Object[]{"plussubtwo", new Integer(0x2A27)},
      new Object[]{"plustrif", new Integer(0x2A28)},
      new Object[]{"commaminus", new Integer(0x2A29)},
      new Object[]{"minusdot", new Integer(0x2A2A)},
      new Object[]{"minusfdots", new Integer(0x2A2B)},
      new Object[]{"minusrdots", new Integer(0x2A2C)},
      new Object[]{"opluslhrim", new Integer(0x2A2D)},
      new Object[]{"oplusrhrim", new Integer(0x2A2E)},
      new Object[]{"vectimes", new Integer(0x2A2F)},
      new Object[]{"dottimes", new Integer(0x2A30)},
      new Object[]{"timesbar", new Integer(0x2A31)},
      new Object[]{"btimes", new Integer(0x2A32)},
      new Object[]{"smashtimes", new Integer(0x2A33)},
      new Object[]{"otimeslhrim", new Integer(0x2A34)},
      new Object[]{"otimesrhrim", new Integer(0x2A35)},
      new Object[]{"otimeshat", new Integer(0x2A36)},
      new Object[]{"Otimes", new Integer(0x2A37)},
      new Object[]{"odiv", new Integer(0x2A38)},
      new Object[]{"triangleplus", new Integer(0x2A39)},
      new Object[]{"triangleminus", new Integer(0x2A3A)},
      new Object[]{"triangletimes", new Integer(0x2A3B)},
      new Object[]{"intprod", new Integer(0x2A3C)},
      new Object[]{"intprodr", new Integer(0x2A3D)},
      new Object[]{"fcmp", new Integer(0x2A3E)},
      new Object[]{"amalg", new Integer(0x2A3F)},
      new Object[]{"capdot", new Integer(0x2A40)},
      new Object[]{"uminus", new Integer(0x2A41)},
      new Object[]{"barcup", new Integer(0x2A42)},
      new Object[]{"barcap", new Integer(0x2A43)},
      new Object[]{"capwedge", new Integer(0x2A44)},
      new Object[]{"cupvee", new Integer(0x2A45)},
      new Object[]{"cupovercap", new Integer(0x2A46)},
      new Object[]{"capovercup", new Integer(0x2A47)},
      new Object[]{"cupbarcap", new Integer(0x2A48)},
      new Object[]{"capbarcup", new Integer(0x2A49)},
      new Object[]{"twocups", new Integer(0x2A4A)},
      new Object[]{"twocaps", new Integer(0x2A4B)},
      new Object[]{"closedvarcup", new Integer(0x2A4C)},
      new Object[]{"closedvarcap", new Integer(0x2A4D)},
      new Object[]{"Sqcap", new Integer(0x2A4E)},
      new Object[]{"Sqcup", new Integer(0x2A4F)},
      new Object[]{"closedvarcupsmashprod", new Integer(0x2A50)},
      new Object[]{"wedgeodot", new Integer(0x2A51)},
      new Object[]{"veeodot", new Integer(0x2A52)},
      new Object[]{"Wedge", new Integer(0x2A53)},
      new Object[]{"Vee", new Integer(0x2A54)},
      new Object[]{"wedgeonwedge", new Integer(0x2A55)},
      new Object[]{"veeonvee", new Integer(0x2A56)},
      new Object[]{"bigslopedvee", new Integer(0x2A57)},
      new Object[]{"bigslopedwedge", new Integer(0x2A58)},
      new Object[]{"wedgemidvert", new Integer(0x2A5A)},
      new Object[]{"veemidvert", new Integer(0x2A5B)},
      new Object[]{"midbarwedge", new Integer(0x2A5C)},
      new Object[]{"midbarvee", new Integer(0x2A5D)},
      new Object[]{"doublebarwedge", new Integer(0x2A5E)},
      new Object[]{"wedgebar", new Integer(0x2A5F)},
      new Object[]{"wedgedoublebar", new Integer(0x2A60)},
      new Object[]{"varveebar", new Integer(0x2A61)},
      new Object[]{"doublebarvee", new Integer(0x2A62)},
      new Object[]{"veedoublebar", new Integer(0x2A63)},
      new Object[]{"dsub", new Integer(0x2A64)},
      new Object[]{"rsub", new Integer(0x2A65)},
      new Object[]{"eqqplus", new Integer(0x2A71)},
      new Object[]{"pluseqq", new Integer(0x2A72)},
      new Object[]{"interleave", new Integer(0x2AF4)},
      new Object[]{"nhVvert", new Integer(0x2AF5)},
      new Object[]{"threedotcolon", new Integer(0x2AF6)},
      new Object[]{"trslash", new Integer(0x2AFB)},
      new Object[]{"sslash", new Integer(0x2AFD)},
      new Object[]{"talloblong", new Integer(0x2AFE)},
      // relations
      new Object[]{"ast", new Integer(0x002A)},
      new Object[]{"less", new Integer(0x003C)},
      new Object[]{"equal", new Integer(0x003D)},
      new Object[]{"greater", new Integer(0x003E)},
      new Object[]{"closure", new Integer(0x2050)},
      new Object[]{"vertoverlay", new Integer(0x20D2)},
      new Object[]{"leftarrow", new Integer(0x2190)},
      new Object[]{"gets", new Integer(0x2190)},
      new Object[]{"uparrow", new Integer(0x2191)},
      new Object[]{"rightarrow", new Integer(0x2192)},
      new Object[]{"to", new Integer(0x2192)},
      new Object[]{"downarrow", new Integer(0x2193)},
      new Object[]{"leftrightarrow", new Integer(0x2194)},
      new Object[]{"updownarrow", new Integer(0x2195)},
      new Object[]{"nwarrow", new Integer(0x2196)},
      new Object[]{"nearrow", new Integer(0x2197)},
      new Object[]{"searrow", new Integer(0x2198)},
      new Object[]{"swarrow", new Integer(0x2199)},
      new Object[]{"nleftarrow", new Integer(0x219A)},
      new Object[]{"nrightarrow", new Integer(0x219B)},
      new Object[]{"leftwavearrow", new Integer(0x219C)},
      new Object[]{"rightwavearrow", new Integer(0x219D)},
      new Object[]{"twoheadleftarrow", new Integer(0x219E)},
      new Object[]{"twoheaduparrow", new Integer(0x219F)},
      new Object[]{"twoheadrightarrow", new Integer(0x21A0)},
      new Object[]{"twoheaddownarrow", new Integer(0x21A1)},
      new Object[]{"leftarrowtail", new Integer(0x21A2)},
      new Object[]{"rightarrowtail", new Integer(0x21A3)},
      new Object[]{"mapsfrom", new Integer(0x21A4)},
      new Object[]{"mapsup", new Integer(0x21A5)},
      new Object[]{"mapsto", new Integer(0x21A6)},
      new Object[]{"mapsdown", new Integer(0x21A7)},
      new Object[]{"hookleftarrow", new Integer(0x21A9)},
      new Object[]{"hookrightarrow", new Integer(0x21AA)},
      new Object[]{"looparrowleft", new Integer(0x21AB)},
      new Object[]{"looparrowright", new Integer(0x21AC)},
      new Object[]{"leftrightsquigarrow", new Integer(0x21AD)},
      new Object[]{"nleftrightarrow", new Integer(0x21AE)},
      new Object[]{"downzigzagarrow", new Integer(0x21AF)},
      new Object[]{"Lsh", new Integer(0x21B0)},
      new Object[]{"Rsh", new Integer(0x21B1)},
      new Object[]{"Ldsh", new Integer(0x21B2)},
      new Object[]{"Rdsh", new Integer(0x21B3)},
      new Object[]{"curvearrowleft", new Integer(0x21B6)},
      new Object[]{"curvearrowright", new Integer(0x21B7)},
      new Object[]{"circlearrowleft", new Integer(0x21BA)},
      new Object[]{"circlearrowright", new Integer(0x21BB)},
      new Object[]{"leftharpoonup", new Integer(0x21BC)},
      new Object[]{"leftharpoondown", new Integer(0x21BD)},
      new Object[]{"upharpoonright", new Integer(0x21BE)},
      new Object[]{"restriction", new Integer(0x21BE)},
      new Object[]{"upharpoonleft", new Integer(0x21BF)},
      new Object[]{"rightharpoonup", new Integer(0x21C0)},
      new Object[]{"rightharpoondown", new Integer(0x21C1)},
      new Object[]{"downharpoonright", new Integer(0x21C2)},
      new Object[]{"downharpoonleft", new Integer(0x21C3)},
      new Object[]{"rightleftarrows", new Integer(0x21C4)},
      new Object[]{"updownarrows", new Integer(0x21C5)},
      new Object[]{"leftrightarrows", new Integer(0x21C6)},
      new Object[]{"leftleftarrows", new Integer(0x21C7)},
      new Object[]{"upuparrows", new Integer(0x21C8)},
      new Object[]{"rightrightarrows", new Integer(0x21C9)},
      new Object[]{"downdownarrows", new Integer(0x21CA)},
      new Object[]{"leftrightharpoons", new Integer(0x21CB)},
      new Object[]{"rightleftharpoons", new Integer(0x21CC)},
      new Object[]{"nLeftarrow", new Integer(0x21CD)},
      new Object[]{"nLeftrightarrow", new Integer(0x21CE)},
      new Object[]{"nRightarrow", new Integer(0x21CF)},
      new Object[]{"Leftarrow", new Integer(0x21D0)},
      new Object[]{"Uparrow", new Integer(0x21D1)},
      new Object[]{"Rightarrow", new Integer(0x21D2)},
      new Object[]{"Downarrow", new Integer(0x21D3)},
      new Object[]{"Leftrightarrow", new Integer(0x21D4)},
      new Object[]{"Updownarrow", new Integer(0x21D5)},
      new Object[]{"Nwarrow", new Integer(0x21D6)},
      new Object[]{"Nearrow", new Integer(0x21D7)},
      new Object[]{"Searrow", new Integer(0x21D8)},
      new Object[]{"Swarrow", new Integer(0x21D9)},
      new Object[]{"Lleftarrow", new Integer(0x21DA)},
      new Object[]{"Rrightarrow", new Integer(0x21DB)},
      new Object[]{"leftsquigarrow", new Integer(0x21DC)},
      new Object[]{"rightsquigarrow", new Integer(0x21DD)},
      new Object[]{"leadsto", new Integer(0x21DD)},
      new Object[]{"barleftarrow", new Integer(0x21E4)},
      new Object[]{"rightarrowbar", new Integer(0x21E5)},
      new Object[]{"circleonrightarrow", new Integer(0x21F4)},
      new Object[]{"downuparrows", new Integer(0x21F5)},
      new Object[]{"rightthreearrows", new Integer(0x21F6)},
      new Object[]{"nvleftarrow", new Integer(0x21F7)},
      new Object[]{"nvrightarrow", new Integer(0x21F8)},
      new Object[]{"nvleftrightarrow", new Integer(0x21F9)},
      new Object[]{"nVleftarrow", new Integer(0x21FA)},
      new Object[]{"nVrightarrow", new Integer(0x21FB)},
      new Object[]{"nVleftrightarrow", new Integer(0x21FC)},
      new Object[]{"leftarrowtriangle", new Integer(0x21FD)},
      new Object[]{"rightarrowtriangle", new Integer(0x21FE)},
      new Object[]{"leftrightarrowtriangle", new Integer(0x21FF)},
      new Object[]{"in", new Integer(0x2208)},
      new Object[]{"notin", new Integer(0x2209)},
      new Object[]{"smallin", new Integer(0x220A)},
      new Object[]{"ni", new Integer(0x220B)},
      new Object[]{"owns", new Integer(0x220B)},
      new Object[]{"nni", new Integer(0x220C)},
      new Object[]{"smallni", new Integer(0x220D)},
      new Object[]{"propto", new Integer(0x221D)},
      new Object[]{"varpropto", new Integer(0x221D)},
      new Object[]{"mid", new Integer(0x2223)},
      new Object[]{"shortmid", new Integer(0x2223)},
      new Object[]{"nmid", new Integer(0x2224)},
      new Object[]{"nshortmid", new Integer(0x2224)},
      new Object[]{"parallel", new Integer(0x2225)},
      new Object[]{"shortparallel", new Integer(0x2225)},
      new Object[]{"nparallel", new Integer(0x2226)},
      new Object[]{"nshortparallel", new Integer(0x2226)},
      new Object[]{"Colon", new Integer(0x2237)},
      new Object[]{"dashcolon", new Integer(0x2239)},
      new Object[]{"dotsminusdots", new Integer(0x223A)},
      new Object[]{"kernelcontraction", new Integer(0x223B)},
      new Object[]{"sim", new Integer(0x223C)},
      new Object[]{"thicksim", new Integer(0x223C)},
      new Object[]{"backsim", new Integer(0x223D)},
      new Object[]{"nsim", new Integer(0x2241)},
      new Object[]{"eqsim", new Integer(0x2242)},
      new Object[]{"simeq", new Integer(0x2243)},
      new Object[]{"nsime", new Integer(0x2244)},
      new Object[]{"cong", new Integer(0x2245)},
      new Object[]{"simneqq", new Integer(0x2246)},
      new Object[]{"ncong", new Integer(0x2247)},
      new Object[]{"approx", new Integer(0x2248)},
      new Object[]{"thickapprox", new Integer(0x2248)},
      new Object[]{"napprox", new Integer(0x2249)},
      new Object[]{"approxeq", new Integer(0x224A)},
      new Object[]{"approxident", new Integer(0x224B)},
      new Object[]{"backcong", new Integer(0x224C)},
      new Object[]{"asymp", new Integer(0x224D)},
      new Object[]{"Bumpeq", new Integer(0x224E)},
      new Object[]{"bumpeq", new Integer(0x224F)},
      new Object[]{"doteq", new Integer(0x2250)},
      new Object[]{"Doteq", new Integer(0x2251)},
      new Object[]{"doteqdot", new Integer(0x2251)},
      new Object[]{"fallingdotseq", new Integer(0x2252)},
      new Object[]{"risingdotseq", new Integer(0x2253)},
      new Object[]{"coloneq", new Integer(0x2254)},
      new Object[]{"eqcolon", new Integer(0x2255)},
      new Object[]{"eqcirc", new Integer(0x2256)},
      new Object[]{"circeq", new Integer(0x2257)},
      new Object[]{"arceq", new Integer(0x2258)},
      new Object[]{"wedgeq", new Integer(0x2259)},
      new Object[]{"veeeq", new Integer(0x225A)},
      new Object[]{"stareq", new Integer(0x225B)},
      new Object[]{"triangleq", new Integer(0x225C)},
      new Object[]{"eqdef", new Integer(0x225D)},
      new Object[]{"measeq", new Integer(0x225E)},
      new Object[]{"questeq", new Integer(0x225F)},
      new Object[]{"ne", new Integer(0x2260)},
      new Object[]{"neq", new Integer(0x2260)},
      new Object[]{"equiv", new Integer(0x2261)},
      new Object[]{"nequiv", new Integer(0x2262)},
      new Object[]{"Equiv", new Integer(0x2263)},
      new Object[]{"leq", new Integer(0x2264)},
      new Object[]{"le", new Integer(0x2264)},
      new Object[]{"geq", new Integer(0x2265)},
      new Object[]{"ge", new Integer(0x2265)},
      new Object[]{"leqq", new Integer(0x2266)},
      new Object[]{"geqq", new Integer(0x2267)},
      new Object[]{"lneqq", new Integer(0x2268)},
      new Object[]{"lvertneqq", new Integer(0x2268)},
      new Object[]{"gneqq", new Integer(0x2269)},
      new Object[]{"gvertneqq", new Integer(0x2269)},
      new Object[]{"ll", new Integer(0x226A)},
      new Object[]{"gg", new Integer(0x226B)},
      new Object[]{"between", new Integer(0x226C)},
      new Object[]{"nasymp", new Integer(0x226D)},
      new Object[]{"nless", new Integer(0x226E)},
      new Object[]{"ngtr", new Integer(0x226F)},
      new Object[]{"nleq", new Integer(0x2270)},
      new Object[]{"ngeq", new Integer(0x2271)},
      new Object[]{"lesssim", new Integer(0x2272)},
      new Object[]{"gtrsim", new Integer(0x2273)},
      new Object[]{"nlesssim", new Integer(0x2274)},
      new Object[]{"ngtrsim", new Integer(0x2275)},
      new Object[]{"lessgtr", new Integer(0x2276)},
      new Object[]{"gtrless", new Integer(0x2277)},
      new Object[]{"nlessgtr", new Integer(0x2278)},
      new Object[]{"ngtrless", new Integer(0x2279)},
      new Object[]{"prec", new Integer(0x227A)},
      new Object[]{"succ", new Integer(0x227B)},
      new Object[]{"preccurlyeq", new Integer(0x227C)},
      new Object[]{"succcurlyeq", new Integer(0x227D)},
      new Object[]{"precsim", new Integer(0x227E)},
      new Object[]{"succsim", new Integer(0x227F)},
      new Object[]{"nprec", new Integer(0x2280)},
      new Object[]{"nsucc", new Integer(0x2281)},
      new Object[]{"subset", new Integer(0x2282)},
      new Object[]{"supset", new Integer(0x2283)},
      new Object[]{"nsubset", new Integer(0x2284)},
      new Object[]{"nsupset", new Integer(0x2285)},
      new Object[]{"subseteq", new Integer(0x2286)},
      new Object[]{"supseteq", new Integer(0x2287)},
      new Object[]{"nsubseteq", new Integer(0x2288)},
      new Object[]{"nsupseteq", new Integer(0x2289)},
      new Object[]{"subsetneq", new Integer(0x228A)},
      new Object[]{"varsubsetneq", new Integer(0x228A)},
      new Object[]{"supsetneq", new Integer(0x228B)},
      new Object[]{"varsupsetneq", new Integer(0x228B)},
      new Object[]{"sqsubset", new Integer(0x228F)},
      new Object[]{"sqsupset", new Integer(0x2290)},
      new Object[]{"sqsubseteq", new Integer(0x2291)},
      new Object[]{"sqsupseteq", new Integer(0x2292)},
      new Object[]{"vdash", new Integer(0x22A2)},
      new Object[]{"dashv", new Integer(0x22A3)},
      new Object[]{"assert", new Integer(0x22A6)},
      new Object[]{"models", new Integer(0x22A7)},
      new Object[]{"vDash", new Integer(0x22A8)},
      new Object[]{"Vdash", new Integer(0x22A9)},
      new Object[]{"Vvdash", new Integer(0x22AA)},
      new Object[]{"VDash", new Integer(0x22AB)},
      new Object[]{"nvdash", new Integer(0x22AC)},
      new Object[]{"nvDash", new Integer(0x22AD)},
      new Object[]{"nVdash", new Integer(0x22AE)},
      new Object[]{"nVDash", new Integer(0x22AF)},
      new Object[]{"prurel", new Integer(0x22B0)},
      new Object[]{"scurel", new Integer(0x22B1)},
      new Object[]{"vartriangleleft", new Integer(0x22B2)},
      new Object[]{"vartriangleright", new Integer(0x22B3)},
      new Object[]{"trianglelefteq", new Integer(0x22B4)},
      new Object[]{"trianglerighteq", new Integer(0x22B5)},
      new Object[]{"origof", new Integer(0x22B6)},
      new Object[]{"imageof", new Integer(0x22B7)},
      new Object[]{"multimap", new Integer(0x22B8)},
      new Object[]{"bowtie", new Integer(0x22C8)},
      new Object[]{"backsimeq", new Integer(0x22CD)},
      new Object[]{"Subset", new Integer(0x22D0)},
      new Object[]{"Supset", new Integer(0x22D1)},
      new Object[]{"pitchfork", new Integer(0x22D4)},
      new Object[]{"equalparallel", new Integer(0x22D5)},
      new Object[]{"lessdot", new Integer(0x22D6)},
      new Object[]{"gtrdot", new Integer(0x22D7)},
      new Object[]{"lll", new Integer(0x22D8)},
      new Object[]{"llless", new Integer(0x22D8)},
      new Object[]{"ggg", new Integer(0x22D9)},
      new Object[]{"gggtr", new Integer(0x22D9)},
      new Object[]{"lesseqgtr", new Integer(0x22DA)},
      new Object[]{"gtreqless", new Integer(0x22DB)},
      new Object[]{"eqless", new Integer(0x22DC)},
      new Object[]{"eqgtr", new Integer(0x22DD)},
      new Object[]{"curlyeqprec", new Integer(0x22DE)},
      new Object[]{"curlyeqsucc", new Integer(0x22DF)},
      new Object[]{"npreccurlyeq", new Integer(0x22E0)},
      new Object[]{"nsucccurlyeq", new Integer(0x22E1)},
      new Object[]{"nsqsubseteq", new Integer(0x22E2)},
      new Object[]{"nsqsupseteq", new Integer(0x22E3)},
      new Object[]{"sqsubsetneq", new Integer(0x22E4)},
      new Object[]{"sqsupsetneq", new Integer(0x22E5)},
      new Object[]{"lnsim", new Integer(0x22E6)},
      new Object[]{"gnsim", new Integer(0x22E7)},
      new Object[]{"precnsim", new Integer(0x22E8)},
      new Object[]{"succnsim", new Integer(0x22E9)},
      new Object[]{"nvartriangleleft", new Integer(0x22EA)},
      new Object[]{"nvartriangleright", new Integer(0x22EB)},
      new Object[]{"ntrianglelefteq", new Integer(0x22EC)},
      new Object[]{"ntrianglerighteq", new Integer(0x22ED)},
      new Object[]{"vdots", new Integer(0x22EE)},
      new Object[]{"adots", new Integer(0x22F0)},
      new Object[]{"ddots", new Integer(0x22F1)},
      new Object[]{"disin", new Integer(0x22F2)},
      new Object[]{"varisins", new Integer(0x22F3)},
      new Object[]{"isins", new Integer(0x22F4)},
      new Object[]{"isindot", new Integer(0x22F5)},
      new Object[]{"varisinobar", new Integer(0x22F6)},
      new Object[]{"isinobar", new Integer(0x22F7)},
      new Object[]{"isinvb", new Integer(0x22F8)},
      new Object[]{"isinE", new Integer(0x22F9)},
      new Object[]{"nisd", new Integer(0x22FA)},
      new Object[]{"varnis", new Integer(0x22FB)},
      new Object[]{"nis", new Integer(0x22FC)},
      new Object[]{"varniobar", new Integer(0x22FD)},
      new Object[]{"niobar", new Integer(0x22FE)},
      new Object[]{"bagmember", new Integer(0x22FF)},
      new Object[]{"frown", new Integer(0x2322)},
      new Object[]{"smallfrown", new Integer(0x2322)},
      new Object[]{"smile", new Integer(0x2323)},
      new Object[]{"smallsmile", new Integer(0x2323)},
      new Object[]{"APLnotslash", new Integer(0x233F)},
      new Object[]{"vartriangle", new Integer(0x25B5)},
      new Object[]{"perp", new Integer(0x27C2)},
      new Object[]{"bsolhsub", new Integer(0x27C8)},
      new Object[]{"suphsol", new Integer(0x27C9)},
      new Object[]{"upin", new Integer(0x27D2)},
      new Object[]{"pullback", new Integer(0x27D3)},
      new Object[]{"pushout", new Integer(0x27D4)},
      new Object[]{"DashVDash", new Integer(0x27DA)},
      new Object[]{"dashVdash", new Integer(0x27DB)},
      new Object[]{"multimapinv", new Integer(0x27DC)},
      new Object[]{"vlongdash", new Integer(0x27DD)},
      new Object[]{"longdashv", new Integer(0x27DE)},
      new Object[]{"cirbot", new Integer(0x27DF)},
      new Object[]{"UUparrow", new Integer(0x27F0)},
      new Object[]{"DDownarrow", new Integer(0x27F1)},
      new Object[]{"acwgapcirclearrow", new Integer(0x27F2)},
      new Object[]{"cwgapcirclearrow", new Integer(0x27F3)},
      new Object[]{"rightarrowonoplus", new Integer(0x27F4)},
      new Object[]{"longleftarrow", new Integer(0x27F5)},
      new Object[]{"longrightarrow", new Integer(0x27F6)},
      new Object[]{"longleftrightarrow", new Integer(0x27F7)},
      new Object[]{"Longleftarrow", new Integer(0x27F8)},
      new Object[]{"Longrightarrow", new Integer(0x27F9)},
      new Object[]{"Longleftrightarrow", new Integer(0x27FA)},
      new Object[]{"longmapsfrom", new Integer(0x27FB)},
      new Object[]{"longmapsto", new Integer(0x27FC)},
      new Object[]{"Longmapsfrom", new Integer(0x27FD)},
      new Object[]{"Longmapsto", new Integer(0x27FE)},
      new Object[]{"longrightsquigarrow", new Integer(0x27FF)},
      new Object[]{"nvtwoheadrightarrow", new Integer(0x2900)},
      new Object[]{"nVtwoheadrightarrow", new Integer(0x2901)},
      new Object[]{"nvLeftarrow", new Integer(0x2902)},
      new Object[]{"nvRightarrow", new Integer(0x2903)},
      new Object[]{"nvLeftrightarrow", new Integer(0x2904)},
      new Object[]{"twoheadmapsto", new Integer(0x2905)},
      new Object[]{"Mapsfrom", new Integer(0x2906)},
      new Object[]{"Mapsto", new Integer(0x2907)},
      new Object[]{"downarrowbarred", new Integer(0x2908)},
      new Object[]{"uparrowbarred", new Integer(0x2909)},
      new Object[]{"Uuparrow", new Integer(0x290A)},
      new Object[]{"Ddownarrow", new Integer(0x290B)},
      new Object[]{"leftbkarrow", new Integer(0x290C)},
      new Object[]{"rightbkarrow", new Integer(0x290D)},
      new Object[]{"leftdbkarrow", new Integer(0x290E)},
      new Object[]{"dashleftarrow", new Integer(0x290E)},
      new Object[]{"dbkarow", new Integer(0x290F)},
      new Object[]{"dashrightarrow", new Integer(0x290F)},
      new Object[]{"drbkarow", new Integer(0x2910)},
      new Object[]{"rightdotarrow", new Integer(0x2911)},
      new Object[]{"baruparrow", new Integer(0x2912)},
      new Object[]{"downarrowbar", new Integer(0x2913)},
      new Object[]{"nvrightarrowtail", new Integer(0x2914)},
      new Object[]{"nVrightarrowtail", new Integer(0x2915)},
      new Object[]{"twoheadrightarrowtail", new Integer(0x2916)},
      new Object[]{"nvtwoheadrightarrowtail", new Integer(0x2917)},
      new Object[]{"nVtwoheadrightarrowtail", new Integer(0x2918)},
      new Object[]{"lefttail", new Integer(0x2919)},
      new Object[]{"righttail", new Integer(0x291A)},
      new Object[]{"leftdbltail", new Integer(0x291B)},
      new Object[]{"rightdbltail", new Integer(0x291C)},
      new Object[]{"diamondleftarrow", new Integer(0x291D)},
      new Object[]{"rightarrowdiamond", new Integer(0x291E)},
      new Object[]{"diamondleftarrowbar", new Integer(0x291F)},
      new Object[]{"barrightarrowdiamond", new Integer(0x2920)},
      new Object[]{"nwsearrow", new Integer(0x2921)},
      new Object[]{"neswarrow", new Integer(0x2922)},
      new Object[]{"hknwarrow", new Integer(0x2923)},
      new Object[]{"hknearrow", new Integer(0x2924)},
      new Object[]{"hksearow", new Integer(0x2925)},
      new Object[]{"hkswarow", new Integer(0x2926)},
      new Object[]{"tona", new Integer(0x2927)},
      new Object[]{"toea", new Integer(0x2928)},
      new Object[]{"tosa", new Integer(0x2929)},
      new Object[]{"towa", new Integer(0x292A)},
      new Object[]{"rightcurvedarrow", new Integer(0x2933)},
      new Object[]{"leftdowncurvedarrow", new Integer(0x2936)},
      new Object[]{"rightdowncurvedarrow", new Integer(0x2937)},
      new Object[]{"cwrightarcarrow", new Integer(0x2938)},
      new Object[]{"acwleftarcarrow", new Integer(0x2939)},
      new Object[]{"acwoverarcarrow", new Integer(0x293A)},
      new Object[]{"acwunderarcarrow", new Integer(0x293B)},
      new Object[]{"curvearrowrightminus", new Integer(0x293C)},
      new Object[]{"curvearrowleftplus", new Integer(0x293D)},
      new Object[]{"cwundercurvearrow", new Integer(0x293E)},
      new Object[]{"ccwundercurvearrow", new Integer(0x293F)},
      new Object[]{"acwcirclearrow", new Integer(0x2940)},
      new Object[]{"cwcirclearrow", new Integer(0x2941)},
      new Object[]{"rightarrowshortleftarrow", new Integer(0x2942)},
      new Object[]{"leftarrowshortrightarrow", new Integer(0x2943)},
      new Object[]{"shortrightarrowleftarrow", new Integer(0x2944)},
      new Object[]{"rightarrowplus", new Integer(0x2945)},
      new Object[]{"leftarrowplus", new Integer(0x2946)},
      new Object[]{"rightarrowx", new Integer(0x2947)},
      new Object[]{"leftrightarrowcircle", new Integer(0x2948)},
      new Object[]{"twoheaduparrowcircle", new Integer(0x2949)},
      new Object[]{"leftrightharpoonupdown", new Integer(0x294A)},
      new Object[]{"leftrightharpoondownup", new Integer(0x294B)},
      new Object[]{"updownharpoonrightleft", new Integer(0x294C)},
      new Object[]{"updownharpoonleftright", new Integer(0x294D)},
      new Object[]{"leftrightharpoonupup", new Integer(0x294E)},
      new Object[]{"updownharpoonrightright", new Integer(0x294F)},
      new Object[]{"leftrightharpoondowndown", new Integer(0x2950)},
      new Object[]{"updownharpoonleftleft", new Integer(0x2951)},
      new Object[]{"barleftharpoonup", new Integer(0x2952)},
      new Object[]{"rightharpoonupbar", new Integer(0x2953)},
      new Object[]{"barupharpoonright", new Integer(0x2954)},
      new Object[]{"downharpoonrightbar", new Integer(0x2955)},
      new Object[]{"barleftharpoondown", new Integer(0x2956)},
      new Object[]{"rightharpoondownbar", new Integer(0x2957)},
      new Object[]{"barupharpoonleft", new Integer(0x2958)},
      new Object[]{"downharpoonleftbar", new Integer(0x2959)},
      new Object[]{"leftharpoonupbar", new Integer(0x295A)},
      new Object[]{"barrightharpoonup", new Integer(0x295B)},
      new Object[]{"upharpoonrightbar", new Integer(0x295C)},
      new Object[]{"bardownharpoonright", new Integer(0x295D)},
      new Object[]{"leftharpoondownbar", new Integer(0x295E)},
      new Object[]{"barrightharpoondown", new Integer(0x295F)},
      new Object[]{"upharpoonleftbar", new Integer(0x2960)},
      new Object[]{"bardownharpoonleft", new Integer(0x2961)},
      new Object[]{"leftharpoonsupdown", new Integer(0x2962)},
      new Object[]{"upharpoonsleftright", new Integer(0x2963)},
      new Object[]{"rightharpoonsupdown", new Integer(0x2964)},
      new Object[]{"downharpoonsleftright", new Integer(0x2965)},
      new Object[]{"leftrightharpoonsup", new Integer(0x2966)},
      new Object[]{"leftrightharpoonsdown", new Integer(0x2967)},
      new Object[]{"rightleftharpoonsup", new Integer(0x2968)},
      new Object[]{"rightleftharpoonsdown", new Integer(0x2969)},
      new Object[]{"leftharpoonupdash", new Integer(0x296A)},
      new Object[]{"dashleftharpoondown", new Integer(0x296B)},
      new Object[]{"rightharpoonupdash", new Integer(0x296C)},
      new Object[]{"dashrightharpoondown", new Integer(0x296D)},
      new Object[]{"updownharpoonsleftright", new Integer(0x296E)},
      new Object[]{"downupharpoonsleftright", new Integer(0x296F)},
      new Object[]{"rightimply", new Integer(0x2970)},
      new Object[]{"equalrightarrow", new Integer(0x2971)},
      new Object[]{"similarrightarrow", new Integer(0x2972)},
      new Object[]{"leftarrowsimilar", new Integer(0x2973)},
      new Object[]{"rightarrowsimilar", new Integer(0x2974)},
      new Object[]{"rightarrowapprox", new Integer(0x2975)},
      new Object[]{"ltlarr", new Integer(0x2976)},
      new Object[]{"leftarrowless", new Integer(0x2977)},
      new Object[]{"gtrarr", new Integer(0x2978)},
      new Object[]{"subrarr", new Integer(0x2979)},
      new Object[]{"leftarrowsubset", new Integer(0x297A)},
      new Object[]{"suplarr", new Integer(0x297B)},
      new Object[]{"leftfishtail", new Integer(0x297C)},
      new Object[]{"rightfishtail", new Integer(0x297D)},
      new Object[]{"upfishtail", new Integer(0x297E)},
      new Object[]{"downfishtail", new Integer(0x297F)},
      new Object[]{"rtriltri", new Integer(0x29CE)},
      new Object[]{"ltrivb", new Integer(0x29CF)},
      new Object[]{"vbrtri", new Integer(0x29D0)},
      new Object[]{"lfbowtie", new Integer(0x29D1)},
      new Object[]{"rfbowtie", new Integer(0x29D2)},
      new Object[]{"fbowtie", new Integer(0x29D3)},
      new Object[]{"lftimes", new Integer(0x29D4)},
      new Object[]{"rftimes", new Integer(0x29D5)},
      new Object[]{"dualmap", new Integer(0x29DF)},
      new Object[]{"lrtriangleeq", new Integer(0x29E1)},
      new Object[]{"eparsl", new Integer(0x29E3)},
      new Object[]{"smeparsl", new Integer(0x29E4)},
      new Object[]{"eqvparsl", new Integer(0x29E5)},
      new Object[]{"gleichstark", new Integer(0x29E6)},
      new Object[]{"ruledelayed", new Integer(0x29F4)},
      new Object[]{"veeonwedge", new Integer(0x2A59)},
      new Object[]{"eqdot", new Integer(0x2A66)},
      new Object[]{"dotequiv", new Integer(0x2A67)},
      new Object[]{"equivVert", new Integer(0x2A68)},
      new Object[]{"equivVvert", new Integer(0x2A69)},
      new Object[]{"dotsim", new Integer(0x2A6A)},
      new Object[]{"simrdots", new Integer(0x2A6B)},
      new Object[]{"simminussim", new Integer(0x2A6C)},
      new Object[]{"congdot", new Integer(0x2A6D)},
      new Object[]{"asteq", new Integer(0x2A6E)},
      new Object[]{"hatapprox", new Integer(0x2A6F)},
      new Object[]{"approxeqq", new Integer(0x2A70)},
      new Object[]{"eqqsim", new Integer(0x2A73)},
      new Object[]{"Coloneq", new Integer(0x2A74)},
      new Object[]{"eqeq", new Integer(0x2A75)},
      new Object[]{"eqeqeq", new Integer(0x2A76)},
      new Object[]{"ddotseq", new Integer(0x2A77)},
      new Object[]{"equivDD", new Integer(0x2A78)},
      new Object[]{"ltcir", new Integer(0x2A79)},
      new Object[]{"gtcir", new Integer(0x2A7A)},
      new Object[]{"ltquest", new Integer(0x2A7B)},
      new Object[]{"gtquest", new Integer(0x2A7C)},
      new Object[]{"leqslant", new Integer(0x2A7D)},
      new Object[]{"geqslant", new Integer(0x2A7E)},
      new Object[]{"lesdot", new Integer(0x2A7F)},
      new Object[]{"gesdot", new Integer(0x2A80)},
      new Object[]{"lesdoto", new Integer(0x2A81)},
      new Object[]{"gesdoto", new Integer(0x2A82)},
      new Object[]{"lesdotor", new Integer(0x2A83)},
      new Object[]{"gesdotol", new Integer(0x2A84)},
      new Object[]{"lessapprox", new Integer(0x2A85)},
      new Object[]{"gtrapprox", new Integer(0x2A86)},
      new Object[]{"lneq", new Integer(0x2A87)},
      new Object[]{"gneq", new Integer(0x2A88)},
      new Object[]{"lnapprox", new Integer(0x2A89)},
      new Object[]{"gnapprox", new Integer(0x2A8A)},
      new Object[]{"lesseqqgtr", new Integer(0x2A8B)},
      new Object[]{"gtreqqless", new Integer(0x2A8C)},
      new Object[]{"lsime", new Integer(0x2A8D)},
      new Object[]{"gsime", new Integer(0x2A8E)},
      new Object[]{"lsimg", new Integer(0x2A8F)},
      new Object[]{"gsiml", new Integer(0x2A90)},
      new Object[]{"lgE", new Integer(0x2A91)},
      new Object[]{"glE", new Integer(0x2A92)},
      new Object[]{"lesges", new Integer(0x2A93)},
      new Object[]{"gesles", new Integer(0x2A94)},
      new Object[]{"eqslantless", new Integer(0x2A95)},
      new Object[]{"eqslantgtr", new Integer(0x2A96)},
      new Object[]{"elsdot", new Integer(0x2A97)},
      new Object[]{"egsdot", new Integer(0x2A98)},
      new Object[]{"eqqless", new Integer(0x2A99)},
      new Object[]{"eqqgtr", new Integer(0x2A9A)},
      new Object[]{"eqqslantless", new Integer(0x2A9B)},
      new Object[]{"eqqslantgtr", new Integer(0x2A9C)},
      new Object[]{"simless", new Integer(0x2A9D)},
      new Object[]{"simgtr", new Integer(0x2A9E)},
      new Object[]{"simlE", new Integer(0x2A9F)},
      new Object[]{"simgE", new Integer(0x2AA0)},
      new Object[]{"Lt", new Integer(0x2AA1)},
      new Object[]{"Gt", new Integer(0x2AA2)},
      new Object[]{"partialmeetcontraction", new Integer(0x2AA3)},
      new Object[]{"glj", new Integer(0x2AA4)},
      new Object[]{"gla", new Integer(0x2AA5)},
      new Object[]{"ltcc", new Integer(0x2AA6)},
      new Object[]{"gtcc", new Integer(0x2AA7)},
      new Object[]{"lescc", new Integer(0x2AA8)},
      new Object[]{"gescc", new Integer(0x2AA9)},
      new Object[]{"smt", new Integer(0x2AAA)},
      new Object[]{"lat", new Integer(0x2AAB)},
      new Object[]{"smte", new Integer(0x2AAC)},
      new Object[]{"late", new Integer(0x2AAD)},
      new Object[]{"bumpeqq", new Integer(0x2AAE)},
      new Object[]{"preceq", new Integer(0x2AAF)},
      new Object[]{"succeq", new Integer(0x2AB0)},
      new Object[]{"precneq", new Integer(0x2AB1)},
      new Object[]{"succneq", new Integer(0x2AB2)},
      new Object[]{"preceqq", new Integer(0x2AB3)},
      new Object[]{"succeqq", new Integer(0x2AB4)},
      new Object[]{"precneqq", new Integer(0x2AB5)},
      new Object[]{"succneqq", new Integer(0x2AB6)},
      new Object[]{"precapprox", new Integer(0x2AB7)},
      new Object[]{"succapprox", new Integer(0x2AB8)},
      new Object[]{"precnapprox", new Integer(0x2AB9)},
      new Object[]{"succnapprox", new Integer(0x2ABA)},
      new Object[]{"Prec", new Integer(0x2ABB)},
      new Object[]{"Succ", new Integer(0x2ABC)},
      new Object[]{"subsetdot", new Integer(0x2ABD)},
      new Object[]{"supsetdot", new Integer(0x2ABE)},
      new Object[]{"subsetplus", new Integer(0x2ABF)},
      new Object[]{"supsetplus", new Integer(0x2AC0)},
      new Object[]{"submult", new Integer(0x2AC1)},
      new Object[]{"supmult", new Integer(0x2AC2)},
      new Object[]{"subedot", new Integer(0x2AC3)},
      new Object[]{"supedot", new Integer(0x2AC4)},
      new Object[]{"subseteqq", new Integer(0x2AC5)},
      new Object[]{"supseteqq", new Integer(0x2AC6)},
      new Object[]{"subsim", new Integer(0x2AC7)},
      new Object[]{"supsim", new Integer(0x2AC8)},
      new Object[]{"subsetapprox", new Integer(0x2AC9)},
      new Object[]{"supsetapprox", new Integer(0x2ACA)},
      new Object[]{"subsetneqq", new Integer(0x2ACB)},
      new Object[]{"varsubsetneqq", new Integer(0x2ACB)},
      new Object[]{"supsetneqq", new Integer(0x2ACC)},
      new Object[]{"varsupsetneqq", new Integer(0x2ACC)},
      new Object[]{"lsqhook", new Integer(0x2ACD)},
      new Object[]{"rsqhook", new Integer(0x2ACE)},
      new Object[]{"csub", new Integer(0x2ACF)},
      new Object[]{"csup", new Integer(0x2AD0)},
      new Object[]{"csube", new Integer(0x2AD1)},
      new Object[]{"csupe", new Integer(0x2AD2)},
      new Object[]{"subsup", new Integer(0x2AD3)},
      new Object[]{"supsub", new Integer(0x2AD4)},
      new Object[]{"subsub", new Integer(0x2AD5)},
      new Object[]{"supsup", new Integer(0x2AD6)},
      new Object[]{"suphsub", new Integer(0x2AD7)},
      new Object[]{"supdsub", new Integer(0x2AD8)},
      new Object[]{"forkv", new Integer(0x2AD9)},
      new Object[]{"topfork", new Integer(0x2ADA)},
      new Object[]{"mlcp", new Integer(0x2ADB)},
      new Object[]{"forks", new Integer(0x2ADC)},
      new Object[]{"forksnot", new Integer(0x2ADD)},
      new Object[]{"shortlefttack", new Integer(0x2ADE)},
      new Object[]{"shortdowntack", new Integer(0x2ADF)},
      new Object[]{"shortuptack", new Integer(0x2AE0)},
      new Object[]{"vDdash", new Integer(0x2AE2)},
      new Object[]{"dashV", new Integer(0x2AE3)},
      new Object[]{"Dashv", new Integer(0x2AE4)},
      new Object[]{"DashV", new Integer(0x2AE5)},
      new Object[]{"varVdash", new Integer(0x2AE6)},
      new Object[]{"Barv", new Integer(0x2AE7)},
      new Object[]{"vBar", new Integer(0x2AE8)},
      new Object[]{"vBarv", new Integer(0x2AE9)},
      new Object[]{"barV", new Integer(0x2AEA)},
      new Object[]{"Vbar", new Integer(0x2AEB)},
      new Object[]{"Not", new Integer(0x2AEC)},
      new Object[]{"bNot", new Integer(0x2AED)},
      new Object[]{"revnmid", new Integer(0x2AEE)},
      new Object[]{"cirmid", new Integer(0x2AEF)},
      new Object[]{"midcir", new Integer(0x2AF0)},
      new Object[]{"nhpar", new Integer(0x2AF2)},
      new Object[]{"parsim", new Integer(0x2AF3)},
      new Object[]{"lllnest", new Integer(0x2AF7)},
      new Object[]{"gggnest", new Integer(0x2AF8)},
      new Object[]{"leqqslant", new Integer(0x2AF9)},
      new Object[]{"geqqslant", new Integer(0x2AFA)},
      new Object[]{"circleonleftarrow", new Integer(0x2B30)},
      new Object[]{"leftthreearrows", new Integer(0x2B31)},
      new Object[]{"leftarrowonoplus", new Integer(0x2B32)},
      new Object[]{"longleftsquigarrow", new Integer(0x2B33)},
      new Object[]{"nvtwoheadleftarrow", new Integer(0x2B34)},
      new Object[]{"nVtwoheadleftarrow", new Integer(0x2B35)},
      new Object[]{"twoheadmapsfrom", new Integer(0x2B36)},
      new Object[]{"twoheadleftdbkarrow", new Integer(0x2B37)},
      new Object[]{"leftdotarrow", new Integer(0x2B38)},
      new Object[]{"nvleftarrowtail", new Integer(0x2B39)},
      new Object[]{"nVleftarrowtail", new Integer(0x2B3A)},
      new Object[]{"twoheadleftarrowtail", new Integer(0x2B3B)},
      new Object[]{"nvtwoheadleftarrowtail", new Integer(0x2B3C)},
      new Object[]{"nVtwoheadleftarrowtail", new Integer(0x2B3D)},
      new Object[]{"leftarrowx", new Integer(0x2B3E)},
      new Object[]{"leftcurvedarrow", new Integer(0x2B3F)},
      new Object[]{"equalleftarrow", new Integer(0x2B40)},
      new Object[]{"bsimilarleftarrow", new Integer(0x2B41)},
      new Object[]{"leftarrowbackapprox", new Integer(0x2B42)},
      new Object[]{"rightarrowgtr", new Integer(0x2B43)},
      new Object[]{"rightarrowsupset", new Integer(0x2B44)},
      new Object[]{"LLeftarrow", new Integer(0x2B45)},
      new Object[]{"RRightarrow", new Integer(0x2B46)},
      new Object[]{"bsimilarrightarrow", new Integer(0x2B47)},
      new Object[]{"rightarrowbackapprox", new Integer(0x2B48)},
      new Object[]{"similarleftarrow", new Integer(0x2B49)},
      new Object[]{"leftarrowapprox", new Integer(0x2B4A)},
      new Object[]{"leftarrowbsimilar", new Integer(0x2B4B)},
      new Object[]{"rightarrowbsimilar", new Integer(0x2B4C)},
   };

   private static final Object[][] PUNCTUATION = new Object[][]
   {
      // punctuation
      new Object[]{"ldotp", new Integer(0x002E)},
      new Object[]{"colon", new Integer(0x003A)},
   };

   private static final Object[][] SMALL_INTEGRALS = new Object[][]
   {
      // integrals
      new Object[]{"smallint", new Integer(0x222B)},
      new Object[]{"smalliint", new Integer(0x222C)},
      new Object[]{"smalliiint", new Integer(0x222D)},
      new Object[]{"smalloint", new Integer(0x222E)},
      new Object[]{"smalloiint", new Integer(0x222F)},
      new Object[]{"smalloiiint", new Integer(0x2230)},
      new Object[]{"smallintclockwise", new Integer(0x2231)},
      new Object[]{"smallvarointclockwise", new Integer(0x2232)},
      new Object[]{"smallointctrclockwise", new Integer(0x2233)},
      new Object[]{"smallsumint", new Integer(0x2A0B)},
      new Object[]{"smalliiiint", new Integer(0x2A0C)},
      new Object[]{"smallintbar", new Integer(0x2A0D)},
      new Object[]{"smallintBar", new Integer(0x2A0E)},
      new Object[]{"smallfint", new Integer(0x2A0F)},
      new Object[]{"smallcirfnint", new Integer(0x2A10)},
      new Object[]{"smallawint", new Integer(0x2A11)},
      new Object[]{"smallrppolint", new Integer(0x2A12)},
      new Object[]{"smallscpolint", new Integer(0x2A13)},
      new Object[]{"smallnpolint", new Integer(0x2A14)},
      new Object[]{"smallpointint", new Integer(0x2A15)},
      new Object[]{"smallsqint", new Integer(0x2A16)},
      new Object[]{"smallintlarhk", new Integer(0x2A17)},
      new Object[]{"smallintx", new Integer(0x2A18)},
      new Object[]{"smallintcap", new Integer(0x2A19)},
      new Object[]{"smallintcup", new Integer(0x2A1A)},
      new Object[]{"smallupint", new Integer(0x2A1B)},
      new Object[]{"smalllowint", new Integer(0x2A1C)},
   };

   private static final Object[][] INTEGRALS = new Object[][]
   {
      new Object[]{"int", new Integer(0x222B)},
      new Object[]{"iint", new Integer(0x222C)},
      new Object[]{"iiint", new Integer(0x222D)},
      new Object[]{"oint", new Integer(0x222E)},
      new Object[]{"oiint", new Integer(0x222F)},
      new Object[]{"oiiint", new Integer(0x2230)},
      new Object[]{"intclockwise", new Integer(0x2231)},
      new Object[]{"varointclockwise", new Integer(0x2232)},
      new Object[]{"ointctrclockwise", new Integer(0x2233)},
      new Object[]{"sumint", new Integer(0x2A0B)},
      new Object[]{"iiiint", new Integer(0x2A0C)},
      new Object[]{"intbar", new Integer(0x2A0D)},
      new Object[]{"intBar", new Integer(0x2A0E)},
      new Object[]{"fint", new Integer(0x2A0F)},
      new Object[]{"cirfnint", new Integer(0x2A10)},
      new Object[]{"awint", new Integer(0x2A11)},
      new Object[]{"rppolint", new Integer(0x2A12)},
      new Object[]{"scpolint", new Integer(0x2A13)},
      new Object[]{"npolint", new Integer(0x2A14)},
      new Object[]{"pointint", new Integer(0x2A15)},
      new Object[]{"sqint", new Integer(0x2A16)},
      new Object[]{"intlarhk", new Integer(0x2A17)},
      new Object[]{"intx", new Integer(0x2A18)},
      new Object[]{"intcap", new Integer(0x2A19)},
      new Object[]{"intcup", new Integer(0x2A1A)},
      new Object[]{"upint", new Integer(0x2A1B)},
      new Object[]{"lowint", new Integer(0x2A1C)},
   };

   private static final Object[][] BIG_OPERATORS = new Object[][]
   {
      // big operators
      new Object[]{"Bbbsum", new Integer(0x2140)},
      new Object[]{"prod", new Integer(0x220F)},
      new Object[]{"coprod", new Integer(0x2210)},
      new Object[]{"sum", new Integer(0x2211)},
      new Object[]{"bigwedge", new Integer(0x22C0)},
      new Object[]{"bigvee", new Integer(0x22C1)},
      new Object[]{"bigcap", new Integer(0x22C2)},
      new Object[]{"bigcup", new Integer(0x22C3)},
      new Object[]{"leftouterjoin", new Integer(0x27D5)},
      new Object[]{"rightouterjoin", new Integer(0x27D6)},
      new Object[]{"fullouterjoin", new Integer(0x27D7)},
      new Object[]{"bigbot", new Integer(0x27D8)},
      new Object[]{"bigtop", new Integer(0x27D9)},
      new Object[]{"xsol", new Integer(0x29F8)},
      new Object[]{"xbsol", new Integer(0x29F9)},
      new Object[]{"bigodot", new Integer(0x2A00)},
      new Object[]{"bigoplus", new Integer(0x2A01)},
      new Object[]{"bigotimes", new Integer(0x2A02)},
      new Object[]{"bigcupdot", new Integer(0x2A03)},
      new Object[]{"biguplus", new Integer(0x2A04)},
      new Object[]{"bigsqcap", new Integer(0x2A05)},
      new Object[]{"bigsqcup", new Integer(0x2A06)},
      new Object[]{"conjquant", new Integer(0x2A07)},
      new Object[]{"disjquant", new Integer(0x2A08)},
      new Object[]{"bigtimes", new Integer(0x2A09)},
      new Object[]{"modtwosum", new Integer(0x2A0A)},
      new Object[]{"Join", new Integer(0x2A1D)},
      new Object[]{"bigtriangleleft", new Integer(0x2A1E)},
      new Object[]{"zcmp", new Integer(0x2A1F)},
      new Object[]{"zpipe", new Integer(0x2A20)},
      new Object[]{"zproject", new Integer(0x2A21)},
      new Object[]{"biginterleave", new Integer(0x2AFC)},
      new Object[]{"bigtalloblong", new Integer(0x2AFF)},
   };

   private static final Object[][] DELIMITERS = new Object[][]
   {
      // delimiters
      new Object[]{"lbrace", new Integer(0x007B)},
      new Object[]{"backslash", new Integer(0x005C)},
      new Object[]{"rbrace", new Integer(0x007D)},
      new Object[]{"lceil", new Integer(0x2308)},
      new Object[]{"lfloor", new Integer(0x230A)},
      new Object[]{"lmoustache", new Integer(0x23B0)},
      new Object[]{"lbrbrak", new Integer(0x2772)},
      new Object[]{"lBrack", new Integer(0x27E6)},
      new Object[]{"langle", new Integer(0x27E8)},
      new Object[]{"lAngle", new Integer(0x27EA)},
      new Object[]{"lgroup", new Integer(0x27EE)},
      new Object[]{"lBrace", new Integer(0x2983)},
      new Object[]{"lParen", new Integer(0x2985)},
      new Object[]{"rceil", new Integer(0x2309)},
      new Object[]{"rfloor", new Integer(0x230B)},
      new Object[]{"rmoustache", new Integer(0x23B1)},
      new Object[]{"rbrbrak", new Integer(0x2773)},
      new Object[]{"rBrack", new Integer(0x27E7)},
      new Object[]{"rangle", new Integer(0x27E9)},
      new Object[]{"rAngle", new Integer(0x27EB)},
      new Object[]{"rgroup", new Integer(0x27EF)},
      new Object[]{"rBrace", new Integer(0x2984)},
      new Object[]{"rParen", new Integer(0x2986)},
      new Object[]{"vert", new Integer(0x007C)},
      new Object[]{"Vert", new Integer(0x2016)},
      new Object[]{"|", new Integer(0x2016)},
      new Object[]{"Vvert", new Integer(0x2980)},
      new Object[]{"uparrow", new Integer(0x2191)},
      new Object[]{"downarrow", new Integer(0x2193)},
      new Object[]{"updownarrow", new Integer(0x2195)},
      new Object[]{"Uparrow", new Integer(0x21D1)},
      new Object[]{"Downarrow", new Integer(0x21D3)},
      new Object[]{"Updownarrow", new Integer(0x21D5)},
      new Object[]{"Uuparrow", new Integer(0x290A)},
      new Object[]{"Ddownarrow", new Integer(0x290B)},
      new Object[]{"UUparrow", new Integer(0x27F0)},
      new Object[]{"DDownarrow", new Integer(0x27F1)},
   };

   private static final Object[][] OTHER_BRACES = new Object[][]
   {
      new Object[]{"ulcorner", new Integer(0x231C)},
      new Object[]{"urcorner", new Integer(0x231D)},
      new Object[]{"llcorner", new Integer(0x231E)},
      new Object[]{"lrcorner", new Integer(0x231F)},
      new Object[]{"Lbrbrak", new Integer(0x27EC)},
      new Object[]{"Rbrbrak", new Integer(0x27ED)},
      new Object[]{"llparenthesis", new Integer(0x2987)},
      new Object[]{"rrparenthesis", new Integer(0x2988)},
      new Object[]{"llangle", new Integer(0x2989)},
      new Object[]{"rrangle", new Integer(0x298A)},
      new Object[]{"lbrackubar", new Integer(0x298B)},
      new Object[]{"rbrackubar", new Integer(0x298C)},
      new Object[]{"lbrackultick", new Integer(0x298D)},
      new Object[]{"rbracklrtick", new Integer(0x298E)},
      new Object[]{"lbracklltick", new Integer(0x298F)},
      new Object[]{"rbrackurtick", new Integer(0x2990)},
      new Object[]{"langledot", new Integer(0x2991)},
      new Object[]{"rangledot", new Integer(0x2992)},
      new Object[]{"lparenless", new Integer(0x2993)},
      new Object[]{"rparengtr", new Integer(0x2994)},
      new Object[]{"Lparengtr", new Integer(0x2995)},
      new Object[]{"Rparenless", new Integer(0x2996)},
      new Object[]{"lblkbrbrak", new Integer(0x2997)},
      new Object[]{"rblkbrbrak", new Integer(0x2998)},
      new Object[]{"lvzigzag", new Integer(0x29D8)},
      new Object[]{"rvzigzag", new Integer(0x29D9)},
      new Object[]{"Lvzigzag", new Integer(0x29DA)},
      new Object[]{"Rvzigzag", new Integer(0x29DB)},
      new Object[]{"lcurvyangle", new Integer(0x29FC)},
      new Object[]{"rcurvyangle", new Integer(0x29FD)},
      new Object[]{"lbrbrak", new Integer(0x2772)},
      new Object[]{"rbrbrak", new Integer(0x2773)},
      new Object[]{"lbag", new Integer(0x27C5)},
      new Object[]{"rbag", new Integer(0x27C6)},
      new Object[]{"Lbrbrak", new Integer(0x27EC)},
      new Object[]{"Rbrbrak", new Integer(0x27ED)},
   };

   private static final Object[][] ACCENTS = new Object[][]
   {
      new Object[]{"grave", new Integer(0x0300)},
      new Object[]{"acute", new Integer(0x0301)},
      new Object[]{"hat", new Integer(0x0302)},
      new Object[]{"tilde", new Integer(0x0303)},
      new Object[]{"bar", new Integer(0x0304)},
      new Object[]{"breve", new Integer(0x0306)},
      new Object[]{"dot", new Integer(0x0307)},
      new Object[]{"ddot", new Integer(0x0308)},
      new Object[]{"ovhook", new Integer(0x0309)},
      new Object[]{"mathring", new Integer(0x030A)},
      new Object[]{"check", new Integer(0x030C)},
      new Object[]{"candra", new Integer(0x0310)},
      new Object[]{"oturnedcomma", new Integer(0x0312)},
      new Object[]{"ocommatopright", new Integer(0x0315)},
      new Object[]{"droang", new Integer(0x031A)},
      new Object[]{"leftharpoonaccent", new Integer(0x20D0)},
      new Object[]{"rightharpoonaccent", new Integer(0x20D1)},
      new Object[]{"leftarrowaccent", new Integer(0x20D6)},
      new Object[]{"vec", new Integer(0x20D7)},
      new Object[]{"rightarrowaccent", new Integer(0x20D7)},
      new Object[]{"leftrightarrowaccent", new Integer(0x20E1)},
      new Object[]{"dddot", new Integer(0x20DB)},
      new Object[]{"ddddot", new Integer(0x20DC)},
      new Object[]{"annuity", new Integer(0x20E7)},
      new Object[]{"widebridgeabove", new Integer(0x20E9)},
      new Object[]{"asteraccent", new Integer(0x20F0)},
   };

   private static final Object[][] WIDE_ACCENTS = new Object[][]
   {
      new Object[]{"widehat", new Integer(0x0302)},
      new Object[]{"widetilde", new Integer(0x0303)},
      new Object[]{"widecheck", new Integer(0x030C)},
      new Object[]{"overleftarrow", new Integer(0x20D6)},
      new Object[]{"overrightarrow", new Integer(0x20D7)},
      new Object[]{"underrightarrow", new Integer(0x20EF)},
      new Object[]{"underleftarrow", new Integer(0x20EE)},
      new Object[]{"overleftrightarrow", new Integer(0x20E1)},
      new Object[]{"underleftrightarrow", new Integer(0x034D)},
      new Object[]{"overleftharpoon", new Integer(0x20D0)},
      new Object[]{"overrightharpoon", new Integer(0x20D1)},
      new Object[]{"underleftharpoon", new Integer(0x20EC)},
      new Object[]{"underrightharpoon", new Integer(0x20ED)},
   };

   private static final Object[][] WIDE_BRACKETS = new Object[][]
   {
      new Object[]{"overbracket", new Integer(0x23B4)},
      new Object[]{"overparen", new Integer(0x23DC)},
      new Object[]{"overbrace", new Integer(0x23DE)},
      new Object[]{"underbracket", new Integer(0x23B5)},
      new Object[]{"underparen", new Integer(0x23DD)},
      new Object[]{"underbrace", new Integer(0x23DF)},
   };

   private static final Object[][] RADICALS = new Object[][]
   {
      new Object[]{"sqrt", new Integer(0x221A)},
      new Object[]{"longdivision", new Integer(0x27CC)},
   };
}
