/*
 * TEMPO - Topographic Eeg Mapping PrOgram.
 * 
 * Copyright (C) 1995, 1996, 2003, 2004 Aleksandar B. Samardzic
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <assert.h>
#include <ctype.h>
#include <float.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fft.h"
#include "input.h"

/* EDF format version string. */
#define VERSION "0       "

Input          *
input_create(char *name, Sensors * sensors)
{
	Input          *input;	/* Input object. */
	FILE           *file;	/* Input file. */
	char            buffer[81];	/* Input buffer. */
        time_t          date_and_time;	/* Calendar time for start date
                                           and time of recording. */
	int             count;	/* Number of signals in file. */
	Signal         *signal;	/* Signal object. */
	int             result;	/* Result returned by input functions. */
	int             i, j;	/* Loop indices. */

	/* Create input object. */
	input = (Input *) malloc(sizeof(Input));
	assert(input != NULL);

	/* Open input file. */
	file = fopen(name, "r");
	assert(file != NULL);

	/* Read and check version string. */
	result = fread(buffer, 8, 1, file);
	assert(result == 1);
	assert(strncmp(buffer, VERSION, 8) == 0);

	/* Read patient information. */
	result = fread(buffer, 80, 1, file);
	assert(result == 1);
	buffer[80] = 0;
	for (i = 79; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
	input->patient = (char *)malloc((i + 2) * sizeof(char));
	assert(input->patient != NULL);
	strcpy(input->patient, buffer);

	/* Read recording information. */
	result = fread(buffer, 80, 1, file);
	assert(result == 1);
	buffer[80] = 0;
	for (i = 79; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
	input->recording = (char *)malloc((i + 2) * sizeof(char));
	assert(input->recording != NULL);
	strcpy(input->recording, buffer);

	/* Read start date and time. */
	result = fread(buffer, 8, 1, file);
	assert(result == 1);
	buffer[8] = 0;
	for (i = 7; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
        assert(sscanf(buffer, "%d.%d.%d", &input->date_and_time.tm_mday, &input->date_and_time.tm_mon, &input->date_and_time.tm_year) == 3);
	result = fread(buffer, 8, 1, file);
	assert(result == 1);
	buffer[8] = 0;
	for (i = 7; i >= 0 && buffer[i] == ' '; i--)
		buffer[i] = 0;
        assert(sscanf(buffer, "%d.%d.%d", &input->date_and_time.tm_hour, &input->date_and_time.tm_min, &input->date_and_time.tm_sec) == 3);
        assert((date_and_time = mktime(&input->date_and_time)) != (time_t) -1);
        assert(localtime_r(&date_and_time, &input->date_and_time) != NULL);

	/* Read number of bytes in header. */
	result = fread(buffer, 8, 1, file);
	assert(result == 1);
	buffer[8] = 0;
	input->offset = atoi(buffer);

	/* Read first reserved block. */
	result = fread(buffer, 44, 1, file);
	assert(result == 1);

	/* Read number of data records and verify that it is greater than 0. */
	result = fread(buffer, 8, 1, file);
	assert(result == 1);
	buffer[8] = 0;
	input->data = atoi(buffer);
	assert(input->data > 0);

	/* Read duration of data record. */
	result = fread(buffer, 8, 1, file);
	assert(result == 1);
	buffer[8] = 0;
	input->duration = atoi(buffer);

	/* Read number of signals in file. */
	result = fread(buffer, 4, 1, file);
	assert(result == 1);
	buffer[4] = 0;
	count = atoi(buffer);

	/* Allocate memory for signal objects and sensor indices. */
	input->count = 0;
	input->signals = (Signal **) malloc(count * sizeof(Signal *));
	input->sensors = (int *)malloc(count * sizeof(int));

	/* Create signal objects. */
	for (i = 0; i < count; i++) {
		/* Read signal name. */
		result = fread(buffer, 16, 1, file);
		assert(result == 1);
		buffer[16] = 0;
		for (j = 15; j >= 0 && buffer[j] == ' '; j--)
			buffer[j] = 0;

		/* Compare signal name with all known sensor names... */
		for (j = 0; j < sensors->count; j++)
			if (!strcmp(buffer, sensors->names[j])) {
				/*
				 * ...and if match found, create signal
				 * object and read signal information from
				 * file,...
				 */
				signal = signal_create();
				signal_load_edf(signal, name, i);

				/*
				 * ...then remember signal object and index
				 * of corresponding sensor.
				 */
				input->signals[input->count] = signal;
				input->sensors[input->count] = j;
				input->count++;
				break;
			}
	}

	/* Adjust size of signal objects and sensor indices arrays. */
	input->signals = (Signal **) realloc(input->signals, input->count * sizeof(Signal *));
	input->sensors = (int *)realloc(input->sensors, input->count * sizeof(int));

	/* Check that at least minimal number of signals exist in input file. */
	assert(input->count >= MIN_COUNT);

	/* Calculate last record index. */
	for (i = 1; i < input->count; i++)
		assert(input->signals[i]->count == input->signals[0]->count);
	input->last = input->data * input->signals[0]->count - 1;

	/* Calculate sampling frequency. */
	input->frequency = (float)input->signals[0]->count / input->duration;

	/* Calculate physical minimum and maximum for whole input file. */
	input->physical_lo = FLT_MAX;
	input->physical_hi = -FLT_MAX;
	for (i = 0; i < input->count; i++) {
		if (input->physical_lo > input->signals[i]->physical_lo)
			input->physical_lo = input->signals[i]->physical_lo;
		if (input->physical_hi < input->signals[i]->physical_hi)
			input->physical_hi = input->signals[i]->physical_hi;
	}

	/* Close input file. */
	fclose(file);

	return input;
}

void
input_destroy(Input * input)
{
	int             i;	/* Loop index. */

	/* Free memory used for patient and recording information. */
	free(input->patient);
	free(input->recording);

	/*
	 * Destroy signal objects and free memory used for signal objects and
	 * sensor indices arrays.
	 */
	for (i = 0; i < input->count; i++)
		signal_destroy(input->signals[i]);
	free(input->signals);
	free(input->sensors);

	/* Free memory used for input object. */
	free(input);
}

void
input_values(Input * input, int at, int score, int count, float *values,...)
{
	va_list         ap;	/* Variable argument list. */
	int             window;	/* DFT window. */
	int             frequency;	/* Frequency to return values for. */
	int             p;	/* DFT window specified as power of number 2. */
	float          *in_re, *in_im;	/* Input vector for DFT calculation. */
	float          *out_re, *out_im;	/* Output vector for DFT
						 * calculation. */
	int             i;	/* Loop index. */

	/* Depending on score, calculate requested values. */
	switch (score) {
	case RAW_POTENTIAL:
		/* Read raw potential values from input file. */
		for (i = 0; i < count; i++)
			signal_values(input->signals[i], at, 1, &values[i]);
		break;

	case DFT_AMPLITUDE:
		/* Extract rest of function arguments. */
		va_start(ap, values);
		window = va_arg(ap, int);
		frequency = va_arg(ap, int);
		va_end(ap);

		/* Check that DFT window is power of 2 and find which one. */
		assert(at >= window - 1);
		assert(window >= 2);
		for (p = 0; (window & (1 << p)) == 0; p++);
		assert((window & ~(1 << p)) == 0);

		/* Allocate memory for input and output DFT vectors. */
		in_re = (float *)malloc(window * sizeof(float));
		assert(in_re != NULL);
		in_im = (float *)malloc(window * sizeof(float));
		assert(in_im != NULL);
		out_re = (float *)malloc(window * sizeof(float));
		assert(out_re != NULL);
		out_im = (float *)malloc(window * sizeof(float));
		assert(out_im != NULL);

		/*
		 * For each signal, read raw potential values from input
		 * file, calculate DFT of these values and extract DFT
		 * amplitude at requested frequency as result.
		 */
		memset(in_im, 0, window * sizeof(float));
		for (i = 0; i < count; i++) {
			signal_values(input->signals[i], at - (window - 1), window, in_re);
			fft(p, in_re, in_im, out_re, out_im);
			values[i] = (float)sqrt(out_re[frequency] * out_re[frequency] + out_im[frequency] * out_im[frequency]);
		}

		/* Free memory used for input and output vectors. */
		free(in_re);
		free(in_im);
		free(out_re);
		free(out_im);
		break;

	case DFT_PHASE:
		/* Extract rest of function arguments. */
		va_start(ap, values);
		window = va_arg(ap, int);
		frequency = va_arg(ap, int);
		va_end(ap);

		/* Check that DFT window is power of 2 and find which one. */
		assert(at >= window - 1);
		assert(window >= 2);
		for (p = 0; (window & (1 << p)) == 0; p++);
		assert((window & ~(1 << p)) == 0);

		/* Allocate memory for input and output DFT vectors. */
		in_re = (float *)malloc(window * sizeof(float));
		assert(in_re != NULL);
		in_im = (float *)malloc(window * sizeof(float));
		assert(in_im != NULL);
		out_re = (float *)malloc(window * sizeof(float));
		assert(out_re != NULL);
		out_im = (float *)malloc(window * sizeof(float));
		assert(out_im != NULL);

		/*
		 * For each signal, read raw potential values from input
		 * file, calculate DFT of these values and extract DFT phase
		 * at requested frequency as result.
		 */
		memset(in_im, 0, window * sizeof(float));
		for (i = 0; i < count; i++) {
			signal_values(input->signals[i], at - (window - 1), window, in_re);
			fft(p, in_re, in_im, out_re, out_im);
			values[i] = (float)atan2(out_im[frequency], out_re[frequency]);
		}

		/* Free memory used for input and output vectors. */
		free(in_re);
		free(in_im);
		free(out_re);
		free(out_im);
		break;
	}
}
