
/*

    File: procinfo.c

    Copyright (C) 2005  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <ctype.h>
#include <signal.h>
#include <wait.h>
#include <errno.h>
#include <time.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <syslog.h>
#include <sys/time.h>



#include "lib.h"
#include "ip-lib.h"
#include "tcpproxy.h"
#include "procinfo.h"


char	varname[40] =		"PROXY_";

char	statdir[200] =		"";
char	sessiondir[200] =	"";

procinfo_t pi;


struct _errortype {
    int		num;
    char	name[20];
    } errortype[] = {
	{ ERR_ACCESS, "ACCESS" },
	{ ERR_CONNECT, "CONNECT" },
	{ ERR_TIMEOUT, "TIMEOUT" },
	{ ERR_SERVER, "SERVER" },
	{ ERR_CLIENT, "CLIENT" },
	{ ERR_PROXY, "PROXY" },
	{ ERR_INFO, "INFO" },
	{ ERR_OK, "OK" },
	{ ERR_ANY, "ANY" },
	{ ERR_ANY, "ALL" },
	{ 1, "NONE" },
	{ 0, "" }
    };


int init_procinfo(char *vp)
{
	memset(&pi, 0, sizeof(procinfo_t));
	copy_string(varname, vp, sizeof(varname));

	return (0);
}

FILE *getstatfp(void)
{
	if (*statdir == 0)
		return (NULL);

	if (*pi.statfile == 0) {
		snprintf(pi.statfile, sizeof(pi.statfile) - 2, "%s/%s-%05d.stat",
				statdir, program, getpid());
		if ((pi.statfp = fopen(pi.statfile, "w")) == NULL) {
			printerror(0 | ERR_INFO, "-INFO", "can't open statfile %s, error= %s",
					pi.statfile, strerror(errno));
			}
		}

	return (pi.statfp);
}

int geterrcode(char *name)
{
	int	i;

	for (i=0; errortype[i].num != 0; i++) {
		if (strcasecmp(errortype[i].name, name) == 0)
			return (errortype[i].num);
		}

	return (0);
}

int set_sessionmode(char *word, char *filename, int lineno)
{
	if (strcmp(word, "none") == 0  ||  strcmp(word, "off") == 0  ||
	    strcmp(word, "no") == 0) {
		pi.errormode = ERRMODE_NONE;
		}
	else if (strcmp(word, "keep") == 0)
		pi.errormode = ERRMODE_KEEP;
	else if (strcmp(word, "handler") == 0)
		pi.errormode = ERRMODE_HANDLER;
	else
		printerror(1, "-ERR", "bad keyword: %s, %s:%d", word, filename, lineno);

	return (pi.errormode);
}

char *set_exithandler(char *handler)
{
	if (handler != NULL)
		copy_string(pi.errorhandler, handler, sizeof(pi.errorhandler));

	return (pi.errorhandler);
}

int set_exitcodes(int mask)
{
	pi.errortypes = mask;
	return (pi.errortypes);
}



int run_errorhandler(int error)
{
	int	pid;

	if (pi.sessionfp == NULL)
		return (0);	/* Nothing to do. */

	if ((error & pi.errortypes) == 0)
		return (0);	/* Not configured to run for this errorcode. */

	if (pi.errormode == ERRMODE_KEEP) {
		*pi.sessionfile = 0;	/* Clear filename to prevent removal on exit. */
		return (0);
		}

	if (pi.errormode != ERRMODE_HANDLER)
		return (0);
	else if (*pi.errorhandler == 0  ||  pi.sessionfp == NULL)
		return (0);	/* Not configured to run at all. */

	pi.errorcode = error;
	if ((pid = fork()) < 0) {
		*pi.errorhandler = 0;
		printerror(1, "-ERR", "can't fork, error= %s", strerror(errno));
		}
	else if (pid == 0) {
		int	i, fd, argc;
		char	*argv[32];

/*		set_variables(x); */
		setnumvar(varname, "STATUSCODE", error);
		for (i=0; errortype[i].num != 0; i++) {
			if ((error & errortype[i].num) != 0) {
				setvar(varname, "STATUS", errortype[i].name);
				break;
				}
			}

/*		if ((fd = open(pi.sessionfile, O_RDONLY)) < 0) { */
		if ((fd = fileno(pi.sessionfp)) < 0  ||  lseek(fd, 0, SEEK_SET) != 0) {
			printerror(1, "-ERR", "can't open session logfile %s, error= %s",
					pi.sessionfile, strerror(errno));
			}

		dup2(fd, 0);
		argc = split(pi.errorhandler, argv, ' ', 30);
		argv[argc] = NULL;

		execvp(argv[0], argv);
		printerror(1, "-ERR", "can't start error handler %s, error= %s",
				argv[0], strerror(errno));
		}

	return (0);
}

char *setpidfile(char *pidfile)
{
	if (pidfile == NULL  ||  *pidfile == 0)
		snprintf (pi.pidfile, sizeof(pi.pidfile) - 2, "/var/run/%s.pid", program);
	else
		copy_string(pi.pidfile, pidfile, sizeof(pi.pidfile));

	return (pi.pidfile);
}


void exithandler(void)
{
	if (debug != 0)
		printerror(0, "+DEBUG", "exithandler mainpid= %u, statfile= %s", pi.mainpid, pi.statfile);

	if (pi.mainpid == getpid()) {
		if (*pi.pidfile != 0) {
			if (unlink(pi.pidfile) != 0) {
				printerror(0, "-ERR", "can't unlink pidfile %s, error= %s",
						pi.pidfile, strerror(errno));
				}
			}
		}

	if (pi.statfp != NULL) {
		rewind(pi.statfp);
		fprintf (pi.statfp, "\n");
		fclose (pi.statfp);

		if ((pi.statfp = fopen(pi.statfile, "w")) != NULL) {
			fprintf (pi.statfp, "%s", "");
			fclose (pi.statfp);
			}
		}

	if (*pi.statfile != 0) {
		if (unlink(pi.statfile) != 0) {
			printerror(0, "-INFO", "can't unlink statfile %s, error= %s",
					pi.statfile, strerror(errno));
			}
		}

	if (pi.sessionfp != NULL)
		fclose (pi.sessionfp);

	if (*pi.sessionfile != 0  &&  pi.errormode != ERRMODE_KEEP)
		unlink(pi.sessionfile);

	return;
}


